package API::ClientOptions;

# $Id$

=head1 NAME

    API::ClientOptions

=head1 SYNOPSIS

    $result = API::ClientOptions::set(1566, {'api_enabled' => 'No'});
    $result = API::ClientOptions::mass_set([1565, 1566], {'api_enabled' => 'No'});
    $result = API::ClientOptions::multi_set([{ClientID =>1565, 'api_enabled' => 'No'}]);
    $options = API::ClientOptions::get_one(1565, 'api_enabled');
    $options = API::ClientOptions::get(1566, ['api_enabled']);
    $options = API::ClientOptions::mass_get([1566, 1567], ['api_enabled']);

=head1 DESCRIPTION

    Работа с настройками API для клиента

=cut

use Direct::Modern;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Validate qw/ is_valid_id is_valid_int /;

use base qw/Exporter/;
our @EXPORT = qw//;

my %fields_hash = (
    ClientID         => \&is_valid_id,
    api_enabled      => \&is_valid_api_enabled,
    api5_units_daily => \&is_valid_int,
);

=head2 get_one($client_id, $field)

    Получает ровно одну настройку клиента по ClientID

=cut

sub get_one {
    my ($client_id, $field) = @_;
    my $options = mass_get([$client_id], [$field]);
    return $options->[0]{$field};
}

=head2 get($client_id, $fields)

    Получает настройки клиента по ClientID
    Возвращает ссылку на хеш с настройками

=cut

sub get {
    my ($client_id, $fields) = @_;
    my $options = mass_get([$client_id], $fields);
    return $options->[0];
}

=head2 mass_get($clients_ids, $fields)

    Получает настройки клиентов по списку ClientID

=cut

sub mass_get {
    my ($clients_ids, $fields) = @_;
    croak "empty fields" unless $fields && @$fields;
    my @wrong_fields = grep {!exists $fields_hash{$_}} @$fields;
    croak "wrong fields".join(',', @wrong_fields) if scalar @wrong_fields;

    my $fields_str = join(',', @$fields);
    my $options = get_all_sql(PPC(ClientID => $clients_ids),
        ["SELECT $fields_str FROM clients_api_options",
         WHERE => {ClientID => SHARD_IDS}]);
    return $options;
}

=head2 set($client_id, $options)

    Сохраняет настройки клиента
    Принимет ClientID и ссылку на хеш с настройками

=cut

sub set {
    my ($client_id, $options) = @_;
    $options->{ClientID} = $client_id;
    return mass_set([$client_id], $options);
}

=head2 mass_set ($clients_ids, $options)

    Сохраняет настройки клиентов
    полезна если нескольким клиентам нужно сохранить одинаковую настройку
    Принимет список ClientID и ссылку на хеш с настройками

=cut

sub mass_set {
    my ($clients_ids, $options) = @_;
    validate($options);
    my $result = do_update_table(PPC(ClientID => $clients_ids),
            'clients_api_options', $options,
             where => {ClientID => SHARD_IDS});
    return $result > 0 ? 1 : 0;
}


=head2 add

    Добавляет клиенту настройки

=cut

sub add {
    my ($client_id, $options) = @_;
    $options->{ClientID} = $client_id;
    validate($options);
    my $result = do_insert_into_table(PPC(ClientID => $client_id),
            'clients_api_options', $options,
            on_duplicate_key_update => 1,
            key => 'ClientID'
    );
    return $result > 0 ? 1 : 0;
}

=head2 validate

    Проверяет правильность переданных настроек

=cut

sub validate {
    my $options = shift;
    for my $option (keys %$options){
        croak "wrong option" unless exists $fields_hash{$option};
        my $is_valid = $fields_hash{$option}->($options->{$option});
        croak "option has wrong value" unless $is_valid;
    }
    return;
}

=head2 is_valid_api_enabled

    Проверяет api_enabled

=cut

sub is_valid_api_enabled {
    my $api_enabled = shift;
    if ($api_enabled =~ /(Yes|No|Default)/) {
        return 1;
    }
    return 0;
    
}

1;
