package API::ConvertFrom;

# $Id$

=pod

    Модуль для методов которые конвертируют данные из формата API во внутренний формат

=cut

use strict;
use warnings;

use Yandex::Validate;
use Yandex::I18n;

use TextTools;
use Settings;

use Phrase;
use TimeTarget;
use URLDomain qw/clear_banner_href/;

use APICommon;
use API::Errors;
use API::Settings;
use API::Strategies qw/
    strategy_sets_from_API_v4_format
    strategy_sets_from_API_format
    context_strategy_sets_from_API_format/;

require Exporter;

use base qw/Exporter/;

our @EXPORT_OK = qw(
    convert_params
);

use utf8;


=head2 convert_params

  convert_params('имя преобразования', $hash_ref_params, 'флаг - не удалять ли старые параметры кот. были преобразованы');

  %gl_cnv_tbl - хеш хешей:
     "название преобразования" =>
           "имя поля параметров" => "новое имя" или ссылка на ф-цию преобразования

=cut

{

my %gl_cnv_tbl = (

    campaign_v4 =>
    {    qw/

            Login                   ulogin
            CampaignID              cid
            FIO                     fio
            Name                    name

            StatusOpenStat          statusOpenStat

            StatusBehavior          statusBehavior
            DisabledIps             disabledIps
            DisabledDomains         DontShow

            ContextPricePercent     ContextPriceCoef

            AutoOptimization        autoOptimization

            MinusKeywords           campaign_minus_words

           /,

           Strategy       => \&strategy_sets_from_API_v4_format,

           StatusContextStop       => sub {
                my $hash = shift;
                $hash->{statusContextStop} = $hash->{StatusContextStop} eq 'Yes'? 1: undef;
           },

           StatusMetricaControl => sub {
                my $hash = shift;
                $hash->{statusMetricaControl} = $hash->{StatusMetricaControl} eq 'Yes'? 1: undef;
            },
           ContextLimit => sub {
                my $hash = shift;
                my $value = 0;
                my $disable_context = 'No';
                if($hash->{ContextLimit} eq 'Unlimited'){
                    $value = 255;
                } elsif($hash->{ContextLimit} eq 'Limited'){
                    if (defined $hash->{ContextLimitSum}) {
                        $disable_context = $hash->{ContextLimitSum} == 0 ? 'Yes' : 'No';
                        $value = $hash->{ContextLimitSum};
                    } else {
                        $value = 0;
                    }
                } elsif($hash->{ContextLimit} eq 'Default'){
                    $value = 0;
                }
                $hash->{ContextLimit} = $value;
                $hash->{dontShowYacontext} = $disable_context;
            },

           SmsNotification => sub {
                my $hash = shift;

                my %external2internal_sms = (
                    'MoneyOutSms' => 'active_orders_money_out_sms',
                    'MoneyInSms' => 'notify_order_money_in_sms',
                    'ModerateResultSms' => 'moderate_result_sms',
                    'MetricaSms' => 'notify_metrica_control_sms'
                );

                foreach my $k (keys %external2internal_sms) {
                    if (defined $hash->{SmsNotification}->{$k}
                             && $hash->{SmsNotification}->{$k} eq "Yes") {

                        $hash->{$external2internal_sms{$k}} = 1;
                    }
                }

                my $sms_time_from = $hash->{SmsNotification}->{SmsTimeFrom};
                if($sms_time_from){
                    $sms_time_from =~ /^(\d\d)\:(\d\d)$/;
                    $hash->{sms_time_hour_from} = $1;
                    $hash->{sms_time_min_from} = $2;
                }

                my $sms_time_to = $hash->{SmsNotification}->{SmsTimeTo};
                if($sms_time_to){
                    $sms_time_to =~ /^(\d\d)\:(\d\d)$/;
                    $hash->{sms_time_hour_to} = $1;
                    $hash->{sms_time_min_to} = $2;
                }
            },
            EmailNotification => sub {
                my $hash = shift;

                smartstrip ( $hash->{EmailNotification}->{Email} );

                if (defined $hash->{EmailNotification}->{SendAccNews} && $hash->{EmailNotification}->{SendAccNews} eq 'Yes') {
                    $hash->{sendAccNews} = 'Yes';
                }

                if (defined $hash->{EmailNotification}->{sendWarn} && $hash->{EmailNotification}->{sendWarn} eq 'Yes') {
                    $hash->{sendWarn} = 'Yes';
                }
                $hash->{email}                = $hash->{EmailNotification}->{Email};
                $hash->{warnPlaceInterval}    = $hash->{EmailNotification}->{WarnPlaceInterval};
                $hash->{money_warning_value}  = $hash->{EmailNotification}->{MoneyWarningValue};
            },
            TimeTarget => \&prepare_api_timetarget,
            StartDate => sub {
                my $hash = shift;
                # TODO: вынести проверку в API::Validate
                dieSOAP('BadDate', iget('Поле StartDate некорректно')) if ! is_valid_date($hash->{StartDate});
                @{$hash}{ qw/yyyy mm dd/ } = $hash->{StartDate} =~ /^(\d\d\d\d)-?(\d\d)-?(\d\d)$/;
            },

            ConsiderTimeTarget => sub {
                my $hash = shift;
                $hash->{fairAuction} = 'Yes' if $hash->{ConsiderTimeTarget} eq 'Yes';
            },
    },

    campaign =>
    {    qw/

            Login                   ulogin
            CampaignID              cid
            FIO                     fio
            Name                    name

            StatusOpenStat          statusOpenStat

            StatusBehavior          statusBehavior
            DisabledIps             disabledIps
            DisabledDomains         DontShow

            AutoOptimization        autoOptimization

            MinusKeywords           campaign_minus_words
            StartDate               start_date
            FinishDate              finish_date
            MobileBidAdjustment     mobile_multiplier_pct


           /,

           MobileBidAdjustment => sub {
                my $hash = shift;
                $hash->{mobile_multiplier_pct} = $hash->{MobileBidAdjustment} == 100
                                            ? undef : $hash->{MobileBidAdjustment};
           },

           ClickTrackingEnabled => sub {
                my $hash = shift;
                $hash->{status_click_track} =
                    (defined $hash->{ClickTrackingEnabled})
                        ? ( $hash->{ClickTrackingEnabled} eq 'Yes'? 1: 0 )
                        : undef
           },

           DayBudget => sub {
                my $hash = shift;

                if (defined $hash->{DayBudget}
                    && ($hash->{DayBudget}{Amount} || 0) > 0
                    && $hash->{Strategy}{StrategyName} =~ /^(HighestPosition|RightBlockHighest|LowestCostGuarantee|LowestCost|LowestCostPremium|ShowsDisabled)$/
                    && $hash->{ContextStrategy}{StrategyName} =~ /^(MaximumCoverage|Default|ShowsDisabled)$/) {
                    $hash->{day_budget} = $hash->{DayBudget}{Amount};
                    $hash->{day_budget_show_mode} = lc($hash->{DayBudget}{SpendMode} || "Default");
                } else {
                    $hash->{day_budget} = 0;
                }
           },

           AccountDayBudget => sub {
                my $hash = shift;

                if (defined $hash->{AccountDayBudget} && ($hash->{AccountDayBudget}{Amount} || 0) > 0) {
                    $hash->{day_budget} = $hash->{AccountDayBudget}{Amount};
                    $hash->{day_budget_show_mode} = lc($hash->{AccountDayBudget}{SpendMode} || "Default");
                } else {
                    $hash->{day_budget} = 0;
                }
           },

           Strategy       => \&strategy_sets_from_API_format,

           ContextStrategy => \&context_strategy_sets_from_API_format,

           StatusContextStop       => sub {
                my $hash = shift;
                $hash->{statusContextStop} = $hash->{StatusContextStop} eq 'Yes'? 1: undef;
           },

           StatusMetricaControl => sub {
                my $hash = shift;
                $hash->{statusMetricaControl} = $hash->{StatusMetricaControl} eq 'Yes'? 1: undef;
           },

           SmsNotification => sub {
                my $hash = shift;

                my %external2internal_sms = (
                    'MoneyOutSms' => 'active_orders_money_out_sms',
                    'MoneyInSms' => 'notify_order_money_in_sms',
                    'ModerateResultSms' => 'moderate_result_sms',
                    'MetricaSms' => 'notify_metrica_control_sms',
                    'PausedByDayBudgetSms' => 'paused_by_day_budget_sms',
                );

                foreach my $k (keys %external2internal_sms) {
                    if (defined $hash->{SmsNotification}->{$k}
                             && $hash->{SmsNotification}->{$k} eq "Yes") {

                        $hash->{$external2internal_sms{$k}} = 1;
                    }
                }

                my $sms_time_from = $hash->{SmsNotification}->{SmsTimeFrom};
                if($sms_time_from){
                    $sms_time_from =~ /^(\d\d)\:(\d\d)$/;
                    $hash->{sms_time_hour_from} = $1;
                    $hash->{sms_time_min_from} = $2;
                }

                my $sms_time_to = $hash->{SmsNotification}->{SmsTimeTo};
                if($sms_time_to){
                    $sms_time_to =~ /^(\d\d)\:(\d\d)$/;
                    $hash->{sms_time_hour_to} = $1;
                    $hash->{sms_time_min_to} = $2;
                }

            },

            EmailNotification => sub {
                my $hash = shift;

                smartstrip ($hash->{EmailNotification}->{Email});
                # -- IF defined добавлено потому, что иначе в error логе появлялось сообщение при неустановленных полях:
                # -- Use of uninitialized value in string eq at /var/www/...../protected/APICommon.pm line 3104.
                # -- Use of uninitialized value in string eq at /var/www/...../protected/APICommon.pm line 3105.
                $hash->{sendAccNews}          = $hash->{EmailNotification}->{SendAccNews} if defined $hash->{EmailNotification}->{SendAccNews} && $hash->{EmailNotification}->{SendAccNews} eq 'Yes';
                $hash->{sendWarn}             = $hash->{EmailNotification}->{SendWarn}    if defined $hash->{EmailNotification}->{SendWarn} && $hash->{EmailNotification}->{SendWarn} eq 'Yes';
                $hash->{email}                = $hash->{EmailNotification}->{Email};
                $hash->{warnPlaceInterval}    = $hash->{EmailNotification}->{WarnPlaceInterval};
                $hash->{money_warning_value}  = $hash->{EmailNotification}->{MoneyWarningValue};
                $hash->{paused_by_day_budget_email} = 1 if defined $hash->{EmailNotification}{PausedByDayBudget} && $hash->{EmailNotification}{PausedByDayBudget} eq 'Yes';
            },

            TimeTarget => \&prepare_api_timetarget,

            StartDate => sub {
                my $hash = shift;
                # TODO: вынести проверку в API::Validate
                dieSOAP('BadDate', iget('Поле StartDate некорректно')) if ! is_valid_date($hash->{StartDate});
                @{$hash}{ qw/yyyy mm dd/ } = $hash->{StartDate} =~ /^(\d\d\d\d)-?(\d\d)-?(\d\d)$/;
            },

            ConsiderTimeTarget => sub {
                my $hash = shift;
                $hash->{fairAuction} = 'Yes' if $hash->{ConsiderTimeTarget} eq 'Yes';
            },

            ExtendedAdTitleEnabled => sub {
                my $hash = shift;
                $hash->{opts}{no_title_substitute} = $hash->{ExtendedAdTitleEnabled} eq 'No' ? 1 : 0 if defined $hash->{ExtendedAdTitleEnabled};
            },

            MaintainNetworkCPC => sub {
               my $hash = shift;
               if (defined $hash->{MaintainNetworkCPC}){
                    if ($hash->{MaintainNetworkCPC} eq 'No') {
                         $hash->{opts}{enable_cpc_hold} = 0;
                    } else {
                         $hash->{opts}{enable_cpc_hold} = 1;
                    }
               }
            },
    },
    #:::::::::::::::::::::::::::::::::::::::
    banners =>
    {    qw/
            CampaignID          cid
            BannerID            bid
            Title               title
            Title2      title_extension
            Text                body
            Href                href
            Domain              domain
            GeoID               geo_id
            Type                banner_type

            AdGroupID           pid
            AdGroupName         group_name

            Phrases             phrases
            Phrase              phrase
            PhraseID            id
            Price               price
            ContextPrice        price_context
            RubricID            id
            AutoBudgetPriority  autobudgetPriority

            Apart         apart
            MetroStationID metro
            Build         build
            City          city
            CityCode      city_code
            ContactEmail  contact_email
            ContactPerson contactperson
            Country       country
            CountryCode   country_code
            ExtraMessage  extra_message
            House         house
            IMClient      im_client
            IMLogin       im_login
            CompanyName   name
            Phone         phone
            PhoneExt      ext
            Street        street
            OGRN          ogrn
            MinusKeywords banner_minus_words

            MediaplanAdID mbid
            SourceAdID    source_bid
            AdGroupMobileBidAdjustment mobile_multiplier_pct
        /,

        Type => sub {
            my $hash = shift;
            $hash->{banner_type} = $hash->{Type} ? lc($hash->{Type}) : 'desktop';
        },

        PointOnMap => sub {
            my $hash = shift;
            if ($hash->{PointOnMap}) {
                $hash->{manual_point} = join(',', map {sprintf("%.6f", $_)} @{$hash->{PointOnMap}}{qw/x y/});
                $hash->{manual_bounds} = join(',', map {sprintf("%.6f", $_)} @{$hash->{PointOnMap}}{qw/x1 y1 x2 y2/});
            }
        },

        Geo => sub {
            my $hash = shift;
            $hash->{geo} = $hash->{Geo};
        },

        Sitelinks => sub {
            my $hash = shift;
            $hash->{sitelinks} = [];
            foreach my $sl (@{$hash->{Sitelinks} || []}) {
                push @{$hash->{sitelinks}}, { title => smartstrip2($sl->{title}), href => clear_banner_href($sl->{href}) };
            }
        },

        #..................................
        ContactInfo => sub {
            my $hash = shift;
            my $ContactInfo = $hash->{ContactInfo};

            if($ContactInfo){

                $hash->{need_validate_ci} = 1;
                for my $k (keys %$ContactInfo) {
                    if ($k eq 'IMClient') {
                        $hash->{$k} = lc($ContactInfo->{$k});
                    } else {
                        $hash->{$k} = $ContactInfo->{$k};
                    }

                }
            }
        },

        #..................................
        WorkTime => sub {
            my $hash = shift;

            my @arr = split /;/, $hash->{WorkTime};
            $hash->{worktime} = '';
            while (@arr) {
                $hash->{worktime} .= join('#', splice(@arr, 0, 6)) . ';';
            }
            $hash->{worktime} =~ s/;$//;
        },
    },

    retargeting_ext => {
        %APICommon::RETARGETING_EXT_CONVERSION,
        is_accessible => sub {
            my $hash = shift;
            $hash->{IsAccessible} = $hash->{is_accessible} ? 'Yes' : 'No';
        },
    },

    retargeting_int => {
        reverse %APICommon::RETARGETING_EXT_CONVERSION,
    },

    banner_retargeting => {
        AdID => 'bid',
        RetargetingConditionID => 'ret_cond_id',
        RetargetingID => 'ret_id',
        StatusPaused => sub {
            my $hash = shift;
            $hash->{is_suspended} = ($hash->{StatusPaused} && $hash->{StatusPaused} eq 'Yes')? 1 : 0;
        },
        AutoBudgetPriority => sub {
            my $hash = shift;
            $hash->{autobudgetPriority} = $hash->{AutoBudgetPriority} ? $Phrase::PRIORITY_VALUES{$hash->{AutoBudgetPriority}} : undef;

        },
        ContextPrice => 'price_context',
        Currency  => 'currency',
    },

);

# хитрый хак, который нужен для перехода с precise на trusty:
# на precise порядок ключей в %gl_cnv_tbl всегда был одинаковый, и код был написан так,
# чтобы правильно работать при таком порядке (порядок влияет на порядок вызова анонимных процедур,
# которые работают с одними и теми же данными)
# на trusty порядок ключей меняется от запроса к запросу, так что написали код, который фиксирует
# порядок ключей как на precise
# почему мы так можем: этот код относится к APIv4, который через какое-то время отключим
# -- andy-ilyin 20150826
my %gl_cnv_tbl_order = (
    banner_retargeting => [ qw(
        Currency ContextPrice RetargetingConditionID RetargetingID AutoBudgetPriority StatusPaused AdID
    ) ],
    retargeting_int => [ qw(
        Time IsAccessible GoalID Type RetargetingConditionName RetargetingConditionID RetargetingCondition
        Goals RetargetingConditionDescription
    ) ],
    retargeting_ext => [ qw(
        goal_id condition_desc time ret_cond_id condition_name is_accessible type goals condition
    ) ],
    campaign => [ qw(
        EmailNotification StatusBehavior DisabledDomains Login Strategy
        MinusKeywords TimeTarget StatusOpenStat ClickTrackingEnabled CampaignID
        MobileBidAdjustment ExtendedAdTitleEnabled StartDate ContextStrategy StatusMetricaControl
        DisabledIps AutoOptimization FIO DayBudget FinishDate StatusContextStop ConsiderTimeTarget
        SmsNotification Name
    ) ],
    banners => [ qw(
        ExtraMessage ContactPerson AdGroupName Street Text ContextPrice Type AutoBudgetPriority
        BannerID MediaplanAdID IMLogin Domain Price Phone OGRN SourceAdID PhoneExt CompanyName
        Href Phrases RubricID House City GeoID MinusKeywords Title CityCode IMClient PhraseID
        CampaignID Sitelinks ContactEmail WorkTime Build ContactInfo AdGroupMobileBidAdjustment
        Phrase Country AdGroupID Geo Apart MetroStationID PointOnMap CountryCode
    ) ],
    campaign_v4 => [ qw(
        EmailNotification StatusBehavior DisabledDomains Login Strategy
        MinusKeywords TimeTarget StatusOpenStat CampaignID StartDate ContextLimit
        StatusMetricaControl DisabledIps AutoOptimization FIO StatusContextStop ConsiderTimeTarget
        SmsNotification Name ContextPricePercent
    ) ],
);

my %gl_cnv_tbl_order_idx_map;
for my $gl_cnv_tbl_key ( keys %gl_cnv_tbl_order ) {
    my %key_map;
    for my $idx ( 0 .. $#{ $gl_cnv_tbl_order{$gl_cnv_tbl_key} } ) {
        $key_map{ $gl_cnv_tbl_order{$gl_cnv_tbl_key}->[$idx] } = $idx + 1;
    }
    $gl_cnv_tbl_order_idx_map{$gl_cnv_tbl_key} = \%key_map;
}

sub convert_params
{
    my ($type, $data, $dont_delete_old_converted_keys) = @_;

    if (ref($data) eq '') {
        return $data;
    }

    if (ref($data) eq 'ARRAY') {
        my @data_copy = @$data;
        for (@data_copy) {
            $_ = convert_params($type, $_, $dont_delete_old_converted_keys);
        }
        return \@data_copy;
    }

    my $cnv_tbl = $gl_cnv_tbl{$type} or dieSOAP('500');
    my $idx_map = $gl_cnv_tbl_order_idx_map{$type};

    if (ref($data) eq 'HASH') {
        my %data_copy = %$data;

        # здесь как раз сортировка в порядке %gl_cnv_tbl_order: сначала ключи, которые
        # там есть, по заданному в таблице порядку; потом все остальные ключи
        # в лексикографическом порядке.
        my @unordered_keys = keys %data_copy;
        my @ordered_keys = (
            ( sort { $idx_map->{$a} <=> $idx_map->{$b} } grep { exists $idx_map->{$_} } @unordered_keys ),
            ( sort { $a cmp $b } grep { ! exists $idx_map->{$_} } @unordered_keys ),
        );

        for my $key (@ordered_keys) {
            $data_copy{$key} = convert_params($type, $data_copy{$key}, $dont_delete_old_converted_keys);
            if (exists $cnv_tbl->{$key}) {
                if (ref($cnv_tbl->{$key}) ne 'CODE') {
                    $data_copy{$cnv_tbl->{$key}} = $data_copy{$key};
                } else {
                    $cnv_tbl->{$key}->(\%data_copy);
                }
                delete $data_copy{$key} unless $dont_delete_old_converted_keys;
            }

        }

        return \%data_copy;
    }
}


}


=head2 prepare_api_timetarget

    Преобразовать пришедшую структуру TimeTarget к внутреннему формату

=cut

sub prepare_api_timetarget {
    my $h = shift;

    return if ! defined $h->{TimeTarget};

    my $tt = $h->{TimeTarget};

    if (defined $tt->{ShowOnHolidays}) {
        if ($tt->{ShowOnHolidays} eq 'No') {
            $h->{time_target_holiday_dont_show} = 1;
            $h->{time_target_holiday} = 1;
        }
    }
    if (defined $tt->{HolidayShowFrom} || defined $tt->{HolidayShowTo}) {
        $h->{time_target_holiday} = 1;
        $h->{time_target_holiday_from} = $tt->{HolidayShowFrom};
        $h->{time_target_holiday_to} = $tt->{HolidayShowTo};
    }

    if (defined $tt->{WorkingHolidays}) {
        $h->{time_target_working_holiday} = $tt->{WorkingHolidays} eq 'Yes' ? 1 : 0;
    }

    if ($tt->{DaysHours}) {
        my $hours = {};
        for my $dh (@{$tt->{DaysHours}}) {
            for my $day (@{$dh->{Days}}) {
                for (my $i = 0; $i < scalar @{$dh->{Hours}}; $i++) {
                    if (! defined $dh->{BidCoefs} || $dh->{BidCoefs}[$i] > 0) {
                        $hours->{int($day)}->{int($dh->{Hours}[$i])} = defined $dh->{BidCoefs} ? $dh->{BidCoefs}[$i] : 100;
                    }
                }
            }
        }
        $h->{timeTarget} = '';
        for my $day (sort keys %$hours) {
            my @hours = keys %{$hours->{$day}};
            next if !@hours;
            $h->{timeTarget} .= $day.join("", sort map {TimeTarget::hour2letter($_).( _bidcoef_letter($hours->{$day}->{$_}) )} @hours);
        }
    }
    if ($tt->{TimeZone}) {
        my $tz = TimeTarget::get_timezone(timezone => $tt->{TimeZone});
        $h->{timezone_id} = $tz->{timezone_id};
    }

}

=head2 _bidcoef_letter(coef)

    Вернуть пустую строку, если коэффициент равен 100, иначе вернуть соответствующую букву

=cut

sub _bidcoef_letter {
    my $coef = shift;
    return '' if ($coef == 100);
    return TimeTarget::coef2letter($coef);
}

1;
