package API::Errors;
## no critic (TestingAndDebugging::RequireUseWarnings)

=pod

    $Id$
    
=head1 NAME
    
    API::Errors - содержит коды ошибок, возвращаемых в АПИ

=head1 DESCRIPTION

    Модуль содержит все коды ошибок с кратким описанием, возвращаемых в API
    А также функции, обрабатывающие ошибки единообразным для API образом.

=cut

use strict;

require Exporter;
use SOAP::Lite;
use Yandex::I18n;
use JSON;
use API::Methods::Keyword::Limits;

our @ISA = qw(Exporter);
our @EXPORT = qw(
    dieSOAP
    get_error_object
    get_warning_object
    dieSOAP_by_object
);

use utf8;

our %WARNINGS = (
    'NewMediaplanBanner' => {
            code => 201,
            string => iget('Создано новое объявление медиаплана'),
        },
    'UpdateMediaplanKeyword' => {
            code => 202,
            string => iget('Существующая ключевая фраза была обновлена'),
        },
    'EqMediplanKeywordsJoined' => {
            code => 203,
            string => iget("Эквивалентные ключевые фразы были объединены"),
        },
    'RemovedWhileUpdateMediaplanKeyword' => {
            code => 204,
            string => iget("Существующее ключевое слово было удалено, создано новое ключевое слово"),
        },
    'UpdateMediaplanRubric' => {
            code => 205,
            string => iget("Существующая рубрика каталога была обновлена"),
        },
    'EqMediplanRubricsJoined' => {
            code => 206,
            string => iget("Дубликаты рубрик были объединены"),
        },
    'RemovedWhileUpdateMediaplanRubric' => {
            code => 207,
            string => iget("Существующая рубрика каталога была удалена, создана новая рубрика каталога"),
        },
    'UpdateRetargeting' => {
            code => 208,
            string => iget('Существующее условие ретаргетинга было обновлено'),
        },
    'RetargetingContextPriceIgnored' => {
            code => 209,
            string => iget('Ставка для условия ретаргетинга была проигнорирована, так как включен автобюджет'),
        },
    'RetargetingConditionAlreadyExistsInAdGroup' => {
            code => 210,
            string => iget('Условие не было добавлено. Группа уже содержит указанное условие'),
        },
    'RetargetingConditionIDIgnored' => {
            code => 211,
            string => iget('Идентификатор условия RetargetingConditionID был проигнорирован')
        },
    "KeywordAlreadySuspended" => {
            code => 212,
            string => iget('Ключевое слово уже остановлено')
        },
    "KeywordNotSuspended" => {
            code => 213,
            string => iget('Ключевое слово не остановлено')
        },
);

#
# Диапазоны кодов ошибок
#
# stat        1-40
# common      41-70
# forecast    71-110
# campaigns   111-150
# banners     151-190
# phrases     191-240
# prices      241-250
# clients     251-270
#


our %ERRORS = (

    'BadCampaignID' => { 
            code => 1,
            string => iget('Неверный CampaignID')
         },    
    'NoStat' => { 
            code => 2,
            string => iget('Нет статистики для данной кампании')
         },    
    'BadDate' => { 
            code => 3,
            string => iget('Неверная дата')
         },    
    'BadTimeInterval' => { 
            code => 5,
            string => iget('Неверный временной интервал')
         },    
    'BadLimits' => { 
            code => 8,
            string => iget('Указанные лимиты неверны')
         },
    'NotArray' => { 
            code => 9,
            string => iget('Поле должно быть массивом')
         },   
    'BadBannerFilter' => { 
            code => 10,
            string => iget('Неверное условие фильтрации баннера')
         },
    'BadGeoFilter' => { 
            code => 11,
            string => iget('Неверное условие фильтрации геотаргетинга')
         },
    'BadPageNameFilter' => { 
            code => 12,
            string => iget('Неверное условие фильтрации площадки')
         },
    'BadPageTypeFilter' => { # на начало 2013 не используется. Дублируется №15.
            code => 13,	
            string => iget('Неверное условие фильтрации типа площадки')		
         },
    'BadPhraseFilter' => { 
            code => 14,
            string => iget('Неверное условие фильтрации параметров фразы')
         },
    'BadPageType' => { 
            code => 15,
            string => iget('Неверный тип площадки')
         },
    'BadGroupByDate' => { 
            code => 16,
            string => iget('Неверно указаны параметры группировки по дате')
         },
    'BadColumnName' => { 
            code => 17,
            string => iget('Неверное название поля отчёта')
         },
    'BadOffset' => { 
            code => 18,
            string => iget('Неверно указан отступ')
         },
    'BadTypeResultReport' => { 
            code => 20,
            string => iget('Неверный тип отчета')
         },
    'BadCompressionReport' => { 
            code => 21,
            string => iget('Неверно указан способ компрессии отчета')
         },
    'BadReportID' => { 
            code => 22,
            string => iget('Неверный ReportID')
         },
    'BadBids' => { 
            code => 23,
            string => iget('Неверный BannerID')
         },      
    'NoReport' => { 
            code => 24,
            string => iget('Указанный отчет не существует')
         },

    'NotYesNo' => { 
            code => 25,
            string => iget('Поле может содержать значения: Yes и No')
         },
    
    'NotDecPercentInt' => { 
            code => 26,
            string => iget('Значение должно быть от 0 до 100 и кратным 10')
         },
    'BadBannerID' => { 
            code => 27,
            string => iget('Неверный BannerID')
         }, 
    'BadPhraseID' => { 
            code => 28,
            string => iget('Неверный PhraseID')
         },          
    'BadCampaignType' => { 
            code => 29,
            string => iget('Неверно указан тип кампании')
         },          
    'EmptyArray' => { 
            code => 30,
            string => iget('Массив не должен быть пустым')
         },
    'ReportsStackFilled' => {
            code => 31,
            string => iget('Достигнут лимит отчетов в очереди')
        },
    'BadPositionType' => {
            code => 32,
            string => iget('Неверный PositionType')
        },
    'BadStatGoalsFilter' => {
            code => 33,
            string => iget('Неверное условие фильтрации по целям')
        },
    'BadEmail' => {
            code => 34,
            string => iget('Неверный формат e-mail')
        },
    'BadPhone' => {
            code => 35,
            string => iget('Неверный формат номера телефона')
        },
    'BadTimestamp' => {
            code => 36,
            string => iget('Неверный формат временной метки')
        },
    'BadStrategy' => {
            code => 37,
            string => iget('Параметры стратегии указаны неверно')
        },
    'NotificationIdNotFound' => {
            code => 38,
            string => iget('Подписка на нотификации с заданными параметрами не найдена')
        },
    'BadAdID' => {
            code => 39,
            string => iget('Неверный идентификатор объявления')
    },
    'BadMediaplanAdID' => {
            code => 40,
            string => iget('Неверный идентификатор объявления медиаплана')
    },
    'BadMediaplanKeywordID' => {
            code => 41,
            string => iget('Неверный идентификатор ключевого слова медиаплана')
    },
    'BadMediaplanCategoryID' => {
            code => 42,
            string => iget('Неверный идентификатор категории медиаплана')
    },
    'BadMediaplan' => {
            code => 43,
            string => iget('Некорректный медиаплан')
    },
    'BadMediaplanAd' => {
            code => 44,
            string => iget('Некорректное объявление медиаплана')
    },
    'BadAdID' => {
            code => 45,
            string => iget('Неверный идентификатор объявления')
    },
    'BadRetargetingID' => {
            code => 46,
            string => iget('Неверный идентификатор ретаргетинга')
    },
    'BadRetargetingConditionID' => {
            code => 47,
            string => iget('Неверный идентификатор условия ретаргетинга')
    },
    'BadFilter' => {
            code => 48,
            string => iget('Неверное условие фильтрации')
    },
    'BadAdGroupID' => {
            code => 49,
            string => iget("Неверный идентификатор группы объявлений")
    },
         
# --- common ---

    'AuthTmpUnavail' => {
        code => 52,
        string => iget('Аутентификация временно недоступна')
        },

    'UserInvalid' => { 
            code => 53,
            string => iget('Ошибка авторизации')
         },   
    'NoRights' => { 
            code => 54,
            string => iget('Недостаточно прав')
         },   
    'MethodNotExist' => { 
            code => 55,
            string => iget('Метод не существует')
         },
    'LimitExceed' => {
            code => 56,
            string => iget('Превышен лимит запросов')
        },
    'ConcurrentLimitExceed' => {
            code => 57,
            string => iget('Превышен лимит одновременных запросов метода')
        },
    'AppNotRegistered' => { 
            code => 58,
            string => iget('Нет доступа')
         },
    MethodDeprecated => {
            code => 59,
            string => iget('Метод yстарел')
        },

# --- forecast ---

    'BadParams' => { 
            code => 71,
            string => iget('Параметры запроса указаны неверно')
         },
    'BadForecastID' => { 
            code => 72,
            string => iget('Неверный ForecastID')
         },
    'NoForecast' => { 
            code => 73,
            string => iget('Указанный отчет с прогнозом бюджета не существует')
         },
    'ForecastIsPending' => { 
            code => 74,
            string => iget('Отчет с прогнозом в процессе подготовки')
         },
    'BadCategoryID' => { 
            code => 75,
            string => iget('Неверный CategoryID')
         },
    'BadPhrase' => { 
            code => 76,
            string => iget('Неверно указана фраза ')
         },
    'BadGeo' => { 
            code => 77,
            string => iget('Неверный GeoID')
         },
    'ReportTmpUnavailable' => { 
            code => 78,
            string => iget('Отчет временно недоступен')
         },
    'BadRubric' => {
            code => 79,
            string => iget('Неверно указана рубрика каталога')
        },
    'CannotDeleteObject' => {
            code => 80,
            string => iget('Не удалось удалить объект')
        },
    'BadArrayValues' => {
        code => 81,
        string => iget_noop('Значения массива некорректны')
    },
                                                                                                                                                                
#--- wordstat report ---

    'NoWordstatReport' => { 
            code => 91,
            string => iget('Указанный отчет wordstat не существует')
         },
    'WordstatReportIsPending' => { 
            code => 92,
            string => iget('Отчет wordstat в процессе подготовки')
         },
    'BadWordstatReportID' => { 
            code => 93,
            string => iget('Неверный WordstatReportID')
         },     
    'EmptyCampaignsStat' => {
            code => 94,
            string => iget('Статистика для указанных кампаний отсутствует')
    },

# --- campaigns ---     111-150

    'BadCampaign' => { 
            code => 111,
            string => iget('Неверно указаны параметры кампании')
         },

    'CampaignOnBalanceError' => {
            code => 112,
            string => iget('Невозможно создать кампанию в балансе')
        },

    'BadCampaignFilter' => {
            code => 113,
            string => iget('Неверное условие фильтрации кампании')
        },

    'TooManyCampaigns' => {
            code => 114,
            string => iget('Превышен лимит количества кампаний')
    },
    'StrategyNotAllowed' => {
            code => 115,
            string => iget('Подключение стратегии недоступно')
    },

# --- banners ---       151-190

    'BadBanner' => { 
            code => 151,
            string => iget('Неверно указаны параметры баннера')
         },
         
    'NotEnoughUnits' => { 
            code => 152,
            string => iget('Недостаточно баллов')
         },           
         
    'TooManyBanners' => { 
            code => 153,
            string => iget('Превышен лимит баннеров в запросе')
         },    

    'BadPointOnMap' => {
            code => 154,
            string => iget('Структура PointOnMap заполнена неверно')
        },
         
     'NotAllowedOp' => {
            code => 155,
            string => iget('Операция для рекламной кампании или объявления недоступна')
     },
     
     'ArchiveEdit' => {
            code => 156,
            string => iget('Не позволяется изменение архивных кампаний или объявлений')
     },

    'TooManyAdGroups' => { 
            code => 157,
            string => iget('Превышен лимит объявлений в кампании')
         },    

    'TooManyAdsInGroup' => { 
            code => 158,
            string => iget('Превышен лимит объявлений в группе')
    },    


# --- phrases ---       191-240
    'NotAllowPriceChange' => {
            code => 191,
            string => iget('Запрещено менять ContextPrice для активных на поиске фраз')
    },

    'BadMinusKeywords' => {
            code => 192,
            string => iget('Неверно указаны минус-слова')
        },
    'TooManyKeywords' => {
            code => 193,
            string => iget(
                'Превышено допустимое количество в %i ключевых слов',
                $API::Methods::Keyword::Limits::MAX_KEYWORDS_PER_QUERY
            )
    },
    'KeywordNotFound' => {
            code => 194,
            string => iget('Ключевое слово не найдено')
    },
    'ActiveKeywordOrRetargetingConditionRequired' => {
            code => 195,
            string => iget('Группа должна содержать как минимум одно активное ключевое слово или условие ретаргетинга')
    },

# --- prices ---        241-250

    'LongMass' => {
            code => 241,
            string => iget('Превышен допустимый размер массива')
    },

    'BadPrice' => { 
            code => 242,
            string => iget('Неверно указана цена')
     },
    
    'PhrasesNotFound' => {
            code => 244,
            string => iget('Не найдены фразы для обновления ставок')
    },

    'BadCurrency' => {
            code => 245,
            string => iget('Недопустимое значение валюты')
    },
# --- clients --- 

    'BadLogin' => { 
            code => 251,
            string => iget('Неверный логин пользователя')
    },

    'LoginOccupied' => {
            code => 252,
            string => iget('Логин занят')
    },

    'CantCreateLogin' => {
            code => 253,
            string => iget('Ошибка при создании логина')
    },

    'CantCreateClient' => {
            code => 254,
            string => iget('Ошибка при создании клиента')
    },
    'CantCreateRelation' => {
            code => 255,
            string => iget('Невозможно связать клиента с агентством')
    },

    'AgencyNotExist' => {
            code => 256,
            string => iget('Указанное агентство не существует')
    },
    
    'CantUpdateRights' => {
            code => 257,
            string => iget('Невозможно обновить права пользователя')
    },

    'CantCreateClientAssociation' => {
            code => 258,
            string => iget('Ошибка при создании клиента')
    },
    
    'BadClient' => { 
            code => 259,
            string => iget('Клиент не существует')
    },

# Images

    'BadImageHash' => {
            code => 271,
            string => iget('Изображение не найдено')
    },

    'BadImage' => {
            code => 272,
            string => iget('Неверное изображение')
    },
    'ImagePoolExceeded' => {
            code => 273,
            string => iget('Превышен размер пула изображений')
    },
    'ImageSaveError' => {
            code => 274,
            string => iget('Ошибка при сохранении файла')
    },

# Finance Operations

    'FinanceTokenInvalid' => {
            code => 350,
            string => iget('Неверный токен для финансовых операций')
    },
    'InvalidFinOperationNum' => {
            code => 351,
            string => iget('Неверный номер финансовой операции')
    },
    'FinOpsNotAvailable' => {
            code => 352,
            string => iget('Финансовые операции недоступны для вашего аккаунта')
    },
    'BadTransferMoneyRequest' => {
            code => 353,
            string => iget('Неверный запрос на перенос денег')
        },
    'BadPayCamp' => {
            code => 354,
            string => iget('Неверный запрос на выставление счета')
        },
    'CreditLimitExceeded' => {
            code => 355,
            string => iget('Превышен доступный кредитный лимит')
        },
    'BadContract' => {
            code => 356,
            string => iget('Неверный идентификатор договора')
        },
    'BadPayCampFromLimits' => {
            code => 357,
            string => iget('Неверный запрос на оплату кампаний из кредитных лимитов')
        },
    'FinOpsTmpUnavail' => {
            code => 358,
            string => iget('Финансовые операции временно недоступны')
        },
    'NoMasterToken' => {
            code => 359,
            string => iget('MasterToken для данного клиента не существует')
        },

    'PayTokenNotProvided' => {
            code => 360,
            string => iget('Не предоставлен платежный токен')
        },
    'InvalidPaymentToken' => {
            code => 361,
            string => iget('Неверный платежный токен')
        },
    'ExpiredPaymentToken' => {
            code => 362,
            string => iget('Требуется обновить платежный токен')
        },
    'OverdraftNotAvailable' => {
            code => 363,
            string => iget('Овердрафт недоступен')
        },
    'OverdraftLimitExceeded' => {
            code => 364,
            string => iget('Превышена сумма доступного овердрафта')
        },
    'PaymentProcError' => {
            code => 365,
            string => iget('Не удалось провести платеж')
        },
    'PaymentDifferentCampaignsType' => {
        code => 366,
        string => iget('В одном запросе не допускается оплата кампаний, имеющих разный тип')
    },
    'BadOAuthPayToken' => {
            code => 367,
            string => iget('Неверный платежный токен')
    },
    'BadPayMethodID' => {
            code => 368,
            string => iget('Неверный платежный метод')
    },
    'PaymentTransactionAlreadyExists' => {
            code => 369,
            string => iget('Транзакция уже существует')
    },
    'NoSuchPaymentTransaction' => {
            code => 370,
            string => iget('Транзакция не существует')
    },
    'PaymentCurrencyForbiddenForClient' => {
            code => 371,
            string => iget('Оплата в переданной валюте недоступна для клиента')
    },
    'AnothersPaymentToken' => {
            code => 372,
            string => iget('Логин в Яндекс.Деньгах не совпадает с логином в Директе')
    },

# ---  ---
    '500' => { 
            code => 500,
            string => iget('Внутренняя ошибка сервера')
    },
    'BadRequest' => { 
            code => 501,
            string => iget('Неверный запрос')
    },

    '503' => { 
            code => 503,
            string => iget('Сервис временно недоступен')
    },

    'TmpCmdUnavailable' => { 
            code => 504,
            string => iget('Плановый перерыв в работе сервера. Повторите запрос позже')
    },

    '506' => { 
            code => 506,
            string => iget('Превышен лимит одновременных запросов')
    },

    'NotSupportedVersion' => {
            code => 508,
            string => iget("Запрашиваемая версия API недоступна")
    },

    'MethodNotAvailable' => {
            code => 509,
            string => iget("Метод недоступен в данной версии API")
    },
    
    # вопрос отдавать такую ошибку, либо прятать за NoRights
    'AccessDenied' => {
            code => 510,
            string => iget("Доступ запрещен")
    },

    'UnknownLocale' => {
            code => 511,
            string => iget("Неизвестный язык")
    },
    'BadRequestMethod' => {
            code => 512,
            string => iget('Неверный метод запроса')
        },
    'UnknownUser' => {
            code => 513,
            string => iget('Ваш логин не подключен к Яндекс.Директу'),
        },
    'InternalLogicError' => { # ошибка, не устраняющаяся со временем самостоятельно
            code => 514,
            string => iget('Внутренняя ошибка сервера. Пожалуйста, обратитесь в службу поддержки.')
        },
    'NoWallet' => {
            code => 515,
            string => iget('Требуется подключить общий счёт.')
        },
    'AccountNotUpdated' => {
            code => 517,
            string => iget('Невозможно обновить информацию по данному AccountID.')
        },
    'AgencyNotValidCurrency' => {
            code => 518,
            string => iget('Валюта недоступна агентству.')        
        },
    'WalletExists' => {
            code => 519,
            string => iget('Общий счёт уже подключен.')
        },
    'CantEnableWalletNoCampaigns' => {
            code => 520,
            string => iget('Для подключения общего счёта необходимо создать хотя бы одну кампанию.')
        },
    'AccountIsDisabledByAgency' => {
            code => 521,
            string => iget('Работа с общим счётом запрещена. Агентству необходимо включить возможность подключения общего счёта у клиентов.')
        },
    'NotSupported' => {
            code => 3_500, # код из API5
            string => iget('Не поддерживается.')
        },
    'LimitAccess' => {
            code => 3_600, # код из API5
            string => iget('Ограниченная функциональность')
        },

#     '' => { 
#             code => 0,
#             string => ''
#     },
);

=head2 dieSOAP(code, detail, OPT)

    Исполняет ошибку в АПИ.
    По-умолчанию выполняется die с текстом-структурой SOAP::Fault, содержащей поля - faultcode, faultstring, detail

    С параметром OPT{dont_die} = 1 - возвращает структуру в ответ

=cut

sub dieSOAP
{
    my ($errorTextCode, $errorDetail, %OPT) = @_;

    return if !defined $errorTextCode;

    $errorDetail ||= '';
    $errorDetail = $OPT{DETAIL_PREFIX} . $errorDetail if $OPT{DETAIL_PREFIX};

    if (! defined $ERRORS{$errorTextCode}) {
        print STDERR "No exists error in API::Errors ! - $errorTextCode / $errorDetail";

        $errorTextCode = '500';
    }
    #print STDERR Carp::longmess("dieSOAP beta $ERRORS{ $errorTextCode }->{code} $ERRORS{ $errorTextCode }->{string}") if EnvTools::is_beta();
    print STDERR Carp::longmess("dieSOAP 500") if $errorTextCode eq '500';
    print STDERR Carp::longmess("dieSOAP 514") if $errorTextCode eq 'InternalLogicError';

    my $errobj = SOAP::Fault   -> faultcode( $ERRORS{ $errorTextCode }->{code} || 0 )
                      -> faultstring( iget($ERRORS{ $errorTextCode }->{string}) || $errorTextCode )
                      -> faultdetail($errorDetail || '');

    if ($OPT{dont_die}) {
        return $errobj;
    } else {
        die $errobj;
    }
}

=head2 dieSOAP_by_object(error_object)

    Исполняет ошибку в АПИ по объекту из get_error_object

=cut

sub dieSOAP_by_object($)
{
    my $error_object = shift;
    return if !defined $error_object;
    my $errorCode = $error_object->{FaultCode};

    my %errors_by_code = reverse map {$_ => $ERRORS{$_}->{code}} keys %ERRORS;

    my $errobj = SOAP::Fault -> faultcode($errorCode)
        -> faultstring($error_object->{FaultString})
        -> faultdetail($error_object->{FaultDetail});

    if (! ($errorCode && defined $errors_by_code{$errorCode})) {
        print STDERR "Error code not found in API::Errors! — $errorCode / $error_object->{FaultDetail}";

        $errorCode = '500';
        $errobj = SOAP::Fault -> faultcode($errorCode)
        -> faultstring(iget($ERRORS{500}->{string}))
        -> faultdetail('');
    }
    print STDERR Carp::longmess("dieSOAP 500") if $errorCode eq '500';
    print STDERR Carp::longmess("dieSOAP 514") if $errorCode eq '514';

    die $errobj;
}

sub get_error_object
{
    my ($errorTextCode, $errorDetail) = @_;

    return {} if !defined $errorTextCode;

    warn "unknown errorTextCode $errorTextCode" unless exists $ERRORS{$errorTextCode};

    return {
            FaultCode => $ERRORS{ $errorTextCode }->{code} || 0,
            FaultString => iget($ERRORS{ $errorTextCode }->{string}) || $errorTextCode,
            FaultDetail => $errorDetail || '',
           };
}

sub get_warning_object
{
    my ($warningTextCode, $warningObject) = @_;
    return {} if !defined $warningTextCode;

    return {
            WarningCode => $WARNINGS{ $warningTextCode }->{code} || 0,
            WarningString => iget($WARNINGS{ $warningTextCode }->{string}) || $warningTextCode,
            Description => to_json($warningObject || {}, { canonical => 1 }),
           };
}

1;
