package API::Filter;

# $Id$

=head1 NAME

API::Filter - Модуль для преобразования форматов выходных данных из внутреннего формата представления данных

=cut

use strict;
use warnings;

use List::MoreUtils qw/uniq any/;

use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::Trace qw/current_trace/;
use Yandex::Clone qw/yclone/;

use Settings;
use Campaign qw/CalcCampStatus/;
use Client qw/get_client_data/;
use Currencies;
use Currency::Rate;
use Phrase;
use TextTools qw/smartstrip round2s/;
use PlacePrice qw//;

use API::Settings;

use Property;

use vars qw($VERSION @ISA @EXPORT);

use base qw(Exporter);
@EXPORT = qw(
    filter_stat_goals
    filter_retargeting_object
    filter_adimage_assoc_object
    filter_adimage_pool_object
    filter_adimage_queue_object
    filter_adimage_limits_object
    filter_account_management_get_object
    filter_banner_object
    filter_phrase_object
    filter_forecast_report
    filter_reports_list
);

use utf8;

our %VCARD_FIELDS = (
    country => 'Country',
    city => 'City',
    name => 'CompanyName',
    street => 'Street',
    house => 'House',
    build => 'Build',
    apart => 'Apart',
    metro => 'MetroStationID',
    contactperson => 'ContactPerson',
    worktime => 'WorkTime',
    im_client => 'IMClient',
    im_login => 'IMLogin',
    extra_message => 'ExtraMessage',
    contact_email => 'ContactEmail'
);


# список полей баннера, используется при формировании объекта и фильтрации,
# а так же при валидации поля FieldsNames ##Type
our %BANNERS_FIELDS = map {$_ => 1} qw/
    CampaignID BannerID Title Type Href Domain Text Geo AdImageHash
    StatusPhoneModerate StatusBannerModerate StatusPhrasesModerate StatusSitelinksModerate StatusAdImageModerate
    StatusActivating StatusShow IsActive StatusArchive
    Sitelinks ContactInfo MinusKeywords
    AdWarnings FixedOnModeration ModerateRejectionReasons
    AgeLabel
    AdGroupID AdGroupName AdGroupMobileBidAdjustment
/;

our %MEDIAPLAN_FIELDS = map {$_ => 1} qw/
    CampaignID MediaplanAdID SourceAdID
    Title Title2 Href Domain Text Geo Type
    Sitelinks ContactInfo MinusKeywords
/;

our %MEDIAPLAN_KEYWORD_FIELDS = map {$_ => 1} qw/
    CampaignID MediaplanAdID MediaplanKeywordID
    Phrase LowCTR Shows Position
    Min Max PremiumMin PremiumMax
    MinCTR MaxCTR PremiumMinCTR PremiumMaxCTR
    Currency
/;

our %MEDIAPLAN_POSITIONS = (
    10 => 'Premium',
    11 => 'Premium',
    12 => 'Premium',
    13 => 'Premium',
    20 => 'FirstPlaceGuarantee',
    21 => 'Guarantee',
    22 => 'Guarantee',
    23 => 'Guarantee',
);

my %conversion_ext = (
    campaignID => 'CampaignID',
    endDate => 'EndDate',
    startDate => 'StartDate',
    'stat' => 'Stat'
);
my %conversion_int = (
    bid => 'BannerID',
    clicks => 'Clicks',
    clicks_0 => 'ClicksSearch',
    clicks_1 => 'ClicksContext',
    shows => 'Shows',
    shows_0 => 'ShowsSearch',
    shows_1 => 'ShowsContext',
    stat_date => 'StatDate',
    phrase => 'Phrase',
    sum => 'Sum',
    sum_0 => 'SumSearch',
    sum_1 => 'SumContext',
    category_id => 'RubricID',
    bids_id => 'PhraseID',
    ret_id => 'RetargetingID',
    dyn_cond_id => 'WebpageID',
    perf_id => 'DynamicMediaAdFilterID',
    banner_image_type => 'StatType',
    fp_clicks_avg_pos => 'ClicksAveragePosition',
    fp_shows_avg_pos => 'ShowsAveragePosition',
    device_type => 'DeviceType'
);

my %STATUS_REPORT_NAME = (
    'forecast' => {id => "ForecastID", status => "StatusForecast"},
    'report'   => {id => "ReportID",   status => "StatusReport"},
    'wordstat' => {id => "ReportID",   status => "StatusReport"},
);

=head2 filter_stat_goals($self, $campaigns_goals_list, $accessible_goal_ids_by_cids, %options)

    Общая часть сбора статистики метода GetStatGoals для одной или нескольких кампаний

=cut

sub filter_stat_goals($$$;%) {
    my ($self, $campaigns_goals_list, $accessible_goal_ids_by_cids, %O) = @_;
    my @result;
    foreach my $cid (keys %{$campaigns_goals_list})
    {
        foreach (values %{$campaigns_goals_list->{$cid}}) {
            next if ($accessible_goal_ids_by_cids->{$cid} && !$accessible_goal_ids_by_cids->{$cid}{$_->{goal_id}}); # пропускаем недоступные цели, если указан список доступных
            my $tmp = {
                            GoalID => $_->{goal_id},
                            Name => $_->{goal_name},
                      };
            if ($self->{api_version_full} > 4) {
                $tmp->{AvailableForStrategy} = $_->{goal_status} eq 'Deleted' ? 'No' : 'Yes';
                $tmp->{AvailableForContextStrategy} = $_->{goal_status} eq 'Deleted' ? 'No' : 'Yes';
                $tmp->{GoalsReached} = $_->{goals_count} || 0;
                $tmp->{ContextGoalsReached} = $_->{context_goals_count} || 0;
                $tmp->{CampaignID} = $cid if ($O{include_campaignIDS});
            }

            push @result, $tmp;
        }
    }
    return \@result;
}

sub filter_retargeting_object {
    my ($object, $pid2bid, %O) = @_;
    my $result_object = {
        AdGroupID => $object->{pid},
        AdID => $pid2bid->{$object->{pid}},
        RetargetingConditionID => $object->{ret_cond_id},
        RetargetingID => $object->{ret_id},
        StatusPaused => $object->{is_suspended} ? 'Yes' : 'No',
        AutoBudgetPriority => $object->{autobudgetPriority} ? $Phrase::PRIORITY_REVERSE_VALUES{$object->{autobudgetPriority}} : undef,
    };
    if ($object->{price_context}) {
        $result_object->{ContextPrice} = (($object->{currency} || 'YND_FIXED') eq ($O{export_currency} || 'YND_FIXED'))? $object->{price_context}
            : currency_price_rounding(
                    convert_currency($object->{price_context}, $object->{currency}, $O{export_currency})
                    , $O{export_currency}
                    , up => 1);
    } else {
        $result_object->{ContextPrice} = $object->{price_context};
    }

    $result_object->{Currency} = $O{export_currency} if $O{export_currency} ne 'YND_FIXED';

    return $result_object;
}

sub filter_adimage_assoc_object {
    my ($object) = @_;
    my $result = {
        AdID => $object->{bid},
        CampaignID => $object->{cid},
        AdImageHash => $object->{image_hash},
        StatusAdImageModerate => status_moderate_int_to_ext([$object->{statusModerate}])->[0],
        Login => $object->{login},
    };

    if ($object->{ModerateRejectionReasons}) {
        $result->{ModerateRejectionReasons} = $object->{ModerateRejectionReasons};
    }
    return $result;
}

sub filter_adimage_pool_object {
    my ($object) = @_;
    my $result =  {
        Login => $object->{login},
        AdImageURL => $object->{url},
        AdImageHash => $object->{image_hash},
        Name => $object->{name},
        Assigned => $object->{assigned} ? 'Yes' : 'No',
    };

    return $result;
}

sub filter_adimage_queue_object {
    my ($object) = @_;
    my $result =  {
        Login => $object->{login},
        AdImageURL => $object->{url},
        AdImageHash => $object->{image_hash},
        Name => $object->{name},
        AdImageUploadTaskID => $object->{job_id},
        SourceURL => $object->{source_url},
    };

    if ($object->{status} =~ /^(Grabbed|New)$/) {
        $result->{Status} = 'Pending';
    } elsif ($object->{status} eq 'Finished') {
        $result->{Status} = 'Done';
    } else {
        $result->{Status} = 'Error';
        $result->{Error} = $object->{error_obj};
    }

    return $result;
}

sub filter_adimage_limits_object
{
    my ($object) = @_;

    return {
        Login => $object->{login},
        Utilized => $object->{cnt},
        Capacity => $object->{total},
    };
}

sub filter_account_management_get_object
{
    my ($object) = @_;

    my $result = [];
    
    # -- просмотрено
    my $ansichten = {};
    foreach my $wo (@{ $object->{WalletsOptions}}) {

        # -- фильтрация повторных кошельков
        next if exists $ansichten->{$wo->{AccountID}};
        $ansichten->{$wo->{AccountID}} = 1;

        my $row = {
            Login       => $object->{Login},
            Discount    => $object->{Discount},
            exists $wo->{AgencyName} ? (AgencyName => $wo->{AgencyName}) : (AgencyName => '')
        };

        if ($wo->{sms_time} =~ /^(\d+\:\d+)\:(\d+\:\d+)$/) {
            $row->{SmsNotification}->{SmsTimeFrom} = $1;
            $row->{SmsNotification}->{SmsTimeTo} = $2;
        }

        my $sms_param_names = {
            active_orders_money_out_sms     => 'MoneyOutSms',
            notify_order_money_in_sms       => 'MoneyInSms',
            paused_by_day_budget_sms        => 'PausedByDayBudgetSms',
        };

        foreach my $flag_name (keys %$sms_param_names) {
            my $external_name = $sms_param_names->{$flag_name};
            $row->{SmsNotification}->{$external_name} = $wo->{sms_flags}{$flag_name} ? 'Yes' : 'No';
        }

        $row->{EmailNotification} = {
            Email             => $wo->{email},
            SendWarn          => $object->{sendWarn},
            MoneyWarningValue => $wo->{money_warning_value},
            PausedByDayBudget => $wo->{email_notifications}{paused_by_day_budget} ? 'Yes' : 'No',
        };

        if ($wo->{day_budget}{sum} > 0) {
            $row->{AccountDayBudget} = {
                Amount    => $wo->{day_budget}{sum},
                SpendMode => ucfirst($wo->{day_budget}{show_mode}),
            };
        }

        @{$row}{qw/AmountAvailableForTransfer Amount AccountID /} = @{$wo}{qw/SumAvailableForTransfer Amount AccountID/};
        $row->{Currency} = $wo->{Currency} if $wo->{Currency} && $wo->{Currency} ne 'YND_FIXED';

        if ( $row->{AmountAvailableForTransfer} < 0) {
            warn 'API::Filter::filter_account_management_get_object [reqid:' . Yandex::Trace::trace_id()
                    . qq~] AmountAvailableForTransfer less than 0! DIRECT-26163 Login = '$row->{Login}', sum = '$row->{AmountAvailableForTransfer}'~;
            $row->{AmountAvailableForTransfer} = undef;
        }

        push @{$result}, $row;
    }

    return $result;
}

=head2 filter_banner_object

    Возвращает унифицированый объект баннера/медиаплана
    Формирует общие структуры. Отличающиеся поля должны быть добавлены в объект раньше.

    options:
        mediaplan => 1|0 - является ли переданный объект медиапланом

=cut

{
    # кеширование получаемых из базы точек
    my %_map_cache;
sub filter_banner_object
{

    my ( $self, $banner, %O ) = @_;
    my $new = {};

    my %map = (Title => 'title',
               Title2 => 'title_extension',
               Text => 'body',
               Href => 'href',
               Domain => 'domain',
               CampaignID => 'cid',
               Geo => 'geo',
               AdImageHash => 'image',
               AdGroupID => 'pid',
               AdGroupName => 'group_name',
               AdGroupMobileBidAdjustment => 'mobile_multiplier_pct'
    );
    $banner->{Sitelinks} = [];

    while ((my $key, my $value) = each %map) {
        $banner->{$key} = $banner->{$value}
    }

    if ($banner->{sitelinks_set_id}) {
        foreach (@{$banner->{sitelinks}}) {
            push @{$banner->{Sitelinks}}, {Title => $_->{title}, Href => $_->{href}};
        }
    }
    $banner->{Type} = ucfirst($banner->{banner_type});

    # преобразуем время работы в csv формат
    $banner->{worktime} =~ s/\#/\;/g if defined $banner->{worktime};
    if ($self->{api_version_full} > 4.5) {
        $banner->{worktime} = join ";", map {$_ + 0} split ";", $banner->{worktime};
    }

    my %filter_by = %BANNERS_FIELDS;
    if ($O{mediaplan}) {
        %filter_by = %MEDIAPLAN_FIELDS;
    }

    foreach (keys %filter_by) {
        $new->{$_} = defined $banner->{$_} ? $banner->{$_} : undef;
    }

    $new->{MinusKeywords} = $banner->{banner_minus_words};

    if (! scalar @{$banner->{Sitelinks}}) {
        delete $new->{StatusSitelinksModerate};
    }

    # собираем контактную информацию
    if ($banner->{phone}) {
        foreach my $field (keys %VCARD_FIELDS) {
            $new->{ContactInfo}->{$VCARD_FIELDS{$field}} = defined $banner->{$field} ? $banner->{$field} : undef;
        }
        delete $new->{ContactInfo}->{MetroStationID} if $self->{api_version_full} < 5;

        $new->{ContactInfo}->{OGRN} = $banner->{ogrn};

        my $vphone = VCards::parse_phone($banner->{phone});

        $new->{ContactInfo}->{IMLogin} = '' unless defined $new->{ContactInfo}->{IMLogin};

        @{$new->{ContactInfo}}{qw/Phone CountryCode CityCode PhoneExt/} = @{$vphone}{qw/phone country_code city_code ext/};

        if ($banner->{map_id}
                && ( ($banner->{map_id_auto}||'') ne $banner->{map_id} || $banner->{precision} && $banner->{precision} =~ /^(exact|street|number)$/i )
        ) {
            if (!exists $_map_cache{$banner->{map_id}}) {
                $_map_cache{$banner->{map_id}} = get_one_line_sql(PPC(cid => $banner->{cid}), "select x, y, x1, y1, x2, y2 from maps where mid = ?", $banner->{map_id});
            }
            # DIRECT-39685, в случае если address.map_id задан но в maps по этому mid
            # ничего нет делаем вид что address.map_id есть NULL, т.е. незаполняем PointOnMap
            $new->{ContactInfo}->{PointOnMap} = yclone($_map_cache{$banner->{map_id}})
                if defined $_map_cache{$banner->{map_id}};
        }

    } else {
        delete $new->{StatusPhoneModerate};
        delete $new->{ContactInfo};
    }
    if ($self->{api_version_full} < 4.5 || ! $new->{AgeLabel}) {
        delete $new->{AgeLabel};
    }
    if ($self->{api_version_full} < 4.5) {
        delete $new->{AdImageHash};
        delete $new->{AdGroupID};
        delete $new->{AdGroupName};
        delete $new->{StatusAdImageModerate};
        delete $new->{Type} if exists $new->{Type};
        delete $new->{AdGroupMobileBidAdjustment} if exists $new->{AdGroupMobileBidAdjustment};
        delete $new->{StatusAdImageModerate};
    }
    return $new;
}
}

=head2 _convert_and_round_up(price, old_currency, export_currency)

    Переводит цену и округляет вверх до шага торгов

=cut

sub _convert_and_round_up($$$)
{
    my ($price, $old, $new) = @_;
    $price = convert_currency($price, $old, $new);
    $price = currency_price_rounding($price, $new, up => 1);
    return $price;
}

=head2 filter_phrase_object

    Возвращает объект PhraseObject:

        PhraseID => id
        CampaignID => cid
        BannerID => bid
        Phrase => фраза
        IsRubric => Yes/No
        Price => выставленная ставка
        Clicks =>123
        Shows => 1234
        Priority => High/Medium/Low
        AutoBroker => Yes

    TODO: плохо, что часть данных обсчитывается здесь

=cut

sub filter_phrase_object($;$%)
{
    my ( $vars, $params, %O ) = @_;

    my $profile = Yandex::Trace::new_profile('api:filter', tags => 'filter_phrase_object');

    if ( defined $vars && ref $vars eq 'HASH' ) {

        die 'no currency given' unless $vars->{currency};

        # multicurrency: что делать в 5-ой версии?
        my $export_currency = $params->{api_version_full} > 4 ? $params->{ExportCurrency} || 'YND_FIXED' : undef;
        my $need_currency_conversion = ($vars->{currency} ne ($export_currency || 'YND_FIXED'))? 1 : 0;

        my $phrase = { Currency => $export_currency || 'YND_FIXED' };
        if ( ! defined $vars->{is_forecast} ) {
            my $price = $vars->{price} ? convert_currency($vars->{price}, $vars->{currency}, $export_currency || 'YND_FIXED') : 0;
            $phrase->{PhraseID} = $vars->{id};
            $phrase->{CampaignID} = $vars->{CampaignID};
            $phrase->{BannerID} = $vars->{BannerID};
            $phrase->{Price} = round2s($price);
            $phrase->{AutoBroker} = 'Yes';
            
            $phrase->{ContentRestricted} = $vars->{disabled_tragic} ? 'Yes' : 'No';
            if ($params->{api_version_full} > 4)
            {
                $phrase->{StatusPaused} = $vars->{is_suspended} ? 'Yes' : 'No';
                $phrase->{ContextClicks} = $vars->{ctx_clicks} || 0;
                $phrase->{ContextShows} = $vars->{ctx_shows} || 0;
                $phrase->{AdGroupID} = $vars->{AdGroupID};
            }
            $phrase->{StatusPhraseModerate} = $vars->{statusModerate} || 'New';
        }

        $phrase->{Phrase} = $vars->{phrase};
        $phrase->{IsRubric} = 'No';
        $phrase->{Clicks} = $vars->{clicks} || 0;
        $phrase->{PremiumClicks} = $vars->{premium_clicks} || 0;
        $phrase->{FirstPlaceClicks} = $vars->{first_place_clicks} || 0;
        $phrase->{Shows} = $vars->{shows} || 0;

        if (scalar grep {defined $vars->{lc($_)} } @Models::Phrase::BIDS_HREF_PARAMS ) {
            foreach my $param_name (@Models::Phrase::BIDS_HREF_PARAMS) {
                next unless defined $vars->{lc($param_name)};
                $phrase->{UserParams}{$param_name} = $vars->{lc($param_name)};
            }
        }

        $phrase->{ContextLowCTR} = $vars->{context_stop_flag} ? 'Yes': 'No';

        if (!defined $vars->{autobudgetPriority}) {
            # -- Для того что бы корректно отдавать NULL поля, подразумевается значение по умолчанию
            $phrase->{ AutoBudgetPriority } = $Phrase::PRIORITY_REVERSE_VALUES{3};
        }
        else {
            $phrase->{ AutoBudgetPriority } = $vars->{autobudgetPriority} ? $Phrase::PRIORITY_REVERSE_VALUES{$vars->{autobudgetPriority}} : '';
        }

        if ($params->{strategy} && $params->{strategy} eq 'different_places') {

            if (! $vars->{context_stop_flag}) {
                $phrase->{ContextCoverage} = [];

                if ($vars->{price_for_coverage} && %{$vars->{price_for_coverage}}
                    && $vars->{pokazometer_data}
                    && $vars->{pokazometer_data}->{shows_cnt} > 0
                ) {

                    foreach my $proc (sort {$b <=> $a} keys %{$vars->{price_for_coverage}}) {
                        my $price = $vars->{price_for_coverage}->{$proc};
                        if ($need_currency_conversion) {
                            $price = currency_price_rounding(convert_currency($price, $vars->{currency}, $export_currency || 'YND_FIXED'),
                                $export_currency || 'YND_FIXED', up => 1);
                        }
                        push @{$phrase->{ContextCoverage}}, {Price => $price, Probability => $proc};
                    }
                }
            }
        } else {
            $phrase->{ContextCoverage} = [];
        }

        # Возвращаем undef, если приходит нулевая цена
        $vars->{price_context} = undef if ! $vars->{price_context} || ! ($vars->{price_context} + 0);

        my $price_context = 0;
        if ($vars->{price_context}) {
            unless ($need_currency_conversion) {
                $price_context = $vars->{price_context};
            } else {
                $price_context = convert_currency($vars->{price_context}, $vars->{currency}, $export_currency || 'YND_FIXED');
            }
            $price_context = round2s($price_context);
        }

        my $premium = $vars->{premium};
        my $guarantee = $vars->{guarantee};

        # DIRECT-67242
        if (defined $premium && @$premium < 4) { # no data for P14
            push @$premium, { %{ $premium->[ $#$premium ] } }; # make a copy of last element in @premium
        }

        if (defined $premium && defined $guarantee) { # если ходили в торги
            my $i = 1;
            foreach my $position_bids (@$premium) {
                my $bids = {
                    Position => ('P1'.$i),
                    Bid => $position_bids->{bid_price} / 1e6,
                    Price => $position_bids->{amnesty_price} / 1e6,
                };
                push @{$phrase->{AuctionBids}}, $bids;
                $i++;
            }

            $i = 1;
            foreach my $position_bids (@$guarantee) {
                my $bids = {
                    Position => ('P2'.$i),
                    Bid => $position_bids->{bid_price} / 1e6,
                    Price => $position_bids->{amnesty_price} / 1e6,
                };
                push @{$phrase->{AuctionBids}}, $bids;
                $i++;
            }

            if ($O{mediaplan}) { # некрасиво, но через год уже будет неважно
                $phrase->{Min} = $guarantee->[ $#$guarantee ]->{amnesty_price} / 1e6;
                $phrase->{Max} = $guarantee->[0]->{amnesty_price} / 1e6;
                $phrase->{PremiumMin} = $premium->[ $#$premium ]->{amnesty_price} / 1e6;
                $phrase->{PremiumMax} = $premium->[0]->{amnesty_price} / 1e6;
            } else {
                $phrase->{Min} = $guarantee->[ $#$guarantee ]->{bid_price} / 1e6;
                $phrase->{Max} = $guarantee->[0]->{bid_price} / 1e6;
                $phrase->{PremiumMin} = $premium->[ $#$premium ]->{bid_price} / 1e6;
                $phrase->{PremiumMax} = $premium->[0]->{bid_price} / 1e6;
            }
        }

        $phrase->{LowCTRWarning} = 'No';
        $phrase->{LowCTR} = 'No';
        $phrase->{ContextPrice} = undef;

        if ($params->{strategy} && $params->{strategy} eq 'different_places') {
            $phrase->{ContextPrice} = $price_context;
        }

        $phrase->{CTR} = round2s($vars->{ctr}) if defined $vars->{ctr};

        if (! defined $vars->{is_forecast} ) {

            my ($coverage, $larr_prices) = split('\|', $vars->{larr} || '');

            if (exists $vars->{guarantee} && defined $vars->{guarantee}) { # без торгов тут особо нечего делать
                my @arr_prices = map {$_->{bid_price}} @{$vars->{guarantee}}, @{$vars->{premium}};

                $phrase->{Prices} = [
                    map { round2s($_) }
                    map { $need_currency_conversion? _convert_and_round_up($_/1e6, $vars->{currency}, $export_currency || 'YND_FIXED') : $_/1e6 }
                    sort {$b <=> $a} 
                    uniq grep {$_} @arr_prices
                ];
            }

            my @processed_coverages = ();
            my @coverages_pairs = split(',', $coverage || '');

            foreach my $i (@coverages_pairs) {
                my ($price, $prob) = split ':', $i;

                next unless $price && $prob;

                $price /= 1e6;
                if ($need_currency_conversion) {
                    $price = _convert_and_round_up($price, $vars->{currency}, $export_currency || 'YND_FIXED');
                }

                if ($params->{api_version_full} > 4.5) { # latest - not! добавляем только функциональность, а не минорные исправления!
                    push @processed_coverages, { Price => round2s($price), Probability => ($prob/1e4) };
                } else {
                    push @processed_coverages, { Price => round2s($price), Probability => ($prob/1e6) };
                }
            };

            $phrase->{Coverage} = \@processed_coverages;

            $phrase->{CurrentOnSearch} = $vars->{broker} && $vars->{broker} > 0 ?  $vars->{broker} : undef;
            $phrase->{MinPrice} = $vars->{min_price} || get_currency_constant($vars->{currency}, 'MIN_PRICE');
        } else {
            $phrase->{PremiumCTR} = round2s($vars->{p_ctr}) if defined $vars->{p_ctr};
            $phrase->{FirstPlaceCTR} = round2s($vars->{first_place_ctr}) if defined $vars->{first_place_ctr};
        }

        foreach my $p (qw/Min Max PremiumMin PremiumMax MinPrice CurrentOnSearch/) {
            next unless $phrase->{$p};

            if ($vars->{currency} eq ($export_currency || 'YND_FIXED')) {
                $phrase->{$p} = round2s($phrase->{$p});
                next;
            }

            if ($need_currency_conversion) {
                $phrase->{$p} = convert_currency($phrase->{$p}, $vars->{currency}, $export_currency || 'YND_FIXED');
                $phrase->{$p} = currency_price_rounding($phrase->{$p}, $export_currency || 'YND_FIXED', up => 1);
            } else {
                $phrase->{$p} = round2s($phrase->{$p});
            }
        }

        if (exists $phrase->{AuctionBids} && defined $phrase->{AuctionBids}) {
            foreach my $auction_bid (@{$phrase->{AuctionBids}}) {
                foreach my $bid_type (qw/Bid Price/) {
                    if ($need_currency_conversion) {
                        $auction_bid->{$bid_type} = convert_currency($auction_bid->{$bid_type}, $vars->{currency}, $export_currency || 'YND_FIXED');
                        $auction_bid->{$bid_type} = currency_price_rounding($auction_bid->{$bid_type}, $export_currency || 'YND_FIXED', up => 1);
                    } else {
                        $auction_bid->{$bid_type} = round2s($auction_bid->{$bid_type});
                    }
                }
            }
        }

        # multicurrency: удаляем обозначение фишек
        if ($phrase->{Currency} && $phrase->{Currency} eq 'YND_FIXED') {
            delete $phrase->{Currency};
        }

        if ($O{mediaplan}) {
            $phrase->{MinCTR} = round2s($guarantee->[$#$guarantee]{ctr}) if defined $guarantee->[$#$guarantee]{ctr};
            $phrase->{MaxCTR} = round2s($guarantee->[0]{ctr}) if defined $guarantee->[0]{ctr};
            $phrase->{PremiumMinCTR} = round2s($premium->[$#$premium]{ctr}) if defined $premium->[$#$premium]{ctr};
            $phrase->{PremiumMaxCTR} = round2s($premium->[0]{ctr}) if defined $premium->[0]{ctr};
            $phrase->{MediaplanAdID} = $vars->{mbid};
            $phrase->{MediaplanKeywordID} = $vars->{id};
            $phrase->{Position} = $vars->{place} ? place_to_mediaplan_position($vars->{place}) : undef; # 0 значит не задано
            return hash_cut $phrase, keys %MEDIAPLAN_KEYWORD_FIELDS;
        } else {
            return $phrase;
        }
    }

    return;
}

=head2 place_to_mediaplan_position

    Маппим позицию из торгов в ее текстовое название в Медиапланах.
    Позиция в торгах числом вида NM где N - номер блока показа (1 - премиум, 2
    - гарания), M - номер позиции в блоке
    Если у конкретной позиции есть имя, то отдаем его, если нет, целочисленно
    делим на 10 и отдаем номер блока

=cut

sub place_to_mediaplan_position {
    my $place = shift;
    $place = PlacePrice::set_new_place_style($place) if $place < 10; # Старый, до VCG-шный формат нумерации блоков
    my $position = $MEDIAPLAN_POSITIONS{ $place } or die "unknown place $place";
    return $position
}

=head2 filter_forecast_report

    Фильтруем созданный отчет

=cut

sub filter_forecast_report
{
    my ($self, $params, %O) = @_;

    my $currency = $params->{currency};
    die 'no currency given' unless $currency;

    # фильтруем выходные данные
    my ( $filtered, $cat_filter, $phrase_filter );

    my @phrases_fields = _get_forecast_fields($self->{api_version_full}, map {$_ => $O{$_}} qw/advanced_forecast auction_bids/);

    my $is_get_data_from_bs = Property->new('forecast_is_get_data_from_bs')->get();

    my (%all_clicks, %all_sums, $all_shows);
    foreach my $ph ( @{ $params->{Phrases} } ) {
        $ph->{geo} = $params->{geo};
        $ph->{currency} = $currency;
        $ph->{is_forecast} = 1;

        my $phrase_obj;
        if ($is_get_data_from_bs) {
            $phrase_obj = filter_phrase_object( $ph, (
                hash_merge { ExportCurrency => $currency }, 
                    hash_cut $self, qw/api_version_full/) 
            );
        } else {
            $phrase_obj = filter_phrase_object_for_forecast($ph);
        }

        $phrase_obj = hash_cut($phrase_obj, @phrases_fields);

        push @$phrase_filter, $phrase_obj;
    }


    $filtered->{Common} = _prepare_common($params, $phrase_filter, $is_get_data_from_bs);

    $filtered->{Phrases} = $phrase_filter;

    return $filtered;
}
sub _get_forecast_fields {
    my ($api_version, %O) = @_;

    my @phrases_fields = qw/
        Phrase IsRubric Clicks FirstPlaceClicks PremiumClicks
        Shows Min Max PremiumMin PremiumMax CTR FirstPlaceCTR PremiumCTR
    /;

    if ($api_version > 4) {
        push @phrases_fields, qw/Currency/;
    }

    if ($O{advanced_forecast}) {
        push @phrases_fields, qw/ContextCoverage/;
    }

    if ($O{auction_bids}) { # не удалять структуру AuctionBids
        push @phrases_fields, qw/AuctionBids/;
    }
    return \@phrases_fields;
}

sub _prepare_common {
    my ($params, $phrase_filter, $is_get_data_from_bs) = @_;
    my (%all_clicks, %all_sums);

    foreach my $phrase ( @$phrase_filter ) {

        $all_clicks{premium} += $phrase->{PremiumClicks};
        $all_clicks{fp} += $phrase->{FirstPlaceClicks};
        $all_clicks{gar} += $phrase->{Clicks};

        $all_sums{premium} += $phrase->{PremiumClicks} * $phrase->{PremiumMin};
        $all_sums{fp} += $phrase->{FirstPlaceClicks} * $phrase->{Max};
        $all_sums{gar} += $phrase->{Clicks} * $phrase->{Min};
    }

    my $result = {
                Min => $all_sums{gar},
                Max => $all_sums{fp},
         PremiumMin => $all_sums{premium},
             Clicks => $all_clicks{gar},
      PremiumClicks => $all_clicks{premium},
   FirstPlaceClicks => $all_clicks{fp},
                Geo => $params->{geo},
    };

    $result->{Shows} = $is_get_data_from_bs ? _get_shows_for_bs($params) : _get_shows_for_advq($params);
    return $result;
}

sub _get_shows_for_advq {
    my ($params) = @_;
    my $all_shows = 0;
    $all_shows += $_->{shows} for @{ $params->{Phrases} };
    return $all_shows;
}

sub _get_shows_for_bs {
    my ($params) = @_;
    return $params->{shows};
}



=head2 filter_phrase_object_for_forecast

    Возвращает объект PhraseObject:

        PhraseID => id
        CampaignID => cid
        BannerID => bid
        Phrase => фраза
        IsRubric => Yes/No
        Price => выставленная ставка
        Clicks =>123
        Shows => 1234
        Priority => High/Medium/Low
        AutoBroker => Yes

    TODO: плохо, что часть данных обсчитывается здесь

=cut

sub filter_phrase_object_for_forecast
{
    my ($vars) = @_;

    my $profile = Yandex::Trace::new_profile('api:filter', tags => 'filter_phrase_object_for_forecast');

    if ( defined $vars && ref $vars eq 'HASH' ) {

        die 'no currency given' unless $vars->{currency};

        my $phrase = { Currency => $vars->{currency} || 'YND_FIXED' };

        $phrase->{Phrase} = $vars->{phrase};
        $phrase->{IsRubric} = 'No';
        $phrase->{Shows} = $vars->{shows} || 0;

        my $positions = $vars->{positions};

        my $premium_entry_pos = PlacePrice::get_premium_entry_place(forecast_style => 1);
        my $guarantee_entry_pos = PlacePrice::get_guarantee_entry_place(forecast_style => 1);
        $phrase->{Clicks} = $positions->{$guarantee_entry_pos}{clicks} || 0;
        $phrase->{PremiumClicks} = $positions->{$premium_entry_pos}{clicks} || 0;
        $phrase->{FirstPlaceClicks} = $positions->{first_place}{clicks} || 0;

        $phrase->{Min} = $positions->{$guarantee_entry_pos}{bid} / 1e6;
        $phrase->{Max} = $positions->{first_place}{bid} / 1e6;
        $phrase->{PremiumMin} = $positions->{$premium_entry_pos}{bid} / 1e6;
        $phrase->{PremiumMax} = $positions->{first_premium}{bid} / 1e6;

        my $no_fourth_premium = (keys %$positions) < 8;

        foreach my $advq_position (sort keys %PlacePrice::ADVQ_TO_FORECAST_POSITION) {
            my $position = $PlacePrice::ADVQ_TO_FORECAST_POSITION{$advq_position};
            if (defined $positions->{$position}) {
                my $price = $positions->{$position}{clicks}
                    ? ($positions->{$position}{budget} / $positions->{$position}{clicks}) / 1e6 : 0;
                my $bids = {
                    Position => $advq_position,
                    Bid => $positions->{$position}{bid} / 1e6,
                    Price => $price
                };
                push @{$phrase->{AuctionBids}}, $bids;

                # DIRECT-67242
                if ($no_fourth_premium && $advq_position eq 'P13') {
                    push @{$phrase->{AuctionBids}}, { %$bids, Position => 'P14' };
                }
            }
        }

        _recalc_ctr($phrase, 'CTR', $positions->{$guarantee_entry_pos});
        _recalc_ctr($phrase, 'PremiumCTR', $positions->{$premium_entry_pos});
        _recalc_ctr($phrase, 'FirstPlaceCTR', $positions->{first_place});

        foreach my $p (qw/Min Max PremiumMin PremiumMax/) {
            next unless $phrase->{$p};
            $phrase->{$p} = round2s($phrase->{$p});
        }

        if (exists $phrase->{AuctionBids} && defined $phrase->{AuctionBids}) {
            foreach my $auction_bid (@{$phrase->{AuctionBids}}) {
                foreach my $bid_type (qw/Bid Price/) {
                    $auction_bid->{$bid_type} = round2s($auction_bid->{$bid_type});
                }
            }
        }
        
        # multicurrency: удаляем обозначение фишек
        if ($phrase->{Currency} && $phrase->{Currency} eq 'YND_FIXED') {
            delete $phrase->{Currency};
        }
        
        return $phrase;
    }

    return;
}

sub _recalc_ctr {
    my ($phrase, $ctr_name, $position) = @_;
    if (defined $position && defined $position->{shows} && defined $position->{clicks}) {
        if ($position->{shows}) {
            $phrase->{$ctr_name} = round2s(100 * $position->{clicks} / $position->{shows});
        } else {
            $phrase->{$ctr_name} = 0;
        }
    }
}

=head2 filter_reports_list(type, reportslist)

    По массиву записей в БД формируем ответ пользователю в методах Get(Report|Forecast|Wordstat)List

=cut

sub filter_reports_list
{
    my ($type, $reports, %OPT) = @_;

    my @result;

    foreach my $job (sort {$a->job_id <=> $b->job_id} @$reports) {
        next if $job->is_revoked;

        my $status =
            $job->is_finished ? 'Done' :
            $job->is_failed ? 'Failed' :
            'Pending';

        my $report_url;
        if ( $job->is_finished && $type eq 'report') {
            $report_url = prepare_report_public_url( $job, $OPT{hostname} );
        }

        my $result_id = $job->job_id % $API::Settings::API4_JOB_ID_THRESHOLD;
        push @result, {
            $STATUS_REPORT_NAME{$type}->{id} => $result_id || undef,
            $STATUS_REPORT_NAME{$type}->{status} => $status,
            ( $report_url ? ( Url => $report_url ) : () ),
        };
    }

    return \@result;
}

=head2 status_moderate_int_to_ext

    Преобразует статусы модерации к внешнему формату

=cut

sub status_moderate_int_to_ext
{
    my $statuses = shift;
    my @result;
    for my $status (@$statuses) {
        if ($status =~ m/^(New|Yes|No)$/) {
            push @result, $status;
        } else {
            push @result, 'Pending';
        }
    }
    return \@result;
}

=head2 prepare_report_public_url($job, $hostname)

    Формирует внешнюю ссылку для доступа к отчету статистики
    Если hostname не указан, то домен по умолчанию — $Settings::API_SERVER_PATH

=cut

sub prepare_report_public_url {
    my ($job, $hostname) = @_;

    $hostname ||= "https://".$Settings::API_SERVER_PATH;

    my $filename = $job->result->{filename};
    my $extension = $job->args->{CompressReport} ? 'gz' : 'xml';

    return $hostname . '/storage/' . $job->ClientID . '/api_report_stat/' . $filename . '.' . $extension;
}

1;
