package API::JSON;

=head1 NAME

    API::JSON

=head1 DESCRIPTION

    Модуль подгружается в apache и позволяет использовать API Директа
    через JSON сериализацию

=cut

use strict;
use warnings;

use Data::Dumper;
use UNIVERSAL;
use Encode qw/decode_utf8 decode/;
use JSON;

use Yandex::Trace;
use Yandex::XsUtils qw//;
use Yandex::I18n;

use API;
use API::Errors;
use APICommon qw/:subs/;

use utf8;

# for debug
our $DEBUG = 0;

my %NON_NUMERIC_RESPONCE_METHODS = map { $_ => 1 } qw/GetNormalizedKeywords GetNormalizedKeywordsData/;


=head2 handler

    На входе должен быть POST запрос с такими параметрами:
        login, token - параметры аутентификации
        method - имя вызываемого метода
        version - порядковый номер версии АПИ
        param - параметр, передаваемый в функцию и запакованный JSON
    Если запрос выполнился правильно, на выходе
        Status: 200 OK
        тело ответа - результат, запакованный JSON
    Если в процессе выполнения произошла ошибка, то на выходе
        Status: 500
        тело ответа - запакованная структура {error_code => ..., error_str => ..., error_detail => ...}

=cut
sub handler
{
    my ($r) = @_;

    # Инициализируем локаль по умолчанию en для JSON. Поскольку основная инициализация производится позже, все ошибки отсюда
    # будут возвращаться в локали, установленной в данный момент.
    Yandex::I18n::init_i18n('en');

    my $err;
    if ($r->method ne 'POST') {
        $err = dieSOAP('BadRequestMethod', iget("Метод HTTP запроса должен быть POST"), dont_die => 1);
    }

    # получаем данные
    my $data = $r->content;

    print STDERR Dumper {len => length($data), readed_data => $data} if $DEBUG;

    # обыкновенная передача дополнительных параметров в soap-handler работает подозрительно,
    # поэтому дополнительные параметры (версию, объект Plack::Request) передаем через глобальные переменные
    # получаем версию апи, к которой обращается пользователь
    # apache2,plack: с libplack-perl=0.9982-2ya  $r->uri содержит странное -- регулярное выражение из LocationMatch 0_0.
    # в Геоконтексте (libplack-perl=0.9985-1) такого вроде бы нет. TODO разобраться, обновиться, снова использовать $r->uri вместо $r->request_uri
    local ($API::api_version, $API::latest, $API::api_version_full) = get_api_version_by_uri($r->request_uri || '');
    # объект Plack::Request
    local $API::plack_request = $r;

    my $form = {};

    if ($data) {
         eval {
             my $profile = Yandex::Trace::new_profile('api_json:handler', tags => 'from_json');
             $form = from_json($data)
         };
    }

    if ($@) {
        $err = dieSOAP('BadRequest', iget('Некорректный JSON запрос'), dont_die => 1);
    } else {
        eval { $form = deep_decode_utf8($form) };
        if ($@) {
            $err = dieSOAP('BadRequest', iget('Кодировка запроса отличается от UTF8'), dont_die => 1);
        }
    }

    # инициализируем профайлер
    my $log_method = $form->{method} && $form->{method} =~ /^[a-z0-9_]+$/i ? $form->{method} : 'MethodIncorrected';
    $r->env->{trace}->method("$log_method");

    my @ret;
    my $method = $form->{method};
    if (! $err) {
        eval {
            dieSOAP('MethodNotExist') if !$method || $method !~ /^[a-z0-9_]+$/i;

            my $data = $form->{param};

            # вызываем метод АПИ
            @ret = API->$method($data, {
                                            token => $form->{token}
                                            , persistent_token => $form->{persistent_token}
                                            , login => $form->{login}
                                            , fake_login => $form->{fake_login}
                                            , application_id => $form->{application_id}
                                            , finance_token => $form->{finance_token}
                                            , operation_num => $form->{operation_num}
                                            , locale => $form->{locale}
                                            , payment_token => $form->{payment_token}
                                            , agcl => $form->{agcl}
                                       });
        };

        # если ошибка случилась при выполнении метода
        $err ||= $@;
    }

    if ($err) {
        my $err_ret = {
            error_code => 500,
            error_str => iget('Внутренняя ошибка сервера')
        };

        if (UNIVERSAL::isa($err, 'SOAP::Fault')) {
            $err_ret = {error_code => $err->faultcode, error_str => $err->faultstring, error_detail => $err->faultdetail};
        }

        warn Dumper {err => to_json($err_ret)} if $DEBUG;

        return {json => $err_ret};
    } else {
        # фильтруем SOAP-заголовки
        # если объект-скаляр, то JSON его пропускает...
        # пропускаем soap::headers
        if (scalar @ret) {
            # берем только ответ АПИ без заголовков
            my $body_answer = shift @ret;

            # собираем ответ сервера пользователю
            my $response = {data => $body_answer};
            if (ref $body_answer ne 'CODE') {
                # трансляция имеет смысл только для нестримовых ответов
                if ($NON_NUMERIC_RESPONCE_METHODS{$method}) {
                    t10n('_array_of_strings' => $response->{data});
                } else {
                    t10n('_data' => $response->{data});
                }
            }
            warn Dumper {ans=>to_json($response)} if $DEBUG;

            # для данных методов - ответ печатается внутри
            # отличать ошибку от результата - по Content-Type
            if ($API::STREAM_METHODS{$method}) {
                return $body_answer;
            } else {
                return {json => $response};
            }

        } else {
            # кажется это условие никогда не выполнится
            # но если выполнится, то не факт, что метод выполнился в ошибкой - поэтому ошибку не печатаем

            return {json => {data => "empty"}};
        }
    }
}

my %TYPES = (
    'digit' => [qw/ _data
                    Price ContextPrice Coverage Probability
                    Min Max PremiumMin PremiumMax FirstPlaceCTR PremiumCTR FirstPlaceClicks PremiumClicks
                    Prices MinPrice CurrentOnSearch
                    MaxPrice AveragePrice WeeklySumLimit GoalID ClicksPerWeek AverageCPA Crr ROICoef ReserveReturn Profitability
                    CampaignIDS BannerIDS TagIDS
                    NotFound Updated NotUpdated
                    Days Hours BidCoefs AdditionalMetrikaCounters
                    WarnPlaceInterval MoneyWarningValue HolidayShowTo HolidayShowFrom
                    Shows Clicks ContextShows ContextClicks Sum Rest SumAvailableForTransfer
                    RegionID BannerID PhraseID CampaignID RubricID ParentID TagID
                    ContextPricePercent ContextLimitSum
                    MetroStation x y x1 y1 x2 y2
                    Discount UnitsRest Payed GMTOffset VersionNumber
                    Limit LimitSpent
                    ReportID ForecastID Timeout UserID
                    ClicksSearch ClicksContext SumSearch SumContext ShowsSearch ShowsContext
                    MediaplanAdID MediaplanKeywordID MediaplanCategoryID MediaplanID
                    FaultCode
                    VATRate
                    PremiumMinCTR PremiumMaxCTR MinCTR MaxCTR CTR
                    AdID
                    RetargetingConditionID RetargetingID Time
                    Capacity Utilized AdImageUploadTaskID
                    AdGroupID
                    BonusDiscount
                    AccountID
                    MobileBidAdjustment AdGroupMobileBidAdjustment
                    ShowsAveragePosition ClicksAveragePosition Revenue ROI
                    OverdraftSumAvailable OverdraftSumAvailableInCurrency
                    WebpageID DynamicMediaAdFilterID
                    Bid
                    /],
);

my %DIGIT_RTYPES = map {$_ => 1} @{$TYPES{digit}};

=head2 t10n

    Простая типизация структуры перед сериализацией в JSON.

    Превращает числовые ключи из словаря -- к цифровому представлению (сериализуется без кавычек),
        и остальные -- к строковому (в результате будут в кавычках).

=cut

sub t10n
{
    my $_profile = Yandex::Trace::new_profile('api_json:handler', tags => 't10n');
    Yandex::XsUtils::t10n(\%DIGIT_RTYPES, $_[0], $_[1]);
    #if (! ref $v) {
    #    if (defined $v) {
    #        if (exists $DIGIT_RTYPES{$k} && is_float($v)) {
    #            $v = 0 + $v;
    #        } else {
    #            $v = "$v";
    #        }
    #    }
    #} elsif (ref($v) eq 'HASH') {
    #    foreach my $k1 (keys %$v) {
    #        $v->{$k1} = t10n($k1, $v->{$k1});
    #    }
    #} elsif (ref($v) eq 'ARRAY') {
    #    foreach my $i (@$v) {
    #        $i = t10n($k, $i);
    #    }
    #} else {
    #    warn "JSON::t10n ($k) ".ref($v);
    #}
    #return $v;
}

=head2 deep_decode_utf8($binary_data);

    "глубокое" перекодирование перловой структуры

=cut

sub deep_decode_utf8 {
    my $v = shift;
    if (ref($v) eq 'ARRAY') {
        $_ = deep_decode_utf8($_) for @$v;
    } elsif (ref($v) eq 'HASH') {
        #$_ = deep_decode_utf8($_) for values %$v;
        $v->{decode("utf8", $_)} = deep_decode_utf8(delete $v->{$_}) for keys %$v;
    } elsif (!ref $v) {
        $v = decode("utf8", $v);
    }
    return $v;
}

1;
