package API::Methods::AccountManagement;

# $Id$

=head1 NAME

API::Methods::AccountManagement - общий счёт

=head1 DESCRIPTION

Методы для работы с общим счётом

=cut

use strict;
use warnings;

use List::MoreUtils qw(any none uniq);

use Yandex::I18n;

use Settings;

use RBACElementary;
use RBACDirect;
use RBAC2::DirectChecks;
use Primitives;
use PrimitivesIds;
use Client;

use APICommon qw(:subs);
use API::Filter;
use API::Errors;
use API::Validate;
use API::ValidateTools qw/_check_fields_exist/;
use API::ConvertFrom qw/convert_params/;
use Agency qw/get_agency_representatives/;
use Direct::Validation::DayBudget qw/validate_wallet_day_budget/;
use Yandex::DBTools;
use Yandex::DBShards qw/SHARD_IDS/;
use Tools qw(sms_time2string);
use API::Methods::Common;
use Wallet;

use utf8;

sub AccountManagement
{
    my ($self, $params) = @_;

    my %dispatch = (
        Get           => \&_account_management_get,
        Update        => \&_account_management_update,
        Invoice       => \&_account_management_invoice,
        Deposit       => \&_account_management_deposit,
        DepositByCard => \&_account_management_deposit,
        CheckPayment  => \&_account_management_check_payment,
        TransferMoney => \&_account_management_transfer
    );

    exists $dispatch{$params->{Action}} ? &{$dispatch{$params->{Action}}}($self, $params) : dieSOAP('InternalLogicError', 'AccountManagement');

    return $self->{ret};
}

# -- ACTION методы

# -- Get
sub _account_management_get
{
    my ($self, $params) = @_;

    my ($logins, $account_ids) = ([], []);
    if (defined $params->{SelectionCriteria}) {
        $logins      = defined( $params->{SelectionCriteria}{Logins} )
                            ? $params->{SelectionCriteria}{Logins}
                            : [];
        $account_ids = defined( $params->{SelectionCriteria}{AccountIDS} )
                            ? $params->{SelectionCriteria}{AccountIDS}
                            : [];
    }

    my %filter_wallet_options_by_wallet_id;

    # Права должны быть уже проверены - поле Logins может существовать для всех ролей >= агентство
    if (@$logins) {
        my $wallets = _get_accounts_by_logins($self, $logins);
        foreach my $wallet (@$wallets) {
            push @{$self->{preprocess}{clientuid2wallet}{ $wallet->{uid} }}, $wallet;
        }
    }
    elsif (@$account_ids) {
        my $wallet_by_id = Wallet::get_wallets_by_cids(
            $account_ids,
            DirectContext->new({
                UID          => $self->{uid},
                rbac         => $self->{rbac},
                login_rights => $self->{rbac_login_rights},
            }),
            additional_wallet_fields => [qw/agency_fio agency_name AgencyID currency sum sum_spent uid sms_time/],
            get_sum_available => 1,
        );

        my $account_id2client_id = get_cid2clientid(cid => $account_ids);
        my @client_ids = uniq values %$account_id2client_id;
        my $client_id_convert_soon = Client::mass_is_client_converting_soon(\@client_ids);
        my $client_id_must_convert = Client::mass_client_must_convert(\@client_ids);

        my $i = 0;
        foreach my $account_id (@$account_ids) {
            if (!exists $wallet_by_id->{$account_id}) {
                $self->{ret}{ActionsResult}[$i]{AccountID} = $account_id;
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights');
                next;
            }

            my $client_id = $account_id2client_id->{$account_id};
            if ($client_id_convert_soon->{$client_id}) {
                $self->{ret}{ActionsResult}[$i]{AccountID} = $account_id;
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights', APICommon::msg_converting_in_progress);
                next;
            }

            if ($client_id_must_convert->{$client_id}) {
                $self->{ret}{ActionsResult}[$i]{AccountID} = $account_id;
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights', APICommon::msg_must_convert);
                next;
            }

            my $wallet = $wallet_by_id->{$account_id}{wallet_camp};

            # можно ли данному пользователю смотреть данную кампанию?
            if (! _user_allowed_for_camp($self, $wallet->{uid}, $wallet->{wallet_cid})) {
                $self->{ret}{ActionsResult}[$i]{AccountID} = $wallet->{wallet_cid};
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights');
                next;
            }

            push @{$self->{preprocess}{clientuid2wallet}{ $wallet->{uid} }}, $wallet;
        }
    }
    else {
        # NB: только клиент может не передавать условия отбора

        my $client_chief_uid = rbac_get_chief_rep_of_client_rep($self->{uid});
        my $clientid         = get_clientid(uid => $client_chief_uid);
        my $client_currency  = get_client_currencies($clientid)->{work_currency};

        my $c = DirectContext->new({
            is_direct        => 1,
            uid              => $client_chief_uid,
            UID              => $self->{uid},
            client_chief_uid => $client_chief_uid,
            client_client_id => $clientid,
            rbac             => $self->{rbac},
            login_rights     => $self->{rbac_login_rights},
            user_ip          => $self->{user_ip},
        });

        my @contexts_array;

        # достанем "собственный" кошелек клиента
        push @contexts_array, { c => $c, agency_client_id => 0, client_currency => $client_currency };

        # DIRECT-27431
        my $subclient_perms = rbac_mass_get_subclients_rights(
                                    $self->{rbac},
                                    $self->{UID},
                                    {
                                        $client_chief_uid => $clientid
                                    },
                                    # NB: пользователь должен видеть все ОС по всем агентствам
                                    #     (если нет права редактировать кампании - то только часть параметров ОС)
                                    not_filter_by_perm => 1
                                )->{$clientid};
        $subclient_perms ||= {};

        for my $agency_ClientID ( keys %$subclient_perms ) {
            push @contexts_array, { c => $c, agency_client_id => $agency_ClientID, client_currency => $client_currency };
        }

        my $camps = Wallet::get_wallets_by_uids(
            \@contexts_array,
            additional_wallet_fields => [qw/currency sum sum_spent uid sms_time agency_name/],
            get_sum_available => 1,
            without_spent_today => 1,
        );

        foreach my $camp (@$camps) {
            my $wallet = $camp->{wallet_camp};

            next if ! $wallet->{enabled};
            
            push @{$self->{preprocess}{clientuid2wallet}{ $client_chief_uid }}, $wallet;

            $filter_wallet_options_by_wallet_id{ $wallet->{wallet_cid} } = 1
                if $camp->{agency_client_id}
                    && ! $subclient_perms->{ $camp->{agency_client_id} }{isSuperSubClient};
        }

        if (! keys %{ $self->{preprocess}{clientuid2wallet} }) { # client hasn't any active wallet
            push @{$self->{ret}{ActionsResult}[0]{Errors}}, get_error_object('NoWallet');
        }
    }

    my $clients_uids = [ keys %{ $self->{preprocess}{clientuid2wallet} } ];

    my $client_objects = APICommon::get_client_object( $self, uid => $clients_uids, get_ext_params => 1 );

    my @cids;
    my $i = 0;
    foreach my $co ( @$client_objects ) {
        my $wallets = filter_account_management_get_object($co);

        foreach my $wallet (@$wallets) {
            if (exists $filter_wallet_options_by_wallet_id{ $wallet->{AccountID} }) {
                # если нет права редактировать кампании - покажем только часть параметров ОС
                $wallet = { map { $_ => $wallet->{ $_ } } qw/ AccountID Amount Currency AgencyName Login / };
            }
            $self->{ret}->{Accounts}->[$i++] = $wallet;
            push @cids, $wallet->{AccountID};
        }
    }

    $self->{cluid} = $clients_uids;
    $self->{syslog_data}->{cid} = [uniq @cids] if (@cids);

    return 0;
}

# -- Update
sub _account_management_update
{
    my ($self, $params) = @_;

    my $accounts = $params->{Accounts};

    for ( my $i = 0; $i < scalar(@$accounts); $i++ ) {

        if (exists $self->{ret}{ActionsResult}
                && exists $self->{ret}{ActionsResult}[$i]
                && exists $self->{ret}{ActionsResult}[$i]{Errors}
                && @{$self->{ret}{ActionsResult}[$i]{Errors}}
        ) {
            next;
        }

        my $camp = convert_params(campaign => $accounts->[ $i ], 1);

        my $aid = $camp->{AccountID};

        api_check_limit_by_method($self, $aid, 'AccountManagement__update', 1);

        my $old_camp = $self->{_preprocessed}{account_by_id}{$aid};

        if (defined $camp->{AccountDayBudget}) {
            # NB: валидация расположена тут, т.к. validate_wallet_day_budget требуются входные данные уже после конвертации

            my $vr = validate_wallet_day_budget(
                cid => $aid,
                new_day_budget_data => $camp,
                old_day_budget_data => {
                    day_budget => $old_camp->{day_budget}{sum},
                    day_budget_daily_change_count => $old_camp->{day_budget}{daily_change_count}
                },
                currency => $old_camp->{currency},
            );

            foreach my $err ( @{ $vr->get_errors() } ) {
                push @{ $self->{ret}{ActionsResult}[$i]{Errors} }, {
                    FaultCode => $err->code,
                    FaultString => $err->text,
                    FaultDetail => $err->description,
                };
            }

            next if not $vr->is_valid();
            
            my $client_chief_uid = $self->{_preprocessed}{chief_by_account_id}{$aid};

            Campaign::save_day_budget($camp, $old_camp, $self->{UID}, $client_chief_uid, $aid); 
        }

        my $to_update = {};

        # -- SmsNotification
        # -- если валидация была пройдена хотя бы одним из параметров(SmsTimeFrom или SmsTimeTo),
        # -- то имеет смысл устанавливать этот параметр, иначе grep даст в итоге FALSE
        my @sms_time_fields = qw/sms_time_hour_from sms_time_min_from sms_time_hour_to sms_time_min_to/;

        if ( grep { defined $_ } @$camp{ @sms_time_fields } ) {
            # -- Если хотя бы один из этих параметров окажется undef, функция установит время по умолчанию: 9-21
            $to_update->{sms_time} = sms_time2string( @{$camp}{ @sms_time_fields });

        }

        for my $flag (qw/active_orders_money_out_sms notify_order_money_in_sms/) {
            next if !defined $camp->{$flag};

            # -- собирать в массив а потом делать join в данном случае не очень правильно
            $to_update->{sms_flags} .= defined $to_update->{sms_flags} && length($to_update->{sms_flags}) > 0 ? ','.$flag : $flag;
        }

        # -- смс флаги не будут выставлены, если active_orders_money_out_sms и notify_order_money_in_sms приехали как No
        $to_update->{sms_flags} = '' if exists $camp->{SmsNotification} && !exists $to_update->{sms_flags};

        # -- EmailNotification
        # -- Для кошелька передаются параметры Email и MoneyWarningValue
        for my $flag (qw/email money_warning_value/) {
            next if !defined $camp->{$flag};

            $to_update->{$flag} = $camp->{$flag};
        }

        # -- метод вернет либо количество строк, задействованных с помощью UPDATE, либо undef
        my $affected = Wallet::update_wallet_sms_email_settings(
            $camp->{AccountID},
            $to_update,
            paused_by_day_budget_email => $camp->{paused_by_day_budget_email} ? 1 : 0,
            paused_by_day_budget_sms   => $camp->{paused_by_day_budget_sms} ? 1 : 0,
        );

        if (!defined $affected || $affected == 0 ) {
            push @{$self->{ret}{ActionsResult}[$i]{Errors}}, get_error_object('AccountNotUpdated', $camp->{AccountID});
        }
        else {
            $self->{ret}{ActionsResult}[$i]{AccountID} = $camp->{AccountID};
        }

        api_update_limit($self, $camp->{AccountID}, 'AccountManagement__update', 1);
    }

    return;
}

# -- Invoice
=head2 _account_management_invoice(self, params)
    Выставление счёта по params->{Payments}
    Заполняет self->{ret}->[]
=cut

sub _account_management_invoice($$)
{
    my ($self, $params) = @_;

    my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};

    foreach my $i(0 .. $#account_ids) {
        next if exists $self->{ret}->{ActionsResult}->[$i]->{Errors};

        my $limit_error = api_check_limit_by_method($self, $account_ids[$i], 'FinOperations__daily', 1, object_errors => 1);
        if (ref $limit_error) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [$limit_error];
            next;
        }

        my $format_payment = [{CampaignID => $account_ids[$i], Currency => $params->{Payments}->[$i]->{Currency}, Sum => $params->{Payments}->[$i]->{Amount}}];

        my $result = create_pay_campaign_request($self, $format_payment);

        if (defined $result->{error}) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [$result->{error}];
        } elsif (!defined $result->{result}) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('InternalLogicError', 'AccountManagement Invoice')];
        } else {
            $self->{ret}->{ActionsResult}->[$i]->{URL} = $result->{result};
            api_update_limit($self, $account_ids[$i], 'FinOperations__daily', 1);
        }
    }
}

# -- Deposit
=head2 _account_management_deposit(self, params)
    Оплата кампаний по params->{Payments} через ЯД, банк или овердрафт
    Заполняет self->{ret}->[]
=cut

sub _account_management_deposit($$)
{
    my ($self, $params) = @_;

    my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};

    foreach my $i (0 .. $#account_ids) {
        next if exists $self->{ret}->{ActionsResult}->[$i]->{Errors};

        my $limit_error = api_check_limit_by_method($self, $account_ids[$i], 'FinOperations__daily', 1, object_errors => 1);
        if (ref $limit_error) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [$limit_error];
            next;
        }

        my $format_payment = [{CampaignID => $account_ids[$i], Currency => $params->{Payments}->[$i]->{Currency}, Sum => $params->{Payments}->[$i]->{Amount}}];
        my $pay_method = (defined $params->{Payments}->[$i]->{Origin})? $params->{Payments}->[$i]->{Origin} : 'Bank';
        my $contract = (defined $params->{Payments}->[$i]->{Origin})? undef : $params->{Payments}->[$i]->{Contract};
        my $format_request = {Payments => $format_payment, ContractID => $contract, PayMethod => $pay_method};

        if ( $params->{Action} eq 'DepositByCard' ) {
            $format_request->{PayMethod}   = 'LinkedCard';
            $format_request->{PayMethodID} = $params->{PayMethodID};
            $format_request->{CustomTransactionID} = $params->{CustomTransactionID};
            $format_request->{Version} = $params->{Version} if exists $params->{Version};
        }

        my $result = API::Methods::Common::payment_common($self, $format_request);

        if (defined $result->{error}) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [$result->{error}];
        } elsif (!defined $result->{result}) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('InternalLogicError', 'AccountManagement Deposit')];
            API::Methods::Common::pay_notification_common($self, $format_payment); # нотификации в случаях успеха
        } else {
            $self->{ret}->{ActionsResult}->[$i]->{AccountID} = $account_ids[$i];
            api_update_limit($self, $account_ids[$i], 'FinOperations__daily', 1);
            if ( $params->{Action} eq 'DepositByCard' ) {
                $self->{ret}->{ActionsResult}->[$i]->{PurchaseToken} = $result->{result};
            }
        }
    }
}

sub _account_management_check_payment
{
    my ($self, $params) = @_;

    return if exists $self->{ret}->{ActionsResult}->[0]->{Errors};

    my $transaction_id = $params->{CustomTransactionID};

    # my $limit_error = api_check_limit_by_method($self, $transaction_id, 'FinOperations__daily', 1, object_errors => 1);
    # if (ref $limit_error) {
    #     $self->{ret}->{ActionsResult}->[0]->{Errors} = [$limit_error];
    #     next;
    # }

    my $request = { CustomTransactionID => $transaction_id };

    my $result = API::Methods::Common::check_payment_common($self, $request);

    if (defined $result->{error}) {
        $self->{ret}->{ActionsResult}->[0]->{Errors} = [$result->{error}];
    } else {
        $self->{ret}->{ActionsResult}->[0]->{Status} = $result->{result};
        #api_update_limit($self, $transaction_id, 'FinOperations__daily', 1);
    }
}

# -- TransferMoney
=head2 _account_management_transfer(self, params)
    Перенос денег по params->{Transfers}
    Заполняет self->{ret}->[]
=cut

sub _account_management_transfer($$)
{
    my ($self, $params) = @_;

    return if exists $self->{ret}->{ActionsResult}->[0]->{Errors};

    my $request_currency = $params->{Transfers}->[0]->{Currency} || 'YND_FIXED';
    my $format_from = [{CampaignID => $params->{Transfers}->[0]->{FromAccountID}, Sum => $params->{Transfers}->[0]->{Amount}, Currency => $request_currency}];
    my $format_to = [{CampaignID => $params->{Transfers}->[0]->{ToAccountID}, Sum => $params->{Transfers}->[0]->{Amount}, Currency => $request_currency}];
    my $format_transfer = {FromCampaigns => $format_from, ToCampaigns => $format_to};

    my $limit_error = api_check_limit_by_method($self, $params->{Transfers}->[0]->{FromAccountID}, 'AccountManagement__transfer', 1,
        object_errors => 1, obj_name => 'AccountManagement Transfer');
    if (ref $limit_error) {
        $self->{ret}->{ActionsResult}->[0]->{Errors} = [$limit_error];
        return;
    }

    $limit_error = api_check_limit_by_method($self, $params->{Transfers}->[0]->{ToAccountID}, 'AccountManagement__transfer', 1,
        object_errors => 1, obj_name => 'AccountManagement Transfer');
    if (ref $limit_error) {
        $self->{ret}->{ActionsResult}->[0]->{Errors} = [$limit_error];
        return;
    }

    my $result = API::Methods::Common::transfer_common($self, $format_transfer);

    if (defined $result->{error}) {
        $self->{ret}->{ActionsResult}->[0]->{Errors} = [$result->{error}];
    } elsif (!defined $result->{result}) {
        $self->{ret}->{ActionsResult}->[0]->{Errors} = [get_error_object('InternalLogicError', 'AccountManagement TransferMoney')];
    } else {
        $self->{ret}->{ActionsResult}->[0]->{AccountID} = $params->{Transfers}->[0]->{FromAccountID};
        api_update_limit($self, $params->{Transfers}->[0]->{FromAccountID}, 'AccountManagement__transfer', 1);
        api_update_limit($self, $params->{Transfers}->[0]->{ToAccountID}, 'AccountManagement__transfer', 1);

        my $compiled_params_for_notifications = $result->{result};

        API::Methods::Common::transfer_notification_common($self, $params, $compiled_params_for_notifications);
    }
}

# -- вспомогательные методы
sub _get_accounts_by_logins
{
    my ($self, $logins) = @_;

    return [] if ! @$logins;

    my $result    = [];
    my $uid2login = {};
    my $login2uid = get_login2uid(login => $logins);
    my $i = 0;
    for my $login (@$logins) {
        my $client_uid = $login2uid->{$login};

        if (!defined $client_uid) {
            $self->{ret}{ActionsResult}[$i]{Login} = $login;
            push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('BadClient');
            delete $login2uid->{$login};
            next;
        }

        $uid2login->{$client_uid} = $login;
    }

    my $uid2clientid = get_uid2clientid(uid => [values %$login2uid]);

    my ($is_agency, $agency_client_id, $agency_login) = (0, undef, undef);

    if ($self->{rbac_login_rights}->{role} && $self->{rbac_login_rights}->{role} eq 'agency') {

        if (! ($agency_client_id = rbac_get_agency_clientid_by_uid( $self->{uid})) ) {
            dieSOAP("BadParams", iget("Агентство не найдено"));
        }

        $agency_login = get_login(uid => $self->{uid});

        $is_agency = 1;
    }

    my @clientids = values %$uid2clientid;
    my $clients_currencies = mass_get_client_currencies(\@clientids);

    my $wallets_request_params = [];
    my $wallets_request_opts   = {
        additional_wallet_fields => defined $agency_client_id
                ? [qw/agency_fio agency_name AgencyID currency sum sum_spent uid sms_time/]
                : [qw/currency sum sum_spent uid sms_time/],
        get_sum_available => 1,
    };

    my %wallet_request_params_keys; # чтобы не спрашивать кошелёк для одной и той же пары {client_client_id, agency_client_id}

    # проверить что логины соответствуют агентству
    my $client_uid2chief = rbac_get_chief_reps_of_client_reps([values %$login2uid]);
    my $client_uid2is_agency_client = $is_agency && %$login2uid
        ? RBACDirect::rbac_is_agency_of_client_multi(rbac_get_chief_rep_of_agency_rep($self->{uid}), [values %$login2uid])
        : {};

    while ( my ($login, $client_uid) = each %$login2uid ) {

        if ($is_agency) {
            # У агентства не может быть общего счета
            if ($login eq $agency_login ) {
                $self->{ret}{ActionsResult}[$i]{Login} = $login;
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights', iget("У агентства не может быть общего счёта"));
                next;
            }

            if ( !$client_uid2is_agency_client->{$client_uid} ) {
                $self->{ret}{ActionsResult}[$i]{Login} = $login;
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights', iget("Не является клиентом агентства"));
                next;
            }
        }

        my $client_chief_uid = $client_uid2chief->{$client_uid};
        # поскольку они оба смотрят на один client_id, то смело можно сделать так
        if ($client_chief_uid != $client_uid) {
            $uid2clientid->{$client_chief_uid} = $uid2clientid->{$client_uid};
            $uid2login->{$client_chief_uid."_chef_of"} = $uid2login->{$client_uid};
        }

        # Теперь в поле client_chief_uid именно uid главного представителя
        my $c = DirectContext->new({
            is_direct        => 1,
            uid              => $client_uid,
            UID              => $self->{uid},
            client_chief_uid => $client_chief_uid,
            client_client_id => $uid2clientid->{$client_chief_uid},
            rbac             => $self->{rbac},
            login_rights     => $self->{rbac_login_rights},
            user_ip          => $self->{user_ip},
        });

        unless (exists $wallet_request_params_keys{$uid2clientid->{$client_chief_uid}.'-'.($agency_client_id || '0')}) {
            push @{$wallets_request_params}, {
                c => $c,
                defined $agency_client_id ? ( agency_client_id => $agency_client_id ): (),
            };
            $wallet_request_params_keys{$uid2clientid->{$client_chief_uid}.'-'.($agency_client_id || '0')} = undef;
        }
    }

    # -- Если это не агенство и клиент прислал свой собственный логин...
    if ( !$is_agency && (scalar(keys(%{$uid2login})) == 1) &&  (exists($uid2login->{$self->{uid}})) ) {
        my $client_id = $uid2clientid->{$self->{uid}};
        # -- для свободных клиентов или клиентов без агенств, нужно передавать 0, в конце
        if ( (rbac_check_freedom($self->{rbac}, $self->{uid}) || ! rbac_has_agency($self->{rbac}, $self->{uid}))
            && (! (exists $wallet_request_params_keys{$client_id.'-0'}))) {
            push @{$wallets_request_params}, {
                c => DirectContext->new({
                    is_direct        => 1,
                    uid              => $self->{uid},
                    UID              => $self->{uid},
                    client_chief_uid => rbac_get_chief_rep_of_client_rep($self->{uid} ),
                    client_client_id => $client_id,
                    rbac             => $self->{rbac},
                    login_rights     => $self->{rbac_login_rights},
                    user_ip          => $self->{user_ip},
                }),
                agency_client_id => 0,
                client_currency => $clients_currencies->{$client_id}->{work_currency},
            };
            $wallet_request_params_keys{$client_id.'-0'} = undef;
        }
    }

    my $user_not_alowed_for_login = {};
    foreach my $wallet ( @{Wallet::get_wallets_by_uids( $wallets_request_params, %{$wallets_request_opts}, without_spent_today => 1 )} ) {

        # -- campaigns_count - количество кампаний, под кошельком!!!

        if ($wallet->{campaigns_count} > 0 && !$wallet->{wallet_camp}{enabled}) {
            $self->{ret}{ActionsResult}[$i]{Login} = $uid2login->{$wallet->{uid}};
            push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoWallet');
            next;
        }
        elsif ( $wallet->{campaigns_count} == 0 ) {

            $self->{ret}{ActionsResult}[$i]{Login} = $uid2login->{$wallet->{uid}};

            # -- если вызов был от имени представителя....
            if (exists $uid2login->{$wallet->{uid}."_chef_of"}) {
                $self->{ret}{ActionsResult}[$i]{Login} = $uid2login->{$wallet->{uid}."_chef_of"};
            }

            if ( !$wallet->{wallet_camp}{enabled} ) {
                # -- Нет кошелька
                if ( $wallet->{wallet_cid} == 0 ) {
                    push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoWallet');

                }else{
                    push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('CantEnableWalletNoCampaigns');
                }
            }else {
                push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoWallet');
            }

            next;
        }

        # -- Проверяем - а можно ли данному пользователю смотреть данную кампанию?
        if ( !_user_allowed_for_camp($self, $wallet->{wallet_camp}->{uid}, $wallet->{wallet_cid}) ) {
            my $_login = $uid2login->{$wallet->{uid}};
            $user_not_alowed_for_login->{ $_login } = 1 if !exists $user_not_alowed_for_login->{ $_login };
            next;
        }

        push @{$result}, $wallet->{wallet_camp};
    }

    if (scalar(keys(%{$user_not_alowed_for_login}))>0) {
        foreach my $login (keys(%{$user_not_alowed_for_login})) {
            $self->{ret}{ActionsResult}[$i]{Login} = $login;
            push @{$self->{ret}{ActionsResult}[$i++]{Errors}}, get_error_object('NoRights', iget("Вы не имеете прав на просмотр запрошенных данных. Обратитесь в службу технической поддержки"));
            next;
        }
    }

    return $result;
}

# -- Проверка прав
sub validate_account_management
{
    my ($self, $params) = @_;

    my %dispatch = (
        Get           => \&_get_rights,
        Update        => [
            \&_update_rights,
            \&_update_object,
        ],
        Invoice       => [
            \&_fill_preprocessed_ids_from_payments,
            \&_check_payment_rights,
            \&_check_wallets,
            \&_validate_payment_currencies
        ],
        Deposit       => [
            \&_fill_preprocessed_ids_from_payments,
            \&_check_payment_rights,
            \&_check_wallets,
            \&_validate_payment_currencies,
        ],
        DepositByCard => [
            \&_validate_custom_transaction_id,
            \&_fill_preprocessed_ids_from_payments,
            \&_check_payment_rights,
            \&_check_wallets,
            \&_validate_card_payment_currencies,
        ],
        TransferMoney => [
            \&_fill_preprocessed_ids_from_transfers,
            \&_check_transfer_rights,
            \&_check_wallets,
            \&_validate_transfer_currencies
        ],
        CheckPayment => [
            \&_validate_custom_transaction_id,
            sub { $_[0]->{ret} = {ActionsResult => []}; return; }
        ],
    );

    # -- Проверка прав может временно отсутствовать
    if ( exists $dispatch{$params->{Action}} ) {
        if (ref $dispatch{$params->{Action}} eq 'CODE' ) {
            my @errors = &{$dispatch{$params->{Action}}}($self, $params);
            return (@errors) if @errors;
        }
        elsif (ref $dispatch{$params->{Action}} eq 'ARRAY' ) {
            foreach my $sub (@{$dispatch{$params->{Action}}}) {
                if ($sub) {
                    my @errors = &{$sub}($self, $params);
                    return (@errors) if @errors;
                }
            }
        }
    }

    return;
}

=head2 _user_allowed_for_camp(self, uid, cid)

    Проверка прав пользователя вызвавшего метода, на кампанию

=cut

sub _user_allowed_for_camp
{
    my ($self, $uid, $cid) = @_;

    return !RBAC2::DirectChecks::rbac_cmd_by_owners_of_cid_through_uid(
        # -- rbac
        $self->{rbac},
        # -- vars
        {
            UID     => $self->{uid},
            uid     => $uid,
            rights  => $self->{rbac_login_rights},
            cid     => $cid
        }
    );
}

=head2 _get_rights(self, params)

    Проверка прав на получение информации о счетах клиентов

=cut

sub _get_rights
{
    my ($self, $params) = @_;

    $self->{ret} = {ActionsResult => [], Accounts => [] };

    my $client_has_agency = rbac_has_agency($self->{rbac}, $self->{uid});
    my $is_client_free    = ! rbac_check_freedom($self->{rbac}, $self->{uid});

    my ($logins, $account_ids) = ([], []);
    if (defined $params->{SelectionCriteria}) {
        $logins = defined( $params->{SelectionCriteria}{Logins} )
                    ? $params->{SelectionCriteria}{Logins}
                    : [];

        $account_ids = defined( $params->{SelectionCriteria}{AccountIDS} )
                        ? $params->{SelectionCriteria}{AccountIDS}
                        : [];
    }

    if ($self->{rbac_login_rights}{role} =~ /^(?:user|client)$/) {
        # клиент может вообще ни чего не прислать
        if (@$logins && @$account_ids) {
            return ('BadParams', iget("Необходимо указывать либо параметр Logins, либо параметр AccountsIDS"));
        }
        elsif (@$logins) {
            if (@$logins > 1) {
                return ('BadParams', iget("Неверный список входных параметров: Клиент может передавать только один логин"));
            }

            if ($self->{uid} ne (get_uid_by_login2($logins->[0]) // '')) {
                return ('NoRights', iget("Вы не имеете права использовать данный логин"));
            }
        }
        elsif (@$account_ids) {
            if (@$account_ids > 1 && !$is_client_free) {
                return ('BadParams', iget("Неверный список входных параметров: Клиент может передать только один AccountID"));
            }
        }
    }
    else {
        if (! @$logins && ! @$account_ids) {
            return ('BadParams', iget("Необходимо указать хотя бы один параметр"));
        }

        if (@$logins && @$account_ids) {
            return ('BadParams', iget("Необходимо указывать либо параметр Logins, либо параметр AccountsIDS"));
        }
    }

    return;
}

=head2 _update_rights(self, params)

    Проверка прав на обновление информации об общем счете

=cut

sub _update_rights
{
    my ($self, $params) = @_;

    $self->{ret} = {ActionsResult => []};

    if (! scalar @{$params->{Accounts}} ) {
        return ('EmptyArray', iget('Массив Accounts пуст') );
    }

    my @right_errors;
    my @convert_soon_errors;
    my @must_convert_errors;

    my $account_ids = [ uniq map { $_->{AccountID} } @{$params->{Accounts}} ];

    my $account_id2client_id = get_cid2clientid(cid => $account_ids);
    my @client_ids = uniq values %$account_id2client_id;
    my $client_id_convert_soon = Client::mass_is_client_converting_soon(\@client_ids);
    my $client_id_must_convert = Client::mass_client_must_convert(\@client_ids);

    my $camps = Wallet::get_wallets_by_cids(
        $account_ids,
        DirectContext->new({
            UID          => $self->{uid},
            rbac         => $self->{rbac},
            login_rights => $self->{rbac_login_rights},
        }),
        additional_wallet_fields => [qw/ClientID uid currency/],
        get_sum_available => 1,
    );

    my ($cluids, $cids);
    for ( my $i = 0; $i < scalar(@{$params->{Accounts}}); $i++ ) {

        my $update_data = $params->{Accounts}[$i];

        my $aid = $update_data->{AccountID};

        if ($self->{rbac_login_rights}{role} =~ /^(superreader)$/) { # super|support|media|placer|
            push @right_errors, $aid;
            next;
        }

        # -- А может таковой кампании вообще нет? :)
        my $wallet_data = $camps->{$aid};
        if (!$wallet_data) {
            push @right_errors, $aid;
            next;
        }

        my $client_id = $account_id2client_id->{$aid};
        if ($client_id_convert_soon->{$client_id}) {
            push @convert_soon_errors, $aid;
            next;
        }

        if ($client_id_must_convert->{$client_id}) {
            push @must_convert_errors, $aid;
            next;
        }

        # NB: проверка _user_allowed_for_camp весьма подозрительна, т.к. основана на том факте, 
        # что в структуре, описывающей ОС без кампаний, не возвращается uid (! exists $camps->{$aid}{wallet_camp}{uid})
        # поэтому решено явно запрещать редактировать отключенные ОС (в том числе ОС без кампаний)
        if ( ! $wallet_data->{wallet_camp}{enabled} ) {
            push @right_errors, $aid;
            next;
        }

        # могут настраивать ДБ на ОС:
        # * самоходные клиенты - всегда
        # * субклиенты - при наличии разрешения от агентства
        # * менеджеры для своих клиентов - всегда
        # * агентства при наличии прав на кампанию - всегда 
        # * представители агентства - при разрешении от агентства
        if ( $wallet_data->{agency_client_id} ) { # агентский ОС
            if ( $self->{rbac_login_rights}{role} =~ /^client$/ ) { # оператор - субклиент
                # настройка бюджета
                if ( defined($update_data->{AccountDayBudget}) && ! $wallet_data->{wallet_camp}{allow_transfer_money} ) {
                    push @right_errors, $aid;
                    next;
                }        

                # настройка уведомлений
                if ( ( defined($update_data->{SmsNotification}) || defined($update_data->{EmailNotification}) )
                     && ! $wallet_data->{wallet_camp}{is_super_subclient} )
                {
                    push @right_errors, $aid;
                    next;
                }
            }
            elsif ( $self->{rbac_login_rights}{role} =~ /^agency$/ ) # оператор - представитель агентства
            {
                my $agency_reps = get_agency_representatives($wallet_data->{agency_client_id});
                my $curr_rep = $agency_reps->{ $self->{uid} } // {};
                # настройка бюджета
                if ( defined($update_data->{AccountDayBudget}) && $curr_rep->{agency_disallow_money_transfer} ) {
                    push @right_errors, $aid;
                    next;
                }      
            }
        }

        if ( !_user_allowed_for_camp($self, $wallet_data->{wallet_camp}{uid}, $aid) ) {
            push @right_errors, $aid;
            next;
        }

        $self->{_preprocessed}{account_by_id}{$aid} = $camps->{$aid}{wallet_camp};
        $self->{_preprocessed}{chief_by_account_id}{$aid} = $camps->{$aid}{wallet_camp}{uid};

        push @$cluids, $camps->{$aid}{wallet_camp}{uid};
        push @$cids, $aid;
    }

    return ('NoRights', iget("Вы не имеете прав на редактирование общих счетов: ".join(', ', @right_errors) )) if @right_errors;
    return ('NoRights', APICommon::msg_converting_in_progress) if @convert_soon_errors;
    return ('NoRights', APICommon::msg_must_convert) if @must_convert_errors;

    $self->{cluid} = $cluids if (defined $cluids);
    $self->{syslog_data}->{cid} = $cids if (defined $cids);

    return;
}

=head2 _update_object(self, params)

    Проверка объектов для обновления информации об общих счетах

=cut

sub _update_object
{
    my ($self, $params) = @_;

    for (my $i = 0; $i < scalar(@{ $params->{Accounts} }); $i++) {
        my $new_account = $params->{Accounts}->[$i];

        if (defined $new_account->{AccountDayBudget}) {

            my $day_budget = $new_account->{AccountDayBudget};

            if (my @err = _check_fields_exist($day_budget, [qw/Amount/], type => 'float', def => 1)) {
                push @{$self->{ret}{ActionsResult}[$i]{Errors}}, get_error_object(@err);
            }

            if (my @err = _check_fields_exist($day_budget, [qw/SpendMode/], type => 'list', list => [qw/Default Stretched/], def => 1)) {
                push @{$self->{ret}{ActionsResult}[$i]{Errors}}, get_error_object(@err);
            }
        }

        if (my @err = API::Validate::_validate_sms_notification($self, $new_account, lite => 1)) {
            push @{$self->{ret}{ActionsResult}[$i]{Errors}}, get_error_object(@err);
        }

        if (my @err = API::Validate::_validate_email_notification($self, $new_account, lite => 1)) {
            push @{$self->{ret}{ActionsResult}[$i]{Errors}}, get_error_object(@err);
        }
    }

    return;
}

=head2 _validate_custom_transaction_id(self, params)

    Валидация CustomTransactionID

=cut

sub _validate_custom_transaction_id {
    my ($self, $params) = @_;

    if (my @errors = _check_fields_exist($params, [qw/CustomTransactionID/], def => 1, type => 'string', not_empty => 1)) {
        return @errors;
    }

    my $transaction_id = $params->{CustomTransactionID};

    return ('BadParams', iget("Поле %s должно содержать строку длиной %s символа, состоящую из цифр и букв латинского алфавита", 'CustomTransactionID', 32))
        unless $transaction_id && $transaction_id =~ m/^[0-9a-zA-Z]{32}$/;

    return;
}

=head2 _fill_preprocessed_ids_from_payments(self, params)

Заполнение хеша self->{_preprocessed} необходимыми для проверок данными из базы по идентификаторам кошелька из $params->{Payments}. Ключом служит id кошелька.
self->{ret}->{ActionsResult}->[] заполняется ошибками там, где не нашлось кошелька.
Здесь же заполняются cluid и cid для журналирования

=cut

sub _fill_preprocessed_ids_from_payments($$) {
    my ($self, $params) = @_;
    $self->{ret} = {ActionsResult => []};

    my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};
    my $wallet_ids = get_hashes_hash_sql(PPC(cid => \@account_ids), ["select cid, IFNULL(currency, 'YND_FIXED') currency, type, uid from campaigns", where => {cid => SHARD_IDS}]);

    my (@cluids, @cids);
    foreach my $i(0 .. $#account_ids) {
        $self->{ret}->{ActionsResult}->[$i] = {};
        if (exists $wallet_ids->{$account_ids[$i]}) {
            if ((!defined $wallet_ids->{$account_ids[$i]}->{type}) || $wallet_ids->{$account_ids[$i]}->{type} ne 'wallet') {
                $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('NoRights')];
            } else {
                $self->{_preprocessed}->{$account_ids[$i]} = $wallet_ids->{$account_ids[$i]};
                push @cluids, $wallet_ids->{$account_ids[$i]}->{uid};
                push @cids, $wallet_ids->{$account_ids[$i]}->{cid};
            }
        } else {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('NoRights')];
        }
    }
    $self->{cluid} = [uniq @cluids] if (@cluids);
    $self->{syslog_data}->{cid} = [uniq @cids] if (@cids);

    return;
}

=head2 _check_payment_rights(self, params)

    Проверка прав на оплату кошельков (кампаний) по self->{_preprocessed}->{ids}
    self->{ret}->{ActionsResult}->[] заполняется ошибками там, где нет прав.
    Не обрабатывает элементы, содержащие ошибки

=cut

sub _check_payment_rights($$) {
    my ($self, $params) = @_;
    my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};
    foreach my $i(0 .. $#account_ids) {
        next if exists $self->{ret}->{ActionsResult}->[$i]->{Errors};

        my $cid = $account_ids[$i];
        my $error_code = check_rbac_rights($self, 'api_payCamp', {
            UID => $self->{uid},
            uid => $self->{_preprocessed}->{$account_ids[$i]}->{uid},
            rights => $self->{rbac_login_rights},
            cid => $cid,
        });

        my $error_text;
        if ($error_code) {
            $error_text = '';
        } else {
            my $client_id = get_clientid(cid => $cid);
            if (Client::is_client_converting_soon($client_id)) {
                $error_text = APICommon::msg_converting_in_progress;
            } elsif (Client::client_must_convert($client_id)) {
                $error_text = APICommon::msg_must_convert;
            }
        }
        if (defined $error_text) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('NoRights', $error_text)];
        }
    }
    return;
}

=head2 _prohibit_payments_in_uah

    Запрет на платежи в гривнах (DIRECT-66515)

=cut

sub _prohibit_payments_in_uah {
    my ($self, $idx, $currency) = @_;

    if (($currency // '') eq 'UAH') {
        $self->{ret}->{ActionsResult}->[$idx]->{Errors} = [get_error_object('BadCurrency')];
        return 1;
    }

    return 0;
}

=head2 _validate_payment_currencies(self, params)

    Проверка на то, что запросы указаны либо в допустимой валюте, либо без неё
    Пополняет self->{ret} в случае ошибок
    Не обрабатывает элементы, содержащие ошибки

=cut

sub _validate_payment_currencies($$) {
    my ($self, $params) = @_;

    my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};
    foreach my $i (0 .. $#account_ids) {
        next if exists $self->{ret}->{ActionsResult}->[$i]->{Errors};
        
        my $camp_currency = $self->{_preprocessed}->{$account_ids[$i]}->{currency};
        next if _prohibit_payments_in_uah($self, $i, $camp_currency);

        my $request_currency = $params->{Payments}->[$i]->{Currency} || 'YND_FIXED';
        if ($request_currency eq 'YND_FIXED' || $camp_currency ne $request_currency) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('BadCurrency')];
            next;
        }
    }

    return;
}

=head2 _validate_card_payment_currencies(self, params)

    Проверка на то, что указанная в запросе валюта совпадает с валютой счёта.
    Если в запросе валюта не указана, то подставляется валюта счета.
    Не обрабатывает элементы, содержащие ошибки

=cut

sub _validate_card_payment_currencies($$) {
    my ($self, $params) = @_;

    my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};
    foreach my $i (0 .. $#account_ids) {
        next if exists $self->{ret}->{ActionsResult}->[$i]->{Errors};

        my $camp_currency = $self->{_preprocessed}->{$account_ids[$i]}->{currency};
        next if _prohibit_payments_in_uah($self, $i, $camp_currency);

        my $req_currency  = $params->{Payments}->[$i]->{Currency} || 'YND_FIXED';
        if ($req_currency eq 'YND_FIXED' || $req_currency ne $camp_currency) {
            $self->{ret}->{ActionsResult}->[$i]->{Errors} = [get_error_object('BadCurrency')];
            next;
        }

        $params->{Payments}->[$i]->{Currency} ||= 'RUB'; # ||= $camp_currency;
    }

    return;
}

=head2 _fill_preprocessed_ids_from_transfers(self, params)

    Заполнение хеша self->{_preprocessed} необходимыми для проверок данными из базы по идентификаторам кошелька из $params->{Transfers}. Ключом служит id кошелька.
    self->{ret}->{ActionsResult}->[0]->{Errors} заполняется ошибками «NoRights», если не нашлось кошелька.

=cut

sub _fill_preprocessed_ids_from_transfers($$) {
    my ($self, $params) = @_;
    $self->{ret} = {ActionsResult => []};

    my @account_ids = ($params->{Transfers}->[0]->{FromAccountID}, $params->{Transfers}->[0]->{ToAccountID});
    my $wallet_ids = get_hashes_hash_sql(PPC(cid => \@account_ids), ["select cid, IFNULL(currency, 'YND_FIXED') currency, type, uid, agencyuid from campaigns", where => {cid => SHARD_IDS}]);
    $self->{ret}->{ActionsResult}->[0] = {};

    if ((scalar uniq @account_ids) == 1) {
        push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('BadTransferMoneyRequest', iget("Номера общих счетов должны различаться при переносе денег"));
    }

    my (@cluids, @cids);
    foreach my $i(0 .. $#account_ids) {
        if (exists $wallet_ids->{$account_ids[$i]}) {
            if ((!defined $wallet_ids->{$account_ids[$i]}->{type}) || $wallet_ids->{$account_ids[$i]}->{type} ne 'wallet') {
                if (exists $self->{ret}->{ActionsResult}->[0]->{Errors}) {
                    push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('NoRights', iget("Общий счёт %d", $account_ids[$i]));
                } else {
                    $self->{ret}->{ActionsResult}->[0]->{Errors} = [get_error_object('NoRights', iget("Общий счёт %d", $account_ids[$i]))];
                }
            } else {
                $self->{_preprocessed}->{$account_ids[$i]} = $wallet_ids->{$account_ids[$i]};
                push @cluids, $wallet_ids->{$account_ids[$i]}->{uid};
                push @cids, $wallet_ids->{$account_ids[$i]}->{cid};
            }
        } else {
            if (exists $self->{ret}->{ActionsResult}->[0]->{Errors}) {
                push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('NoRights', iget("Общий счёт %d", $account_ids[$i]));
            } else {
                $self->{ret}->{ActionsResult}->[0]->{Errors} = [get_error_object('NoRights', iget("Общий счёт %d", $account_ids[$i]))];
            }
        }
    }

    $self->{cluid} = [uniq @cluids] if (@cluids);
    $self->{syslog_data}->{cid} = [uniq @cids] if (@cids);

    return;
}

=head2 _check_transfer_rights(self, params)

    Проверки прав (избыточные)
    self->{ret}->{ActionsResult}->[0]->{Errors} заполняется ошибками в случае ошибок
    Не обрабатывает элементы, содержащие ошибки
    Эти проверки также будет вызваны в &RBACDirect::rbac_cmd_transfer и в &MoneyTransfer::validate_transfer_money

=cut

sub _check_transfer_rights($$) {
    my ($self, $params) = @_;
    return if exists $self->{ret}->{ActionsResult}->[0]->{Errors};

    if ($self->{rbac_login_rights}->{role} ne 'agency' && $self->{rbac_login_rights}->{role} ne 'super') {
        push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('NoRights', iget("Операция переноса денег c общего счёта доступна только агентствам"));
        return;
    }

    foreach my $account_id(keys %{$self->{_preprocessed}}) {
        my $error_text;

        unless (rbac_is_owner_of_camp($self->{rbac}, $self->{_preprocessed}->{$account_id}->{uid}, $account_id)
        && rbac_is_owner($self->{rbac}, $self->{uid}, $self->{_preprocessed}->{$account_id}->{uid})) {
            $error_text = iget("Общий счёт %d", $account_id);
        } elsif (!rbac_user_allow_edit_camp($self->{rbac}, $self->{uid}, $account_id) ) {
            $error_text = iget("Общий счёт %d", $account_id);
        } else {
            my $client_id = get_clientid(cid => $account_id);
            if (Client::is_client_converting_soon($client_id)) {
                $error_text = APICommon::msg_converting_in_progress;
            } elsif (Client::client_must_convert($client_id)) {
                $error_text = APICommon::msg_must_convert;
            }
        }

        if (defined $error_text) {
            push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('NoRights', $error_text);
        }
    }

    return;
}

=head2 _validate_transfer_currencies(self, params)

    Проверка на то, что запрос указан либо в допустимой валюте, либо без неё
    self->{ret}->{ActionsResult}->[0]->{Errors} заполняется ошибками в случае ошибок
    Не обрабатывает элементы, содержащие ошибки

=cut

sub _validate_transfer_currencies($$) {
    my ($self, $params) = @_;
    return if exists $self->{ret}->{ActionsResult}->[0]->{Errors};
    my $request_currency = $params->{Transfers}->[0]->{Currency} || 'YND_FIXED';
    if ($request_currency eq 'YND_FIXED') {
        $self->{ret}->{ActionsResult}->[0]->{Errors} //= [];
        push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('BadCurrency');
        return;
    }

    my $camps_currency;
    foreach my $account_id(keys %{$self->{_preprocessed}}) {
        my $camp_currency = $self->{_preprocessed}->{$account_id}->{currency};
        if ($camp_currency ne $request_currency) {
            if (exists $self->{ret}->{ActionsResult}->[0]->{Errors}) {
                push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('BadCurrency', iget("Общий счёт %d", $account_id));
            } else {
                $self->{ret}->{ActionsResult}->[0]->{Errors} = [get_error_object('BadCurrency', iget("Общий счёт %d", $account_id))];
            }
        }
        if($camps_currency && $camps_currency ne $camp_currency) {
            if (exists $self->{ret}->{ActionsResult}->[0]->{Errors}) {
                push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('BadCurrency', iget('Запрещен перенос денег между общими счетами в разных валютах'));
            } else {
                $self->{ret}->{ActionsResult}->[0]->{Errors} = [get_error_object('BadCurrency', iget('Запрещен перенос денег между общими счетами в разных валютах'))];
            }
        } else {
            $camps_currency = $camp_currency;
        }
    }
    return;
}

=head2 _check_wallets(self, params)

    Проверка включенности кошельков
    self->{ret}->{ActionsResult}->[0]->{Errors} заполняется ошибками в случае ошибок
    Не обрабатывает элементы, содержащие ошибки

=cut

sub _check_wallets($$) {
    my ($self, $params) = @_;
    my $c = DirectContext->new({
        is_direct        => 1,# Директ/Баян
        UID              => $self->{uid},
        uid              => undef,
        client_chief_uid => undef,
        rbac             => $self->{rbac},
        rights           => $self->{rbac_rights},
        login_rights     => $self->{rbac_login_rights},
    });

    if ($params->{Action} ne 'TransferMoney') {
        my @account_ids = map {$_->{AccountID}} @{$params->{Payments}};
        my $wallets = Wallet::get_wallets_by_cids(\@account_ids, $c);
        foreach my $i(0 .. $#account_ids) {
            next if exists $self->{ret}->{ActionsResult}->[$i]->{Errors};
            unless ($wallets->{$account_ids[$i]}->{wallet_camp} && $wallets->{$account_ids[$i]}->{wallet_camp}->{enabled}) {
                push @{$self->{ret}->{ActionsResult}->[$i]->{Errors}}, get_error_object('NoWallet', iget('Операции с общим счётом %s недоступны', $account_ids[$i]));
            }
        }
    } else { # TransferMoney
        my $wallets = Wallet::get_wallets_by_cids([keys %{$self->{_preprocessed}}], $c);
        return if exists $self->{ret}->{ActionsResult}->[0]->{Errors};
        foreach my $account_id(keys %{$self->{_preprocessed}}) {
            unless ($wallets->{$account_id}->{wallet_camp} && $wallets->{$account_id}->{wallet_camp}->{enabled}) {
                push @{$self->{ret}->{ActionsResult}->[0]->{Errors}}, get_error_object('NoWallet', iget('Операции с общим счётом %s недоступны', $account_id));
            }
        }
    }

    return;
}

1;
