package API::Methods::AdImage;

# $Id$

=head1 NAME
    
    API::Methods::AdImage

=head1 DESCRIPTION

    Методы для работы с картинкой в API

=cut

use Direct::Modern;

use List::MoreUtils qw(any none uniq);
use MIME::Base64;

use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::DBTools;
use Yandex::DBShards qw/SHARD_IDS/;

use Settings;

use RBACDirect;
use RBAC2::DirectChecks;
use PrimitivesIds;
use EnvTools;
use HttpTools;

use Models::Banner qw/check_banner_ignore_moderate/;
use BannersCommon qw/get_banners_short_hash/;
use BannerImages;
use BannerImages::Pool;
use BannerImages::Queue;
use Campaign::Types;
use Direct::Validation::Image;
use Client qw/mass_is_client_converting_soon mass_client_must_convert/;

use API::Filter;
use API::Preprocess;
use API::Errors;
use API::Validate::Ids qw/validate_ids_detail/;
use APICommon;
use API::Settings qw//;

my $AD_IMAGE_ASSOC_GET_LIMIT = 10000;

sub AdImage {
    my ($self, $params) = @_;
    my $result = {};
    my $hostname;
    if (!is_production()) {
        $hostname = http_server_host($self->{plack_request}, no_port => 1);
        $hostname =~ s/^api\.//;
    }
    if ($params->{Action} eq 'Get') {
        my $clientid2logins = $self->{preprocess}{clientid2logins};
        my @filtered_data;

        for my $item (@{$self->{preprocess}{pool}}) {
            $item->{url} = BannerImages::get_image_url($item, {host => $hostname});
            if ($self->{rbac_login_rights}{role} eq 'client') {
                $item->{login} = $self->{user_info}{login};
                push @filtered_data, filter_adimage_pool_object($item);
            } else {
                for my $login (@{$clientid2logins->{$item->{ClientID}}}) {
                    $item->{login} = $login;
                    push @filtered_data, filter_adimage_pool_object($item);
                }
            }
        }
        $result = {AdImages => \@filtered_data, TotalObjectsCount => $self->{preprocess}{total_images_in_pool}};
        
    } elsif ($params->{Action} eq 'Delete') {        
        my $imp_ids = [map {$_->{imp_id}} @{$self->{preprocess}{pool}}];
        BannerImages::Pool::delete_items($imp_ids);
        
    } elsif ($params->{Action} eq 'GetLimits') {        
        my $clientid2logins = $self->{preprocess}{clientid2logins};

        my $limits = $self->{preprocess}{limits};

        my @filtered_data;
        for my $item (@$limits) {
            if ($self->{rbac_login_rights}{role} eq 'client') {
                $item->{login} = $self->{user_info}{login};
                push @filtered_data, filter_adimage_limits_object($item);
            } else {
                for my $login (@{$clientid2logins->{$item->{ClientID}}}) {
                    $item->{login} = $login;
                    push @filtered_data, filter_adimage_limits_object($item);
                }
            }
        }

        $result = {AdImageLimits => \@filtered_data};

    } elsif ($params->{Action} eq 'UploadRawData') {
        for (my $i = 0; $i < scalar @{$params->{AdImageRawData}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            
            my $item = $params->{AdImageRawData}[$i];
            
            my $clientid = $self->{user_info}{ClientID};
            if ($self->{rbac_login_rights}{role} ne 'client') {
                $clientid = $self->{preprocess}{login2clientid}{$item->{Login}};
            }
            
            my $fname = $item->{Name};
            my $img_data = banner_prepare_save_image($self->{preprocess}{images}[$i], { ClientID => $clientid });

            my $bipid;
            if ($img_data->{error}) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('BadImage', iget($img_data->{error}));
            } else {
                my $pool = BannerImages::Pool::add_items([{ ClientID => $clientid, image_hash => $img_data->{md5}, name => $fname }])->[0];
                $bipid = $pool->{imp_id};
            }
            if ($bipid) {
                $self->{ret}[$i]{AdImageHash} = $img_data->{md5};
            }
        }
        $result = {ActionsResult => $self->{ret}};

    } elsif ($params->{Action} eq 'Upload') {
        my @image_urls;
        for (my $i = 0; $i < scalar @{$params->{AdImageURLData}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $item = $params->{AdImageURLData}[$i];
            my $clientid = $self->{user_info}{ClientID};
            if ($self->{rbac_login_rights}{role} ne 'client') {
                $clientid = $self->{preprocess}{login2clientid}{$item->{Login}};
            }
            my $fname = $item->{Name};
            push @image_urls, { ClientID => $clientid, url => $item->{URL}, name => $fname };
        }
        my $job_ids = BannerImages::Queue::add_items(\@image_urls, UID => $self->{uid});
        for (my $i = 0; $i < scalar @{$params->{AdImageURLData}}; $i++) {
            $self->{ret}[$i]{AdImageUploadTaskID} = $job_ids->[$i] % $API::Settings::API4_JOB_ID_THRESHOLD if ($job_ids->[$i]);
        }
        $result = {ActionsResult => $self->{ret}};
    } elsif ($params->{Action} eq 'CheckUploadStatus') {
        my $clientids = [uniq grep {$_} values %{$self->{preprocess}{login2clientid} || {}}];
        my $task_ids = $params->{SelectionCriteria}{AdImageUploadTaskIDS};
        my %orig_task_ids;
        $task_ids = undef if $task_ids && !@$task_ids;
        for my $task_id (@{$task_ids // []}) {
            my $orig_task_id = $task_id;
            $task_id %= $API::Settings::API4_JOB_ID_THRESHOLD;
            $task_id += $API::Settings::API4_JOB_ID_SHIFT if !is_sandbox();
            $orig_task_ids{$task_id} = $orig_task_id;
        }
        my ($selected_tasks_num, $tasks) = BannerImages::Queue::get_items_status(
                $self->{uid},
                $clientids, 
                $task_ids,
                limit => $params->{SelectionCriteria}->{Limit} || $AD_IMAGE_ASSOC_GET_LIMIT,
                offset => $params->{SelectionCriteria}->{Offset} || 0,
                role => $self->{rbac_login_rights}{role},
            );

        my $clientid2logins = $self->{preprocess}{clientid2logins};
        my @filtered_data;
        for my $item (@$tasks) {
            if ($orig_task_ids{$item->{job_id}}) {
                $item->{job_id} = $orig_task_ids{$item->{job_id}};
            } else {
                $item->{job_id} %= $API::Settings::API4_JOB_ID_THRESHOLD;
            }

            if ($item->{image_hash}) {
                $item->{url} = BannerImages::get_image_url($item, {host => $hostname});
            }
            if ($item->{error}) {
                $item->{error_obj} = get_error_object('BadImage', iget($item->{error}));
            }
            if ($self->{rbac_login_rights}{role} eq 'client') {
                $item->{login} = $self->{user_info}{login};
                push @filtered_data, filter_adimage_queue_object($item);
            } else {
                for my $login (@{$clientid2logins->{$item->{ClientID}}}) {
                    $item->{login} = $login;
                    push @filtered_data, filter_adimage_queue_object($item);
                }
            }
        }
        $result = {AdImageUploads => \@filtered_data, TotalObjectsCount => $selected_tasks_num};
    }

    return $result;
}

sub preprocess_adimage {
    my ($self, $params) = @_;

    # если нужно препроцессить логины, заполняется
    my $logins;
    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete' || $params->{Action} eq 'CheckUploadStatus' || $params->{Action} eq 'GetLimits') {
        if ($self->{rbac_login_rights}{role} ne 'client') {
            $logins = $params->{SelectionCriteria}{Logins} || [];
        } elsif ($self->{rbac_login_rights}{role} eq 'client') {
            $logins = [$self->{user_info}{login}];
            #если клиент очищаем Logins чтоб дальше по коду не мешались
            $params->{SelectionCriteria}{Logins} = [];
        }
    } elsif ($params->{Action} eq 'UploadRawData') {
        if ($self->{rbac_login_rights}{role} ne 'client') {
            $logins = [uniq grep {$_} map {$_->{Login}} @{$params->{AdImageRawData}}];
        } else {
            $logins = [$self->{user_info}{login}];
        }

        for (my $i = 0; $i < scalar @{$params->{AdImageRawData}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};

            my $item = $params->{AdImageRawData}[$i];
            $self->{preprocess}{images}[$i] = MIME::Base64::decode_base64($item->{RawData});
            $self->{preprocess}{image_obj}[$i] = banner_check_image($self->{preprocess}{images}[$i]);
        }
    } elsif ($params->{Action} eq 'Upload') {
        if ($self->{rbac_login_rights}{role} ne 'client') {
            $logins = [uniq grep {$_} map {$_->{Login}} @{$params->{AdImageURLData}}];
        } else {
            $logins = [$self->{user_info}{login}];
            
        }
    }
    if (@$logins) {
        $self->{preprocess} = hash_merge $self->{preprocess}, 
                                        preprocess_logins(
                                            $self, 
                                            $logins,
                                            clientid2chiefuid => 1, 
                                            chiefuid2login => 1,
                                            login2uid => 1, 
                                            uid2role => 1,
                                        );
    }
    my $p = $self->{preprocess};
    my $clientids = scalar keys %{$p->{login2clientid}} ? [
            uniq 
            map {$ p->{login2clientid}{$_} } 
            grep {$_ && $p->{uid2role}{$p->{login2uid}{$_}} eq 'client'} 
            keys %{ $p->{login2clientid} }
            ] : [];
    return ('BadLogin', iget('Не передано ни одного валидного логина'))
        unless scalar @$clientids;

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        my ($total, $pool) = (0, []);

        my $image_hashes = scalar @{ $params->{SelectionCriteria}{AdImageHashes} || [] } ? $params->{SelectionCriteria}{AdImageHashes} : undef;

        if (scalar @$clientids) {
            my %limit_offset = ();
            my %assigned_filter = ();
            if ($params->{Action} eq 'Get') {
                my $assigned_hash = {map {$_ => 1} @{$params->{SelectionCriteria}{Assigned} || []}};
                if ($assigned_hash->{'Yes'} && ! $assigned_hash->{'No'}) {
                    $assigned_filter{assigned} = 1;
                } elsif ($assigned_hash->{'No'} && ! $assigned_hash->{'Yes'}) {
                    $assigned_filter{assigned} = 0;
                }

                %limit_offset = (
                    limit => $params->{SelectionCriteria}->{Limit} || $AD_IMAGE_ASSOC_GET_LIMIT,
                    offset => $params->{SelectionCriteria}->{Offset} || 0,
                );
            }

            ($total, $pool) = BannerImages::Pool::get_items($clientids, $image_hashes,
                            with_assigned_info => 1, 
                            clientid2chiefuid => $self->{preprocess}{clientid2chiefuid},
                            exclude_types => ['image_ad'],
                            %assigned_filter,
                            %limit_offset,
                            );
        }
        $self->{preprocess}{total_images_in_pool} = $total;
        $self->{preprocess}{pool} = $pool;
    }

    if ($params->{Action} eq 'Upload' || $params->{Action} eq 'UploadRawData' || $params->{Action} eq 'GetLimits') {
        $self->{preprocess}{limits} = BannerImages::Pool::get_users_pool_limits($self->{preprocess}{clientid2chiefuid});

        if ($params->{Action} eq 'Upload' || $params->{Action} eq 'UploadRawData') {
            my $limits_by_clientid;
            for my $item (@{$self->{preprocess}{limits}}) {
                $limits_by_clientid->{$item->{ClientID}} = $item;
            }
            $self->{preprocess}{limits_by_clientid} = $limits_by_clientid;
        }
    }

    return;
}

sub validate_adimage_rights {
    my ($self, $params) = @_;

    if ($params->{Action} =~ /^(Get|Delete|CheckUploadStatus|GetLimits)$/) {
        my $uniqed_uids = [uniq values %{$self->{preprocess}{clientid2chiefuid} || {}}];

        if (scalar @$uniqed_uids && !rbac_mass_is_owner($self->{rbac}, $self->{uid}, $uniqed_uids)) {
            return ('NoRights');
        }

        my @client_ids = uniq keys %{$self->{preprocess}{clientid2chiefuid} || {}};
        dieSOAP('NoRights', APICommon::msg_converting_in_progress) if Client::is_any_client_converting_soon(\@client_ids);
        dieSOAP('NoRights', APICommon::msg_must_convert) if Client::is_any_client_must_convert(\@client_ids);
    }

    if ($params->{Action} eq 'Delete') {
        if ($self->{rbac_login_rights}{role} =~ m/^(media|superreader)$/) {
            return ('NoRights');
        }
    } elsif ($params->{Action} eq 'Upload' || $params->{Action} eq 'UploadRawData') {
        my @items_to_iterate;
        if ($params->{Action} eq 'Upload') {
            @items_to_iterate = @{$params->{AdImageURLData}};
        } elsif ($params->{Action} eq 'UploadRawData') {
            @items_to_iterate = @{$params->{AdImageRawData}};
        }

        if ($self->{rbac_login_rights}{role} ne 'client') {

            my @uids = uniq values %{$self->{preprocess}{login2uid}};
            my $isowner;
            if (@uids) {
               $isowner = RBACDirect::rbac_is_owner_of_users($self->{uid}, \@uids);
            }
            
            my @clientids = uniq values %{$self->{preprocess}{login2clientid}};

            dieSOAP('NoRights', APICommon::msg_converting_in_progress) if Client::is_any_client_converting_soon(\@clientids);
            dieSOAP('NoRights', APICommon::msg_must_convert) if Client::is_any_client_must_convert(\@clientids);

            for (my $i=0; $i<scalar @items_to_iterate; $i++) {
                next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};

                my $item = $items_to_iterate[$i];

                my $clientid = $self->{preprocess}{login2clientid}{$item->{Login}};
                my $uid = $self->{preprocess}{login2uid}{$item->{Login}};

                if ($self->{preprocess}{uid2role}{$uid} ne 'client') {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights', iget("Пользователь должен быть клиентом Яндекс.Директа"));
                } elsif ($self->{rbac_login_rights}{role} =~ m/^(media|superreader)$/) {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights');
                } elsif ($self->{rbac_login_rights}{role} !~ m/^(super|support|placer)$/ && ! $isowner->{$uid}) {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights', iget("Нет прав на данного клиента"));
                }
            }
        }

        for (my $i=0; $i<scalar @items_to_iterate; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $item = $items_to_iterate[$i];
            
            my $clientid;

            # если клиент, то всегда берем его ClientID, в противном случае из запроса
            if ($self->{rbac_login_rights}{role} ne 'client') {
                $clientid = $self->{preprocess}{login2clientid}{$item->{Login}};
            } else {
                $clientid = $self->{preprocess}{login2clientid}{$self->{user_info}{login}};
            }
            
            my $limits = $self->{preprocess}{limits_by_clientid}{$clientid};
            $limits->{cnt}++;

            if ($limits->{cnt} > $limits->{total}) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('ImagePoolExceeded');
            }
        }
    }

    return;
}

sub validate_adimage {
    my ($self, $params) = @_;
    if (exists $params->{SelectionCriteria}) {
        my @logins = keys %{$self->{preprocess}{login2clientid}};
        for my $login (@logins){
            return ('BadLogin', iget($login))
                unless $self->{preprocess}{login2clientid}{$login};
        }
    }
    if ($params->{Action} eq 'Delete') {
        for my $item (@{$self->{preprocess}{pool}}) {
            if ($item->{assigned}) {
                return ('CannotDeleteObject', iget('Невозможно удалить привязанные картинки'));
            }
        }

        my @uniqed_hashes = uniq @{$params->{SelectionCriteria}{AdImageHashes} || []};

        if (scalar @uniqed_hashes != scalar @{$self->{preprocess}{pool}}) {
            return ('BadImageHash', iget('Неверный AdImageHash'));
        }
    } elsif ($params->{Action} eq 'UploadRawData') {
        for (my $i=0; $i<scalar @{$params->{AdImageRawData}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}} || !$self->{preprocess}{image_obj}[$i];
            if ($self->{preprocess}{image_obj}[$i]{error}) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('BadImage', $self->{preprocess}{image_obj}[$i]{error});
            } else {
                if ($self->{preprocess}{image_obj}[$i]{size} > $BannerImages::MAX_IMAGE_FILE_SIZE) {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('BadImage', iget("Размер изображения больше допустимого (%d)", $BannerImages::MAX_IMAGE_FILE_SIZE));
                }
                my $dimensions = $self->{preprocess}{image_obj}[$i];
                if (Direct::Validation::Image::is_image_size_valid_for_image_ad($dimensions->{width}, $dimensions->{height}) ||
                    !banner_image_check_size($dimensions->{width}, $dimensions->{height}))
                {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('BadImage', iget('Размер изображения некорректен'));
                }
            }
        }
    }

    if ($params->{Action} eq 'UploadRawData' || $params->{Action} eq 'Upload') {
        my @items_to_iterate;
        if ($params->{Action} eq 'Upload') {
            @items_to_iterate = @{$params->{AdImageURLData}};
        } elsif ($params->{Action} eq 'UploadRawData') {
            @items_to_iterate = @{$params->{AdImageRawData}};
        }

        for (my $i=0; $i<scalar @items_to_iterate; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $item = $items_to_iterate[$i];
            
            my $clientid;

            # если клиент, то всегда берем его ClientID, в противном случае из запроса
            if ($self->{rbac_login_rights}{role} ne 'client') {
                $clientid = $self->{preprocess}{login2clientid}{$item->{Login}};
            } else {
                $clientid = $self->{preprocess}{login2clientid}{$self->{user_info}{login}};
            }

            my $chiefuid = $self->{preprocess}{clientid2chiefuid}{$clientid};
            $self->{uhost}->reserve_units_by_type($chiefuid, 'load_image', 1);

            if (!$self->{uhost}->have_enough_user_units($chiefuid, allow_monitor_fault => 1)) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('NotEnoughUnits');
                $self->{uhost}->unreserve_units_by_type($chiefuid, 'load_image', 1);
            }
        }
    }
    return;
}


sub AdImageAssociation {
    my ($self, $params) = @_;

    my $result;

    if ($params->{Action} eq 'Get') {
        my $sc = $params->{SelectionCriteria};

        my %where;
        $where{uid} = scalar values %{$self->{preprocess}{clientid2chiefuid}} ? [uniq values %{$self->{preprocess}{clientid2chiefuid}}] : undef;
        $where{bid} = $sc->{AdIDS} if defined $sc->{AdIDS} && scalar @{$sc->{AdIDS}};
        $where{cid} = $sc->{CampaignIDS} if defined $sc->{CampaignIDS} && scalar @{$sc->{CampaignIDS}};
        $where{image_hash} = $sc->{AdImageHashes} if defined $sc->{AdImageHashes} && scalar @{$sc->{AdImageHashes}};
        $where{statusModerate} = APICommon::status_moderate_ext_to_int($sc->{StatusAdImageModerate}) if defined $sc->{StatusAdImageModerate} && scalar @{$sc->{StatusAdImageModerate}};
        my ($selected_banners_num, $data) = BannerImages::get_banners_images(
                %where, 
                limit => $sc->{Limit} || $AD_IMAGE_ASSOC_GET_LIMIT,
                offset => $sc->{Offset} || 0);

        my @filtered_data;
        if (scalar @$data) {
            my @cids = uniq map {$_->{cid}} @$data; 

            # TODO: сделать универсальную процедуру
            my $cid2logins;
            my $cluids;
            if ($self->{rbac_login_rights}{role} ne 'client') {
                my $cid2uid = get_cid2uid(cid => \@cids);
                $cluids = [uniq values %$cid2uid];

                if (! scalar @{$params->{SelectionCriteria}{Logins} || []}) {
                    my $uid2login = get_uid2login(uid => [values %$cid2uid]);
                    for my $cid (@cids) {
                        push @{$cid2logins->{$cid}}, $uid2login->{$cid2uid->{$cid}};
                    }
                } else {
                    my $uid2clientid = get_uid2clientid(uid => [values %$cid2uid]);

                    for my $cid (@cids) {
                        for my $login (@{$self->{preprocess}{clientid2logins}{$uid2clientid->{$cid2uid->{$cid}}}}) {
                            push @{$cid2logins->{$cid}}, $login;
                        }
                    }
                }
            } else {
                $cluids = [$self->{uid}];
            }

            for my $item (@$data) {
                $item->{StatusAdImageModerate} = $item->{statusModerate};
                $item->{BannerID} = $item->{bid};
                $item->{image_statusModerate} = $item->{statusModerate};
                APICommon::get_modreasons_into_banners([$item], image => 1);

                if ($self->{rbac_login_rights}{role} eq 'client') {
                    $item->{login} = $self->{user_info}{login};
                    push @filtered_data, filter_adimage_assoc_object($item);
                } else {
                    for my $login (@{$cid2logins->{$item->{cid}}}) {
                        $item->{login} = $login;
                        push @filtered_data, filter_adimage_assoc_object($item);
                    }
                }
            }

            $self->{syslog_data}->{cid} = \@cids;
            $self->{cluid} = $cluids if defined $cluids;
        }
        $result = {AdImageAssociations => \@filtered_data, TotalObjectsCount => $selected_banners_num};
    } elsif ($params->{Action} eq 'Set') {
        my $old_images = get_banners_images(bid => $self->{preprocess}{bids});
        my $old_banners_hash = $self->{preprocess}{old_banners_hash};
        my %old_images_hash = map {$_->{bid} => $_} @$old_images;
        my @images_to_save;
        my @image_bids_to_delete;

        my $image_type_by_hash = get_hash_sql(PPC(bid => $self->{preprocess}{bids}), [
            q{SELECT image_hash, image_type FROM banner_images_formats},
            where => {
                Image_hash => [ map { $_->{AdImageHash} } @{ $params->{AdImageAssociations} } ]
            }
        ]);

        for (my $i=0; $i<scalar @{$params->{AdImageAssociations} || []}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $image_hash = $params->{AdImageAssociations}[$i]{AdImageHash};
            my $bid = $params->{AdImageAssociations}[$i]{AdID};
            if ($old_images_hash{$bid}{image_hash} && ! $image_hash) {
                push @image_bids_to_delete, $params->{AdImageAssociations}[$i]{AdID};
            } elsif ($image_hash &&
                        (! $old_images_hash{$bid}
                         || ! defined $old_images_hash{$bid}{image_hash}
                         || $old_images_hash{$bid}{image_hash} ne $image_hash
                        )
                    ) {
                if (!exists $image_type_by_hash->{$image_hash} ||
                    $image_type_by_hash->{$image_hash} eq 'small' ||
                    $image_type_by_hash->{$image_hash} eq 'image_ad')
                {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('BadImage', iget('Размер изображения некорректен'));
                    next;
                }
                my $ignore_moderate = check_banner_ignore_moderate({ 
                        statusModerate => $old_banners_hash->{$bid}{statusModerate},
                        statusEmpty => $old_banners_hash->{$bid}{statusEmpty},
                    });

                my $statusAdImageModerate = $ignore_moderate ? 'New' : 'Ready';
                push @images_to_save, {
                    bid => $bid,
                    image_hash => $image_hash,
                    statusModerate => $statusAdImageModerate,
                };
            }

            $self->{ret}[$i]->{AdID} = $params->{AdImageAssociations}[$i]{AdID};
        }
        mass_banner_assign_image(@images_to_save);
        mass_banner_remove_image(\@image_bids_to_delete) if @image_bids_to_delete;
        $result = {ActionsResult => $self->{ret}};
    }

    return $result;
}

sub preprocess_adimageassociation {
    my ($self, $params) = @_;
    if ($params->{Action} eq 'Get') {
        my $logins = [$self->{user_info}{login}];
        if ($self->{rbac_login_rights}{role} ne 'client' && scalar @{$params->{SelectionCriteria}{Logins} || []}) {
            $logins = $params->{SelectionCriteria}{Logins} || [],
        }
        $self->{preprocess} = hash_merge {}, 
                                            preprocess_logins(
                                                $self, 
                                                $logins,
                                                clientid2chiefuid => 1, 
                                                chiefuid2login => 1
                                            );
        
        my $clientids = [uniq values %{$self->{preprocess}{login2clientid}}];
        return ('BadLogin', iget('Неверно задан логин владельца изображения'))
            unless scalar @$clientids;
        if (scalar @{$params->{SelectionCriteria}{CampaignIDS} || []}) {
            $self->{preprocess}{cids} = $params->{SelectionCriteria}{CampaignIDS}
        } elsif (scalar @{$params->{SelectionCriteria}{AdIDS} || []}) {
            $self->{preprocess}{cids} = get_cids(bid => $params->{SelectionCriteria}{AdIDS});
        }
    } elsif ($params->{Action} eq 'Set') {
        my @bids;
        for (my $i=0; $i<scalar @{$params->{AdImageAssociations} || []}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            push @bids, $params->{AdImageAssociations}[$i]{AdID} if $params->{AdImageAssociations}[$i]{AdID};
        }
        @bids = uniq @bids;
        $self->{preprocess}{bids} = \@bids;
        $self->{preprocess}{bid2cid} = get_bid2cid(bid => \@bids);
        $self->{preprocess}{old_banners_hash} = get_banners_short_hash($self->{preprocess}{bids}, ['statusModerate'], ['statusEmpty', 'uid']);
        $self->{syslog_data}->{cid} = [uniq values %{$self->{preprocess}->{bid2cid}}];
        $self->{cluid} = [uniq map {$_->{uid}} values %{$self->{preprocess}->{old_banners_hash}}];
    }

    return;
}

sub validate_adimageassociation_rights {
    my ($self, $params) = @_;

    if ($params->{Action} eq 'Get') {
        my $uniqed_uids = [uniq values %{$self->{preprocess}{clientid2chiefuid} || {}}];

        if (! scalar @$uniqed_uids) {
            return ('BadLogin', iget('Неверно задан логин владельца изображения'));
        }

        if (scalar @$uniqed_uids && !rbac_mass_is_owner($self->{rbac}, $self->{uid}, $uniqed_uids)) {
            return ('NoRights');
        }
        
        my @preprocess_cids = uniq grep {$_} @{$self->{preprocess}{cids} || []};
        
        if (scalar @preprocess_cids) {
            my $types = get_hash_sql(PPC(cid => \@preprocess_cids), ["select cid, type from campaigns"
                                            , where => {
                                                cid => SHARD_IDS
                                                , statusEmpty => 'No'
                                                , type__not_in => ['wallet']
                                            } ] );

            foreach my $cid (@preprocess_cids) {
                unless ($types->{$cid}) {
                    return ('NoRights');
                }
            }

            if (RBAC2::DirectChecks::rbac_cmd_user_allow_show_camps($self->{rbac}, {cid => \@preprocess_cids, UID => $self->{uid}})) {
                return ('NoRights');
            }

            foreach my $cid (@preprocess_cids) {
                unless (camp_kind_in(type => $types->{$cid}, 'api_edit_geo')) {
                    return ('NotSupported');
                }
                if ($types->{$cid} eq 'geo' && !(APICommon::is_api_geo_allowed($self) eq 'Yes')) {
                    return ('NotSupported');
                }
            }
        }

        my $banner_type_validation_result = validate_ids_detail( { bids => $params->{SelectionCriteria}{AdIDS}, banner_types => ['text'] } );
        if ( $banner_type_validation_result->{not_supported} && @{ $banner_type_validation_result->{not_supported} } ) {
            return ('NotSupported', iget('Тип объявления не поддерживается. Объявления: %s', join(', ', @{$banner_type_validation_result->{not_supported}})));
        }

    } elsif ($params->{Action} eq 'Set') {
        my $bid2cid = $self->{preprocess}{bid2cid};
        my @cids = uniq values %{$self->{preprocess}{bid2cid}};
        my $camp_rights = @cids ? rbac_user_allow_edit_camps_detail($self->{rbac}, $self->{uid}, \@cids) : {};
        my $cid2uid = get_cid2uid(cid => \@cids);
        my $uid2clientid = get_uid2clientid(uid => [values %$cid2uid]);
        my %cid2clientid = map { $_ => $uid2clientid->{ $cid2uid->{$_} } } @cids;
        my @client_ids = uniq values %$uid2clientid;
        my $client_id_convert_soon = mass_is_client_converting_soon(\@client_ids);
        my $client_id_must_convert = mass_client_must_convert(\@client_ids);
        my $types = get_hash_sql(PPC(cid => \@cids), ["select cid, type from campaigns", where => {cid => SHARD_IDS, statusEmpty => 'No'}]);

        # для каждого bid в запросе неправильного типа: bid => 1
        my %ad_type_not_supported;

        my $banner_type_validation_result = validate_ids_detail( { bids => $self->{preprocess}{bids}, banner_types => ['text'] } );
        if ( $banner_type_validation_result->{not_supported} && @{ $banner_type_validation_result->{not_supported} } ) {
            for my $bid ( @{$banner_type_validation_result->{not_supported}} ) {
                $ad_type_not_supported{$bid} = 1;
            }
        }

        my @image_hashes;
        for (my $i=0; $i<scalar @{$params->{AdImageAssociations} || []}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            push @image_hashes, $params->{AdImageAssociations}[$i]{AdImageHash} if $params->{AdImageAssociations}[$i]{AdImageHash};
        }
        @image_hashes = uniq @image_hashes;

        my ($total, $pool) = BannerImages::Pool::get_items([values %$uid2clientid], \@image_hashes);

        my $pool_hash;
        for my $pool_item (@$pool) {
            $pool_hash->{$pool_item->{ClientID}}{$pool_item->{image_hash}} = $pool_item;
        }

        my $old_banners_hash = $self->{preprocess}{old_banners_hash};
        my $is_api_geo_allowed = (APICommon::is_api_geo_allowed($self) eq 'Yes');
        for (my $i = 0; $i < scalar @{$params->{AdImageAssociations} || []}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};

            my $cid = $bid2cid->{$params->{AdImageAssociations}[$i]{AdID}};

            if (!$cid || !$camp_rights->{$cid} || $self->{rbac_login_rights}{role} eq 'media') {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights');
            } elsif ($client_id_convert_soon->{$cid2clientid{$cid}}) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights', APICommon::msg_converting_in_progress);
            } elsif ($client_id_must_convert->{$cid2clientid{$cid}}) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights', APICommon::msg_must_convert);
            } elsif (!$types->{$cid} || !camp_kind_in(type => $types->{$cid}, 'api_edit_geo')
                || ($types->{$cid} eq 'geo' && !$is_api_geo_allowed)) {
                push @{$self->{ret}->[$i]->{Errors}}, get_error_object('NotSupported');
            } elsif ( $ad_type_not_supported{$params->{AdImageAssociations}[$i]{AdID}} ) {
                push @{$self->{ret}->[$i]->{Errors}}, get_error_object('NotSupported', iget('Тип объявления не поддерживается'));
            } else {
                my $clientid = $uid2clientid->{$cid2uid->{$cid}};
                if ($params->{AdImageAssociations}[$i]{AdImageHash} && 
                    (! exists $pool_hash->{$clientid} || ! exists $pool_hash->{$clientid}{$params->{AdImageAssociations}[$i]{AdImageHash}})) {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('BadImageHash', iget("Указанное изображение не найдено"));
                }
            }
        }

        for (my $i=0; $i<scalar @{$params->{AdImageAssociations} || []}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $item = $params->{AdImageAssociations}[$i];
            if ($old_banners_hash->{$item->{AdID}}{statusModerate} ne 'New') {
                my $type;
                if (! $item->{AdImageHash}) {
                    $type = 'image_set_remove';
                } else {
                    $type = 'image_set_update';
                }

                $self->{uhost}->reserve_units_by_type($old_banners_hash->{$item->{AdID}}{uid}, $type, 1);
                if (!$self->{uhost}->have_enough_user_units($old_banners_hash->{$item->{AdID}}{uid}, allow_monitor_fault => 1)) {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('NotEnoughUnits');
                    $self->{uhost}->unreserve_units_by_type($old_banners_hash->{$item->{AdID}}{uid}, $type, 1);
                }
            }
        }
    }
    return;
}

1;
