package API::Methods::Clients;

=pod

    $Id$

=head1 NAME

    API::Methods::Clients

=head1 DESCRIPTION

    Методы API Директа для работы с клиентами и списками клиентов

=cut

use strict;
use warnings;

use API::Settings;
use API::Errors;
use APICommon qw(:subs);
use APIUnits;
use API::ClientOptions;

use API::Limits qw/has_spec_limits get_spec_limit/;

use Yandex::I18n;
use TextTools;
use HashingTools;
use Settings;
use Primitives;
use PrimitivesIds;
use RBACElementary;
use RBACDirect;

use User;
use Staff;
use Client;
use ServicedClient;

use Yandex::Passport;
use Yandex::TVM2;
use Yandex::Balance;
use Yandex::SendMail qw(send_alert);
use Direct::Validation::Client;

use List::MoreUtils qw/any uniq/;
use JSON;

use utf8;

=head2 CreateNewSubclient

    Создает субклиента для агенства в паспорте, директе и балансе

    Входные данные:
        {
            Login => 'john_smith', # обязательный параметр
            AgencyLogin => 'my_agency', # обязательный параметр только для менеджера
            Name => 'John',
            Surname => 'Smith',
        }

    Выходные данные:
        {
            Login => 'john_smith',
            Password => 'sdfr54h',
            FIO => 'John Smith',
            Email => 'john_smith@yandex.ru',
            ClientID => '1234567',
            uid => '123123', # скрытый параметр, возвращается только менеджерским ролям
        }

=cut

sub CreateNewSubclient
{
    my ($self, $params) = @_;

    if (!$self->{rbac_login_rights}->{manager_control} && !$self->{rbac_login_rights}->{agency_control}) {
        dieSOAP('NoRights');
    }

    # создаем ли мы прямого менеджерского клиента, или агентского
    my $create_direct_client = 0;
    my $login_exists = 0;
    if ($self->{rbac_login_rights}->{role} eq 'manager') {
        $create_direct_client = $params->{ServicedClient} && $params->{ServicedClient} eq 'Yes' ? 1 : 0;
        $login_exists = $params->{LoginExists} && $params->{LoginExists} eq 'Yes' ? 1 : 0;
    }

    # api_update_limit вызывается после вызова метода в API::AUTOLOAD
    api_check_limit_by_method( $self, $self->{uid}, 'CreateNewSubclient', 1, obj_name => $self->{operator_login});

    #заменяем точку на дефис в логине
    $params->{Login} = normalize_login($params->{Login});
    my $uid_by_login_passport = get_uid_by_login($params->{Login});
    my $uid_by_login_direct = get_uid(login => $params->{Login});
    if ($self->{rbac_login_rights}->{manager_control} && $login_exists && ! $uid_by_login_passport) {
        dieSOAP('BadLogin', iget('Логин не существует'));
    }

    # если нужно создать логин, проверяем заведен ли пользователь в паспорте
    if ( ! $login_exists && $uid_by_login_passport) {
        dieSOAP('LoginOccupied');
    }

    # генерируем пароль
    my $password = get_random_string(length=> 8, alphabets=>'wWd');

    my $fio = $login_exists && !$params->{Name} ? get_info_by_uid_passport($uid_by_login_passport)->{fio} : join(" ", $params->{Name}, $params->{Surname});

    my $new_login = {};

    my $email = "$params->{Login}\@yandex.ru";

    my $CREATE_UPDATE_USER_OBJECT = { 
        _login      => $params->{Login},
        email       => $email,
        UID         => $self->{uid},
        # -- флаг для /protected/User.pm, где идет попытка заново вызвать create_client_id_association
        # -- по идее - если ClientID уже был создан и существует - значит клиент уже был создан
        # -- если он не был создан - значит 10-ю строками выше - он будет создан, а => см.выше
        not_create_client_id_association => 1,
        hide_market_rating => 0,
        # -- Валидация валюты и костыли про неё есть в Validate.pm
        initial_currency => $params->{Currency},
    };

    unless ($self->{rbac_login_rights}->{manager_control} && $login_exists) {
        my $ticket = eval { Yandex::TVM2::get_ticket($Settings::PASSPORT_TVM2_ID) } or die "Cannot get ticket for $Settings::PASSPORT_TVM2_ID: $@";
        $new_login = Yandex::Passport::create_passport_login($params->{Login}, $params->{Name}, $params->{Surname}, $password, ip => $ENV{'X-Real-IP'} || $ENV{REMOTE_ADDR}, tvm_ticket => $ticket);
    }

    # получаем информацию по новому пользователю
    my $client_uid = $login_exists ? $uid_by_login_passport : $new_login->{uid};

    # если логин существует, проверяем заведен ли пользователь в директе
    if ( $self->{rbac_login_rights}->{manager_control}
        && $login_exists
        && $uid_by_login_direct
        && rbac_who_is($self->{rbac}, $client_uid) ne 'empty'
    ) {
        dieSOAP('LoginOccupied', iget('Логин уже существует в Директе'));
    }

    if ( $self->{rbac_login_rights}->{manager_control}
        && $login_exists
        && $uid_by_login_passport
        || $new_login->{status} eq 'ok'
    ) {
        # получаем uid агенства, если менеджер содает субклиента агенства
        my $operator_uid = ! $create_direct_client && $self->{rbac_login_rights}->{role} ne 'agency' ? get_uid_by_login($params->{AgencyLogin}) : $self->{uid};
        my $AgencyID = $create_direct_client ? undef : get_clientid(uid => $operator_uid);

        # создаём клента в Балансе и привязку, если надо
        my $ClientID = $login_exists ? get_clientid_by_uid($client_uid) : undef;

        $CREATE_UPDATE_USER_OBJECT->{ClientID} = $ClientID if $ClientID;

        # Теперь проверяем, что у нас получилось
        my $is_geo_manager = ((is_api_geo_allowed($self) || 'No') eq 'Yes');

        # если ходим от геоменеджера и пользователь новый (валюта не определена), то позволяем создавать в YND_FIXED
        my $is_yndx_fixed_allowed = ($is_geo_manager && !$ClientID && $params->{Currency} eq 'YND_FIXED');
        my $vr = Direct::Validation::Client::check_country_and_currency(
            {uid => $client_uid, %$CREATE_UPDATE_USER_OBJECT}, $AgencyID, is_direct => !$is_yndx_fixed_allowed
        );

        if (!$vr->is_valid()) {
            warn "check country and currency failed: " . join(',', map { to_json($_->TO_JSON) } @{$vr->get_errors});
            dieSOAP('CantCreateClient', $vr->get_first_error_description());
        }

        if (!$ClientID && $self->{rbac_login_rights}->{manager_control} && $params->{Currency} eq 'RUB') {
            # для новых рублёвых клиентов, создаваемых менеджером - прописываем Россию
            # временный хак для CSSUPPORT-1637
            $CREATE_UPDATE_USER_OBJECT->{initial_country} = $geo_regions::RUS;
        }

        if (!$ClientID){
            # создаем клиента
            my $new_client = {
                agency      => $create_direct_client ? 'no' : 'yes',
                client_uid  => $client_uid,
                email       => $email,
                name        => $fio,
                currency    => $CREATE_UPDATE_USER_OBJECT->{initial_currency},
                # multicurrency: надо принимать через API параметры страны и валюты клиента
            };

            if ( create_client_id($new_client, $operator_uid) ) {
                warn "CantCreateClient create_client_id failed";
                dieSOAP('CantCreateClient');
            }
            $ClientID = $new_client->{new_clientID};
            if (!create_client_id_association($client_uid, $ClientID, $operator_uid)) {
                dieSOAP('CantCreateClientAssociation');
            }
        }

        $CREATE_UPDATE_USER_OBJECT->{ClientID} = $ClientID;
        $CREATE_UPDATE_USER_OBJECT->{role} = 'client';

        create_update_user( $client_uid, $CREATE_UPDATE_USER_OBJECT );
        API::ClientOptions::add($ClientID, {'api_enabled' => 'Default'});

        # при создании субклиентов агентства проверяем, нужно ли создавать клиента без ОС
        if ($AgencyID) {
            my $agency_options = Direct::AgencyOptions->get_item($AgencyID);
            if ($agency_options->allow_clients_without_wallet()
                && !$agency_options->default_clients_with_wallet()
            ) {
                create_update_client({client_data => {ClientID => $ClientID, create_without_wallet => 1}});
            }
        }

        # создаем клиента в rbac
        if ( ! $create_direct_client ) {
            # ассоциируем клиента с агентством
            if ( rbac_create_agency_subclient($self->{rbac}, $operator_uid, $client_uid, 'commit') ) {
                warn "CantCreateClient rbac_create_agency_subclint failed";
                dieSOAP('CantCreateClient');
            }
        } else {
            if ( rbac_create_manager_client($self->{rbac}, $self->{uid}, $client_uid, 'commit') ) {
                warn "CantCreateClient rbac_create_manager_client failed";
                dieSOAP('CantCreateClient')
            }
        }

        my $result = {
                        Login => $params->{Login},
                        FIO => $fio,
                        Email => $email,
                        ClientID => $ClientID,
                     };

        if (! $login_exists
                && ! (defined $self->{ClientID} && has_spec_limits($self->{ClientID})
                        && get_spec_limit($self->{ClientID}, 'create_client_no_password'))) {
            # возвращаем пароль, если клиент создавался в паспорте
            $result->{Password} = $password;
        }

        # возвращаем дополнительное поле, если к нам обращается менеджер
        if ($self->{rbac_login_rights}->{role} eq 'manager') {
            $result->{uid} = $client_uid;
        }

        return $result;

    } elsif ($new_login->{status} eq 'error' && any {$_ eq 'login.notavailable'} @{$new_login->{errors}}) {
        dieSOAP('LoginOccupied');
    } elsif ($new_login->{status} eq 'error') {
        send_alert("Error creating login $params->{Login}, got result " . to_json($new_login), 'create_passport_login API error');
        dieSOAP('CantCreateLogin');
    } else {
        dieSOAP('CantCreateLogin');
    }
}

=head2  GetClientInfo

    Получить информацию о клиенте

    на вход - для _агенств_ - логины клиента,
               для _обычных_ клиентов - ничего ( вернет информацию о самом клиенте )
     возвращает структуру с информацией о клиенте - client_obj

=cut

sub GetClientInfo
{
    my ($self, $logins) = @_;

    my @client_uids;
    if( defined $logins ) {

        # собираем массив UID'ов валидных логинов
        my @uniq_logins = uniq @$logins;
        my $login2uid = get_login2uid(login => \@uniq_logins);
        push @client_uids, values %$login2uid;

        if (scalar @client_uids != scalar @uniq_logins) {
            dieSOAP('BadLogin');
        }
    } else {
        # показываем информацию о самом себе
        push @client_uids, $self->{uid};
    }

    if (@client_uids) {
        # админ роли могут посмотреть любой логин
        unless ($self->{rbac_login_rights}->{role} =~ /^(media|placer|super|support|superreader)$/) {
            # фильтруем "себя", на себя права всегда есть
            my @uids_to_check = grep {$_ != $self->{uid}} @client_uids;
            dieSOAP('BadLogin') if scalar @uids_to_check && check_rbac_rights( $self, 'api_userSettings', { UID => $self->{uid}, uids => \@uids_to_check } );
        }
    } else {
        dieSOAP('BadLogin');
    }

    my $result = [];

    my $clients = get_client_object($self, uid => \@client_uids, get_ext_params => 1);

    push @$result, @{$clients || []};
    
    my $role = $self->{rbac_login_rights}->{role};
    foreach my $client ( @$result ) {
        if ( $role eq 'client' ) {
            $client->{StatusArch} = 'No';
        }
    }


    return $result;
}

=head2 UpdateClientInfo

    Обновить информацию о клиенте

    на вход - массив ClientInfo

=cut

# для трансляции внешних названий во внутренние
our %INT_NAMES;

$INT_NAMES{UpdateClientInfo} = {
    AgencyName => 'name',
    AgencyUrl => 'agency_url',
    AgencyStatus => 'agency_status',
    CityID => 'geo_id',
    Description => 'description',
};

sub UpdateClientInfo
{
    my ($self, $params) = @_;

    my %REVERSE_INT_NAMES = reverse %{$INT_NAMES{UpdateClientInfo}};

    # соответствия названий прав
    my %client_rights_names;
    @client_rights_names{qw/AllowEditCampaigns AllowImportXLS AllowTransferMoney/} = qw/isSuperSubClient allowImportXLS allowTransferMoney/;

    my $login2uid = get_login2uid(login => [map { $_->{Login} } @$params]);

    my @client_uids = values %$login2uid;

    push @{$self->{cluid}}, uniq @client_uids;

    if (@client_uids) {
        dieSOAP('BadLogin') if check_rbac_rights( $self, 'api_userSettingsModify', { UID => $self->{uid}, uids => \@client_uids } );
    } else {
        dieSOAP('BadLogin');
    }

    # если к методу обращаются под логином агенства, вычисляем его client_id
    my $cur_agency_client_id = $self->{rbac_login_rights}->{agency_control} ? rbac_get_agency_clientid_by_uid( $self->{uid}) : undef;

    my ($agency_logins, $agency_login2uid, $agency_client_ids);

    if ( $self->{rbac_login_rights}->{role} =~ /^(agency|super|manager)$/) {

        # соответствие логинов агенств их uid (только для менеджеров имеет смысл)
        $agency_logins = [ uniq map { $_->{AgencyLogin} } grep {$_->{AgencyLogin}} map { @{$_->{ClientRights}} } grep {$_->{ClientRights}} @$params ];

        if ($self->{rbac_login_rights}->{manager_control}) {
            if ($self->{api_version_full} > 4) { # latest - ok!
                push @$agency_logins, uniq map { $_->{AgencyLogin} } grep {$_->{AgencyLogin}} map { @{$_->{Descriptions}} } grep {$_->{Descriptions}} @$params;
            }
        }

        $agency_login2uid = get_login2uid(login => $agency_logins);
        # соответствие uid агенств и их client_id
        $agency_client_ids = rbac_get_agencies_clientids_by_uids( [values %$agency_login2uid]);
    }

    if ($self->{rbac_login_rights}{role} eq 'manager'
        && scalar keys %$agency_login2uid
        && !rbac_mass_is_owner($self->{rbac}, $self->{uid}, [values %$agency_login2uid])) {

        dieSOAP('NoRights', iget('Недостаточно прав на редактирование параметров клиента одного или нескольких агентств'));
    }

    # соответствие uid клиентов и их client_id
    my $client_client_ids = rbac_get_client_clientids_by_uids([@client_uids]);
    my @client_ids = values %$client_client_ids;
    dieSOAP('NoRights', APICommon::msg_converting_in_progress) if Client::is_any_client_converting_soon(\@client_ids);
    dieSOAP('NoRights', APICommon::msg_must_convert) if Client::is_any_client_must_convert(\@client_ids);

    # соответствие uid редактируемых агенств и их client_id
    my $edited_agency_client_ids = rbac_get_agencies_clientids_by_uids( [@client_uids]);

    # соответствие uid клиентов их client_chief_uids
    my $client_chief_uids = rbac_get_chief_reps_of_client_reps([@client_uids]);

    # соответствие uid и ролей
    my $clients_who_is = rbac_multi_who_is($self->{rbac}, [@client_uids]);

    # если оператор - агентство, проверяем права на редактирвоание паарметров пользователя
    my $agency_can_edit_user_data = {};
    if ($cur_agency_client_id) {
        $agency_can_edit_user_data = rbac_agency_can_edit_user_data($self->{rbac}, $cur_agency_client_id, \@client_ids)
    }

    foreach my $client (@$params) {
        my $cl_uid = $login2uid->{$client->{Login}};

        my $client_client_id = $edited_agency_client_ids->{$cl_uid} || $client_client_ids->{$cl_uid};
        
        my $user_role = $clients_who_is->{$cl_uid};

        # переобзываем пришедшие к нам параметры для сохранения в базу
        my $save_user_data = {};

        # сохранять такие поля может только клиент и агентство для своих несвободных клиентов или агентство и менеджер для себя
        if ( $user_role !~ /client$/ || !$self->{rbac_login_rights}->{agency_control} || $agency_can_edit_user_data->{$client_client_ids->{$cl_uid}} ) {
            @{$save_user_data}{qw/fio email phone/} = @{$client}{qw/FIO Email Phone/};
            # а эти поля имеют смысл только для клиента
            if ($user_role =~ /client$/ ) {
                @{$save_user_data}{qw/sendWarn sendNews sendAccNews/} = @{$client}{qw/SendWarn SendNews SendAccNews/};
            }
        }
        my $client_data = { ClientID => $client_client_ids->{$cl_uid} };

        if ( $self->{rbac_login_rights}->{role} =~ /^(super|support|manager)$/
                && $self->{api_version_full} > 4
        ) { # latest - ok!
            # подставляем данные из базы, если они не указаны пользователем в версии 4-latest
            if ($self->{api_version_full} == 4.5 && $user_role eq 'agency') {
                # TODO: нужно оптимизировать до массовой операции
                my $client_info_db = get_client_data($client_client_id, [qw/name agency_status agency_url/]);
                foreach my $key_name (qw/name agency_status agency_url/) {
                    my $ext_keys_name = $REVERSE_INT_NAMES{$key_name};
                    $client->{$ext_keys_name} = $client_info_db->{$key_name} if !defined $client->{$ext_keys_name};
                }
            }

            # логика для 5-ой версии
            if ($self->{api_version_full} > 4.5) {
                # latest - description может быть пустым - в latest не делаем!
                $save_user_data->{geo_id} ||= $client->{CityID};
                $save_user_data->{description} ||= $client->{Description} if $user_role =~ /(client|agency)$/;
            }

            if ($user_role eq 'agency') { # latest - ok!

                $client->{AgencyName} =~ s/^\s+//;
                $client->{AgencyName} =~ s/\s+$//;
                $client->{AgencyUrl} =~ s/^\s+//;
                $client->{AgencyUrl} =~ s/\s+$//;

                # TODO: может получиться, что ДК затрет статус агентства в 5ой версии ?!
                my $agency_status = $client->{AgencyStatus} && $client->{AgencyStatus} =~ /^(?:SA|AA|HY|WC|NR|ABU)$/ ? $client->{AgencyStatus} : undef;
                $client_data = {
                    ClientID => $edited_agency_client_ids->{$cl_uid},
                    name => $client->{AgencyName},
                    agency_url => $client->{AgencyUrl},
                    agency_status => $agency_status,
                };
            }
        }

        if ($self->{api_version_full} > 4) {
            if (exists $client->{DisplayStoreRating}) {
                my $is_hide_market_rating = $client->{DisplayStoreRating} eq 'No' ? 1 : 0;
                $client_data->{hide_market_rating} = $is_hide_market_rating;
            }
        }
        create_update_client({ client_data => $client_data });
        create_update_user($cl_uid, $save_user_data);
                
        # ветка выполняется только для агенств или менеджеров над клиентами, на которых у них есть права на редактирвоание права
        if ( $self->{rbac_login_rights}->{role} =~ /^(agency|super|manager)$/
             && $user_role =~ /client$/ && rbac_has_agency($self->{rbac}, $cl_uid) && $self->{rbac_login_rights}->{ModifySubClient}) {

            my $subclients_rights = rbac_get_subclients_rights($self->{rbac}, $self->{uid}, $cl_uid);

            # заполняем хэш, маскируя его под форму
            my $rights_data = {};
            foreach my $right (@{$client->{ClientRights}}) {
                my $agency_client_id = $self->{rbac_login_rights}->{agency_control} ? $cur_agency_client_id :
                                                                              $agency_client_ids->{$agency_login2uid->{$right->{AgencyLogin}}};
                # ключи хэша вида [RightName]_[agency_client_id]
                $rights_data->{ join('_', $client_rights_names{$right->{RightName}}, $agency_client_id) } = $right->{Value} eq 'Yes' ? 1 : 0;
            }

            my %save_rights_options = ();

            # сохраняем description
            if ($self->{api_version_full} > 4.5) {
                # latest - description может быть пустым - в latest не делаем!

                if ($self->{rbac_login_rights}->{agency_control}) {
                    $rights_data->{join('_', 'client_description', $cur_agency_client_id)} = $client->{Description};
                } elsif ($self->{rbac_login_rights}->{role} =~ /^(super|manager)$/) {

                    foreach my $desc (@{$client->{Descriptions}}) {
                        my $agency_client_id = $agency_client_ids->{$agency_login2uid->{$desc->{AgencyLogin}}};

                        # ключи хэша вида [RightName]_[agency_client_id]
                        $rights_data->{ join('_', 'client_description', $agency_client_id) } = $desc->{Description};
                    }
                }
                $save_rights_options{save_description} = 1;
            }

            if (my $error = save_subclients_rights( $self->{rbac},
                                                    $cl_uid, $self->{uid},
                                                    $self->{rbac_login_rights},
                                                    $client_client_ids->{$cl_uid},
                                                    $subclients_rights,
                                                    $rights_data,
                                                    %save_rights_options )) {
                dieSOAP('CantUpdateRights', $error);
            }
        }
        save_status_not_resident($self->{rbac},
                                    serviced => ServicedClient::is_user_serviced($client_chief_uids->{$cl_uid}),
                                    client_client_id => $client_client_ids->{$cl_uid},
                                    agency_client_id => $edited_agency_client_ids->{$cl_uid},
                                    not_resident => ($client->{NonResident} || '') eq 'Yes' ? 'Yes' : 'No',
                                    user_role => $user_role,
                                    login_rights => $self->{rbac_login_rights},
                                );
    }
    return 1;
}

=pod head2 GetSubClients

 Вызывается для получения списка "прямых клиентов" и агенств у пользователя,
 подчиненных ему "на прямую" менеджеров

 на вход {Login => 'Login'}

 на выходе ArrayOfShortClientInfo
 [
    {
    Login => 'sidorov',
    FIO  => 'Сидоров Андрей Петрович',
    Role  => 'agency'
    },
    {
     Login => 'ivanov',
     FIO  => 'Иванов Иван Иванович',
     Role  => 'client'
    },
 ]

=cut

sub GetSubClients
{
    my ($self, $params) = @_;

    my $cl_uid;

    if ($params && $params->{Login}){
        my $client_login = $params->{Login};

        $cl_uid = get_uid_by_login2($client_login);
        dieSOAP('BadClient') unless defined $cl_uid;
        dieSOAP('BadClient') if $self->{rbac_login_rights}->{role} !~ /^(media|placer)$/ && check_rbac_rights( $self, 'api_getSubClients', { UID => $self->{uid}, uid => $cl_uid } );
        #dieSOAP( 'NoRights' ) if $self->{uid} != $cl_uid && ! rbac_is_owner($self->{rbac}, $self->{uid}, $cl_uid);
    } else {
        if(! $self->{uid} ) {
            dieSOAP('UserInvalid'); # лучше еще раз, иначе лишнего можем сболтнуть ;-(
        }
        $cl_uid = $self->{uid};
    }

    my ($staff_roles, $clients_roles) = (1, 1);
    if ($params && $params->{RoleFilter}) {
        $clients_roles = 0 if $params->{RoleFilter} eq 'Staff';
        $staff_roles = 0 if $params->{RoleFilter} eq 'Clients';
    }

    my $clients = [];
    my $managers = [];
    my $agencies = [];

    my $res = rbac_who_is_detailed($self->{rbac}, $cl_uid);

    if ($res->{role} eq 'manager'){
        if ($staff_roles) {
            if ($res->{is_superteamleader}){
                # получаем тимлидеров у супертимлидера
                $managers = rbac_get_team_of_superteamleader($self->{rbac}, $cl_uid);
            } elsif ($res->{is_teamleader}){
                $managers = rbac_get_managers_of_teamleader_with_idm($self->{rbac}, $cl_uid);
            }
        }

        if ($clients_roles) {
            $clients = rbac_get_clients_of_manager($self->{rbac}, $cl_uid);

            my $chiefs_uids = @$clients ? rbac_get_chief_reps_of_client_reps($clients) : {};
            $clients = [uniq values %$chiefs_uids];

            # rbac_get_chief_reps_of_agencies
            $agencies = rbac_get_agencies_uids($self->{rbac}, $cl_uid);
        }

    } elsif ($res->{role} eq 'agency'){
        if( $res->{is_agency_limited} ){
            $clients  = rbac_get_subclients_uids($self->{rbac}, $cl_uid);

        } elsif( $res->{is_agency_main} ){
            my $ClientID = rbac_get_agency_clientid_by_uid( $cl_uid) || dieSOAP('500', iget("Указанный клиент не найден"));
            $clients  = rbac_get_subclients_uids($self->{rbac}, $cl_uid);
            $agencies = rbac_get_limited_reps_of_agency($ClientID);

        } elsif( $res->{is_agency_chief} ) {
            my $ClientID = rbac_get_agency_clientid_by_uid( $cl_uid) || dieSOAP('500', iget("Указанный клиент не найден"));
            $clients  = rbac_get_subclients_uids($self->{rbac}, $cl_uid);
            $agencies = rbac_get_limited_reps_of_agency($ClientID);
            my $unlimited_agency_reps = rbac_get_main_reps_of_agency($ClientID) || [];
            push(@$agencies, grep {$_ != $cl_uid} @$unlimited_agency_reps);
        }

    } elsif($res->{role} eq 'client'){
        my $chief_uid = rbac_get_chief_rep_of_client_rep($cl_uid);
        my $cl_client_id = rbac_get_client_clientid_by_uid($cl_uid) || dieSOAP('500', iget("Указанный клиент не найден"));
        $clients  = rbac_get_main_reps_of_client($cl_client_id);
        $clients = [grep {$_ != $cl_uid && $_ != $chief_uid} @$clients];

    }

    $clients = filter_geo_mcb_users($clients);

    my $clients_info;
    if ($clients && ref( $clients ) eq 'ARRAY'){
        my $agency_uid = $res->{role} eq 'agency' ? $cl_uid : undef;

        $clients_info = get_short_client_object($self, {
                            uid => [(@$clients, @$agencies, @$managers)]
                            , cl_uid => $cl_uid
                            , agency_uid => $agency_uid
                        });

        my $no_arch = 0;
        $no_arch = 1 if $params->{Filter} && $params->{Filter}->{StatusArch} eq 'No';

        $clients_info = [ grep {
                                 if ($no_arch && $_->{StatusArch} eq 'Yes') {
                                    0;
                                 }
                                 else {
                                    delete $_->{StatusArch}; #ненужный параметр
                                    1;
                                 }

                               } @$clients_info ];

    }

    return $clients_info;
}


=head2 GetClientsList

    возвращает массив структур с информацией о клиенте [ client_obj ]
    для НЕагенств возвращает SOAP-ошибку - `NoRights`

=cut

sub GetClientsList
{
    # возвращает массив структур с информацией о клиенте [ client_obj ]
    # для НЕагенств возвращает SOAP-ошибку - `NoRights`
    my ($self, $params) = @_;

    my $error_code = 0;
    if( $error_code
            || $self->{rbac_login_rights}->{role} ne 'agency' ) {
        dieSOAP( 'NoRights' ) ;
    }

    if(! $self->{uid} ) {
        dieSOAP('UserInvalid'); # лучше еще раз, иначе лишнего можем сболтнуть ;-(
    }

    my $uid_subclients = rbac_get_subclients_uids($self->{rbac}, $self->{uid}) || []; # список из RBAC

    $uid_subclients = filter_geo_mcb_users($uid_subclients);

    my @result;

    if (defined $uid_subclients && ref( $uid_subclients ) eq 'ARRAY' && scalar @$uid_subclients) {

        my %OPT = (uid => \@{$uid_subclients});

        if ($self->{api_version_full} == 4) {
            $OPT{get_ext_params} = 1;
        }

        my $clients = get_client_object( $self, %OPT ) || [];

        if ($params->{Filter} && $params->{Filter}->{StatusArch} eq 'No') {
            push @result, grep { $_->{StatusArch} eq 'No' } @$clients;
        }
        else {
            push @result, @$clients;
        }
    }

    return \@result;
}

sub GetClientsUnits {
    my ($self, $params) = @_;
    $params ||= [];

    my (@logins, $login2uid);
    if (!@$params) {
        $login2uid->{ $self->{operator_login} } = $self->{uid};
        push @logins, $self->{operator_login};
    } else {
        @logins = uniq @$params;
        $login2uid = get_login2uid(login => $params);
    }

    my @cl_uids = values %$login2uid;
    my $uid2login = { reverse %$login2uid };

    if (@logins && @logins != scalar keys %$login2uid) {
        dieSOAP('BadLogin', iget('Несуществующие логины: %s', join(', ', grep {!$login2uid->{$_}} @logins) ));
    }

    if (! @cl_uids ||
        ( $self->{rbac_login_rights}->{role} !~ /^(media|placer|super|support|superreader)$/
          && check_rbac_rights( $self, 'api_userSettingsModify', {UID => $self->{uid}, uids => \@cl_uids})
        )
       ) {
        dieSOAP('NoRights');
    }

    # rep_uid to chief_rep_uid
    my $chief_client_reps = rbac_get_chief_reps_of_client_reps(\@cl_uids);

    # rep_uid to client_id
    my $agency_client_ids = rbac_get_agencies_clientids_by_uids( \@cl_uids);

    my $chief_agency_reps = (scalar values %$agency_client_ids) # client_id to chief_rep_uid
        ? rbac_get_chief_reps_of_agencies([values %$agency_client_ids])
        : {};

    # учитываем что баллы хранятся на главном представителе, при этом в запросе
    # может быть указан любой представитель, равно как и несколько представителей
    # имеющих одного главного представителя, в этом случае нужно вернуть
    # баллы для представителей с указанием логинов из запроса.
    # поэтому для каждого логина в запросе нужно знать uid главного представителя,
    # на которого будем получать баллы
    my $login2chief_rep_uid = {};

    foreach my $rep_uid (keys %$chief_client_reps) {
        $login2chief_rep_uid->{ $uid2login->{$rep_uid} } = $chief_client_reps->{$rep_uid};
    }

    foreach my $rep_uid (keys %$agency_client_ids) {
        my $clid = $agency_client_ids->{$rep_uid};
        $login2chief_rep_uid->{ $uid2login->{$rep_uid} } =  $chief_agency_reps->{$clid};
    }

    my $units = $self->{uhost}->check_or_init_user_units(uniq(values %$chief_client_reps, values %$chief_agency_reps));

    return [
        map {{
            Login => $_,
            UnitsRest => $units->{$login2chief_rep_uid->{$_}}->{units_rest}
        }} @logins
    ];
}

1;
