package API::Methods::Common;

# $Id$

=head1 NAME

    API::Methods::Common

=head1 DESCRIPTION

    Модуль для общих функций внутри API::Methods

=cut

use APICommon qw(:subs);
use API::Errors;
use User;
use MoneyTransfer;
use RBACElementary;

use Yandex::I18n;
use Yandex::Balance qw/
    balance_get_request_choices
    balance_create_invoice
    balance_create_fast_payment
    /;
use Yandex::Balance::Simple qw/
    balance_simple_create_basket
    balance_simple_pay_basket
    balance_simple_check_basket
    /;
use utf8;
use strict;
use warnings;
use Settings;
use Yandex::Clone qw/yclone/;
use Primitives;
use PrimitivesIds;
use TextTools;
use Client;
use Yandex::DBTools qw/do_insert_into_table get_one_line_sql do_update_table/;
use Currency::Rate;

use List::MoreUtils qw(any);

our $FIN_OPS_API_TIMEOUT        = 310; # seconds
our $BALANCE_SIMPLE_API_TIMEOUT = 10;   # seconds

=head2 create_user_payment_transaction

    Сохраняет запись по платежу картой

=cut

sub create_user_payment_transaction
{
    my ( $uid, $transaction_id, $payment_id ) = @_;

    return do_insert_into_table( PPC( uid => $uid ), 'user_payment_transactions',
        {
            uid => $uid,
            transaction_id => $transaction_id,
            payment_id => $payment_id,
        }
    );
}

=head2 get_user_payment_transaction

    Возвращает данные по платежу картой по uid и transaction_id

=cut

sub get_user_payment_transaction
{
    my ( $uid, $tid ) = @_;

    return get_one_line_sql( PPC( uid => $uid ),
        [
            'SELECT uid, transaction_id, payment_id, status FROM user_payment_transactions',
            WHERE => {
                uid => $uid,
                transaction_id => $tid,
            }
        ]
    );
}

=head2 update_user_payment_transaction_status

    Обновляет статус платежа картой по uid и transaction_id

=cut

# List of final payment statuses, 
# NB: length of status name must be less or equal 32 chars
my %PAYMENT_FINAL_STATUSES = (
    Success        => 1,
    Cancelled      => 1,
    Refund         => 1,
    Timeout        => 1,
    NotEnoughFunds => 1,
    ExpiredCard    => 1,
    LimitExceeded  => 1,
    Error          => 1,
);

sub update_user_payment_transaction_status
{
    my ( $uid, $tid, $status ) = @_;

    die "Unknown payment status: $status" unless exists $PAYMENT_FINAL_STATUSES{ $status };

    return do_update_table( PPC( uid => $uid ), 'user_payment_transactions',
        { status => $status },
        where => {
            uid => $uid,
            transaction_id => $tid,
        }
    );
}

=head2 payment_common(self, params)

    Функция оплаты кампаний/счетов
    ожидает в params структуру {Payments => ..., Contract => ..., PayMethod => ...}
    Payments — как в APICommon::create_pay_campaign_request

=cut

sub payment_common($$) {
    my ($self, $params) = @_;
    my $result = 0;

    if ($params->{PayMethod} eq 'Bank') {

        # multicurrency: ok!
        my $billing_response = create_pay_campaign_request($self, $params->{Payments});
        if ($billing_response->{error}) {
            return $billing_response;
        }
        my $billing_url = $billing_response->{result};
        my ($request_id) = $billing_url =~ /request_id%3D(\d+)%26/;

        my $client_id = get_one_user_field($self->{uid}, 'ClientID');

        my ($request_choices, $contract_options);

        eval {
            $request_choices = balance_get_request_choices($self->{uid}, $request_id, $params->{ContractID});
        };

        if ($@) {
            return {error => get_error_object('FinOpsTmpUnavail')};
        }

        if ($request_choices eq 'CONTRACT_NOT_FOUND') {
            return {error => get_error_object('BadContract', iget('Указанный договор %s не найден', $params->{ContractID}))};
        } elsif ($request_choices eq 'ILLEGAL_REQUEST_CONTRACT') {
            return {error => get_error_object('BadContract')};
        }

        if (ref($request_choices) eq 'HASH' && scalar keys %$request_choices) {

            if (!$request_choices->{credits}{is_available}) {
                return {error => get_error_object('CreditLimitExceeded')};
            }

            my $pcp;

            foreach (@{$request_choices->{pcp_list}}) {
                if ($_->{contract}->{external_id} eq $params->{ContractID} ) {
                    $pcp = $_;
                    last;
                }
            }

            $contract_options->{contract_id} = $pcp->{contract}->{id};
            $contract_options->{person_id} = $pcp->{contract}->{person_id};

            my (@bank_paysyses, @card_paysyses);
            foreach (@{$pcp->{paysyses}}) {
                if ($_->{payment_method}{cc} eq 'bank') {
                    push @bank_paysyses, $_;
                } elsif ($_->{payment_method}{cc} eq 'card') {
                    push @card_paysyses, $_;
                }
            }
            if (scalar @bank_paysyses) {
                $contract_options->{paysys_id} = $bank_paysyses[0]->{id};
            } elsif (scalar @card_paysyses) {
                $contract_options->{paysys_id} = $card_paysyses[0]->{id};
            } else {
                return {error => get_error_object('BadContract', iget('Оплата в кредит с помощью данного договора невозможна'))};
            }
        }

        if ($contract_options && scalar keys %$contract_options) {

            my $response;
            eval {
                $response =  balance_create_invoice( [ $self->{uid}, {
                                                    RequestID => $request_id,
                                                    PaysysID => $contract_options->{paysys_id},
                                                    PersonID => $contract_options->{person_id},
                                                    Credit => 1,
                                                    ContractID => $contract_options->{contract_id}
                                                }], timeout => $FIN_OPS_API_TIMEOUT);
            };

            if ($@) {
                return {error => get_error_object('FinOpsTmpUnavail')};
            }

            if ($response eq 'CREDIT_LIMIT_EXEEDED') {
                return {error => get_error_object('CreditLimitExceeded')};
            } elsif ($response eq 'ILLEGAL_CONTRACT') {
                return {error => get_error_object('BadContract')};
            } elsif ($response =~ /^\d+$/) {
                $result = 1;
            } else {
                return {error => get_error_object('BadPayCampFromLimits')};
            }
        } else {
            return {error => get_error_object('BadContract')};
        }

    } elsif ($params->{PayMethod} eq 'YM' || $params->{PayMethod} eq 'Overdraft') {

        return {error => get_error_object('PayTokenNotProvided')} if ! $self->{payment_token} && $params->{PayMethod} eq 'YM';

        my $paysys_id = $Yandex::Balance::PAYSYS_IDS{YM};
        my $overdraft = $params->{PayMethod} eq 'Overdraft' ? 1 : 0;

        my $billing_response = create_pay_campaign_request($self, $params->{Payments}, prepare_only => 1);
        if ($billing_response->{error}) {
            return $billing_response;
        }
        my $options = $billing_response->{result};

        my @items;

        for my $payment (@{$options->{CreateRequest}}) {

            push @items, {
                service_id => $Settings::SERVICEID{direct},
                service_order_id => $payment->{ServiceOrderID},
                qty => $payment->{Qty},
            };
        }

        my $request_options = {
            passport_id => $self->{uid},
            client_id => $options->{client_id},
            paysys_id => $paysys_id,
            payment_token => $self->{payment_token},
            overdraft => $overdraft,
            items => \@items,
        };

        my $response;
        eval {
            $response = balance_create_fast_payment($request_options, timeout => $FIN_OPS_API_TIMEOUT);
        };

        if ($@) {
            return {error => get_error_object('FinOpsTmpUnavail')};
        } else {
            if ($response eq 'OVERDRAFT_NOT_AVAILABLE') {
                return {error => get_error_object('OverdraftNotAvailable')};
            } elsif ($response eq 'OVERDRAFT_LIMIT_EXCEEDED') {
                return {error => get_error_object('OverdraftLimitExceeded')};
            } elsif ($response eq 'YM_AUTH_API_ERROR' || $response eq 'YM_PAYMENT_API_ERROR') {
                return {error => get_error_object('FinOpsTmpUnavail', iget('Платеж через Яндекс.Деньги временно недоступен'))};
            } elsif ($response eq 'PAYMENT_REG_ERROR' || $response eq 'PAYMENT_PROC_ERROR') {
                return {error => get_error_object('PaymentProcError')};
            } elsif ($response eq 'REFRESH_TOKEN') {
                return {error => get_error_object('ExpiredPaymentToken')};
            } elsif ($response eq 'AMOUNT_TOO_BIG') {
                return {error => get_error_object('PaymentProcError', iget('Сумма для зачисления превышает доступный лимит'))};
            } elsif ($response eq 'LIMIT_EXCEEDED') {
                return {error => get_error_object('PaymentProcError', iget('Превышен допустимый лимит платежа'))};
            } elsif ($response eq 'NOT_ENOUGH_FUNDS') {
                return {error => get_error_object('PaymentProcError', iget('Недостаточно средств на счете'))};
            }
        }

        if (ref($response) eq 'HASH') {
            $result = 1;
        }
    } elsif ($params->{PayMethod} eq 'LinkedCard') {

        dieSOAP('NoRights') if $self->{rbac_login_rights}->{role} && $self->{rbac_login_rights}->{role} ne 'client';

        my $is_old_app = $params->{Version} ? 0 : 1; # в поле Version получаем версию мобильного приложения, сделавшего запрос,
                                                     # например 20502A, символ в конце A - Андроид, I - IOS, сделали на будущее, сейчас не используется
        if ($is_old_app) {
            return { error => get_error_object('PaymentProcError', iget('Пожалуйста, обновите версию приложения, чтобы пополнить счёт')) }
        }

        # TODO: !!! надо отключить доступность метода без платежного токена !!!

        # TODO: нужно ли проверять OAuth токен - при обращении к АПИ он уже проверяется?
        # return { error => get_error_object('OAuthPayTokenNotProvided') } if not defined $self->{authorization_token};

        my $payment_transaction = get_user_payment_transaction( $self->{uid}, $params->{CustomTransactionID} );

        # не разрешаем user-у создавать несколько платежей с одним и тем же CustomTransactionID
        if ( defined $payment_transaction ) {
            return { error => get_error_object('PaymentTransactionAlreadyExists') };
        }

        my $billing_response = create_pay_campaign_request($self, $params->{Payments}, prepare_only => 1);
        if ($billing_response->{error}) {
            return $billing_response;
        }

        # конвертируем параметры запросы для создания корзины на оплату заказов
        my @basket_orders;
        for my $payment (@{$billing_response->{result}->{CreateRequest}}) {
            push @basket_orders, {
                service_order_id => $payment->{ServiceOrderID},
                qty => $payment->{Qty}
            };
        }

        # создаем заказ-корзину для оплаты в Балансе
        my $create_basket_options = {
            uid => $self->{uid},
            paymethod_id => $params->{PayMethodID},
            user_ip => $self->{plack_request}->address,
            currency => $params->{Payments}->[0]->{Currency},
            orders => \@basket_orders,
            $is_old_app ? () : (wait_for_cvn => 1),
            #no_charge => 1, # FOR TESTING ONLY
        };

        my $create_basket_responce;
        eval {
            $create_basket_responce = balance_simple_create_basket( $create_basket_options, timeout => $BALANCE_SIMPLE_API_TIMEOUT, recall => 2 );
        };

        if ($@) {
            return { error => get_error_object('FinOpsTmpUnavail') };
        }

        my ( $status, $scode ) = @$create_basket_responce{qw/ status status_code /};
        if ( $status ne 'success' ) {
            my $error;
            if ( $status eq 'no_auth' ) {
                $error = 'BadOAuthPayToken';
            } elsif ( $status eq 'error' and $scode eq 'invalid_payment_method' ) {
                $error = 'BadPayMethodID';
            } elsif ( $status eq 'forbidden' and $scode eq 'invalid_currency' ) {
                $error = 'PaymentCurrencyForbiddenForClient';
            } else {
                $error = 'PaymentProcError';
            }
            return { error => get_error_object( $error ) };
        }

        create_user_payment_transaction( $self->{uid}, $params->{CustomTransactionID}, $create_basket_responce->{trust_payment_id} );

        # пытаемся оплатить корзину выбранным способом
        my $pay_basket_options = {
            uid => $self->{uid},
            user_ip => $self->{plack_request}->address,
            trust_payment_id => $create_basket_responce->{trust_payment_id},
        };

        my $pay_basket_responce;
        eval {
            $pay_basket_responce = balance_simple_pay_basket( $pay_basket_options, timeout => $BALANCE_SIMPLE_API_TIMEOUT, recall => 2 );
        };

        if ($@) {
            return { error => get_error_object('FinOpsTmpUnavail') };
        }

        ( $status, $scode ) = @$pay_basket_responce{qw/ status status_code /};
        if ( $status ne 'wait_for_notification'
                and $status ne 'success'
                and $status ne 'cancelled'
                and $status ne 'refund'
        ) {
            my $error = $status eq 'no_auth' ? 'BadOAuthPayToken' : 'PaymentProcError';
            return { error => get_error_object( $error ) };
        }

        $result = $create_basket_responce->{purchase_token};
    }

    return { result => $result };
}

=head2 check_payment_common(self, params)

    Функция статуса оплаты кампаний/счетов
    ожидает в params структуру {PaymentId => ...}

=cut

sub check_payment_common($$) {
    my ($self, $params) = @_;

    my $payment_transaction = get_user_payment_transaction( $self->{uid}, $params->{CustomTransactionID} );

    if ( not defined $payment_transaction ) {
        return { error => get_error_object('NoSuchPaymentTransaction') };
    }

    # возвращаем сохраненный статус платежа если он есть
    if ( $payment_transaction->{status} ) {
        return { result => $payment_transaction->{status} };
    }

    # проверям статус оплаты в Балансе
    my $check_basket_options = {
        token => $self->{authorization_token},
        user_ip => $self->{plack_request}->address,
        trust_payment_id => $payment_transaction->{payment_id},
    };

    my $check_basket_responce;
    eval {
        $check_basket_responce = balance_simple_check_basket( $check_basket_options, timeout => $BALANCE_SIMPLE_API_TIMEOUT );
    };

    if ($@) {
        return { error => get_error_object('FinOpsTmpUnavail') };
    }

    my ( $status, $scode ) = @$check_basket_responce{qw/ status status_code /};

    if ( $status eq 'no_auth' ) {
        return { error => get_error_object('BadOAuthPayToken') };
    }

    my $payment_status;
    my $is_final_status = 0;
    if ( $status eq 'wait_for_notification' ) {
        $payment_status = 'InProgress';
    } elsif ( $status eq 'success' ) {
        $is_final_status = 1;
        $payment_status  = 'Success';
    } elsif ( $status eq 'cancelled' ) {
        $is_final_status = 1;
        $payment_status  = 'Cancelled';
    } elsif ( $status eq 'refund' ) {
        $is_final_status = 1;
        $payment_status  = 'Refund';
    } elsif ( $status eq 'error' ) {
        if ( $scode eq 'payment_timeout' ) {
            $is_final_status = 1;
            $payment_status  = 'Timeout';
        } elsif ( $scode eq 'not_enough_funds' ) {
            $is_final_status = 1;
            $payment_status  = 'NotEnoughFunds';
        } elsif ( $scode eq 'expired_card' ) {
            $is_final_status = 1;
            $payment_status  = 'ExpiredCard';
        } elsif ( $scode eq 'limit_exceeded' ) {
            $is_final_status = 1;
            $payment_status  = 'LimitExceeded';
        # ошибки Директа или Биллинга
        } elsif ( $scode eq 'payment_not_found'
                    or $scode eq 'invalid_service_order_id'
                    or $scode eq 'order_not_found'
                    or $scode eq 'invalid_service_product'
        ) {
            $is_final_status = 1;
            $payment_status  = 'Error';
        # ошибки Биллинга или платежной системы
        } elsif ( $scode eq 'authorization_reject'
                    or $scode eq 'payment_refused'
                    or $scode eq 'technical_error'
        ) {
            $is_final_status = 1;
            $payment_status  = 'Error';
        # ошибки, которые не должнвы появляться при платеже картой в Директе
        } elsif ( $scode eq 'another_order_exists'
                    or $scode eq 'already_purchased'
                    or $scode eq 'already_pending'
        ) {
            $is_final_status = 1;
            $payment_status  = 'Error';
        } else {
            $payment_status = 'Error';
        }
    } else {
        $payment_status = 'Error';
    }

    if ( $is_final_status ) {
        update_user_payment_transaction_status( $self->{uid}, $params->{CustomTransactionID}, $payment_status );
    }

    return { result => $payment_status };
}

=head2 pay_notification_common(self, params; O)

    Функция постановки уведомлений об оплате в очередь
    ожидает в params структуру {Payments => ..., PayMethod => ...}
    Payments — как в APICommon::create_pay_campaign_request
    Если появится шаблон для общего счёта, выбор шаблона можно осуществить через третий параметр

=cut

sub pay_notification_common($$;%) {
    my ($self, $params) = @_;
    my $cids2uids;
    # для нотификации получаем логины пользователей кампаний

    foreach my $camp(map {$self->{_preprocessed}->{$_->{CampaignID}}} @{$params->{Payments}}) {
        $cids2uids->{$camp->{cid}} = $camp->{uid};
    }
    my $uids2logins = get_uid2login(uid => [values %$cids2uids]);

    foreach my $payment (@{$params->{Payments}}) {
        $payment->{Login} = $uids2logins->{$cids2uids->{$payment->{CampaignID}}};
    }

    if ($params->{PayMethod} eq 'Bank' || $params->{PayMethod} eq 'Overdraft') {
        notification_api_finance($self->{uid}, $self->{rbac}, $self->{uid}, 'api_pay_campaign', $params);
    }
    # TODO: письмо про оплату YM
    return;
}

=head2 transfer_common(self, params)

    Функция переноса средств между кампаниями. Должна корректно обрабатывать общие счета
    Сейчас сама проверяет права на перенос
    В случае успеха возвращает в {result => ...} структуру, частично подходящую для нотификаций
    Структура params:
    {"FromCampaigns": [
        {  /* PayCampElement */
            "CampaignID": (int),
            "Sum": (float),
            "Currency": (string)
        }
        ...
    ],
    "ToCampaigns": [
        {  /* PayCampElement */
            "CampaignID": (int),
            "Sum": (float),
            "Currency": (string)
        }
        ...
    ]}

    Структура $self->{_preprocessed} как для get_hashes_hash_sql("SELECT cid, name, IFNULL(currency, 'YND_FIXED') currency, uid, agencyuid FROM campaigns c", ...)

=cut

sub transfer_common($$) {
    my ($self, $params) = @_;
    my $compiled_params;

    $compiled_params->{camps_names} = $self->{_preprocessed};

    my $cid2currency = { map { $_ => $compiled_params->{camps_names}->{$_}->{currency} } keys %{$compiled_params->{camps_names} || {}} };

    my $source_client_cid = $params->{FromCampaigns}->[0]->{CampaignID};
    my $client_discount = get_client_discount(get_clientid(cid => $source_client_cid)) || 0;

    foreach my $item (@{$params->{FromCampaigns}}) {

        my $sum;
        if (($cid2currency->{$item->{CampaignID}} eq 'YND_FIXED')
            || (($self->{api_version_full} > 4) && ($item->{Currency} || 'YND_FIXED') ne 'YND_FIXED')) {
            $sum = $item->{Sum};
        } else {
            $sum = convert_currency($item->{Sum}, 'YND_FIXED', $cid2currency->{$item->{CampaignID}});
            $sum = Currencies::remove_bonus($sum, $client_discount);
        }

        $compiled_params->{from}->{$item->{CampaignID}} = round2s($sum); # можно переносить копейки
    }

    foreach my $item (@{$params->{ToCampaigns}}) {

        my $sum;
        if (($cid2currency->{$item->{CampaignID}} eq 'YND_FIXED')
            || (($self->{api_version_full} > 4) && ($item->{Currency} || 'YND_FIXED') ne 'YND_FIXED')) {
            $sum = $item->{Sum};
        } else {
            $sum = convert_currency($item->{Sum}, 'YND_FIXED', $cid2currency->{$item->{CampaignID}});
            $sum = Currencies::remove_bonus($sum, $client_discount);
        }

        $compiled_params->{to}->{$item->{CampaignID}} = round2s($sum);
    }

    # Здесь был комментарий про получение uid агентства вместо клиентского, если кампании агентские. Где-то это могло использоваться

    # cid первой кампании с которой будем переводить деньги
    my $fromcid = [ keys %{$compiled_params->{from}} ]->[0];
    my $client_uid = $compiled_params->{camps_names}->{$fromcid}->{uid};
    my $currency_source = $compiled_params->{camps_names}->{$fromcid}->{currency};

    # cid первой кампании на которую будем переводить деньги
    my $tocid = [ keys %{$compiled_params->{to}} ]->[0];
    my $client_uid_to = $compiled_params->{camps_names}->{$tocid}->{uid};
    my $currency_to = $compiled_params->{camps_names}->{$tocid}->{currency};

    # хэши кампаний для проверки в rbac
    my %from_camps = map {'from__'.$_ => $compiled_params->{from}{$_}} keys %{$compiled_params->{from}};
    my %to_camps = map {'to__'.$_ => $compiled_params->{to}{$_}} keys %{$compiled_params->{to}};

    my $error_code = check_rbac_rights( $self, 'api_transferMoney', {
            UID => $self->{uid}, uid => $client_uid, client_from => get_login(uid => $client_uid),
            client_to => get_login(uid => $client_uid_to), %from_camps, %to_camps } );

    return {error => get_error_object('NoRights')} if $error_code;

    $compiled_params->{total_sum} = 0.0;
    foreach my $from_cid(keys %{$compiled_params->{from}}) {
        $compiled_params->{total_sum} += $compiled_params->{from}{$from_cid};
    }

    # общий счёт: пользуемся тем, что перевод средств только между одним или двумя uid (и тем, что права уже проверили)
    my $agency_client_id = 0;
    if ($self->{rbac_login_rights}->{role} && $self->{rbac_login_rights}->{role} eq 'agency') {
        $agency_client_id = rbac_get_agency_clientid_by_uid( $self->{uid})
            || return {error => get_error_object("BadParams", iget("Клиент не найден"))};
    } elsif ($self->{rbac_login_rights}->{role} eq 'manager') {
        if ($self->{rbac_login_rights}->{is_any_teamleader}) {
            my $agency_uid = $compiled_params->{camps_names}->{$fromcid}->{agencyuid};
            $agency_client_id = rbac_get_agency_clientid_by_uid( $agency_uid)
                || return {error => get_error_object("BadParams", iget("Клиент не найден"))};
        } elsif (any {$_} map {$self->{_preprocessed}->{$_}->{agencyuid}} keys %{$self->{_preprocessed}}) {
            return {error => get_error_object("NoRights")};
        }
    } elsif ($self->{rbac_login_rights}->{role} eq 'super') {
        my $agency_uid = $compiled_params->{camps_names}->{$fromcid}->{agencyuid};
        $agency_client_id = rbac_get_agency_clientid_by_uid( $agency_uid)
            || return {error => get_error_object("BadParams", iget("Клиент не найден"))};
    }

    my $wallet_from = get_wallet_camp($client_uid, $agency_client_id, $currency_source);
    my $wallet_to = get_wallet_camp($client_uid_to, $agency_client_id, $currency_to);

    my $wallet_params = {};
    my $compiled_params_for_notifications = yclone($compiled_params);

    if ($wallet_from->{is_enabled}) { # если клиент использует общий счёт, нам без разницы, каким методом ему переносят деньги
        undef $compiled_params->{from};
        return {error => get_error_object('500')} unless $wallet_from->{wallet_cid}; # мы не должны заходить в этот метод, если нет кампании-счёта, но он включен
        $compiled_params->{from}->{$wallet_from->{wallet_cid}} = $compiled_params->{total_sum};
    }

    if ($wallet_to->{is_enabled}) { # это работает в предположении, что при наличии кампании и включенном общем счёте сама кампания общего счёта заведена
        undef $compiled_params->{to};
        return {error => get_error_object('500')} unless $wallet_to->{wallet_cid};
        $compiled_params->{to}->{$wallet_to->{wallet_cid}} = $compiled_params->{total_sum};
    }

    if (($wallet_to->{is_enabled} || $wallet_from->{is_enabled}) && ($client_uid == $client_uid_to)) {
        return {error => get_error_object('BadTransferMoneyRequest', iget('При использовании общего счёта возможен перенос только между счетами разных клиентов'))};
    }

    prepare_transfer_money($self->{rbac}, $client_uid, $compiled_params);

    my $error = validate_transfer_money($self->{rbac}, $self->{uid}, $compiled_params);

    if (!$error) {
        $error = process_transfer_money($self->{rbac}, $client_uid, $self->{uid}, $compiled_params, timeout => $FIN_OPS_API_TIMEOUT);
    }

    if ($error) {
        return {error => get_error_object('BadTransferMoneyRequest', $error)};
    }
    $compiled_params_for_notifications->{client_uid} = $client_uid;
    $compiled_params_for_notifications->{client_uid_to} = $client_uid_to;
    return {result => $compiled_params_for_notifications};
}

=head2 transfer_notification_common(self, params, compiled_params_for_notifications; O)

    Функция постановки уведомлений о переносе денег в очередь
    ожидает в params структуру {Payments => ..., PayMethod => ...}
    Payments — как в APICommon::create_pay_campaign_request
    compiled_params_for_notifications имеют структуру
    {
        to => {cid => sum, ...}    # sum -- сумма для переноса без НДС и без скидочного бонуса
        from => {cid => sum, ...}    # sum -- сумма для переноса без НДС и без скидочного бонуса
        client_uid => uid того, с чьих кампаний ушли деньги
        client_uid_to => uid того, на чьи кампаний ушли деньги
    }
    Если появится шаблон для общего счёта, выбор шаблона можно осуществить через четвёртый параметр

=cut

sub transfer_notification_common($$$;%) {
    my ($self, $params, $compiled_params_for_notifications) = @_;
    # параметры для шаблона письма
    my ($client_uid, $client_uid_to) = ($compiled_params_for_notifications->{client_uid}, $compiled_params_for_notifications->{client_uid_to});
    $compiled_params_for_notifications->{to_login} = get_login(uid => $client_uid_to);
    $compiled_params_for_notifications->{from_login} = get_login(uid => $client_uid);
    $compiled_params_for_notifications->{operator_role} = $self->{rbac_login_rights}->{role};
    $compiled_params_for_notifications->{currency} = ($self->{api_version_full} > 4)? ($params->{FromCampaigns}->[0]->{Currency} || 'YND_FIXED') : 'YND_FIXED';

    notification_api_finance($self->{uid}, $self->{rbac}, $self->{uid}, 'api_transfer_money', $compiled_params_for_notifications);
    return;
}
1;
