package API::Methods::Events;

# $Id$

=head1 NAME

    API::Methods::Events

=head1 DESCRIPTION

    Методы для работы с событиями и push нотификациями в API

=cut

use strict;
use warnings;
use POSIX qw/strftime/;

use JSON;

use Settings;
use EventLog;
use RBACDirect;
use Client;
use Property;

use APICommon;
use API::Errors;

use Currencies;
use Currency::Rate;
use HashingTools qw//;
use TextTools;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::DateTime;
use Yandex::I18n;
use Yandex::TimeCommon qw/mysql_round_day/;
use List::MoreUtils qw/uniq/;

use PrimitivesIds;

use utf8;

=head2 SaveSubscription

    Сохраняет подписку на уведомления

=cut

sub SaveSubscription
{
    my ($self, $params) = @_;

    if ($self->{rbac_login_rights}->{role} !~ m/client/) {
        dieSOAP('NoRights', iget('Метод доступен только клиентам'));
    }

    my $uid = $self->{uid};
    my %reversed_tokens = reverse %APICommon::EVENT_TYPE2TOKEN;

    my @converted_event_types;
    foreach my $event_type (uniq @{$params->{EventTypes}}) {
        push @converted_event_types, $reversed_tokens{$event_type};
    }

    $params->{EventTypes} = \@converted_event_types;

    $params->{Tag} = HashingTools::generate_push_uuid($params->{SubscriptionResource}); # нужен для запасного способа подписки и поиска уже существующих подписок
    $params->{Uuid} //= $params->{Tag};

    my $data = to_json(hash_cut $params, qw/Locale Timeout EventTypes Options Uuid DeviceId Tag/);

    do_replace_into_table(PPC(uid => $uid), 'users_notifications_register_queue',
                        {uid => $uid,
                        type => $params->{SubscriptionType},
                        resource => $params->{SubscriptionResource},
                        data => $data,
                        action => 'add'
                        });

    return 1;
}

=head2 DeleteSubscription

    Удаляет подписку на уведомления

=cut

sub DeleteSubscription
{
    my ($self, $params) = @_;

    if ($self->{rbac_login_rights}->{role} !~ m/client/) {
        dieSOAP('NoRights', iget('Метод доступен только клиентам'));
    }

    my $uid = $self->{uid};
    my $tag = HashingTools::generate_push_uuid($params->{SubscriptionResource});
    my $temp_data = {
        Uuid => $params->{Uuid} // $tag,
        Tag => $tag,
        DeviceId => $params->{DeviceId},
    };

    my $data = to_json($temp_data);

    do_replace_into_table(PPC(uid => $uid), 'users_notifications_register_queue',
                        {uid => $uid,
                        type => $params->{SubscriptionType},
                        resource => $params->{SubscriptionResource},
                        data => $data,
                        action => 'remove'
                        });

    return 1;
}

=head2 GetSubscription

    Метод больше для отладки. Возвращает текущие параметры подписки.

=cut

sub GetSubscription
{
    my ($self, $params) = @_;

    if ($self->{rbac_login_rights}->{role} !~ m/client/) {
        dieSOAP('NoRights', iget('Метод доступен только клиентам'));
    }

    my $uid = $self->{uid};
    my $tag = HashingTools::generate_push_uuid($params->{SubscriptionResource});

    my @conditions = (tag => $tag);
    push @conditions, (device_id => $params->{DeviceId}) if ($params->{DeviceId});

    my $subscription = get_one_line_sql(PPC(uid => $uid), ['select * from users_notifications',
                        where => {
                                    type => $params->{SubscriptionType},
                                    _OR => \@conditions,
                                    uid => $uid
                                 }, "ORDER BY notification_id DESC LIMIT 0,1"]);
    if ($subscription) {
        my $details = get_one_column_sql(PPC(uid => $uid), ['select event_type from users_notifications_details',
                            where => {notification_id => $subscription->{notification_id}}, 'order by event_type']) || [];

        return {
            SubscriptionResource => $params->{SubscriptionResource},
            SubscriptionType => $params->{SubscriptionType},
            Timeout => $subscription->{timeout},
            Locale => $subscription->{locale},
            Options => $subscription->{options},
            EventTypes => [map {$APICommon::EVENT_TYPE2TOKEN{$_}} @$details],
        };
    } else {
        dieSOAP('NotificationIdNotFound');
    }

}

=head2 GetEventsLog

    Метод для получения лога событий

=cut

sub GetEventsLog
{
    my ($self, $params) = @_;

    my %token2eventtype = reverse(%APICommon::EVENT_TYPE2TOKEN);

    my $timestamp_from = iso8601_2_mysql($params->{TimestampFrom});
    my $timestamp_to = $params->{TimestampTo}
        ? iso8601_2_mysql($params->{TimestampTo})
        : strftime("%Y%m%d", localtime(time() + 86400));
    my $client_uids = $self->{rbac_login_rights}{role} =~ /^(agency|super|support|superreader|manager)$/
        ? get_uids(login => $params->{Logins})
        : [$self->{uid}];

    dieSOAP( 'NoRights' ) if $self->{uid} != $client_uids && ! rbac_mass_is_owner($self->{rbac}, $self->{uid}, $client_uids);

    my $client_ids = get_clientids(uid => $client_uids);

    my $filter;

    if ($params->{Filter}{CampaignIDS} && scalar @{$params->{Filter}{CampaignIDS}}) {
        $filter->{cid} = $params->{Filter}{CampaignIDS};
    }

    if ($params->{Filter}{BannerIDS} && scalar @{$params->{Filter}{BannerIDS}}) {
        $filter->{bid} = $params->{Filter}{BannerIDS};
    }

    if ($params->{Filter}{PhraseIDS} && scalar @{$params->{Filter}{PhraseIDS}}) {
        $filter->{bids_id} = $params->{Filter}{PhraseIDS};
    }

    if ($params->{Filter}{AccountIDS} && scalar @{$params->{Filter}{AccountIDS}}) {
        if (exists $filter->{cid} && scalar(@{$filter->{cid}}) > 0) {
            push @{$filter->{cid}}, @{$params->{Filter}{AccountIDS}};
        }else{
            $filter->{cid} = $params->{Filter}{AccountIDS};
        }
    }
    add_account_filters($params->{Filter}{EventType});

    if ($params->{Filter}{EventType} && scalar @{$params->{Filter}{EventType}}) {
        $filter->{type} = [map {$EventLog::EVENTS{$token2eventtype{$_}}{type}} @{$params->{Filter}{EventType}} ];
    } else {
        $filter->{type} = \@EventLog::API_EVENTS;
    }

    my %eventlog_params = (
        ClientID => $client_ids,
        date_from => $timestamp_from,
        date_to => $timestamp_to,
        conditions => $filter,
    );

    if ($params->{LastEventOnly} && $params->{LastEventOnly} eq 'Yes') {
        $eventlog_params{only_last_event} = 1;
    }

    if (defined $params->{Limit}) {
        $eventlog_params{limit} = $params->{Limit};
        if (defined $params->{Offset}) {
            $eventlog_params{offset} = $params->{Offset};
        }
    }

    my $events = EventLog::get_events(%eventlog_params);

    my $client_nds_schedules = mass_get_client_nds_schedule($client_ids);

    my $text_descrs;

    if ($params->{WithTextDescription} && $params->{WithTextDescription} eq 'Yes') {
        $text_descrs = EventLog::get_events_text_descr($events);
    }

    my @events_parsed;
    my $timezone = DateTime::TimeZone->new(name=>'local');

    my $cids = [map{ $_->{cid} if defined $_->{cid} }@{$events}];

    my $wallet_cids = get_hashes_hash_sql(PPC(cid => $cids),
            ["SELECT cid FROM campaigns", where => {cid => SHARD_IDS, type__in => ['wallet']}]);

    my $bids = [map{ $_->{bid} if defined $_->{bid} }@{$events}];
    my $deleted_bids = get_hash_sql(PPC(ClientID => $client_ids), ["SELECT bid,1 FROM deleted_banners b", where => {bid => $bids}]);

    my $CHANGE_NOTIFICATION_MESSAGES_FOR_DAYS_WARNING_WALLET = Property->new('CHANGE_NOTIFICATION_MESSAGES_FOR_DAYS_WARNING_WALLET');

    foreach my $event (@$events) {

        # -- Отфильтровываем события по удаленным банерам, иначе ломается Коммандер DIRECT-27218
        next if defined $event->{bid} && exists $deleted_bids->{$event->{bid}};

        my $is_wallet  = exists $wallet_cids->{$event->{cid}} ? 1 : 0;

        my $event_name = $EventLog::EVENTS{$event->{slug}}{name};

        my $parsed_event = {
                            !$is_wallet ? (BannerID => $event->{bid} || undef) : (),
                            !$is_wallet ? (PhraseID  => $event->{bids_id} || undef):(),
                            # -- wallet company ?
                            $is_wallet ? (
                                AccountID => $event->{cid}
                            ):
                            (
                                CampaignID => $event->{cid} || undef
                            ),
                            EventType => $APICommon::EVENT_TYPE2TOKEN{$event->{slug}},
                            EventName => iget($event_name)
                           };

        if ($params->{WithTextDescription} && $params->{WithTextDescription} eq 'Yes') {
            $parsed_event->{TextDescription} = $text_descrs->{$event->{id}}{descr};
        }

        $parsed_event->{Timestamp} = mysql2iso8601($event->{eventtime}, $timezone);

        foreach my $attr (keys %{$event->{params}}) {
            if ($attr =~ /^(?:finish_date|old_place)$/) {
                $parsed_event->{Attributes}{$APICommon::EVENT_ATTR2TOKEN{$attr}} = $event->{params}{$attr};
            }
            if ($attr =~ /^(?:sum_rest|sum_payed|new_cpm_limit)$/) {
                # multicurrency - ok!
                my $export_currency = $params->{Currency} || 'YND_FIXED';
                my $value = $event->{params}{$attr};

                if ($value) {

                    my $from_currency = $event->{params}{currency} || $event->{params}{new_cpm_limit_currency} || "YND_FIXED";
                    if ($from_currency ne "YND_FIXED") {
                        my $client_nds = _get_client_nds_on_date($client_nds_schedules->{$event->{ClientID}}, mysql_round_day($event->{eventtime}));
                        $value = remove_nds($value, $client_nds);
                    }
                    my $converted_value = convert_currency($value, $from_currency, $export_currency);
                    if ($attr eq 'new_cpm_limit') {
                        $value = round_price_to_currency_step( $converted_value, $export_currency, up => 1 );
                    } else {
                        $value = round2s($converted_value);
                    }

                }

                $parsed_event->{Attributes}{$APICommon::EVENT_ATTR2TOKEN{$attr}} = $value;
                $parsed_event->{Attributes}{Currency} = $export_currency if $export_currency ne 'YND_FIXED';
            }
            if ($attr eq 'bs_stop_time') {
                $parsed_event->{Attributes}{$APICommon::EVENT_ATTR2TOKEN{$attr}} = mysql2iso8601($event->{params}{$attr}, $timezone);
            }
            if ($attr eq 'is_edited_by_moderator') {
                $parsed_event->{Attributes}{$APICommon::EVENT_ATTR2TOKEN{$attr}} = $event->{params}{$attr} ? 'Yes' : 'No';
            }
            if ($event->{type} == $EventLog::EVENTS{banner_moderated}{type}) {
                $parsed_event->{Attributes}{ModerationResult} = $APICommon::EVENT_MODRES2TOKEN{$event->{params}{results}{global}};
            }
            if ($event->{type} == $EventLog::EVENTS{money_warning_wallet}{type} 
                    && $attr eq 'days_left'
                    && $CHANGE_NOTIFICATION_MESSAGES_FOR_DAYS_WARNING_WALLET->get(60)) {

                my $value = $event->{params}{$attr};
                if ($value eq 'ONE_DAY_REMAIN') {
                    $event_name = iget_noop('При текущем уровне расходов средств на счёте хватит на 1 день');
                } elsif ($value eq 'THREE_DAYS_REMAIN') {
                    $event_name = iget_noop('При текущем уровне расходов средств на счёте хватит на 3 дня');
                }
                $parsed_event->{EventName} = iget($event_name);

            }
        }
        push @events_parsed, $parsed_event;
    }

    return \@events_parsed;
}

=head2 _get_client_nds_on_date($nds_schedule, $date)

    По графику НДС клиента определяем НДС на указанную дату

=cut


sub _get_client_nds_on_date {
    my ($nds_schedule, $date) = @_;

    return undef if !$nds_schedule || !$date;

    foreach my $row (@$nds_schedule) {
        return $row->{nds} if $date >= $row->{date_from} && $date <= $row->{date_to};
    }

    return undef;
}

=head2 add_account_filters($@filters)

    Принимает список фильтров, добавляет в него фильтры по событиям обшего счета
    если пользователь указал фильтрацию по событиям MoneyOut, MoneyWarning или MoneyIn
    то автоматически добавятся события MoneyOutAccount, MoneyWarningAccount и MoneyInAccount
    сделано для обратной совместимости


=cut

sub add_account_filters($){
    my $filters = shift;
    my %filters = map {$_ => 1} @$filters;
    for my $eventtype (keys %filters){
        if (($eventtype eq 'MoneyOut' || $eventtype eq 'MoneyWarning' || $eventtype eq 'MoneyIn')
            && !exists $filters{$eventtype.'Account'}
        ) {
            push @$filters, $eventtype.'Account';
        }
    }
    return $filters;
}

1;
