package API::Methods::Finance;

# $Id$

=head1 NAME

    API::Methods::Finance

=head1 DESCRIPTION

    Финансовые методы в API

=cut

use Direct::Modern;

use Data::Dumper;

use Yandex::Balance qw/
    balance_get_client_credit
    /;

use Yandex::I18n;

use Settings;
use APICommon qw(:subs);
use API::Errors;
use Campaign qw/get_camp_info get_camp_sum_available/;
use Primitives;
use PrimitivesIds;
use Client;
use Wallet;
use DirectContext;
use RBACElementary;
use RBACDirect;
use TextTools;
use Currencies;
use Currency::Rate;
use API::Methods::Common;
use BalanceQueue;
use LogTools qw/log_stacktrace/;

use Models::CampaignOperations qw/check_block_money_camp/;
use List::MoreUtils qw/uniq firstval/;

sub GetBalance
{
    my ($self, $cids) = @_;
   
    log_stacktrace("api_old_code", "GetBalance");
    my $result = [];

    my $camps = get_camp_info($cids);
    my @client_ids = uniq map { $_->{ClientID} } @$camps;
    my $clients_discounts = mass_get_client_discount(\@client_ids);
    my $clients_nds = mass_get_client_NDS(\@client_ids);
    my @wallet_cids = grep {$_} map {$_->{wallet_cid}} @$camps;
    my $wallets = get_all_wallet_camps(cid => \@wallet_cids);

    foreach my $camp (@$camps) {

        # multicurrency
        my $export_currency = $self->{api_version_full} < 5 ? 'YND_FIXED' : $camp->{currency};
        if ($camp->{wallet_cid}) {
            my $record = firstval {$_->{wallet_cid} == $camp->{wallet_cid}} @$wallets;
            if ($record->{is_enabled}) {
                $camp->{wallet_total} = $record->{wallet_total};
                $camp->{wallet_enabled} = 1;
            }
        }

        if ($export_currency ne 'YND_FIXED') { # данные в валюте не включают НДС и скидочный бонус
            campaign_remove_nds_and_add_bonus($camp
            , client_nds => $clients_nds->{$camp->{ClientID}}
            , client_discount => undef);
        } elsif ($camp->{currency} ne 'YND_FIXED') { # данные при конвертации в фишки включают НДС фишек, а не клиента.
            campaign_remove_nds_and_add_bonus($camp
            , client_nds => $clients_nds->{$camp->{ClientID}}
            , client_discount => ($clients_discounts->{$camp->{ClientID}} || 0));
        }

        my $camp_own_rest = $camp->{sum} - $camp->{sum_spent};
        $camp_own_rest = 0 if $camp_own_rest < 0;
        my $item = {
            CampaignID => $camp->{cid},
            Rest => ($camp->{wallet_enabled})? ($camp_own_rest + $camp->{wallet_total}) : $camp_own_rest,
            Sum => ($camp->{wallet_enabled})? $camp->{sum_spent} : $camp->{sum},
        };

        if ($self->{api_version_full} > 4) { # latest ok
            if ($camp->{wallet_enabled}) {
                # -- Не показываем для кошельков сумму переноса DIRECT-26163
                $item->{SumAvailableForTransfer} = undef;
            } else {
                $camp->{money_type} = check_block_money_camp($camp) ? 'blocked' : 'real';
                my $available = get_camp_sum_available($camp, sums_without_nds => 1, dont_round => 1);
                $item->{SumAvailableForTransfer} = $available >= 0 ? $available : undef;

                # Сумма для переноса возвращяется без НДС и скидочного бонуса, поэтому при конвертации валюты в фишки нужно доначислить бонус
                if ($export_currency eq 'YND_FIXED' && $camp->{currency} ne 'YND_FIXED') {
                    $item->{SumAvailableForTransfer} = Currencies::add_bonus($item->{SumAvailableForTransfer}, $clients_discounts->{$camp->{ClientID}} || 0);
                }
            }
        }

        if ($export_currency) {
            foreach my $f (qw/Sum Rest SumAvailableForTransfer/) {
                next unless $item->{$f};

                if (($export_currency eq 'YND_FIXED') && ($camp->{currency} ne 'YND_FIXED')) {

                    # НДС начисляется автоматически при конвертации в фишки внутри функции
                    $item->{$f} = convert_currency($item->{$f}, $camp->{currency}, $export_currency);
                }

                if ($self->{api_version_full} > 4) {
                    $item->{Currency} = $export_currency if $export_currency ne 'YND_FIXED';
                }
            }
        }

        foreach my $f (qw/Sum Rest SumAvailableForTransfer/) {
            next unless $item->{$f};

            $item->{$f} = round2s($item->{$f});
        }

        push @$result, $item;
    }

    return $result;
}

# multicurrency: текущий код будет рассматривать для мультивалютных кампаний переданные суммы как суммы без НДС
sub TransferMoney
{
    my ($self, $params) = @_;

    my @cids = map {$_->{CampaignID}} @{$params->{ToCampaigns}};

    # проверяем наш технологический лимит
    foreach my $cid (@cids) {
        api_check_limit_by_method( $self, $cid, 'FinOperations__daily', 1);
    }

    my $result = API::Methods::Common::transfer_common($self, $params);
    if ($result->{error}) {
        dieSOAP_by_object($result->{error});
    }
    my $compiled_params_for_notifications = $result->{result};

    # апдейтим наш технологический лимит
    foreach my $cid (@cids) {
        api_update_limit( $self, $cid, 'FinOperations__daily', 1);
    }

    API::Methods::Common::transfer_notification_common($self, $params, $compiled_params_for_notifications);

    my $syslog_data = get_syslog_data({cid => \@cids});

    $self->{syslog_data}->{cid} = \@cids;
    $self->{cluid} = $syslog_data->{cluid};

    return 1;
}

=head2 CreateInvoice

    Метод для выставления счетов.
    Входные параметры:
        {
            Payments =>
            [
                { CampaignID => 123, Sum => 500 },
                { CampaignID => 124, Sum => 500 },
            ]
        }

    Возвращает url в случае успеха, или ошибку

=cut

sub CreateInvoice
{
    my ($self, $params) = @_;

    my @cids = map {$_->{CampaignID}} @{$params->{Payments}};

    # проверяем наш технологический лимит
    foreach my $cid (@cids) {
        api_check_limit_by_method( $self, $cid, 'FinOperations__daily', 1);
    }

    # multicurrency: ok!
    my $billing_url = create_pay_campaign_request($self, $params->{Payments});

    if ($billing_url->{error}) {
        dieSOAP_by_object($billing_url->{error});
    }

    if ($billing_url->{result}) {
        # апдейтим  наш технологический лимит
        foreach my $cid (@cids) {
            api_update_limit( $self, $cid, 'FinOperations__daily', 1);
        }

        my $syslog_data = get_syslog_data({cid => \@cids});

        $self->{syslog_data}->{cid} = \@cids;
        $self->{cluid} = $syslog_data->{cluid};

        return $billing_url->{result};
    }
}

=head2 PayCampaigns

    Метод для оплаты кампаний с помощью кредитных лимитов и овердрафта.
    В будущем Биллинг будет готов принимать платежи через банковские карты (говорили об этом в январе 2012 года).

    Способы оплаты: пока принимает только значения - Bank/Overdraft (плюс недокументированый способ - YM).
        Метод оплаты через ЯД не документирован, потому что он происходит не очень хорошим способом - транслированием через Директ платежного токена ЯДа (внешне конечно Яндекс един, но все же платежный токен дело самих Денег) - об этом была договоренность с СБ.

    Входные параметры:
    {
        ContractID - идентификатор договора, получаемый из GetCreditLimits
        PayMethod - метод оплаты кампаний.
        Payments - массив хэшей [{CampaignID => 123123, Sum=>345345}, ...]
    }

    Возвращает 1 или 0

=cut

sub PayCampaigns
{
    my ($self, $params) = @_;

    # проверяем наш технологический лимит
    foreach my $payment (@{$params->{Payments}}) {
        api_check_limit_by_method( $self, $payment->{CampaignID}, 'FinOperations__daily', 1);
    }

    my $result = API::Methods::Common::payment_common($self, $params);

    if ($result->{error}) {
        dieSOAP_by_object($result->{error});
    }

    if ($result->{result}) {
        # апдейтим наш технологический лимит
        foreach my $payment (@{$params->{Payments}}) {
            api_update_limit( $self, $payment->{CampaignID}, 'FinOperations__daily', 1);
        }

        # отправляем письма про оплату в кредит
        # TODO: нужно ли отправлять письмо при оплате картой?
        API::Methods::Common::pay_notification_common($self, $params);

        my $syslog_data = get_syslog_data({cid => [map {$_->{CampaignID}} @{$params->{Payments}}]});

        $self->{syslog_data}->{cid} = $syslog_data->{cid};
        $self->{cluid} = $syslog_data->{cluid};
        return $result->{result};
    }
}

=head2 PayCampaignsByCard

    Метод для оплаты кампаний с помощью карт.

    Вызывает PayCampaigns со способом оплаты LinkedCard.

    Входные параметры:
    {
        PayMethodID - идентификатор транзакции, генерируется клиентом, уникален в рамка всех его платежных транзакций
        CustomTransactionID - идентификатор транзакции, генерируется клиентом, уникален в рамка всех его платежных транзакций,
        Payments - массив хэшей [{CampaignID => 123123, Sum=>345345, Currency => 'RUB'}, ...]
    }

    Возвращает 1 или 0

=cut

sub PayCampaignsByCard
{
    my ($self, $params) = @_;

    $params->{PayMethod} = 'LinkedCard';

    return PayCampaigns($self, $params);
}

=head2 CheckPayment

    Метод для проверки оплаты кампании с помощью карт.

    Входные параметры:
    {
        CustomTransactionID - идентификатор транзакции, переданный при вызове метода PayCampaignsByCard
    }

    Возвращает статус оплаты или ошибку.

=cut

sub CheckPayment
{
    my ($self, $params) = @_;

    # # проверяем наш технологический лимит
    # foreach my $payment (@{$params->{Payments}}) {
    #     api_check_limit_by_method( $self, $payment->{CampaignID}, 'FinOperations__daily', 1);
    # }

    my $result = API::Methods::Common::check_payment_common($self, $params);
    if ($result->{error}) {
        dieSOAP_by_object($result->{error});
    }

    # # апдейтим наш технологический лимит
    # foreach my $payment (@{$params->{Payments}}) {
    #     api_update_limit( $self, $payment->{CampaignID}, 'FinOperations__daily', 1);
    # }

    return $result->{result};
}

=head2 GetCreditLimits

    Для текущего клиента возвращает список договоров и кредитных лимитов по ним

=cut

sub GetCreditLimits
{
    my $self = shift;

    my $client_id = $self->{ClientID} || get_clientid(uid => $self->{uid});
    $self->{cluid} = [$self->{uid}];

    my $client_currencies = get_client_currencies($client_id);
    my $client_info = get_client_data($client_id, [qw/is_using_quasi_currency/]);

    my $product = product_info(type => 'text', currency => $client_currencies->{work_currency}, quasi_currency => $client_info->{is_using_quasi_currency});

    my $limits_data;
    eval {
        $limits_data = balance_get_client_credit($client_id, $product->{ProductID});
    };

    if ($@) {
        warn Dumper $@;
        dieSOAP('FinOpsTmpUnavail');
    }

    if ($limits_data && ref($limits_data) eq 'HASH') {

        my $result = {};
        foreach my $limit (@{$limits_data->{LIMITS}}) {
            push @{$result->{Limits}}, {ContractID => $limit->{CONTRACT_EID}, Limit => $limit->{LIMIT_TOTAL}, LimitSpent => $limit->{LIMIT_SPENT}};
        }

        $result->{Currency} = $limits_data->{CURRENCY};

        # multicurrency: конвертация не требуется - метод всегда работал с валютой договора агентства
        return $result;

    } else {
        dieSOAP('BadParams');
    }
}

sub EnableSharedAccount
{
    my ($self, $params) = @_;

    my $client_uid = get_uid_by_login2($params->{Login});
    my $clientid = get_clientid(uid => $client_uid);

    my $agency_client_id = get_clientid(uid => $self->{uid});

    $self->{ret} = {};

    if (!defined $agency_client_id) {
        push @{$self->{ret}->{Errors}}, get_error_object('NoRights');
    }

    return $self->{ret} if $self->{ret}->{Errors} && scalar @{$self->{ret}->{Errors}};

    # Agency
    my $agency_chief_uid = 0; # -- не придется отдельно проверять роль "супера", только "агентство".DIRECT-26270
    if ($self->{rbac_login_rights}{role} eq 'agency') {

        my $agency_id = rbac_is_agency_client_id_of_client($self->{rbac}, $agency_client_id, $client_uid);

        if (!defined $agency_id) {
            push @{$self->{ret}->{Errors}}, get_error_object('NoRights', iget('Не является клиентом агентства'));
            return $self->{ret};
        }
        $agency_chief_uid = rbac_get_chief_rep_of_agency($agency_id);
    }

    # Client
    dieSOAP('NoRights', APICommon::msg_converting_in_progress) if Client::is_client_converting_soon($clientid);
    dieSOAP('NoRights', APICommon::msg_must_convert) if Client::client_must_convert($clientid);

    my $client_chief_uid    = rbac_get_chief_rep_of_client($clientid);
    my $client_currencies   = get_client_currencies($clientid, allow_initial_currency => 1, uid => $client_uid);

    # Теперь в поле client_chief_uid именно uid главного представителя
    my $c = DirectContext->new({
        is_direct        => 1,
        uid              => $client_uid,
        UID              => $self->{uid},
        client_chief_uid => $client_chief_uid,
        client_client_id => $clientid,
        rbac             => $self->{rbac},
        login_rights     => $self->{rbac_login_rights},
    });

    my $r = Wallet::enable_wallet(
        $c,
        $client_currencies->{work_currency},
        $agency_chief_uid
    );

    if( defined $r->{error}) {
        if (exists($r->{code})) {

            if ($r->{code} == 519) {
                push @{$self->{ret}->{Errors}}, get_error_object('WalletExists', iget($r->{error}));
            } elsif ($r->{code} == 520) {
                push @{$self->{ret}->{Errors}}, get_error_object('CantEnableWalletNoCampaigns', iget($r->{error}));
            } elsif ($r->{code} == 521) {
                push @{$self->{ret}->{Errors}}, get_error_object('AccountIsDisabledByAgency', iget($r->{error}));
            } else {
                push @{$self->{ret}->{Errors}}, get_error_object('NoRights', iget($r->{error}));    
            }

        } else {
            push @{$self->{ret}->{Errors}}, get_error_object('NoRights', iget($r->{error}));
        }
        return $self->{ret};
    }

    @{$self->{ret}}{qw/Login AccountID/} = ($params->{Login}, $r->{wallet_cid});

    BalanceQueue::add_to_balance_info_queue($self->{uid}, 'cid', $r->{wallet_cid}, BalanceQueue::PRIORITY_CAMP_ON_ENABLE_WALLET_API);

    $self->{cluid} = [$client_uid];
    $self->{syslog_data}->{cid} = [$r->{wallet_cid}];

    return $self->{ret};
}

1;
