package API::Methods::Keyword;

# $Id$

=head1

    Модуль для методов, работающих с ключевыми словами

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq any/;

use Yandex::I18n;

use Settings;

use BannersCommon;
use PhraseDoubles;
use Lang::Unglue;
use PhraseText;

use Direct::Validation::Keywords qw/base_validate_keywords/;
use API::Methods::Keyword::ActionResults;
use API::Settings;


=head2 RearrangeKeywords
    Метод обрабатывает список фраз аналогично тому, как они обрабатываются при добавлении банера и выводит результом список обработанных фраз

    Request: {
        Keywords : [ (string), ..., (string) ],
        AutoMinusWords : "Yes" /* optional */
    }

    Response: {
        Keywords : [ (string), ..., (string) ]
    }
=cut

sub RearrangeKeywords($$) {
    my ($self, $params) = @_;
    my $keywords = [
        map { $_->{'phrase'} }
        glue_minus_words(
            _prepare_for_glue_minus_words( $params->{Keywords} )
        )
    ];
    if (defined $params->{AutoMinusWords}
            && $params->{AutoMinusWords} eq 'Yes')
    {
        my $unglued = _prepare_for_unglue_phrases($keywords);
        my $status = unglue_phrases( $unglued, -1 );
        $keywords = [
            map {
                $_->{phrase}
                    . (
                        $status
                            ? ($_->{phrase_unglued_suffix}||'')
                            :''
                        )
            } @{ $unglued->[0]->{Phrases} }
        ];

    }
    return { Keywords => $keywords };
}

=head2 _validate_rearrange_keywords_rights

    Проверка прав на вызов RearrangeKeywords
    
=cut

sub _validate_rearrange_keywords_rights($) {
    my $self = shift;
    # никому кроме пользователей из списка
    return ('AccessDenied') unless is_in_rearrange_keywords_allow_apps($self->{application_id});
    return;
}

=head2 функция is_in_rearrange_keywords_allow_apps

    проверяет доступен ли приложению метод RearrangeKeywords

=cut

sub is_in_rearrange_keywords_allow_apps {
    my $application_id = shift;
    my @apps = @$API::Settings::REARRANGE_KEYWORDS_ALLOW_APPS;
    if (any { $_ eq $application_id } @apps) {
        return 1;
    }
    return 0;
}

=head2 _validate_rearrange_keywords
    Проверка данных для RearrangeKeywords
=cut

sub _validate_rearrange_keywords($$) {
    my ($self, $params) = @_;

    my $validation_result = base_validate_keywords($params->{Keywords}); 
    return (BadParams => $validation_result->get_first_error_description) unless $validation_result->is_valid; 
    
    return;
}

=head2 _prepare_for_glue_minus_words 
    Приводит массив строк (ключевых слов), к виду 
    [
        { phrase => 'keyword1', is_new => 0|1 }
        ..
        { phrase => 'keywordn', is_new => 0|1 }
    ]
    готовому для скармливания методу Lang::Unglue::glue_minus_words
=cut

sub _prepare_for_glue_minus_words {
    my $keywords = shift;
    [
        map {
            { phrase => $_, is_new => '1' }
        } @$keywords
    ]
}

=head2 _prepare_unglue_phrases
    Приводит массив строк (ключевых слов), к виду 
    [
        {
            Phrases => [
                { phrase => 'keyword1', md5sum => '..' }
                ..
                { phrase => 'keywordn', md5sum => '..' }
            ]
        }
    ]
    готовому для скармливания методу Lang::Unglue::unglue_phrases
    (эмулируются фразы упаковынные в баннеры при пакетном добавлении банеров)
=cut

sub _prepare_for_unglue_phrases {
    my $keywords = shift;
    return [
        {
            Phrases => [
                map {
                    { phrase => $_, md5 => get_phrase_props($_)->{md5} }
                } @$keywords
            ]
        }
    ]
}


sub GetKeywordsIntersection {
    my ($self, $params) = @_;

    my $phrases;
    my $with_phrases;
    if ($params->{WithCampaignIDS}) {
        my ($banners, undef) = get_banners({cid=>$params->{WithCampaignIDS}, arch_banners => 0}, {no_pokazometer_data=>1});
        $with_phrases = [map {@{$_->{phrases} || []}} @$banners];
    } elsif ($params->{WithKeywords}) {
        $with_phrases = [map { {phrase=>$_} } @{$params->{WithKeywords}}];
    }
    if ($params->{CampaignIDS}) {
        my ($banners, undef) = get_banners({cid=>$params->{CampaignIDS}, arch_banners => 0}, {no_pokazometer_data=>1});
        $phrases = [map {@{$_->{phrases} || []}} @$banners];
    } elsif ($params->{Keywords}) {
        $phrases = [map { {phrase=>$_} } @{$params->{Keywords}}];
    }

    my $result;
    if ($with_phrases) {
        $result = get_all_doubles($with_phrases, phrases_for_check => $phrases);
    } else {
        $result = get_all_doubles($phrases);
    }
    
    my @response;
    while (my ($keyword, $doubles) = each %$result) {
        push @response, [map { {Keyword => $_->{phrase}, KeywordID => $_->{id}} } @$doubles];
    }
    return {IntersectingKeywords => \@response};
}

1;
