package API::Methods::Mediaplan;

# $Id: KeywordCategory.pm 34890 2012-09-13 13:21:54Z metallic $

=head1 NAME
    
    API::Methods::Keyword

=head1 DESCRIPTION

    Методы для работы с ключевыми словами и категориями

=cut

use strict;
use warnings;
use utf8;

use feature 'state';

use Yandex::DBTools;
use Yandex::I18n;
use Yandex::HashUtils;
use Yandex::DBShards qw/SHARD_IDS/;

use Settings;
use API::Errors;
use API::ConvertFrom qw(convert_params);
use API::Filter;
use Mediaplan;
use VCards;
use OrgDetails;
use CommonMaps;
use Sitelinks;
use RBACElementary;
use MinusWords;
use PlacePrice qw//;

use PrimitivesIds;

use BannersCommon qw/get_banners_short_hash/;

=head2 position_to_place

    Метод обратный API::Filter::place_to_mediaplan_position

=cut

sub position_to_place {
    my $position = shift;
    # Маппим апишные значения в словарь понятный PlacePrice
    state $place_map = {
        FirstPlaceGuarantee => $PlacePrice::PLACES{GUARANTEE1},
        Guarantee => PlacePrice::get_guarantee_entry_place(),
        Premium => PlacePrice::get_premium_entry_place()
    };
    return $place_map->{$position} || $PlacePrice::PLACES{ROTATION};
}

sub MediaplanWrapper {
    my ($self, $params) = @_;
    my $result = {};
    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Finish') {
        my $criteria = $params->{SelectionCriteria};
        my $data = get_mediaplans_stats($params->{SelectionCriteria}{CampaignIDS}, $self->{rbac});

        if ($params->{Action} eq 'Get') {
            foreach my $cid (keys %$data) {
                my $mediaplan = $data->{$cid};

                push @{$result->{Mediaplans}}, {
                    CampaignID => $mediaplan->{cid},
                    defined $mediaplan->{manager_login} ? (ManagerLogin => $mediaplan->{manager_login}) : (),
                    MediaplannerLogin => $mediaplan->{mediaplanner_login} || undef,
                    MediaplanStatus => $mediaplan->{mediaplan_status}
                };
            }
        } elsif ($params->{Action} eq 'Finish') {

            my $mediaplans_in_process = get_hashes_hash_sql(PPC(cid => $params->{SelectionCriteria}{CampaignIDS}), [
                "SELECT cid, COUNT(ms.mpid) as cnt, c.uid 
                FROM mediaplan_stats ms 
                LEFT JOIN campaigns c using (cid)
                ", 
                where => {cid => SHARD_IDS, accepted => 'No'},
                "group by cid"
                ]);

            my $res;
            foreach my $cid (@{$params->{SelectionCriteria}{CampaignIDS}}) {
                $res->{$cid}{CampaignID} = $cid;
                if (!$mediaplans_in_process->{$cid}{cnt}) {
                    push @{$res->{$cid}{Errors}}, get_error_object('BadMediaplan', iget("Для кампании %d не найден медиаплан для завершения", $cid));
                } else {
                    my $mediaplan = get_user_mediaplan($mediaplans_in_process->{$cid}{uid}, $cid, {no_forecast => 1, is_api => 1});
                    my $result_check = validate_mediaplan($mediaplan, use_multierrors_format => 1);

                    for my $k (keys %$result_check) {
                        if ($k eq 'common') {
                            for my $error (@{$result_check->{$k}}) {
                                push @{$res->{$cid}{Errors}}, get_error_object('BadMediaplan', $error);
                            }
                        } else {
                            for my $field (keys %{$result_check->{$k}}) {
                                for my $error (@{$result_check->{$k}{$field}}) {
                                    push @{$res->{$cid}{Errors}}, get_error_object('BadMediaplanAd', iget("%s (%s) в объявлении медиаплана %d", $error, $field, $k));
                                }
                            }
                        }
                    }
                }
            }

            # уже провалидировано, можем завершить все медиапланы из запроса
            foreach my $cid (keys %$data) {
                if (! $res->{$cid}{Errors} || ! scalar @{$res->{$cid}{Errors}}) {
                    end_mediaplan($self->{rbac}, $cid, $params->{Options}{Comment}, uid => $self->{uid});
                }
            }
            $result->{ActionsResult} = [values %$res];

        }
    }
    return $result;
}

sub MediaplanAdGroupWrapper {
    my ($self, $params) = @_;

}

=head2 MediaplanAdWrapper

    Добавление/Обновление баннера медиаплана

    При указание SourceAdID при добавлении, баннер источник подцепится автоматически, при обновлении проигнорируется,
    но на всякий случай поле явно удаляется.

=cut

sub MediaplanAdWrapper {
    my ($self, $params) = @_;
    my $result = {};
    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        my $criteria = $params->{SelectionCriteria};
        # получаем баннеры медиаплана без фраз
        my $mediaplan_banners = get_mediaplan_banners(
            $criteria->{CampaignIDS} || [], 
            {
                bids => $criteria->{MediaplanAdIDS} || [],
                no_phrases => 1,
                not_add_minus_to_minus_words => 1,
                is_api => 1,
            }
        );

        if ($params->{Action} eq 'Get') {
            $result->{MediaplanAds} = [];
            foreach my $mbanner (@$mediaplan_banners) {
                $mbanner->{MediaplanAdID} = $mbanner->{mbid};
                if ($mbanner->{source_bid}) {
                    $mbanner->{SourceAdID} = $mbanner->{source_bid};
                }

                push @{$result->{MediaplanAds}}, filter_banner_object( $self, $mbanner, mediaplan => 1 );
            }
            
        # Используем данные из предыдущего шага, не очень хорошо, можно вытащить все данные за один запросы, вместо
        # 1 + banners_num. Но, кажется, что операция не частая, и можно просто код реюзнуть
        } elsif ($params->{Action} eq 'Delete') {
            my $cids_mbids;
            foreach my $mbanner (@$mediaplan_banners) {
                push @{$cids_mbids->{$mbanner->{cid}}}, $mbanner->{mbid};
            }
            foreach my $cid (keys %$cids_mbids) {
                delete_mediaplan_banners($self->{uid}, $cid, $cids_mbids->{$cid}, with_vcards => 1);
                update_mediaplan_stats($self->{rbac}, $cid, $self->{uid}, $self->{rbac_login_rights}, 1);
            }
        }
    } elsif ($params->{Action} eq 'Update' || $params->{Action} eq 'Add') {

        my $banners2campaigns;
        my $cids;

        my $banners = convert_params(banners => $params->{MediaplanAds});

        my @source_bids = map { $_->{source_bid} ? $_->{source_bid} : () } @$banners;
        my $banners_models_by_id = get_banners_short_hash(\@source_bids, ['type', 'pid']);
        for my $banner (@$banners) {
            my $banner_model = $banners_models_by_id->{$banner->{source_bid}};
            if ($banner_model) {
                $banner->{source_pid} = $banner_model->{pid};
                $banner->{banner_type} = $banner_model->{type};
            } else {
                delete $banner->{source_bid};
                $banner->{banner_type} = 'desktop';
            }
        }
        if ($params->{Action} eq 'Add') {
            $cids = [ grep {/^\d+$/} map {$_->{cid}} @$banners ];

            foreach my $banner (@$banners) {
                $banner->{mbid} = 0;
                # дефолтное значение geo
                $banner->{geo} = "0" unless defined $banner->{geo};
            }
            $banners2campaigns = get_hashes_hash_sql(PPC(bid => [map {$_->{source_bid}} grep {$_->{source_bid}} @$banners]),
                ['select bid, cid from banners', where => {bid => SHARD_IDS}]);
            
        }
        my $old_banners_hash;
        if ($params->{Action} eq 'Update') {
            my $old_banners = get_mediaplan_banners(
                [],
                {
                    bids => [map {$_->{mbid}} @$banners], 
                    no_phrases => 1,
                    not_add_minus_to_minus_words => 1,
                    is_api => 1,
                }
            );
            foreach my $old_banner (@$old_banners) {
                $old_banners_hash->{ $old_banner->{mbid} } = $old_banner;
                push @$cids, $old_banner->{cid};
            }
        }

        my $cid2uid = get_cid2uid(cid => $cids);
        # Определяем ClientID (массово) для сайт-линков, минус-слов по сid из баннеров
        my $cids2clientids = get_cid2clientid(cid => $cids);

        my @missing_fields = ('sitelinks', 'geo', grep {$_ ne 'type'} @Mediaplan::MEDIAPLAN_BANNER_FIELDS);
        my %cids_to_update_stats;
        for (my $i = 0; $i < scalar @{$banners}; $i++) {
            if (! $self->{ret}[$i]{Errors} || ! scalar @{$self->{ret}[$i]{Errors}}) {
                my $banner = $banners->[$i];
                # немного дублирования кода из APIMethods и DoCmdMediaplan
                # TODO: поправить при реализации нового подхода для групп
                my $ClientID;
                if ($params->{Action} eq 'Add') {
                    if ($banner->{source_bid} && $banners2campaigns->{$banner->{source_bid}}{cid} != $banner->{cid}) {
                        delete $banner->{source_bid};
                    }
                    if (! defined $banner->{banner_minus_words}) {
                        $banner->{banner_minus_words} = [];
                    }
                    $ClientID = $cids2clientids->{ $banner->{cid} };
                
                } elsif ($params->{Action} eq 'Update') {
                    delete $banner->{source_bid};
                    if (! defined $banner->{banner_minus_words}) {
                        $banner->{banner_minus_words} = $old_banners_hash->{$banner->{mbid}}{banner_minus_words};
                    }
                    # ClientID для минус-слов и сайтл-линков. Запоминаем, пока не удалили cid.
                    $ClientID = $cids2clientids->{ $old_banners_hash->{ $banner->{mbid} }->{cid} };
                    # поле cid при апдейте всегда лишнее, удаляем, если было передано
                    delete $banner->{cid};
                    # непереданные поля заменяем взятыми из базы
                    # пробегаемся по всем полям из @Mediaplan::MEDIAPLAN_BANNER_FIELDS, хотя там немного лишнего :)

                    for my $field (@missing_fields) {
                        if (!defined $banner->{$field}) {
                            $banner->{$field} = $old_banners_hash->{$banner->{mbid}}{$field};
                        }
                    }

                    # если передано полe ContactInfo, будет выставлен этот флаг, 
                    # если его нет, берем визитку из предыдущей версии баннера
                    if (!$banner->{need_validate_ci}) {
                        for my $field (@$VCARD_FIELDS_FORM) {
                            if (!defined $banner->{$field}) {
                                if ($field eq 'phone') {
                                    hash_merge $banner, parse_phone($old_banners_hash->{$banner->{mbid}}{$field});
                                } else {
                                    $banner->{$field} = $old_banners_hash->{$banner->{mbid}}{$field};
                                }
                            }
                        }
                    }
                }

                $banner->{banner_minus_words} = MinusWords::polish_minus_words_array($banner->{banner_minus_words});

                my $map = CommonMaps::check_address_map(
                    hash_cut($banner, @$VCARD_FIELDS_FORM),
                    { ClientID => $ClientID }
                );

                my $sitelinks_set_id = undef;
                if (Sitelinks::need_save_sitelinks_set($banner->{sitelinks})) {
                    $sitelinks_set_id = Sitelinks::save_sitelinks_set($banner->{sitelinks}, $ClientID);
                }

                my @errors = validate_mediabanner($banner, skip_contactinfo => 1, ClientID => $ClientID);

                if ($banner->{need_validate_ci} && scalar grep {$banner->{$_}} @$VCARD_FIELDS_FORM) {
                    push @errors, validate_contactinfo_api($banner);
                }

                if ($banner->{country_code} || $banner->{city_code} || $banner->{phone} || $banner->{ext}) {
                    $banner->{phone} = VCards::compile_phone($banner);
                }

                my $vcard_values = hash_cut($banner, @$VCARD_FIELDS_DB);

                my $uid_for_org_details;
                if ($params->{Action} eq 'Update') {
                    $uid_for_org_details = $cid2uid->{ $old_banners_hash->{ $banner->{mbid} }->{cid} };
                } elsif ($params->{Action} eq 'Add') {
                    $uid_for_org_details = $cid2uid->{ $banner->{cid} };
                }
                $vcard_values->{org_details_id} = add_org_details({
                    ogrn => $banner->{ogrn}, 
                    uid => $uid_for_org_details,
                });
                $vcard_values->{address_id} = ref($map) && $map->{aid} ? $map->{aid} : undef;
                if (scalar @errors) {
                    foreach my $error (@errors) {
                        push @{$self->{ret}[$i]->{Errors}}, get_error_object('BadParams', $error);
                    }
                }
                if (! $self->{ret}[$i]{Errors} || ! scalar @{$self->{ret}[$i]{Errors}}) {
                    $self->{ret}[$i]->{MediaplanAdID} = update_mediaplan_banner(
                        $banner->{mbid}, 
                        $banner, 
                        $vcard_values, 
                        $sitelinks_set_id, 
                        $ClientID,
                        undef,
                        {is_api => 1},
                    );
                    $cids_to_update_stats{$banner->{cid}} = undef;
                }
            }
        }

        foreach my $cid (keys %cids_to_update_stats) {
            update_mediaplan_stats($self->{rbac}, $cid, $self->{uid}, $self->{rbac_login_rights}, 1);
        }

        $result->{ActionsResult} = $self->{ret};
    }

    return $result;
}

sub MediaplanKeywordWrapper {
    my ($self, $params) = @_;

    my $unglue_flag = exists $params->{Options} && exists $params->{Options}{AutoMinusWords} && $params->{Options}{AutoMinusWords} eq 'Yes';
    my $result = {};
    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        my $phrases = _get_mediaplan_criteria(@_, 'keyword');
        if ($params->{Action} eq 'Get') {
            $result->{MediaplanKeywords} = $phrases;
        } else {
            my $cids_phrases;
            foreach my $phrase (@$phrases) {
                push @{$cids_phrases->{$phrase->{CampaignID}}}, $phrase->{MediaplanKeywordID};
            }
            foreach my $cid (keys %$cids_phrases) {
                delete_mediaplan_phrases($self->{uid}, $cid, ids => $cids_phrases->{$cid});
            }
        }
    # TODO: склейка-расклейка минус слов
    } elsif ($params->{Action} eq 'Add') {
        my @keywords_to_add;

        for (my $i = 0; $i < scalar @{$params->{MediaplanKeywords}}; $i++) {

            if (! $self->{ret}[$i]{Errors} || ! scalar @{$self->{ret}[$i]{Errors}}) {
                my $keyword = $params->{MediaplanKeywords}[$i];
                my $parsed_keyword = {
                    mbid => $keyword->{MediaplanAdID},
                    source_id => $keyword->{SourceKeywordID},
                    phrase => $keyword->{Phrase},
                    place => position_to_place($keyword->{Position}),
                    is_new => 1,
                    id_in_result => $i,
                };

                push @keywords_to_add, $parsed_keyword;
            }
        }

        my $ret = update_mediaplan_phrases_wrapper(
            $self->{rbac}, 
            $self->{uid}, 
            $self->{rbac_login_rights}, 
            rbac_get_chief_rep_of_client_rep($self->{uid}), 
            add => \@keywords_to_add,
            unglue_phrases => $unglue_flag,
        );

        foreach my $p (@keywords_to_add) {
            $self->{ret}[$p->{id_in_result}]->{MediaplanKeywordID} = $ret->{$p->{mbid}}{add}{$p->{md5}}{id};

            if ($ret->{$p->{mbid}}{add}{$p->{md5}}{mbid} != $p->{mbid}) {
                push @{$self->{ret}[$p->{id_in_result}]{Warnings}}, 
                    API::Errors::get_warning_object('NewMediaplanBanner', {new_id => int($ret->{$p->{mbid}}{add}{$p->{md5}}{mbid})});
            }

            if ($ret->{$p->{mbid}}{add}{$p->{md5}}{updated}) {
                push @{$self->{ret}[$p->{id_in_result}]{Warnings}}, 
                    API::Errors::get_warning_object('UpdateMediaplanKeyword', {id => int($ret->{$p->{mbid}}{add}{$p->{md5}}{id})});
            }

            if ($ret->{$p->{mbid}}{add}{$p->{md5}}{cnt} > 1) {
                push @{$self->{ret}[$p->{id_in_result}]{Warnings}}, 
                    API::Errors::get_warning_object('EqMediplanKeywordsJoined', {});
            }
        }
        $result->{ActionsResult} = $self->{ret};

    } elsif ($params->{Action} eq 'Update') {
        my @keywords_to_update;

        for (my $i = 0; $i < scalar @{$params->{MediaplanKeywords}}; $i++) {

            if (! $self->{ret}[$i]{Errors} || ! scalar @{$self->{ret}[$i]{Errors}}) {
                my $keyword = $params->{MediaplanKeywords}[$i];
                my $parsed_keyword = {
                    old_id => $keyword->{MediaplanKeywordID},
                    phrase => $keyword->{Phrase},
                    place => position_to_place($keyword->{Position}),
                    id_in_result => $i,
                };
                push @keywords_to_update, $parsed_keyword;
            }
        }
        my $ret = update_mediaplan_phrases_wrapper(
            $self->{rbac}, 
            $self->{uid}, 
            $self->{rbac_login_rights}, 
            rbac_get_chief_rep_of_client_rep($self->{uid}),
            update => \@keywords_to_update,
            unglue_phrases => $unglue_flag,
        );

        foreach my $p (@keywords_to_update) {
            $self->{ret}[$p->{id_in_result}]->{MediaplanKeywordID} = $ret->{$p->{mbid}}{update}{$p->{old_id}}{id};

            if ($ret->{$p->{mbid}}{update}{$p->{old_id}}{mbid} != $p->{mbid}) {
                push @{$self->{ret}[$p->{id_in_result}]{Warnings}}, 
                    API::Errors::get_warning_object('NewMediaplanBanner', {new_id => int($ret->{$p->{mbid}}{update}{$p->{old_id}}{mbid})});
            }

            # TODO: mbid сейчас заполняется в wrapper'е, надо бы сделать это более явным
            if ($p->{old_id} != $ret->{$p->{mbid}}{update}{$p->{old_id}}{id}) {
                push @{$self->{ret}[$p->{id_in_result}]->{Warnings}}, 
                    API::Errors::get_warning_object('RemovedWhileUpdateMediaplanKeyword', {old_id => int($p->{old_id}), new_id => int($ret->{$p->{mbid}}{update}{$p->{old_id}}{id})});
            }
        }
        $result->{ActionsResult} = $self->{ret};
    }
    return $result;
}

sub MediaplanCategoryWrapper {
    my ($self, $params) = @_;

    if ($params->{Action} eq 'Get'){
        return  {
                    MediaplanCategories =>[],
                };
    } elsif ($params->{Action} eq 'Delete') {
        return {};
    } elsif ($params->{Action} eq 'Add') {
        return {
                    ActionsResult => []
                };
    } elsif ($params->{Action} eq 'Update') {
        return {
                    ActionsResult => []
                };
    }
    return {};
}

# type = 'keyword' - вернуть ключевые слова медиаплана
# type = 'category' - вернуть категории медиаплана

sub _get_mediaplan_criteria {
    my ($self, $params, $type) = @_;
    
    my @result;

    my $criteria = $params->{SelectionCriteria};
    # собираем массив mbids
    my $mbids = $criteria->{MediaplanAdIDS} || [];

    my $ids = [@{$criteria->{MediaplanKeywordIDS} || []}, @{$criteria->{MediaplanCategoryIDS} || []}];

    if (!scalar @$mbids && scalar @$ids) {

        $mbids = get_one_column_sql(PPC(shard => 'all'), [
            "select distinct mbid from mediaplan_bids", 
            where => {id => $ids}
            ]) || [];
    }

    # получаем полный список фраз для фильтрации
    my $where;
    $where->{cid} = $criteria->{CampaignIDS} if (scalar @{$criteria->{CampaignIDS} || []});
    $where->{mbid} = $mbids if @$mbids;
    $where->{id} = $ids if scalar @$ids;

    my $ids_to_filter = get_hashes_hash_sql(PPC(shard => 'all'), ["select id from mediaplan_bids", where => $where]);

    # получаем баннеры медиаплана с фразами
    # TODO: фильтрация по фразам, чтобы не выбирать и не обрабатывать лишнего
    my $mediaplan_banners = get_mediaplan_banners(
        $criteria->{CampaignIDS} || [], 
        {no_phrases => 0, bids => $mbids, is_api => 1}
    );

    foreach my $mbanner (@$mediaplan_banners) {

        foreach my $phrase (@{$mbanner->{Phrases}}) {
            if (exists $ids_to_filter->{$phrase->{id}} && $type eq 'keyword') {
                # TODO: временно, внутри filter_phrase_object надо использовать cid
                $phrase->{CampaignID} = $phrase->{cid};
                $phrase->{mbid} = $mbanner->{mbid};
                $phrase->{currency} = $mbanner->{currency};
                
                my $filter_params = hash_merge { ExportCurrency => $params->{Options}{Currency} }, hash_cut($self, qw/api_version_full/);
                
                my $phrase_object = filter_phrase_object( $phrase, $filter_params, mediaplan => 1);
                
                if ($type eq 'category') {
                    $phrase_object->{RubricID} = $phrase_object->{Phrase};
                    $phrase_object->{MediaplanCategoryID} = $phrase_object->{MediaplanKeywordID};
                    delete $phrase_object->{MediaplanKeywordID};
                    delete $phrase_object->{Phrase};
                }
                push @result, $phrase_object;
            }
        }
    }
    return \@result;
}

1;
