package API::Methods::Retargeting;

#$Id$

=head1 NAME

    API::Methods::Retargeting

=head1 DESCRIPTION

    Методы для работы с условиями ретаргетинга

=cut

use Direct::Modern;

use List::MoreUtils qw(any none uniq);
use Scalar::Util qw(looks_like_number);

use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut/;
use Yandex::ListUtils qw/nsort/;
use Yandex::DBTools;
use Yandex::DBShards;

use Settings;

use Retargeting;
use RBACElementary;
use RBACDirect;
use RBAC2::DirectChecks;
use Primitives qw/get_goal_type_by_goal_id/;
use PrimitivesIds;
use PhrasePrice;
use Campaign;
use Models::AdGroup;
use Direct::RetargetingConditions;
use Direct::Validation::RetargetingConditions;
use Client ();

use Currencies;
use Currency::Rate;

use APICommon qw/get_syslog_data/;
use API::Filter;
use API::Errors;
use API::Validate::Structure;
use API::ValidateTools;
use API::ConvertFrom qw(convert_params);
use API::ObjectRelations;

our $METRIKA_CLIENT_TIMEOUT = 60;

sub GetRetargetingGoals {
    my ($self, $params) = @_;

    state $goal_type_internal_to_api_map = {
        goal => 'goal',
        segment => 'segment',
        audience => 'audience_segment',
        ecommerce => 'ecommerce',               # https://github.yandex-team.ru/Metrika/metrika-api/pull/1229
    };
    my $uid2login;
    if ($self->{rbac_login_rights}{role} eq 'client') {
        for my $uid (@{rbac_get_client_uids_by_clientid($self->{user_info}{ClientID})}) {
            $uid2login->{$uid} = $self->{user_info}{login};
        }
    } else {
        my $login2clientid = get_login2clientid(login => $params->{Logins});
        while (my ($login, $clientid) = each %$login2clientid) {
            for my $uid (@{rbac_get_client_uids_by_clientid($clientid)}) {
                $uid2login->{$uid} = $login;
            }
        }
    }

    my $uid2goals = Retargeting::get_metrika_goals_by_uid([keys %$uid2login], timeout => $METRIKA_CLIENT_TIMEOUT);

    my $result = [];
    foreach my $uid ( nsort keys %$uid2goals) {
        foreach my $goal ( sort {$a->{goal_id} <=> $b->{goal_id} } @{$uid2goals->{$uid}} ) {
            my $type = $goal_type_internal_to_api_map->{ $goal->{goal_type} } or next;
            push @$result, {
                GoalID => $goal->{goal_id},
                Name => $goal->{goal_name},
                GoalDomain => $goal->{goal_domain},
                Login => $uid2login->{$uid},
                Type  => $type
            };
        }
    }

    return $result;
}

sub validate_get_retargeting_goals_rights {
    my ($self, $params) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    if ($self->{rbac_login_rights}{role} ne 'client') {
        my $uids = get_uids(login => $params->{Logins});
        if (my $error_code = APICommon::check_rbac_rights( $self, 'api_userSettings', { UID => $self->{uid}, uids => $uids } )) {
            return ('NoRights');
        }
    }
    return;
}

sub RetargetingCondition {
    my ($self, $params) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    my $result = {};
    if ($params->{Action} eq 'Get') {
        my $clientid2login = $self->{preprocess}{clientid2login};
        my $conditions = $self->{preprocess}{conditions};

        my $res = [];
        for my $clientid (nsort keys %$conditions) {
            for my $conditionid (nsort keys %{$conditions->{$clientid}}) {
                # для каждого логина клиента - своя копия
                for my $login (@{$clientid2login->{$clientid}}) {
                    # копируем и конвертируем данные
                    my $condition = convert_params(retargeting_ext => $conditions->{$clientid}{$conditionid});
                    delete $condition->{ClientID};
                    delete $condition->{properties};
                    $condition->{Login} = $login;
                    my $condition_with_filtered_goals = _filter_excess_goal_fields($condition); # в структуре цели отдаем только поля Time и GoalID
                    push @$res, $condition_with_filtered_goals;
                }
            }
        }
        $result = {RetargetingConditions => $res};
    } elsif ($params->{Action} eq 'Delete') {
        my $clientids = $self->{preprocess}{criteria}{clientids};
        if (! @$clientids) {
            $clientids =
                get_clientids(ret_cond_id => $self->{preprocess}{criteria}{conditionIDS});
        }
        Retargeting::delete_retargeting_condition_by_ClientIDS(
            $clientids, $self->{preprocess}{criteria}{conditionIDS}
        );
    } elsif ($params->{Action} eq 'Add' || $params->{Action} eq 'Update') {

        for (my $i=0; $i<scalar @{$params->{RetargetingConditions} || []}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $cond = $params->{RetargetingConditions}[$i];
            my $cond_converted = $self->{preprocess}{converted}[$i];

            if ($params->{Action} eq 'Add') {
                my $error = Retargeting::validate_retargeting_condition_dublicate(
                    $cond_converted,
                    [values %{$self->{preprocess}{exists_cond}{$cond_converted->{ClientID}}}],
                    error_text_variant => 'api4',
                );
                if ($error) {
                    push @{$self->{ret}[$i]{Errors}}, get_error_object('BadParams', $error);
                }
            }
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};

            my $conditionid = Retargeting::save_retargeting_condition('ClientID', $cond_converted->{ClientID}, $cond_converted);
            $self->{ret}[$i]->{RetargetingConditionID} = $conditionid;

            if ($params->{Action} eq 'Add') {
                # добавляем условие в список существующих, чтобы дальше валидировать с его учетом
                $self->{preprocess}{exists_cond}{$cond_converted->{ClientID}}{$conditionid} = {%$cond_converted, ret_cond_id => $conditionid};
            }
        }

        $result = {ActionsResult => $self->{ret}};
    }
    return $result;
    # Actions: Get (for uid), Add (to uid), Update(for retargetingid), Delete (retargetingid),
}

sub preprocess_retargeting_condition {
    my ($self, $params) = @_;

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        my $conditionIDS;
        if ($params->{SelectionCriteria}{RetargetingConditionIDS} && scalar @{$params->{SelectionCriteria}{RetargetingConditionIDS}}) {
            $conditionIDS = $params->{SelectionCriteria}{RetargetingConditionIDS};
        }

        my $logins;
        if ($params->{SelectionCriteria}{Logins}) {
            $logins = $params->{SelectionCriteria}{Logins};
        }

        my $clientids = [];
        my $login2clientid = {};
        if ($self->{rbac_login_rights}{role} eq 'client') {
            $login2clientid = {$self->{user_info}{login} => $self->{user_info}{ClientID}};
            $clientids = [$self->{user_info}{ClientID}];
        } else {
            if ($logins) {
                $login2clientid = get_login2clientid(login => $logins);
                $clientids = [values %$login2clientid];
            }
        }

        $clientids = [grep {$_} @$clientids];

        my $clientid2login = {};
        for my $login (keys %$login2clientid) {
            push @{$clientid2login->{$login2clientid->{$login}}}, $login;
        }

        # сразу получаем условия, они нам понадобятся при проверке прав все равно
        my $conditions = $clientids && scalar @$clientids ? Retargeting::mass_get_retargeting_conditions_by_ClientIDS(
            $clientids,
            ret_cond_id => $conditionIDS) : {};

        # поскольку в запросе могут отсутствовать логины, забираем соответствия дополнительно
        # TODO: Здесь нужны только чифы
        if (! scalar keys %$clientid2login) {
            $clientid2login = get_clientid2logins(ClientID => [keys %$conditions]);
        }
        $self->{preprocess}{clientid2login} = $clientid2login;
        $self->{preprocess}{conditions} = $conditions;
        $self->{preprocess}{criteria} = {conditionIDS => $conditionIDS, clientids => $clientids};

    } elsif ($params->{Action} eq 'Add') {
        my $login2clientid;
        if ($self->{rbac_login_rights}{role} eq 'client') {
            $login2clientid = {$self->{user_info}{login} => $self->{user_info}{ClientID}};
        } else {
            $login2clientid = get_login2clientid(login => [grep {$_} map {$_->{Login}} @{$params->{RetargetingConditions}}]);
        }
        my $clientids = [uniq values %$login2clientid];
        my $exists_cond = scalar @$clientids ? Retargeting::mass_get_retargeting_conditions_by_ClientIDS($clientids, short => 1) : {};

        $self->{preprocess}{login2clientid} = $login2clientid;
        $self->{preprocess}{exists_cond} = $exists_cond;

        for (my $i=0; $i<scalar @{$params->{RetargetingConditions}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};

            my $condition_with_filtered_goals = _filter_excess_goal_fields($params->{RetargetingConditions}[$i]);
            # во входных данных удаляем для цели все поля, кроме Time и GoalID
            my $cond_converted = convert_params(retargeting_int => $condition_with_filtered_goals);
            _prepare_goals($cond_converted);

            delete $cond_converted->{ret_cond_id};
            if ($self->{rbac_login_rights}{role} eq 'client') {
                $cond_converted->{ClientID} = $self->{user_info}{ClientID};
            } else {
                $cond_converted->{ClientID} = $login2clientid->{$cond_converted->{Login}};
            }
            $self->{preprocess}{converted}[$i] = $cond_converted;
        }
    } elsif ($params->{Action} eq 'Update') {
        # сначала достаем ClientIDS
        my $old_conds_by_clientid = Retargeting::mass_get_retargeting_conditions_by_ClientIDS(
            [],ret_cond_id=>[grep {$_} map {$_->{RetargetingConditionID}} @{$params->{RetargetingConditions}}]);

        # теперь все существующие условия этих клиентов (нужно для валидации)
        $old_conds_by_clientid = scalar keys %$old_conds_by_clientid ? Retargeting::mass_get_retargeting_conditions_by_ClientIDS([keys %$old_conds_by_clientid], short => 1) : {};
        $self->{preprocess}{exists_cond} = $old_conds_by_clientid;

        my $old_conds;
        while (my ($clientid, $conditions) = each %$old_conds_by_clientid) {
            while (my ($condid, $condition) = each %$conditions) {
                $old_conds->{$condid} = $condition;
            }
        }
        for (my $i=0; $i<scalar @{$params->{RetargetingConditions}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};

            my $condition_with_filtered_goals = _filter_excess_goal_fields($params->{RetargetingConditions}[$i]); # во входных данных удаляем для цели все поля, кроме Time и GoalID
            my $cond_converted = convert_params(retargeting_int => $condition_with_filtered_goals);
            _prepare_goals($cond_converted);

            $cond_converted->{ClientID} = $old_conds->{$cond_converted->{ret_cond_id}}{ClientID};

            my %fields_to_update = map {$_=>1} @{$cond_converted->{Fields}};

            if (scalar keys %fields_to_update) {
                for my $field (keys %{$old_conds->{$cond_converted->{ret_cond_id}}}) {
                    if (! exists $APICommon::RETARGETING_EXT_CONVERSION{$field} || ! $fields_to_update{$APICommon::RETARGETING_EXT_CONVERSION{$field}}) {
                        $cond_converted->{$field} = $old_conds->{$cond_converted->{ret_cond_id}}{$field};
                    }
                }
            }

            $self->{preprocess}{converted}[$i] = $cond_converted;
        }
    }

    return;
}

=head2 _filter_excess_goal_fields($condition)

    Удаляем из целей все поля, кроме Time и GoalID. В функцию передается ссылка хэш вида:
    {
        IsAccessible         => "Yes",
        Login                => "Login",
        RetargetingCondition => [
            {
                Goals => [
                    {
                        GoalID => 8,
                        Time   => 1,
                        Type   => 'goal'
                    },
                    {
                        GoalID => 16,
                        Time   => 1,
                        Type   => 'segment'
                    }
                ],
                Type => "or"
            }
        ],
        RetargetingConditionDescription => "описание условия 1",
        RetargetingConditionID          => 250276,
        RetargetingConditionName        => "условие 1"
    }

    Из функции возвращается эта-же структура, но в целях удалены все поля, кроме Time и GoalID:
    {
        IsAccessible         => "Yes",
        Login                => "Login",
        RetargetingCondition => [
            {
                Goals => [
                    {
                        GoalID => 8,
                        Time   => 1
                    },
                    {
                        GoalID => 16,
                        Time   => 1
                    }
                ],
                Type => "or"
            }
        ],
        RetargetingConditionDescription => "описание условия 1",
        RetargetingConditionID          => 250276,
        RetargetingConditionName        => "условие 1"
    }

=cut

sub _filter_excess_goal_fields {
    my ($condition) = @_;

    if (_is_array_in_field($condition, 'RetargetingCondition')) {
        foreach my $item (@{$condition->{RetargetingCondition}}) {
            if (_is_array_in_field($item, 'Goals')) {
                foreach my $goal (@{$item->{Goals}}) {
                    my @fields = get_goal_type_by_goal_id($goal->{GoalID}) eq 'audience' ? 'GoalID' : qw/Time GoalID/;
                    $goal = hash_cut($goal, @fields); # формируем новый хэш, переносим в него только поля Time и GoalID
                }
            }
        }
    }

    return $condition;
}

=head2 _repare_goals($condition)

    Подготавливает цели чтобы после преобразования в json, выглядели как в базе

=cut

sub _prepare_goals {
    my ($condition) = @_;

    if (_is_array_in_field($condition, 'condition')) {
        foreach my $item (@{$condition->{condition}}) {
            if (_is_array_in_field($item, 'goals')) {
                foreach my $goal (@{$item->{goals}}) {
                    _add_goal_type_field($goal);
                    _convert_goal_id_and_time_to_int($goal);
                }
            }
        }
    }
    return $condition;
}

=head2 _add_goal_type_field($condition)

    Добавляет для каждой цели поле Type

=cut

sub _add_goal_type_field {
    my ($goal) = @_;

    $goal->{goal_type} = get_goal_type_by_goal_id($goal->{goal_id})
        if defined $goal->{goal_id};

    return $goal;
}

=head2 _convert_goal_id_and_time_to_int($goal)

    Преобразует поля goal_id и time из строк в числа

=cut

sub _convert_goal_id_and_time_to_int {
    my ($goal) = @_;

    $goal->{goal_id} += 0 if defined $goal->{goal_id} && looks_like_number($goal->{goal_id});
    $goal->{time} += 0 if defined $goal->{time} && looks_like_number($goal->{time});

    return $goal;
}


=head2 _is_array_in_field($hash, $field_name)

    Функция показывает, содержит ли поле $field_name в хэше $hash массив

    if (_is_array_in_field($hash, 'field_name')) {
        my @array = @{$hash->{field_name}};
    }

=cut

sub _is_array_in_field {
    my ($hash, $field_name) = @_;

    return exists $hash->{$field_name} && ref $hash->{$field_name} eq 'ARRAY' ? 1 : 0;
}

sub validate_retargeting_condition {
    my ($self, $params) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    if ($params->{Action} eq 'Add' || $params->{Action} eq 'Update') {
        my $p = $self->{preprocess};

        for (my $i=0; $i<@{$params->{RetargetingConditions}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $cond = $self->{preprocess}{converted}[$i];
            if ($params->{Action} eq 'Update') {
                $cond->{old} = $self->{preprocess}{exists_cond}{$cond->{ClientID}}{$cond->{ret_cond_id}};
            }

            my $exists_cond = $p->{exists_cond}{$cond->{ClientID}};
            foreach(my $j = 0; $j < @{$params->{RetargetingConditions}[$i]{RetargetingCondition} || []}; $j++) {
                my $condition = $params->{RetargetingConditions}[$i]{RetargetingCondition}[$j];
                foreach(my $g = 0; $g < @{$condition->{Goals}}; $g++) {
                    my $goal = $condition->{Goals}[$g];
                    my $validate_for_type;
                    if(get_goal_type_by_goal_id($goal->{GoalID}) eq 'audience') {
                        $validate_for_type = 'RetargetingConditionAudience';
                        delete $goal->{Time};
                    } else {
                        $validate_for_type = 'RetargetingConditionMetrikaGoal';
                    }
                    if(my @errors = API::Validate::Structure::api_validate_structure($self,
                        $condition->{Goals}[$g],
                        $validate_for_type,
                        "RetargetingConditions[$i].RetargetingCondition[$j].Goals[$g]")
                    ) {
                        push @{$self->{ret}[$i]{Errors}}, get_error_object(@$_) foreach @errors;
                    }
                }
            }
            if ($params->{Action} eq 'Add') {
                # будем валидировать непосредственно при добавлении
            } else {
                if (!$self->{ret}[$i]{Errors} || !@{$self->{ret}[$i]{Errors}}) {
                    ## если была ошибка структуры, ничего больше проверять не надо

                    # сначала проверяем пересечения по названиям и наборам целей, потом no_db проверки
                    my $error = Retargeting::validate_retargeting_condition_dublicate(
                        $cond,
                        [values %$exists_cond],
                        no_db => 1, error_text_variant => 'api4'
                    );
                    if ($error) {
                        push @{$self->{ret}[$i]{Errors}}, get_error_object('BadParams', $error);
                    } else {
                        $error = Retargeting::validate_retargeting_condition_without_db($cond, count_of_exists_cond => 0);
                        if ($error) {
                            push @{$self->{ret}[$i]{Errors}}, get_error_object('BadParams', $error);
                        }
                    }
                }
            }
        }
    }

    return;
}

sub validate_retargeting_condition_rights {
    my ($self, $params) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    if ($params->{Action} eq 'Add' || $params->{Action} eq 'Update') {
        my $clientids = [keys %{$self->{preprocess}{exists_cond}}];

        my $clientid2uid;
        $clientid2uid = rbac_get_chief_reps_of_clients($clientids) if @$clientids;

        my $uniqed_uids = [uniq values %$clientid2uid];
        my $uid2role;
        $uid2role = rbac_multi_who_is($self->{rbac}, $uniqed_uids) if @$uniqed_uids;

        my $isowner;
        if (@$uniqed_uids) { 
            $isowner = RBACDirect::rbac_is_owner_of_users($self->{uid}, $uniqed_uids);
        }

        my $is_client_converting_soon = Client::mass_is_client_converting_soon($clientids);
        my $client_id2must_convert = Client::mass_client_must_convert($clientids);

        for (my $i=0; $i<scalar @{$params->{RetargetingConditions}}; $i++) {
            next if $self->{ret}[$i]{Errors} && scalar @{$self->{ret}[$i]{Errors}};
            my $clientid = $self->{preprocess}{converted}[$i]{ClientID};
            my $uid = $clientid2uid->{$clientid};

            my $error_text;
            if (!$uid || ($uid2role->{$uid} ne 'client')) {
                $error_text = iget("Невозможно добавить условие пользователю, не являющемуся клиентом");
            } elsif ($self->{rbac_login_rights}{role} =~ m/^(media|superreader)$/) {
                $error_text = '';
            } elsif ($self->{rbac_login_rights}{role} !~ m/^(super|support|placer)$/ && ! $isowner->{$uid}) {
                $error_text = iget("Нет прав на данного клиента");
            } elsif ($is_client_converting_soon->{$clientid}) {
                $error_text = APICommon::msg_converting_in_progress;
            } elsif ($client_id2must_convert->{$clientid}) {
                $error_text = APICommon::msg_must_convert;
            }

            if (defined $error_text) {
                push @{$self->{ret}[$i]{Errors}}, get_error_object('NoRights', $error_text);
            }
        }
    } elsif ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {

        my @clientids = keys %{$self->{preprocess}{clientid2login}};
        if (scalar @clientids) {
            my $clientid2uid = rbac_get_chief_reps_of_clients(\@clientids);
            my $uniqed_uids = [uniq values %$clientid2uid];

            if (!rbac_mass_is_owner($self->{rbac}, $self->{uid}, $uniqed_uids)) {
                return ('NoRights');
            }
        }
        if ($params->{Action} eq 'Delete') {
            if ($self->{rbac_login_rights}{role} =~ m/^(media|superreader)$/) {
                return ('NoRights');
            }
            my @ret_cond_ids;
            for my $clid (keys %{$self->{preprocess}{conditions}}) {
                push @ret_cond_ids, keys %{$self->{preprocess}{conditions}{$clid}};
            }
            if (@ret_cond_ids) {
                my $exists_cond = Retargeting::get_used_ret_cond(\@ret_cond_ids);

                if (scalar keys %$exists_cond) {
                    return ('BadParams', iget('Одно или несколько условий, соответствующих критерию, используются'));
                }
            }
        }
    }
    return ;
}

=head2 Retargeting
    Выполнение запрошенного действия и формирование результата:
        * Get - формирование результата из ранее полученных данных и, если требуется,
            конвертация цен в тебуемую валюту
        * Delete - удаление ретаргетингов и отметка затронутых групп для переотправки в БК
        * Add & Update - проверки:
                - не обновляем цену у ретаргетинга, который относится к кампании с автобюджетом
                - не пытаемся создать новый ретаргетинг, относящийся к условиям показа и группе,
                    если похожий уже существует.
                - не создаем за запрос несколько ретаргетингов, относящийся к одинаковым условиям
                    показа и группе
            После проверок добавляем новый ретаргетинг

=cut

sub Retargeting {
    my ($self, $params) = @_;

    my $action     = $params->{Action};
    my $preprocess = $self->{preprocess};

    my $result = {};
    if ( $action eq 'Get' ) {

        my $pid2bid  = $preprocess->{pid2bid};
        my $currency = $params->{Options}{Currency} || 'YND_FIXED';

        my @ret;
        foreach my $retargeting ( values %{ $preprocess->{retargetings} } ) {
            push @ret, filter_retargeting_object( $retargeting, $pid2bid, export_currency => $currency );
        }

        $result->{Retargetings} = \@ret;

    } elsif ( $action eq 'Delete' ) {

        Retargeting::delete_group_retargetings( [ values %{ $preprocess->{retargetings} } ] );

    } elsif ( $action eq 'Add' || $action eq 'Update' ) {

        my $camps         = $preprocess->{camps};
        my $pid2cid       = $preprocess->{pid2cid};
        my $converted     = $preprocess->{converted};
        my $camp_strategy = $preprocess->{camp_strategy};

        my $old_retargetings        = $preprocess->{old_retargetings};
        my $old_retargetings_by_pid = $preprocess->{old_retargetings_by_pid_hash};

        # добавляем ретаргетинги по одному, чтобы отследить изменения идентификаторов
        # TODO: массовость

        # Собираем список на обновления, для каждого условия внутри группы
        # делаем обновление один раз, данными из последнего баннера
        my $retargetings_to_update = {}; # { pid => { ret_cond_id => {} } }

        my $size = @$converted;
        for ( my $i = 0; $i < $size; $i++ ) {

            next if $self->{ret}[ $i ]{Errors} && scalar @{ $self->{ret}[ $i ]{Errors} };

            my $retargeting = $converted->[ $i ];

            my $pid         = $retargeting->{pid};
            my $cid         = $pid2cid->{ $pid };
            my $ret_cond_id = $retargeting->{ret_cond_id};

            my $old_retargeting;
            if ( $action eq 'Add' && exists $old_retargetings_by_pid->{ $pid }{ $ret_cond_id } ) {

                # все таки обновляем, если условие с данным ret_cond_id уже существует
                $old_retargeting = $old_retargetings_by_pid->{ $pid }{ $ret_cond_id };

                my $old_ret_id = $old_retargeting->{ret_id};
                $retargeting->{ret_id} = $old_ret_id;

                push @{ $self->{ret}[ $i ]{Warnings} }, get_warning_object('UpdateRetargeting', { id => int($old_ret_id) } );

            } elsif ( $action eq 'Update') {

                $old_retargeting = $old_retargetings->{ $retargeting->{ret_id} };

            }

            if ( $camp_strategy->{ $cid }{is_autobudget}
                    && $retargeting->{price_context}
                    && (! defined $old_retargeting || $retargeting->{price_context} != $old_retargeting->{price_context} )
            ) {
                push @{ $self->{ret}[ $i ]{Warnings} }, get_warning_object('RetargetingContextPriceIgnored');
            }

            $retargetings_to_update->{ $pid } ||= {};
            $retargetings_to_update->{ $pid }->{ $ret_cond_id } = {
                seq => $i,
                pid => $pid,
                bid => $retargeting->{bid},
                cid => $cid,
                currency => $camps->{ $cid }{currency},
                retargetings => [ $retargeting, ],
            };
        }

        # Ищем условия которые не попали в retargetings_to_update, как
        # дубликаты и проставляем для них warning
        for ( my $i = 0; $i < $size; $i++ ) {

            my $retargeting = $converted->[ $i ];

            my $pid         = $retargeting->{pid};
            my $ret_cond_id = $retargeting->{ret_cond_id};

            if ( $pid
                   && exists $retargetings_to_update->{ $pid }
                   && exists $retargetings_to_update->{ $pid }{ $ret_cond_id }
                   && $retargetings_to_update->{ $pid }{ $ret_cond_id }{seq} != $i
            ) {
                push @{ $self->{ret}[ $i ]{Warnings} }, get_warning_object('RetargetingConditionAlreadyExistsInAdGroup');
            }
        }

        foreach my $ret_by_pid ( values %$retargetings_to_update ) {
            foreach my $ret ( values %$ret_by_pid ) {

                my $i = delete $ret->{seq};

                my $ret_id = Retargeting::update_group_retargetings(
                    $ret, insert_only => 1, camp_strategy => $camp_strategy->{ $ret->{cid} }
                );

                $self->{ret}[ $i ]->{RetargetingID} = $ret_id->[0];
            }
        }

        $result->{ActionsResult} = $self->{ret};
    }

    return $result;
}

=head2 preprocess_retargeting
    Сбор данных, необходимых для дальнейших проверок и выполнения запрошенных действий
    Если к API обращается клиент, то его логин добавляется к параметрам
=cut

sub preprocess_retargeting {
    my ($self, $params) = @_;

    # так как для шардинга ретаргетинга нужен ClientID который можно получить из логина
    # а просить его дополнительно у пользователя не хочется, подсовываем его в параметры
    # агентства и внутренние роли могут работать с разными пользователями,
    # поэтому они передают логин в параметрах
    if ( $self->{rbac_login_rights}{role} eq 'client' ) {
        $params->{Login} = $self->{user_info}{login};
    }

    my $client_id = get_clientid( login => $params->{Login} );

    my $action = $params->{Action};
    if ( $action eq 'Add' || $action eq 'Update' ) {

        my $converted = convert_params( banner_retargeting => $params->{Retargetings} );

        if ( $action eq 'Add' ) {

            my $bid2pid = get_bid2pid( bid => [ map { $_->{bid} } @$converted ]);

            my $pids = [ uniq values %$bid2pid ];
            my $old_retargetings_by_pid = Retargeting::get_group_retargeting( pid => $pids );

            my $old_retargetings_by_pid_hash;

            foreach my $pid ( keys %$old_retargetings_by_pid ) {

                my %ret_by_ret_cond;
                foreach my $ret( @{ $old_retargetings_by_pid->{ $pid } } ) {
                    $ret_by_ret_cond{ $ret->{ret_cond_id} } = $ret;
                }
                $old_retargetings_by_pid_hash->{ $pid } = \%ret_by_ret_cond;

            }

            my @retargeting_condition_ids = map {$_->{ret_cond_id}} @$converted;
            my $retargeting_conditions_by_id = {};

            if (@retargeting_condition_ids) {
                $retargeting_conditions_by_id = Direct::RetargetingConditions->
                get_by(id => \@retargeting_condition_ids)
                ->items_by('id');
            }

            $self->{preprocess}{retargeting_conditions_by_id} = $retargeting_conditions_by_id;
            $self->{preprocess}{old_retargetings_by_pid_hash} = $old_retargetings_by_pid_hash;

            my $size = @$converted;
            for ( my $i = 0; $i < $size; $i++ ) {

                next if ( $self->{ret}[ $i ]{Errors} && scalar @{ $self->{ret}[ $i ]{Errors} } );

                my $ret = $converted->[ $i ];

                if ( exists $bid2pid->{ $ret->{bid} } ) {
                    $ret->{pid} = $bid2pid->{ $ret->{bid} };
                } else {
                    push @{ $self->{ret}[ $i ]{Errors} }, get_error_object('NoRights');
                }

                # для добавления удаляем идентификатор, если вдруг пришел
                delete $converted->[ $i ]{ret_id};
            }

            $self->{preprocess}{adgroup_types} = get_hash_sql(PPC(pid => $pids),
                ["SELECT pid, adgroup_type FROM phrases", WHERE => {pid => SHARD_IDS}]
            ) if @$pids;
        } elsif ( $action eq 'Update' ) {

            my $old_retargetings = Retargeting::get_group_retargeting_ret_id_hash(
                ret_id   => [ map { $_->{ret_id} } @$converted ],
                ClientID => $client_id,
            );

            my $size = @$converted;
            for ( my $i = 0; $i < $size; $i++ ) {

                next if ( $self->{ret}[ $i ]{Errors} && scalar @{ $self->{ret}[ $i ]{Errors} } );

                my $retargeting = $converted->[ $i ];
                if ( exists $old_retargetings->{ $retargeting->{ret_id} } ) {

                    my $old_retargeting = $old_retargetings->{ $retargeting->{ret_id} };

                    # нельзя обновлять ret_cond_id, заменяем на старое значение
                    my $ret_cond_id     = $retargeting->{ret_cond_id};
                    my $old_ret_cond_id = $old_retargeting->{ret_cond_id};

                    if ( $ret_cond_id and $ret_cond_id ne $old_ret_cond_id ) {
                        push @{ $self->{ret}[ $i ]{Warnings} }, get_warning_object('RetargetingConditionIDIgnored');
                    }

                    $retargeting->{ret_cond_id} = $old_ret_cond_id;

                    $retargeting->{pid} = $old_retargeting->{pid};
                    $retargeting->{bid} = $old_retargeting->{bid};

                    my %fields = map { $_ => 1 } @{ $retargeting->{Fields} || [] };

                    $fields{Currency} = 1 if $fields{ContextPrice};

                    if ( keys %fields ) {
                        for my $field ( keys %$old_retargeting ) {
                            if ( $APICommon::AD_RETARGETING_CONVERSION{ $field }
                                    && ! $fields{ $APICommon::AD_RETARGETING_CONVERSION{ $field } }
                            ) {
                                $retargeting->{ $field } = $old_retargeting->{ $field };
                            }
                        }
                    }

                    # далее по коду происходит валидация этих данных, как будто их прислал клиент
                    #   поскольку снаружи пользователи не могут прислать такое значение -- то неправильно его добавлять из предыдущей версии объекта
                    delete $retargeting->{currency} if $retargeting->{currency} && $retargeting->{currency} eq 'YND_FIXED';
                }
            }

            $self->{preprocess}{old_retargetings} = $old_retargetings;
        }

        # смотрим на новые идентификаторы, могли обновиться при Update
        my $pid2cid = get_pid2cid( pid => [ uniq grep { defined $_ } map { $_->{pid} } @$converted ] );
        $self->{preprocess}{pid2cid} = $pid2cid;

        my @cids       = uniq values %$pid2cid;
        my $camps_info = get_camp_info( \@cids, undef, short => 1 );
        my %camps      = map { $_->{cid} => $_ } @$camps_info;
        my %cid2clientid = map { $_->{cid} => $_->{ClientID} } @$camps_info;

        $self->{preprocess}{camps}         = \%camps;
        $self->{preprocess}{camp_strategy} = Campaign::mass_campaign_strategy( \@cids );
        $self->{preprocess}{cid2clientid} = \%cid2clientid;

        # multicurrency - ok!
        my $size = @$converted;
        for ( my $i = 0; $i < $size; $i++ ) {

            next if ( $self->{ret}[ $i ]{Errors} && scalar @{ $self->{ret}[ $i ]{Errors} } );

            my $ret = $converted->[ $i ];

            my $camp_currency;
            if ( defined $ret->{pid} && exists $pid2cid->{ $ret->{pid} } && exists $camps{ $pid2cid->{ $ret->{pid} } } ) {
                $camp_currency = $camps{ $pid2cid->{ $ret->{pid} } }->{currency};
            }
            $camp_currency ||= 'YND_FIXED';

            if ( $camp_currency ne ( $ret->{currency} || 'YND_FIXED' ) ) {
                #конвертируем ставку от клиента в валюту кампании
                if ( $ret->{price_context} + 0 ) {
                    my $price = convert_currency( $ret->{price_context}, $ret->{currency} || 'YND_FIXED', $camp_currency );
                    $ret->{price_context} = currency_price_rounding( $price, $camp_currency, up => 1 );
                }
            }

            $ret->{price_currency} = $camp_currency;
        }
        $self->{preprocess}{converted} = $converted;
    } else {

        my %condition;
        $condition{ClientID} = $client_id;

        my $criteria = $params->{SelectionCriteria}; # mandatory field

        if ( $criteria->{AdIDS} && scalar @{ $criteria->{AdIDS} } ) {
            my $bid2pid = get_bid2pid( bid => $criteria->{AdIDS} );
            $condition{pid} = [ uniq values %$bid2pid ];
            $self->{preprocess}{pid2bid} = { reverse %$bid2pid };
        }

        if ( $criteria->{RetargetingConditionIDS} && scalar @{ $criteria->{RetargetingConditionIDS} } ) {
            $condition{ret_cond_id} = $criteria->{RetargetingConditionIDS};
        }

        if ( $criteria->{RetargetingIDS} && scalar @{$criteria->{RetargetingIDS}}) {
            $condition{ret_id} = $criteria->{RetargetingIDS};
        }

        my $retargetings = Retargeting::get_group_retargeting_ret_id_hash( %condition );

        $self->{preprocess}{retargetings} = $retargetings;

        if ( ! exists $self->{preprocess}{pid2bid} ) {
            my @pids = map { $_->{pid} } values %$retargetings;
            $self->{preprocess}{pid2bid} = Primitives::get_main_banner_ids_by_pids(@pids);
        }
    }

    return;
}

=head2 validate_retargeting_rights
    Проверка прав:
        - запрещено добавлять ретаргетинг к баннеру, относящемуся к кампании,
            на которую у пользователя нет права на редактирование
        - запрещено добавлять ретаргетинг к баннеру, относящемуся к
            несуществующей кампании
        - запрещено при вызове Add указывать id не существующих условий ретаргетинга
        - запрещено при вызове Update указывать id не существующего ретаргетинга
        - оператору с ролью media запрещено вызывать Add, Update и Delete
        - оператору с ролью superreader запрещено вызывать Delete
        - пользователю запрещено вызывать Get или Delete если он не имеет
            прав на просмотр кампаний, к которым относятся ретаргетинги, для
            которых вызывается метод
=cut

sub validate_retargeting_rights {
    my ($self, $params) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    my $action = $params->{Action};
    if ( $action eq 'Add' || $action eq 'Update' ) {

        my $pid2cid          = $self->{preprocess}{pid2cid};
        my $converted        = $self->{preprocess}{converted};
        my $old_retargetings = $self->{preprocess}{old_retargetings};
        my $cid2clientid     = $self->{preprocess}{cid2clientid};

        my @cids        = uniq values %$pid2cid;
        my $camp_rights = @cids ? rbac_user_allow_edit_camps_detail( $self->{rbac}, $self->{uid}, \@cids ) : {};
        my @clientids = uniq values %$cid2clientid;
        my $clientid2must_convert = Client::mass_client_must_convert(\@clientids);
        my $clientid2converting_soon = Client::mass_is_client_converting_soon(\@clientids);

        my ( $pid2client_id, $existing_conditions ); #, $isowner);
        if ( $action eq 'Add' ) {

            my $cid2uid       = get_cid2uid(cid => \@cids);
            my $uid2client_id = get_uid2clientid(uid => [ uniq values %$cid2uid ]);

            foreach my $pid ( keys %$pid2cid ) {
                my $cid = $pid2cid->{ $pid };

                my $uid = $cid2uid->{ $cid };
                next unless $uid;

                my $client_id = $uid2client_id->{ $uid };
                next unless $client_id;

                $pid2client_id->{ $pid } = $client_id;
            }

            my @clientids = uniq values %$pid2client_id;
            if ( @clientids ) {
                $existing_conditions = Retargeting::mass_get_retargeting_conditions_by_ClientIDS( \@clientids, short => 1 );
                #$isowner = rbac_mass_is_owner_of_client_id( $self->{rbac}, $self->{uid}, \@clientids );
            }

            $self->{cluid} = [uniq values %$cid2uid];
        }

        my $size = @$converted;
        for ( my $i = 0; $i < $size; $i++ ) {

            next if ( $self->{ret}[ $i ]{Errors} && scalar @{ $self->{ret}[ $i ]{Errors} } );

            my $curr_elem = $converted->[ $i ];

            my $pid = $curr_elem->{pid};
            if ( $action eq 'Add' && ! exists $pid2cid->{ $pid }
                 || exists $pid2cid->{ $pid } && !$camp_rights->{ $pid2cid->{ $pid } }
            ) {
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object('NoRights');
            } elsif (exists $pid2cid->{ $pid } && $clientid2must_convert->{ $cid2clientid->{ $pid2cid->{ $pid } } }) {
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object('NoRights', APICommon::msg_must_convert);
            } elsif (exists $pid2cid->{ $pid } && $clientid2converting_soon->{ $cid2clientid->{ $pid2cid->{ $pid } } }) {
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object('NoRights', APICommon::msg_converting_in_progress);
            } else {
                if ( $action eq 'Update' && ! exists $old_retargetings->{ $curr_elem->{ret_id} } ) {
                    push @{ $self->{ret}[ $i ]{Errors} }, get_error_object( 'BadRetargetingID', iget('Несуществующий RetargetingID') );
                }

                if ( $action eq 'Add' ) {
                    my $client_id = $pid2client_id->{ $pid };
                    #if ( !$isowner->{ $self->{uid} }{ $client_id } ) {
                    #    push @$errors, get_error_object('NoRights');
                    #} els
                    my $client_conditions = $existing_conditions->{ $client_id };
                    if ( ! $client_conditions || ! $client_conditions->{ $curr_elem->{ret_cond_id} } ) {
                        push @{ $self->{ret}[ $i ]{Errors} }, get_error_object( 'BadRetargetingConditionID', iget('Несуществующее условие ретаргетинга') );
                    }
                }
            }
        }

        $self->{syslog_data}->{cid} = \@cids;
        unless (defined $self->{cluid}) {
            my $syslog_data = get_syslog_data({cid => \@cids});

            $self->{cluid} = $syslog_data->{cluid};
        }
    } else {

        my $cids = get_cids(pid => [ uniq map { $_->{pid} } values %{ $self->{preprocess}{retargetings} } ]);
        if ( RBAC2::DirectChecks::rbac_cmd_user_allow_show_camps( $self->{rbac}, { cid => $cids, UID => $self->{uid} } ) ) {
            return ('NoRights');
        }

        $self->{syslog_data}->{cid} = $cids;
        my $syslog_data = get_syslog_data({cid => $cids});
        $self->{cluid} = $syslog_data->{cluid};

    }

    return;
}

=head2 pre_validate_retargeting
    Превалидация:
        - если к API обращается агентство или супер-пользователь,
            то обязательно должен быть передан логин
        - если обращаются к методам Add или Update и в параметрах
            указана ContextPrice, то проверяется цена
=cut

sub pre_validate_retargeting {
    my ($self, $params) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    my $action = $params->{Action};
    my $role   = $self->{rbac_login_rights}{role};

    if ( ( $role eq 'media' or $role eq 'superreader' ) and $action ne 'Get' ) {
        return ('NoRights');
    }

    # логин нужен для шардинга
    # агентства и супер-пользователи могут работать с данными разных пользователей
    # поэтому они передают логин в параметрах
    if ( $role ne 'client' ) {
        my @errors = _check_fields_exist( $params, [qw/Login/], def => 1, not_empty => 1, type => 'login' );
        return @errors if ( @errors );
    }

    if ( $action eq 'Add' || $action eq 'Update' ) {

        my @retargetings = @{ $params->{Retargetings} || [] };

        my $size = @retargetings;
        for ( my $i = 0; $i < $size; $i++ ) {

            my $ret = $retargetings[ $i ];

            my $context_price = $ret->{ContextPrice};

            next if ( ! defined $context_price || $context_price eq '' );

            # Q: ecли $ret->{ContextPrice} есть, но $ret->{Fields} !~ 'ContextPrice', то цена не валидируется
            # A: оставить такое поведение, расчет на то, что поля, не указанные в Fields, не обновляются
            my $need_check_cp;
            if ( $ret->{Fields} ) {
                for my $field ( @{ $ret->{Fields} } ) {
                    $need_check_cp = 0 if $field eq 'ContextPrice';
                }
            } else {
                $need_check_cp = 1;
            }
            next if !$need_check_cp;

            # multicurrency: валидируем сумму до конвертации по исходным данным
            my $error = validate_phrase_price( $context_price, $ret->{Currency} || 'YND_FIXED' );
            if ( $error ) {
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object( 'BadPrice', $error );
            }
        }
    }

    return;
}

=head2 validate_retargeting
    Валидация ретаргетингов:
        - разрешено добавлять ретаргетинг только в группы определенных типов
        - запрещено добавлять или менять ретаргетинг к архивированной группе
            (группа, где все баннеры имеют статус архивированных)
        - при добавлении или обновлении ретаргетинга к баннеру, относящемуся к
            кампании, чья стратегия не является автобюджетной, валидируется
            валюта и цена ретаргетинга
        - если при запросе списка ретаргетингов указана опция - валюта для
            отображения цен ретаргетингов, то проверяется допустимость значения
            опции и то, что у всех ретаргетингов в ответе одинаковая валюта ценя
=cut

sub validate_retargeting {
    my ( $self, $params ) = @_;

    local $Direct::Validation::RetargetingConditions::ERROR_TEXT_VARIANT = 'api4';

    my $action = $params->{Action};
    if ( $action eq 'Add' || $action eq 'Update' ) {

        my $camps         = $self->{preprocess}{camps};
        my $pid2cid       = $self->{preprocess}{pid2cid};
        my $converted     = $self->{preprocess}{converted} || [];
        my $camp_strategy = $self->{preprocess}{camp_strategy};
        my $adgroup_types = $self->{preprocess}{adgroup_types} // {};
        my $retargeting_conditions_by_id = $self->{preprocess}{retargeting_conditions_by_id} // {};

        my %not_arch_groups = map { $_ => 1 } @{ filter_arch_groups([ keys %$pid2cid ]) };

        my $size = @$converted;
        for ( my $i = 0; $i < $size; $i++ ) {

            next if ( $self->{ret}[ $i ]{Errors} && scalar @{ $self->{ret}[ $i ]{Errors} } );

            my $curr_elem = $converted->[ $i ];

            if ($action eq 'Add') {
                my $retargeting_condition = $retargeting_conditions_by_id->{$curr_elem->{ret_cond_id}};
                unless (exists $adgroup_types->{$curr_elem->{pid}}) {
                    push @{$self->{ret}[ $i ]{Errors}}, get_error_object('NoRights');
                    next;
                } elsif (!API::ObjectRelations::is_eligible_container_item('adgroup', $adgroup_types->{$curr_elem->{pid}}, 'retargeting')) {
                    push @{$self->{ret}[ $i ]{Errors}},
                        get_error_object('NotSupported', iget('Тип группы объявлений не поддерживается. Объявление: %s', $curr_elem->{bid}));
                    next;
                } else {
                    if ($retargeting_condition) {
                        if ($retargeting_condition->is_negative) {
                            push @{$self->{ret}[ $i ]{Errors}},
                            get_error_object('BadParams',
                                iget('Это условие ретаргетинга можно использовать только для корректировки ставок')
                            );
                        }
                    }
                }
            }

            if ( not exists $not_arch_groups{ $curr_elem->{pid} } ) {
                my $msg = iget( 'Все объявления группы %d перенесены в архив и поэтому группа недоступна для редактирования', $curr_elem->{pid} );
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object( 'ArchiveEdit', $msg );
            }

            my $cid = $pid2cid->{ $curr_elem->{pid} };
            next if $camp_strategy->{ $cid }{is_autobudget};

            if (!$curr_elem->{currency} || $curr_elem->{currency} eq 'YND_FIXED') {
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object('BadCurrency');
                next;
            }

            # multicurrency
            if ( defined $curr_elem->{currency} ) {

                my $camp_info = $camps->{ $cid };
                my @allowed_currencies = $camp_info->{currency} && $camp_info->{currency} ne 'YND_FIXED' ? ( $camp_info->{currency} ) : ();

                # копируем в отдельный хэш, чтобы функция формировала корректное собощение об ошибке
                my $currency_params = { Currency => $curr_elem->{currency} };

                my @err = _check_fields_exist( $currency_params, [qw/Currency/], type => 'currency', list => \@allowed_currencies );
                if ( @err ) {
                    push @{ $self->{ret}[ $i ]{Errors} }, get_error_object( @err );
                    # если валюта указана неверная - дальше не имеет смысла проверять ставку
                    next;
                }
            }

            # multicurrency: в preprocessed_retargeting происходит конвертация данных в валюту кампании - поэтому MIN_PRICE и MAX_PRICE верны.
            # Проверка оставлена для потенциальной проверки других логик.
            my $err = validate_phrase_price( $curr_elem->{price_context},  $curr_elem->{price_currency} || 'YND_FIXED');
            if ( $err ) {
                push @{ $self->{ret}[ $i ]{Errors} }, get_error_object( 'BadPrice', $err );
            }
        }
    } elsif ( $action eq 'Get' ) {

        return ('BadCurrency') if !$params->{Options}{Currency} || $params->{Options}{Currency} eq 'YND_FIXED';

        my $retargetings = $self->{preprocess}{retargetings};
        if ( defined $params->{Options}{Currency} && scalar keys %$retargetings ) {

            my @ret_currencies     = uniq map { $_->{currency} } values %$retargetings;
            my @allowed_currencies = grep { $_ ne 'YND_FIXED' } @ret_currencies;

            return ('BadCurrency') if scalar @ret_currencies > 1;

            my @errors = _check_fields_exist( $params->{Options}, [qw/Currency/], type => 'currency', list => \@allowed_currencies );
            return @errors if ( @errors );
        }
    }

    return;
}

1;
