package API::Methods::Tags;

# $Id$

=head1 NAME
    
    API::Methods::Tags

=head1 DESCRIPTION

    Методы для работы с метками

=cut

use strict;
use warnings;

use List::MoreUtils qw/uniq/;

use Settings;
use APICommon qw(:subs);
use API::Errors;
use PrimitivesIds;
use Tag;

use utf8;

=head2 GetCampaignsTags

    Возвращает список меток для указанной кампании или нескольких кампаний.
    http://api.yandex.ru/direct/doc/live/GetCampaignsTags.xml

=cut
sub GetCampaignsTags {
    my ($self, $params) = @_;

    my @cids = uniq @{ $params->{CampaignIDS} };

    my $syslog_data = get_syslog_data({cid => \@cids});

    $self->{syslog_data}->{cid} = \@cids;
    $self->{cluid} = $syslog_data->{cluid};

    return get_campaigns_tags_struct(\@cids);
}

=head2 UpdateCampaignsTags

    Формирует список меток для указанной кампании или списки для нескольких кампаний.
    http://api.yandex.ru/direct/doc/live/UpdateCampaignsTags.xml

=cut
sub UpdateCampaignsTags {
    my ($self, $params) = @_;

    my @cids = map {$_->{CampaignID}} @$params;

    my $error_code = check_rbac_rights( $self, 'api_editCamps', {cid => \@cids, UID => $self->{uid} } );
    dieSOAP( 'NoRights' ) if $error_code;

    foreach my $campaign_tags (@$params) {
        my @tags;
        foreach my $tag (@{ $campaign_tags->{Tags} }) {
            my $tag_text = Tag::trim($tag->{Tag});
            next unless $tag_text;

            push @tags, { name => $tag_text, tag_id => $tag->{TagID} };
        }
        Tag::save_campaign_tags($campaign_tags->{CampaignID}, \@tags);
    }

    my $syslog_data = get_syslog_data({cid => \@cids});

    $self->{syslog_data}->{cid} = \@cids;
    $self->{cluid} = $syslog_data->{cluid};

    return get_campaigns_tags_struct(\@cids);
}

=head2 GetBannersTags

    Возвращает идентификаторы меток, присвоенных объявлениям.
    http://api.yandex.ru/direct/doc/live/GetBannersTags.xml

=cut
sub GetBannersTags {
    my ($self, $params) = @_;

    my $bid2pid = {};
    if ($params->{BannerIDS} && scalar @{$params->{BannerIDS}}) {
        my $bids = $params->{BannerIDS};
        if ($params->{CampaignIDS} && scalar @{$params->{CampaignIDS}}) {
            # если указаны и CampaignIDS и BannerIDS - фильтруем баннеры по номерам кампаний
            my %cids = map {$_=>1} @{$params->{CampaignIDS}};
            my $bid2cid = get_bid2cid(bid => $bids);
            $bids = [grep {$bid2cid->{$_} && $cids{$bid2cid->{$_}}} @$bids];
        }
        $bid2pid = get_bid2pid(bid => $bids);
    } elsif ($params->{CampaignIDS} && scalar @{$params->{CampaignIDS}}) {
        $bid2pid = get_bid2pid(cid => $params->{CampaignIDS});
    }

    my $pids = [uniq values %$bid2pid];
    my $group_tags = Tag::get_groups_tags(pid => $pids);

    my @result;
    foreach my $bid (sort keys %{$bid2pid}) {
        push @result, {BannerID => $bid, TagIDS => $group_tags->{$bid2pid->{$bid}} || []};
    }
    
    my $syslog_data = get_syslog_data({bid => [keys %$bid2pid]});

    $self->{syslog_data}->{cid} = $syslog_data->{cid};
    $self->{cluid} = $syslog_data->{cluid};

    return \@result;
}

=head2 UpdateBannersTags

    Присваивает метки объявлениям.
    http://api.yandex.ru/direct/doc/live/UpdateBannersTags.xml

=cut
sub UpdateBannersTags {
    my ($self, $params) = @_;

    my %save_tags;
    my %tags_pids;

    my @bids = map {$_->{BannerID}} @$params;
    my $bid2pid = get_bid2pid(bid => \@bids);
    foreach my $banner (@$params) {
        my $pid = $bid2pid->{$banner->{BannerID}};
        foreach my $tag_id (@{$banner->{TagIDS} || []}) {
            $save_tags{$tag_id}{$pid} = 1;
        }
        $tags_pids{$pid} = 1;
    }

    foreach my $tag_id (keys %save_tags) {
        $save_tags{$tag_id} = [keys %{$save_tags{$tag_id}}];
    }
    my $cids = get_cids(pid => [keys %tags_pids]);

    my $error_code = check_rbac_rights( $self, 'api_editCamps', {cid => $cids, UID => $self->{uid} } );
    dieSOAP( 'NoRights' ) if $error_code;

    Tag::save_tags_groups([keys %tags_pids], \%save_tags);

    my $syslog_data = get_syslog_data({cid => $cids});

    $self->{syslog_data}->{cid} = $cids;
    $self->{cluid} = $syslog_data->{cluid};

    return 1;
}

1;
