package API::ReportCommon;
## no critic (TestingAndDebugging::RequireUseWarnings)

=pod

    $Id$

=head1 NAME

API::ReportCommon - Функции для работы с отчётами, очередями отчётов в API

=head1 DESCRIPTION

    Авторы:
        Vasiliy Bryadov <mirage@yandex-team.ru>
        Alexey Ziyangirov <metallic@yandex-team.ru>

=cut

use Direct::Modern;

use Carp qw/croak/;

use Settings;
use API::Settings;

use Direct::Storage;
use Yandex::I18n;

use ADVQ6;
use Digest::MD5 qw(md5_hex);
use EnvTools;
use Forecast;

use Yandex::DBQueue;
use Yandex::DBShards;
use Yandex::Trace;

use Encode qw/decode_utf8/;
use YAML ();
use List::Util qw(minstr max);
use Primitives;
use PrimitivesIds;
use Property;

use base qw( Exporter );
our @EXPORT = qw(

    create_queue_report_request

    get_queue_report_db
    get_queue_reports_list
    get_queue_report_file

    parse_forecast_options

    sync_wordstat
    get_forecast

    report_options_load

    wordstat_common_report
    calculate_priority
);

# таблицы для разных типов отчетов
our %TABLE_BY_REPORT_TYPE = (
    'forecast' => "api_queue_forecast",
);

# типы заданий в таблице dbqueue_jobs для разных типов отчётов
our %JOB_TYPE_BY_REPORT_TYPE = (
    'report' => 'api_report',
    'wordstat' => 'api_wordstat',
    'forecast' => 'api_forecast',
);

# сколько секунд ждать ответа ADVQ
my $ADVQ_TIMEOUT_PER_PHRASE = 3.0;

=head2 check_count_reports(type, uid)

    Функция считает кол-во отчетов заданного типа у пользователя

=cut

sub check_count_reports($$)
{
    my $type = shift;
    my $uid = shift;

    die "Report type must be specified" unless validate_report_type($type);

    my $ClientID = get_clientid(uid => $uid);
    my $queue = Yandex::DBQueue->new(PPC(ClientID => $ClientID), $JOB_TYPE_BY_REPORT_TYPE{$type});
    return $queue->count_jobs(ClientID => $ClientID, uid => $uid);
}

=head2 get_client_reports_limit(client_id)

    Количество одновременно хранимых на сервере отчетов, заказанных по
    CreateNewReport

    В случае если не задано в БД для клиента используется
    значение по умолчанию из API::Settings::MAX_NUM_REPORTS Данный метод
    предназанчен ТОЛЬКО для проверки лимита на отчеты CreateNewReport (в то
    время как переменная MAX_NUM_REPORTS также является максимумом для Forecast
    и WordStat отчетов)

=cut

sub get_client_reports_limit {
    my $client_id = shift;
    return query_special_user_option($client_id, "api_reports_limit") // $API::Settings::MAX_NUM_REPORTS;
}


=head2 create_queue_report_request(type, uid, options)

      Добавляет в очередь отчетов на обработку новую запись.
      Параметры:
          type = forecast/wordstat|reports
          options

=cut

sub create_queue_report_request($$$)
{
    my ($type, $uid, $options) = @_;

    die "create_queue_report_request: invalid type of entry: $type" unless validate_report_type($type);

    my $count_reports = check_count_reports($type, $uid);

    my $client_id = get_clientid(uid => $uid);

    my $limit = $type eq 'report' ? get_client_reports_limit($client_id) : $API::Settings::MAX_NUM_REPORTS;
    if ($count_reports >= $limit) {
        return ['ReportsStackFilled', iget('Очередь отчетов уже содержит %s заказов', $count_reports)];
    }

    # здесь можно задаться вопросом, что обозначает priority для отличных от wordstat отчётов.
    my $advq_priority = query_special_user_option($client_id, 'advq_report_queue_priority') || $API::Settings::WORDSTAT_DEFAULT_PRIORITY;

    my $queue = Yandex::DBQueue->new(PPC(ClientID => $client_id), $JOB_TYPE_BY_REPORT_TYPE{$type});
    my $job = $queue->insert_job( {
        job_id => get_new_id('job_id'),
        ClientID => $client_id,
        uid => $uid,
        args => $options,
        priority => $advq_priority,
    } );

    return $job->job_id % $API::Settings::API4_JOB_ID_THRESHOLD;
}

=head2 parse_forecast_options($self, $params)

    $self -- такой же $self, как в методах APIMethods
    $params -- данные от пользователя

=cut

sub parse_forecast_options($$) {
    my ($self, $params) = @_;

    # multicurrency: принимать в параметрах метода желаемую валюту
    my $report_currency = $self->{api_version_full} > 4 ? $params->{Currency} || 'YND_FIXED' : undef;

    my $options = {
        geo => $params->{GeoID} ? join(',', grep {/^-?\d+$/} @{$params->{GeoID}}) : 0,
        currency => $report_currency || 'YND_FIXED',
        auction_bids => ($params->{AuctionBids} && lc($params->{AuctionBids}) eq 'yes') ? 1 : 0
    };

    $options->{Phrases} = defined $params->{Phrases} ? [ map { { phrase => $_ } } grep{/\S+/} @{$params->{Phrases}} ] : [];

    return $options;
}

=head2 get_forecast($options, %O)

    Обёртка над Forecast::forecast_calc для коммандеровской версии API

=cut

sub get_forecast($%) {
    my ($options, %O) = @_;
    %O = () unless keys %O;
    my $is_get_data_from_bs = Property->new('forecast_is_get_data_from_bs')->get();

    if ( $is_get_data_from_bs ) {
        Forecast::forecast_calc([ $options ], %O );
    } else {
        Forecast::forecast_calc_new([ $options ], positions => [qw/P11 P12 P13 P14 P21 P22 P23 P24/], with_nds => 0, %O );
    }

    return $options;
}

=head2 get_queue_report_db(type, id, uid)

    Возвращает из базы запрос на отчет

    table.options - остается в сериализованом виде (YAML)

=cut

sub get_queue_report_db($$$)
{
    my ($type, $id, $uid) = @_;

    die "Incorrect report type or report_id" unless validate_report_type($type) && $id && $id =~ /^\d+$/;

    die "Missing required parameter: uid" unless $uid;

    my $ClientID = get_clientid(uid => $uid);
    my $queue = Yandex::DBQueue->new(PPC(ClientID => $ClientID), $JOB_TYPE_BY_REPORT_TYPE{$type});
    $id %= $API::Settings::API4_JOB_ID_THRESHOLD;
    $id += $API::Settings::API4_JOB_ID_SHIFT if !is_sandbox();
    my $job = $queue->find_job_by_id($id);

    if ($job && $job->ClientID == $ClientID && $job->uid == $uid) {
        if ($job->status eq 'Revoked') {
            return undef;
        }
        return $job;
    }

    return undef;
}

=head2 get_queue_report_file(type, id, uid; file_name)

Читает содержимое готового файла с отчетом и возвращает.
Путь к файлу формируется автоматически, если не задан, согласно типу отчета.

=cut

sub get_queue_report_file($$$;$)
{
    my $type = shift || return;
    my $id = shift || return;
    my $uid = shift or croak "uid required";
    my $file_name = shift;
    my $client_id = get_clientid(uid => $uid);

    die "Incorrect report type or report_id" unless validate_report_type($type) && $id && $id =~ /^\d+$/;

    die "Not implemented for type = report" if $type eq 'report';

    unless (defined $file_name) {
        $file_name = $type . '_' . $id;
    }

    my $storage = Direct::Storage->new();
    my $file;
    if ($file_name =~ /_nameless_/) {
        $file = $storage->get_file("api_$type".'_nameless', filename => $file_name, ClientID => $client_id);
    } else {
        $file = $storage->get_file("api_$type", filename => $file_name, ClientID => $client_id);
    }
    if ($file) {
        return $file->content ? report_options_load(decode_utf8($file->content)) : "";
    }
    else {
        return undef;
    }
}

=head2 get_queue_reports_list(type, uid)

    Возвращает данные по отчетам указанного типа для данного пользователя

=cut

sub get_queue_reports_list
{
    my ($type, $uid) = @_;

    my $profile = Yandex::Trace::new_profile('api_report:get_queue_reports_list');

    die "get_queue_report_list: invalid type of entry: $type" unless validate_report_type($type);

    my $jobs;
    my $ClientID = get_clientid( uid => $uid );
    my $queue = Yandex::DBQueue->new( PPC( ClientID => $ClientID ), $JOB_TYPE_BY_REPORT_TYPE{$type} );
    $jobs = $queue->find_jobs(ClientID => $ClientID, uid => $uid, status__not_in => ['Revoked']);

    return $jobs;
}

=head2 validate_report_type($type)

    Возвращает 1 или 0: есть ли такой тип отчёта?

=cut

sub validate_report_type($)
{
    my ($type) = @_;

    return 0 unless $type && $type =~ /^(forecast|wordstat|report)$/;

    return 1;
}

=head2 wordstat_common_report(phrases, geo)

обёртка над _wordstat_common с заранее подготовленными параметрами для «медленных» отчётов

=cut

sub wordstat_common_report {
    my ($phrases, $geo) = @_;
    return _wordstat_common($ADVQ_TIMEOUT_PER_PHRASE, $phrases, $geo, 300);
}

=head2 _wordstat_common(phrase_timeout, phrases, geo)

Общая часть синхронных и асинхронных запросов к методам, обращающимся к AdvQ

=cut

sub _wordstat_common($$$$) {
    my ($phrase_timeout, $phrases, $geo, $ph_page_size)=@_;
    my $geo_string = join(',', @{$geo || []});
    my $res = advq_get_phrases_detailed_stat($geo_string, $phrases, 0, $ph_page_size, $phrase_timeout);

    my @filtered;

    foreach my $ph (@$res) {
        my $item = {Phrase => $ph->{phrase}, GeoID => $geo || []};

        foreach (@{$ph->{including_phrases}}) {
            push @{$item->{SearchedWith}}, {Phrase => $_->{phrase}, Shows => $_->{cnt}};
        }

        foreach (@{$ph->{associations}}) {
            push @{$item->{SearchedAlso}}, {Phrase => $_->{phrase}, Shows => $_->{cnt}};
        }

        # для фразы в запросе в кавычках, проверяем, возвращаем ли мы direct match (поле count в ответе вордстата),
        # и отдаем его первым элементом
        if (! scalar grep {$_->{phrase} eq $ph->{phrase}} @{$ph->{including_phrases}} ) {
            unshift @{$item->{SearchedWith}}, {Phrase => $ph->{phrase}, Shows => $ph->{count}};
        }

        push @filtered, $item;
    }
    return (\@filtered, $res);
}

=head2 sync_wordstat

Создаёт и отдаёт отчет AdvQ без промежуточного сохранения

=cut

sub sync_wordstat($$) {
    my ($phrases, $geo)=@_;

    my $ADVQ_TIMEOUT_PER_PHRASE = 1;
    my $max_length = max map {length($_)} @$phrases;
    $ADVQ_TIMEOUT_PER_PHRASE *= int($max_length / 1_000);
    $ADVQ_TIMEOUT_PER_PHRASE = 1 unless $ADVQ_TIMEOUT_PER_PHRASE;

    my ($result, undef) = _wordstat_common($ADVQ_TIMEOUT_PER_PHRASE, $phrases, $geo, 600);
    return $result;
}

my %age_translation = (
    undefined => 'AGE_UNKNOWN',
    '0-17' => 'AGE_0_17',
    '18-24' => 'AGE_18_24',
    '25-34' => 'AGE_25_34',
    '35-44' => 'AGE_35_44',
    '45-' => 'AGE_45',
);

my %gender_translation = (
    undefined => 'GENDER_UNKNOWN',
    male => 'GENDER_MALE',
    female => 'GENDER_FEMALE',
);

my %detailed_device_type_translation = (
    undefined => 'OS_TYPE_UNKNOWN',
    android => 'ANDROID',
    ios => 'IOS',
    other => 'OS_TYPE_UNKNOWN',
);

my %connection_type_translation = (
    undefined => 'CARRIER_TYPE_UNKNOWN',
    mobile => 'CELLULAR',
    stationary => 'STATIONARY',
);

=head2 report_options_load

    Декодирование параметров отчёта

=cut

sub report_options_load {
    return YAML::Load(shift);
}

=head2 calculate_priority

Для демонов: зная uniq и шаг приоритетов, понять, к какой группе по приоритету относится воркер.

=cut

sub calculate_priority {
    my ($UNIQ) = @_;
    my $groups_number;
    $groups_number = 101 / $API::Settings::WORDSTAT_PRIORITY_STEP;
    $groups_number = int($groups_number) + 1 if ($groups_number != int $groups_number); # округление вверх

    my $group = $UNIQ % $groups_number;
    $group = $groups_number unless $group; # если групп n, то нулевая на самом деле n-тая
    return (($group - 1) * $API::Settings::WORDSTAT_PRIORITY_STEP);
}

1;
