package API::Reports::Builder;
use Direct::Modern;

=head1 NAME

API::Reports::Builder

=head1 DESCRIPTION

Построитель отчётов: берёт API::Reports::InternalRequestRepresentation и строит отчёт,
который можно куда-то выводить, например, отдавать прямо клиенту (онлайн) или сохранять в MDS (офлайн).

Единственная процедура в интерфейсе, build_report, берёт API::Reports::InternalRequestRepresentation
и возвращает API::Reports::Result.

=cut

use Yandex::I18n;
use Yandex::HashUtils;

use API::Reports::DataRules;
use API::Reports::Result;
use Stat::CustomizedArray;

=head2 $class->build_report($request)

=cut

sub build_report {
    my ( $class, $request ) = @_;

    die "Invalid format: " . $request->response_format unless $request->response_format eq 'tsv';

    my $lang_guard = Yandex::I18n::init_i18n_guard( $request->lang );

    my ( $stat_iterator, $total_rows, $start_date, $end_date, $stat_source );
    if ( $request->need_to_request_report_from_provider ) {
        my $report_params = hash_copy($request->provider_request_params);
        $report_params->{options}->{external_countable_fields_override} = API::Reports::DataRules::get_countable_fields(@{$request->{displayed_field_names}});
        my $reporter = Stat::CustomizedArray::StreamedExt->new( %$report_params );

        $stat_iterator = $reporter->generate_streamed_ext_iterator({no_need_days_num => 1});
        $total_rows = $stat_iterator->headers->{row_num};
        $start_date = $stat_iterator->headers->{totals}->{stat_date};
        $end_date = $stat_iterator->headers->{totals}->{stat_date_end};
        $stat_source = $stat_iterator->headers->{stat_source};
    } else {
        my $report_params = $request->provider_request_params;

        $stat_iterator = undef;
        $total_rows = 0;
        $start_date = $report_params->{start_date};
        $end_date = $report_params->{end_date};
        $stat_source = 'no_stat';
    }

    return API::Reports::Result->new(
        title => $class->_make_report_title($request->report_name, $start_date, $end_date),
        format => $request->response_format,
        header => $request->displayed_field_names,
        stat_source => $stat_source,
        _lang_guard => $lang_guard,
        _requested_report_fields => $request->field_names_in_provider_response,
        _need_conversion_to_micro_units => $request->return_money_in_micros,
        _date_aggregation_by => $request->provider_request_params->{date_aggregation_by},
        _uac_oids => $request->provider_request_params->{uac_oid},
        _report_type => $request->report_type,
        _stat_iterator => $stat_iterator,
        _total_rows => $total_rows,
        _start_date => $start_date,
        _end_date => $end_date,
    );
}

sub _make_report_title {
    my ( $class, $name, $start_date, $end_date ) = @_;

    if ( defined $start_date ) {
        return sprintf('"%s (%s - %s)"', $name, $start_date, $end_date);
    }

    return sprintf('"%s"', $name);
}

1;
