package API::Reports::DataRules;

=head1 NAME

API::Reports::DataRules

=head1 DESCRIPTION

Константы и функции для описания, как Reports работает с данными: что есть в разных
полях клиентского запроса (FieldNames, OrderBy, Filter и т. п.), как это превращать
в запрос в CustomizedArray и как превращать ответ в ответ клиента.

Все эти данные-функции используются в нескольких разных местах, но в частности,
в Reports.pm и в Reports/Validation.pm.

=cut

use Direct::Modern;
use Exporter qw/import/;

use List::MoreUtils qw/any uniq/;
use Hash::Util qw/lock_hash_recurse/;
use Yandex::DateTime qw/now/;
use Yandex::DBTools;
use Yandex::I18n;
use Yandex::ListUtils qw/xisect/;

use my_inc '../../../', for => 'api/services/v5';

use Stat::Const qw/
    %AGES
    %BANNER_IMAGE_TYPES
    $BS_SEARCH_QUERIES_LAST_DAYS
    %CLICK_PLACES
    %CONNECTION_TYPES
    %CONTEXT_COND_TYPES
    %CRITERION_TYPES
    %DETAILED_DEVICE_TYPES
    %DEVICE_TYPES
    %GENDERS
    %MATCH_TYPES
    %POSITION_TYPES
    %TARGET_TYPES
    %CAMP_TYPES
    %ATTRIBUTION_MODEL_TYPES
    %TARGETING_CATEGORIES
    %PRISMA_INCOME_GRADES
/;
use Stat::Fields;

use Settings;

our @EXPORT_OK = qw/
    get_all_field_names
    get_all_operators
    get_all_order_directions
    get_all_report_types
    get_all_date_ranges

    %OPERATORS_MAP_TO_INTERNAL
    %ORDER_DIRECTION_MAP
    %DATE_RANGE_TYPE_MAP
    %RESULT_MAP_BY_FIELD
    %DATE_AGGREGATION_MAP
    %FILTER_PARAMS_MAP
    %FIELD_DESC
    %OPERATOR_DESC_MAP
    %MONEY_FIELD_MAP
    %MONEY_INTERNAL_FIELD_MAP
    %FORMAT_MAP
    %RESULT_FIELD_MAP
    %GROUP_BY_FIELD_MAP
    %MANDATORY_GROUP_BY_FIELDS
    %ORDER_BY_FIELD_MAP
    %INCOMPATIBLE_FIELDS_MAP
    $MONEY_MULT
    %SINGLE_VALUE_OPERATOR
    %SPECIAL_VALUE_FOR_RELATED_KEYWORD
    %INTERNAL_FIELD_TYPE_MAP
    %FIELD_TYPE_MAP
    %REPORT_TYPE_CONF
    %FIELDS_DEPENDENT_ON_GOAL_MAP
    %ATTRIBUTION_MODEL_TYPE_MAP
    @REPORT_TYPES
    @CRITERIA_RELATED_FILTER_FIELDS
    %COUNTABLE_FIELDS
    %MANDATORY_FIELDS
    %UAC_FIELDS_TO_HIDE

    TYPE_STRING
    TYPE_ID
    TYPE_INTEGER
    TYPE_FLOAT
    TYPE_BOOLEAN
    TYPE_MONEY
    TYPE_SIGNED_MONEY
    TYPE_GEO_REGION

    DEFAULT_SORT_ORDER
    EMPTY_NUMERIC_VALUE
    EMPTY_STRING_VALUE
    CAMPAIGN_PERFORMANCE_REPORT_TYPE
    STRATEGY_PERFORMANCE_REPORT_TYPE
    CUSTOM_DATE_RANGE_TYPE
    CUSTOM_REPORT_TYPE
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE
    ADGROUP_PERFORMANCE_REPORT_TYPE
    ACCOUNT_PERFORMANCE_REPORT_TYPE
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE

    AD_NETWORK_TYPE_SEARCH
    PLACEMENT_YANDEX

    AD_NETWORK_TYPE_INTERNAL_FIELD
    CRITERIA_ID_INTERNAL_FIELD
    CRITERIA_TYPE_AUDIENCE_TARGET
    CRITERIA_TYPE_AUTOTARGETING
    CRITERIA_TYPE_INTERNAL_FIELD
    CRITERIA_TYPE_KEYWORD
    MATCH_TYPE_INTERNAL_FIELD
    PLACEMENT_INTERNAL_FIELD_FOR_FILTER
    QUERY_INTERNAL_FIELD
    TARGET_LOCATION_NAME_INTERNAL_FIELD
    TARGET_LOCATION_NOT_EXACTLY_DETECTED_INTERNAL_FIELD
    LOCATION_OF_PRESENCE_NAME_INTERNAL_FIELD
    LOCATION_OF_PRESENCE_NOT_EXACTLY_DETECTED_INTERNAL_FIELD
    NOT_EXACTLY_DETECTED_LOCATION_PREFIX
/;

our %EXPORT_TAGS = (
    types => [qw/TYPE_STRING TYPE_ID TYPE_INTEGER TYPE_FLOAT TYPE_BOOLEAN TYPE_MONEY TYPE_GEO_REGION TYPE_SIGNED_MONEY/],
);

use constant TYPE_STRING => 'String';
use constant TYPE_ID => 'Id';
use constant TYPE_INTEGER => 'Integer';
use constant TYPE_FLOAT => 'Float';
use constant TYPE_BOOLEAN => 'Boolean';
use constant TYPE_MONEY => 'Money';
use constant TYPE_SIGNED_MONEY => 'SignedMoney'; # отличие в валидации по сравнению с типом Money (DIRECT-92379)
use constant TYPE_GEO_REGION => 'Region';

use constant DEFAULT_SORT_ORDER => 'ASCENDING';
use constant EMPTY_NUMERIC_VALUE => '--';
use constant EMPTY_STRING_VALUE => '';
use constant CUSTOM_DATE_RANGE_TYPE => 'CUSTOM_DATE';
use constant ACCOUNT_PERFORMANCE_REPORT_TYPE => 'ACCOUNT_PERFORMANCE_REPORT';
use constant AD_PERFORMANCE_REPORT_TYPE => 'AD_PERFORMANCE_REPORT';
use constant CAMPAIGN_PERFORMANCE_REPORT_TYPE => 'CAMPAIGN_PERFORMANCE_REPORT';
use constant STRATEGY_PERFORMANCE_REPORT_TYPE => 'STRATEGY_PERFORMANCE_REPORT';
use constant CRITERIA_PERFORMANCE_REPORT_TYPE => 'CRITERIA_PERFORMANCE_REPORT';
use constant CUSTOM_REPORT_TYPE => 'CUSTOM_REPORT';
use constant SEARCH_QUERY_PERFORMANCE_REPORT_TYPE => 'SEARCH_QUERY_PERFORMANCE_REPORT';
use constant ADGROUP_PERFORMANCE_REPORT_TYPE => 'ADGROUP_PERFORMANCE_REPORT';
use constant REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE => 'REACH_AND_FREQUENCY_PERFORMANCE_REPORT';

use constant US_LOCALE => 'en-US';

use constant API_COMSUMER_KEY_NAME => 'api';
use constant TRUE => 1;

use constant AD_ID_INTERNAL_FIELD_FOR_GROUP_BY => 'banner';
use constant ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY => 'adgroup';
use constant CAMPAIGN_ID_INTERNAL_FIELD_FOR_GROUP_BY => 'campaign';
use constant STRATEGY_ID_INTERNAL_FIELD => 'strategy_id';

use constant AD_NETWORK_TYPE_SEARCH => 'SEARCH';

use constant AD_NETWORK_TYPE_INTERNAL_FIELD => 'targettype';
use constant CRITERIA_ID_INTERNAL_FIELD => 'PhraseID';
use constant CRITERIA_ID_OWN_MOD_INTERNAL_FIELD => 'bs_criterion_id';
use constant CRITERION_ID_MOD_INTERNAL_FIELD => CRITERIA_ID_OWN_MOD_INTERNAL_FIELD;
use constant CRITERIA_TYPE_INTERNAL_FIELD => 'contexttype';
use constant CRITERION_TYPE_INTERNAL_FIELD => 'criterion_type';
use constant CRITERIA_TYPE_ORIG_INTERNAL_FIELD => 'contexttype_orig';
use constant CRITERIA_MOD_INTERNAL_FIELD => 'contextcond_mod';
use constant CRITERIA_EXT_MOD_INTERNAL_FIELD => 'contextcond_ext_mod';
use constant CRITERIA_ORIG_MOD_INTERNAL_FIELD => 'contextcond_orig_mod';
use constant MATCH_TYPE_INTERNAL_FIELD => 'match_type';
use constant MATCHED_KEYWORD_INTERNAL_FIELD => 'matched_phrase';
use constant CRITERIA_TYPE_AUDIENCE_TARGET => 'AUDIENCE_TARGET';
use constant CRITERIA_TYPE_KEYWORD => 'KEYWORD';
use constant CRITERIA_TYPE_DYNAMIC_TEXT_AD_TARGET => 'DYNAMIC_TEXT_AD_TARGET';
use constant CRITERIA_TYPE_AUTOTARGETING => 'AUTOTARGETING';
use constant CRITERION_TYPE_KEYWORD => 'KEYWORD';
use constant CRITERION_TYPE_AUTOTARGETING => 'AUTOTARGETING';
use constant CRITERION_TYPE_FEED_FILTER => 'FEED_FILTER';
use constant CRITERION_TYPE_WEBPAGE_FILTER => 'WEBPAGE_FILTER';
# название фильтра постепенно переезжает на page_name, см. описание Placement в %FIELD_DESC
use constant PLACEMENT_INTERNAL_FIELD_FOR_FILTER => 'page';
use constant QUERY_INTERNAL_FIELD => 'search_query';
use constant TARGET_LOCATION_NAME_INTERNAL_FIELD => 'region_name';
use constant TARGET_LOCATION_NOT_EXACTLY_DETECTED_INTERNAL_FIELD => 'region_not_exactly_detected';
use constant LOCATION_OF_PRESENCE_NAME_INTERNAL_FIELD => 'physical_region_name';
use constant LOCATION_OF_PRESENCE_NOT_EXACTLY_DETECTED_INTERNAL_FIELD => 'physical_region_not_exactly_detected';
use constant NOT_EXACTLY_DETECTED_LOCATION_PREFIX => 'original_';
use constant TARGETING_CATEGORY_FIELD => 'targeting_category';
use constant INCOME_GRADE_FIELD => 'prisma_income_grade';

use constant PLACEMENT_YANDEX => 'Яндекс';

our @CRITERIA_RELATED_FILTER_FIELDS = qw/AudienceTargetId CriteriaType CriterionType DynamicTextAdTargetId Keyword SmartBannerFilterId SmartAdTargetId/;

our $MONEY_MULT = 1000000;

our %DATE_AGGREGATION_MAP = (
    Date => 'day',
    Week => 'week',
    Month => 'month',
    Quarter => 'quarter',
    Year => 'year',
);

our @AD_NETWORK_TYPE_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %TARGET_TYPES;
our @AGE_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %AGES;
our @CARRIER_TYPE_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %CONNECTION_TYPES;
our @CLICK_TYPE_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %CLICK_PLACES;
our @CRITERIA_TYPE_ENUM = sort(uniq(map { $_->{+API_COMSUMER_KEY_NAME} } values %CONTEXT_COND_TYPES));
our @CRITERION_TYPE_ENUM = sort(uniq(map { $_->{+API_COMSUMER_KEY_NAME} } values %CRITERION_TYPES));
our @SEARCH_QUERY_PERFORMANCE_REPORT_CRITERIA_TYPE_ENUM = sort @{xisect(\@CRITERIA_TYPE_ENUM, [CRITERIA_TYPE_KEYWORD, CRITERIA_TYPE_DYNAMIC_TEXT_AD_TARGET, CRITERIA_TYPE_AUTOTARGETING])};
our @SEARCH_QUERY_PERFORMANCE_REPORT_CRITERION_TYPE_ENUM = sort @{xisect(\@CRITERION_TYPE_ENUM, [CRITERION_TYPE_KEYWORD, CRITERION_TYPE_WEBPAGE_FILTER, CRITERION_TYPE_FEED_FILTER, CRITERION_TYPE_AUTOTARGETING])};
our @DEVICE_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %DEVICE_TYPES;
our @GENDER_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %GENDERS;
our @MOBILE_PLATFORM_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %DETAILED_DEVICE_TYPES;
our @SLOT_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %POSITION_TYPES;
our @AD_FORMAT_ENUM = sort grep {defined $_} map { $_->{+API_COMSUMER_KEY_NAME} } values %BANNER_IMAGE_TYPES;
our @MATCH_TYPE_ENUM = sort(uniq(map { $_->{+API_COMSUMER_KEY_NAME} } values %MATCH_TYPES));
our @CAMP_TYPE_ENUM = sort(uniq(map { $_->{+API_COMSUMER_KEY_NAME} } values %CAMP_TYPES));
our @ATTRIBUTION_MODEL_TYPE_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %ATTRIBUTION_MODEL_TYPES;
our @TARGETING_CATEGORY_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %TARGETING_CATEGORIES;
our @PRISMA_INCOME_GRADES_ENUM = sort map { $_->{+API_COMSUMER_KEY_NAME} } values %PRISMA_INCOME_GRADES;

our %ATTRIBUTION_MODEL_TYPE_MAP =  map { $_->{+API_COMSUMER_KEY_NAME} =>  $_->{bs} } values %ATTRIBUTION_MODEL_TYPES;

our %UAC_FIELDS_TO_HIDE = map { $_ => 1 } ( qw/adgroup_id adgroup_name bid/, CRITERION_ID_MOD_INTERNAL_FIELD);

our %OPERATOR_DESC_MAP = (
    EQUALS => {
        internal => 'eq',
        takes_array => 0,
    },
    NOT_EQUALS => {
        internal => 'ne',
        takes_array => 0,
    },
    IN => {
        internal => 'eq',
        takes_array => 1,
    },
    NOT_IN => {
        internal => 'ne',
        takes_array => 1,
    },
    GREATER_THAN => {
        internal => 'gt',
        takes_array => 0,
    },
    LESS_THAN => {
        internal => 'lt',
        takes_array => 0,
    },
    STARTS_WITH_IGNORE_CASE => {
        internal => 'starts_with',
        takes_array => 0,
    },
    DOES_NOT_START_WITH_IGNORE_CASE => {
        internal => 'not_starts_with',
        takes_array => 0,
    },
    STARTS_WITH_ANY_IGNORE_CASE => {
        internal => 'starts_with',
        takes_array => 1,
    },
    DOES_NOT_START_WITH_ALL_IGNORE_CASE => {
        internal => 'not_starts_with',
        takes_array => 1,
    },
);

our %OPERATORS_MAP_TO_INTERNAL = map {$_ => $OPERATOR_DESC_MAP{$_}->{internal}} keys %OPERATOR_DESC_MAP;

our %SINGLE_VALUE_OPERATOR = map { $OPERATORS_MAP_TO_INTERNAL{$_} => TRUE } qw/GREATER_THAN LESS_THAN/;
lock_hash_recurse(%SINGLE_VALUE_OPERATOR);

=head2 %REPORT_TYPE_CONF
    Хэш, описывающий конфигурацию для типов отчетов, поддерживаемых сервисом Reports.
    Ключи хэша - названия типов отчетов, значения - хэш с конфигурацией для указанного типа отчета.
    В конфигурации могут быть следующие поля:
        fields - хеш (для удобства фильтрации), в котором ключами являются названия полей, доступных в запросе отчета данного типа (т.е. это все поля, которые в разных частях запроса можно использовать);
        field_overwrite - хэш, с переопределенными, относительно %FIELD_DESC, полями, структура, описывающая поле аналогична %FIELD_DESC;
        mandatory_group_by - массив со списком полей, по которым выполняется неотключаемая группировка.
        availability_date_getter - процедура, возвращающая дату в формате YYYY-MM-DD, начиная с которой доступны данные отчёта этого типа
        not_available_error_msg_getter - процедура ($report_type, $param_name, $availability_date), возвращающая сообщение об ошибке о том, что отчёт такого типа за такой период нельзя и почему

=cut

our %REPORT_TYPE_CONF = (
    ADGROUP_PERFORMANCE_REPORT_TYPE() => {
        mandatory_group_by => [ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY],
        fields => {
            map { $_ => TRUE } qw/
                AdFormat AdGroupId AdGroupName AdNetworkType Age AttributionModels AvgEffectiveBid AvgCpc AvgClickPosition 
                AvgImpressionPosition AvgPageviews AvgTimeToConversion
                BounceRate Bounces ClientLogin
                CampaignId CampaignName CampaignType CampaignUrlPath CarrierType ClickType Clicks Conversions ConversionRate Cost CostPerConversion CriteriaType CriterionType Ctr
                Date Device ExternalNetworkName Gender Goals GoalsRoi ImpressionShare Impressions
                LocationOfPresenceId LocationOfPresenceName MatchType MobilePlatform Month Placement Profit Quarter Revenue Sessions Slot StrategyId
                TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    AD_PERFORMANCE_REPORT_TYPE() => {
        mandatory_group_by => [AD_ID_INTERNAL_FIELD_FOR_GROUP_BY],
        fields => {
            map { $_ => TRUE } qw/
                AdFormat AdGroupId AdGroupName AdId AdNetworkType Age AttributionModels AvgClickPosition AvgEffectiveBid AvgCpc
                AvgImpressionPosition AvgPageviews AvgTimeToConversion
                BounceRate Bounces CampaignId CampaignName CampaignType CampaignUrlPath CarrierType Clicks ClickType ConversionRate Conversions ClientLogin
                Cost CostPerConversion CriteriaType CriterionType Ctr Date Device ExternalNetworkName 
                Gender Goals GoalsRoi Impressions LocationOfPresenceId LocationOfPresenceName MatchType
                MobilePlatform Month Placement Profit Quarter Revenue Sessions Slot StrategyId TargetingLocationId
                TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    ACCOUNT_PERFORMANCE_REPORT_TYPE() => {
        fields => {
            map { $_ => TRUE } qw/
                AdFormat AdNetworkType Age AttributionModels AvgClickPosition AvgEffectiveBid AvgCpc AvgImpressionPosition AvgPageviews
                AvgTimeToConversion BounceRate Bounces ClientLogin
                CampaignType CarrierType Clicks ClickType Conversions ConversionRate Cost CostPerConversion CriteriaType CriterionType Ctr
                Date Device ExternalNetworkName Gender Goals GoalsRoi Impressions
                LocationOfPresenceId LocationOfPresenceName MatchType MobilePlatform Month Placement Profit Quarter Revenue Sessions Slot
                TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    CRITERIA_PERFORMANCE_REPORT_TYPE() => {
        mandatory_group_by => sub {
            my ($field_names) = @_;
            my $criterion_type_field = (any { $_ eq 'CriterionType' } @$field_names) ? CRITERION_TYPE_INTERNAL_FIELD : CRITERIA_TYPE_ORIG_INTERNAL_FIELD;
            my $criteria_field = (any { $_ eq 'Criteria' || $_ eq 'CriteriaId' } @$field_names)
                ? CRITERIA_MOD_INTERNAL_FIELD : CRITERIA_ORIG_MOD_INTERNAL_FIELD;
            return [ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY, $criterion_type_field, $criteria_field];
        },
        fields => {
            map { $_ => TRUE } qw/
                AdGroupId AdGroupName AdNetworkType Age AttributionModels AudienceTargetId AvgEffectiveBid AvgCpc AvgClickPosition 
                AvgImpressionPosition AvgPageviews AvgTimeToConversion BounceRate Bounces ClientLogin
                CampaignId CampaignName CampaignType CampaignUrlPath CarrierType Clicks ClickType Conversions ConversionRate Cost CostPerConversion
                Criteria CriteriaId CriteriaType Criterion CriterionId CriterionType Ctr
                Date Device DynamicTextAdTargetId ExternalNetworkName Gender Goals GoalsRoi ImpressionShare Impressions
                Keyword LocationOfPresenceId LocationOfPresenceName MatchType MobilePlatform Month Placement Profit
                Quarter Revenue RlAdjustmentId Sessions Slot SmartBannerFilterId SmartAdTargetId
                TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    SEARCH_QUERY_PERFORMANCE_REPORT_TYPE() => {
        mandatory_group_by => [ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY, QUERY_INTERNAL_FIELD],
        fields => {
            map { $_ => TRUE } qw/
                AdGroupId AdGroupName AdId AttributionModels AvgEffectiveBid AvgCpc AvgClickPosition AvgImpressionPosition 
                AvgPageviews AvgTimeToConversion BounceRate Bounces ClientLogin
                CampaignId CampaignName CampaignType CampaignUrlPath Clicks Conversions ConversionRate Cost CostPerConversion Criteria
                CriteriaId CriteriaType Criterion CriterionId CriterionType Ctr Date DynamicTextAdTargetId Goals GoalsRoi Impressions Keyword
                MatchedKeyword MatchType Month Placement Profit Quarter Query Revenue Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        field_overwrite => {
            CriteriaId => {
                result => {
                    name => CRITERIA_ID_OWN_MOD_INTERNAL_FIELD,
                    special_value_for_relevant_keyword => EMPTY_NUMERIC_VALUE,
                },
                type => TYPE_ID,
                group_by => [ CRITERIA_EXT_MOD_INTERNAL_FIELD, ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY ],
            },
            Criteria => {
                result => 'phrase',
                group_by => CRITERIA_EXT_MOD_INTERNAL_FIELD,
            },
            CriteriaType => {
                result => CRITERIA_TYPE_INTERNAL_FIELD,
                group_by => CRITERIA_TYPE_ORIG_INTERNAL_FIELD,
                order_by => CRITERIA_TYPE_ORIG_INTERNAL_FIELD,
                type => TYPE_STRING,
                filter => {
                    name => CRITERIA_TYPE_ORIG_INTERNAL_FIELD,
                    ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
                    enum => \@SEARCH_QUERY_PERFORMANCE_REPORT_CRITERIA_TYPE_ENUM,
                },
            },
            CriterionId => {
                result => CRITERION_ID_MOD_INTERNAL_FIELD,
                group_by => [ CRITERIA_ORIG_MOD_INTERNAL_FIELD, ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY ],
            },
            Criterion => {
                result => 'phrase',
                group_by => CRITERIA_ORIG_MOD_INTERNAL_FIELD,
            },
            CriterionType => {
                result => CRITERION_TYPE_INTERNAL_FIELD,
                group_by => CRITERION_TYPE_INTERNAL_FIELD,
                order_by => CRITERION_TYPE_INTERNAL_FIELD,
                type => TYPE_STRING,
                filter => {
                    name => CRITERION_TYPE_INTERNAL_FIELD,
                    ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
                    enum => \@SEARCH_QUERY_PERFORMANCE_REPORT_CRITERION_TYPE_ENUM,
                },
            },
        },
        availability_date_getter => sub {
            return now()->subtract( days => $BS_SEARCH_QUERIES_LAST_DAYS - 1 )->ymd('-');
        },
        not_available_error_msg_getter => sub {
            my ($report_type, $param_name, $availability_date) = @_;
            return iget('Для типа отчета %1$s статистика доступна за последние 180 дней. Дата в параметре %2$s должна быть не ранее %3$s.',
                $report_type, $param_name, $availability_date);
        },
    },
    STRATEGY_PERFORMANCE_REPORT_TYPE()     => {
        mandatory_group_by => [ STRATEGY_ID_INTERNAL_FIELD ],
        fields             => {
            map { $_ => TRUE } qw/
                AdFormat AdNetworkType Age AttributionModels AvgClickPosition AvgEffectiveBid AvgCpc AvgImpressionPosition
                AvgPageviews AvgTimeToConversion BounceRate Bounces ClientLogin
                StrategyId CarrierType Clicks ClickType ConversionRate Conversions
                Cost CostPerConversion CriteriaType CriterionType Ctr Date Device ExternalNetworkName Gender Goals GoalsRoi ImpressionShare Impressions
                LocationOfPresenceId LocationOfPresenceName MatchType MobilePlatform Month Placement Profit
                Quarter Revenue Sessions Slot TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    CAMPAIGN_PERFORMANCE_REPORT_TYPE() => {
        mandatory_group_by => [CAMPAIGN_ID_INTERNAL_FIELD_FOR_GROUP_BY],
        fields => {
            map { $_ => TRUE } qw/
                AdFormat AdNetworkType Age AttributionModels AvgClickPosition AvgEffectiveBid AvgCpc AvgImpressionPosition 
                AvgPageviews AvgTimeToConversion BounceRate Bounces ClientLogin
                CampaignId CampaignName CampaignType CampaignUrlPath CarrierType Clicks ClickType ConversionRate Conversions
                Cost CostPerConversion CriteriaType CriterionType Ctr Date Device ExternalNetworkName Gender Goals GoalsRoi ImpressionShare Impressions
                LocationOfPresenceId LocationOfPresenceName MatchType MobilePlatform Month Placement Profit
                Quarter Revenue Sessions Slot TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    REACH_AND_FREQUENCY_PERFORMANCE_REPORT_TYPE() => {
        mandatory_fields => [qw/CampaignId/],
        fields => {
            map { $_ => TRUE } qw/
                AdGroupId AdGroupName AdId Age AvgEffectiveBid AvgEffectiveCpmBid AvgCpc AvgCpm AvgImpressionFrequency 
                AvgPageviews AvgTimeToConversion BounceRate Bounces ClientLogin
                CampaignId CampaignName CampaignType CampaignUrlPath Clicks Conversions ConversionRate Cost CostPerConversion Ctr
                Date Device Gender GoalsRoi Impressions Month Quarter ImpressionReach Profit Revenue Sessions
                TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume
                AuctionHits AuctionWins AuctionWinRate ImpToWinRate ImpReachRate
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
    CUSTOM_REPORT_TYPE() => {
        fields => {
            map { $_ => TRUE } qw/
                AdGroupId AdGroupName AdId AdNetworkType Age AttributionModels AudienceTargetId AvgEffectiveBid AvgCpc AvgClickPosition 
                AvgImpressionPosition AvgPageviews AvgTimeToConversion
                BounceRate Bounces ClientLogin
                StrategyId CampaignId CampaignName CampaignType CampaignUrlPath CarrierType ClickType Clicks Conversions ConversionRate Cost CostPerConversion Criteria CriteriaId CriteriaType
                Criterion CriterionId CriterionType Ctr
                Date Device DynamicTextAdTargetId ExternalNetworkName Gender Goals GoalsRoi AdFormat Impressions
                Keyword LocationOfPresenceId LocationOfPresenceName MatchType MobilePlatform Month Placement Profit Quarter Revenue RlAdjustmentId Sessions Slot
                SmartBannerFilterId SmartAdTargetId
                TargetingLocationId TargetingLocationName Week Year
                WeightedCtr WeightedImpressions AvgTrafficVolume
                AuctionHits AuctionWins AuctionWinRate ImpToWinRate ImpReachRate TargetingCategory IncomeGrade
            /
        },
        availability_date_getter => \&_bs_stat_availability_date_getter,
        not_available_error_msg_getter => \&_not_available_error_msg_getter,
    },
);
lock_hash_recurse(%REPORT_TYPE_CONF);

our @REPORT_TYPES = sort keys %REPORT_TYPE_CONF;

our %MANDATORY_GROUP_BY_FIELDS = map { $_ => exists $REPORT_TYPE_CONF{$_}{mandatory_group_by} ? $REPORT_TYPE_CONF{$_}{mandatory_group_by} : [] } @REPORT_TYPES;
lock_hash_recurse(%MANDATORY_GROUP_BY_FIELDS);

our %MANDATORY_FIELDS = map { $_ => exists $REPORT_TYPE_CONF{$_}{mandatory_fields} ? $REPORT_TYPE_CONF{$_}{mandatory_fields} : undef } @REPORT_TYPES;
lock_hash_recurse(%MANDATORY_FIELDS);

our %FIELD_DESC = (
    Date => {
        result => 'stat_date',
        group_by => 'date',
        order_by => 'date',
    },
    Week => {
        result => 'stat_date',
        group_by => 'date',
        order_by => 'date',
    },
    Month => {
        result => 'stat_date',
        group_by => 'date',
        order_by => 'date',
    },
    Quarter => {
        result => 'stat_date',
        group_by => 'date',
        order_by => 'date',
    },
    Year => {
        result => 'stat_date',
        group_by => 'date',
        order_by => 'date',
    },
    CampaignType => {
        type => TYPE_STRING,
        result => 'campaign_type',
        group_by => 'campaign_type',
        order_by => 'campaign_type',

        filter => {
            name => 'campaign_type',
            ops => [qw/EQUALS IN/],
            enum => \@CAMP_TYPE_ENUM,
        },
    },
    CampaignId => {
        result => 'cid',
        group_by => CAMPAIGN_ID_INTERNAL_FIELD_FOR_GROUP_BY,
        order_by => 'campaign',
        type => TYPE_ID,

        filter => {
            name => 'cid',
            ops => [qw/EQUALS IN/],
        },
    },
    CampaignName => {
        type => TYPE_STRING,
        result => 'camp_name',
        group_by => CAMPAIGN_ID_INTERNAL_FIELD_FOR_GROUP_BY,
        order_by => 'camp_name',
    },
    StrategyId => {
        type     => TYPE_ID,
        result   => STRATEGY_ID_INTERNAL_FIELD,
        group_by => STRATEGY_ID_INTERNAL_FIELD,
        order_by => STRATEGY_ID_INTERNAL_FIELD,
        filter   => {
            name => STRATEGY_ID_INTERNAL_FIELD,
            ops  => [qw/EQUALS IN NOT_EQUALS NOT_IN/]
        }
    },
    CampaignUrlPath => {
        type => TYPE_STRING,
        result => 'camp_url_path',
        group_by => CAMPAIGN_ID_INTERNAL_FIELD_FOR_GROUP_BY,
        order_by => 'camp_url_path',
    },
    AdGroupId => {
        result => 'adgroup_id',
        group_by => ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY,
        order_by => 'adgroup',
        type => TYPE_ID,

        filter => {
            name => 'adgroup',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    AdGroupName => {
        type => TYPE_STRING,
        result => 'adgroup_name',
        group_by => ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY,
        order_by => 'adgroup_name',
    },
    AdId => {
        result => 'bid',
        group_by => 'banner',
        order_by => 'banner',
        type => TYPE_ID,
        filter => {
            name => 'banner',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    CriteriaId => {
        result => {
            name => CRITERIA_ID_OWN_MOD_INTERNAL_FIELD,
            special_value_for_relevant_keyword => EMPTY_NUMERIC_VALUE,
        },
        type => TYPE_ID,
        group_by => [ CRITERIA_MOD_INTERNAL_FIELD, ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY ],
        incompatible_with => [qw/Criterion CriterionId CriterionType/],
    },
    Criteria => {
        type => TYPE_STRING,
        result => {
            name => 'phrase',
            special_value_for_relevant_keyword => EMPTY_STRING_VALUE,
        },
        group_by => CRITERIA_MOD_INTERNAL_FIELD,
        incompatible_with => [qw/Criterion CriterionId CriterionType/],
    },
    CriteriaType => {
        result => CRITERIA_TYPE_INTERNAL_FIELD,
        group_by => CRITERIA_TYPE_ORIG_INTERNAL_FIELD,
        order_by => CRITERIA_TYPE_ORIG_INTERNAL_FIELD,
        type => TYPE_STRING,
        filter => {
            name => CRITERIA_TYPE_ORIG_INTERNAL_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@CRITERIA_TYPE_ENUM,
        },
        incompatible_with => [qw/CriterionType/],
    },
    CriterionType => {
        result => CRITERION_TYPE_INTERNAL_FIELD,
        group_by => CRITERION_TYPE_INTERNAL_FIELD,
        order_by => CRITERION_TYPE_INTERNAL_FIELD,
        type => TYPE_STRING,
        filter => {
            name => CRITERION_TYPE_INTERNAL_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@CRITERION_TYPE_ENUM,
        },
    },
    Keyword => {
        type => TYPE_STRING,
        filter => {
            name => 'phrase_orig',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN STARTS_WITH_IGNORE_CASE STARTS_WITH_ANY_IGNORE_CASE DOES_NOT_START_WITH_IGNORE_CASE DOES_NOT_START_WITH_ALL_IGNORE_CASE/],
        },
    },
    AudienceTargetId => {
        type => TYPE_ID,
        filter => {
            name => 'bs_criterion_id',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    DynamicTextAdTargetId => {
        type => TYPE_ID,
        filter => {
            name => 'dynamic',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    # deprecated
    # дублирует SmartAdTargetId
    SmartBannerFilterId => {
        type => TYPE_ID,
        filter => {
            name => 'performance',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    SmartAdTargetId => {
        type => TYPE_ID,
        filter => {
            name => 'performance',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    AdNetworkType => {
        result => AD_NETWORK_TYPE_INTERNAL_FIELD,
        group_by => AD_NETWORK_TYPE_INTERNAL_FIELD,
        order_by => AD_NETWORK_TYPE_INTERNAL_FIELD,
        type => TYPE_STRING,
        filter => {
            name => AD_NETWORK_TYPE_INTERNAL_FIELD,
            ops => [qw/EQUALS IN/],
            enum => \@AD_NETWORK_TYPE_ENUM,
        },
    },
    Placement => {
        result => 'page_name',
        # переезжает на page_name в Stat::CustomizedArray::StreamedExt::group_by_stat_stream_ext
        # после включения фичи на 100% можно здесь поменять
        group_by => 'page_group',
        type => TYPE_STRING,
        filter => {
            # переезжает на page_name в API::Service::Reports::_convert_request_data_to_report_params
            name => PLACEMENT_INTERNAL_FIELD_FOR_FILTER,
            ops => [qw/EQUALS NOT_EQUALS IN NOT_IN/],
        },
        # переезжает на page_name в Stat::CustomizedArray::StreamedExt::_order_by_stat_stream_ext
        # после включения фичи на 100% можно здесь поменять
        order_by => 'page_group',
    },
    ExternalNetworkName => {
        result => 'ssp',
        group_by => 'ssp',
        order_by => 'ssp',
        type => TYPE_STRING,
        filter => {
            name => 'ssp',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    TargetingLocationId => {
        result => 'region',
        group_by => 'region',
        order_by => 'region_id',
        type => TYPE_GEO_REGION,
        filter => {
            name => 'region_id',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    TargetingLocationName => {
        type => TYPE_STRING,
        result => TARGET_LOCATION_NAME_INTERNAL_FIELD,
        group_by => 'region',
        order_by => 'region',
    },
    LocationOfPresenceId => {
        result => 'physical_region',
        group_by => 'physical_region',
        order_by => 'physical_region_id',
        type => TYPE_GEO_REGION,
        filter => {
            name => 'physical_region',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    LocationOfPresenceName => {
        type => TYPE_STRING,
        result => LOCATION_OF_PRESENCE_NAME_INTERNAL_FIELD,
        group_by => 'physical_region',
        order_by => 'physical_region',
    },
    Slot => {  # Position, ImpressionType
        result => 'position',
        group_by => 'position',
        order_by => 'position',
        type => TYPE_STRING,
        filter => {
            name => 'position',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@SLOT_ENUM,
        },
    },
    AdFormat => {
        result => 'banner_image_type',
        group_by => 'banner_image_type',
        order_by => 'banner_image_type',
        type => TYPE_STRING,
        filter => {
            name => 'banner_image_type',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@AD_FORMAT_ENUM,
        },
    },
    Device => {
        result => 'device_type',
        group_by => 'device_type',
        order_by => 'device_type',
        type => TYPE_STRING,
        filter => {
            name => 'device_type',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@DEVICE_ENUM,
        },
    },
    MobilePlatform => {
        result => 'detailed_device_type',
        group_by => 'detailed_device_type',
        order_by => 'detailed_device_type',
        type => TYPE_STRING,
        filter => {
            name => 'detailed_device_type',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@MOBILE_PLATFORM_ENUM,
        },
    },
    CarrierType => {
        result => 'connection_type',
        group_by => 'connection_type',
        order_by => 'connection_type',
        type => TYPE_STRING,
        filter => {
            name => 'connection_type',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@CARRIER_TYPE_ENUM,
        },
    },
    Gender => {
        result => 'gender',
        group_by => 'gender',
        order_by => 'gender',
        type => TYPE_STRING,
        filter => {
            name => 'gender',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@GENDER_ENUM,
        },
    },
    Age => {
        result => 'age',
        group_by => 'age',
        order_by => 'age',
        type => TYPE_STRING,
        filter => {
            name => 'age',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@AGE_ENUM,
        },
    },
    ClickType => {
        result => 'click_place',
        group_by => 'click_place',
        order_by => 'click_place',
        type => TYPE_STRING,
        filter => {
            name => 'click_place',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@CLICK_TYPE_ENUM,
        },
        incompatible_with => [qw/Impressions Ctr AvgImpressionPosition WeightedCtr WeightedImpressions AvgTrafficVolume/],
    },
    TargetingCategory => {
        result => TARGETING_CATEGORY_FIELD,
        group_by => TARGETING_CATEGORY_FIELD,
        order_by => TARGETING_CATEGORY_FIELD,
        type => TYPE_STRING,
        filter => {
            name => TARGETING_CATEGORY_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@TARGETING_CATEGORY_ENUM,
        },
    },
    IncomeGrade => {
        result => INCOME_GRADE_FIELD,
        group_by => INCOME_GRADE_FIELD,
        order_by => INCOME_GRADE_FIELD,
        type => TYPE_STRING,
        filter => {
            name => INCOME_GRADE_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@PRISMA_INCOME_GRADES_ENUM,
        },
    },
    # Агрегируемые поля
    Clicks => {
        result => 'clicks',
        order_by => 'clicks',
        type => TYPE_INTEGER,
        filter => {
            name => 'clicks',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    ImpressionShare => {
        result => 'winrate',
        group_by => CAMPAIGN_ID_INTERNAL_FIELD_FOR_GROUP_BY,
        order_by => 'winrate',
        type => TYPE_FLOAT,
        filter => {
            name => 'winrate',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        incompatible_with => [qw/AdFormat AdId Age CarrierType Gender MobilePlatform RlAdjustmentId/],
        is_a_countable_field => 1,
    },
    Impressions => {
        result => 'shows',
        order_by => 'shows',
        type => TYPE_INTEGER,
        filter => {
            name => 'shows',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    Ctr => {
        result => 'ctr',
        order_by => 'ctr',
        type => TYPE_FLOAT,
        filter => {
            name => 'ctr',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    Cost => {
        result => 'sum',
        order_by => 'sum',
        type => TYPE_MONEY,
        filter => {
            name => 'sum',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    AvgCpc => {
        result => 'av_sum',
        order_by => 'av_sum',
        type => TYPE_MONEY,
        filter => {
            name => 'av_sum',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    AvgImpressionPosition => {
        result => 'fp_shows_avg_pos',
        order_by => 'fp_shows_avg_pos',
        type => TYPE_FLOAT,
        filter => {
            name => 'fp_shows_avg_pos',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    AvgClickPosition => {
        result => 'fp_clicks_avg_pos',
        order_by => 'fp_clicks_avg_pos',
        type => TYPE_FLOAT,
        filter => {
            name => 'fp_clicks_avg_pos',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    Bounces => {
        result => 'bounces',
        order_by => 'bounces',
        type => TYPE_INTEGER,
#   фильтрация по Bounces пока не реализована в БК
#         filter => {
#             name => 'bounces',
#             ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
#             non_negative => 1,
#             positive_lt => 1,
#         },
        is_a_countable_field => 1,
    },
    BounceRate => {
        result => 'bounce_ratio',
        order_by => 'bounce_ratio',
        type => TYPE_FLOAT,
        filter => {
            name => 'bounce_ratio',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    AvgPageviews => {
        order_by => 'adepth',
        result => 'adepth',
        type => TYPE_FLOAT,
        filter => {
            name => 'adepth',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    ClientLogin => {
        result => 'client_login',
        group_by => 'client_login',
        order_by => 'client_login',
        type => TYPE_STRING,
        filter => {
            name => 'client_login',
            ops => [qw/EQUALS IN/],
        },
    },
    Conversions => {
        result => 'agoalnum',
        order_by => 'agoalnum',
        type => TYPE_INTEGER,
        filter => {
            name => 'agoalnum',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    ConversionRate => {
        result => 'aconv',
        order_by => 'aconv',
        type => TYPE_FLOAT,
        filter => {
            name => 'aconv',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    CostPerConversion => {
        result => 'agoalcost',
        order_by => 'agoalcost',
        type => TYPE_MONEY,
        filter => {
            name => 'agoalcost',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    GoalsRoi => {
        result => 'agoalroi',
        order_by => 'agoalroi',
        type => TYPE_FLOAT,
        filter => {
            name => 'agoalroi',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    Revenue => {
        result => 'agoalincome',
        order_by => 'agoalincome',
        type => TYPE_MONEY,
        filter => {
            name => 'agoalincome',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
        is_a_countable_field => 1,
    },
    Sessions => { # количество сессий (визитов)
        type => TYPE_INTEGER,
        result => 'asesnum',
        order_by => 'asesnum',
        is_a_countable_field => 1,
    },

    RlAdjustmentId => {
        result => 'coef_ret_cond_id',
        group_by => 'retargeting_coef',
        order_by => 'coef_ret_cond_id',
        type => TYPE_ID,
        filter => {
            name => 'retargeting_coef',
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
        },
    },
    Query => { # поисковый запрос
        result => QUERY_INTERNAL_FIELD,
        group_by => QUERY_INTERNAL_FIELD,
        order_by => QUERY_INTERNAL_FIELD,
        type => TYPE_STRING,
        filter => {
            name => QUERY_INTERNAL_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN STARTS_WITH_IGNORE_CASE STARTS_WITH_ANY_IGNORE_CASE DOES_NOT_START_WITH_IGNORE_CASE DOES_NOT_START_WITH_ALL_IGNORE_CASE/],
        },
    },
    MatchType => {
        result => MATCH_TYPE_INTERNAL_FIELD,
        group_by => MATCH_TYPE_INTERNAL_FIELD,
        order_by => MATCH_TYPE_INTERNAL_FIELD,
        type => TYPE_STRING,
        filter => {
            name => MATCH_TYPE_INTERNAL_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN/],
            enum => \@MATCH_TYPE_ENUM,
        },
    },
    MatchedKeyword => {
        result => MATCHED_KEYWORD_INTERNAL_FIELD,
        group_by => MATCHED_KEYWORD_INTERNAL_FIELD,
        order_by => MATCHED_KEYWORD_INTERNAL_FIELD,
        type => TYPE_STRING,
        filter => {
            name => MATCHED_KEYWORD_INTERNAL_FIELD,
            ops => [qw/EQUALS IN NOT_EQUALS NOT_IN STARTS_WITH_IGNORE_CASE STARTS_WITH_ANY_IGNORE_CASE DOES_NOT_START_WITH_IGNORE_CASE DOES_NOT_START_WITH_ALL_IGNORE_CASE/],
        },
    },
    Criterion => {
        type => TYPE_STRING,
        result => 'phrase',
        group_by => CRITERIA_ORIG_MOD_INTERNAL_FIELD,
        incompatible_with => [qw/Criteria CriteriaId/],
    },
    CriterionId => {
        type => TYPE_ID,
        result => CRITERION_ID_MOD_INTERNAL_FIELD,
        group_by => [ CRITERIA_ORIG_MOD_INTERNAL_FIELD, ADGROUP_ID_INTERNAL_FIELD_FOR_GROUP_BY ],
        incompatible_with => [qw/Criteria CriteriaId/],
    },
   # not_in_field_names нужно для теста Reports/xsd.t когда сравниваем ключи этого хеша с FieldNames в xsd-файле эти поля учитывать не нужно
    AttributionModels => {
        not_in_field_names => 1, 
        type => TYPE_STRING,
        filter => {
            name => 'attribution_models',
            ops => [qw/IN/],
            enum => \@ATTRIBUTION_MODEL_TYPE_ENUM,
        },
    },
    Goals => {
        not_in_field_names => 1,
        type => TYPE_STRING,
        filter => {
            name => 'goal_ids',
            ops => [qw/IN/],
        },
    },
    WeightedCtr => {
        result => 'ectr',
        is_countable => 1,
        order_by => 'ectr',
        type => TYPE_FLOAT,
        filter => {
            name => 'ectr',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    WeightedImpressions => {
        result => 'eshows',
        is_countable => 1,
        order_by => 'eshows',
        type => TYPE_FLOAT,
        filter => {
            name => 'eshows',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    AvgTrafficVolume => {
        result => 'avg_x',
        is_countable => 1,
        order_by => 'avg_x',
        type => TYPE_FLOAT,
        filter => {
            name => 'avg_x',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    AvgCpm => {
        result => 'avg_cpm',
        is_countable => 1,
        order_by => 'avg_cpm',
        type => TYPE_MONEY,
        filter => {
            name => 'avg_cpm',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
    },
    ImpressionReach => {
        result => 'uniq_viewers',
        is_countable => 1,
        order_by => 'uniq_viewers',
        type => TYPE_INTEGER,
        filter => {
            name => 'uniq_viewers',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    AvgImpressionFrequency => {
        result => 'avg_view_freq',
        is_countable => 1,
        order_by => 'avg_view_freq',
        type => TYPE_FLOAT,
        filter => {
            name => 'avg_view_freq',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    AvgTimeToConversion => {
        result => 'avg_time_to_conv',
        order_by => 'avg_time_to_conv',
        type => TYPE_FLOAT,
        filter => {
            name => 'avg_time_to_conv',
            ops => [qw/GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
        not_in_field_names => 1,
        is_a_countable_field => 1,
    },
    AvgEffectiveBid => {
        result => 'avg_bid',
        is_countable => 1,
        order_by => 'avg_bid',
        type => TYPE_MONEY,
        filter => {
            name => 'avg_bid',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
    },
    AvgEffectiveCpmBid => {
        result => 'avg_cpm_bid',
        is_countable => 1,
        order_by => 'avg_cpm_bid',
        type => TYPE_MONEY,
        filter => {
            name => 'avg_cpm_bid',
            ops => [qw/GREATER_THAN LESS_THAN/],
            positive_lt => 1,
        },
    },
    Profit => {
        result => 'agoals_profit',
        order_by => 'agoals_profit',
        type => TYPE_SIGNED_MONEY,
        filter => {
            name => 'agoals_profit',
            ops => [qw/GREATER_THAN LESS_THAN/],
        },
        is_a_countable_field => 1,
    },
    AuctionHits => {
        not_in_field_names => 1,
        result => 'auction_hits',
        is_countable => 1,
        order_by => 'auction_hits',
        type => TYPE_INTEGER,
        filter => {
            name => 'auction_hits',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    AuctionWins => {
        not_in_field_names => 1,
        result => 'auction_wins',
        is_countable => 1,
        order_by => 'auction_wins',
        type => TYPE_INTEGER,
        filter => {
            name => 'auction_wins',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    AuctionWinRate => {
        not_in_field_names => 1,
        result => 'auction_win_rate',
        is_countable => 1,
        order_by => 'auction_win_rate',
        type => TYPE_INTEGER,
        filter => {
            name => 'auction_win_rate',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    ImpToWinRate => {
        not_in_field_names => 1,
        result => 'imp_to_win_rate',
        is_countable => 1,
        order_by => 'imp_to_win_rate',
        type => TYPE_INTEGER,
        filter => {
            name => 'imp_to_win_rate',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },
    ImpReachRate => {
        not_in_field_names => 1,
        result => 'imp_reach_rate',
        is_countable => 1,
        order_by => 'imp_reach_rate',
        type => TYPE_INTEGER,
        filter => {
            name => 'imp_reach_rate',
            ops => [qw/EQUALS IN GREATER_THAN LESS_THAN/],
            non_negative => 1,
            positive_lt => 1,
        },
    },

);

our @FIELDS_DEPENDENT_ON_GOAL = qw/Conversions ConversionRate CostPerConversion GoalsRoi Revenue AvgTimeToConversion/;

our %FIELDS_DEPENDENT_ON_GOAL_MAP = map { $_ => $FIELD_DESC{$_}{result}} @FIELDS_DEPENDENT_ON_GOAL;

our %ORDER_DIRECTION_MAP = (
    ASCENDING => 'asc',
    DESCENDING => 'desc',
);

our %FORMAT_MAP = (
    TSV => 'tsv',
);

our (%RESULT_FIELD_MAP, %ORDER_BY_FIELD_MAP, %RESULT_MAP_BY_FIELD, %MONEY_FIELD_MAP, %FILTER_PARAMS_MAP, %GROUP_BY_FIELD_MAP, %INCOMPATIBLE_FIELDS_MAP, %SPECIAL_VALUE_FOR_RELATED_KEYWORD, %COUNTABLE_FIELDS);
our (%INTERNAL_FIELD_TYPE_MAP, %FIELD_TYPE_MAP);
foreach my $field (keys %FIELD_DESC) {
    my $field_data = $FIELD_DESC{$field};
    my $result = exists $field_data->{result} ? $field_data->{result} : undef;
    my $filter = exists $field_data->{filter} ? $field_data->{filter} : undef;
    my $group_by = exists $field_data->{group_by} ? $field_data->{group_by} : undef;
    my $order_by = exists $field_data->{order_by} ? $field_data->{order_by} : undef;
    my $type = exists $field_data->{type} ? $field_data->{type} : undef;
    my $incompatible_with = exists $field_data->{incompatible_with} ? $field_data->{incompatible_with} : undef;
    foreach my $report_type (@REPORT_TYPES) {
        next unless exists $REPORT_TYPE_CONF{$report_type}{fields}{$field};
        my $field_overwrite = exists $REPORT_TYPE_CONF{$report_type}{field_overwrite} && exists $REPORT_TYPE_CONF{$report_type}{field_overwrite}{$field} ? $REPORT_TYPE_CONF{$report_type}{field_overwrite}{$field} : undef;
        my $internal_field;
        if (my $custom_result = $field_overwrite && exists $field_overwrite->{result} && $field_overwrite->{result} || $result) {
            if (ref $custom_result) {
                $internal_field = $custom_result->{name};
                $RESULT_FIELD_MAP{$report_type}{$field} = $internal_field;
                if (exists $custom_result->{map}) {
                    $RESULT_MAP_BY_FIELD{$report_type}{$internal_field} = $custom_result->{map};
                }
                if (exists $custom_result->{special_value_for_relevant_keyword}) {
                    $SPECIAL_VALUE_FOR_RELATED_KEYWORD{$report_type}{$internal_field} = $custom_result->{special_value_for_relevant_keyword};
                }
            } else {
                $RESULT_FIELD_MAP{$report_type}{$field} = $internal_field = $custom_result;
            }
        }

        my $custom_type;
        if ($field_overwrite && exists $field_overwrite->{type} && $field_overwrite->{type} || $type) {
            $custom_type = exists $field_overwrite->{type} && $field_overwrite->{type} || $type;
            $INTERNAL_FIELD_TYPE_MAP{$report_type}{$internal_field} = $custom_type if $internal_field;
            $FIELD_TYPE_MAP{$report_type}{$field} = $custom_type;
        }
        if ($custom_type && ($custom_type eq TYPE_MONEY || $custom_type eq TYPE_SIGNED_MONEY) && $internal_field) {
            $MONEY_FIELD_MAP{$report_type}{$field} = $internal_field;
        }

        if (my $custom_filter = $field_overwrite && exists $field_overwrite->{filter} && $field_overwrite->{filter} || $filter) {
            $FILTER_PARAMS_MAP{$report_type}{$field} = $custom_filter;
        }

        if (my $custom_group_by = $field_overwrite && exists $field_overwrite->{group_by} && $field_overwrite->{group_by} || $group_by) {
            $GROUP_BY_FIELD_MAP{$report_type}{$field} = $custom_group_by;
        }

        if (my $custom_order_by = exists $field_overwrite->{order_by} && $field_overwrite->{order_by} || $order_by) {
            $ORDER_BY_FIELD_MAP{$report_type}{$field} = $custom_order_by;
        }

        if (my $custom_incompatible_with = exists $field_overwrite->{incompatible_with} && $field_overwrite->{incompatible_with} || $incompatible_with) {
            $INCOMPATIBLE_FIELDS_MAP{$report_type}{$field} = $custom_incompatible_with;
        }
    }

    $COUNTABLE_FIELDS{$field} = 1 if $field_data->{is_countable};
}

lock_hash_recurse(%INTERNAL_FIELD_TYPE_MAP);
lock_hash_recurse(%FIELD_TYPE_MAP);
lock_hash_recurse(%FILTER_PARAMS_MAP);
lock_hash_recurse(%INCOMPATIBLE_FIELDS_MAP);
lock_hash_recurse(%GROUP_BY_FIELD_MAP);
lock_hash_recurse(%MONEY_FIELD_MAP);
lock_hash_recurse(%ORDER_BY_FIELD_MAP);
lock_hash_recurse(%RESULT_FIELD_MAP);
lock_hash_recurse(%RESULT_MAP_BY_FIELD);
lock_hash_recurse(%SPECIAL_VALUE_FOR_RELATED_KEYWORD);
lock_hash_recurse(%COUNTABLE_FIELDS);

our %MONEY_INTERNAL_FIELD_MAP = map { $_ => { reverse %{$MONEY_FIELD_MAP{$_}} } } keys %MONEY_FIELD_MAP;
lock_hash_recurse(%MONEY_INTERNAL_FIELD_MAP);

our %DATE_RANGE_TYPE_MAP = (
    TODAY               => sub { my $now = shift; return ( $now->ymd(), $now->ymd() ); },
    YESTERDAY           => sub { my $now = shift; $now->subtract(days => 1); return ( $now->ymd(), $now->ymd() ); },
    LAST_3_DAYS         => sub { my $now = shift; return ( $now->clone->subtract(days => 3)->ymd(),  $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_5_DAYS         => sub { my $now = shift; return ( $now->clone->subtract(days => 5)->ymd(),  $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_7_DAYS         => sub { my $now = shift; return ( $now->clone->subtract(days => 7)->ymd(),  $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_14_DAYS        => sub { my $now = shift; return ( $now->clone->subtract(days => 14)->ymd(), $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_30_DAYS        => sub { my $now = shift; return ( $now->clone->subtract(days => 30)->ymd(), $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_90_DAYS        => sub { my $now = shift; return ( $now->clone->subtract(days => 90)->ymd(), $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_365_DAYS       => sub { my $now = shift; return ( $now->clone->subtract(days => 365)->ymd(), $now->subtract(days => 1)->ymd() ); }, # today excluded
    LAST_WEEK           => sub { my $now = shift; return ( $now->truncate(to => 'week')->subtract(weeks => 1)->ymd, $now->add(days => 6)->ymd() ); },
    LAST_BUSINESS_WEEK  => sub { my $now = shift; return ( $now->set_locale(US_LOCALE)->truncate(to => 'local_week')->subtract(days => 6)->ymd, $now->add(days => 4)->ymd() ); },
    THIS_MONTH          => sub { my $now = shift; return ( $now->clone->truncate(to => 'month')->ymd(), $now->ymd() ); },
    LAST_MONTH          => sub { my $now = shift; return ( $now->truncate(to => 'month')->subtract(months => 1)->ymd, DateTime->last_day_of_month(year => $now->year, month => $now->month)->ymd() ); },
    THIS_WEEK_SUN_TODAY => sub { my $now = shift; return ( $now->clone->set_locale(US_LOCALE)->truncate(to => 'local_week')->ymd, $now->ymd() ); },
    THIS_WEEK_MON_TODAY => sub { my $now = shift; return ( $now->clone->truncate(to => 'week')->ymd, $now->ymd() ); },
    LAST_WEEK_SUN_SAT   => sub { my $now = shift; return ( $now->set_locale(US_LOCALE)->subtract(days => 7)->truncate(to => 'local_week')->ymd, $now->add(days => 6)->ymd() ); },
    ALL_TIME            => sub { return (); },
    CUSTOM_DATE_RANGE_TYPE() => undef,
    AUTO => sub {
        my ( $now, $OrderIds ) = @_;

        my $default_start_date = $now->clone->subtract(days => 3)->ymd();
        my $default_end_date = $now->clone->subtract(days => 1)->ymd();

        my $min_border_date;
        if ($OrderIds && @$OrderIds) {
            $min_border_date = get_one_field_sql( PPCDICT(), [
                'SELECT MIN(border_date) FROM stat_rollbacks',
                WHERE => {
                    OrderID => $OrderIds,
                    rollback_time__ge__dont_quote => 'CURDATE() - INTERVAL 1 DAY',
                    rollback_time__lt__dont_quote => 'CURDATE()',
                },
            ] );
        }

        if ( ! defined $min_border_date || $min_border_date gt $default_start_date ) {
            return ( $default_start_date, $default_end_date );
        }

        return ( $min_border_date, $default_end_date );
    },
);

sub _bs_stat_availability_date_getter {
    my $date = Stat::Const::BS_STAT_START_DATE();
    $date =~ s/^(\d{4})(\d{2})(\d{2})$/$1-$2-$3/;
    return $date;
}

sub _not_available_error_msg_getter {
    my ($report_type, $param_name, $availability_date) = @_;
    return iget('Для типа отчета %1$s статистика доступна за последние три года от текущего месяца. Дата в параметре %2$s должна быть не ранее %3$s.',
        $report_type, $param_name, $availability_date);
}

=head3 get_all_field_names()

    Возвращает список имен всех доступных полей

=cut

sub get_all_field_names {
    return [ sort grep { !exists $FIELD_DESC{$_}{not_in_field_names} } keys %FIELD_DESC ];
}


=head3 get_all_operators()

    Вернуть ссылку на массив имён операторов, которые можно использовать в фильтрах

=cut

sub get_all_operators {
    return [ sort keys %OPERATOR_DESC_MAP ];
}


=head3 get_all_order_directions()

    Вернуть ссылку на массив направлений сортировки

=cut

sub get_all_order_directions {
    return [ sort keys %ORDER_DIRECTION_MAP ];
}


=head3 get_all_report_types()

    Вернуть ссылку на массив имён типов отчётов

=cut

sub get_all_report_types {
    return \@REPORT_TYPES;
}


=head3 get_all_date_ranges()

    Вернуть ссылку на массив всех возможных имён временных интервалов

=cut

sub get_all_date_ranges {
    return [ sort keys %DATE_RANGE_TYPE_MAP ];
}

=head3 get_countable_fields()

    Вернуть ссылку на массив countable_fields для Stat::StreamExtended по переданному списку ожидаемых полей

=cut

sub get_countable_fields {
    my @displayed_field_names = @_;
    my @internal_names;

    for my $field (@displayed_field_names) {
        if (exists $FIELD_DESC{$field}->{result} && ($FIELD_DESC{$field}->{is_a_countable_field} || $FIELD_DESC{$field}->{is_countable})) {
            push @internal_names, $FIELD_DESC{$field}->{result};
        }
    }

    return Stat::Fields::get_countable_fields_for_override_by_fields_array(\@internal_names) // [];
}

1;
