package API::Reports::OfflineReportTask;
use Direct::Modern;
use JSON;
use Mouse;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::I18n;
use Yandex::Trace;

use API::Reports::ReportNameCollisionException;
use API::Reports::QueueLimitExceededException;
use API::Settings;
use Direct::Errors::Messages;
use Primitives;
use Settings;
use Yandex::DBQueue;

has 'ClientID' => ( is => 'ro', isa => 'Int' );
has 'ReportName' => ( is => 'ro', isa => 'Str' );
has 'dbqueue_job_id' => ( is => 'ro', isa => 'Int' );
has 'just_created' => ( is => 'ro', isa => 'Bool' );
has 'dbqueue_job' => ( is => 'ro', isa => 'Yandex::DBQueue::Job' );

=head2 DBQUEUE_JOB_TYPE

Тип задания в DBQueue

=cut

use constant DBQUEUE_JOB_TYPE => 'api5_report';

=head2 STORAGE_FILE_TYPE

Тип файла в MDS

=cut

use constant STORAGE_FILE_TYPE => 'api5_offline_report';

=head2 $class->get_or_create($params)

На вход метод получает вспомогательный объект с параметрами задания $params

Либо возвращает задание с указанным именем из базы, либо создаёт новое и возвращает его. Понять,
что произошло, можно по флагу just_created в результате.

Поле dbqueue_job в задании в любом случае заполнено соответствующими данными Yandex::DBQueue::Job.

Ожидается, что в job_args будет структура, которую можно "просто" сериализовать и положить в базу,
а потом оттуда достать. Функции to_plain_struct и from_plain_struct в API::Reports::InternalRequestRepresentation
помогают конвертировать (туда и обратно) объекты того типа в такие "простые" структуры.

установленный флаг dont_check_limit позволяет не проверять ограничение на количество оффлайн-отчетов в очереди

=cut

sub get_or_create {
    my ( $class, $params ) = @_;

    my $queue = Yandex::DBQueue->new( PPC( ClientID => $params->client_id ), DBQUEUE_JOB_TYPE );

    my $self;
    my $exception;
    do_in_transaction sub {
        my $mapping_row = get_one_line_sql(
            PPC( ClientID => $params->client_id ),
            'SELECT ClientID, ReportName, dbqueue_job_id, request_data_md5 FROM api_reports_offline WHERE ClientID = ? AND ReportName = ? FOR UPDATE',
            $params->client_id, $params->report_name );

        ## если строка есть в таблице с мэпингом и в нужной таблице в DBQueue,
        ## возвращаем запись
        if ($mapping_row) {
            if (!$mapping_row->{request_data_md5}) {
                die 'no md5 in row: '.to_json($mapping_row);
            } elsif ($mapping_row->{request_data_md5} ne $params->request_data_md5) {
                $exception = API::Reports::ReportNameCollisionException->new(
                    defect => error_BadParams(
                        iget('Отчет с таким названием, но с отличающимися параметрами уже сформирован или находится в очереди. Измените значение в параметре %s', 'ReportName')
                    ),
                );
                return;
            }

            my $job = $queue->find_job_by_id( $mapping_row->{dbqueue_job_id} );

            if ($job) {
                $self = $class->new(
                    ClientID => $mapping_row->{ClientID},
                    ReportName => $mapping_row->{ReportName},
                    dbqueue_job_id => $mapping_row->{dbqueue_job_id},
                    just_created => 0,
                    dbqueue_job => $job,
                );
                return;
            }
        }

        ## иначе создаём задачу и запись в таблице с мэппингом

        my $count_reports = $queue->count_jobs( ClientID => $params->client_id );
        my $limit = query_special_user_option( $params->client_id, 'api_reports_limit' ) // $API::Settings::MAX_NUM_REPORTS;

        # но сначала проверим, что можно, т. е. что клиент не упёрся в лимит
        if (!$params->dont_check_limit && $count_reports >= $limit) {
            # не бросаем исключение, а сохраняем его, потому что do_in_transaction все исключения
            # ловит и приводит к строкам
            $exception = API::Reports::QueueLimitExceededException->new(
                defect => error_OfflineTaskQueueLimitExceeded( iget(
                    '%s отчетов ожидают формирования в режиме офлайн. Дождитесь, пока хотя бы один из них будет сформирован',
                    $count_reports ) ),
                reports_count => $count_reports
            );
            return;
        }

        do {
            my $profile = Yandex::Trace::new_profile('api:new_offline_report_request');
            my $job = $queue->insert_job( {
                job_id => get_new_id('job_id'),
                ClientID => $params->client_id,
                args => $params->job_args,
                priority => query_special_user_option($params->client_id, 'advq_report_queue_priority') || $API::Settings::WORDSTAT_DEFAULT_PRIORITY,
            } );

            do_replace_into_table( PPC( ClientID => $params->client_id ), 'api_reports_offline',
                { ClientID => $params->client_id, ReportName => $params->report_name, dbqueue_job_id => $job->job_id, request_data_md5 => $params->request_data_md5 } );

            $self = $class->new(
                ClientID => $params->client_id,
                ReportName => $params->report_name,
                dbqueue_job_id => $job->job_id,
                just_created => 1,
                dbqueue_job => $job,
            );
            $count_reports++;
        };
    };

    die $exception if $exception;
    return $self;
}

=head2 find_task

Находит задание в базе; если задания нет, возвращает undef

=cut

sub find_task {
    my ( $class, $ClientID, $ReportName ) = @_;

    my $queue = Yandex::DBQueue->new( PPC( ClientID => $ClientID ), DBQUEUE_JOB_TYPE );

    my $mapping_row = get_one_line_sql(
        PPC( ClientID => $ClientID ),
        'SELECT ClientID, ReportName, dbqueue_job_id FROM api_reports_offline WHERE ClientID = ? AND ReportName = ?',
        $ClientID, $ReportName );

    return undef unless $mapping_row;

    ## если строка есть в таблице с мэпингом и в нужной таблице в DBQueue,
    ## возвращаем запись
    if ( my $job = $queue->find_job_by_id( $mapping_row->{dbqueue_job_id} ) ) {
        return $class->new(
            ClientID => $mapping_row->{ClientID},
            ReportName => $mapping_row->{ReportName},
            dbqueue_job_id => $mapping_row->{dbqueue_job_id},
            just_created => 0,
            dbqueue_job => $job,
        );
    }

    return undef;
}

=head2 get_resulting_file

Возвращает Direct::Storage::File со сделанным отчётом. Если отчёт ещё не сделан, порождает исключение.

=cut

sub get_resulting_file {
    my ($self) = @_;

    my $job = $self->dbqueue_job;
    die "job isn't done yet" unless $job->is_finished;

    return Direct::Storage->new->get_file( STORAGE_FILE_TYPE,
        ClientID => $job->ClientID,
        filename => $job->result->{filename} )
}

1;
