package API::Reports::ReportRowProcessor;

=head1 NAME

API::Reports::ReportRowProcessor

=head1 DESCRIPTION

    Класс для обработки строки отчета перед выдачей клиенту.
    Преобразует названия полей из внутреннего формата, преобразует значения enum'ов (которые мы мапим сами),
    подставляет исходное название региона для неточно определенных регионов, заменяет, при необходимости,
    текст и идентификатор условия показа для АДФ, пересчитывает, при необходимости, значения денежных полей в микроединицы.

=cut

use Direct::Modern;

use Mouse;
use List::Util qw/any/;

use API::Reports::ReportTypeConstraints;

use API::Reports::DataRules qw/
    %INTERNAL_FIELD_TYPE_MAP
    %MONEY_INTERNAL_FIELD_MAP
    %RESULT_MAP_BY_FIELD
    $MONEY_MULT
    %SPECIAL_VALUE_FOR_RELATED_KEYWORD

    CRITERIA_ID_INTERNAL_FIELD
    CRITERIA_TYPE_INTERNAL_FIELD
    TARGET_LOCATION_NAME_INTERNAL_FIELD
    TARGET_LOCATION_NOT_EXACTLY_DETECTED_INTERNAL_FIELD
    LOCATION_OF_PRESENCE_NAME_INTERNAL_FIELD
    LOCATION_OF_PRESENCE_NOT_EXACTLY_DETECTED_INTERNAL_FIELD
    NOT_EXACTLY_DETECTED_LOCATION_PREFIX
    EMPTY_NUMERIC_VALUE
    EMPTY_STRING_VALUE
    TYPE_STRING
    TYPE_ID
    TYPE_INTEGER
    TYPE_FLOAT
    TYPE_MONEY
    TYPE_SIGNED_MONEY
    %UAC_FIELDS_TO_HIDE
/;

use constant TRUE => 1;
use constant FALSE => 0;

has requested_fields => (isa => 'ArrayRef', is => 'ro', required => TRUE);
has uac_oids => ( isa => 'HashRef', is => 'ro', default => sub {{}} );
has need_conversion_to_micro_units => (isa => 'Bool', is => 'ro', default => FALSE);
has report_type => (isa => 'ReportTypes', is => 'ro', required => TRUE);
has _type_by_internal_field => (isa => 'HashRef', is => 'ro', default => sub {{}});
has _result_map_by_field => (isa => 'HashRef', is => 'ro', default => sub {{}});
has _money_field_map => (isa => 'HashRef', is => 'ro', default => sub {{}});
has _special_value_for_relevant_keyword => (isa => 'HashRef', is => 'ro', default => sub {{}});


around BUILDARGS => sub {
    my ($orig, $class, %args) = @_;

    if (exists $args{report_type}) {
        my $report_type = $args{report_type};
        $args{_type_by_internal_field} = $INTERNAL_FIELD_TYPE_MAP{$report_type} if exists $INTERNAL_FIELD_TYPE_MAP{$report_type};
        $args{_result_map_by_field} = $RESULT_MAP_BY_FIELD{$report_type} if exists $RESULT_MAP_BY_FIELD{$report_type};
        $args{_money_field_map} = $MONEY_INTERNAL_FIELD_MAP{$report_type} if exists $MONEY_INTERNAL_FIELD_MAP{$report_type};
        $args{_special_value_for_relevant_keyword} = $SPECIAL_VALUE_FOR_RELATED_KEYWORD{$report_type}
            if exists $SPECIAL_VALUE_FOR_RELATED_KEYWORD{$report_type};
    }

    return $class->$orig(%args);
};

sub _has_special_value {
    my ($self, $field) = @_;

    return $self->_special_value_for_relevant_keyword && exists $self->_special_value_for_relevant_keyword->{$field};
}

sub _get_special_value {
    my ($self, $field) = @_;
    return $self->_special_value_for_relevant_keyword->{$field};
}

sub _has_result_map {
    my ($self, $field) = @_;
    return exists $self->_result_map_by_field->{$field};
}

sub _get_result_value {
    my ($self, $field, $value) = @_;
    return exists $self->_result_map_by_field->{$field}{$value} ? $self->_result_map_by_field->{$field}{$value} : undef;
}

sub _get_type {
    my ($self, $field) = @_;
    return exists $self->_type_by_internal_field->{$field} ? $self->_type_by_internal_field->{$field} : undef;
}

sub _has_type {
    my ($self, $field) = @_;
    return exists $self->_type_by_internal_field->{$field};
}

sub _is_money_field {
    my ($self, $field) = @_;
    return exists $self->_money_field_map->{$field};
}

=head2 process_report_row($row)

    На основе данных из БК формируем данные для клиента, преобразуем значения Enum'ов

=cut

sub process_report_row {
    my ($self, $result_row) = @_;
    my @report_row;

    my $is_related_keyword_row = exists $result_row->{+CRITERIA_ID_INTERNAL_FIELD} && ($result_row->{+CRITERIA_ID_INTERNAL_FIELD} // 0) <= 1;
    my $is_autotargeting_row = (defined $result_row->{+CRITERIA_TYPE_INTERNAL_FIELD} 
            && $result_row->{+CRITERIA_TYPE_INTERNAL_FIELD} eq "AUTOTARGETING") ? 1 : 0;
    my $uac_oids = $self->uac_oids;

    foreach my $field (@{$self->requested_fields}) {
        my $value = $result_row->{$field};
        my $field_without_postfix = $field;
        $field_without_postfix =~ s/(\w+)_\d+_\d/$1/;

        if ($is_related_keyword_row && !$is_autotargeting_row && $self->_has_special_value($field_without_postfix)) {
            # заменяем пустыми значениями текст и идентификатор у АДФ
            $value = $self->_get_special_value($field_without_postfix);
        } elsif ($self->need_conversion_to_micro_units && $self->_is_money_field($field_without_postfix)) {
            $value *= $MONEY_MULT if defined $value;
#       } elsif ($self->_has_result_map($field)) {
#           $value = $self->_get_result_value($field, $value);
        } elsif ( ($field_without_postfix eq TARGET_LOCATION_NAME_INTERNAL_FIELD && $result_row->{+TARGET_LOCATION_NOT_EXACTLY_DETECTED_INTERNAL_FIELD})
            || ($field_without_postfix eq LOCATION_OF_PRESENCE_NAME_INTERNAL_FIELD && $result_row->{+LOCATION_OF_PRESENCE_NOT_EXACTLY_DETECTED_INTERNAL_FIELD}) ) {
            # ядро возвращает часть регионов как неточно определенные и добавляет им строковый комментарий в название,
            # при этом оригинальное название без комментария сохраняется в отдельном ключе original_xxx
            # мы возвращаем всегда оригинальное название без комментария
            $value = $result_row->{NOT_EXACTLY_DETECTED_LOCATION_PREFIX.$field_without_postfix};
        }
        
        if ($UAC_FIELDS_TO_HIDE{$field} && $result_row->{OrderID} && $uac_oids->{$result_row->{OrderID}}) {
            $value = undef;
        }

        my $has_type = $self->_has_type($field_without_postfix);
        my $type = $self->_get_type($field_without_postfix);
        if ( $has_type && $type eq TYPE_ID ) {
            push @report_row, $value || EMPTY_NUMERIC_VALUE;
        } else {
            push @report_row, $value // EMPTY_NUMERIC_VALUE;
        }
    }

    return \@report_row;
}

__PACKAGE__->meta->strict_constructor(1);
__PACKAGE__->meta->make_immutable;

1;
