package API::Reports::ResultingDataIterator;
use Direct::Modern;

=head1 NAME

API::Reports::ResultingDataIterator

=head1 DESCRIPTION

Итератор по строкам отчёта. Даёт возможность один раз пройтись циклом по всем строкам;
возврат назад или удаление-изменение строк не поддерживаются.

=head1 SYNOPSIS

    use API::Reports::FormatTsv 'tsv_line';
    my $iterator = API::Reports::Builder->build_report(...)->data_iterator;
    while ($iterator->has_next_row) {
        print tsv_line($iterator->next_row);
    }

=cut

use API::Reports::ReportRowProcessor;

our $STAT_CHUNK_SIZE //= 20000;

=head2 new

Конструктор. Хоть и публичный, инстанциировать этот объект стоит только из API::Reports::Result (что он и делает),
из других мест создавать объекты не надо. Все поля в объекте приватные.

=cut
sub new {
    my ( $class, %args ) = @_;

    my %uac_oids_hash = map { $_ => 1 } @{$args{uac_oids} // []};

    return bless {
        _processor => API::Reports::ReportRowProcessor->new(
            requested_fields => $args{requested_report_fields},
            uac_oids => \%uac_oids_hash,
            need_conversion_to_micro_units => $args{need_conversion_to_micro_units},
            report_type => $args{report_type},
        ),
        _buffer => undef,
        _stat_iterator => $args{stat_iterator},
        _exhausted => !$args{stat_iterator},
        _date_aggregation_by => $args{date_aggregation_by},
    }, $class;
}

=head2 _try_refill_buffer

Пытается заполнить $self->{_buffer} правильно обработанными данными из итератора статистики.
Если в итераторе статистики данные кончились, $self->{_buffer} устанавливается в undef.

Возвращаемого значения нет.

=cut

sub _try_refill_buffer {
    my ($self) = @_;

    $self->{_buffer} = $self->{_stat_iterator}->next_chunk( chunk_size => $STAT_CHUNK_SIZE );

    return;
}

=head2 has_next_row

=cut

sub has_next_row {
    my ($self) = @_;

    return 0 if $self->{_exhausted};

    unless ( defined $self->{_buffer} ) {
        $self->_try_refill_buffer;
        unless ( defined $self->{_buffer} ) {
            $self->{_exhausted} = 1;
            return 0;
        }
    }

    return 1;
}

=head2 next_row

Возвращает arrayref со значениями в текущей строке. Для вывода в TSV стоит передать это в tsv_line.

Если данные кончились, порождает исключение; предполагается, что потребитель перед вызовом этого метода позовёт
has_next_row и с этим исключением не столкнётся никогда.

=cut

sub next_row {
    my ($self) = @_;

    if ( $self->{_exhausted} ) {
        die 'Iterator index out of bounds';
    }

    unless ( defined $self->{_buffer} ) {
        $self->_try_refill_buffer;
        unless ( defined $self->{_buffer} ) {
            die 'Iterator index out of bounds';
        }
    }

    my $row = shift @{ $self->{_buffer} };
    $self->{_buffer} = undef unless @{ $self->{_buffer} };

    return $self->{_processor}->process_report_row($row);
}

1;
