package API::Soapize;
## no critic (TestingAndDebugging::RequireUseWarnings)

=pod

    $Id$

=head1 NAME

    API::Soapize

=head1 DESCRIPTION

    Содеражит функции соапизации данных

=cut

use strict;

use SOAP::Lite;

use Yandex::ListUtils;

require Exporter;
our @ISA = qw(Exporter);
our @EXPORT = qw(

     set_soap_types
     soapize

     soapize_banner
     soapize_phrase
     soapize_client
     soapize_camp_param
     soapize_short_camp
     soapize_short_client
);

our $contact_info_soapize_tbl = {
        string  => [qw(WorkTime ExtraMessage ContactPerson Build IMLogin Street House City CityCode Phone PhoneExt IMClient Country CompanyName Apart ContactEmail CountryCode OGRN)],
    };

sub set_soap_types
{
    my ($type, $data, $api_version) = @_;
    my $result;

    my %INT_METHODS = (
        GetVersion => 1,
        PingAPI => 1,
        ModerateBanners => 1,
        DeleteForecastReport => 1,
        CreateNewForecast => 1,
        CreateNewWordstatReport => 1,
        CreateNewReport => 1,
        DeleteReport => 1,
        DeleteWordstatReport => 1,
        UpdatePrices => 1,
        TransferMoney => 1,
        UpdateClientInfo => 1,
        PayCampaigns => 1,
        SaveSubscription => 1,
        DeleteSubscription => 1,
        UpdateBannersTags => 1,
        AddEvents => 1,
    );

    if($type eq 'GetRegions'){
        $data = [map { API::Soapize::Data->type(RegionInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfRegionInfo => $data);

    } elsif($type eq 'GetMetroStations'){
        $data = [map { API::Soapize::Data->type(MetroStationInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfMetroStationInfo => $data);

    } elsif($type eq 'SetAutoPrice'){
        my $soapize_table = {
            int         => [qw(CampaignID)],
            long        => [qw/BannerID PhraseID/],
            float       => [qw(Price ContextPrice)],
        };

        $data = [map {soapize($_, $soapize_table); $_} @$data];

        $data = [map { API::Soapize::Data->type(PhrasePriceInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfPhrasePriceInfo => $data);

    } elsif($type eq 'GetRubrics'){
        $data = [map { API::Soapize::Data->type(RubricInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfRubricInfo => $data);


    } elsif($type eq 'GetTimeZones'){
        my $soapize_table = {
            int =>      [qw(GMTOffset)],
            string =>   [qw(TimeZone Name)]
        };

        @$data = map {soapize($_, $soapize_table); $_} xuniq {$_->{TimeZone}} @$data;

        $data = [map { API::Soapize::Data->type(TimeZoneInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfTimeZoneInfo => $data);

    } elsif ($type eq 'GetCampaignsList' || $type eq 'GetCampaignsListFilter'){

        for my $camp (@$data) {
            $camp = soapize_short_camp($camp);

            if ($camp->{TimeTarget}) {
                $camp->{TimeTarget} = API::Soapize::Data->type(TimeTargetInfo => $camp->{TimeTarget} );
                $camp->{TimeTarget}->{DaysHours} = API::Soapize::Data->type(ArrayOfTimeTargetItem =>
                                                 [map {API::Soapize::Data->type(TimeTargetItem => $_)} @{$camp->{TimeTarget}->{DaysHours}}]
                    );
            }
        }

        $result = API::Soapize::Data->type(ArrayOfShortCampaignInfo => [map { API::Soapize::Data->type(ShortCampaignInfo => $_ ) } @$data]);

    } elsif ($type eq 'GetBanners'){

        foreach my $d (@$data){
            soapize_banner($d);
            $d = API::Soapize::Data->type(BannerInfo => $d);
        }

        $result = API::Soapize::Data->type(ArrayOfBannerInfo => $data);

    } elsif ($type eq "GetClientsList"){
        foreach my $c (@$data){
            $c = soapize_client($c);
        }

        $data = [map { API::Soapize::Data->type(ClientInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfClientInfo => $data);
    } elsif ($type eq 'SearchClients') {
        foreach my $c (@{$data->{Clients}}){
            $c = soapize_client($c);
        }
        $data->{Clients} = API::Soapize::Data->type(ArrayOfClientInfo => $data->{Clients});
        $data->{Count} = API::Soapize::Data->type(int => $data->{Count});
        $result = API::Soapize::Data->type(SearchClientsResponse => $data);
    } elsif ($type eq "GetReportList"){

        foreach my $d (@$data){
            $d->{Filter} = API::Soapize::Data->type(NewReportFilterInfo => $d->{Filter} ) if $d->{Filter};
            $d = API::Soapize::Data->type(ReportInfo => $d);
        }
        $result = API::Soapize::Data->type(ArrayOfReportInfo => $data);

    } elsif ($type eq "GetClientsUnits"){

        $data = [map { API::Soapize::Data->type(ClientsUnitInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfClientsUnitInfo => $data);

    } elsif ($type eq "GetClientInfo") {

        foreach my $d (@$data) {
            $d = soapize_client($d);
        }

        $data = [map { API::Soapize::Data->type(ClientInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfClientInfo => $data);

    } elsif ($type eq "GetBalance"){
        my $soapize_table = {
            int =>      [qw(CampaignID)],
            float =>    [qw(Rest Sum SumAvailableForTransfer)]
        };

        $data = [map {soapize($_, $soapize_table); $_} @$data];

        $data = [map { API::Soapize::Data->type(CampaignBalanceInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfCampaignBalanceInfo => $data);

    } elsif ($type eq "GetForecast" || $type eq "GetForecastSync"){
        foreach my $d (@{$data->{Phrases}}){
            soapize_phrase($d);
            $d = API::Soapize::Data->type(BannerPhraseInfo => $d );
        }

        $data->{Phrases} = API::Soapize::Data->type(ArrayOfBannerPhraseInfo => $data->{Phrases});
        $data->{Categories} = API::Soapize::Data->type(ArrayOfBannerPhraseInfo => []);

        my $soapize_table = {
            int     => [qw/Clicks Shows PremiumClicks FirstPlaceClicks/],
            float   => [qw/PremiumMin Min Max/],
            string  => [qw/Geo/],
        };
        soapize($data->{Common}, $soapize_table);
        $data->{Common} = API::Soapize::Data->type(ForecastCommonInfo => $data->{Common});

        $result = API::Soapize::Data->type(GetForecastInfo => $data);

    } elsif ($type eq "GetRubricsRequest"){
        $data = [map { API::Soapize::Data->type(RubricInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfRubricInfo => $data);

    } elsif ($type eq "GetForecastList") {
        $data = [map { API::Soapize::Data->type(ForecastStatusInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfForecastStatusInfo => $data);

    } elsif ($type eq "GetSubClients") {
            $data = [map {soapize_short_client($_)} @$data];
            $data = [map { API::Soapize::Data->type(ShortClientInfo => $_ ) } @$data];
            $result = API::Soapize::Data->type(ArrayOfShortClientInfo => $data);

    } elsif ($type eq "GetCampaignParams") {
        $result = soapize_camp_info($data, $api_version);
    } elsif ($type eq "GetCampaignsParams") {
        $result = [ map { soapize_camp_info($_, $api_version) } @$data ];
        $result = API::Soapize::Data->type(ArrayOfCampaignInfo => $result);

    } elsif ($type eq "CreateOrUpdateBanners"){
        $result = API::Soapize::Data->type(ArrayOfLong => $data);

    } elsif ($type eq "GetBannerPhrases" || $type eq "GetBannerPhrasesFilter") {

        foreach my $d (@$data){
            soapize_phrase($d);
            $d = API::Soapize::Data->type(BannerPhraseInfo => $d );
        }

        $result = API::Soapize::Data->type(ArrayOfBannerPhraseInfo => $data);

    } elsif($type eq 'GetAvailableVersions') {

        my $soapize_table = {
            int => [qw/VersionNumber/],
            date => [qw/BorderDate/],
            string => [qw/LinkWSDL/]
        };
        $data = [map {soapize($_, $soapize_table); $_} @$data];

        $data = [map { API::Soapize::Data->type(VersionDesc => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfVersionDesc => $data);

    } elsif($type eq 'GetKeywordsSuggestion') {

        # так можно ловить ошибку Schema/namespace for type \'GetKeywordsSuggestionInfo\' is not specified
        #$data = [map { API::Soapize::Data->type(GetKeywordsSuggestionInfo => $_ ) } @$data];
        #$result = API::Soapize::Data->type(ArrayOfString => $data);

        $data = [map {API::Soapize::Data->type(string => $_)} @$data];
        $result = API::Soapize::Data->type(ArrayOfString => $data);
    } elsif($type eq 'GetNormalizedKeywords') {
        $data = [map {API::Soapize::Data->type(string => $_)} @$data];
        $result = API::Soapize::Data->type(ArrayOfString => $data);
    } elsif($type eq 'GetNormalizedKeywordsData') {
        for my $items (@$data){
            for my $item (@$items){
                my $soapize_table = {
                    ArrayOfString => [qw/Lemmas Formas/],
                    string => [qw/Word/]
                };
                $item->{Lemmas} = [ map { API::Soapize::Data->type("string" => $_) } @{$item->{Lemmas} || []} ];
                $item->{Formas} = [ map { API::Soapize::Data->type("string" => $_) } @{$item->{Formas} || []} ];
                soapize($item, $soapize_table);
            }
            $items = API::Soapize::Data->type(ArrayOfNormalizedKeywordsData => $items);
        }
        $result = API::Soapize::Data->type(ArrayOfNormalizedPhraseData => $data);
    } elsif ($type eq 'GetKeywordsIntersection') {
        for my $item (@{$data->{IntersectingKeywords}}) {
            for my $keyword (@$item) {
                my $soapize_table = {
                    long   => [qw/KeywordID/],
                    string => [qw/Keyword/],
                };
                soapize($keyword, $soapize_table);
            }
            $item = API::Soapize::Data->type(ArrayOfKeywordItem => $item);
        }
        $data->{IntersectingKeywords} = API::Soapize::Data->type(KeywordsIntersection => $data->{IntersectingKeywords});
        $result = API::Soapize::Data->type(KeywordIntersectionResponse => $data);

    } elsif ($type eq 'RearrangeKeywords') {
        $data->{Keywords} = API::Soapize::Data->type(ArrayOfString => 
            [
                map {
                    API::Soapize::Data->type(string => $_)
                } @{$data->{Keywords}}
            ]
        );

        $result = API::Soapize::Data->type(
            RearrangeKeywordsResponse => $data
        );
    } elsif ($type eq 'Keyword') {
        for my $keyword (@{$data->{ActionsResult}}) {

            my $soapize_table = {
                long => [qw/KeywordID/],
            };
            my $qi_table = {
                decimal => [qw/Value/]
            };
            soapize($keyword, $soapize_table);
            soapize($keyword->{QualityIndex}, $qi_table);
        }

        $result = API::Soapize::Data->type(
            KeywordResponse => $data
        );
    } elsif($type eq 'CreateNewSubclient') {

        foreach my $item (qw/Login Password FIO Email/) {
            $data->{$item} = API::Soapize::Data->type(string => $data->{$item});
        }

        $result = API::Soapize::Data->type(CreateNewSubclientResponse => $data);
    } elsif ($type eq 'GetEventsLog') {

        foreach my $item (@$data) {
            if ($item->{Attributes}) {
                my $attr_soapize_table = {
                    float => [qw(Rest Payed MinPrice)],
                    date =>  [qw(FinishDate)],
                    string => [qw/ModerationResult OldPlace StopTime/],
                };

                soapize($item->{Attributes}, $attr_soapize_table);
                $item->{Attributes} = API::Soapize::Data->type(EventsLogItemAttributes => $item->{Attributes});
            }

            my $soapize_table = {
                long        => [qw/BannerID PhraseID/],
                int         => [qw/CampaignID AccountID/],
                string      => [qw/EventType Timestamp EventName TextDescription/],
            };

            soapize($item, $soapize_table);

            $item = API::Soapize::Data->type(EventsLogItem => $item);
        }

        $result = API::Soapize::Data->type(ArrayOfEventsLogItem => $data);


    } elsif ($type eq 'GetCampaignsTags' || $type eq 'UpdateCampaignsTags') {

        foreach my $item (@$data) {

            foreach my $tag (@{$item->{Tags}}) {
                my $soapize_table = {
                    int => [qw/TagID/],
                    string => [qw/Tag/],
                };
                soapize($tag, $soapize_table);

                $tag = API::Soapize::Data->type(TagInfo => $tag);
            }

            my $soapize_table = {
                int => [qw/CampaignID/],
                ArrayOfTagInfo => [qw/Tags/],
            };

            soapize($item, $soapize_table);

            $item = API::Soapize::Data->type(CampaignTagsInfo => $item);
        }

        $result = API::Soapize::Data->type(ArrayOfCampaignTagsInfo => $data);


    } elsif ($type eq 'GetBannersTags') {

        foreach my $banner (@$data) {

            soapize($banner, {
                long => [qw/BannerID/],
                ArrayOfInteger => [qw/TagIDS/],
            });

            $banner = API::Soapize::Data->type(BannerTagsInfo => $banner);
        }

        $result = API::Soapize::Data->type(ArrayOfBannerTagsInfo => $data);

    } elsif ($INT_METHODS{$type}) {
        $result = API::Soapize::Data->type(int => $data);

    }  elsif ($type =~ /^CreateInvoice$/) {
        $result = API::Soapize::Data->type(string => $data);
    }  elsif ($type eq "GetCreditLimits"){

        foreach my $item (@{$data->{Limits}}) {
            $item->{ContractID} = API::Soapize::Data->type(string => $item->{ContractID});
            $item->{Limit} = API::Soapize::Data->type(float => $item->{Limit});
            $item->{LimitSpent} = API::Soapize::Data->type(float => $item->{LimitSpent});

            $item = API::Soapize::Data->type(CreditLimitsItem => $item);
        }

        $data->{Limits} = API::Soapize::Data->type(ArrayOfCreditLimitsItem => $data->{Limits});

        $result = API::Soapize::Data->type(CreditLimitsInfo => $data);

    } elsif ($type eq "GetWordstatReportList"){
        $data = [map { API::Soapize::Data->type(WordstatReportStatusInfo => $_ ) } @$data];
        $result = API::Soapize::Data->type(ArrayOfWordstatReportStatusInfo => $data);

    } elsif ($type eq "GetWordstatReport" || $type eq "GetWordstatSync") {

        foreach my $item (@$data) {
            $item->{GeoID} = API::Soapize::Data->type(ArrayOfInteger => $item->{GeoID});
            $item->{Phrase} = API::Soapize::Data->type(string => $item->{Phrase});

            foreach my $wordstat (@{$item->{SearchedWith}}, @{$item->{SearchedAlso}}) {
                $wordstat->{Phrase} = API::Soapize::Data->type(string => $wordstat->{Phrase});
                $wordstat = API::Soapize::Data->type(WordstatItem => $wordstat);
            }

            $item->{SearchedWith} = API::Soapize::Data->type(ArrayOfWordstatItem => $item->{SearchedWith});
            $item->{SearchedAlso} = API::Soapize::Data->type(ArrayOfWordstatItem => $item->{SearchedAlso});

            $item = API::Soapize::Data->type(WordstatReportInfo => $item);
        }

        $result = API::Soapize::Data->type(ArrayOfWordstatReportInfo => $data);

    } elsif ($type eq "GetStatGoals") {
        foreach my $item (@$data) {
            $item->{GoalID} = API::Soapize::Data->type(int => $item->{GoalID});
            if (exists $item->{CampaignID}) {
                $item->{CampaignID} = API::Soapize::Data->type(int => $item->{CampaignID});
            }
            $item->{Name} = API::Soapize::Data->type(string => $item->{Name});

            if ($item->{AvailableForStrategy}) {
                $item->{AvailableForStrategy} = API::Soapize::Data->type(string => $item->{AvailableForStrategy});
            }

            $item = API::Soapize::Data->type(StatGoalInfo => $item);
        }

        $result = API::Soapize::Data->type(ArrayOfStatGoalInfo => $data);

    } elsif ($type eq "GetSummaryStat") {

        my $soapize_table = {
            int =>   [qw(CampaignID ShowsSearch ShowsContext ClicksSearch ClicksContext)],
            float => [qw(SumSearch SumContext SessionDepthSearch SessionDepthContext GoalConversionContext GoalConversionSearch GoalCostContext GoalCostSearch)],
            date =>  [qw(StatDate)],
        };

        foreach my $stat_item (@$data) {
            soapize($stat_item, $soapize_table);
            $stat_item = API::Soapize::Data->type(StatItem => $stat_item);
        }

        $result = API::Soapize::Data->type(ArrayOfStatItem => $data);

    } elsif ($type eq "GetBannersStat") {

        my $soapize_table = {
            long        =>   [qw/BannerID PhraseID/],
            int         =>   [qw(Shows ShowsSearch ShowsContext Clicks ClicksSearch ClicksContext RubricID RetargetingID WebpageID DynamicMediaAdFilterID)],
            float       =>   [qw(Sum SumSearch SumContext ClicksAveragePosition ShowsAveragePosition Revenue ROI)],
            date        =>   [qw(StatDate)],
            string      =>   [qw(Phrase StatType DeviceType)]
        };

        foreach my $stat_item (@{$data->{Stat}}) {
            soapize($stat_item, $soapize_table);
            $stat_item = API::Soapize::Data->type(BannersStatItem => $stat_item);
        }
        $data->{Stat} = API::Soapize::Data->type(ArrayOfBannersStatItem => $data->{Stat});

        $data->{CampaignID} = API::Soapize::Data->type(int  => $data->{CampaignID});
        $data->{StartDate}  = API::Soapize::Data->type(date => $data->{StartDate});
        $data->{EndDate}    = API::Soapize::Data->type(date => $data->{EndDate});

        $result = API::Soapize::Data->type(GetBannersStatResponse => $data);

    } elsif ($type eq "GetSubscription") {
        $data->{EventTypes} = API::Soapize::Data->type(ArrayOfString => $data->{EventTypes});
        $result = API::Soapize::Data->type(Subscription => $data);
    } elsif ($type eq "PingAPI_X") {
        $result = [map {soapize($_, {'date'=>['DateValue']}); $_} @$data];
    } elsif ($type eq 'Mediaplan') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{CampaignID}) {
                    $res->{CampaignID} = API::Soapize::Data->type(int => $res->{CampaignID});
                }
                $res = API::Soapize::Data->type(MediaplanActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfMediaplanActionResult => $data->{ActionsResult});
        }
        if ($data->{Mediaplans}) {

            foreach my $mediaplan (@{$data->{Mediaplans}}) {
                $mediaplan = soapize_mediaplan($mediaplan);
            }

            $data->{Mediaplans} = API::Soapize::Data->type(ArrayOfMediaplan => $data->{Mediaplans});
        }
        $result = API::Soapize::Data->type(MediaplanResponse => $data);
    } elsif ($type eq 'MediaplanAd') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{MediaplanAdID}) {
                    $res->{MediaplanAdID} = API::Soapize::Data->type(long => $res->{MediaplanAdID});
                }
                $res = API::Soapize::Data->type(MediaplanAdActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfMediaplanAdActionResult => $data->{ActionsResult});
        }
        if ($data->{MediaplanAds}) {

            foreach my $banner (@{$data->{MediaplanAds}}) {
                $banner = soapize_mediaplan_ad($banner);
            }

            $data->{MediaplanAds} = API::Soapize::Data->type(ArrayOfMediaplanAd => $data->{MediaplanAds});
        }
        $result = API::Soapize::Data->type(MediaplanAdResponse => $data);
    } elsif ($type eq 'MediaplanKeyword') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{MediaplanKeywordID}) {
                    $res->{MediaplanKeywordID} = API::Soapize::Data->type(long => $res->{MediaplanKeywordID});
                }
                $res = API::Soapize::Data->type(MediaplanKeywordActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfMediaplanKeywordActionResult => $data->{ActionsResult});

        }
        if ($data->{MediaplanKeywords}) {

            foreach my $keyword (@{$data->{MediaplanKeywords}}) {
                $keyword = soapize_mediaplan_keyword($keyword);
            }

            $data->{MediaplanKeywords} = API::Soapize::Data->type(ArrayOfMediaplanKeyword => $data->{MediaplanKeywords});
        }
        $result = API::Soapize::Data->type(MediaplanKeywordResponse => $data);
    } elsif ($type eq 'MediaplanCategory') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{MediaplanCategoryID}) {
                    $res->{MediaplanCategoryID} = API::Soapize::Data->type(long => $res->{MediaplanCategoryID});
                }
                $res = API::Soapize::Data->type(MediaplanCategoryActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfMediaplanCategoryActionResult => $data->{ActionsResult});
        }
        if ($data->{MediaplanCategories}) {

            foreach my $category (@{$data->{MediaplanCategories}}) {
                $category = soapize_mediaplan_category($category);
            }

            $data->{MediaplanCategories} = API::Soapize::Data->type(ArrayOfMediaplanCategory => $data->{MediaplanCategories});
        }
        $result = API::Soapize::Data->type(MediaplanCategoryResponse => $data);
    } elsif ($type eq 'GetRetargetingGoals') {
        my $retargeting_goal_soapize_table = {
            long     => [qw/GoalID/],
            string  => [qw/Name GoalDomain Login/],
        };

        foreach my $goal (@{$data}) {
            soapize($goal, $retargeting_goal_soapize_table);
            $goal = API::Soapize::Data->type(RetargetingGoal => $goal);
        }
        $result = API::Soapize::Data->type(ArrayOfRetargetingGoal => $data);
    } elsif ($type eq 'RetargetingCondition') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{RetargetingConditionID}) {
                    $res->{RetargetingConditionID} = API::Soapize::Data->type(int => $res->{RetargetingConditionID});
                }
                $res = API::Soapize::Data->type(RetargetingConditionActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfRetargetingConditionActionResult => $data->{ActionsResult});
        }
        if ($data->{RetargetingConditions}) {
            foreach my $ret (@{$data->{RetargetingConditions}}) {
                $ret = soapize_retargeting_condition($ret);
            }

            $data->{RetargetingConditions} = API::Soapize::Data->type(ArrayOfRetargetingCondition => $data->{RetargetingConditions});
        }
        $result = API::Soapize::Data->type(RetargetingConditionResponse => $data);
    } elsif ($type eq 'Retargeting') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{RetargetingID}) {
                    $res->{RetargetingID} = API::Soapize::Data->type(int => $res->{RetargetingID});
                }
                $res = API::Soapize::Data->type(RetargetingActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfRetargetingActionResult => $data->{ActionsResult});
        }
        if ($data->{Retargetings}) {
            foreach my $ret (@{$data->{Retargetings}}) {
                $ret = soapize_retargeting($ret);
            }

            $data->{Retargetings} = API::Soapize::Data->type(ArrayOfRetargeting => $data->{Retargetings});
        }
        $result = API::Soapize::Data->type(RetargetingResponse => $data);
    } elsif ($type eq 'AdImage') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{AdImageUploadTaskID}) {
                    $res->{AdImageUploadTaskID} = API::Soapize::Data->type(int => $res->{AdImageUploadTaskID});
                }
                $res = API::Soapize::Data->type(AdImageActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfAdImageActionResult => $data->{ActionsResult});
        }
        if ($data->{AdImages}) {
            foreach my $item (@{$data->{AdImages}}) {
                $item = soapize_ad_image($item);
            }

            $data->{AdImages} = API::Soapize::Data->type(ArrayOfAdImage => $data->{AdImages});
        }
        if ($data->{AdImageUploads}) {
            foreach my $item (@{$data->{AdImageUploads}}) {
                $item = soapize_ad_image_upload($item);
            }

            $data->{AdImageUploads} = API::Soapize::Data->type(ArrayOfAdImageUpload => $data->{AdImageUploads});
        }
        if ($data->{AdImageLimits}) {
            foreach my $item (@{$data->{AdImageLimits}}) {
                $item = soapize_ad_image_limit($item);
            }

            $data->{AdImageLimits} = API::Soapize::Data->type(ArrayOfAdImageLimit => $data->{AdImageLimits});
        }
        $result = API::Soapize::Data->type(AdImageResponse => $data);
    } elsif ($type eq 'AdImageAssociation') {
        if ($data->{ActionsResult}) {
            foreach my $res (@{$data->{ActionsResult}}) {
                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }
                if ($res->{Warnings}) {
                    $res->{Warnings} = soapize_warnings($res->{Warnings});
                }
                if ($res->{AdID}) {
                    $res->{AdID} = API::Soapize::Data->type(long => $res->{AdID});
                }
                $res = API::Soapize::Data->type(AdImageAssociationActionResult => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfAdImageAssociationActionResult => $data->{ActionsResult});
        }
        if ($data->{AdImageAssociations}) {
            foreach my $item (@{$data->{AdImageAssociations}}) {
                $item = soapize_ad_image_association($item);
            }

            $data->{AdImageAssociations} = API::Soapize::Data->type(ArrayOfAdImageAssociation => $data->{AdImageAssociations});
        }
        $result = API::Soapize::Data->type(AdImageAssociationResponse => $data);

    } elsif ($type eq 'EnableSharedAccount') { # метод стиля v4

        if ($data->{Errors}) {
            $data->{Errors} = soapize_errors($data->{Errors});
        }
        if ($data->{AccountID}) {
            $data->{AccountID} = API::Soapize::Data->type(int => $data->{AccountID});
        }
        if ($data->{Login}) {
            $data->{Login} = API::Soapize::Data->type(string => $data->{Login});
        }
        $result = API::Soapize::Data->type(EnableSharedAccountResponse => $data);

    } elsif ($type eq 'AccountManagement') {

        if ($data->{ActionsResult}) {

            foreach my $res (@{$data->{ActionsResult}}) {
                soapize($res, {
                    int    => [qw(AccountID)],
                    string => [qw(Login URL)],
                });

                if ($res->{Errors}) {
                    $res->{Errors} = soapize_errors($res->{Errors});
                }

                $res = API::Soapize::Data->type('AccountActionResult' => $res);
            }
            $data->{ActionsResult} = API::Soapize::Data->type(ArrayOfAccountActionResult => $data->{ActionsResult});
        }

        if ($data->{Accounts}) {

            foreach my $res (@{$data->{Accounts}}) {

                if ($res) {
                    soapize($res, {
                        int                     => [qw(AccountID)],
                        float                   => [qw(Amount AmountAvailableForTransfer Discount)],
                        string                  => [qw(Login AgencyName SendWarn Currency)],
                        SmsNotificationInfo     => [qw(SmsNotification)],
                        EmailNotificationInfo   => [qw(EmailNotification)],
                    });

                    $res = API::Soapize::Data->type('Account' => $res);
                }

            }
            $data->{Accounts} = API::Soapize::Data->type(ArrayOfAccount => $data->{Accounts});
        }
        $result = API::Soapize::Data->type('AccountManagementResponse' => $data);

    } else {
        warn "Undefined soap type: $type";
        $result = $data;
    }
    # if (ref $result && my $type = $result->type()) {
        # $result->name($type);
    # }
    return $result;
}

sub soapize {
    my $vars = shift;
    my $tbl = shift;
#    my $typize = shift;

    foreach my $type (keys %$tbl){
        foreach my $key (@{$tbl->{$type}}){
            if (exists $vars->{$key}) {
#                if ($typize) {
#                    # для json'a делаем цифры цифрами, а строки строками
#                    if (defined $vars->{$key}) {
#                        if ($type =~ /^(int|float|double)$/) {
#                            $vars->{$key} += 0;
#                        } elsif ($type =~ /^(string|date)$/) {
#                            $vars->{$key} .= "";
#                        }
#                    }
#                } else {
                    $vars->{$key} = API::Soapize::Data->type($type => $vars->{$key});
#                }
            }
        }
    }
}

sub soapize_errors
{
    my ($errors) = @_;
    return if ! $errors;

    foreach my $error (@$errors) {
        $error = soapize_error($error);
    }

    return API::Soapize::Data->type(ArrayOfError => $errors);
}

sub soapize_error
{
    my ($error) = @_;
    my $soapize_table = {
        int => [qw/FaultCode/],
        string => [qw/FaultString FaultDetail/]
    };
    soapize($error, $soapize_table);
    return API::Soapize::Data->type(Error => $error);
}

sub soapize_warnings
{
    my ($warnings) = @_;
    return if ! $warnings;

    my $soapize_table = {
        int => [qw/WarningCode/],
        string => [qw/WarningString Description/]
    };

    foreach my $warning (@$warnings) {
        soapize($warning, $soapize_table);
        $warning = API::Soapize::Data->type(Warning => $warning);
    }

    return API::Soapize::Data->type(ArrayOfWarning => $warnings);
}

sub soapize_camp_info
{
    my ($info, $api_version) = @_;

    $info = soapize_camp_param($info);
    $info->{MinusKeywords} = [map {API::Soapize::Data->type(string => $_)} @{ $info->{MinusKeywords} || [] }] if $info->{MinusKeywords};
    $info->{MinusKeywords} = API::Soapize::Data->type(ArrayOfString => $info->{MinusKeywords} ) if $info->{MinusKeywords};
    $info->{EmailNotification} = API::Soapize::Data->type(EmailNotificationInfo => $info->{EmailNotification} ) if $info->{EmailNotification};
    $info->{SmsNotification} = API::Soapize::Data->type(SmsNotificationInfo => $info->{SmsNotification} ) if $info->{SmsNotification};

    $info->{AdditionalMetrikaCounters} = API::Soapize::Data->type(ArrayOfInteger => $info->{AdditionalMetrikaCounters} ) if $info->{AdditionalMetrikaCounters};

    if ($info->{Strategy}) {

        my $strategy_soapize_table = {
            int => [qw/ClicksPerWeek GoalID/], 
            float => [qw/MaxPrice AveragePrice AverageCPA WeeklySumLimit ROICoef ReserveReturn Profitability/], 
            string => [qw/StrategyName/]
        };

        soapize($info->{Strategy}, $strategy_soapize_table);

        $info->{Strategy} = API::Soapize::Data->type(CampaignStrategy => $info->{Strategy} );
    }

    if ($info->{DayBudget}) {
        my $day_budget_soapize_table = {
            float => [qw/Amount/],
            string => [qw/SpendMode/]
        };
        soapize($info->{DayBudget}, $day_budget_soapize_table);
        $info->{DayBudget} = API::Soapize::Data->type(DayBudgetInfo => $info->{DayBudget} );
    }

    if ($info->{ContextStrategy}) {

        my $strategy_soapize_table = {
            float => [qw/MaxPrice AveragePrice WeeklySumLimit/],
            int => [qw/ContextPricePercent ContextLimitSum GoalID ClicksPerWeek/],
            string => [qw/StrategyName ContextLimit/]
        };

        soapize($info->{ContextStrategy}, $strategy_soapize_table);

        $info->{ContextStrategy} = API::Soapize::Data->type(CampaignContextStrategy => $info->{ContextStrategy} );
    }

    if ($info->{TimeTarget}) {
        foreach my $tt_item (@{$info->{TimeTarget}->{DaysHours}}) {
            foreach (qw/Days Hours/) {
                $tt_item->{$_} = API::Soapize::Data->type(ArrayOfInteger => $tt_item->{$_});
            }
            if ($api_version > 4) {
                $tt_item->{BidCoefs} = API::Soapize::Data->type(ArrayOfInteger => $tt_item->{BidCoefs});
            }

            $tt_item = API::Soapize::Data->type(TimeTargetItem => $tt_item);
        }

        $info->{TimeTarget} = API::Soapize::Data->type(TimeTargetInfo => $info->{TimeTarget} );
    }

    return API::Soapize::Data->type(CampaignInfo => $info);
}

sub soapize_client
{
    my $client = shift;

    if (exists $client->{ManagersLogins}) {
        $client->{ManagersLogins} = API::Soapize::Data->type('ArrayOfString' => [
            map {API::Soapize::Data->type('string' => $_)} @{$client->{ManagersLogins} || []}
        ]);
    }

    if (exists $client->{CampaignEmails}) {
        $client->{CampaignEmails} = API::Soapize::Data->type('ArrayOfString' => [
            map {API::Soapize::Data->type('string' => $_)} @{$client->{CampaignEmails} || []}
        ]);
    }

    if (exists $client->{ClientRights}) {
        $client->{ClientRights} = API::Soapize::Data->type('ArrayOfClientRight' => [
            map {API::Soapize::Data->type('ClientRight' => $_)} @{$client->{ClientRights} || []}
        ]);
    }

    if (exists $client->{Descriptions}) {
        foreach my $item ( @{$client->{Descriptions}} ) {
            soapize($item, {string => [qw/Description AgencyLogin/]});
            $item = API::Soapize::Data->type('ClientDescription' => $item);
        }
        $client->{Descriptions} = API::Soapize::Data->type('ArrayOfClientDescription' => $client->{Descriptions});
    }

    my $soapize_table = {
                            string => [qw(Login FIO Email Phone Role IsGoodUser NonResident AgencyName AgencyUrl AgencyStatus AllowAgencyMassADVQ SendNews SendAccNews SendWarn StatusArch SendAgencyMcbLetters SendAgencyDirectLetters SmsPhone Description
                            SharedAccountEnabled)],
                            int => [qw(CityID)],
                            date => [qw(DateCreate)],
                            float => [qw(Discount OverdraftSumAvailable OverdraftSumAvailableInCurrency)],
                        };
    soapize($client, $soapize_table);
    return $client;
}

=head2 soapize_contact_info($contact_info_structure)

    Соапизируем ContactInfo из BannerInfo, пользоваться soapize сразу нельзя, т.к. 
    структура сложная, содержит тип MapPoint, который не simple, указывать его в 
    $contact_info_soapize_tbl нельзя, в случае undef это приведет, к ошибки 
    сереализации, т.к. SOAP::Lite не сможет угадать тип данных

=cut

sub soapize_contact_info {
    my $contact_info = shift;

    soapize($contact_info, $contact_info_soapize_tbl);

    if(defined $contact_info->{PointOnMap}) {
        my $map = $contact_info->{PointOnMap};
        soapize($map, { float => [qw/x y x1 y1 x2 y2/] });
        $contact_info->{PointOnMap} = API::Soapize::Data->type(MapPoint => $map);
    }
    return;
}

sub soapize_banner
{
    my $banner = shift;

    my $banner_soapize_tbl = {
                int     => ['CampaignID'],
                long    => [qw(BannerID AdGroupID)],
                string  => [qw(Text Geo Href Domain Title
                            StatusActivating StatusArchive StatusBannerModerate
                            StatusPhoneModerate StatusPhrasesModerate
                            StatusSitelinksModerate StatusShow IsActive AdGroupName)],
              };

    soapize($banner, $banner_soapize_tbl);
    soapize_contact_info($banner->{ContactInfo}) if (exists $banner->{ContactInfo});
    soapize_sitelinks($banner->{Sitelinks}) if (exists $banner->{Sitelinks});

    $banner->{ContactInfo} = API::Soapize::Data->type(ContactInfo => $banner->{ContactInfo}) if defined $banner->{ContactInfo};
    $banner->{Sitelinks} = [map {API::Soapize::Data->type(Sitelink => $_)} @{ $banner->{Sitelinks} || [] }] if defined $banner->{Sitelinks};
    $banner->{Sitelinks} = API::Soapize::Data->type(ArrayOfSitelink => $banner->{Sitelinks}) if defined $banner->{Sitelinks};
    $banner->{MinusKeywords} = [map {API::Soapize::Data->type(string => $_)} @{ $banner->{MinusKeywords} || [] }] if $banner->{MinusKeywords};
    $banner->{MinusKeywords} = API::Soapize::Data->type(ArrayOfString => $banner->{MinusKeywords} ) if $banner->{MinusKeywords};
    if (defined $banner->{Phrases}) {
        foreach my $d (@{$banner->{Phrases}}){
            soapize_phrase($d);
            $d = API::Soapize::Data->type(BannerPhraseInfo => $d );
        }

        $banner->{Phrases} = API::Soapize::Data->type(ArrayOfBannerPhraseInfo => $banner->{Phrases});
    }

    return $banner;
}

sub soapize_mediaplan_ad
{
    my $banner = shift;
    my $mediaplan_ad_soapize_tbl = {
                int     => [qw(CampaignID)],
                long    => [qw(MediaplanAdID SourceAdID )],
                string  => [qw(Text Geo Href Domain Title)],
              };
    soapize($banner, $mediaplan_ad_soapize_tbl);
    soapize($banner->{ContactInfo}, $contact_info_soapize_tbl) if (exists $banner->{ContactInfo});
    soapize_sitelinks($banner->{Sitelinks}) if (exists $banner->{Sitelinks});
    $banner->{ContactInfo} = API::Soapize::Data->type(ContactInfo => $banner->{ContactInfo}) if defined $banner->{ContactInfo};
    $banner->{Sitelinks} = [map {API::Soapize::Data->type(Sitelink => $_)} @{ $banner->{Sitelinks} || [] }] if defined $banner->{Sitelinks};
    $banner->{Sitelinks} = API::Soapize::Data->type(ArrayOfSitelink => $banner->{Sitelinks}) if defined $banner->{Sitelinks};
    $banner->{MinusKeywords} = [map {API::Soapize::Data->type(string => $_)} @{ $banner->{MinusKeywords} || [] }] if $banner->{MinusKeywords};
    $banner->{MinusKeywords} = API::Soapize::Data->type(ArrayOfString => $banner->{MinusKeywords} ) if $banner->{MinusKeywords};

    return API::Soapize::Data->type(MediaplanAd => $banner);
}

sub soapize_mediaplan_keyword
{
    my $keyword = shift;

    my $mediaplan_keyword_soapize_table = {
        long        => [qw/MediaplanAdID MediaplanKeywordID/],
        int         => [qw/CampaignID/],
        float       => [qw/PremiumMax PremiumMin Min Max MinCTR MaxCTR PremiumMinCTR PremiumMaxCTR/],
        string      => [qw/Phrase LowCTR Position/],
    };

    soapize($keyword, $mediaplan_keyword_soapize_table);

    return API::Soapize::Data->type(MediaplanKeyword => $keyword);
}

sub soapize_mediaplan_category
{
    my $category = shift;

    my $mediaplan_category_soapize_table = {
        long        => [qw/MediaplanAdID MediaplanCategoryID/],
        int         => [qw/RubricID CampaignID/],
        float       => [qw/PremiumMax PremiumMin Min Max MinCTR MaxCTR PremiumMinCTR PremiumMaxCTR/],
        string      => [qw/PLowCTR Position/],
    };

    soapize($category, $mediaplan_category_soapize_table);

    return API::Soapize::Data->type(MediaplanCategory => $category);
}

sub soapize_mediaplan
{
    my $mediaplan = shift;
    my $mediaplan_soapize_table = {
        int     => [qw/CampaignID/],
        string  => [qw/ManagerLogin MediaplannerLogin MediaplanStatus/],
    };
    soapize($mediaplan, $mediaplan_soapize_table);

    return API::Soapize::Data->type(Mediaplan => $mediaplan);
}

sub soapize_retargeting_condition
{
    my $ret_cond = shift;
    my $retargeting_condition_soapize_table = {
        int     => [qw/RetargetingConditionID/],
        string  => [qw/RetargetingConditionName RetargetingConditionDescription Login IsAccesible/],
    };

    my $retargeting_condition_item_soapize_table = {
        string  => [qw/Type/],
    };

    my $retargeting_condition_item_goal_soapize_table = {
        int  => [qw/Time/],
        long => [qw/GoalID/],
    };

    for my $item (@{$ret_cond->{RetargetingCondition}}) {

        for my $goal (@{$item->{Goals}}) {
            soapize($goal, $retargeting_condition_item_goal_soapize_table);
            $goal = API::Soapize::Data->type(RetargetingConditionGoalItem => $goal);
        }
        soapize($item, $retargeting_condition_item_soapize_table);
        $item->{Goals} = API::Soapize::Data->type(ArrayOfRetargetingConditionItem => $item->{Goals});
        $item = API::Soapize::Data->type(RetargetingConditionItem => $item);
    }
    soapize($ret_cond, $retargeting_condition_soapize_table);
    $ret_cond->{RetargetingCondition} = API::Soapize::Data->type(ArrayOfRetargetingConditionItem => $ret_cond->{RetargetingCondition});
    return API::Soapize::Data->type(RetargetingCondition => $ret_cond);
}

sub soapize_retargeting
{
    my $ret = shift;
    my $retargeting_soapize_table = {
        int     => [qw/RetargetingID RetargetingConditionID/],
        long    => [qw/AdGroupID AdID/],
        float   => [qw/ContextPrice/],
        string  => [qw/StatusPaused AutoBudgetPriority/],
    };
    soapize($ret, $retargeting_soapize_table);

    return API::Soapize::Data->type(Retargeting => $ret);
}

sub soapize_ad_image
{
    my $ad_image = shift;

    my $ad_image_soapize_table = {
        string  => [qw/Login AdImageHash AdImageURL Assigned Name/],
    };
    soapize($ad_image, $ad_image_soapize_table);
    return API::Soapize::Data->type(AdImage => $ad_image);
}

sub soapize_ad_image_upload
{
    my $ad_image_upload = shift;

    my $ad_image_upload_soapize_table = {
        int     => [qw/AdImageUploadTaskID/],
        string  => [qw/Login AdImageHash AdImageURL SourceURL Name Status/],
    };
    soapize($ad_image_upload, $ad_image_upload_soapize_table);
    if ($ad_image_upload->{Error}) {
        $ad_image_upload->{Error} = soapize_error($ad_image_upload->{Error});
    }

    return API::Soapize::Data->type(AdImageUpload => $ad_image_upload);
}

sub soapize_ad_image_limit
{
    my $ad_image_limit = shift;

    my $ad_image_limit_soapize_table = {
        int     => [qw/Capacity Utilized/],
        string  => [qw/Login/],
    };
    soapize($ad_image_limit, $ad_image_limit_soapize_table);

    return API::Soapize::Data->type(AdImageLimit => $ad_image_limit);
}

sub soapize_ad_image_association
{
    my $ad_image_assoc = shift;

    my $ad_image_assoc_soapize_table = {
        int     => [qw/CampaignID/],
        long    => [qw/AdID/],
        string  => [qw/AdImageHash Login StatusAdImageModerate/],
    };
    soapize($ad_image_assoc, $ad_image_assoc_soapize_table);
    if ($ad_image_assoc->{ModerateRejectionReasons}) {
        $ad_image_assoc->{ModerateRejectionReasons} = soapize_modreasons($ad_image_assoc->{ModerateRejectionReasons});
    }

    return API::Soapize::Data->type(AdImageLimit => $ad_image_assoc);
}

sub soapize_modreasons
{
    my $modreasons = shift;

    for my $reason (@$modreasons) {
        $reason = API::Soapize::Data->type(RejectReason => $reason);
    }

    return API::Soapize::Data->type(ArrayOfRejectReason => $modreasons);
}

sub soapize_sitelinks
{
    my $sitelinks = shift;
    my $soapize_table = {
                            string => [qw/Href Title/],
                        };

    foreach (@{$sitelinks}) {
        soapize($_, $soapize_table);
    }
}

sub soapize_phrase
{
    my $phrase = shift;

    my $phrase_soapize_table = {
        long        => [qw/BannerID AdGroupID PhraseID/],
        int         => [qw/Clicks Shows PremiumClicks FirstPlaceClicks CampaignID/],
        float       => [qw/CTR Market Price PremiumMax PremiumMin CurrentOnSearch MinPrice Min Max PremiumCTR FirstPlaceCTR ContextPrice/],
        string      => [qw/AutoBroker AdGroupName Phrase IsRubric StatusPhraseModerate AutoBudgetPriority LowCTR LowCTRWarning ContextLowCTR/],
    };

    soapize($phrase, $phrase_soapize_table);

    if ($phrase->{ContextCoverage}) {
        foreach (@{$phrase->{ContextCoverage}}) {
            soapize($_, {int => ['Probability'], float => ['Price']});
        }

        $phrase->{ContextCoverage} = API::Soapize::Data->type(ArrayOfCoverageInfo => [(map {API::Soapize::Data->type(CoverageInfo => $_)} @{$phrase->{ContextCoverage}})] ) if $phrase->{ContextCoverage};
    }

    if ($phrase->{Coverage}) {
        foreach (@{$phrase->{Coverage}}) {
            soapize($_, {float => ['Price', 'Probability']});
        }

        $phrase->{Coverage} = API::Soapize::Data->type(ArrayOfCoverageInfo => [(map {API::Soapize::Data->type(CoverageInfo => $_)} @{$phrase->{Coverage}})] ) if $phrase->{Coverage};
    }

    if ($phrase->{Prices}) {
        $_ = API::Soapize::Data->type('float' => $_) foreach @{$phrase->{Prices}};
    }

    if (defined $phrase->{UserParams}) {
        soapize($phrase->{UserParams}, {string => [qw/Param1 Param2/]});

        $phrase->{UserParams} = API::Soapize::Data->type(PhraseUserParams => $phrase->{UserParams});
    }

    if (exists $phrase->{ContentRestricted}) {
        soapize($phrase, {string => [qw/ContentRestricted/]});
    }

    if (exists $phrase->{StatusPaused}) {
        soapize($phrase, {string => [qw/StatusPaused/]});
    }

    if (exists $phrase->{ContextClicks}) {
        soapize($phrase, {int => [qw/ContextClicks/]});
    }

    if (exists $phrase->{ContextShows}) {
        soapize($phrase, {int => [qw/ContextShows/]});
    }
    
    if (exists $phrase->{AuctionBids}) {
        foreach my $auction_bids (@{$phrase->{AuctionBids}}) {
            $auction_bids = soapize_auction_bid($auction_bids);
        }
    }
}

=head2 soapize_auction_bid

    Типизация структуры PhraseAuctionBids

=cut

sub soapize_auction_bid
{
    my $auction_bids = shift;
    my $soapize_table = {
        string => [qw/Position/],
        float  => [qw/Bid Price/],
    };
    soapize($auction_bids, $soapize_table);
    return $auction_bids;
}

=head2 soapize_short_client

    Типизация структуры ShortClientInfo

=cut

sub soapize_short_client
{
    my $client = shift;
    my $soapize_table = {
                            string => [qw(Login FIO Role)],
                        };
    soapize($client, $soapize_table);
    return $client;
}

=head2 soapize_short_camp

    Типизация структуры ShortCampaignInfo

=cut

sub soapize_short_camp
{
    my $camp = shift;
    my $campaign_soapize_table = {
        int     => [qw(Shows Clicks CampaignID)],
        float   => [qw(Rest Sum SumAvailableForTransfer AmountAvailableForTransfer BonusDiscount)],
        string  => [qw(ManagerName AgencyName StatusModerate IsActive StatusArchive Login StatusActivating Name StatusShow Status)],
        date    => [qw(StartDate FinishDate)],
    };
    soapize($camp, $campaign_soapize_table);
    return $camp;
}

=head2 soapize_camp_param

    Типизация структуры CampaignInfo api.v2

=cut

sub soapize_camp_param
{
    my $camp_params = shift;
    my $campaign_soapize_table = {
        int     => [qw(CampaignID Shows Clicks ContextLimitSum ContextPricePercent MobileBidAdjustment)],
        float   => [qw(Sum Rest AutobudgetMaxPrice AutobudgetSum SumAvailableForTransfer AmountAvailableForTransfer BonusDiscount)],
        string  => [qw(Name Login SmsTimeFrom SmsTimeTo Status StatusShow StatusModerate StatusActivating StatusArchive IsActive ManagerName AgencyName StatusBehavior Autobudget FIO ContextLimit ContextPriceCoef AutoOptimization StatusMetricaControl DisabledDomains DisabledIps ConsiderTimeTarget StatusOpenStat StatusContextStop AgencyLogin)],
        date    => [qw(StartDate FinishDate)],
    };
    soapize($camp_params, $campaign_soapize_table);
    return $camp_params;
}

1;

package API::Soapize::Data;

=head1 API::Soapize::Data — подкласс SOAP::Data

    Создаёт структуру, которая в нашем случае использования SOAP::Serializer производит идентичный XML-результат.
    1. _attr не пишется
    2. В объекты SOAP::Data преобразуются только скаляры

=cut

use base qw/SOAP::Data/;
my %attr = ();
sub type {
    my (undef, $type, $value) = @_;
=off отключено до следующего раза когда опять закончатся int-ы
    no warnings 'once';
    my $is_long = ($API::api_version == 4 && $API::api_wsdl_subversion == 2) ? 1 : 0;
    if ($type eq 'long_or_int') {
        $type = $is_long ? 'long' : 'int';
    } elsif ($type eq 'ArrayOfLongOrInt') {
        if ($is_long) {
            $type = 'ArrayOfLong';
            $value = [ map { API::Soapize::Data->type('long' => $_) } @{$value//[]} ];
        } else {
            $type = 'ArrayOfInt';
        }
    }
=cut

    if ($type eq 'ArrayOfLong') {
        $value = [ map { API::Soapize::Data->type('long' => $_) } @{$value//[]} ];
    }

    if (ref $value) {
        return bless($value, $type);
    } else {
        return bless({_value => [$value], _type => $type, _attr => \%attr}, 'SOAP::Data');
    }
}
1;
