package API::Strategies;

# $Id$


use strict;
use warnings;

use List::Util qw/max/;

use Yandex::HashUtils;
use Yandex::I18n;

use Settings;
use API::Settings;

require Exporter;

use base qw/Exporter/;

our @EXPORT_OK = qw(
        is_autobudget
        strategy_sets_from_API_v4_format
        strategy_sets_from_API_format
        context_strategy_sets_from_API_format
        prepare_context_strategy_sets
);

use utf8;


# названия стратегий в API - к названиям стратегий из функции detect_strategy
our %CONVERT_STRATEGY_NAMES_HASH = (
            default => 'HighestPosition',
            autobudget => 'WeeklyBudget',
            autobudget_week_bundle => 'WeeklyPacketOfClicks',
            autobudget_avg_click => 'AverageClickPrice',
            autobudget_avg_cpa => 'AverageCPAOptimization',
            autobudget_crr => 'CRROptimization',
            autobudget_roi => 'ROIOptimization',
            strategy_no_premium_highest => 'RightBlockHighest',
            stop => 'ShowsDisabled',
            cpa_optimizer => 'CPAOptimizer',

            # TODO: заглушки для поддержки версий API до v5
            different_places => 'IndependentControl',
            remove_in_api_v5 => 'NoPremiumPosition'
        );

# названия стратегий для сети в API - к названиям стратегий из функции detect_context_strategy
our %CONVERT_CONTEXT_STRATEGY_NAMES_HASH = (
            default => 'Default',
            stop => 'ShowsDisabled',
            maximum_coverage => 'MaximumCoverage',

            cpa_optimizer => 'CPAOptimizer',
            autobudget => 'WeeklyBudget',
            autobudget_week_bundle => 'WeeklyPacketOfClicks',
            autobudget_avg_click => 'AverageClickPrice',
            autobudget_avg_cpa => 'AverageCPAOptimization',
            autobudget_crr => 'CRROptimization',
            autobudget_roi => 'ROIOptimization',
        );

our %STRATEGIES_AVAILABLE_WITH_CONTEXT = (
            HighestPosition => 1,
            LowestCostPremium => 1,
            LowestCost => 1,
            RightBlockHighest => 1,
            LowestCostGuarantee => 1,
            ShowsDisabled => 1,
        );

=pod
    функция convert_params(campaign => $params, 1)
    создает хеш с параметрами кампаний которые нужны ядру
    здесь описаны только  те из них, которые относятся к стратегиям
=cut
our %STRATEGIES_FIELDS = (
            autobudget => 1,
            autobudget_bid => 1,
            autobudget_sum => 1,
            autobudget_limit_clicks => 1,
            autobudget_goal_id => 1,
            autobudget_avg_bid => 1,
            autobudget_avg_cpa => 1,
            autobudget_crr => 1,
            autobudget_roi_coef => 1,
            autobudget_reserve_return => 1,
            autobudget_profitability => 1,
            ContextLimit => 1,
            ContextPriceCoef => 1,
            strategy_no_premium => 1,
            strategy => 1,
            is_search_stop => 1,

        );

our %AUTOBUDGET_STRATEGIES = (
            WeeklyBudget => 1,
            WeeklyPacketOfClicks => 1,
            AverageClickPrice => 1,
            CPAOptimizer => 1,
            AverageCPAOptimization => 1,
            CRROptimization => 1,
            ROIOptimization => 1
        );

=head2 is_autobudget(strategy_name)

    Проверяет является ли стратегия автобюджетной
    strategy_name название стратегии в API!

=cut

sub is_autobudget($){
    my $strategy_name = shift;
    if ($AUTOBUDGET_STRATEGIES{$strategy_name}) {
        return 1;
    } else {
        return 0;
    }
}

our %strategies_with_BID = (
    WeeklyBudget => 1,
    CPAOptimizer => 1,
    WeeklyPacketOfClicks => 1,
    AverageCPAOptimization => 1,
    ROIOptimization => 1
);

our %strategies_with_AVG_BID = (
    WeeklyPacketOfClicks => 1,
    AverageClickPrice => 1
);

my %strategies_with_SUM = (
    WeeklyBudget => 1,
    CPAOptimizer => 1,
    AverageClickPrice => 1,
    AverageCPAOptimization => 1,
    CRROptimization => 1,
    ROIOptimization => 1
);

my %strategies_with_GOAL_ID = (
    CPAOptimizer => 1,
    AverageCPAOptimization => 1,
    CRROptimization => 1,
    ROIOptimization => 1
);

=head2 strategy_sets_from_API_v4_format(api_params)

    Преобразует настройки стратегий из формата API v4 во внутренний формат 

=cut

sub strategy_sets_from_API_v4_format{
    my $api_params = shift;
    if (exists $api_params->{Strategy}) {
        my $strategy = $api_params->{Strategy};
    
        $api_params->{autobudget} = 'No';
        if ( $strategy->{StrategyName}) {
            $api_params->{autobudget} = is_autobudget($strategy->{StrategyName}) ? 'Yes' : 'No';
            $api_params->{autobudget_bid} = $strategy->{MaxPrice}
                if $strategies_with_BID{$strategy->{StrategyName}};
                
            $api_params->{autobudget_sum} = $strategy->{WeeklySumLimit}
                if $strategies_with_SUM{$strategy->{StrategyName}};
                
            $api_params->{autobudget_avg_bid} = $strategy->{AveragePrice}
                if $strategies_with_AVG_BID{$strategy->{StrategyName}};

            $api_params->{autobudget_limit_clicks} = $strategy->{ClicksPerWeek}
                if $strategy->{StrategyName} eq 'WeeklyPacketOfClicks';

            # TODO: убрать нижеследующие заглушки для поддержки версий API до v5
            $api_params->{strategy_no_premium} = 'in APIv5'
                if $strategy->{StrategyName} eq 'NoPremiumPosition';
            $api_params->{strategy} = 'different_places'
                if $strategy->{StrategyName} eq 'IndependentControl';
        }
    }
}

=head2 strategy_sets_from_API_format(api_params)

    Преобразует настройки стратегий из формата API во внутренний формат 

=cut

sub strategy_sets_from_API_format{
    my $api_params = shift;
    if (exists $api_params->{Strategy}) {
        my $strategy = $api_params->{Strategy};
        $api_params->{autobudget} = 'No';
        if ( $strategy->{StrategyName}) {
            $api_params->{autobudget} = is_autobudget($strategy->{StrategyName}) ? 'Yes' : 'No';
            if ($api_params->{autobudget} eq 'Yes') {
                my $autobudget_sets = _autobudget_sets_from_API_format($strategy);
                hash_merge $api_params, $autobudget_sets;
            }      
            $api_params->{is_search_stop} = $strategy->{StrategyName} eq 'ShowsDisabled' ? 1 : 0;

            $api_params->{strategy_no_premium} = 'highest_place'
                if $strategy->{StrategyName} eq 'RightBlockHighest';
        }
    }
}

=head2 context_strategy_sets_from_API_format(api_params)

    Преобразует настройки контекстных стратегий из формата API во внутренний формат 

=cut

sub context_strategy_sets_from_API_format{
    my $api_params = shift;
    
    if ($api_params->{ContextStrategy}){
        my $strategy = $api_params->{ContextStrategy};
        if ($strategy->{StrategyName} eq 'Default') {
            $api_params->{ContextPriceCoef} = $strategy->{ContextPricePercent} || 100;
            $api_params->{dontShowYacontext} = 'No';

            my $value = 0;
            if(defined $strategy->{ContextLimit}){
                if($strategy->{ContextLimit} eq 'Unlimited'){
                    $value = 255;
                } elsif($strategy->{ContextLimit} eq 'Limited'){
                    $value = $strategy->{ContextLimitSum};
                    $api_params->{dontShowYacontext} = 'Yes'  if $value == 0; 
                } elsif($strategy->{ContextLimit} eq 'Default'){
                    $value = 0;
                }
            }
            $api_params->{ContextLimit} = $value;
        } elsif ($strategy->{StrategyName} eq 'ShowsDisabled') {
            $api_params->{ContextLimit} = 0;
            $api_params->{dontShowYacontext} = 'Yes';
            $api_params->{ContextPriceCoef} = 100;
        } elsif ($strategy->{StrategyName} eq 'MaximumCoverage') {
            $api_params->{strategy} = 'different_places';
            $api_params->{ContextLimit} = 0;
            $api_params->{ContextPriceCoef} = 100;
        } else {
            $api_params->{strategy} = 'different_places';
            $api_params->{autobudget} = is_autobudget($strategy->{StrategyName}) ? 'Yes' : 'No';
            if ($api_params->{autobudget} eq 'Yes') {
                my $autobudget_sets = _autobudget_sets_from_API_format($strategy);
                hash_merge $api_params, $autobudget_sets;
            } 
        }
    } 
}

=head2 _autobudget_sets_from_API_format(strategy)

    Преобразует настройки автобюджетных стратегий из формата API во внутренний формат 

=cut

sub _autobudget_sets_from_API_format($){
    my $strategy = shift;
    my $sets;
    $sets->{autobudget_bid} = $strategy->{MaxPrice}
        if $strategies_with_BID{$strategy->{StrategyName}};
    $sets->{autobudget_sum} = $strategy->{WeeklySumLimit}
        if $strategies_with_SUM{$strategy->{StrategyName}};
    $sets->{autobudget_avg_bid} = $strategy->{AveragePrice}
        if $strategy->{StrategyName} =~ /^(WeeklyPacketOfClicks|AverageClickPrice)$/;
    $sets->{autobudget_avg_cpa} = $strategy->{AverageCPA}
        if $strategy->{StrategyName} eq 'AverageCPAOptimization';
    $sets->{autobudget_limit_clicks} = $strategy->{ClicksPerWeek}
        if $strategy->{StrategyName} =~ /^(?:WeeklyPacketOfClicks)$/x;
    $sets->{autobudget_goal_id} = $strategy->{GoalID}
        if $strategies_with_GOAL_ID{$strategy->{StrategyName}};
    $sets->{autobudget_crr} = $strategy->{Crr}
        if $strategy->{StrategyName} eq 'CRROptimization';
    if ($strategy->{StrategyName} eq 'ROIOptimization') {
        $sets->{autobudget_reserve_return} = $strategy->{ReserveReturn};
        $sets->{autobudget_profitability} = $strategy->{Profitability};
        $sets->{autobudget_roi_coef} = $strategy->{ROICoef};
    }
    return $sets;
}

1;
