package API::Validate;

=pod

    $Id$

    Модуль для проверки правильности входных данных от пользователя
    Авторы:
        Vasiliy Bryadov <mirage@yandex-team.ru>
        Alexey Ziyangirov <metallic@yandex-team.ru>

=cut

use Direct::Modern;

use Date::Calc qw/Delta_Days/;

use Carp;
use Settings;
use Yandex::DBTools;
use Yandex::DBShards;

use API::Settings;
use API::Filter;
use API::ValidateTools qw/_check_array _check_fields_exist is_yes_no is_dec_percent/;
use API::Validate::Ids qw/validate_id validate_ids_detail/;
use API::Validate::Structure;

use RBACElementary;
use RBACDirect;

use API::ReportCommon;
use API::ValidateRights;
use APICommon qw(:subs);

use List::MoreUtils qw(any none uniq);
use MinusWords;
use Primitives;
use PrimitivesIds;
use Currencies;
use Sitelinks;
use EnvTools;
use TextTools;
use Tools;
use GeoTools;
use Campaign;
use Campaign::Types;
use Client;
use Common qw//;
use Phrase qw//;
use Primitives qw//;
use Metro qw/validate_metro/;
use API::Errors;
use Tag;
use TextTools qw//;
use Retargeting;
use Agency;
use BannersCommon;
use vars qw($VERSION @ISA @EXPORT);

use Direct::Validation::Keywords qw/validate_keywords_for_api_forecasts base_validate_keywords/;
use Direct::Validation::SitelinksSets qw//;
use Direct::Validation::HierarchicalMultipliers qw//;
use Direct::Validation::MinusWords qw/validate_keyword_minus_words/;

use Yandex::TimeCommon;
use Yandex::ListUtils;
use Yandex::DateTime;
use Yandex::I18n;
use Yandex::Balance;

use API::Methods::Retargeting;
use API::Methods::AdImage;
use API::Methods::Keyword;

use API::Limits qw/has_spec_limits get_spec_limit get_clients_with_limits/;

use Stat::Const ();

use base qw(Exporter);
@EXPORT = qw(
    validate_params
);

my $SECONDS_IN_A_DAY = 86400;

my %cmd_checks = (

    'ModerateBanners' => [\&_validate_moderate_banners],

    'GetKeywordsSuggestion' => [\&_validate_keywords_suggestion],

    'CreateNewWordstatReport' => [\&_validate_create_new_wordstat_report],
    'GetWordstatReport' => [\&_validate_get_wordstat_report],
    'GetWordstatReportList' => [\&_validate_get_wordstat_report_list],
    'DeleteWordstatReport' => [\&_validate_delete_wordstat_report],
    'GetWordstatSync' => [
                          \&_validate_get_wordstat_sync_access,
                          \&_validate_get_wordstat_sync,
                          \&_validate_get_wordstat_sync_limits
                        ],
    'GetForecastSync' => [
                          \&_validate_get_forecast_sync_access,
                          \&_validate_create_new_forecast,
                          \&_validate_extended_forecast,
                        ],
    'GetStatGoals' => [\&_validate_get_stat_goals],
    'CreateNewSubclient' => [\&_validate_create_new_subclient],

    'CreateNewForecast' => [\&_validate_create_new_forecast],
    'GetForecast' => [\&_validate_get_forecast],
    'GetForecastList' => [\&_validate_get_forecast_list],
    'DeleteForecastReport' => [\&_validate_delete_forecast_report],

    'CreateNewReport' => [\&_validate_create_new_report, \&_validate_campaign_request_currency],
    'DeleteReport' => [\&_validate_delete_report],
    'GetBannersStat' => [\&_validate_get_banners_stat, \&_validate_campaign_request_currency],

    'GetClientsUnits' => [\&_validate_get_clients_units],
    'GetClientInfo' => [\&_validate_get_client_info],
    'UpdateClientInfo' => [\&_validate_update_client_info],
    'TransferMoney' => [\&_validate_transfer_money],
    'CreateInvoice' => [\&_validate_create_invoice],
    'PayCampaigns' => [\&_validate_pay_campaigns],
    'PayCampaignsByCard' => [\&_validate_pay_campaigns_by_card],
    'CheckPayment' => [\&_validate_check_payment],
    'GetBalance' => [\&_validate_get_balance],
    'GetSubClients' => [\&_validate_get_subclients],
    'GetClientsList' => [\&_validate_get_clients_list],
    'SetAutoPrice' => [\&API::Methods::Prices::validate_set_auto_price],
    'GetSummaryStat' => [\&_validate_get_summary_stat, \&_validate_campaign_request_currency],
    'GetEventsLog' => [\&_validate_get_events_log],

    'GetCampaignsTags' => [\&_validate_get_campaigns_tags],
    'UpdateCampaignsTags' => [\&_validate_update_campaigns_tags],
    'GetBannersTags' => [\&_validate_get_banners_tags],
    'UpdateBannersTags' => [\&_validate_update_banners_tags],

    'GetMetroStations' => [\&_validate_get_metro_stations],

    'SaveSubscription' => [\&_validate_save_subscription],
    'DeleteSubscription' => [\&_validate_subscription_base_params],
    'GetSubscription' => [\&_validate_subscription_base_params],

    'AddEvents' => [\&_validate_add_events],

    'SearchClients' => [\&_validate_search_clients],
    'GetNormalizedKeywords' => [\&_validate_get_normalized_keywords],
    'GetNormalizedKeywordsData' => [\&_validate_get_normalized_keywords],
    'GetKeywordsIntersection' => [
            \&_validate_get_keywords_intersection,
            \&API::ValidateRights::validate_get_keywords_intersection_rights
        ],

    'RearrangeKeywords' => [
              \&API::Methods::Keyword::_validate_rearrange_keywords_rights
            , \&API::Methods::Keyword::_validate_rearrange_keywords
        ],

    'MediaplanKeyword' => [
                    \&API::ValidateRights::validate_mediaplan_allow,
                    \&_validate_mediaplan_keyword_wrapper,
                    \&API::ValidateRights::validate_mediaplan_limited_rights,
                    \&API::ValidateRights::validate_mediaplan_keyword_rights
                    ],
    'MediaplanCategory' => [
                    \&API::ValidateRights::validate_mediaplan_allow,
                    \&API::ValidateRights::validate_mediaplan_limited_rights,
                    ],
    'MediaplanAd' => [
                    \&API::ValidateRights::validate_mediaplan_allow,
                    \&_validate_mediaplan_ad_wrapper,
                    \&API::ValidateRights::validate_mediaplan_limited_rights,
                    \&API::ValidateRights::validate_mediaplan_ad_rights,
                    ],
    'Mediaplan' => [
                    \&API::ValidateRights::validate_mediaplan_allow,
                    \&_validate_mediaplan_wrapper,
                    \&API::ValidateRights::validate_mediaplan_limited_rights,
                    \&API::ValidateRights::validate_mediaplan_rights,
                    ],
    'GetRetargetingGoals' => [
                    \&API::Methods::Retargeting::validate_get_retargeting_goals_rights,
                    ],
    'RetargetingCondition' => [
                    \&API::Methods::Retargeting::preprocess_retargeting_condition,
                    \&API::Methods::Retargeting::validate_retargeting_condition_rights,
                    \&API::Methods::Retargeting::validate_retargeting_condition,
                    ],
    'Retargeting' => [
                    \&API::Methods::Retargeting::pre_validate_retargeting,
                    \&API::Methods::Retargeting::preprocess_retargeting,
                    \&API::Methods::Retargeting::validate_retargeting_rights,
                    \&API::Methods::Retargeting::validate_retargeting,
    ],
    'AdImage' => [
                    \&API::Methods::AdImage::preprocess_adimage,
                    \&API::Methods::AdImage::validate_adimage_rights,
                    \&API::Methods::AdImage::validate_adimage,
    ],
    'AdImageAssociation' => [
                    \&API::Methods::AdImage::preprocess_adimageassociation,
                    \&API::Methods::AdImage::validate_adimageassociation_rights,
    ],

    'EnableSharedAccount' => [
        \&API::ValidateRights::validate_enable_shared_account,
    ],

    'AccountManagement' => [
        \&API::Methods::AccountManagement::validate_account_management,
    ]

);

my %campaigns_list_filter_values = (
    StatusModerate => {
                        New => 1,
                        Yes => 1,
                        No => 1,
                        Pending => 1,
                    },
    StatusActivating => {
                        Yes => 1,
                        Pending => 1,
                    },
    StatusShow => {
                        Yes => 1,
                        No => 1,
                    },
    IsActive => {
                        Yes => 1,
                        No => 1,
                    },
    StatusArchive => {
                            Yes => 1,
                            No => 1,
                            Pending => 1,
                            CurrencyConverted => 1
                     },
);

=head2 validate_params($$$;%)

     Валидирует запрос к методу по офисаннной в модуле API::Validate::Structure структуре INPUT_STRUCTURES

=cut

sub validate_params($$$;%)
{
    my ($self, $method, $params, %O) = @_;

    my @validate_structure_errors = API::Validate::Structure::api_validate_structure($self, $params, $method);

    if (@validate_structure_errors) {
        return @{$validate_structure_errors[0]};
    }

    if ($method && defined $cmd_checks{$method}) {

        my $validate_functions;

        if (ref $cmd_checks{$method} eq 'ARRAY') {
            $validate_functions = $cmd_checks{$method};
        } elsif (ref $cmd_checks{$method} eq 'CODE') {
            $validate_functions = [$cmd_checks{$method}];
        } else {
            foreach my $version (keys %{$cmd_checks{$method}}) {
                if ($version eq $self->{api_version} || $self->{api_version} > $version && $cmd_checks{$method}{$version}{more}) {
                    $validate_functions = $cmd_checks{$method}{$version}{sub};
                }
            }
        }
        if (ref $validate_functions eq 'ARRAY') {
            foreach my $val_func (@$validate_functions) {
                my @validation_result = $val_func->($self, $params);
                return @validation_result if scalar @validation_result;
            }
        }
    }

    if ($O{not_die_on_no_validate}) {
        return; #ok
    } else {
        die "Unknown validate check for method!";
    }
}

sub _validate_campaigns_currency
{
    my ($self, $params, $camp_currencies) = @_;

    # multicurrency
    if ($self->{api_version_full} > 4) {

        if (scalar keys %$camp_currencies > 1) {
            return ('BadCurrency', iget('Недопустим запрос для кампаний в разных валютах'));
        }

        delete $camp_currencies->{YND_FIXED} if exists $camp_currencies->{YND_FIXED};

        my @allowed_currencies = keys %$camp_currencies;

        if (!defined $params->{Currency} && @allowed_currencies && !$self->{app_has_access_override}) { # не задана валюта для запроса по кампаниям в валюте и нет специального разрешения у приложения
            return ('BadCurrency');
        }

        if (my @fields_exist_errors = _check_fields_exist($params, [qw/Currency/], type => 'currency', list => [shift @allowed_currencies])) {
            return (@fields_exist_errors);
        }
    }

    return;
}

=head3 _validate_keywords_suggestion

    Метод GetKeywordsSuggestion
    На вход:
              обязательно массив ключевых слов,
              необязательно: Iteration, NumberOfKeywords

=cut
sub _validate_keywords_suggestion
{
    my ($self, $params) = @_;

    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                               && get_spec_limit($self->{ClientID}, 'keywords_suggestion_disallow');

    my $MAX_KEYWORDS_PER_QUERY = 100;
    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Keywords/], def => 1)){
        return @fields_exist_errors;
    }

    my $keywords = $params->{Keywords};

    if (my @array_errors = _check_array($keywords, 'Keywords', max => $MAX_KEYWORDS_PER_QUERY, type => 'string')){
        return @array_errors;
    }

    # удаляем пустые строки и 0 элементы
    @{$keywords} = grep {$_} @{$keywords};
    my $validation_result = base_validate_keywords($keywords);
    unless ($validation_result->is_valid) {
        my $errors = join '. ', @{$validation_result->one_error_description_by_objects};
        return ('BadParams', $errors);
    } else {
        return; #ok
    }
}

sub _validate_get_keywords_intersection
{
    my ($self, $params) = @_;
    my $MAX_KEYWORDS_PER_QUERY = 1000;
    my $MAX_CAMPAIGNS_PER_QUERY = 1000;
    return ('BadRequest') if (ref($params) ne 'HASH');

    if (defined $params->{Keywords} && defined $params->{CampaignIDS}) {
        return ('BadParams', iget('Только одно поле из %s может быть указано', join ', ', 'Keywords', 'CampaignIDS'));
    }

    if (defined $params->{WithKeywords} && defined $params->{WithCampaignIDS}) {
        return ('BadParams', iget('Только одно поле из %s может быть указано', join ', ', 'WithKeywords', 'WithCampaignIDS'));
    }

    for my $field (qw/Keywords WithKeywords/) {
        if (defined $params->{$field}) {
            if (my @array_errors = _check_array($params->{$field}, $field, max => $MAX_KEYWORDS_PER_QUERY, not_empty => 1)){
                return @array_errors;
            }
        }
    }
    unless (defined $params->{Keywords} || defined $params->{CampaignIDS}) {
        return ('BadParams', iget('Одно поле из %s должно быть указано', join ', ', 'Keywords', 'CampaignIDS'));
    }

    for my $field (qw/CampaignIDS WithCampaignIDS/) {
        if (defined $params->{$field}) {
            if (my @array_errors = _check_array($params->{$field}, $field, max => $MAX_KEYWORDS_PER_QUERY, not_empty => 1, type => 'campaignid')){
                return @array_errors;
            }
        }
    }

    return;
}

=head3 _validate_create_new_wordstat_report(self, params, options)

    Метод CreateNewWordstatReport
    На вход:
              обязательно: массив фраз
              необязательно: массив geo

=cut

sub _validate_create_new_wordstat_report($$;%)
{
    my ($self, $params, %O) = @_;
    $O{max_keywords} = $Settings::API_MAX_KEYWORDS_PER_ADVQ_QUERY unless $O{max_keywords};
    return ('BadRequest') if (ref($params) ne 'HASH');

    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                               && get_spec_limit($self->{ClientID}, 'wordstat_disallow');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Phrases/], def => 1)){
        return @fields_exist_errors;
    }

    if (defined $params->{GeoID}) {

        if (my @array_errors = _check_array($params->{GeoID}, 'GeoID', type => 'int')){
            return @array_errors;
        }

    }

    if ( $params->{Phrases} ){

        if (my @array_errors = _check_array($params->{Phrases}, 'Phrases')){
            return @array_errors;
        }

        # удаляем пустые строки, undef и 0 элементы
        @{$params->{Phrases}} = grep {$_} @{$params->{Phrases}};

        if (my @array_errors = _check_array($params->{Phrases}, 'Phrases', not_empty => 1, max => $O{max_keywords}, type => 'string')){
            return @array_errors;
        }

        my $keywords_vr = validate_keywords_for_api_forecasts($params->{Phrases});
        unless ($keywords_vr->is_valid) {
            my $errors = join '. ', @{$keywords_vr->one_error_description_by_objects};
            return ('BadParams', $errors);
        } else {
            return; #ok
        }
    }
}

sub _validate_get_forecast_sync_access {
    my ($self, $params) = @_;
    # разрешаем доступ внутренним пользвателяи и приложению Я.Услуги
    unless ($self->{rbac_login_rights}{is_internal_user} || $self->{application_id} eq $API::Settings::YNDX_SERVICES_APP_ID) {
        return ('NoRights');
    }
    return;
}

sub _validate_get_wordstat_sync_access {
    my ($self, $params) = @_;
    my $limit_resource_key = $API::Settings::GET_WORDSTAT_SYNC_LIMIT_KEY;
    unless ($self->{rbac_login_rights}{is_internal_user} || $self->{special_options}->{method_limits}->{$limit_resource_key}) {
        return ('AccessDenied');
    }
    return;
}

=head3 _validate_get_wordstat_sync(self, params)

    Метод GetWordstatSync
    На вход:
              обязательно: массив фраз
              необязательно: массив geo

=cut
sub _validate_get_wordstat_sync($$) {
    my ($self, $params) = @_;

    local $Settings::MAX_PHRASE_LENGTH = 30_000;
    local $Settings::GROUP_MINUS_WORDS_LIMIT = 30_000;

    return _validate_create_new_wordstat_report($self, $params, max_keywords => 100);
}

sub _validate_get_wordstat_sync_limits {
    my ($self, $params) = @_;
    return if $self->{rbac_login_rights}{is_internal_user};

    my $limit_resource_key = $API::Settings::GET_WORDSTAT_SYNC_LIMIT_KEY;
    my $limit = $self->{special_options}->{method_limits}->{$limit_resource_key};
    my $limit_claimer_key = "CLIENT_ID" . $self->{ClientID};

    api_check_limit($self, $limit_claimer_key, $limit_resource_key, $limit, 1, obj_name => $self->{operator_login});
    api_update_limit( $self, $limit_claimer_key, $limit_resource_key, 1);

    return;
}

=head3 _validate_get_wordstat_report

    Метод GetWordstatReport

    Должен передаваться один единственный параметр ReportID типа int > 0

=cut
sub _validate_get_wordstat_report {

    my ($self, $params) = @_;

    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID},'wordstat_disallow');

    if (!$params || $params !~ /^\d+$/) {
        return ('BadWordstatReportID');
    }

    return; #ok
}

sub _validate_get_wordstat_report_list
{
    my ($self, $params) = @_;

    # кроме пользователей из списка
    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID}, 'wordstat_disallow');

    return;
}

sub _validate_delete_wordstat_report
{
    my ($self, $params) = @_;

    # кроме пользователей из списка
    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID} ,'wordstat_disallow');

    return ('BadReportID') unless $params && $params =~ /^\d+$/;

    my $report = get_queue_report_db('wordstat', $params, $self->{uid});

    return ('NoReport') unless $report && $report->{status} ne 'Deleted';

    return;
}

=head3 _validate_get_stat_goals_common(self, CampaignIDS)

    общая часть проверки в _validate_get_stat_goals для разных типов входных данных

=cut

sub _validate_get_stat_goals_common($$) {
    my ($self, $cids) = @_;
    my $existing_camps=get_hashes_hash_sql(PPC(cid => $cids), ['select cid from campaigns', where => {cid__int => SHARD_IDS, statusEmpty => 'No'}]);
    foreach my $cid (@{$cids}) {
        unless ( validate_id( {cid => $cid, check_exist => 1} ) ) {
            return ('BadCampaignID');
        }
        unless (exists $existing_camps->{$cid}) {
            return ('BadCampaignID', iget('Кампания не существует'));
        }
    }
    my $error_code = check_rbac_rights( $self, 'api_showCampStat', { stat_type => 'custom', cid => join(',', @$cids), UID => $self->{uid} } );
    return ($error_code? 'NoRights' : ());
}

=head3 _validate_get_stat_goals

    Метод GetStatGoals

    Принимает хэш с одним элементом CampaignID
    Альтернативно принимает хеш с одним элементом - CampaignIDS, ссылающимся на массив (v4.5+)

=cut

sub _validate_get_stat_goals
{
    my ($self, $params) = @_;
    my $STAT_GOALS_MAX = 100;
    return ('BadRequest') if (ref($params) ne 'HASH');
    if ($self->{api_version_full} >= 5) {
        return ('BadRequest', iget('Поле %s должно быть указано', 'CampaignIDS')) unless defined $params->{CampaignIDS};
    }
    if ($self->{api_version_full} > 4) {
        if (defined $params->{CampaignIDS}) {
            if (my @array_errors = _check_array($params->{CampaignIDS}, 'CampaignIDS', not_empty => 1, type => 'int', max => $STAT_GOALS_MAX)) {
                return @array_errors;
            }
            return _validate_get_stat_goals_common($self, $params->{CampaignIDS});
        }
    }
    unless (defined $params->{CampaignID}) {
        return ('BadParams', iget('Поле %s должно быть указано', 'CampaignID')) if ($self->{api_version_full } == 4);
        return ('BadParams', iget('Одно из полей CampaignIDS или CampaignID должно быть указано'));
    }
    if (!validate_id({cid => $params->{CampaignID}})) {
        return ('BadCampaignID');
    }

    return _validate_get_stat_goals_common($self, [$params->{CampaignID}]);
}

=head3 _validate_create_new_subclient

    Метод CreateNewSubclient
    На вход:
        хэш из 4 элементов:
        Login - желаемый логин
        Name
        Surname
        AgencyLogin - логин агентства, обязателен для менеджера,
            опционален для агентства, но должен совпадать с логином последнего

=cut

sub _validate_create_new_subclient
{
    my ($self, $params) = @_;

    my $MAX_FIO_LENGTH = 20;

    return ('BadRequest') if (ref($params) ne 'HASH');

    my $chief_uid = $self->{uid};

    if ($self->{rbac_login_rights}->{role} eq 'agency') {
        $chief_uid = rbac_get_chief_rep_of_agency_rep($self->{uid});
    } elsif ($self->{rbac_login_rights}->{role} eq 'client') {
        $chief_uid = rbac_get_chief_rep_of_client_rep($self->{uid});
    }

    my $allow_create_subclients = get_one_field_sql(PPC(uid => $chief_uid), "select allow_create_subclients from users_api_options where uid = ?", $chief_uid);

    unless ($allow_create_subclients eq 'Yes') {
        return ('NoRights', iget('Нет разрешения создавать клиентов'));
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Login/], def => 1)){
        return @fields_exist_errors;
    }

    my $login_not_exists_flag = 1;
    if ($self->{rbac_login_rights}->{manager_control}) {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/LoginExists/], type => 'yesno')){
            return @fields_exist_errors;
        }

        $login_not_exists_flag = 0 if $params->{LoginExists} && $params->{LoginExists} =~ /^yes$/i;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Name Surname/], def => ($self->{rbac_login_rights}->{manager_control} && (($params->{LoginExists} || '') eq 'Yes' ? 0 : 1)), type => 'string')){
        return @fields_exist_errors;
    }

    if ( !validate_login( $params->{Login}, lite => ($login_not_exists_flag ? 0 : 1) ) ) {
        return ('BadLogin', iget('Поле %s может содержать только латинские буквы, цифры, одинарный дефис или точку, первый символ должен быть буквой, последний буквой или цифрой, длина не должна превышать 30 символов', 'Login'));
    }

    if (defined $params->{Name} && defined $params->{Surname}) {
        foreach (qw/Name Surname/) {

            if ($params->{$_} =~ m/[&=<>]/) {
                return ('BadParams', iget("Поле %s не должно содержать символы: &=<>", $_));
            }

            $params->{$_} =~ s/^\s+//;
            $params->{$_} =~ s/\s+$//;

            if ( length($params->{$_}) > $MAX_FIO_LENGTH) {
                return ('BadParams', iget("Длина поля %s должна быть не более %s символов", $_, $MAX_FIO_LENGTH));
            }

            if ( length($params->{$_}) == 0 ) {
                return ('BadParams', iget("Поле %s не должно быть пустой строкой", $_));
            }
        }
    }

    if (defined $params->{AgencyLogin} && !validate_login($params->{AgencyLogin}) ) {
        return ('BadLogin', iget('Поле %s может содержать только латинские буквы, цифры, одинарный дефис или точку, первый символ должен быть буквой, последний буквой или цифрой, длина не должна превышать 30 символов', 'AgencyLogin'));
    }

    if ($self->{rbac_login_rights}->{role} eq 'manager') {

        if (defined $params->{ServicedClient}) {
            return ('NotYesNo', iget('Поле %s', 'ServicedClient')) if ! is_yes_no($params->{ServicedClient});
        }

        # -- Если менеджер может управлять гео-кампаниями
        if ((is_api_geo_allowed($self)||'No') eq 'Yes') {
            # -- Создает сервисируемого собой клиента (ServicedClient=Yes)
            if (defined $params->{ServicedClient} && $params->{ServicedClient} eq 'Yes') {
                if (! $login_not_exists_flag) {
                    my $login = TextTools::normalize_login($params->{Login});
                    # зарегистрирован ли в Паспорте такой пользователь
                    my $uid = Primitives::get_uid_by_login($login);
                    if (! $uid) {
                        return ('BadLogin', iget('Логин не существует'));
                    }
                    # существует ли в Балансе клиент, чьим представителем является пользователь
                    my $client_id = Yandex::Balance::get_clientid_by_uid($uid);
                    if ($client_id) {
                        my $existing = Common::get_country_and_currency_from_balance($client_id);
                        # у клиента в Балансе ровно одна валюта
                        if (exists $existing->{currency}) {
                            if ( defined($params->{Currency}) && ($params->{Currency} ne $existing->{currency}) ) {
                                return ('CantCreateClient', iget('Клиент уже использует другую валюту'));
                            }
                            $params->{Currency} = $existing->{currency};
                        }
                    }
                }
                $params->{Currency} ||= 'YND_FIXED';
            }
            # если не задан ServicedClient или он в 'No', то проверяем поля, касающиеся агентства
            elsif (! defined $params->{ServicedClient} || $params->{ServicedClient} eq 'No') {

                if (my @fields_exist_errors = _check_fields_exist($params, [qw/AgencyLogin/], def => 1)){
                    return @fields_exist_errors;
                }

                my $agency_uid = get_uid_by_login($params->{AgencyLogin});

                if (! $agency_uid) {
                    return ('AgencyNotExist');
                } else {
                    my $auid_detail = rbac_who_is_detailed($self->{rbac}, $agency_uid);
                    if ($auid_detail->{role} ne 'agency') {
                        return ('BadParams', iget('Поле %s содержит неверный логин агентства', 'AgencyLogin'));
                    }
                    my $uids = rbac_get_all_reps_agencies_of_manager($self->{rbac}, $self->{uid});
                    my %uids_hash = map {$_ => undef} @{$uids};
                    if (! exists $uids_hash{$agency_uid}){
                        return ('NoRights', iget('У вас нет прав на указанное агентство'));
                    }
                }

                # -- проверять валюту нужно именно у агенства, а не у менеджера
                my $agency_client_id = rbac_get_agency_clientid_by_uid( $agency_uid);

                # -- ac = agency currency
                my $ac = _check_agency_currency($self->{rbac}, $params->{Currency}, $agency_client_id, $self->{api_version_full}, 'Yes');
                if (defined $ac->{errors}) {
                    if (ref($ac->{errors}) eq 'ARRAY'){
                        return @{$ac->{errors}};
                    }
                    return ('BadParams', $ac->{errors});
                } else {
                    $params->{Currency} = $ac->{Currency};
                }
            }

        # -- обычным менеджерам запрещено создавать субклиентов
        } else{
            return ('NoRights', iget('У Вас нет прав для выполнения данной операции'));
        }

    } elsif ($self->{rbac_login_rights}->{role} eq 'agency' ) {

        if (defined $params->{AgencyLogin}) {
            my $agency_chief_uid = rbac_get_chief_rep_of_agency_rep($self->{uid});
            my $agency_uid = get_uid_by_login($params->{AgencyLogin});

            if (! $agency_uid) {
                return ('AgencyNotExist');
            } else {

                my $auid_detail = rbac_who_is_detailed($self->{rbac}, $agency_uid);
                if ($auid_detail->{role} ne 'agency') {
                    return ('BadParams', iget('Поле %s содержит неверный логин агентства', 'AgencyLogin'));
                }

                if ( ! $agency_uid || $self->{uid} != $agency_uid && $agency_chief_uid != $agency_uid ) {
                    return ('NoRights', iget('У указанного агентства нет прав для данной операции'));
                }
            }
        }

        my $agency_client_id = rbac_get_agency_clientid_by_uid( $self->{uid});

        # -- ac = agency currency
        my $ac = _check_agency_currency($self->{rbac}, $params->{Currency}, $agency_client_id, $self->{api_version_full}, 'No');
        if (defined $ac->{errors}) {
            if (ref($ac->{errors}) eq 'ARRAY'){
                return @{$ac->{errors}};
            }
            return ('BadParams', $ac->{errors});
        } else {
            $params->{Currency} = $ac->{Currency};
        }
    } else {
        return ( 'NoRights', iget('Только агентствам и менеджерам разрешено создавать новые логины' ));
    }

    return; #ok
}

=head3 _validate_create_new_report(self, report_options)

    проверяет валидность данных пользователя для создания отчета

=cut

sub _validate_create_new_report
{
    my ($self, $params, %O) = @_;
    return ('BadRequest') if ! $params || ref($params) ne 'HASH' ;

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/StartDate EndDate/], type => 'date_no_future', def => 1 )){
        return @fields_exist_errors;
    }

    # is array
    foreach my $field (qw/GroupByColumns OrderBy/) {
        return ('NotArray', iget('Поле %s', $field)) if defined $params->{$field} && ref $params->{$field} ne 'ARRAY';
    }

    #campaignID
    return ('BadCampaignID') if !defined $params->{CampaignID} || !validate_id( { cid => $params->{CampaignID}, check_exist => 1 } );

    my $error_code = check_rbac_rights($self, 'api_showCampStat', { stat_type => 'custom', cid => $params->{CampaignID}, UID => $self->{uid}});
    return ('NoRights') if $error_code;

    my $camp = get_camp_info($params->{CampaignID}, undef, short => 1);

    return ('NoStat') if !$camp->{OrderID};

    if ((any {lc($_) eq 'clcarriertype' || lc($_) eq 'clmobileplatform'} @{$params->{GroupByColumns}}, @{$params->{OrderBy}})
        && !$O{sync}) { # для GetBannersStat чуть позднее будет собрана другая ошибка
        return ('BadCampaignType', iget("Создание отчетов по типу мобильной операционной системы или по типу мобильной связи доступно только для мобильных кампаний")) unless $camp->{type} && $camp->{type} eq 'mobile_content';
    } elsif (0) {
        return ('NotSupported', iget("Создание отчетов для типа выбранной кампании не поддерживается")) unless (camp_kind_in(type => $camp->{type}, 'api_stat'));
    } else {
        return ('NotSupported', iget("Создание отчетов для типа выбранной кампании не поддерживается")) unless $camp->{type} && $camp->{type} =~ /^(text|geo|mobile_content|dynamic)$/;
    }

    if ( !camp_kind_in(type => $camp->{type}, 'allow_audience') && any { lc($_) eq 'cladjustment' } @{$params->{GroupByColumns}} ) {
        return ('BadCampaignType', iget("Создание отчетов по условиям подбора аудитории недоступно для кампаний данного типа"));
    }

    my $start_date =  mysql2unix($params->{StartDate});
    my $end_date = mysql2unix($params->{EndDate});

    return ('BadTimeInterval') if $end_date < $start_date;

    if (my @error = _validate_stat_within_3years($end_date)) {
        return @error;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/IncludeDiscount IncludeVAT/], type => 'yesno')){
        return @fields_exist_errors;
    }

    return ('BadParams', iget('Поле %s должно быть структурой', 'Filter')) if defined $params->{Filter} && ref $params->{Filter} ne 'HASH';

    if (!$O{sync} && defined $params->{Filter}) {

        # is array
        foreach my $field (qw/Banner Geo PageName Phrase StatGoals Age Gender/) {
            return ('NotArray', iget('Поле %s', $field)) if defined $params->{Filter}{$field} && ref $params->{Filter}{$field} ne 'ARRAY';
        }

        if ($params->{Filter}{StatGoals} && scalar @{$params->{Filter}{StatGoals}} > 1) {
            return ('LongMass', iget('Фильтр %s должен содержать не более одной цели', 'StatGoals'));
        }

        # is scalar
        foreach my $field (qw/PositionType PageType WithImage/) {
            return ('BadParams', iget('Поле %s должно содержать значение типа string', $field)) if defined $params->{$field} && ref $params->{$field};
        }

        # validate values filter
        return ('BadBannerFilter')  if defined $params->{Filter}->{Banner}   && grep{ !/^\d+$/ } @{ $params->{Filter}->{Banner} };
        return ('BadGeoFilter')     if defined $params->{Filter}->{Geo}      && grep{!/^\d+$/} @{ $params->{Filter}->{Geo}};

        return ('BadPageNameFilter')if defined $params->{Filter}->{PageName} && grep{!/^[$Settings::ALLOW_LETTERS\.]+$/} @{ $params->{Filter}->{PageName}};

        if (defined $params->{Filter}{Phrase}) {
            foreach my $phrase (@{$params->{Filter}{Phrase}}) {
                return ('BadPhraseFilter', iget('Превышена допустимая длина строки фильтра в %d символов', $Settings::MAX_PHRASE_LENGTH)) if length($phrase) > $Settings::MAX_PHRASE_LENGTH;
                return ('BadPhraseFilter')  unless $phrase =~ /^[$Settings::ALLOW_LETTERS\- \"!\+]+$/;
            }
        }


        return ('BadStatGoalsFilter')  if defined $params->{Filter}->{StatGoals} && grep{ !/^\d+$/ } @{ $params->{Filter}->{StatGoals} };

        # pageType
        return ('BadPageType') if defined $params->{Filter}->{PageType}
                                    && $params->{Filter}->{PageType} !~ /^(all|search|context)$/;

        # place, position type
        return ('BadPositionType') if defined $params->{Filter}->{PositionType}
                                        && ! defined $APICommon::POSITION_TYPES{ $params->{Filter}->{PositionType} };

        return (
            'BadPositionType',
            iget('Для работы фильтрации PositionType необходимо указать clPositionType в GroupByColumns')
        ) if defined $params->{Filter}->{PositionType}
            and ! grep { $_ eq 'clPositionType' } @{$params->{GroupByColumns}};

        # Image
        return ('BadFilter', iget('Неверное условие фильтрации по наличию изображения')) if defined $params->{Filter}->{WithImage}
                                    && $params->{Filter}->{WithImage} !~ /^(yes|no|both)$/i;

        # Device type
        return ('BadFilter', iget('Неверное условие фильтрации по типу устройства')) if defined $params->{Filter}->{DeviceType}
                                    && $params->{Filter}->{DeviceType} !~ /^(mobile|desktop|tablet)$/i;
        # Age
        return ('BadFilter', iget('Неверное условие фильтрации по возрасту')) if defined $params->{Filter}->{Age}
                                    && any {$_ !~ /^(AGE_0_17|AGE_18_24|AGE_25_34|AGE_35_44|AGE_45|AGE_UNKNOWN)$/i} @{$params->{Filter}->{Age}};
        # Gender
        return ('BadFilter', iget('Неверное условие фильтрации по полу')) if defined $params->{Filter}->{Gender}
                                    && any {$_ !~ /^(GENDER_MALE|GENDER_FEMALE|GENDER_UNKNOWN)$/i} @{$params->{Filter}->{Gender}};
        # MobilePlatform
        return ('BadFilter', iget('Неверное условие фильтрации по типу мобильной операционной системы')) if defined $params->{Filter}->{MobilePlatform}
                                    && any {$_ !~ /^(ANDROID|IOS|OS_TYPE_UNKNOWN)$/i} @{$params->{Filter}->{MobilePlatform}};
        # CarrierType
        return ('BadFilter', iget('Неверное условие фильтрации по типу мобильной связи')) if defined $params->{Filter}->{CarrierType}
                                    && any {$_ !~ /^(CELLULAR|STATIONARY|CARRIER_TYPE_UNKNOWN)$/i} @{$params->{Filter}->{CarrierType}};
        if (defined $params->{Filter}->{CarrierType} || defined $params->{Filter}->{MobilePlatform}) {
            return ('BadFilter') unless $camp->{type} && $camp->{type} eq 'mobile_content';
        }

        if (defined $params->{Filter}->{Phrase}) {
            return ('BadFilter') unless (camp_kind_in(type => $camp->{type}, 'api_stat_phrase_filter'));
        }
    }

    # group by date
    return ('BadGroupByDate') if !$O{sync}
        && defined $params->{GroupByDate}
            && $params->{GroupByDate} !~ /^(day|week|month)$/;
    # specified column name

    # проверяем правильность параметра GroupByColumnns
    # проверяем правильность параметра OrderBy
    for my $column('GroupByColumns', 'OrderBy') {
        if (defined $params->{$column}
            && ref $params->{$column} eq 'ARRAY'
                && scalar @{$params->{$column}}) {
            if ($O{sync}) {
                return ('BadColumnName', iget("Поле %s", $column))
                    if scalar grep {
                        !/^cl(Banner|Phrase|Date|Image|AveragePosition|DeviceType|ROI)$/
                    } @{$params->{$column}};
            } else {
                return ('BadColumnName', iget("Поле %s", $column))
                    if scalar grep {
                        ! /^cl(Banner|Page|Geo|Phrase|Date|StatGoals|GoalConversionsNum|PositionType|Image|AveragePosition|DeviceType|Demographics|ROI|MobilePlatform|CarrierType|Adjustment)$/ 
                    } @{$params->{$column}};
            }
        }
    }

    return ('BadLimits')
        if _check_fields_exist($params, [qw/Limit/], positive => 1 );

    return ('BadOffset') if (defined $params->{Limit} && !defined $params->{Offset}) || (defined $params->{Offset} && $params->{Offset} !~ /^\d+$/);

    if (!$O{sync} && defined $params->{TypeResultReport} && $params->{TypeResultReport} ne 'xml') {
        return ('BadTypeResultReport', iget('Формат отчёта может быть только xml'));
    }

    if ($self->{api_version_full} > 4){
        if ($params->{Currency}) {
            return ('BadCurrency') unless is_valid_currency($params->{Currency});
        }

        if (my @fields_exist_errors = _check_fields_exist($params, [qw/IncludeVAT IncludeDiscount/], type => 'yesno')){
            return @fields_exist_errors;
        }
    }

    return; # ok
}

=head3 _validate_get_banners_stat

    Синхронный аналог CreateNewReport

=cut

sub _validate_get_banners_stat {
    my ($self, $params) = @_;
    my $MAX_DAYS = 7;

    return ('BadRequest') if !$params || ref($params) ne 'HASH';

    return ('BadGroupByDate') if defined $params->{GroupByDate} && $params->{GroupByDate} ne 'day';
    if (my @report_errors = _validate_create_new_report($self, $params, sync => 1)) {
        return (@report_errors);
    }

    my $start_date =  mysql2unix($params->{StartDate});
    my $end_date = mysql2unix($params->{EndDate});
    if ($end_date - $start_date >= $SECONDS_IN_A_DAY * $MAX_DAYS) {
        return ('BadTimeInterval', iget('Можно получить статистику не более, чем за %d дней', $MAX_DAYS));
    }

    if (any {lc($_) eq 'cldevicetype'} @{$params->{GroupByColumns}}) {
        my $device_type_date = mysql2unix($Stat::Const::BS_DEVICE_TYPE_BORDER_DATE);
        if ($device_type_date >= $start_date) {
            return ('NoStat', iget('Невозможно получить статистику по типам устройств за одну из указанных дат'));
        }
    }

    if (any {lc($_) eq 'claverageposition'} @{$params->{GroupByColumns}}) {
        my $avg_pos_date = mysql2unix($Stat::Const::BS_STREAM_AVG_POS_BORDER_DATE);
        if ($avg_pos_date >= $start_date) {
            return ('NoStat', iget('Невозможно получить статистику по средним позициям за одну из указанных дат'));
        }
    }

    if (any {lc($_) eq 'clroi'} @{$params->{GroupByColumns}}) {
        my $roi_date = mysql2unix('20150510'); # DIRECT-42611
        if ($roi_date > $start_date) {
            return ('NoStat', iget('Невозможно получить статистику по ROI за одну из указанных дат'));
        }
    }

    if ($self->{api_version_full} > 4 && $params->{Currency}) {
        return ('BadCurrency') unless is_valid_currency($params->{Currency});
    }

    return; # OK
}

sub _validate_create_new_forecast
{
    my ($self, $opt) = @_;

    return ('BadRequest') if ! $opt || ref($opt) ne 'HASH' ;
    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID}, 'forecast_disallow');

    foreach my $field (qw/Phrases GeoID/) {
        if( defined $opt->{$field} && ref( $opt->{$field} ) ne 'ARRAY') {
            return ('NotArray', iget('Поле %s', $field));
        }
    }

    my $MAX_FORECAST_PHRASES = 100;
    if( defined $opt->{Phrases}){
        if (scalar @{ $opt->{Phrases} || [] }) {
            return ('LongMass', iget("Массив 'Phrases' должен содержать не более %d элементов", $MAX_FORECAST_PHRASES)) if scalar @{ $opt->{Phrases} || [] } > $MAX_FORECAST_PHRASES ;

            my $keywords_vr = validate_keywords_for_api_forecasts($opt->{Phrases});
            $keywords_vr->process_objects_descriptions( keyword => { field => 'Phrases' } );
            unless ($keywords_vr->is_valid) {
                my $errors = join '. ', @{$keywords_vr->one_error_description_by_objects};
                return ('BadParams', $errors);
            }
        } else {
            return ('BadParams', iget("Поле Phrases должно быть не пустым массивом"));
        }
    } else {
        return ('BadParams', iget("Поле Phrases должно быть указано"));
    }
    
    if( defined $opt->{GeoID} ) {
        my $geo_error = validate_geo($opt->{Geo}, undef, {tree => 'api'});
        if ($geo_error) {
            return ('BadGeo');
        }
    }

    if ($self->{api_version_full} > 4) {
        return ('BadCurrency') unless is_valid_currency($opt->{Currency}) && $opt->{Currency} ne 'YND_FIXED';
    }

    if ($self->{api_version_full} > 4) {
        if (my @fields_errors = _check_fields_exist($opt, [qw/AuctionBids/], type => 'yesno')) {
            return @fields_errors;
        }
    }

    return; #ok
}

sub _validate_extended_forecast {
    my ($self, $params) = @_;

    if (defined $params->{PeriodType}) {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/PeriodType/], type => 'list', list => [qw/Month Quarter Year/])) {
            return @fields_exist_errors;
        }
        if ($params->{PeriodType} eq "Month") {
            if (my @fields_exist_errors = _check_fields_exist($params, [qw/PeriodValue/], def => 1, type => 'unsignedint', more_than => 0, max => 12)) {
                return @fields_exist_errors;
            }
        } elsif ($params->{PeriodType} eq "Quarter") {
            if (my @fields_exist_errors = _check_fields_exist($params, [qw/PeriodValue/], def => 1, type => 'unsignedint', more_than => 0, max => 4)) {
                return @fields_exist_errors;
            }
        }
    }

    return; #ok
}

sub _validate_get_forecast
{
    my ($self, $forecast_id) = @_;

    return ('BadForecastID') unless $forecast_id && $forecast_id =~ /^\d+$/;
    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID}, 'forecast_disallow');

    my $report = get_queue_report_db('forecast', $forecast_id, $self->{uid});

    if (!$report || !ref $report) {
        dieSOAP("NoForecast");
    }

    unless ($report->is_finished()) {
        dieSOAP("ForecastIsPending");
    }

    unless ($report->has_result && defined $report->result->{filename}) { # готов, но почему-то без результата
        dieSOAP('InternalLogicError', iget('Невозможно получить отчёт.'));
    }

    $self->{_preprocessed}->{report} = $report;

    return; #ok
}

sub _validate_get_clients_units
{
    my ($self, $params) = @_;

    if (defined $params) {
        if (my @array_errors = _check_array($params, 'LoginList', not_empty => 1, type => 'login')){
            return @array_errors;
        }
    }

    return; #ok
}

sub _validate_get_client_info
{
    my ($self, $params) = @_;
    # максимальное количество клиентов при запросе GetClientInfo
    my $MAX_CLIENTS_IN_QUERY = 1000;
    if (defined $params) {
        if (my @array_errors = _check_array($params, 'LoginList', not_empty => 1, max => $MAX_CLIENTS_IN_QUERY, type => 'login')){
            return @array_errors;
        }
    }

    return; #ok
}

sub _validate_update_client_info
{
    my ($self, $params) = @_;

    if (my @array_errors = _check_array($params, 'ClientInfoList', not_empty => 1, type => 'structure')){
        return @array_errors;
    }

    my %logins_cnt = map { lc($_->{Login}) => 1 } @$params;

    if (scalar keys %logins_cnt < scalar @$params) {
        return ('BadParams', iget('Один и тот же логин передан несколько раз'));
    }

    my $login2uid = get_login2uid(login => [keys %logins_cnt]);

    my @client_uids = values %$login2uid;

    if (scalar @client_uids != scalar keys %logins_cnt) {
        return ('BadLogin', iget('Несуществующие логины: %s', join(', ', grep {!$login2uid->{$_}} sort keys %logins_cnt) ));
    }

    # соответствие uid и ролей
    my $clients_who_is = rbac_multi_who_is($self->{rbac}, [@client_uids]);

    foreach my $client (@$params) {

        if (my @fields_exist_errors = _check_fields_exist($client, [qw/Login/], type => 'login', def => 1)){
            return @fields_exist_errors;
        }

        my $user_role = '';
        if (exists $login2uid->{$client->{Login}} and exists $clients_who_is->{$login2uid->{$client->{Login}}}) {
            $user_role = $clients_who_is->{$login2uid->{$client->{Login}}};
        }

        if (my @fields_exist_errors = _check_fields_exist($client, [qw/FIO/], type => 'string', def => 1)){
            return @fields_exist_errors;
        }

        if (my @fields_exist_errors = _check_fields_exist($client, [qw/Phone/], type => 'phone', def => 1)){
            return @fields_exist_errors;
        }

        if (my @fields_exist_errors = _check_fields_exist($client, [qw/Email/], type => 'email', def => 1)){
            return @fields_exist_errors;
        }
        
        if ($self->{api_version_full} > 4) {
            if (my @fields_errors = _check_fields_exist($client, [qw/DisplayStoreRating/], type => 'yesno')){
                return @fields_errors;
            }
        }
        if ($self->{api_version_full} > 4.5) { # latest - male! bad!
            if (my @fields_exist_errors = _check_fields_exist($client, [qw/Description/], type => 'string', len => $Client::MAX_USER_DESCRIPTION_LENGTH)){
                return @fields_exist_errors;
            }
            if ($self->{rbac_login_rights}->{role} eq 'manager') {
                if (defined $client->{Descriptions}) {

                    if (my @array_errors = _check_array($client->{Descriptions}, 'Descriptions', type => 'structure')){
                        return @array_errors;
                    }

                    foreach my $desc (@{$client->{Descriptions}}) {
                        if (my @fields_exist_errors = _check_fields_exist($desc, [qw/Description/], type => 'string', len => $Client::MAX_USER_DESCRIPTION_LENGTH)){
                            return @fields_exist_errors;
                        }
                        if (my @fields_exist_errors = _check_fields_exist($desc, [qw/AgencyLogin/], type => 'login')){
                            return @fields_exist_errors;
                        }
                    }
                }
            }
        }

        if ($self->{rbac_login_rights}->{role} =~ /^(super|support|manager)$/) {
            if ($user_role =~ /(agency|client)$/ ) {
                if (my @fields_exist_errors = _check_fields_exist($client, [qw/NonResident/], type => 'yesno', def => 1 )){
                    return @fields_exist_errors;
                }
                if ($self->{api_version_full} > 4) { # latest - male! bad!
                    if (my @fields_exist_errors = _check_fields_exist($client, [qw/CityID/], type => 'int' )){
                        return @fields_exist_errors;
                    }
                    if (defined $client->{CityID} && $client->{CityID} != 0 && !is_region_city($client->{CityID})) {
                        return ('BadGeo', iget('В поле CityID должен быть идентификатор города'));
                    }

                    if ($user_role eq 'agency') {
                        # TODO: параметры должны юыть обязательными! иначе затрем в базе значения при пустом значении
                        if (my @fields_exist_errors = _check_fields_exist($client, [qw/AgencyName/], type => 'string', len => 255 )){
                            return @fields_exist_errors;
                        }

                        if (my @fields_exist_errors = _check_fields_exist($client, [qw/AgencyUrl/], type => 'url' )){
                            return @fields_exist_errors;
                        }

                        if (my @fields_exist_errors = _check_fields_exist($client, [qw/AgencyStatus/], type => 'string' )){
                            return @fields_exist_errors;
                        }

                        # TODO: может получиться, что ДК затрет статус агентства ?!
                        my $valid_status = { SA => 1, AA => 1, HY => 1, WC => 1, NR => 1, ABU => 1, };
                        if (defined $client->{AgencyStatus} && !$valid_status->{$client->{AgencyStatus}}) {
                            return('BadParams', iget('Поле %s должно быть неопределено или содержать значения: %s', 'AgencyStatus', join(', ', sort keys %$valid_status)));
                        }
                    }

                }
            }
        }

        if ($user_role =~ /client$/ ) {
            if (my @fields_exist_errors = _check_fields_exist($client, [qw/SendWarn SendNews SendAccNews/], type => 'yesno', def => 1 )){
                return @fields_exist_errors;
            }
        }

        if ($self->{rbac_login_rights}->{role} =~ /^(agency|super|manager)$/
            && $user_role =~ /client$/
            && rbac_has_agency($self->{rbac}, $login2uid->{$client->{Login}} ) ) {

            my %rights_list = map {$_ => {}} qw/AllowEditCampaigns AllowImportXLS AllowTransferMoney/;

            if (my @fields_exist_errors = _check_fields_exist($client, [qw/ClientRights/], def => 1)){
                return @fields_exist_errors;
            }

            if (defined $client->{ClientRights}) {

                if (my @array_errors = _check_array($client->{ClientRights}, 'ClientRights', type => 'structure')){
                    return @array_errors;
                }

                # валидируем состояние
                my $rights_hash = {};

                foreach my $right (@{$client->{ClientRights}}) {
                    if (my @fields_exist_errors = _check_fields_exist($right, [qw/RightName/], type => 'list', list => [sort keys %rights_list], def => 1)){
                        return @fields_exist_errors;
                    }

                    if (my @fields_exist_errors = _check_fields_exist($right, [qw/Value/], type => 'yesno', def => 1)){
                        return @fields_exist_errors;
                    }

                    if ($self->{rbac_login_rights}->{role} =~ /^(super|support|manager)$/) {

                        if (my @fields_exist_errors = _check_fields_exist($right, [qw/AgencyLogin/], type => 'login', def => 1)){
                            return @fields_exist_errors;
                        }

                        if (++$rights_list{$right->{RightName}}{$right->{AgencyLogin}} > 1) {
                            return('BadParams', iget('В списке прав массива ClientRights дублируются названия прав'));
                        }

                    } elsif ($self->{rbac_login_rights}->{role} eq 'agency') {

                        if (++$rights_list{$right->{RightName}}{cnt} > 1) {
                            return('BadParams', iget('В списке прав массива ClientRights дублируются названия прав'));
                        }

                    }

                    # NB: ($right->{AgencyLogin} || '') - имхо, так сделать можно, т.к. всё структуры $right, у которых
                    # $right->{AgencyLogin} = undef, и так попадут под один ключ в хеше $rights_hash.
                    $rights_hash->{($right->{AgencyLogin} || '')}{$right->{RightName}} = $right->{Value};
                }

                while (my ($agency, $hash) = each %$rights_hash) {
                    if (   defined $hash->{AllowImportXLS}
                        && defined $hash->{AllowEditCampaigns}
                        && $hash->{AllowImportXLS} eq 'Yes'
                        && $hash->{AllowEditCampaigns} eq 'No'
                    ) {
                        return ('BadParams', iget('Некорректное состояние прав: AllowImportXLS = Yes, AllowEditCampaigns = No'));
                    }
                }

            }
        }
    }

    return; #ok
}

sub _validate_delete_report
{
    my ($self, $params) = @_;

    return ('BadParams', iget('Не указан ReportID')) if ! defined $params;
    return ('BadReportID') unless defined $params && $params =~ /^\d+$/ && $params > 0;

    return; #ok
}

=head3 _validate_transfer_money(params)

    Метод MoneyTransfer

=cut

sub _validate_transfer_money
{
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/FromCampaigns ToCampaigns/], def => 1)){
        return @fields_exist_errors;
    }

    if (my @array_errors = _check_array($params->{FromCampaigns}, 'FromCampaigns', not_empty => 1, type=>'structure')){
        return @array_errors;
    }

    if (my @array_errors = _check_array($params->{ToCampaigns}, 'ToCampaigns', not_empty => 1, type=>'structure')){
        return @array_errors;
    }

    my ( $sums, $cids ) = ( {}, {} );

    for my $field (qw/FromCampaigns ToCampaigns/) {
        foreach my $item (@{$params->{$field}}) {
            if (!$item->{CampaignID} || !$item->{Sum}) {
                return ('BadTransferMoneyRequest', iget('Каждый элемент массива %s должен содержать поля Sum and CampaignID', $field));
            }

            if (my @fields_exist_errors = _check_fields_exist($item, [qw/Sum/], def => 1, type => 'float')){
                return @fields_exist_errors;
            }

            if ( $item->{Sum} <= 0 ) {
                return ('BadTransferMoneyRequest', iget('Поле Sum должно содержать значение больше 0'));
            }

            return ('BadCampaignID') if !validate_id( { cid => $item->{CampaignID}, check_exist => 1 } );
            return ('BadCurrency') unless is_valid_currency($item->{Currency}) && $item->{Currency} ne 'YND_FIXED';

            return ('BadTransferMoneyRequest', iget("Поле %s в структуре %s должно быть уникальным в запросе", 'CampaignID', $field)) if $cids->{$field}->{$item->{CampaignID}};
            $cids->{$field}{$item->{CampaignID}}++;
        }
    }

    my @camps_cids = map {$_->{CampaignID}} (@{$params->{FromCampaigns}}, @{$params->{ToCampaigns}});

    # -- Разделили запрос - отдельно к кампаниям
    my $camps_info = get_hashes_hash_sql(PPC(cid => \@camps_cids), 
        ["select c.cid, c.agencyuid, c.name, c.uid, c.wallet_cid, IFNULL(c.currency, 'YND_FIXED') as currency, c.type
               , u.ClientID
          from campaigns c
          inner join users u ON c.uid = u.uid
        ",
            where => {cid__in => SHARD_IDS, type__not_in => ['wallet']}
        ]
    );

    return('NoRights') if ((scalar uniq @camps_cids) != (scalar keys %$camps_info));

    my @agency_uids = uniq grep { defined } map {$_->{agencyuid}} values %$camps_info;
    my $agency_uid2clientid = get_uid2clientid(uid => \@agency_uids);

    $_->{agency_clientid} = (($_->{agencyuid}) ? $agency_uid2clientid->{$_->{agencyuid}} : undef) for values %$camps_info;

    $self->{_preprocessed} = $camps_info;

    foreach my $cid (@camps_cids) { # такая же проверка осуществляется потом в &RBACDirect::rbac_cmd_transfer с точностью до общих счетов, если они подключены
        # XXX: почему проверяется доступ uid'а из кампании, а не $self->{uid}?
        # TODO performance: можно сгруппировать и проверять массово через rbac_check_owner_of_camps
        return ('NoRights') if ! rbac_is_owner_of_camp($self->{rbac}, $camps_info->{$cid}->{uid}, $cid);
    }

    if ($self->{rbac_login_rights}->{role} =~ /client$/ && @agency_uids) {
        my $cid2allow_transfer_money = rbac_check_allow_transfer_money_camps($self->{rbac}, $self->{uid}, \@camps_cids);
        if (any { !$cid2allow_transfer_money->{$_} } @camps_cids) {
            return ('NoRights', iget("Недостаточно прав для переноса денег между кампаниями агентства."));
        }
    } else {
        return ('NoRights') unless rbac_user_allow_edit_camp($self->{rbac}, $self->{uid}, \@camps_cids); # также вызовется в &MoneyTransfer::validate_transfer_money
    }

    my @client_ids = uniq map { $_->{ClientID} } values %$camps_info;
    return('NoRights', APICommon::msg_converting_in_progress) if Client::is_any_client_converting_soon(\@client_ids);
    return('NoRights', APICommon::msg_must_convert) if Client::is_any_client_must_convert(\@client_ids);

    my @from_camps_uids = uniq map {$camps_info->{$_->{CampaignID}}->{uid}} @{$params->{FromCampaigns}};
    my @to_camps_uids = uniq map {$camps_info->{$_->{CampaignID}}->{uid}} @{$params->{ToCampaigns}};
    my @wallets_from = uniq grep {$_} map {$camps_info->{$_->{CampaignID}}->{wallet_cid}} @{$params->{FromCampaigns}};
    my @wallets_to = uniq grep {$_} map {$camps_info->{$_->{CampaignID}}->{wallet_cid}} @{$params->{ToCampaigns}};

    foreach my $cid (@camps_cids) {
        return ('NotSupported', iget('Тип кампании не поддерживается. Кампания %d', $cid)) unless (camp_kind_in(type => $camps_info->{$cid}->{type}, 'under_wallet'));
    }

    if (scalar @from_camps_uids > 1) {
        return ('BadTransferMoneyRequest', iget('Не разрешается переносить деньги с кампаний нескольких клиентов'));
    }

    if (scalar @to_camps_uids > 1) {
        return ('BadTransferMoneyRequest', iget('Не разрешается переносить деньги на кампании нескольких клиентов'));
    }

    if ((scalar @wallets_from > 1) || (scalar @wallets_to > 1)) {
        return ('BadTransferMoneyRequest', iget('Операция переноса денег c использованием более одного счета одного клиента невозможна'));
    }

    if ((scalar @wallets_from == 1) && (scalar @wallets_to == 1)) {
        if ($wallets_from[0] == $wallets_to[0]) {
            return ('BadTransferMoneyRequest', iget('Операция переноса денег между кампаниями одного общего счёта невозможна'));
        }
        if ($from_camps_uids[0] == $to_camps_uids[0]) {
            return ('BadTransferMoneyRequest', iget('При использовании общего счёта возможен перенос только между счетами разных клиентов'));
        }
    }

    if ((uniq map {$_->{agency_clientid} // 0} values %$camps_info) > 1) {
        return ('BadTransferMoneyRequest', iget('Допускается перенос только в пределах одного агентства'));
    }

    # хэш используемых в запросе валют
    my %used_currencies;

    for my $field (qw/FromCampaigns ToCampaigns/) {
        foreach my $item (@{$params->{$field}}) {
            my $sum_currency = 'YND_FIXED';
            if ($self->{api_version_full} > 4) {
                $sum_currency = $item->{Currency} || 'YND_FIXED';
            }

            $used_currencies{$sum_currency} = 1;

            $sums->{$field} += round2s($item->{Sum});
        }
    }

    return ('BadCurrency', iget('Все суммы в запросе должны быть указаны в одной валюте')) if scalar keys %used_currencies > 1;

    if ($self->{api_version_full} > 4) {
        my %camps_currencies = map { $_->{currency} => 1 } values %$camps_info;
        return ('BadCurrency', iget('Запрещен перенос денег между кампаниями с разными валютами')) if scalar keys %camps_currencies > 1;

        delete $camps_currencies{YND_FIXED};

        foreach my $item (@{$params->{FromCampaigns}}, @{$params->{ToCampaigns}}) {
            next unless $item->{Currency};

            if (my @fields_exist_errors = _check_fields_exist($item, [qw/Currency/], type => 'currency', list => [sort keys %camps_currencies]) ) {
                return (@fields_exist_errors);
            }
        }
    }

    if (round2s($sums->{FromCampaigns}) != round2s($sums->{ToCampaigns})) {
        return ('BadTransferMoneyRequest', iget('Суммы переносимых денег не равны: %0.2f и %0.2f', $sums->{FromCampaigns}, $sums->{ToCampaigns}));
    }

    return; # ok
}

sub _validate_create_invoice
{
    my ($self, $params) = @_;
    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Payments/], def => 1)) {
        return @fields_exist_errors;
    }

    if (my @array_errors = _check_array($params->{Payments}, 'Payments', not_empty => 1, type => 'structure')) {
        return @array_errors;
    }

    my $cids = {};

    foreach my $item (@{$params->{Payments}}) {

        if (my @fields_exist_errors = _check_fields_exist($item, [qw/CampaignID Sum/], def => 1)){
            return @fields_exist_errors;
        }

        return ('BadCurrency') unless is_valid_currency($item->{Currency}) && $item->{Currency} ne 'YND_FIXED';

        return ('BadCampaignID') if !validate_id( { cid => $item->{CampaignID}, check_exist => 1 } );

        # is integer
        return ('BadParams', iget('Поле %s должно содержать значение типа integer', 'CampaignID')) if ( ref $item->{CampaignID} || $item->{CampaignID} !~ m/^\d+$/ );

        # is float
        return ('BadParams', iget("Поле %s должно содержать значение типа float", 'Sum')) if $item->{Sum} !~ m/^-?\d+(\.\d+)?$/x;

        return ('BadPayCamp', iget("Поле %s в структуре %s должно быть уникальным в запросе", 'CampaignID', 'Payments')) if $cids->{$item->{CampaignID}};
        
        $cids->{$item->{CampaignID}}++;
    }

    my @errors = API::ValidateRights::validate_invoice_rights($self, $params);
    return @errors if @errors;

    my @cids = keys %{$self->{_preprocessed}};
    my @client_ids = uniq map { $_->{ClientID} } values %{$self->{_preprocessed}};
    
    dieSOAP('NoRights', APICommon::msg_converting_in_progress) if Client::is_any_client_converting_soon(\@client_ids);
    dieSOAP('NoRights', APICommon::msg_must_convert) if Client::is_any_client_must_convert(\@client_ids);
    
    return ('BadCurrency', iget('Не допускается оплата кампаний с разными валютами')) if ((scalar uniq map {$self->{_preprocessed}->{$_}->{currency}} @cids) > 1);
    
    my $request_currency;
    if ($self->{api_version_full} > 4) {

        foreach my $item (@{$params->{Payments}}) {

            my $camp_currency = $self->{_preprocessed}->{$item->{CampaignID}}->{currency};

            if (($camp_currency // '') eq 'UAH') {
                return ('BadCurrency');
            }

            unless ($request_currency) {
                my @allowed_first_currencies = grep {$_ ne 'YND_FIXED'} ($camp_currency);
                if ($item->{Currency}) {
                    if (my @fields_exist_errors = _check_fields_exist($item, [qw/Currency/], type => 'currency', list => \@allowed_first_currencies) ) {
                        return (@fields_exist_errors);
                    }
                }
                $request_currency = $item->{Currency} || 'YND_FIXED';
            }
            return ('BadCurrency', iget('Все суммы в запросе должны быть указаны в одной валюте')) if (($item->{Currency} || 'YND_FIXED') ne $request_currency);
        }

    }

    return ('PaymentDifferentCampaignsType') if ((scalar uniq map {$self->{_preprocessed}->{$_}->{type}} @cids) > 1);

    return; #ok
}

sub _validate_pay_campaigns_by_card {
    my ($self, $params) = @_;

    if (my @errors = _validate_pay_campaigns($self, $params, 'LinkedCard')) {
        return @errors;
    }

    return;
}

sub _validate_pay_campaigns {
    my ($self, $params, $method) = @_;

    my %VALID_PAY_METHODS = (Bank => 1, YM => 1, Overdraft => 1);
    if ($self->{api_version_full} == 4) {
        %VALID_PAY_METHODS = (Bank => 1);
    }
    if ($method and $method eq 'LinkedCard') {
        $VALID_PAY_METHODS{LinkedCard} = 1;
    }
    return ('BadRequest') if (ref($params) ne 'HASH');

    $method ||= $params->{PayMethod};

    if (my @fields_exist_errors = _check_fields_exist({PayMethod => $method}, [qw/PayMethod/], def => 1)){
        return @fields_exist_errors;
    }

    return ('BadParams', iget('Поле %s должно содержать значения: %s', 'PayMethod', join ', ', grep {$_ ne 'YM'} sort keys %VALID_PAY_METHODS)) if !$VALID_PAY_METHODS{ $method };

    if ($method eq 'Bank') {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/ContractID/], def => 1, type => 'contract')) {
            return @fields_exist_errors;
        }
    }

    if ($method eq 'LinkedCard') {
        if (my @errors = _check_fields_exist($params, [qw/CustomTransactionID/], def => 1, type => 'string', not_empty => 1)) {
            return @errors;
        }

        my $transaction_id = $params->{CustomTransactionID};
        return ('BadParams', iget("Поле %s должно содержать строку длиной %s символа, состоящую из цифр и букв латинского алфавита", 'CustomTransactionID', 32))
            unless $transaction_id =~ m/^[0-9a-zA-Z]{32}$/;

        if (my @errors = _check_fields_exist($params, [qw/PayMethodID/], def => 1, type => 'string', not_empty => 1)) {
            return @errors;
        }

        if (my @errors = _check_fields_exist($params, [qw/Version/], type => 'string', not_empty => 1)) {
            return @errors;
        }
    }

    my @errors = _validate_create_invoice($self, $params);
    return @errors if @errors;

    # размещено тут, потому что нужные данные в $self->{_preprocessed}
    # заполняются при вызове _validate_create_invoice(...)
    if ($method eq 'LinkedCard') {
        my @bad_cids = map { $_->{cid} } grep { $_->{wallet_cid} } values %{ $self->{_preprocessed} };
        if ( @bad_cids ) {
            return ('BadParams', iget('Для кампаний %s включен общий счёт, пополнение возможно через соответствующий счёт', join ', ' => @bad_cids));
        }

        foreach my $item ( @{ $params->{Payments} } ) {
            my $req_currency  = $item->{Currency};
            my $camp_currency = $self->{_preprocessed}{ $item->{CampaignID} }{currency};
            if ($req_currency and $req_currency ne $camp_currency) {
                return ('BadCurrency');
            }
            $item->{Currency} ||= 'RUB'; # ||= $camp_currency;
        }
    }

    return;
}

sub _validate_check_payment
{
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @errors = _check_fields_exist($params, [qw/CustomTransactionID/], def => 1, type => 'string', not_empty => 1)) {
        return @errors;
    }

    my $transaction_id = $params->{CustomTransactionID};
    return ('BadParams', iget("Поле %s должно содержать строку длиной %s символа, состоящую из цифр и букв латинского алфавита", 'CustomTransactionID', 32))
        unless $transaction_id && $transaction_id =~ m/^[0-9a-zA-Z]{32}$/;

    return;
}

sub _validate_get_forecast_list
{
    my ($self, $params) = @_;

    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID}, 'forecast_disallow');

    return;
}

sub _validate_delete_forecast_report
{
    my ($self, $params) = @_;

    return ('BadParams', iget('Не указан ForecastID')) if ! defined $params;
    return ('BadForecastID') unless $params && $params =~ /^\d+$/;

    return ('AccessDenied') if has_spec_limits($self->{ClientID})
                                && get_spec_limit($self->{ClientID}, 'forecast_disallow');

    # существует ли отчет (если удален = не существует)
    my $report = get_queue_report_db('forecast', $params, $self->{uid});
    if (!$report || !ref $report) {
        return ('NoReport');
    }

    $self->{_preprocessed}->{report} = $report;

    return; #ok
}

sub _validate_get_balance
{
    my ($self, $params) = @_;

    return ('NotArray') if ref $params ne 'ARRAY';

    foreach my $cid ( @{$params} ) {
        return ('BadCampaignID') if !validate_id( { cid => $cid, check_exist => 1 } );
    }

    my $error_code = check_rbac_rights( $self, 'api_showCamp', { cid => $params, UID => $self->{uid}, uid => $self->{uid} } );
    return ( 'NoRights' ) if $error_code;

    # multicurrency - интерфейс метода не возволяет задавать валюту кампании
#     if ($self->{api_version_full} > 4 && defined $params->{Currency}) {
#         return ('BadCurrency') unless is_valid_currency($params->{Currency});
#     }

    return; #ok
}

sub _validate_get_subclients
{
    my ($self, $params) = @_;
    return ('BadRequest') if defined $params && ref($params) ne 'HASH';

    if ($params && $params->{Login}){
        return ('BadLogin') if !validate_login( $params->{Login}, lite => 1 );
    }

    if ($params && $params->{RoleFilter}) {
        return ('BadParams', iget("Неверное значение поля RoleFilter")) unless $params->{RoleFilter} =~ /^(Staff|Clients|All)$/;
    }

    return _validate_client_filter($params);
}

sub _validate_client_filter
{
    my $params=shift;

    if ($params && defined $params->{Filter}) {
        if ( ref ($params->{Filter}) ne 'HASH') {
            return ('BadParams', iget('Неверная структура %s', 'Filter'));
        }

        if ( keys %{ $params->{Filter}  } ) {
             if ( grep { $_ !~ /^(StatusArch)$/  } keys  %{ $params->{Filter} } ) {
                 return ('BadParams', iget('Неизвестные ключи в структуре  %s', 'Filter'));
             }

             if (defined $params->{Filter}{'StatusArch'} && !is_yes_no($params->{Filter}{'StatusArch'})) {
                 return ('NotYesNo', iget('Поле %s', 'StatusArch'));
             }
        }
     }

  return; #ok
}

sub _validate_get_clients_list
{
    my ($self, $params) = @_;
    return ('BadRequest') if defined $params && ref($params) ne 'HASH';

    return _validate_client_filter($params);
}


sub _validate_get_summary_stat
{
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/StartDate EndDate/], def => 1, type => 'date')){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/CampaignIDS/], def => 1)){
        return @fields_exist_errors;
    }

    if (my @array_errors = _check_array($params->{CampaignIDS}, 'CampaignIDS', type => 'unsignedint', not_empty => 1)) {
        return @array_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/IncludeDiscount IncludeVAT/], type => 'yesno')){
        return @fields_exist_errors;
    }


    my $valid_date_re = qr/^(\d{4})\-?(\d{2})\-?(\d{2})$/;
    my $days_delta = Date::Calc::Delta_Days($params->{StartDate} =~ $valid_date_re, $params->{EndDate} =~ $valid_date_re);
    return ('BadTimeInterval') if $days_delta < 0;

    if (my @error = _validate_stat_within_3years(mysql2unix($params->{EndDate}))) {
        return @error;
    }

    my @cids_uniq = uniq @{$params->{CampaignIDS}};

    my $data = get_all_sql(PPC(cid => \@cids_uniq), ['SELECT cid FROM campaigns', where => { cid__int => SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet'] } ]);

    if (scalar @$data != scalar @cids_uniq) {
        return ('BadCampaignID', iget('Некоторые из указанных кампаний не существуют'));
    }

    my $cids_string = join(', ', @cids_uniq);

    my $error_code = check_rbac_rights( $self, 'api_showCampStat', { stat_type => 'campdate', cid => $cids_string, UID => $self->{uid} } );
    return ( 'NoRights' ) if $error_code;

    if ($days_delta * scalar @{$params->{CampaignIDS}} > $APICommon::SUMMARY_STAT_MAX_LINES_IN_REPORT) {
        return ('BadParams', iget('Количество строк в ответе метода не должно превышать %s', $APICommon::SUMMARY_STAT_MAX_LINES_IN_REPORT));
    }

    return; #ok
}

sub _validate_get_events_log
{
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'HASH');

    if ($self->{rbac_login_rights}->{role} !~ /client$/) {
        if (my @array_errors = _check_array($params->{Logins}, 'Logins', type => 'login')){
            return @array_errors;
        }
        return ('BadParams', iget('Поле %s обязательно для агентства', 'Logins'))
            if scalar @{$params->{Logins}} == 0;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/TimestampFrom/], type => 'timestamp', def => 1)){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/TimestampTo/], type => 'timestamp')){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/LastEventOnly WithTextDescription/], type => 'yesno')){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Filter/], type => 'structure')){
        return @fields_exist_errors;
    }

    return ('BadCurrency') unless is_valid_currency($params->{Currency}) && $params->{Currency} ne 'YND_FIXED';

    if (defined $params->{Filter}) {
        if ($params->{Filter}{CampaignIDS}) {
            if (my @array_errors = _check_array($params->{Filter}{CampaignIDS}, 'Filter/CampaignIDS', type => 'int')){
                return @array_errors;
            }

            # проверяем на существование кампании
            my $valid_cids = get_hashes_hash_sql(PPC(cid => $params->{Filter}{CampaignIDS}),
                    ["select cid from campaigns", where => {cid => SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet']}] );

            my @cids = uniq @{ $params->{Filter}{CampaignIDS} };

            foreach my $cid (@cids) {
                unless ($valid_cids->{$cid}) {
                    
                    return ('BadCampaignID', iget('Кампания %s не существует', $cid));
                }
            }
        }
        if ($params->{Filter}{BannerIDS}) {
            if (my @array_errors = _check_array($params->{Filter}{BannerIDS}, 'Filter/BannerIDS', type => 'int')){
                return @array_errors;
            }
        }
        if ($params->{Filter}{PhraseIDS}) {
            if (my @array_errors = _check_array($params->{Filter}{PhraseIDS}, 'Filter/PhraseIDS', type => 'int')){
                return @array_errors;
            }
        }
        if ($params->{Filter}{AccountIDS}) {
            if (my @array_errors = _check_array($params->{Filter}{AccountIDS}, 'Filter/AccountIDS', type => 'int')){
                return @array_errors;
            }
        }
        if ($params->{Filter}{EventType}) {
            if (my @array_errors = _check_array($params->{Filter}{EventType}, 'Filter/EventType', type => 'string')){
                return @array_errors;
            }

            my %reversed_tokens = reverse %APICommon::EVENT_TYPE2TOKEN;
            foreach (@{$params->{Filter}{EventType}}) {
                return ('BadParams', iget('Массив %s структуры %s должен содержать значения: %s', 'EventType', 'Filter', join ', ', sort keys %reversed_tokens)) if !$reversed_tokens{$_};
            }
        }
    }

    return ('BadLimits') if defined $params->{Limit} && ($params->{Limit} !~ /^\d+$/ || $params->{Limit} == 0);
    return ('BadOffset') if defined $params->{Offset} && $params->{Offset} !~ /^\d+$/;

    return;
}

sub _validate_add_events {
    my ($self, $params) = @_;
    return ('BadRequest') if (ref($params) ne 'ARRAY');

    my %reversed_tokens = reverse %APICommon::EVENT_TYPE2TOKEN;

    foreach my $event (@$params) {
        if (my @fields_exist_errors = _check_fields_exist($event, [qw/EventType/], def => 1, type => 'list', list => [sort keys %reversed_tokens])){
            return (@fields_exist_errors);
        }
    }

    return;
}

sub _validate_get_campaigns_tags {
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @array_errors = _check_array($params->{CampaignIDS}, 'CampaignIDS', type => 'int', not_empty => 1)){
        return @array_errors;
    }
    foreach my $cid (@{$params->{CampaignIDS}}) {
        check_uid_cid_bid($self, cid => $cid, campaign_kind => 'base');
    }
    return;
}

sub _validate_update_campaigns_tags {
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'ARRAY');

    foreach my $campaign_tags (@$params) {

        if (my @fields_exist_errors = _check_fields_exist($campaign_tags, [qw/CampaignID/], def => 1)){
            return @fields_exist_errors;
        }

        if ( !validate_id( { cid => $campaign_tags->{CampaignID}, check_exist => 1 } ) ) {
            return ('BadCampaignID');
        }

        check_uid_cid_bid($self, cid => $campaign_tags->{CampaignID}, campaign_kind => 'base');

        if (my @array_errors = _check_array($campaign_tags->{Tags}, 'Tags', type => 'structure')){
            return @array_errors;
        }

        my (@tags_ids, %tag_id_dup_checker);
        foreach my $tag (@{$campaign_tags->{Tags}}) {
            if (my @fields_exist_errors = _check_fields_exist($tag, [qw/TagID/], type => 'unsignedint', def => 1)){
                return @fields_exist_errors;
            }

            if (my @fields_exist_errors = _check_fields_exist($tag, [qw/Tag/], type => 'string', def => 1)){
                return @fields_exist_errors;
            }
            
            return ('BadParams', iget('В списке меток кампании %d дублируются метки c TagID=%d', $campaign_tags->{CampaignID}, $tag->{TagID}))
                if $tag->{TagID} && $tag_id_dup_checker{$tag->{TagID}}++;

            push @tags_ids, { tag_id => $tag->{TagID}, name => $tag->{Tag} };
        }

        # хеш: id метки (или 0 для новых) => массив ошибок
        my $tags_error = Tag::validate_tags($campaign_tags->{CampaignID}, \@tags_ids);

        if (scalar (keys %$tags_error)) {
            return ('BadParams', 
                join ('; ',
                    map {
                        ($_ ? "TagID=$_" : iget('Новые метки')) . ': ' . join (', ', @{ $tags_error->{$_} })
                    } sort keys %$tags_error
                )
            );
        }
    }

    return;
}

sub _validate_get_banners_tags {
    my ($self, $params) = @_;

    return ('BadRequest') if (ref($params) ne 'HASH');

    if (my @banners_request_errors = _validate_banners_request_new($self, $params)) {
        return @banners_request_errors;
    }

    return;
}

sub _validate_update_banners_tags {
    my ($self, $params) = @_;

    if (my @array_errors = _check_array($params, 'BannersList', not_empty => 1, type => 'structure')){
        return @array_errors;
    }


    for my $banner (@$params) {

        if (my @fields_exist_errors = _check_fields_exist($banner, [qw/BannerID/], type => 'int' )){
            return (@fields_exist_errors);
        }

        if ($banner->{TagIDS}) {
            if (my @array_errors = _check_array($banner->{TagIDS}, 'TagIDS', type => 'unsignedint')){
                return @array_errors;
            }
        }

    }

    my $bid2cid = get_bid2cid(bid => [map {$_->{BannerID}} @$params]);
    my @cids = uniq values %$bid2cid;

    foreach my $cid (@cids) {
        check_uid_cid_bid($self, cid => $cid, campaign_kind => 'base');
    }

    for my $banner (@$params) {
        if (!$bid2cid->{$banner->{BannerID}}) {
            return ('BadParams', iget("Баннер %s не найден", $banner->{BannerID}));
        }
        if (my @tags_limit_cnt_error = Tag::check_banner_count_limit(scalar uniq @{$banner->{TagIDS}})) {
            return ('BadParams', @tags_limit_cnt_error);
        }
        if (my @tags_errors = Tag::validate_adgroup_tags($bid2cid->{$banner->{BannerID}}, $banner->{TagIDS})) {
            return ('BadParams', @tags_errors);
        }
    }
    return;
}

sub _validate_get_metro_stations
{
    my ($self, $params) = @_;

    return ('BadRequest') if ($params && ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/TimestampFrom/], type => 'timestamp', def => 1)){
        return @fields_exist_errors;
    }

    if ($params->{RegionIDS}) {
        if (my @array_errors = _check_array($params->{RegionIDS}, 'RegionIDS', type => 'geo_unsigned')){
            return @array_errors;
        }
    }

    return ;
}

sub _validate_save_subscription
{
    my ($self, $params) = @_;

    if (my @base_errors = _validate_subscription_base_params($self, $params)) {
        return @base_errors;
    }

    if (my @array_errors = _check_array($params->{EventTypes}, 'EventTypes', not_empty => 1,  type => 'string')){
        return @array_errors;
    }

    my %reversed_tokens = reverse %APICommon::EVENT_TYPE2TOKEN;
    foreach my $event_type (@{$params->{EventTypes}}) {
        if (!$reversed_tokens{$event_type}) {
            return ('BadParams', iget('Массив %s должен содержать значения: %s', 'EventTypes', join(', ', sort keys %reversed_tokens)));
        }
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Options/], type => 'string')){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Timeout/], type => 'unsignedint', max => 60)){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Locale/], type => 'locale')){
        return @fields_exist_errors;
    }
    return;
}

sub _validate_subscription_base_params
{
    my ($self, $params) = @_;

    return ('BadRequest') if ($params && ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/SubscriptionResource/], type => 'string', def => 1)){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/SubscriptionType/], def => 1)){
        return @fields_exist_errors;
    }

    if (none {$params->{SubscriptionType} eq $_} qw/APNS GCM/) {
        return('BadParams', iget('Поле %s должно содержать значения: %s', 'SubscriptionType', 'APNS, GCM'));
    }

    if ($params->{SubscriptionType} eq 'APNS') {
        if ($params->{SubscriptionResource} !~ m/^[a-f0-9]{64}$/) {
            return('BadParams', iget('Неверный SubscriptionResource'));
        }
    }

    if ($params->{SubscriptionType} eq 'GCM') {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/SubscriptionResource/], not_empty => 1, type => 'string', len => 255, def => 1)) {
            return @fields_exist_errors;
        }
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/DeviceId/], not_empty => 1, type => 'string', len => 255)) {
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Uuid/], not_empty => 1, type => 'string', len => 255)) {
        return @fields_exist_errors;
    }

    return;
}

sub _validate_search_clients
{
    my ($self, $params) = @_;

    return ('BadRequest') if ($params && ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/UserID ClientID CampaignID/], type => 'unsignedint', more_than => 0)){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Login/], type => 'login')){
        return @fields_exist_errors;
    }

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/Name/], type => 'string')){
        return @fields_exist_errors;
    }

    if (defined $params->{ManagerLogins}) {
        if (my @array_errors = _check_array($params->{ManagerLogins}, 'ManagerLogins', type => 'login')) {
            return @array_errors;
        }
    }

    my @fields = qw/UserID ClientID CampaignID Login Name/;

    if ( ! scalar grep {defined $params->{$_}} @fields ) {
        if (! defined $params->{ManagerLogins} || ! scalar @{$params->{ManagerLogins}}) {
            return ('BadParams', iget('Одно из следующих полей должно быть указано: %s, или массив %s не должен быть пустым.', join(', ', @fields), 'ManagerLogins'));
        }
    }

    return;
}

#####################################
####
#### Валидация новых массовых методов
####
#####################################

sub _validate_mediaplan_keyword_wrapper
{
    my ($self, $params) = @_;
    if ($params->{Action} eq 'Add' || $params->{Action} eq 'Update') {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/Options/], def => 1, type => 'structure')){
            return @fields_exist_errors;
        }
        if (my @fields_exist_errors = _check_fields_exist($params->{Options}, [qw/AutoMinusWords/], type => 'yesno')){
            return @fields_exist_errors;
        }
    } elsif ($params->{Action} eq 'Get' && defined $params->{Options}) {

            if (my @fields_exist_errors = _check_fields_exist($params, [qw/Options/], def => 1, type => 'structure')){
                return @fields_exist_errors;
            }

            if (defined $params->{Options}{Currency}) {
                return ('BadCurrency') unless is_valid_currency($params->{Options}{Currency});
            }
    }

    return _validate_mediaplan_keyword($self, $params)
}


sub _validate_mediaplan_keyword
{
    my ($self, $params) = @_;
    my $type = 'keyword';
    my %possible_criteria;

    %possible_criteria = map {$_ => 1} qw /CampaignIDS MediaplanAdIDS MediaplanKeywordIDS/;

    my $list_key = 'MediaplanKeywords';

    my $key = 'MediaplanKeywordID';
    my $source_key = 'SourceKeywordID';

    return ('BadRequest') if ($params && ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist(
            $params, [qw/Action/],
            def => 1, type => 'list', list => [keys %APICommon::AVAILABLE_MEDIAPLAN_KEYWORDS_ACTIONS])){
        return @fields_exist_errors;
    }

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/SelectionCriteria/], def => 1, type => 'structure')){
            return @fields_exist_errors;
        }

        if (! scalar grep {$possible_criteria{$_} && $params->{SelectionCriteria}{$_}} keys %{$params->{SelectionCriteria}}) {
            return ('BadParams', iget("Один из следующих критериев должен быть указан: %s", join ', ', sort keys %possible_criteria))
        }

        foreach my $key (sort keys %possible_criteria) {
            if ($params->{SelectionCriteria}{$key}) {
                if (my @array_errors = _check_array($params->{SelectionCriteria}{$key}, $key, type => 'unsignedint', not_empty => 1)){
                    return @array_errors;
                }
            }
        }

        if (exists $params->{SelectionCriteria}{CampaignIDS}) {
            
            # проверяем на существование кампании
            my $valid_cids = get_hashes_hash_sql(PPC(cid =>  $params->{SelectionCriteria}{CampaignIDS}), 
                ["select cid from campaigns", where => {cid => SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet']}] );

            my @cids = uniq @{ $params->{SelectionCriteria}{CampaignIDS} };
            foreach my $cid (@cids) {
                unless ($valid_cids->{$cid}) {
                    return ('BadCampaignID', iget('Кампания %s не существует', $cid));
                }
            }            
        }

    } elsif ($params->{Action} eq 'Add' || $params->{Action} eq 'Update') {

        foreach my $p (@{$params->{$list_key}}) {
            my $result_object = {};
            if (my @fields_exist_errors = _check_fields_exist($p, [qw/Position/], def => 1, type => 'list',
                            list => [sort(uniq(values %API::Filter::MEDIAPLAN_POSITIONS))]) # скобки нужны иначе uniq внутри sort не работает
            ){
                push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
            }

            if (my @fields_exist_errors = _check_fields_exist($p, [qw/Phrase/], def => 1, type => 'phrase')){
                push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
            }

            if ($params->{Action} eq 'Add') {
                if (my @fields_exist_errors = _check_fields_exist($p, [qw/MediaplanAdID/], def => 1, type => 'unsignedint')){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }
                if (my @fields_exist_errors = _check_fields_exist($p, [$source_key], type => 'unsignedint')){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }
            } elsif ($params->{Action} eq 'Update') {

                if (my @fields_exist_errors = _check_fields_exist($p, [$key], def => 1, type => 'unsignedint')){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }
            }

            push @{$self->{ret}}, $result_object;
        }
    }

    return;
}


sub _validate_mediaplan_ad_wrapper
{
    my ($self, $params) = @_;
    return ('BadRequest') if ($params && ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist(
            $params, [qw/Action/],
            def => 1, type => 'list', list => [keys %APICommon::AVAILABLE_MEDIAPLAN_AD_ACTIONS])){
        return @fields_exist_errors;
    }

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/SelectionCriteria/], def => 1, type => 'structure')){
            return @fields_exist_errors;
        }

        my %possible_criteria = map {$_ => 1} qw /CampaignIDS MediaplanAdIDS/;

        if (! scalar grep {$possible_criteria{$_} && $params->{SelectionCriteria}{$_}} keys %{$params->{SelectionCriteria}}) {
            return ('BadParams', iget("Один из следующих критериев должен быть указан: %s", join ', ', sort keys %possible_criteria))
        }

        foreach my $key (sort keys %possible_criteria) {
            if ($params->{SelectionCriteria}{$key}) {
                if (my @array_errors = _check_array($params->{SelectionCriteria}{$key}, $key, type => 'unsignedint', not_empty => 1)){
                    return @array_errors;
                }
            }
        }

        if (defined $params->{SelectionCriteria}{CampaignIDS}) {
            
            # проверяем на существование кампании
            my $valid_cids = get_hashes_hash_sql(PPC(cid =>  $params->{SelectionCriteria}{CampaignIDS}),
                    ["select cid from campaigns", where => {cid =>  SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet']}] );

            my @cids = uniq @{ $params->{SelectionCriteria}{CampaignIDS} };
            foreach my $cid (@cids) {
                unless ($valid_cids->{$cid}) {
                    return ('BadCampaignID', iget('Кампания %s не существует', $cid));
                }
            }            
        }

    } elsif ($params->{Action} eq 'Add' || $params->{Action} eq 'Update') {
        local $Direct::Validation::SitelinksSets::SITELINKS_NUMBER = 4;

        for (my $i = 0; $i < scalar @{$params->{MediaplanAds}}; $i++) {
            my $banner = $params->{MediaplanAds}[$i];

            my $result_object = {};

            # TODO: код частично дублирует валидацию баннера, можно вынести в общую

            if ($params->{Action} eq 'Add') {
                if (my @fields_exist_errors = _check_fields_exist($banner, [qw/CampaignID/], type => 'int', not_empty => 1, def => 1 )){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }
                if (my @fields_exist_errors = _check_fields_exist($banner, [qw/SourceAdID/], type => 'int')){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }

            } elsif ($params->{Action} eq 'Update') {
                if (my @fields_exist_errors = _check_fields_exist($banner, [qw/MediaplanAdID/], type => 'int', not_empty => 1, def => 1 )){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }
            }
            # -- Хотя в документации и сказано, что CampaignID должен присутствовать только при Action=Add
            # -- но он по прежнему является частью структуры MediaplanAd, а значит указать его ни кто не запрещает
            if (defined $banner->{CampaignID}) {
                # проверяем на существование кампании
                my $valid_cids = get_hash_sql(PPC(cid =>  $banner->{CampaignID}), ["select cid, 1 from campaigns", where => {cid =>  SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet']}] );
                unless ($valid_cids->{$banner->{CampaignID}}) {
                    return ('BadCampaignID', iget('Кампания %s не существует', $banner->{CampaignID}));
                }
            }

            # общие проверки: проверям только при наличии
            foreach my $field (qw/Title Title2 Text/) {
                if (my @fields_exist_errors = _check_fields_exist($banner, [$field], type => 'string' )){
                    push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                }
            }

            if (my @fields_exist_errors = _check_fields_exist($banner, [qw/ContactInfo/], type => 'structure')){
                push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
            }

            # Валидация сайтлинков
            if (defined $banner->{Sitelinks}) {
                if (my @array_errors = _check_array($banner->{Sitelinks}, 'Sitelinks', type => 'structure')) {
                    push @{$result_object->{Errors}}, get_error_object(@array_errors);
                }

                unless (!scalar @{$banner->{Sitelinks}} || scalar @{$banner->{Sitelinks}} > 0 && scalar @{$banner->{Sitelinks}} <= $Direct::Validation::SitelinksSets::SITELINKS_NUMBER) {
                    push @{$result_object->{Errors}}, get_error_object('BadParams', iget("Массив %s должен содержать от 1 до %s элементов или быть пустым", 'Sitelinks', $Direct::Validation::SitelinksSets::SITELINKS_NUMBER));
                }

                foreach my $sl (@{$banner->{Sitelinks}}) {
                    if (my @fields_exist_errors = _check_fields_exist($sl, [qw/Title Href/], type => 'string', not_empty => 1 )){
                        push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
                    }
                }
            }

            if (my @fields_exist_errors = _check_fields_exist($banner, [qw/Geo/], type => 'string' )){
                push @{$result_object->{Errors}}, get_error_object(@fields_exist_errors);
            }

            if (defined $banner->{MinusKeywords}) {
                if (my @array_errors = _check_array($banner->{MinusKeywords}, 'MinusKeywords', type => 'string')){
                    push @{$result_object->{Errors}}, get_error_object(@array_errors);
                }

                if (my @minuswords_errors = @{MinusWords::check_minus_words($banner->{MinusKeywords}, type => 'banner')}) {
                    push @{$result_object->{Errors}}, get_error_object('BadMinusKeywords', join(' ', @minuswords_errors));
                }
            }

            if ($banner->{ContactInfo} && defined $banner->{ContactInfo}{MetroStationID}) {
                push @{$result_object->{Errors}}, get_error_object('BadParams', iget('Не указан город в контактной информации')) unless $banner->{ContactInfo}{City};
                push @{$result_object->{Errors}}, get_error_object('BadParams', iget('Неправильно указана станция метро')) unless (validate_metro($banner->{ContactInfo}{MetroStationID}, $banner->{ContactInfo}{City}));
            }

            push @{$self->{ret}}, $result_object;
        }
    }
    return;
}

sub _validate_mediaplan_wrapper
{
    my ($self, $params) = @_;
    return ('BadRequest') if ($params && ref($params) ne 'HASH');

    if (my @fields_exist_errors = _check_fields_exist(
            $params, [qw/Action/],
            def => 1, type => 'list', list => [keys %APICommon::AVAILABLE_MEDIAPLAN_ACTIONS])){
        return @fields_exist_errors;
    }

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Finish') {
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/SelectionCriteria/], def => 1, type => 'structure')){
            return @fields_exist_errors;
        }

        my %possible_criteria = map {$_ => 1} qw /CampaignIDS/;

        if (! scalar grep {$possible_criteria{$_} && $params->{SelectionCriteria}{$_}} keys %{$params->{SelectionCriteria}}) {
            return ('BadParams', iget("Один из следующих критериев должен быть указан: %s", join ', ', sort keys %possible_criteria))
        }

        foreach my $key (sort keys %possible_criteria) {
            if ($params->{SelectionCriteria}{$key}) {
                if (my @array_errors = _check_array($params->{SelectionCriteria}{$key}, $key, type => 'unsignedint', not_empty => 1)){
                    return @array_errors;
                }
            }
        }

        if (defined $params->{SelectionCriteria}{CampaignIDS}) {
            
            # проверяем на существование кампании
            my $valid_cids = get_hashes_hash_sql(PPC(cid =>  $params->{SelectionCriteria}{CampaignIDS}),
                ["select cid from campaigns", where => {cid =>  SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet']}] );

            my @cids = uniq @{ $params->{SelectionCriteria}{CampaignIDS} };
            foreach my $cid (@cids) {
                unless ($valid_cids->{$cid}) {
                    return ('BadCampaignID', iget('Кампания %s не существует', $cid));
                }
            }            
        }

        if ($params->{Action} eq 'Finish') {

            if (my @fields_exist_errors = _check_fields_exist($params, [qw/Options/], def => 1, type => 'structure')){
                return @fields_exist_errors;
            }

            if (my @fields_exist_errors = _check_fields_exist($params->{Options}, [qw/Comment/], def => 1, not_empty => 1, type => 'string')){
                return @fields_exist_errors;
            }
        }
    }

    return;
}

=head2 _validate_banners_operation

    Валидация операций с баннерами:
        ModerateBanners

    Принимает 2 параметра:
        BannerIDS
        CampaignID`

=cut

sub _validate_banners_operation {

    my ($self, $params) = @_;
    return ('BadRequest') if (ref($params) ne 'HASH');
    my $BANNERS_LIMIT = 2000;

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/BannerIDS/], def => 1)) {
        return @fields_exist_errors;
    }

    if (my @array_errors = _check_array($params->{BannerIDS}, 'BannerIDS', type => 'int', max => $BANNERS_LIMIT)) {
        return @array_errors;
    }

    # валидируем идентификаторы
    foreach my $bid (@{$params->{BannerIDS}}) {
        if ( !validate_id( { bid => $bid } ) ) {
            return ('BadBannerID', "".$bid);
        }
    }

    # проверяем существование объявлений
    my $bids = get_bid2cid(bid => $params->{BannerIDS});

    my @not_exists_bids = grep { $_ && !exists $bids->{$_} }
                            @{ $params->{BannerIDS} };

    return ('BadBannerID', iget("Объявления не найдены: %s", join(", ", @not_exists_bids))) if @not_exists_bids;

    return; #ok

}

sub _validate_campaign_request_currency {
    my ($self, $params) = @_;

    my @cids = $params->{CampaignID} ? ($params->{CampaignID}) : ( $params->{CampaignIDS} ? @{$params->{CampaignIDS}} : () );

    return unless @cids;

    if (exists $params->{Currency} && $self->{api_version_full} <= 4 ) {
        delete $params->{Currency};
    }

    # multicurrency
    if ($self->{api_version_full} > 4) {

        my $camp_currencies = get_hashes_hash_sql(PPC(cid => \@cids), ["select DISTINCT IFNULL(currency, 'YND_FIXED') currency from campaigns", where => { cid => SHARD_IDS } ]);

        if (my @array_errors = _validate_campaigns_currency($self, $params, $camp_currencies)) {
            return @array_errors;
        }
    }

    return;
}

# новая, правильная проверка идентификаторов баннеров и кампаний для --GetBanners-- и GetBannersTags
sub _validate_banners_request_new
{
    my ($self, $params, %O) = @_;

    my $CAMPAIGNS_LIMIT = 10;
    my $BANNERS_LIMIT = 2000;
    my ($cids, $bids) = ($params->{CampaignIDS}, $params->{BannerIDS});
    if ($bids) {
        if (my @bids_errors = _check_array($bids, 'BannerIDS', type => 'positiveint', max => $BANNERS_LIMIT)) {
            return @bids_errors;
        }
        if(get_one_field_sql(PPC(bid => $bids), ["SELECT count(*) FROM deleted_banners" , where => { bid => SHARD_IDS}])){
            return ('BadBanner', iget('Баннеры были удалены'));
        }
        # если баннеры удалены (в том числе из метабазы), то получить cid в неизвестном шарде по bid'ам не получится и вернем "Нет прав"
        # в документации мы не обещам именно ошибку "Баннеры были удалены" в этом методе
        if (scalar @$bids) {
            my $cids = get_cids(bid => $bids);
            if (!@$cids || check_rbac_rights( $self, 'api_showCamp', {cid => $cids, UID => $self->{uid}})) {
                return ( 'NoRights' );
            }
        }
    }

    if ($cids) {
        #в противном случае смотрим кампании
        if (my @cids_errors = _check_array($cids, 'CampaignIDS', type => 'positiveint', max => $CAMPAIGNS_LIMIT)) {
            return @cids_errors;
        }

        my $type_check = validate_ids_detail({cids => $cids, campaign_kind => 'base'});

        if (defined $type_check->{malformed}) {
            return ('BadCampaignID');
        }

        if (defined $type_check->{not_found}) {
            return ('NoRights');
        }

        return ('NoRights') if check_rbac_rights($self, 'api_showCamp', {cid => $cids, UID => $self->{uid}, uid => $self->{uid}});

        if (defined $type_check->{not_supported}) {
            return ('NotSupported');
        }
    }

    if ((! defined $bids || ! scalar @$bids) && (! defined $cids || ! scalar @$cids) ) {
        return ('BadParams', iget('Одно из полей CampaignIDS или BannerIDS должно быть указано'));
    }
    return;
}

sub _validate_moderate_banners
{
    my ($self, $params) = @_;
    return ('BadRequest') if (ref($params) ne 'HASH');

    unless (defined $params->{BannerIDS} || defined $params->{CampaignID}) {
        return ('BadParams', iget('Одно из следующих полей должно быть указано: CampaignID, BannerIDS'));
    }

    if (defined $params->{BannerIDS}) {
        return _validate_banners_operation($self, $params);
    }

    if (defined $params->{CampaignID}) {

        # -- Check not wallet
        return ('BadCampaignID') if !validate_id( { cid => $params->{CampaignID}, check_exist => 1 } );

        if (my @fields_exist_errors = _check_fields_exist($params, [qw/CampaignID/], type => 'unsignedint', more_than => 0)) {
            return @fields_exist_errors;
        }
    }

    # проверка прав в методе

    return; #ok
}

=head2 _validate_get_normalized_keywords(self, params)

Леммер-нормализатор

=cut

sub _validate_get_normalized_keywords($$) {
    my ($self, $params) = @_;
    my $MAX_NORMALIZED_PHRASES = 500;
    if ($self->{rbac_login_rights}->{is_internal_user} || ($self->{application_id} eq $API::Settings::COMMANDER_APP_ID)) {
        return ('BadRequest') if (ref($params) ne 'HASH');
        if (my @fields_exist_errors = _check_fields_exist($params, [qw/Keywords/], def => 1)) {
            return (@fields_exist_errors);
        }
        my $phrases = $params->{Keywords};
        if (my @array_errors = _check_array($phrases, 'Keywords', not_empty => 1, type => 'string', max => $MAX_NORMALIZED_PHRASES)) {
            return @array_errors;
        }

        my $validation_result = base_validate_keywords($phrases);
        unless ($validation_result->is_valid) {
            my $errors = join '. ', @{$validation_result->one_error_description_by_objects};
            return ('BadParams', $errors);  
        }
    } else {
        return ('NoRights');
    }
    return;
}

=head2 _validate_stat_within_3years($end_date_ts)

    $end_date_ts - unix timestampt конец интервала за который запрашиваем статистику
    Метод проверяет, что конец периода за который выбираем статистику не старше 1ого числа текущего месяца 3 года назад.
    Возвращает ошибку массивом (код, текст), либо false

    https://st.yandex-team.ru/DIRECT-51668

=cut

sub _validate_stat_within_3years {
    my $end_date_ts = shift;

    return if $end_date_ts >= mysql2unix(Stat::Const::BS_STAT_START_DATE);
    return ('BadTimeInterval', iget("Нет данных за выбранный период. Статистика доступна за три года"));
}

# -- BUGFIX DIRECT-26428, DIRECT-56447
sub _check_agency_currency {
    my ($rbac, $Currency, $agency_client_id, $api_version_full, $is_geo_manager) = @_;

    my $ac = Agency::get_agency_allowed_currencies_hash($agency_client_id, is_direct => 1);
    my @available_currencies = keys(%{$ac});
    # sort нужен, ниже используется в тексте ошибки
    my @available_currencies_for_client = sort map { $_ eq 'YND_FIXED' ? 'cu' : $_ } @available_currencies;
    my $currencies_str = join(', ', @available_currencies_for_client);

    my $result = {};

    if (!defined $Currency || !$Currency || $Currency eq 'cu') {
        if ($is_geo_manager eq 'Yes') {
            # Геоконтекст не передает валюту вообще, поэтому определяем так:
            #   смотрим на рабочую валюту агентства:
            #       * если фишки, то смотрим на список доступных валют в Балансе:
            #           - если доступна ровно 1 валюта - создаем в субклиента в ней
            #           - если ни одной или несколько - возвращаем ошибку
            #       * не фишки - создаем субклиента в этой валюте

            my $agency_currency = get_client_currencies($agency_client_id);
            if ($agency_currency->{work_currency} eq 'YND_FIXED') {
                if (@available_currencies > 1) {
                    $result->{errors} = ['CantCreateClient', iget('Невозможно определить валюту клиента.')];
                } elsif (@available_currencies == 1) {
                    $Currency = $available_currencies[0];
                } else {
                    $result->{errors} = ['CantCreateClient', iget('Нет доступных валют')];
                }
            } else {
                $Currency = $agency_currency->{work_currency};
            }
        } else {
            # -- Если параметер не передан или переданы фишки, то:
            # --     если валюта единственная, создаём в ней
            # --     если валют несколько, отдаём ошибку
            if (@available_currencies > 1) {
                if ($api_version_full > 4) {
                    $result->{errors} = iget("Поле %s должно содержать значения: %s", 'Currency', $currencies_str);
                } else {
                    $result->{errors} = ['CantCreateClient', iget('Невозможно определить валюту клиента. Используйте метод версии Live 4.')];
                }
            } elsif (@available_currencies == 1) {
                $Currency = $available_currencies[0];
            } else {
                $result->{errors} = ['CantCreateClient', iget('Нет доступных валют')];
            }
        }
    }

    if (!exists $result->{errors} && !exists $ac->{$Currency}) {
        # валюта недоступна
        if (@available_currencies > 1) {
            $result->{errors} = iget("Поле %s должно содержать значения: %s", 'Currency', $currencies_str);
        } else {
            $result->{errors} = iget('Поле Currency должно быть пустым или содержать: %s', $currencies_str);
        }
    }

    $result->{Currency} = $Currency;

    return $result;
}

sub _validate_sms_notification {
    my ($self, $params, %O) = @_;
    
    # -- в AccountManagement приезжают не все поля, а лишь часть, для них используется lite версия
    my $lite = defined $O{'lite'} ? 1: 0;

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/SmsNotification/], type => 'structure')){
        return (@fields_exist_errors);
    }

    my $has_SmsNotification = defined $params->{SmsNotification}->{SmsTimeFrom} || defined $params->{SmsNotification}->{SmsTimeTo} ? 1: 0;

    if ($has_SmsNotification) {
        for my $field (qw/SmsTimeFrom SmsTimeTo/) {
            if ($params->{SmsNotification}->{$field}) {
                if ($params->{SmsNotification}->{$field} =~ /^(\d\d)\:?(\d\d)$/) {
                    return ('BadParams', iget('Поле %s некорректно', $field)) if ($1 > 24 || ($2 % 15 > 0) || $2 > 45);
                } else {
                    return ('BadParams', iget('Поле %s некорректно', $field));
                }
            }
        }
    }

    my @check_fields = $lite ? qw(MoneyOutSms MoneyInSms PausedByDayBudgetSms) : qw(MoneyOutSms MoneyInSms ModerateResultSms MetricaSms);

    for (@check_fields) {
        if( defined $params->{SmsNotification}->{$_}){
            return ('NotYesNo', iget('Поле %s в структуре %s', $_, 'SmsNotification')) if(! is_yes_no($params->{SmsNotification}->{$_}) );
        }
    }
    
    return;
}

sub _validate_email_notification {
    my ($self, $params, %O) = @_;

    # -- в AccountManagement приезжают не все поля, а лишь часть, для них используется lite версия
    my $lite = defined $O{'lite'} ? 1 : 0;

    if (my @fields_exist_errors = _check_fields_exist($params, [qw/EmailNotification/], type => 'structure')){
        return @fields_exist_errors;
    }

    if (exists $params->{EmailNotification}) {
        my $email_notification = $params->{EmailNotification};

        my @check_fields = $lite ? qw(SendWarn PausedByDayBudget) : qw(SendAccNews SendWarn);
        for (@check_fields) {
            if (exists $email_notification->{$_} && defined $email_notification->{$_}
                && !is_yes_no($email_notification->{$_})) {
                return ('NotYesNo', iget('Поле %s в структуре %s', $_, 'EmailNotification'));
            }
        }

        if ( ! $lite ||                                               # для CreateOrUpdateCampaign
             $lite && exists $email_notification->{MoneyWarningValue} # для AccountManagement
        ) {
            my $MoneyWarningValue = int($email_notification->{MoneyWarningValue});
            if ($MoneyWarningValue < 1 || $MoneyWarningValue > 50) {
                return ('BadParams', iget('Значение поля %s в структуре %s должно быть от 1 до 50', 'MoneyWarningValue', 'EmailNotification'));
            }
        }

        if (! $lite && $email_notification->{WarnPlaceInterval} !~ /(15|30|60)/) { # для CreateOrUpdateCampaign
            return ('BadParams', iget('Поле %s в структуре %s должно содержать значения: 15, 30 или 60', 'WarnPlaceInterval', 'EmailNotification'));
        }
    }

    return;
}

1;
