package API::Validate::Ids;
use Direct::Modern;

use List::MoreUtils 'uniq';

use Yandex::DBShards;
use Yandex::DBTools;

use Campaign::Types;
use Settings;

use parent 'Exporter';
our @EXPORT_OK = qw( validate_id validate_ids_detail );

=head3 validate_id

=cut

sub validate_id
{
    my $vars = shift;                # можно указать { cid => .. }, либо { bid => .. }

    if( defined $vars->{cid} ) {
        return 0 unless ($vars->{cid} =~ /^\d+$/
                        && $vars->{cid});
    } elsif( defined $vars->{bid} ) {
        unless ($vars->{bid} =~ /^\d+$/ && $vars->{bid}) {
            return 0;
        }
    }

    if( defined $vars->{ check_exist }) {
        my $exist;
        if( defined $vars->{cid} ) {
            $exist = get_one_field_sql( PPC(cid => $vars->{cid}), ["SELECT 1 FROM campaigns", where => { cid => SHARD_IDS, statusEmpty => 'No', type__not_in => ['wallet']}, "LIMIT 1" ] );
        } elsif ( defined $vars->{bid} ) {
            $exist = get_one_field_sql( PPC(bid => $vars->{bid}), ["SELECT 1 FROM banners", where => { bid => SHARD_IDS }, "LIMIT 1" ] );
        }
        return 1 if defined $exist && $exist;
    return 0;
    }

    return 1;
}

=head2 validate_ids_detail

    переделка validate_id, которая а) принимает на вход более одного id, б) при указании check_exist говорит, что не найдено и где
    на вход на данный момент допустимы: cids, bids
    check_existence — (true/false) проверить наличие в базе
    campaign_kind   — (kind) проверить, что тип кампании объекта соответствует переданному. Подключает проверку check_existence
    banner_types    — (string, ...) проверить, что тип баннера соответствует одному из переданных. Подключает проверку check_existence

=cut

sub validate_ids_detail
{
    my $vars = shift;

    my $response = {};

    my (@ids, @malformed_ids, @fine_ids);

    if (defined $vars->{cids}) {
        @ids = @{$vars->{cids}};
    } elsif (defined $vars->{bids}) {
        @ids = @{$vars->{bids}};
    }

    foreach my $id (uniq @ids) {
        if ($id && $id =~ /^\d+$/) {
            push @fine_ids, $id;
        } else {
            push @malformed_ids, $id;
        }
    }

    $response->{malformed} = \@malformed_ids if @malformed_ids;
    return $response unless @fine_ids;

    my $check_campaign_kind = defined $vars->{cids} && defined $vars->{campaign_kind};
    my $check_banner_type   = defined $vars->{bids} && defined $vars->{banner_types};

    my %allowed_banner_types;
    if ($check_banner_type) {
        %allowed_banner_types = map { $_ => 1 } @{ $vars->{banner_types} };
    }

    if ($vars->{check_existence} || $check_campaign_kind || $check_banner_type) {
        my $id_type;
        if (defined $vars->{cids}) {
            $id_type = get_hash_sql(PPC(cid => \@fine_ids), ['SELECT cid, type FROM campaigns', where => {cid => SHARD_IDS, type__not_in => ['wallet']}]);
        } elsif (defined $vars->{bids}) {
            $id_type = get_hash_sql(PPC(bid => \@fine_ids), ['SELECT bid, banner_type FROM banners', where => {bid => SHARD_IDS}]);
        }

        my @ids_not_found;
        my @ids_not_supported;
        my @ids_ok;
        foreach my $id (@fine_ids) {
            if (exists $id_type->{$id}) {
                my $type = $id_type->{$id};
                if ($check_campaign_kind) {
                    if (camp_kind_in(type => $type, $vars->{campaign_kind})) {
                        push @ids_ok, $id;
                    } else {
                        push @ids_not_supported, $id;
                    }
                } elsif ($check_banner_type) {
                    if (exists $allowed_banner_types{$type}) {
                        push @ids_ok, $id;
                    } else {
                        push @ids_not_supported, $id;
                    }
                } else {
                    push @ids_ok, $id;
                }
            } else {
                push @ids_not_found, $id;
            }
        }

        $response->{ok}            = \@ids_ok || [];
        $response->{not_found}     = \@ids_not_found     if @ids_not_found;
        $response->{not_supported} = \@ids_not_supported if @ids_not_supported;
    } else {
        $response->{ok} = \@fine_ids || [];
    }

    return $response;
}

1;
