package API::Validate::Structure;
use Direct::Modern;

use Hash::Util;
use List::Util qw( any none );

use Yandex::I18n;

use API::Errors;
use API::Methods::Keyword::Limits;
use API::ValidateTools;

=pod

    type => structure, array
    request => структура верхнего уровня
        mandatory => обязательна или нет
    если type == structure
        fields => {} хэш возможных полей
        at_least_one => [] хотя бы одно поле из списка

    если type == array
        element_type => type тип поля из массива
        (все возможные ключи проверок для check_array)

    Field:
        массив структур:
            type => тип поля
            (все возможные ключи проверок для check_fields_exist)
            options => {} дополнительные опции для рекурсивного вызова
            mandatory => обязательно или нет
            condition => при каких условиях проверяется

    Condition =>
        role__ne => [список ролей] роль оператора не из списка
        role => [список ролей] роль оператора из списка
        field__in => [поле, [список значений]] указанное поле структуры этого уровня в значении из списка
        field__not_in => [поле, [список значений]] указанное поле структуры этого уровня в значении не из списка

=cut

no warnings 'once';

# TODO: condition in mandatory request
our %INPUT_STRUCTURES = (
    GetRetargetingGoals => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    options => {not_empty => 1, max => 100, element_type => 'login'},
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                    }
                },
            ]
        }
    },

    RetargetingCondition => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Action => [
                {
                    type => 'list',
                    list => ['Get', 'Delete', 'Update', 'Add'],
                    mandatory => 1,
                },
            ],
            RetargetingConditions => [
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, element_type => 'RetargetingConditionsElement_add'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Add']],
                    }
                },
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, element_type => 'RetargetingConditionsElement_update'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Update']],
                    }
                },
            ],
            SelectionCriteria => [
                {
                    type => 'RetargetingCondition_SelectionCriteria',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get', 'Delete']],
                        role__ne => ['client'],
                    },
                },
                {
                    type => 'RetargetingCondition_SelectionCriteria_client',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get', 'Delete']],
                        role => ['client'],
                    }
                },
            ],
        },
    },

    RetargetingCondition_SelectionCriteria => {
        type => 'structure',
        at_least_one => ['Logins', 'RetargetingConditionIDS'],
        at_least_one_not_empty_array => ['Logins', 'RetargetingConditionIDS'],
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    options => {max => 100, element_type => 'login'},
                },
            ],
            RetargetingConditionIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint'},
                }
            ],
        }
    },

    RetargetingCondition_SelectionCriteria_client => {
        type => 'structure',
        fields => {
            RetargetingConditionIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint'},
                }
            ],
        }
    },

    ArrayOf => {
        type => 'array',
    },

    RetargetingConditionsElement_add => {
        type => 'structure',
        fields => {
            RetargetingConditionName => [
                {
                    type => 'string',
                    not_empty => 1,
                    mandatory => 1,
                },
            ],
            RetargetingConditionDescription => [
                {
                    type => 'string',
                }
            ],
            Login => [
                {
                    type => 'login',
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                    },
                }
            ],
            RetargetingCondition => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'RetargetingConditionElement', not_empty => 1},
                    mandatory => 1,
                }
            ],
        }
    },

    RetargetingConditionsElement_update => {
        type => 'structure',
        fields => {
            RetargetingConditionID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            RetargetingConditionName => [
                {
                    type => 'string',
                    not_empty => 1,
                },
            ],
            RetargetingConditionDescription => [
                {
                    type => 'string',
                }
            ],
            RetargetingCondition => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'RetargetingConditionElement', not_empty => 1},
                    mandatory => 1,
                    condition => {
                        array__contains => ['Fields', 'RetargetingCondition'],
                    }
                },
                {
                    type => 'ArrayOf',
                    options => {element_type => 'RetargetingConditionElement', not_empty => 1},
                    mandatory => 1,
                    condition => {
                        field__in => ['Fields', [undef]],
                    }
                },
            ],
            Fields => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'list', list => ['RetargetingConditionName', 'RetargetingConditionDescription', 'RetargetingCondition']},
                }
            ],
        }
    },

    RetargetingConditionElement => {
        type => 'structure',
        fields => {
            Type => [
                {
                    type => 'list',
                    list => ['not', 'or', 'all'],
                    mandatory => 1,
                },
            ],
            Goals => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'RetargetingConditionBase', not_empty => 1},
                    mandatory => 1,
                }
            ],
        },
    },

    RetargetingConditionBase => {
        type => 'structure',
        fields => {
            GoalID => [
                {
                    type => 'unsignedint',
                    positive => 1,
                    mandatory => 1,
                }
            ],
        },
    },

    RetargetingConditionMetrikaGoal => {
        type => 'structure',
        fields => {
            Time => [
                {
                    type => 'unsignedint',
                    max => $Settings::MAX_RETARGETING_GOAL_TIME_DAYS,
                    positive => 1,
                    mandatory => 1,
                }
            ],
            GoalID => [
                {
                    type => 'unsignedint',
                    positive => 1,
                    mandatory => 1,
                }
            ],
        },
    },

    RetargetingConditionAudience => {
        type => 'structure',
        fields => {
            GoalID => [
                {
                    type => 'unsignedint',
                    positive => 1,
                    mandatory => 1,
                }
            ],
        },
    },

    Retargeting => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Action => [
                {
                    type => 'list',
                    list => ['Get', 'Delete', 'Update', 'Add'],
                    mandatory => 1,
                },
            ],
            Login => [
                {
                    type => 'login',
                }
            ],
            Retargetings => [
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, element_type => 'RetargetingElement_add'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Add']],
                    }
                },
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, element_type => 'RetargetingElement_update'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Update']],
                    }
                },
            ],
            SelectionCriteria => [
                {
                    type => 'Retargeting_SelectionCriteria',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get', 'Delete']],
                    }
                },
            ],
            Options => [
                {
                    type => 'Retargeting_Options',
                    mandatory => 0,
                    condition => {
                        field__in => ['Action', ['Get']],
                    }
                }
            ]
        },
    },

    Retargeting_SelectionCriteria => {
        type => 'structure',
        at_least_one => ['AdIDS', 'RetargetingIDS', 'RetargetingConditionIDS'],
        at_least_one_not_empty_array => ['AdIDS', 'RetargetingIDS', 'RetargetingConditionIDS'],
        fields => {
            AdIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 1000},
                },
            ],
            RetargetingIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 1000},
                }
            ],
            RetargetingConditionIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 1000},
                }
            ],
        }
    },

    Retargeting_Options => {
        type => 'structure',
        fields => {
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                },
            ],
        }
    },

    RetargetingElement_add => {
        type => 'structure',
        fields => {
            AdID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                },
            ],
            RetargetingConditionID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            ContextPrice => [
                {
                    type => 'float',
                }
            ],
            StatusPaused => [
                {
                    type => 'yesno',
                }
            ],
            AutoBudgetPriority => [
                {
                    type => 'list',
                    list => [sort keys %Phrase::PRIORITY_VALUES],
                }
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                }
            ],
        }
    },

    RetargetingElement_update => {
        type => 'structure',
        fields => {
            RetargetingID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                },
            ],
            ContextPrice => [
                {
                    type => 'float',
                }
            ],
            StatusPaused => [
                {
                    type => 'yesno',
                }
            ],
            AutoBudgetPriority => [
                {
                    type => 'list',
                    list => [qw/High Medium Low/],
                }
            ],
            Fields => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'list', list => ['ContextPrice', 'StatusPaused', 'AutoBudgetPriority']},
                }
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                }
            ],
        }
    },

    AdImage => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Action => [
                {
                    type => 'list',
                    list => ['Get', 'Delete', 'Upload', 'CheckUploadStatus', 'UploadRawData', 'GetLimits'],
                    mandatory => 1,
                },
            ],
            AdImageRawData => [
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, element_type => 'AdImageRaw', max => 10},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['UploadRawData']],
                    }
                },
            ],
            AdImageURLData => [
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, element_type => 'AdImageURL', max => 10000},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Upload']],
                    }
                },
            ],

            SelectionCriteria => [
                {
                    type => 'AdImage_SelectionCriteria',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get']],
                    }
                },
                {
                    type => 'AdImage_SelectionCriteriaDelete',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Delete']],
                    }
                },
                {
                    type => 'AdImage_SelectionCriteriaCheckStatus',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['CheckUploadStatus']],
                    }
                },
                {
                    type => 'AdImage_SelectionCriteriaGetLimits',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['GetLimits']],
                    }
                },
            ],
        },
    },

    AdImage_SelectionCriteria => {
        type => 'structure',
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    mandatory => 1,
                    options => {max => 10, element_type => 'login', not_empty => 1},
                    condition => {
                        role__ne => ['client'],
                    }
                },
            ],
            AdImageHashes => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'string', max => 10000},
                }
            ],
            Assigned => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'yesno'},
                }
            ],

            Limit => [
                {
                    max => 10000,
                    type => 'unsignedint',
                }
            ],

            Offset => [
                {
                    type => 'unsignedint',
                }
            ],
        }
    },

    AdImage_SelectionCriteriaDelete => {
        type => 'structure',
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    mandatory => 1,
                    options => {max => 1, element_type => 'login', not_empty => 1},
                    condition => {
                        role__ne => ['client'],
                    }
                },
            ],
            AdImageHashes => [
                {
                    type => 'ArrayOf',
                    mandatory => 1,
                    options => {element_type => 'string', max => 10000},
                }
            ],
        }
    },

    AdImage_SelectionCriteriaCheckStatus => {
        type => 'structure',
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    options => {max => 10, element_type => 'login', not_empty => 1},
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                    }
                },
            ],
            AdImageUploadTaskIDS => [
                {
                    type => 'ArrayOf',
                    options => {max => 10000, element_type => 'unsignedint'},
                }
            ],

            Limit => [
                {
                    max => 10000,
                    type => 'unsignedint',
                }
            ],

            Offset => [
                {
                    type => 'unsignedint',
                }
            ],
        }
    },

    AdImage_SelectionCriteriaGetLimits => {
        type => 'structure',
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    options => {max => 1000, element_type => 'login', not_empty => 1},
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                    }
                },
            ],
        }
    },

    AdImageRaw => {
        type => 'structure',
        isResult => 1,
        fields => {
            Login => [
                {
                    type => 'string',
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                    }
                }
            ],
            RawData => [
                {
                    type => 'base64',
                    len => 15_000_000,
                    mandatory => 1,
                }
            ],
            Name => [
                {
                    type => 'string',
                    not_empty => 1,
                    mandatory => 1,
                }
            ],
        },
    },

    AdImageURL => {
        type => 'structure',
        fields => {
            Login => [
                {
                    type => 'string',
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                    }
                }
            ],
            URL => [
                {
                    type => 'url_ex_ip',
                    mandatory => 1,
                }
            ],
            Name => [
                {
                    type => 'string',
                    not_empty => 1,
                    mandatory => 1,
                }
            ],
        },
    },

    AdImageAssociation => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Action => [
                {
                    type => 'list',
                    list => ['Get', 'Set'],
                    mandatory => 1,
                },
            ],

            AdImageAssociations => [
                {
                    type => 'ArrayOf',
                    options => {isResult => 1, max => 10000, element_type => 'AdImageAssociationItem'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Set']],
                    }
                },
            ],

            SelectionCriteria => [
                {
                    type => 'AdImageAssociation_SelectionCriteria',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get']],
                        role__ne => ['client'],
                    },
                },
                {
                    type => 'AdImageAssociation_SelectionCriteria_client',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get']],
                        role => ['client'],
                    },
                },
            ],
        },
    },

    AdImageAssociation_SelectionCriteria => {
        type => 'structure',
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    options => {max => 10, element_type => 'login', not_empty => 1},
                    mandatory => 1,
                },
            ],
            AdImageHashes => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'string', max => 10000},
                }
            ],
            AdIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 10000},
                }
            ],

            CampaignIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 1000},
                }
            ],

            StatusAdImageModerate => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'list', list => ['New', 'Pending', 'Yes', 'No']},
                }
            ],

            Limit => [
                {
                    max => 10000,
                    type => 'unsignedint',
                }
            ],

            Offset => [
                {
                    type => 'unsignedint',
                }
            ],
        }
    },

    AdImageAssociation_SelectionCriteria_client => {
        type => 'structure',
        fields => {
            AdImageHashes => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'string'},
                }
            ],
            AdIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint'},
                }
            ],

            CampaignIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint'},
                }
            ],

            StatusAdImageModerate => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'list', list => ['New', 'Pending', 'Yes', 'No']},
                }
            ],

            Limit => [
                {
                    max => 10000,
                    type => 'unsignedint',
                }
            ],

            Offset => [
                {
                    type => 'unsignedint',
                }
            ],
        }
    },

    AdImageAssociationItem => {
        type => 'structure',
        fields => {
            AdID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            AdImageHash => [
                {
                    type => 'base64ya',
                    len => 22,
                }
            ],
        },
    },

    EnableSharedAccount => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Login => [
                {
                    type => 'login',
                    mandatory => 1,
                },
            ],
        }
    },

    AccountManagement => {
        type => 'structure',
        request => {
            mandatory => 1
        },
        fields => {
            Action => [
                {
                    type => 'list',
                    list => ['Deposit', 'TransferMoney', 'Invoice', 'Get', 'Update'],
                    mandatory => 1,
                    condition => {
                        role__ne => ['client']
                    }
                },
                {
                    type => 'list',
                    list => ['Deposit', 'Invoice', 'DepositByCard', 'CheckPayment', 'Get', 'Update'],
                    mandatory => 1,
                    condition => {
                        role => ['client']
                    }
                },
            ],
            PayMethodID => [
                {
                    mandatory => 1,
                    type => 'string',
                    not_empty => 1,
                    condition => {
                        field__in => ['Action', ['DepositByCard']],
                    }
                },
            ],
            CustomTransactionID => [
                {
                    type => 'string',
                    mandatory => 1,
                    not_empty => 1,
                    len => 32,
                    condition => {
                        field__in => ['Action', ['DepositByCard','CheckPayment']],
                    }
                },
            ],
            Version => [
                {
                    type => 'string',
                    not_empty => 1,
                    len => 32,
                    condition => {
                        field__in => ['Action', ['DepositByCard']],
                    }
                },
            ],
            Payments => [
                {
                    type => 'ArrayOf',
                    options => {not_empty => 1, max => 1, element_type => 'PaymentItemDepositByCard'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['DepositByCard']],
                    }
                },
                {
                    type => 'ArrayOf',
                    options => {max => 50, element_type => 'PaymentItemDeposit'},
                    mandatory => 1,
                    condition => {
                        role__ne => ['client'],
                        field__in => ['Action', ['Deposit']],
                    }
                },
                {
                    type => 'ArrayOf',
                    options => {max => 50, element_type => 'PaymentItemDepositClient'},
                    mandatory => 1,
                    condition => {
                        role => ['client'],
                        field__in => ['Action', ['Deposit']],
                    }
                },
                {
                    type => 'ArrayOf',
                    options => {max => 50, element_type => 'PaymentItemInvoice'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Invoice']],
                    }
                },
            ],
            Transfers => [
                {
                    type => 'ArrayOf',
                    options => {not_empty => 1, max => 1, element_type => 'TransferItem'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['TransferMoney']],
                    }
                },
            ],

            SelectionCriteria => [
                {
                    type => 'AccountManagement_SelectionCriteria',
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Get']],
                        role__ne => ['client'],
                    },
                },
                {
                    type => 'AccountManagement_SelectionCriteria_Client',
                    mandatory => 0,
                    condition => {
                        field__in => ['Action', ['Get']],
                        role => ['client'],
                    },
                },
            ],

            Accounts => [
                {
                    type => 'ArrayOf',
                    options => {max => 1000, element_type => 'AccountItem'},
                    mandatory => 1,
                    condition => {
                        field__in => ['Action', ['Update']],
                    }
                }
            ],
        },
    },

    PaymentItemDepositClient => {
        type => 'structure',
        fields => {
            AccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            Amount => [
                {
                    type => 'float',
                    mandatory => 1,
                    positive => 1,
                }
            ],
            Origin => [
                {
                    type => 'list',
                    list => ['Overdraft', 'YM'],
                    mandatory => 1,
                },
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                },
            ],
        },
    },

    PaymentItemDeposit => {
        type => 'structure',
        only_one => ['Origin', 'Contract'],
        fields => {
            AccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            Amount => [
                {
                    type => 'float',
                    mandatory => 1,
                    positive => 1,
                }
            ],
            Origin => [
                {
                    type => 'list',
                    list => ['Overdraft', 'YM'],
                    mandatory => 1,
                    condition => {
                        field__in => ['Contract', [undef]],
                    }
                },
            ],
            Contract => [
                {
                    type => 'contract',
                    mandatory => 1,
                    options => {type => 'contract'},
                    condition => {
                        field__in => ['Origin', [undef]],
                    }
                },
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                },
            ],
        },
    },

    PaymentItemDepositByCard => {
        type => 'structure',
        fields => {
            AccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            Amount => [
                {
                    type => 'float',
                    mandatory => 1,
                    positive => 1,
                }
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                },
            ],
        },
    },

    PaymentItemInvoice => {
        type => 'structure',
        fields => {
            AccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            Amount => [
                {
                    type => 'float',
                    mandatory => 1,
                    positive => 1,
                }
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                },
            ],
        },
    },

    TransferItem => {
        type => 'structure',
        fields => {
            FromAccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            ToAccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                }
            ],
            Amount => [
                {
                    type => 'float',
                    mandatory => 1,
                    positive => 1,
                }
            ],
            Currency => [
                {
                    type => 'currency'
                    , list => [sort keys %Currencies::_CURRENCY_DESCRIPTION]
                },
            ],
        },
    },

    AccountManagement_SelectionCriteria => {
        type => 'structure',
        fields => {
            Logins => [
                {
                    type => 'ArrayOf',
                    options => {max => 50, element_type => 'login', not_empty => 1},
                    condition => {
                        role__ne => ['client'],
                    }
                },
            ],
            AccountIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 100},
                }
            ],
        }
    },
    
    AccountManagement_SelectionCriteria_Client => {
        type => 'structure',
        fields => {
            AccountIDS => [
                {
                    type => 'ArrayOf',
                    options => {element_type => 'unsignedint', max => 100},
                }
            ],
        }
    },

    AccountItem => {
        type => 'structure',
        fields => {
            AccountID => [
                {
                    type => 'unsignedint',
                    mandatory => 1,
                },
            ],
            AccountDayBudget => [
                {
                    type => 'AccountDayBudgetInfo',
                }
            ],
            SmsNotification => [
                {
                    type => 'SmsNotificationInfo',
                }
            ],
            EmailNotification => [
                {
                    type => 'EmailNotificationInfo',
                }
            ],
        }
    },

    RearrangeKeywords => {
        type => 'structure',
        fields => {
            Keywords => [
                {
                    type => 'ArrayOf',
                    options => { element_type => 'string', not_empty => 1, max => $API::Methods::Keyword::Limits::MAX_KEYWORDS_PER_QUERY },
                    mandatory => 1,
                }
            ],
            "AutoMinusWords" => [
                {
                    type => 'yesno'
                }
            ]
        }
    },

);

use warnings 'once';

Hash::Util::lock_keys(%INPUT_STRUCTURES);

=head2 api_validate_structure

    Функция принимает
        $self,
        $params - данные которые нужно провалидировать
        $type - название группы правил из хеша %INPUT_STRUCTURES по которым нужно проверять данные
        $fieldname - если данные структура,  её название
        $rules - набор правил отличный от тех что есть в %INPUT_STRUCTURES
            усли используется своя структура $type можно не передавать

=cut
sub api_validate_structure {
    my ($self, $params, $type, $fieldname, $rules, %OPTIONS) = @_;

    $rules = {} unless $rules;
    $rules->{$type} = $INPUT_STRUCTURES{$type}
        if exists $INPUT_STRUCTURES{$type} && !exists $rules->{$type};
    return unless $rules && $rules->{$type};
    my $s = {%{$rules->{$type}}, %OPTIONS};
    my @result_errors;
    if ($s->{type} eq 'structure') {
        my @local_errors;
        if ($s->{request}) {
            push @local_errors, ['BadRequest'] if $s->{request}{mandatory} && (ref($params) ne 'HASH' || !$params);
            push @local_errors, ['BadRequest'] if !$s->{request}{mandatory} && defined $params && ref($params) ne 'HASH';
        } elsif (defined $params && ref $params ne 'HASH'){
            push @local_errors, ['BadParams', iget('Поле %s должно быть структурой', $fieldname || $type)];
        }

        if (! @local_errors) {
            if ($s->{at_least_one}) {
                if (none {$params->{$_} && (ref($params->{$_}) ne "ARRAY" || scalar @{$params->{$_}})} @{$s->{at_least_one}}) {
                    push @local_errors, ['BadParams', iget("Хотя бы одно из следующих полей: %s в структуре %s должно быть указано", join(', ', @{$s->{at_least_one}}), $fieldname)];
                }
            }
            if ($s->{at_least_one_not_empty_array}) {
                if (none {$params->{$_} && ref($params->{$_}) eq "ARRAY" && scalar @{$params->{$_}}} @{$s->{at_least_one}}) {
                    push @local_errors, ['BadParams', iget("Хотя бы одно из следующих полей: %s в структуре %s должно быть указано", join(', ', @{$s->{at_least_one}}), $fieldname)];
                }
            }
            if ($s->{only_one}) {
                if (scalar(grep {defined $params->{$_}} @{$s->{only_one}}) != 1) {
                    push @local_errors, ['BadParams', iget("Ровно одно из следующих полей: %s в структуре %s должно быть указано", join(', ', @{$s->{only_one}}), $fieldname)];
                }
            }

            for my $field (sort keys %{$s->{fields}}) {
                my $variants = $s->{fields}{$field};
                for my $opt (@$variants) {
                    my $allow_continue = 1;
                    if ($opt->{condition}) {
                        for my $p (keys %{$opt->{condition}}) {
                            my $val = $opt->{condition}{$p};
                            if (
                                   ( $p eq 'role__ne' && any {$self->{rbac_login_rights}{role} eq $_} @$val )
                                || ( $p eq 'role' && none {$self->{rbac_login_rights}{role} eq $_} @$val)
                                || ( $p eq 'field__in' && !is_fields_in($params, $val))
                                || ( $p eq 'field__not_in' && any {$params->{$val->[0]} eq $_ } @{$val->[1]})
                                || ( $p eq 'array__contains' && ( ref($params->{$val->[0]}) ne 'ARRAY' || none {$_ eq $val->[1]} @{$params->{$val->[0]}}))
                            ) {
                                $allow_continue = 0;
                            }
                        }
                    }
                    if ($allow_continue) {
                        my %check_field_opt = (
                            list => $opt->{list},
                            max => $opt->{max},
                            more_than => $opt->{more_than},
                            positive => $opt->{positive},
                            prefix => $fieldname ? $fieldname."." : "",
                            not_empty => $opt->{not_empty},
                            len => $opt->{len},
                        );
                        if ($opt->{mandatory}) {
                            $check_field_opt{def} = 1;
                        }

                        if (exists $rules->{$opt->{type}} || exists $INPUT_STRUCTURES{$opt->{type}} ) {

                            if (my @fields_exist_errors = _check_fields_exist($params, [$field], %check_field_opt )){
                                push @local_errors, \@fields_exist_errors;
                            }
                            
                            if (defined $params->{$field}) {
                                
                                if (my @errors = api_validate_structure($self, $params->{$field}, $opt->{type}, join(".", grep {$_} ($fieldname, $field)), $rules, %{$opt->{options} || {}})) {
                                    push @local_errors, @errors;
                                }
                            }
                        } else {
                            if (my @fields_exist_errors = _check_fields_exist($params, [$field], type => $opt->{type}, %check_field_opt )){
                                push @local_errors, \@fields_exist_errors;
                            }
                        }
                    }
                }
            }
        }
        if ($s->{isResult}) {
            for (@local_errors) {
                push @{$self->{ret}[0]{Errors}}, get_error_object(@$_);
            }
        } else {
            push @result_errors, @local_errors;
        }
    } elsif ($s->{type} eq 'array') {
        my %check_array_opt = (not_empty => $s->{not_empty}, max => $s->{max}, list => $s->{list});
        if (exists $INPUT_STRUCTURES{$s->{element_type}}) {
            my @array_errors = _check_array($params, $fieldname, %check_array_opt);
            if (scalar @array_errors) {
                push @result_errors, \@array_errors;
            } else {
                for (my $i = 0; $i < scalar(@$params); $i++){
                    my @local_errors;
                    my $item = $params->[$i];
                    if (!defined $item || ref $item ne 'HASH') {
                        push @local_errors, ['BadParams', iget('Поле %s должно быть структурой', $fieldname."[$i]")];
                    } elsif (my @errors = api_validate_structure($self, $item, $s->{element_type}, $fieldname."[$i]", $rules)) {
                        push @local_errors, @errors;
                    }

                    if ($s->{isResult}) {
                        for (@local_errors) {
                            push @{$self->{ret}[$i]{Errors}}, get_error_object(@$_);
                        }
                    } else {
                        push @result_errors, @local_errors;
                    }
                }
            }
        } else {
            if (my @array_errors = _check_array($params, $fieldname, type => $s->{element_type}, %check_array_opt)) {
                push @result_errors, \@array_errors;
            }
        }
    }
    return @result_errors;
}

=head2 is_fields_in($val,$params)

    Проверяет соотаетствие параметров условиям field__in
    принимает структуру с условиями и параметры
    Пример структуры с условиями:
    { 'PhrasesType' => ['Network', 'Both'],
      'Mode' => ['Wizard'],
    }
    ['Mode', ['Wizard']]
    Если у всех указанных в условиях полей, в параметрах необходимые значения
    возвращает 1 иначе 0

=cut
sub is_fields_in($$){
    my ($params, $fields) = @_;

    my %fields;
    if (ref $fields eq 'HASH') {
        %fields = %$fields;
    } elsif (ref $fields eq 'ARRAY'){
        %fields = @$fields;
    } else {
        croak('Условия field__in должны быть массивом или хешем');
    }
    for my $field ( keys %fields ){
        if (none {
                   (!defined($params->{$field}) && !defined($_))
                || (defined($params->{$field}) && defined($_) && $params->{$field} eq $_)
            } @{$fields{$field}}
        ) {
            return 0;
        }
    }
    return 1;
}

1;
