package API::ValidateRights;

=pod

    $Id $

    Модуль для проверки прав

=cut

use strict;
use warnings;
use List::MoreUtils qw/uniq/;

use API::Errors;
use Yandex::DBTools;
use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::I18n;
use Mediaplan;

use Settings;
use RBACDirect;
use APICommon qw(:subs);

use utf8;

sub validate_mediaplan_keyword_rights {
    return _validate_mediaplan_keyword_rights(@_);
}

sub _validate_mediaplan_keyword_rights {
    my ($self, $params) = @_;
    my $type = 'keyword';
    my $key = 'MediaplanKeywordID';
    my $keys = 'MediaplanKeywordIDS';
    my $list_key = 'MediaplanKeywords';
    my $error_token = 'BadMediaplanKeywordID';
    my $error_text = iget("Указанная ключевая фраза не существует");
    my $source_key = 'SourceKeywordID';

    my $camp_rights;
    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        # запрещаем менеджеру смотреть кампании не своих клиентов
        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/){
            my $cids = $params->{SelectionCriteria}{CampaignIDS};
            if (! $cids || ! scalar @$cids) {
                $cids = get_one_column_sql(PPC(mbid => \@{$params->{SelectionCriteria}{MediaplanAdIDS}}), ['select cid from mediaplan_banners',
                    where => {mbid => SHARD_IDS}]) || [];
            }
            if (!@$cids) {
                $cids = get_one_column_sql(PPC(shard => 'all'), ['select cid from mediaplan_bids',
                    where => {id => \@{$params->{SelectionCriteria}{$keys}}}]) || [];
            }

            if ( @$cids ) {
                if (!rbac_user_allow_edit_camp($self->{rbac}, $self->{uid}, $cids)) {
                    return ('NoRights');
                }
            }
        }
    } elsif ($params->{Action} eq 'Add') {
        my $existing_banners = get_hashes_hash_sql(PPC(mediaplan_bid => [map {$_->{MediaplanAdID}} @{$params->{$list_key}}]),
            ['select mb.mbid, c.cid, c.uid, c.archived from mediaplan_banners mb left join campaigns c using (cid)',
            where => {mbid => SHARD_IDS, 'c.statusEmpty__not_in' => ['Yes']}]);

        my $source_key2cid = get_hashes_hash_sql(PPC(shard => 'all'), ['select id, cid, uid from bids left join campaigns using (cid)',
            where => {id => [map {$_->{$source_key}} grep {$_->{$source_key} && $_->{$source_key} =~ /^\d+$/} @{$params->{$list_key}}]}]);

        my %cids = map {$_->{cid} => 1} values %$existing_banners;
        my $fa_statuses = get_fa_statuses([keys %cids]);

        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/ && scalar keys %cids) {
            $camp_rights = rbac_user_allow_edit_camps_detail($self->{rbac}, $self->{uid}, [keys %cids]);
        }

        for (my $i = 0; $i < scalar @{$params->{$list_key}}; $i++) {
            my $result_object = $self->{ret}[$i];
            my $p = $params->{$list_key}[$i];

            # пропускаем проверку прав, если не указан идентификатор, проверили это выше
            next unless defined $p->{MediaplanAdID};

            if (!$existing_banners->{$p->{MediaplanAdID}}) {
                push @{$result_object->{Errors}}, get_error_object('BadMediaplanAdID', iget("Указанный баннер не существует"));
            } else {
                if ($existing_banners->{$p->{MediaplanAdID}}{archived} eq 'Yes') {
                    push @{$result_object->{Errors}}, get_error_object('BadParams', iget("Кампания заархивирована"));
                }

                if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/) {
                    if (! $camp_rights->{ $existing_banners->{$p->{MediaplanAdID}}{cid} }) {
                        push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Нет прав на данную кампанию"));
                    }
                }

                # возможно, что не очень правильно. Проверяем, что uid баннера, куда добавляется фраза
                # совпадает с uid кампании фразы, которая названа источником.
                if ($p->{$source_key} && $source_key2cid->{$p->{$source_key}} && $source_key2cid->{$p->{$source_key}}{uid} != $existing_banners->{$p->{MediaplanAdID}}{uid}) {
                    push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Нет прав на данный источник"));
                }
                if ($existing_banners->{$p->{MediaplanAdID}}{cid} && $fa_statuses->{$existing_banners->{$p->{MediaplanAdID}}{cid}}) {
                    push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Кампания на Первой Помощи"));
                }
            }
        }

    } elsif ($params->{Action} eq 'Update') {
        my $existing_phrases = get_hashes_hash_sql(PPC(shard => 'all'), ['select id, cid, archived from mediaplan_bids left join campaigns using (cid)',
                where => {
                    id => [map {$_->{$key}} @{$params->{$list_key}}],
                    'statusEmpty__not_in' => ['Yes']
                }]);

        my %cids = map {$_->{cid} => 1} values %$existing_phrases;
        my $fa_statuses = get_fa_statuses([keys %cids]);

        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/ && scalar keys %cids) {
            $camp_rights = rbac_user_allow_edit_camps_detail($self->{rbac}, $self->{uid}, [keys %cids]);
        }

        for (my $i = 0; $i < scalar @{$params->{$list_key}}; $i++) {
            my $result_object = $self->{ret}[$i];
            my $p = $params->{$list_key}[$i];

            # пропускаем проверку прав, если не указан идентификатор, проверили это выше
            next unless defined $p->{$key};

            if (!$existing_phrases->{$p->{$key}}) {
                push @{$result_object->{Errors}}, get_error_object($error_token, $error_text);
            } else {
                if ($existing_phrases->{$p->{$key}}{archived} eq 'Yes') {
                    push @{$result_object->{Errors}}, get_error_object('BadParams', iget("Кампания заархивирована"));
                }
                if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/) {
                    if (! $camp_rights->{ $existing_phrases->{$p->{$key}}{cid} }) {
                        push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Нет прав на данную кампанию"));
                    }
                }
                if ($existing_phrases->{$p->{$key}}{cid} && $fa_statuses->{$existing_phrases->{$p->{$key}}{cid}}) {
                    push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Кампания на Первой Помощи"));
                }
            }

        }
    }
    return;#
}

sub validate_mediaplan_ad_rights {
    my ($self, $params) = @_;

    my $camp_rights;

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Delete') {
        my $cids = $params->{SelectionCriteria}{CampaignIDS};
        if (! $cids || ! scalar @$cids) {
            $cids = get_one_column_sql(PPC(mediaplan_bid => \@{$params->{SelectionCriteria}{MediaplanAdIDS}}),
                ['select cid from mediaplan_banners', where => {mbid => SHARD_IDS}]) || [];
        }
        my $uids = get_hashes_hash_sql(PPC(cid => $cids), ["select cid, uid from campaigns", where => {cid => SHARD_IDS, statusEmpty => 'No'}] );

        foreach my $cid (@$cids) {
            unless ($uids->{$cid}) {
                return ('BadCampaignID', iget('Кампания %s не существует', $cid));
            }
        }

        # запрещаем менеджеру смотреть кампании не своих клиентов
        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/) {
            if ( $cids && scalar @$cids ) {
                if (!rbac_user_allow_edit_camp($self->{rbac}, $self->{uid}, $cids)) {
                    return ('NoRights');
                }
            }
        }
    } elsif ($params->{Action} eq 'Add') {
        my $existing_campaigns = get_hashes_hash_sql(PPC(cid => [map {$_->{CampaignID}} @{$params->{MediaplanAds}}]),
            ['select cid, archived, statusEmpty from campaigns', where => {cid => SHARD_IDS, statusEmpty__not_in => ['Yes']}]);

        my $fa_statuses = get_fa_statuses([keys %$existing_campaigns]);

        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/ && scalar keys %$existing_campaigns) {
            $camp_rights = rbac_user_allow_edit_camps_detail($self->{rbac}, $self->{uid}, [keys %$existing_campaigns]);
        }

        for (my $i = 0; $i < scalar @{$params->{MediaplanAds}}; $i++) {
            my $result_object = $self->{ret}[$i];
            my $banner = $params->{MediaplanAds}[$i];

            # пропускаем проверку прав, если не указан идентификатор, проверили это выше
            next unless defined $banner->{CampaignID};

            if (!$existing_campaigns->{$banner->{CampaignID}}) {
                push @{$result_object->{Errors}}, get_error_object('BadCampaignID', iget("Кампания не существует"));
            } else {
                if ($existing_campaigns->{$banner->{CampaignID}}{archived} eq 'Yes') {
                    push @{$result_object->{Errors}}, get_error_object('BadParams', iget("Кампания заархивирована"));
                }

                if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/) {
                    if (! $camp_rights->{$banner->{CampaignID}}) {
                        push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Нет прав на данную кампанию"));
                    }
                }

                if ($banner->{CampaignID} && $fa_statuses->{$banner->{CampaignID}}) {
                    push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Кампания на Первой Помощи"));
                }
            }
        }


    } elsif ($params->{Action} eq 'Update') {
        my $existing_banners = get_hashes_hash_sql(PPC(mediaplan_bid => [map {$_->{MediaplanAdID}} @{$params->{MediaplanAds}}]),
            ['select mbid, cid, archived from mediaplan_banners left join campaigns using (cid)',
                where => {mbid => SHARD_IDS, statusEmpty__not_in => ['Yes']}]);

        my %cids = map {$_->{cid} => 1} values %$existing_banners;
        my $fa_statuses = get_fa_statuses([keys %cids]);

        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/ && scalar keys %cids) {
            $camp_rights = rbac_user_allow_edit_camps_detail($self->{rbac}, $self->{uid}, [keys %cids]);
        }

        for (my $i = 0; $i < scalar @{$params->{MediaplanAds}}; $i++) {
            my $result_object = $self->{ret}[$i];
            my $banner = $params->{MediaplanAds}[$i];

            # пропускаем проверку прав, если не указан идентификатор, проверили это выше
            next unless defined $banner->{MediaplanAdID};

            if (!$existing_banners->{$banner->{MediaplanAdID}}) {
                push @{$result_object->{Errors}}, get_error_object('BadMediaplanAdID', iget("Указанный баннер не существует"));
            } else {
                if ($existing_banners->{$banner->{MediaplanAdID}}{archived} eq 'Yes') {
                    push @{$result_object->{Errors}}, get_error_object('BadParams', iget("Кампания заархивирована"));
                }

                if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/) {
                    if (! $camp_rights->{ $existing_banners->{$banner->{MediaplanAdID}}{cid} }) {
                        push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Нет прав на данную кампанию"));
                    }
                }
                if ($existing_banners->{$banner->{MediaplanAdID}}{cid} && $fa_statuses->{$existing_banners->{$banner->{MediaplanAdID}}{cid}}) {
                    push @{$result_object->{Errors}}, get_error_object('NoRights', iget("Кампания на Первой Помощи"));
                }
            }
        }
    }

    return;#
}

sub validate_mediaplan_rights {
    my ($self, $params) = @_;

    if ($params->{Action} eq 'Get' || $params->{Action} eq 'Finish') {
        my $cids = $params->{SelectionCriteria}{CampaignIDS};
        my $uniq_cids = [uniq @$cids];
        my $existing_cids = get_one_column_sql(PPC(cid => $cids), ["select cid from campaigns", where => {cid => SHARD_IDS, statusEmpty => 'No'}]) || [];
        if (scalar @$uniq_cids != scalar @$existing_cids) {
            return ('NoRights');
        }
        if ($self->{rbac_login_rights}->{role} =~ /^(manager)$/) {
            if ( $cids && scalar @$cids ) {
                if (!rbac_user_allow_edit_camp($self->{rbac}, $self->{uid}, $cids)) {
                    return ('NoRights');
                }
            }
        }
    }
    return;
}

sub validate_mediaplan_allow {
    my ($self, $params) = @_;

    if ($self->{rbac_login_rights}->{role} !~ /^(super|media|manager|support|placer|superreader)$/) {
        return ('NoRights');
    }

    return;
}

sub validate_get_keywords_intersection_rights {
    my ($self, $params) = @_;
    my @cids_to_check;
    for my $field (qw/CampaignIDS WithCampaignIDS/) {
        if (defined $params->{$field}) {
            push @cids_to_check, @{get_one_column_sql(PPC(cid => $params->{$field}), [ "select cid from campaigns",  where => { cid => SHARD_IDS, statusEmpty => 'No'} ]) || []};
        }
    }
    if ( @cids_to_check && APICommon::check_rbac_rights( $self, 'api_showCamp', { cid => \@cids_to_check, UID => $self->{uid}, uid => $self->{uid} } ) ) {
        return ( 'NoRights' );
    }
    return;
}

sub validate_mediaplan_limited_rights {
    my ($self, $params) = @_;
    if ($self->{rbac_login_rights}{role} eq 'superreader' && $params->{Action} ne 'Get') {
        return ('NoRights', iget("Недопустимое действие"));
    }
    return;
}

=head2 validate_enable_shared_account

    Перевод на единый счет разрешен только агентствам, всем остальным через интерфейс

=cut


sub validate_enable_shared_account {
    my ($self, $params) = @_;
    # -- Включение ОС под супером DIRECT-26270
    if ($self->{rbac_login_rights}{role} ne 'agency' && $self->{rbac_login_rights}{role} ne 'super') {
        return ('NoRights', iget("Доступно только агентствам"));
    }
    return;
}

=head2 validate_invoice_rights(self, params)

    Проверка прав на оплату кампании, заполнение self->{_preprocessed}

=cut

sub validate_invoice_rights($$) {
    my ($self, $params) = @_;
    my @campaign_ids = map {$_->{CampaignID}} @{$params->{Payments}};

    my $campaigns = get_hashes_hash_sql(PPC(cid => \@campaign_ids),["
            select c.cid, IFNULL(c.currency, 'YND_FIXED') currency, c.type, c.wallet_cid, c.uid
                 , u.ClientID
            from campaigns c
            inner join users u ON c.uid = u.uid
         ", where => {cid => SHARD_IDS}]);
    my $client_uid; # проверка изначально была только для первого uid

    foreach my $payment(@{$params->{Payments}}) {
        if ((!defined ($campaigns->{$payment->{CampaignID}}))
            || ($campaigns->{$payment->{CampaignID}}->{type} && $campaigns->{$payment->{CampaignID}}->{type} eq 'wallet')) {
            return ('NoRights');
        }
        $client_uid = $campaigns->{$payment->{CampaignID}}->{uid};
    }

    my $cids_str = join (',', @campaign_ids);
    my $error_code = check_rbac_rights($self, 'api_payCamp', {UID => $self->{uid}, uid => $client_uid, rights => $self->{rbac_login_rights}, cid => $cids_str});
    return ('NoRights') if $error_code;

    $self->{_preprocessed} = $campaigns;
    return;
}

1;
