package API::ValidateTools;

=pod

    $Id $

    Модуль с функциями для проверок полей

=cut

use utf8;
use strict;
use warnings;
use Settings;
use IpTools;
use TextTools;
use geo_regions;
use GeoTools;

use Yandex::Validate;
use Yandex::DateTime;
use Yandex::IDN qw(is_valid_email is_valid_domain);
use Yandex::I18n;
use Yandex::URL;

use Direct::Validation::Keywords qw/base_validate_keywords/;

use Currencies;
use YaCatalogApi qw/validate_one_rubric/;
use Models::Phrase qw/validate_phrase_one_href_param/;
use vars qw($VERSION @ISA @EXPORT @EXPORT_OK);
use base qw(Exporter);
@EXPORT = qw(
    is_yes_no
    is_dec_percent

    _check_array
    _check_fields_exist
);
@EXPORT_OK = qw(
    is_dec
);

sub is_yes_no{
    my $var = shift;
    return $var =~ /^(Yes|No)$/ ? 1 : 0;
}

sub is_dec_percent {
    my ($var, %OPT) = @_;

    if ($OPT{not_zero}) {
        return is_dec($var, min_value => 10, max_value => 100);
    } else {
        return is_dec($var, min_value => 0, max_value => 100);
    }
}

=head2 is_dec(var; OPT)

    Проверить кратность var 10;
    по ключу max_value задаётся наибольшее допустимое значение, по min_value — наименьшее.

=cut

sub is_dec {
    my ($var, %OPT) = @_;

    return 0 unless is_valid_int($var, $OPT{min_value}, $OPT{max_value});

    return (($var % 10) ? 0 : 1);
}

########## Типовые проверки

=head2 _check_array

    Проверки корректности массива

    Параметры:
        array - ссылка на массив
        field_name - название поля
        checks - дополнительные проверки
            not_empty - не пустой массив
            max => [maximum] - не более maximum элементов
            type => [type] - проверка на тип элементов массива

=cut

sub _check_array($$;%) {

    my ($array, $field_name, %checks) = @_;

    if (ref($array) ne 'ARRAY') {
        return ('NotArray', iget('Поле %s должно быть массивом', $field_name));
    }

    if ($checks{not_empty} && scalar @{$array} == 0) {
        return ('EmptyArray', iget('Массив %s не должен быть пустым', $field_name));
    }

    # валидируем количество элементов в массиве идентификаторов
    if ( $checks{max} && scalar @{$array} > $checks{max} ) {
        return ('LongMass', iget('Массив %s должен содержать не более %s элементов', $field_name, $checks{max} ));
    }

    # валидируем тип данных массива
    if ($checks{type}) {
        if ($checks{type} eq 'positiveint') {
            foreach (@{$array}) {
                return ('BadParams', iget('Массив %s должен содержать положительные значения типа integer', $field_name)) if ($_ == 0 || $_ !~ /^\d+$/);
            }
        } elsif ($checks{type} eq 'unsignedint') {
            foreach (@{$array}) {
                return ('BadParams', iget('Массив %s должен содержать значения типа unsigned integer', $field_name)) if $_ !~ /^\d+$/;
            }
        } elsif ($checks{type} eq 'int') {
            foreach (@{$array}) {
                return ('BadParams', iget('Массив %s должен содержать значения типа integer', $field_name)) if ! is_valid_int($_);
            }
        } elsif ($checks{type} eq 'string') {
            foreach (@{$array}) {
                return ('BadParams', iget('Массив %s должен содержать значения типа string', $field_name)) if ref($_);
            }
        } elsif ($checks{type} eq 'structure') {
            foreach (@{$array}) {
                return ('BadParams', iget('Поле %s должно быть массивом структур', $field_name)) if ref($_) ne 'HASH';
            }
        } elsif ($checks{type} eq 'login') {
            foreach (@{$array}) {
                return ('BadLogin', iget('Поле %s должно быть массивом валидных логинов', $field_name)) if !validate_login( $_, lite => 1 );
            }
        } elsif ($checks{type} eq 'dec_percent') {
            foreach (@{$array}) {
                return ('NotDecPercentInt', iget('Массив %s', $field_name)) if ! is_dec_percent($_);
            }
        } elsif ($checks{type} eq 'dec') {
            foreach (@{$array}) {
                return ('BadArrayValues', iget('Значения массива %s не кратны 10 или не из допустимого промежутка', $field_name))
                    unless is_dec($_, max_value => $checks{max_value}, min_value => $checks{min_value});
            }
        } elsif ($checks{type} eq 'geo') {
            my $geo_error = validate_geo($array);
            if ($geo_error) {
                return ('BadGeo', iget('Массив %s', $field_name));
            }
        } elsif ($checks{type} eq 'geo_unsigned') {
            foreach (@{$array}) {
                return ('BadGeo', iget('Массив %s', $field_name)) if $_ !~ /^\d+$/ || !exists $geo_regions::GEOREG{abs($_)};
            }
        } elsif ($checks{type} eq 'campaignid') {
            foreach (@{$array}) {
                unless ( $_ =~ /^\d+$/ ) {
                    return ('BadCampaignID');  
                }
            }
        } elsif ($checks{type} eq 'yesno') {
            foreach (@{$array}) {
                if ( !is_yes_no($_) ) {
                    return ('BadParams', iget('Массив %s должен содержать значения Yes или No', $field_name));  
                }
            }
        } elsif ($checks{type} eq 'list') {
            my %list = map {$_ => 1} @{$checks{list}};
            foreach (@{$array}) {
                if ( ! defined $list{$_} ) {
                    return ('BadParams', iget('Массив %s должен содержать значения: %s', $field_name, join(', ', @{$checks{list}})));  
                }
            }
        }
    }

    return; #ok
}

=head2 _check_fields_exist

    Простая проверка на существование списка полей
    Параметры:
        params - хэш параметров
        fields - массив проверяемых полей
        checks - проверки
            def - поле существует
            positive - поле больше 0
            more_than - поле больше чем X
            not_empty - поле определено
            max - поле не больше чем X
            type - тип поля
                string
                    len - длинна не больше Х
                    not_empty - строка не пустая и не состоит из пробелов
                unsignedint
                int
                float
                yesno
                email
                date_no_future
                date
                login
                phone
                phrase-params
                structure
                url
                url_ex_ip - url в котором может быть ip-адрес
                timestamp
                locale
                list
                phrase
                rubric
                currency

=cut

sub _check_fields_exist {

    my ($params, $fields, %checks) = @_;

    my $prefix = $checks{prefix} || '';
    foreach my $field (@{$fields}) {

        if ($checks{def} && ! defined $params->{$field}) {
            return ('BadParams', iget('Поле %s должно быть указано', $prefix.$field));
        }

        if ($checks{type}) {
            if ($checks{type} eq 'string') {
                return ('BadParams', iget('Поле %s должно содержать значение типа string', $prefix.$field)) if defined $params->{$field} && ref($params->{$field});
                if ($checks{len}) {
                    return ('BadParams', iget('Длина строки в поле %s не должна превышать %s', $prefix.$field, $checks{len})) if defined $params->{$field} && length($params->{$field}) > $checks{len};
                }
                if ($checks{not_empty} && defined $params->{$field}) {
                    my $field_val = $params->{$field};
                    $field_val =~ s/^\s+//;
                    $field_val =~ s/\s+$//;
                    return ('BadParams', iget('Строка в поле %s не должна быть пустой или состоять только из пробелов', $prefix.$field, $checks{len})) if length($field_val) == 0;
                }
            } elsif ($checks{type} eq 'unsignedint') {
                return ('BadParams', iget('Поле %s должно содержать значение типа unsigned integer', $prefix.$field))
                    if defined $params->{$field} && $params->{$field} !~ /^\d+$/;
            } elsif ($checks{type} eq 'int') {
                return ('BadParams', iget('Поле %s должно содержать значение типа integer', $prefix.$field))
                    if defined $params->{$field} && ! is_valid_int( $params->{$field} );
            } elsif ($checks{type} eq 'float') {
                return ('BadParams', iget('Поле %s должно содержать значение типа float', $prefix.$field))
                    if defined $params->{$field} && ! is_valid_float( $params->{$field} );
            } elsif ($checks{type} eq 'yesno') {
                return ('NotYesNo', iget('Поле %s', $prefix.$field))
                    if defined $params->{$field} && !is_yes_no($params->{$field});
            } elsif ($checks{type} eq 'email') {
                return ('BadEmail', iget('Поле %s', $prefix.$field))
                    if defined $params->{$field} && !is_valid_email($params->{$field});
            } elsif ($checks{type} eq 'date_no_future') {
                return ('BadDate', iget('Поле %s', $prefix.$field))
                    if defined $params->{$field} && !is_valid_date($params->{$field}, no_future => 1);
            } elsif ($checks{type} eq 'date') {
                return ('BadDate', iget('Поле %s', $prefix.$field))
                    if defined $params->{$field} && !is_valid_date($params->{$field});
            }elsif ($checks{type} eq 'login') {
                return ('BadLogin', iget('Поле %s', $prefix.$field))
                    if defined $params->{$field} && !validate_login( $params->{$field}, lite => 1 );
            } elsif ($checks{type} eq 'phone') {
                return ('BadPhone', iget('Поле %s', $prefix.$field))
                    if defined $params->{$field} && !is_valid_phone( $params->{$field} );
            } elsif ($checks{type} eq 'phrase-params') {
                if (defined $params->{$field}) {
                    my $param_error = validate_phrase_one_href_param($params->{$field});
                    return ('BadParams', iget('Поле %s: %s', $prefix.$field, lcfirst($param_error))) if $param_error;
                }
            } elsif ($checks{type} eq 'structure') {
                return ('BadParams', iget('Поле %s должно быть структурой', $prefix.$field))
                    if defined $params->{$field} && ref $params->{$field} ne 'HASH';
            } elsif ($checks{type} eq 'url') {
                return ('BadParams', iget('Поле %s должно содержать url', $prefix.$field))
                    if defined $params->{$field} && !is_valid_domain($params->{$field});
            } elsif ($checks{type} eq 'url_ex_ip') {
                my $host = get_host($params->{$field});
                return ('BadParams', iget('Поле %s должно содержать url', $prefix.$field))
                    if defined $params->{$field}
                        && ( $params->{$field} !~ m!^https?://! || !(is_valid_domain($host) || (is_valid_ip($host) && !is_internal_ip($host))));
            } elsif ($checks{type} eq 'timestamp') {
                if (defined $params->{$field}) {
                    if ($params->{$field} !~ /^\d{4}\-\d{2}\-\d{2}T\d{2}:\d{2}:\d{2}/) {
                        return ('BadParams', iget('Поле %s должно содержать значение в формате ISO 8601 date/time (YYYY-MM-DDThh:mm:ss)', $prefix.$field));
                    }

                    eval {
                        iso8601_2_mysql($params->{$field});
                    };

                    if ($@) {
                        return ('BadTimestamp');
                    }
                }
            } elsif ($checks{type} eq 'locale') {
                if (defined $params->{$field}) {
                    if ( ! Yandex::I18n::is_valid_lang($params->{$field}) ) {
                        return ('UnknownLocale', iget('Неизвестный язык %s', $params->{$field}));
                    }
                }
            } elsif ($checks{type} eq 'list') {
                my %list = map {$_ => 1} @{$checks{list}};
                if (defined $params->{$field} && ! defined $list{$params->{$field}}) {
                    return('BadParams', iget('Поле %s должно содержать значения: %s', $prefix.$field, join(', ', @{$checks{list}})));
                }
            } elsif ($checks{type} eq 'phrase') {
                if (defined $params->{$field}) {
                    my $validation_result = base_validate_keywords([$params->{$field}]);
                    $validation_result->process_objects_descriptions( keyword => { field => $field } );
                    return ('BadPhrase', $validation_result->get_first_error_description) unless $validation_result->is_valid;
                }
            } elsif ($checks{type} eq 'rubric') {
                if (defined $params->{$field}) {
                    my $error = validate_one_rubric($params->{$field});
                    return ('BadRubric', iget("Невозможно добавить рубрику с данным идентификатором (%d)", $params->{$field}))
                        if $error;
                }
            } elsif ($checks{type} eq 'currency') {
                return ('BadCurrency') unless !defined $params->{$field} || is_valid_currency($params->{$field});
                my %list = map {($_ || 'empty')  => 1} @{$checks{list}};

                if (defined $params->{$field} && ! defined $list{$params->{$field}}) {
                    if (scalar grep {$_} @{$checks{list}}) {
                        return('BadCurrency', iget('Допустимые значения %s: %s', $field, join(', ', @{$checks{list}})));
                    } else {
                        return('BadCurrency', iget('Для данного запроса нет допустимых значений валюты'));
                    }
                }
            } elsif ($checks{type} eq 'base64') {
                if (defined $params->{$field}) {
                    return ('BadParams', iget('Поле %s должно содержать бинарные данные, закодированные в Base64', $prefix.$field))
                        unless $params->{$field} && is_valid_base64($params->{$field});
                }
                if ($checks{len}) {
                    return ('BadParams', iget('Длина Base64 строки в поле %s не должна превышать %s', $prefix.$field, $checks{len}))
                        if defined $params->{$field} && length($params->{$field}) > $checks{len};
                }
            # см. HashingTools::md5_base64ya
            } elsif ($checks{type} eq 'base64ya') {
                if (defined $params->{$field}) {
                    return ('BadParams', iget('Поле %s должно содержать бинарные данные, закодированные в Base64', $prefix.$field))
                        unless $params->{$field} && $params->{$field} =~ /^[0-9a-zA-Z\-\_]+$/;
                }
                if ($checks{len}) {
                    return ('BadParams', iget('Длина Base64 строки в поле %s не должна превышать %s', $prefix.$field, 22))
                        if defined $params->{$field} && length($params->{$field}) != 22;
                }
            } elsif ($checks{type} eq 'contract') {
                # is string
                return ('BadParams', iget('Поле %s должно содержать значение типа string', $field)) if (ref $params->{$field});
                return ('BadParams', iget('Поле %s заполнено неверно', $field)) if ($params->{$field} !~ /^[1-9][\d\/]*$/);
            }
        }

        if (defined $params->{$field}) {
            if ($checks{positive}) {
                unless ($params->{$field} > 0) {
                    return ('BadParams', iget('Поле %s должно быть больше 0', $prefix.$field));
                }
            }

            if ($checks{not_empty}) {
                return ('BadParams', iget('Поле %s должно быть указано', $prefix.$field)) if !$params->{$field};
            }

            if (defined $checks{more_than}) {
                return ('BadParams', iget('Поле %s должно быть больше %s', $prefix.$field, $checks{more_than}))
                    if $params->{$field} <= $checks{more_than};
            }
            
            if (defined $checks{min}) {
                return ('BadParams', iget('Поле %s должно быть не меньше %s', $prefix.$field, $checks{min}))
                    if $params->{$field} < $checks{min};
            }

            if (defined $checks{max}) {
                return ('BadParams', iget('Поле %s должно быть не больше %s', $prefix.$field, $checks{max}))
                    if $params->{$field} > $checks{max};
            }
        }
    }

    return; #ok

}

1;
