package APICommon;

# $Id$
# APICommon.pm
# Автор: Vasiliy Bryadov <mirage@yandex-team.ru>

use strict;
use warnings;
use feature 'state';

use POSIX qw/strftime/;
use Data::Dumper;
use Time::HiRes;
use List::Util qw/max min/;
use List::MoreUtils qw/any none uniq firstval/;
use Date::Calc;
use URI::Escape qw/uri_escape_utf8/;
use JSON;
use Try::Tiny;

use Yandex::DateTime;
use Yandex::TimeCommon;
use Yandex::Validate;
use Yandex::HashUtils;
use Yandex::Balance;
use Yandex::TimeCommon qw/today mysql_round_day/;
use Yandex::I18n;
use Yandex::SendSMS;
use Yandex::Blackbox;
use Yandex::YaMoney;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Trace;
use Yandex::Clone qw/yclone/;

use Currencies;
use Currency::Pseudo;
use Currency::Rate;
use Currency::Texts;

use Wallet;
use RBACElementary;
use RBACDirect;
use RBAC2::DirectChecks;
use RBAC2::Extended;
use YAML qw(Dump Load);

use Settings;
use Campaign;
use Campaign::Types;
use Phrase;
use Common qw(:globals :subs);
use IpTools;
use EnvTools;
use TextTools qw/round2s html2string smartstrip normalize_login/;
use HashingTools qw/get_random_string sha256_hex/;
use LogTools qw/log_auctions_limit log_api_tech_limit_update/;
use Tools;
use geo_regions;
use API::Errors;
use API::Settings;
use API::Limits qw/has_spec_limits get_spec_limit get_spec_method_limit/;
use API::ClientOptions;
use APIUnits qw//;
use Fake;

use ModerateChecks;
use ModerateDiagnosis qw/mass_get_diags/;
use MailNotification;
use Primitives;
use PrimitivesIds;
use Sitelinks;
use Stat::Const;
use User;
use MoneyTransfer;
use MailService;
use Client;
use Agency;
use Tag qw/mass_get_all_campaign_tags get_pids_by_tags get_pids_by_tags_text/;

use Yandex::SendMail qw/send_alert/;

use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);
require Exporter;
@ISA = qw(Exporter);
%EXPORT_TAGS = (
           subs => [ qw(
                            api_initialize
                            soap_log_request
                            log_and_dieSOAP
                            get_syslog_data

                            get_new_master_token
                            drop_master_token
                            drop_finance_ops_counter
                            check_finance_token
                            check_payment_token
                            update_finance_operation_stat
                            check_fin_operations_available
                            api_check_user_access
                            check_access_user_version

                            check_rbac_rights
                            check_uid_cid_bid

                            convert_user_data_for_stat

                            get_client_object
                            get_short_client_object

                            api_check_limit_by_method
                            api_check_limit_autoload
                            api_check_limit
                            api_update_limit
                            api_update_limit_uid

                            get_api_version_by_uri
                            get_api_version_label
                            check_avaliable_version

                            get_client_role

                            get_user_advq_queries_limit

                            is_api_geo_allowed

                            create_auto_price_structure
                            get_max_available_api_version
                            create_pay_campaign_request

                            get_user_api_params
                            notification_api_finance

                            get_campaigns_tags_struct
                            check_edit_phrase

                            status_moderate_ext_to_int
                            get_modreasons_into_banners

                       )
                   ]
);
@EXPORT = qw( );
Exporter::export_ok_tags('subs');

use vars @{$EXPORT_TAGS{globals}};

use utf8;

# handler для RBAC, чтобы не создавать объект при каждом запросе
our $rbac;

our $API_LIMIT_MAX_TIME_DELAY ||= 10 * 60; # 10 minutes
our $API_LIMIT_IGNORE_DELAY ||= { 
    map {$_=>1} qw/
        GetBannersStat
        GetSummaryStat__daily
        FinOperations__daily
        SetAutoPrice
        CreateNewSubclient
        AccountManagement__transfer
/};

# максимальное количество строк в ответе метода GetSummaryStat
our $SUMMARY_STAT_MAX_LINES_IN_REPORT ||=1000;

# список методов в API вызов которых ограничивается по uid из AUTOLOAD
our $USER_LIMITS ||= {
    GetKeywordsSuggestion => 3000
  , CreateInvoice => 1000
  , PayCampaigns => 1000
  , TransferMoney => 1000
  , GetCreditLimits => 1000
    };

# список методов в API вызов которых ограничивается из самих методов по uid, cid или чего либо еще...
our %OBJECT_LIMITS;
if (!keys %OBJECT_LIMITS) {
    %OBJECT_LIMITS = (
        CreateNewReport => 300
        , GetSummaryStat => -1
        , GetSummaryStat__daily => 100
        , CreateNewWordstatReport => -1
        , CreateNewSubclient => 100
        , GetBannersStat => 300
        , FinOperations__daily => 30
        , AccountManagement__transfer => 3
        , AccountManagement__update   => 50
    );
} # -1 для того, чтобы не проверяться в AUTOLOAD

# доступные в отчете статистики позиции
our %POSITION_TYPES = (other => 'other', premium => 'prime');


our %EVENT_TYPE2TOKEN = (
        # новые события типа warn_ctr перестали появляться после DIRECT-40716, старые удалены
        # DIRECT-84007: оставлено для валидации параметров, приходящих от имеющихся клиентов
    warn_ctr                    => 'LowCTR',
    money_out                   => 'MoneyOut',
    money_warning               => 'MoneyWarning',
    money_in                    => 'MoneyIn',
    warn_cpm_limit              => 'WarnMinPrice',
    camp_finished               => 'CampaignFinished',
    warn_place                  => 'WarnPlace',
    banner_moderated            => 'BannerModerated',
    paused_by_day_budget        => 'PausedByDayBudget',
    paused_by_day_budget_wallet => 'PausedByDayBudgetAccount',
    money_out_wallet            => 'MoneyOutAccount',
    money_warning_wallet        => 'MoneyWarningAccount',
    money_in_wallet             => 'MoneyInAccount',
    #strategy_data_sum_changed  => недоступен из API
    #daily_budget_sum_changed   => недоступен из API
    custom_message_with_link    => 'InformationalWithLink',
    );

our %EVENT_ATTR2TOKEN = (
    new_cpm_limit          => 'MinPrice',
    sum_rest               => 'Rest',
    sum_payed              => 'Payed',
    finish_date            => 'FinishDate',
    old_place              => 'OldPlace',
    is_edited_by_moderator => 'IsEditedByModerator',
    bs_stop_time           => 'StopTime',
    currency               => 'Currency',
);

our %EVENT_MODRES2TOKEN = (
        accepted => 'Accepted',
        declined => 'Declined',
        declined_partly => 'DeclinedPartly',
);

our %RETARGETING_EXT_CONVERSION = (
    condition => 'RetargetingCondition',
    goals => 'Goals',
    goal_id => 'GoalID',
    time => 'Time',
    type => 'Type',
    condition_desc => 'RetargetingConditionDescription',
    condition_name => 'RetargetingConditionName',
    ret_cond_id => 'RetargetingConditionID',
    is_accessible => 'IsAccessible',
);

our %AD_RETARGETING_CONVERSION = (
        'bid' => 'AdID',
        'ret_cond_id' => 'RetargetingConditionID',
        'ret_id' => 'RetargetingID',
        'is_suspended' => 'StatusPaused',
        'autobudgetPriority' => 'AutoBudgetPriority',
        'price_context' => 'ContextPrice',
        'currency' => 'Currency',
);
our %BANNER_FILTER_VALUES = (
    StatusPhoneModerate => {
                        New => 1,
                        Yes => 1,
                        No => 1,
                        Pending => 1,
                    },

    StatusBannerModerate => {
                        Yes => 1,
                        No => 1,
                        Pending => 1,
                        PreliminaryAccept => 1,
                        New => 1,
                    },
    StatusPhrasesModerate => {
                        Yes => 1,
                        No => 1,
                        New => 1,
                        PreliminaryAccept => 1,
                        Pending => 1,
                    },
    StatusActivating => {
                        Yes => 1,
                        Pending => 1,
                    },
    StatusShow => {
                        Yes => 1,
                        No => 1,
                    },
    IsActive => {
                        Yes => 1,
                        No => 1,
                    },
    StatusArchive => {
                            Yes => 1,
                            No => 1,
                     },
    Tags => 1,
    TagIDS => 1,
);

our @AVAILABLE_CURRENCIES = (
    'cu','RUB'
);

our %AVAILABLE_MEDIAPLAN_KEYWORDS_ACTIONS = map {$_ => 1} qw/Get Update Add Delete/;

our %AVAILABLE_MEDIAPLAN_AD_ACTIONS = map {$_ => 1} qw/Get Update Add Delete/;

our %AVAILABLE_MEDIAPLAN_ACTIONS = map {$_ => 1} qw/Get Finish/;


# поля, которые всегда есть в ответе метода
our %PHRASES_FIELDS_MIN = map {$_ => 1} qw(CampaignID BannerID PhraseID);

# поля, которые передаются стабильно в версии 4live — можно регулировать параметром FieldsNames
our %PHRASES_FIELDS_NEW = map {$_ => 1} qw/StatusPaused ContextShows ContextClicks AdGroupID Currency/;

# поля при наличии который ходим в торги
our @PHRASES_AUCTION_FIELDS = qw/Prices Max Min PremiumMax PremiumMin LowCTR ContextLowCTR Shows Clicks ContextShows ContextClicks
                      CurrentOnSearch LowCTRWarning MinPrice ContextCoverage Coverage ContentRestricted AuctionBids/;

=head2 log_and_dieSOAP

    Логирование ошибок, которые произошли до того, как был вызван код валидации или метода

=cut

sub log_and_dieSOAP
{
    my ($log_data, $errorTextCode, $error_detail) = @_;

    if (! $log_data->{uid} && $log_data->{login}) {     
        $log_data->{uid} = get_uid_by_login2($log_data->{login});       
    }

    $log_data->{soap_status} = $API::Errors::ERRORS{ $errorTextCode }->{code};
    $log_data->{error_detail} = $error_detail;
    $log_data->{cluid} = [];

    soap_log_request($log_data);

    dieSOAP($errorTextCode, $error_detail);
}

=head2 soap_log_request

    Логирование запросов в API.
    Записывает параметры запроса в лог
    Параметры от пользователя сохраняеются в YAML формате, обрезанные до 1000 символов
    syslog_data опционален, используется для записи более подробных данных в syslog

=cut

sub soap_log_request
{
    my ($log_data, $params, $syslog_data) = @_;
    if ($log_data->{dont_log}) { # не засорять журнал дублирующимися данными
        return;
    }

    # Workaround для того, чтобы проходили юнит-тесты (при использовании просто $Settings::SYSLOG_BETA_PREFIX выдаётся ворнинг "Name <...> used only once: possible typo...", если в рабочей копии не подключается SettingsLocal.pm)
    # Вообще не очень хорошо, что есть переменная, которая есть в SettingsDevTest и которой нет в Settings. Это временное решение, пока логи в syslog пишутся только на бетах.
    my $syslog_prefix;
    if ($Settings::BETA_SYSLOG_PREFIX) {
        $syslog_prefix = $Settings::BETA_SYSLOG_PREFIX;
    }
    state $log_syslog = Yandex::Log->new(use_syslog => 1, no_log => 1, syslog_prefix => $syslog_prefix || 'PPCLOG', log_file_name => 'ppclog_api.log');

    # обрезаем детальное описание, поскольку иногда сюда могут попасть длинные строки, вроде фразы, например.
    my $error_detail = substr( $log_data->{error_detail} || '', 0, 200 );
    my $time = time();

    my $cmd = $log_data->{method};
    if (ref $params eq 'HASH') {
        $cmd .= '.'.$params->{Action} if $params->{Action};
    }
    # получаем важный для поиска cid
    my $cids = defined $syslog_data && exists $syslog_data->{cid} && ref $syslog_data->{cid} eq 'ARRAY' ? $syslog_data->{cid} : [];
    my $mdata = {
       log_type => 'api',
        logtime => unix2mysql($time),
            cid => $cids,
            bid => ref($params) eq 'HASH' ? $params->{BannerID} || ($params->{BannerIDS} && ref($params->{BannerIDS}) eq 'ARRAY' && @{$params->{BannerIDS}} && $params->{BannerIDS}->[0] && is_valid_int($params->{BannerIDS}->[0], 0) ? $params->{BannerIDS}->[0] : 0) : 0,
             ip => $ENV{REMOTE_ADDR} || '',
            cmd => $cmd || '',
        runtime => defined $log_data->{start_time} ? ( Time::HiRes::time() - $log_data->{start_time} ) : 0,
          param => Tools::trim_log_data($params),
    http_status => $log_data->{soap_status} || 0,
          cluid => join(',', @{$log_data->{cluid}}),
                reqid => $log_data->{reqid} || '',
                  uid => $log_data->{uid} || 0,
                 host => $EnvTools::hostname,
              proc_id => $$,
        # api only params
        sleeptime => $log_data->{sleep_time} || undef,
        error_detail => $error_detail || '',
        units => $log_data->{units},
        units_stats => YAML::Dump($log_data->{units_stats}),
        api_version => $log_data->{api_version},
        interface   => $log_data->{api_interface},
        application_id => $log_data->{application_id},
    };

    eval {
        $log_syslog->out($mdata);
    };
    if( $@ ) {
        print STDERR Dumper ['$@',$@];
    }
}

=head2 get_syslog_data

По имеющимся данным (подмножество bid, cid, cluid) вернуть все связанные cid и cluid
На входе и выходе ссылки на хеши

=cut

sub get_syslog_data {
    my ($in) = @_;
    my $data = {cid => [], cluid => (exists $in->{cluid} && defined $in->{cluid}) ? $in->{cluid} : []};
    my $cids = $in->{cid};

    unless (defined $cids && @$cids) { # здесь и далее полагаем, что данные на входе в правильном формате
        if (exists $in->{bid} && defined $in->{bid} && @{$in->{bid}}) {
            $cids = get_cids(bid => $in->{bid});
        }
    }

    if (defined $cids && @$cids) {
        $data->{cid} = $cids;
        unless (@{$data->{cluid}}) {
            $data->{cluid} = get_uids(cid => $cids);
        }
    }

    return $data;
}

=head2 check_rbac_rights($self, $method, $params)

    $self содержит ключи rbac, rbac_rights, user_info
    В случае успеха проверки (есть доступ) возвращает false, иначе, в случае ошибки возращает error_code

=cut

sub check_rbac_rights
{
    my ($self, $method, $params) = @_;
    $params ||= {};
    $params->{cmd} = $method;
    $params->{rights} = $self->{rbac_login_rights};
    
    return RBAC2::DirectChecks::rbac_do_cmd( $self->{rbac}, $method, $params, \$self->{rbac_rights}, $self->{user_info} );
}

#...............................................
# check for valid and in rbac
# check_uid_cid_bid(uid => 23656, cid => 534566, bids => [2344,543566,6578]);
# return uid (for ulogin) or undef or die

sub check_uid_cid_bid
{
    my ($self, %params) = @_;

    if (exists $params{ulogin}) {
        dieSOAP('NoRights', iget('Логин некорректен')) if ! defined $params{ulogin} || length($params{ulogin}) == 0;
        $params{uid} = get_uid_by_login2($params{ulogin}) or dieSOAP('NoRights', iget('User login is invalid'));
    }

    if (exists $params{uid} && (! defined $params{uid} || ! $params{uid} =~ /^\d+$/)) {
        dieSOAP('NoRights', iget('Логин некорректен'));
    }

    if (!$params{skip_is_owner} && $params{uid} && ! rbac_is_owner($self->{rbac}, $self->{uid}, $params{uid})) {
        dieSOAP('NoRights', iget('Недостаточно прав на указанный логин'));
    }

    if (exists $params{cid} && (! defined $params{cid} || ! $params{cid} =~ /^\d+$/)) {
        dieSOAP('NoRights', iget('CampaignID некорректен'));
    }

    if ($params{cid} && ! rbac_is_owner_of_camp($self->{rbac}, $self->{uid}, $params{cid})) {
        dieSOAP('NoRights', iget('CampaignID не найден'));
    }

    if (exists $params{bids}) {
        if (ref($params{bids}) eq 'ARRAY' && @{ $params{bids} }) {
            dieSOAP('BadParams', iget('Кампания не найдена')) unless $params{cid};
            dieSOAP('BadParams', iget('BannerID некорректен')) if grep {!/^\d+$/} @{ $params{bids} };

            my $real_bids_cnt = get_one_field_sql(PPC(cid => $params{cid}), ["select count(*) from banners", where => { cid => $params{cid}, bid => $params{bids} }]) || 0;
            my $bids_cnt = scalar @{ $params{bids} };

            dieSOAP('BadParams', iget('баннеры не найдены')) if $bids_cnt != $real_bids_cnt;

        } else {
            dieSOAP('BadParams', iget('BannerIDS некорректен'));
        }
    }
    
    if (exists $params{cid}) {
        my $type = get_camp_type(cid => $params{cid});

        if (!defined $type || is_wallet_camp(type => $type)) {
            return dieSOAP('BadCampaignID');
        } else {
            if (exists $params{campaign_kind}) {
                unless (camp_kind_in(type => $type, $params{campaign_kind})) {
                    dieSOAP('NotSupported');
                }
                if ($type eq 'geo' && !(is_api_geo_allowed($self) eq 'Yes')) {
                    dieSOAP('NotSupported');
                }
            } elsif (! (camp_kind_in(type => $type, 'api_edit') || $type eq 'geo' && is_api_geo_allowed($self) eq 'Yes') ) {
                dieSOAP('BadCampaignType');
            }
        }
    }

    return $params{uid};
}

=head2 is_api_geo_allowed({uid => $uid})

    Разрешено ли пользователю работать с кампаниями типа geo

=cut

sub is_api_geo_allowed
{
    my $self = shift;
    return $self->{user_info} && $self->{user_info}{api_geo_allowed} ? 
        $self->{user_info}{api_geo_allowed}
        : (get_one_user_field($self->{uid}, 'api_geo_allowed') || 'No');
}

# для инициализации RBAC нужен uid !
sub api_initialize
{
    my $self = shift;

    if ($self->{uid}) {
        $rbac = RBAC2::Extended->get_singleton( $self->{uid} );

        $self->{rbac} = $rbac;
        
        if ($self->{uid}) {
            $self->{user_info} = get_user_data($self->{uid}, [qw/login api_allowed_ips api_geo_allowed statusBlocked api_allow_old_versions ClientID/]) || {};
            $self->{special_options} = query_all_api_special_options($self->{user_info}->{ClientID});

            $self->{operator_login} = $self->{user_info}{login};
            $self->{ClientID} = $self->{user_info}->{ClientID};
            $self->{user_ip} = $self->{plack_request}->address;
            
            $self->{client_options} = API::ClientOptions::get($self->{ClientID}, ['api_enabled']);

            my $rbac_res = rbac_login_check( $self->{rbac}, {
                UID => $self->{uid},
                user_info => $self->{user_info},
                is_internal_ip => is_internal_ip($self->{plack_request}->address),
            }, \$self->{rbac_login_rights} );

            if ($rbac_res) {
                my $descr = $rbac_res != 3 ? undef : iget("Доступ возможен только из внутренней сети Яндекса");
                dieSOAP('NoRights', $descr);
            }
            
            # save for MailNotification::mail_notification
            MailNotification::save_UID_host($self->{uid});
        } else {
            dieSOAP('UserInvalid', iget("Пользователь не указан"));
        }
        
        # save for Tools
        Tools::_save_vars($self->{uid});

        # создаём контроллер балльной системы 
        $self->{uhost} = new APIUnits({
                                scheme => "API"
                                , is_internal_user => $self->{rbac_login_rights}->{is_internal_user}
                             }) or dieSOAP('500');
    }
    if($self->{locale} eq 'non'){
        Yandex::I18n::init_i18n('en');
    } else {
        Yandex::I18n::init_i18n($self->{locale});
    }

    return $self;
}

=head2 get_api_version_by_uri(URI)

    Из URI получает номер версии API.
    
    Ожидаемый формат URI: soap.direct.yandex.ru/api/v11/
    Иначе - возвращаем версию = 1.

=cut

sub get_api_version_by_uri
{
    my $uri = shift;
    
    my ($version, $latest, $wsdl_subversion); # последнее для обновлений wsdl-файлов
    if ($uri =~ /^\/(api|json\-api)\/v\d+/) {
        ($version, $latest) = $uri =~ /\/(?:json\-)?api(?:\/v(\d{1,4}))\/?(live)?\/?$/;
    } elsif($uri =~ m!/(live/)?v4/soap/1\/?$!) {
        ($latest, $version, $wsdl_subversion) = ($1,4,1);
    } elsif($uri =~ m!/(live/)?v4/soap/2\/?$!) {
        ($latest, $version, $wsdl_subversion) = ($1,4,2);
    } else {
        ($latest, $version) = $uri =~ /\/(live\/)?v(\d{1,4})\/(?:json|soap)\/?$/;
    }

    $latest = 1 if $latest;

    # для тестов - вместо суффикса live - прибавляем 1000 к номеру версии
    if ($version > 1000) {
        $version -= 1000;
        $latest = 1;
    }
    
    # по-умолчанию(+ поддержка старых форматов URLа) версия 1
    $version ||= 1;

    my $version_full = $version;
    $version_full += 0.5 if $latest;

    return ($version, $latest, $version_full, $wsdl_subversion);
}

=head2 get_new_master_token

=cut

sub get_new_master_token
{
    my $cl_uid = shift;
    my $new_token = get_random_string(alphabets => 'wWd', length => 16);

    do_insert_into_table(PPCDICT, 'api_finance_tokens'
                             , {uid => $cl_uid, master_token_timecreated__dont_quote => 'NOW()', master_token => $new_token || 0}
                             , on_duplicate_key_update => 1
                             , key => 'uid');

    return $new_token;
}

=head2 drop_master_token

=cut

sub drop_master_token
{
    my $cl_uid = shift;

    do_insert_into_table(PPCDICT, 'api_finance_tokens'
                             , {uid => $cl_uid, master_token_timecreated__dont_quote => '000000000000', master_token => ''}
                             , on_duplicate_key_update => 1
                             , key => 'uid');

    return 1;
}

=head2 drop_finance_ops_counter

=cut

sub drop_finance_ops_counter
{
    my $cl_uid = shift;
    do_insert_into_table(PPCDICT, 'api_finance_tokens', 
                    {uid => $cl_uid, api_operation_num => 0, finance_cnt_dropped_time__dont_quote => 'NOW()'}
                    , on_duplicate_key_update => 1
                    , key => 'uid');

    return 1;
}
=head2 check_finance_token

    Проверяет валидность переданного sha2 хэша:
        sha2(master_token, operation_num, method, action, login)

    Возвращает 1, если проверка пройдена
               0, если что-то пошло не так

=cut

sub check_finance_token {
    my ($cl_uid, $finance_token, $method, $operation_num, $login, $action) = @_;

    my @errors = ();

    if ( !$finance_token ) {
        @errors = ('FinanceTokenInvalid', iget('Не передан финансовый токен'));
    } else {

        my $params = get_one_line_sql(PPCDICT, [ "select master_token, 
                                                        api_operation_num
                                                from api_finance_tokens", 
                                                where => {uid => $cl_uid} ]);

        if (!$params->{master_token}) {
            @errors = ('NoMasterToken');
        } else {
            $params->{login} = $login || get_login(uid => $cl_uid);

            my $correct_finance_token = sha256_hex($params->{master_token}, $operation_num, $method, $action, $params->{login});

            if ( $operation_num <= $params->{api_operation_num} ) {
                @errors =  ('InvalidFinOperationNum');
            } elsif ($finance_token ne $correct_finance_token) {

                # в случае неудачи, получаем логин как он есть в паспорте и пытаемся сгенерировать еще один
                # возможный вариант токена
                my $passport_login = bb_userinfo($cl_uid, $ENV{REMOTE_ADDR} || '127.0.0.1', "direct.yandex.ru")->{regname}{content};

                my $correct_finance_token = sha256_hex($params->{master_token}, $operation_num, $method, $action, $passport_login);

                # и если не получается, то увы...
                if ($finance_token ne $correct_finance_token) {
                    @errors =  ('FinanceTokenInvalid');
                }
            }
        }
    }

    return @errors;
}

=head2 check_payment_token

    Проверяет переданный payment_token в ЯД
    Возвращает:
        0, если uid, привязанный в ЯД, соответствует переданному uid
        1, если токен невалиден
        2, во всех остальных случаях

    Для тестирования на ТС и devtest необходимо добавить фэйковые данные:
    perl -Iprotected -MFake -le 'save_fake_data(type => "check_payment_token", id => 64270283, data => 3000104813);'
    perl -Iprotected -MFake -le 'save_fake_data(type => "check_payment_token", id => 50436888, data => 3000106533);'
    perl -Iprotected -MFake -le 'save_fake_data(type => "check_payment_token", id => 65782527, data => 3000105677);'
    perl -Iprotected -MFake -le 'save_fake_data(type => "check_payment_token", id => 131630732, data => 3000108101);'
    perl -Iprotected -MFake -le 'save_fake_data(type => "check_payment_token", id => 167883457, data => 3000156060);'

=cut

sub check_payment_token {
    my ($uid, $payment_token) = @_;
    my $res = Yandex::YaMoney::ym_validate_payment_token($payment_token);
    
    if (is_sandbox()) {
        if ($res && $res->{direct_scope} && $res->{uid} == 1) {
            return 0;
        }
    }

    if (! is_production()) {
        fake_up($uid, type => 'check_payment_token', id => $uid);
    }

    if ($res->{error} && $res->{error} eq 'invalid_token') {
        return 1;
    } elsif($res->{error} && $res->{error} eq 'ym_not_available') {
        return 2;
    } elsif ($res->{direct_scope} && $res->{uid} && $res->{uid} == $uid) {
        return 0;
    } elsif ($res->{uid} && $res->{uid} != $uid) {
        return 3;
    }

    return 2;
}

=head2 update_finance_operation_stat

    Обновляет базу, содержащую токены и дополнительную информацию о финансовых операциях

    на вход:
        success - увеличивает счетчик с порядковыми номерами операций, обнуляет счетчик неудач
        fault - увеличивает счетчик неудач

=cut

sub update_finance_operation_stat
{
    my ($cl_uid, $rbac, %O) = @_;

    if ($O{'success'}) {
        do_sql(PPCDICT, [ 'update api_finance_tokens set api_operation_num = ?, faults = 0', where => {uid => $cl_uid}], $O{operation_num});
    } elsif ($O{'fault'} && ! is_sandbox()) {
        do_sql(PPCDICT, [ 'update api_finance_tokens set faults = faults + 1', where => {uid => $cl_uid}]);
    }

    my $fin_ops_faults = get_one_field_sql(PPCDICT, [ 'select faults from api_finance_tokens', where => {uid => $cl_uid} ]) || 0;

    if ($fin_ops_faults >= $Settings::API_MAX_FIN_OPS_FAULTS) {
        create_update_user($cl_uid, {api_allow_finance_operations => 'No'});
        notification_api_finance($cl_uid, $rbac, $cl_uid, 'api_finance_off');
    }
}

=head2 check_fin_operations_available

=cut

sub check_fin_operations_available
{
    my $cl_uid = shift;

    my $fin_ops_avail = get_one_user_field($cl_uid, 'api_allow_finance_operations') eq 'Yes' ? 1 : 0;
    my $faults = get_one_field_sql(PPCDICT, 'select faults from api_finance_tokens where uid = ?', $cl_uid);

    if (!$fin_ops_avail && $faults) {
        return ('FinOpsNotAvailable', iget('Финансовые операции отключены автоматически из-за большого количества ошибок подряд'));
    } elsif (!$fin_ops_avail && !$faults) {
        return ('FinOpsNotAvailable');
    } 

    return;
}

=head2 msg_converting_in_progress

    Детализация ошибки с сообщением о конвертации

=cut

sub msg_converting_in_progress { iget('Доступ к API закрыт на время перевода кампаний в валюту') }

=head2 msg_must_convert

    Детализация ошибки с сообщением о необходимости конвертации

=cut

sub msg_must_convert { iget('Доступ к API закрыт до конвертации в валюту') }


=head2 api_check_user_access(uid)

    Проверяет наличие доступа к АПИ пользователя
    
    Запрещаем доступ:
        1. пользователям легкого интерфейса - у которых нет спец разрешения на доступ
        2. пользователям, которым это запрещено из интефейса
        3. пользователям с ip, при наличии ограничений в найстройках
        4. пользователям, не существующим в Директе

    Доступ с IP проверяем по $ENV{X-Real-IP} || $ENV{REMOTE_ADDR} - важно, что он транслировался правильно
    
=cut

sub api_check_user_access
{
    my ($self, $agcl) = @_;

    if ($self->{user_info}{statusBlocked} eq 'Yes') {
        return iget("Доступ блокирован");
    }

    return iget("Доступ запрещен")
        unless rbac_can_use_api($self->{rbac}, { uid => $self->{uid}, UID => $self->{uid}, 
                                                 api_enabled => $self->{client_options}{api_enabled},
                                                 ClientID => $self->{ClientID},
                                });

    if ($self->{user_info}) {

        if ($self->{rbac_login_rights}{is_internal_user} 
                  && ! is_ip_in_list(($ENV{'X-Real-IP'} || $ENV{REMOTE_ADDR}), $Settings::INTERNAL_NETWORKS)) {
            # внутренним ролям только из внутренней сети
            return iget("Доступ разрешен только из внутренней сети Яндекса");
        }
        
        # проверяем возможность доступа для пользователя с IP
        # ipv6: IPv6 адреса могут быть без цифр и проверка не сработает: ABCD:ABCD:ABCD:ABCD:ABCD:ABCD:ABCD:ABCD
        if ($self->{user_info}->{api_allowed_ips} && $self->{user_info}->{api_allowed_ips} =~ /\d/) {
            my $request_ip = $ENV{'X-Real-IP'} || $ENV{'REMOTE_ADDR'};

            if (! is_ip_in_networks($request_ip, $self->{user_info}->{api_allowed_ips})) {
                # доступ запрещен
                return iget("Доступ ограничен");
            }
        }

        # блокируем API на время перехода в реальную валюту
        if ($self->{user_info}->{ClientID}) {
            my $convert_queue_data = get_one_line_sql(PPC(ClientID => $self->{user_info}->{ClientID}),
                    [ 'SELECT state, convert_type, new_currency
                       FROM currency_convert_queue
                       WHERE NOW() > start_convert_at - INTERVAL ? MINUTE AND ',
                       { ClientID => $self->{user_info}->{ClientID} } ]
                , $Settings::STOP_OPERATION_MINUTES_BEFORE_CONVERT
            );
            if ($convert_queue_data && $convert_queue_data->{state}) {
                # блокируем API целиком по клиенту на время конвертации в реальную валюту
                # принять страницу успеха не требуем
                if ($convert_queue_data->{state} ne 'DONE') {
                    return get_currency_text($convert_queue_data->{new_currency}, 'api_locked_during_currency_converted');
                }
            }
            return (msg_must_convert, 'NoRights') if Client::client_must_convert($self->{user_info}->{ClientID});
        }
    } else {
        # пользователь не существует в Директе
        # TODO: эта ветка не выполняется т.к. user_options всегда defined
        return iget("Доступ разрешен только клиентам Яндекс.Директа");
    }
    
    return undef;
}

# По-хорошему вместо этих хешей должны быть функции. Но тогда нужен будет модуль API::Common,
# который подцепляется из APICommon и API::ReportCommon. Плюс нет смысла падать при генерации отчёта.

my %age_translation = (
    undefined => 'AGE_UNKNOWN',
    '0-17' => 'AGE_0_17',
    '18-24' => 'AGE_18_24',
    '25-34' => 'AGE_25_34',
    '35-44' => 'AGE_35_44',
    '45-' => 'AGE_45',
);

my %gender_translation = (
    undefined => 'GENDER_UNKNOWN',
    male => 'GENDER_MALE',
    female => 'GENDER_FEMALE',
);

my %detailed_device_type_translation = (
    undefined => 'OS_TYPE_UNKNOWN',
    android => 'ANDROID',
    ios => 'IOS',
);

my %connection_type_translation = (
    undefined => 'CARRIER_TYPE_UNKNOWN',
    mobile => 'CELLULAR',
    stationary => 'STATIONARY',
);

=head2 translate_hash(stat, table)

    Жёсткие хеши и обращение
    stat — переводимый элемент
    table — обратная таблица перевода

=cut

sub translate_hash {
    my ($stat, $table) = @_;
    my %table_reverse = reverse %$table;
    my $new_stat = $table_reverse{uc($stat)}; # пользуемся тем, что наши «как бы enum»-значения всегда в верхнем регистре
    return $new_stat if (defined $new_stat);
    die "no translation for $stat that passed validation";
}

=head2 convert_user_data_for_stat

    конвертирует данные от пользователя для создания отчета ( испозуется стандартная функция из DBStat )

=cut

sub convert_user_data_for_stat
{
    my ($in, $uid, $api_version) = @_;

    my $new = {};

    my %internal_names = (
        'PositionType' => 'position',
        'GoalConversionsNum' => 'agoalnum',
        'DeviceType' => 'device_type',
        'Demographics' => 'age',
        'MobilePlatform' => 'detailed_device_type',
        'CarrierType' => 'connection_type',
        'Adjustment' => 'retargeting_coef',
        'Image' => 'banner_image_type',
    );

    foreach( qw/StartDate EndDate CampaignID/ ) {
        $new->{$_} = $in->{$_};
    }

    my $camp = get_camp_info($in->{CampaignID}, undef, short => 1);

    $new->{OrderID} = $camp->{OrderID};

    # добиваем массив полей для сортировки из массива GroupBy
    my %order_by_fields = map {$_ => 1} @{$in->{OrderBy}};

    foreach my $field ( @{$in->{GroupByColumns}} ) {
        if (!$order_by_fields{$field}) {
            push @{$in->{OrderBy}}, $field;
        }
    }

    for (grep { /^cl(Banner|Page|Geo|Phrase|Date|StatGoals|GoalConversionsNum|PositionType|Image|AveragePosition|DeviceType|Demographics|ROI|MobilePlatform|CarrierType|Adjustment)$/ } @{$in->{GroupByColumns}}) {
        ## no critic (Freenode::DollarAB)
        my ($a) = /^cl(.*)$/;
        $new->{GroupByStatGoals} = 1 if $a eq 'StatGoals';
        $new->{GroupByGoalConversionsNum} = 1 if $a eq 'GoalConversionsNum';
        push @{$new->{GroupByColumns}}, lc($internal_names{$a} || $a);
        if (lc($internal_names{$a} || $a) eq 'phrase') {
            push @{$new->{GroupByColumns}}, grep { $_ ne 'phrase'} @Stat::Const::ANY_PHRASES_GROUP_BY;
        }
        push @{$new->{GroupByColumns}}, 'gender' if ($internal_names{$a} && lc($internal_names{$a}) eq 'age');
    }

    if (defined $new->{GroupByColumns} && (grep {/banner_image_type/} @{$new->{GroupByColumns}}) && !(grep {/banner/} @{$new->{GroupByColumns}})) {
        push @{$new->{GroupByColumns}}, 'banner';
    }

    if( defined $new->{GroupByColumns} && grep{/date/} @{$new->{GroupByColumns}} ) {
        $new->{GroupByDate} = $in->{GroupByDate} || 'day';
    }

    # фильтр по блоку показов - PositionType
    $new->{Filter}->{position} = $in->{Filter}->{PositionType} ? 
                                        lc( $POSITION_TYPES{$in->{Filter}->{PositionType}} ) : undef;

    $new->{Filter}->{page} = lc( join ',', @{ $in->{Filter}->{PageName} || [] } );
    $new->{Filter}->{page_target} = lc( !defined $in->{Filter}->{PageType} || $in->{Filter}->{PageType} eq 'all' ? '' : $in->{Filter}->{PageType} );
    $new->{Filter}->{image} = '';
    if (defined $in->{Filter}->{WithImage}) {
        if (lc($in->{Filter}->{WithImage}) eq 'yes') {
            $new->{Filter}->{banner_image_type} = 'text_image';
        } elsif (lc($in->{Filter}->{WithImage}) eq 'no') {
            $new->{Filter}->{banner_image_type} = 'text_only';
        }
    }

    if (defined $in->{Filter}->{DeviceType}) {
        if (lc($in->{Filter}->{DeviceType}) eq 'mobile') {
            $new->{Filter}->{device_type} = 'mobile';
        } elsif (lc($in->{Filter}->{DeviceType}) eq 'desktop') {
            $new->{Filter}->{device_type} = 'desktop';
        } elsif (lc($in->{Filter}->{DeviceType}) eq 'tablet') {
            $new->{Filter}->{device_type} = 'tablet';
        }
    }

    $new->{Filter}->{phrase} = lc( join ',', @{ $in->{Filter}->{Phrase}||[] } ) || '';
    $new->{Filter}->{goal_id} = lc( join ',', @{ $in->{Filter}->{StatGoals}||[] } ) || '0';

    # geo
    if ( defined $in->{Filter}->{Geo} && @{$in->{Filter}->{Geo}} ) {
        $new->{Filter}->{geo} = join ',', map{ $geo_regions::GEOREG{ $_ }->{name} } grep { /^\d+$/ && exists $geo_regions::GEOREG{abs($_)} } @{$in->{Filter}->{Geo}};
    }

    # banners
    if ( defined $in->{Filter}->{Banner} && @{$in->{Filter}->{Banner}} ) {
        my @bids = grep {/^\d+$/} @{$in->{Filter}->{Banner}};
        if ( @bids ) {
            $new->{Filter}->{banner} = get_bannerids(bid => \@bids);
        }
    }

    # age
    if (defined $in->{Filter}->{Age} && @{$in->{Filter}->{Age}}) {
        my @ages = map {translate_hash($_, \%age_translation)} @{$in->{Filter}->{Age}};
        if (@ages) {
            $new->{Filter}->{age} = \@ages;
        }
    }

    # gender
    if (defined $in->{Filter}->{Gender} && @{$in->{Filter}->{Gender}}) {
        my @genders = map {translate_hash($_, \%gender_translation)} @{$in->{Filter}->{Gender}};
        if (@genders) {
            $new->{Filter}->{gender} = \@genders;
        }
    }

    # detailed_device_type
    if (defined $in->{Filter}->{MobilePlatform} && @{$in->{Filter}->{MobilePlatform}}) {
        my @detailed_device_types = map {translate_hash($_, \%detailed_device_type_translation)} @{$in->{Filter}->{MobilePlatform}};
        if (@detailed_device_types) {
            if (any {$_ eq 'undefined'} @detailed_device_types) {
                push @detailed_device_types, 'other';
            }
            $new->{Filter}->{detailed_device_type} = \@detailed_device_types;
        }
    }

    # connection_type
    if (defined $in->{Filter}->{CarrierType} && @{$in->{Filter}->{CarrierType}}) {
        my @connection_types = map {translate_hash($_, \%connection_type_translation)} @{$in->{Filter}->{CarrierType}};
        if (@connection_types) {
            $new->{Filter}->{connection_type} = \@connection_types;
        }
    }

    my %internal_order_by_names = (%internal_names,
                                   Geo    => 'region_id',
                                   Phrase => 'phrase_id',
                                   Page   => 'page_name',
                                   ROI    => 'agoalroi',
                                   );

    my @order_by = map { lc($internal_order_by_names{$_} || $_) }
            map { s/^cl//r }
            grep { /^(clBanner|clPage|clGeo|clPhrase|clDate|clPositionType|clStatGoals|clImage|clDeviceType|clROI)$/ } 
            @{$in->{OrderBy}};

    $new->{limits} = {
                    order_by => [ map { { 'field' => $_ } } @order_by ],
                    limit => $in->{Limit},
                    offset => $in->{Offset}
    };

    $new->{TypeResultReport} = defined $in->{TypeResultReport} && $in->{TypeResultReport} ? $in->{TypeResultReport} : 'xml';
    $new->{CompressReport} = defined $in->{CompressReport} && $in->{CompressReport} ? 1 : 0;
    
    # чтобы можно было создавать отчеты в разных форматах в зависимости от версии
    $new->{_api_version} = $api_version;

    # hidden option :) - for future
    $new->{TypeReport} = $in->{TypeReport} ? $in->{TypeReport} : 'customReport';

    if ($api_version ne '4') {
        $new->{with_nds} = (($in->{Currency} && $in->{IncludeVAT} && $in->{IncludeVAT} eq 'No'))? 0 : 1;
        $new->{with_discount} = (!$in->{Currency} || ($in->{IncludeDiscount} && $in->{IncludeDiscount} eq 'No'))? 0 : 1;
        $new->{currency} = (defined $in->{Currency})? $in->{Currency} : 'YND_FIXED';
    } else {
        $new->{currency} = 'YND_FIXED';
        $new->{with_nds} = 1;
        $new->{with_discount} = 0;
    }

    $new->{consumer} = 'api4';

    return $new;
}

sub get_short_client_object
{
    my ($self, $vars) = @_;

    my $result = 0;
    if( defined $vars->{uid} ) {
        $result = get_all_sql(PPC(uid => $vars->{uid}), ['SELECT uid, Login, FIO, ClientID, statusArch as StatusArch
                                        FROM users',
                                       where => {
                                                    uid => SHARD_IDS
                                                }]);
    }

    my ($rbac_clients_info, $client_id_by_uid, $cur_agency_client_id);

    # оптимизация производительности метода GetSubClients для агенств
    # получаем список клиентов агенства вместе с правами суб-клиента
    if ($vars->{agency_uid}) {
        $rbac_clients_info = rbac_get_subclients_list_with_info($self->{rbac}, $vars->{agency_uid});
        $client_id_by_uid = rbac_get_chief_reps_of_client_reps([map {$_->{uid}} @$result]);
        $cur_agency_client_id = rbac_get_agency_clientid_by_uid( $vars->{cl_uid});
    }

    my $rbac_res = rbac_multi_who_is_detailed($self->{rbac}, [map {$_->{uid}} @$result]);

    my $clients_info = [];
    
    my ($client_relations, $clientid2arch_status);

    if ($vars->{agency_uid}) {
        $client_relations = get_agency_clients_relations_data([uniq map {$_->{ClientID}} @$result]);

        foreach my $rel (@$client_relations) {
             $clientid2arch_status->{ $rel->{client_client_id} }->{ $rel->{agency_client_id} } = $rel->{client_archived};
        }
    }

    foreach my $u (@$result) {
        my $rbac_agency_client_info;

        if ($vars->{agency_uid}) {
            my $client_chief_uid = $client_id_by_uid->{$u->{uid}} 
                                        || rbac_get_chief_rep_of_client_rep($u->{uid});

            $rbac_agency_client_info = {
                $vars->{agency_uid} => $rbac_clients_info->{$client_chief_uid}
            };
        }

        $u->{Role} = get_client_role($self->{rbac}, $u->{uid}, $vars->{cl_uid}, sub_client_rights => $rbac_agency_client_info, rbac_who_is_detailed => $rbac_res->{$u->{uid}});

        if ( $vars->{agency_uid} && $cur_agency_client_id ) {
            $u->{StatusArch} =  $clientid2arch_status->{$u->{ClientID}}{$cur_agency_client_id} || 'No';
        }

        my $iclient = hash_cut $u, qw/Login FIO Role StatusArch/;

        push @$clients_info, $iclient;
    }

    return $clients_info;
}

=head2 get_client_role

    По uid, либо результату выполнения rbac_who_is_detailed возвращает роль пользователя

=cut

sub get_client_role 
{
    my ($rbac, $uid, $operator_uid, %OPT) = @_;

    my $res;
    if (defined $OPT{rbac_who_is_detailed}) {
        $res = $OPT{rbac_who_is_detailed};
    } else {
        my $check_res = rbac_login_check($rbac, {
            UID => $uid,
            # завязываться на %ENV не очень хорошо. стоит или прокидывать объект запроса или делать эту проверку где-то на более высоком уровне
            is_internal_ip => is_internal_ip($ENV{REMOTE_ADDR}),
        }, \$res);        
        dieSOAP('NoRights') if $check_res;
    }

    my $result = $res->{role};
    if($result eq 'manager'){
        if ($res->{is_teamleader}){
            $result = 'TeamLeader';
        } elsif ($res->{is_superteamleader}){
            $result = 'SuperTeamLeader';
        }
    } elsif($result eq 'agency'){
        $result = 'ChiefRepAgency' if $res->{is_agency_chief};
        $result = 'UnlimitedRepAgency' if $res->{is_agency_main};
        $result = 'LimitedRepAgency' if $res->{is_agency_limited};
    
    } elsif($result eq 'client'){
        my $is_sub_client = 0;
        my $is_super_sub_client = 0; 
        if($operator_uid){
            my $sub_rights = defined $OPT{sub_client_rights} ? 
                                     $OPT{sub_client_rights} 
                                     : rbac_get_subclients_rights($rbac, $operator_uid, $uid);
            
            if (exists $sub_rights->{$operator_uid}){
                $is_sub_client = 1;

                # решили не называть роли подобным образом
                # $is_super_sub_client = 1 if $sub_rights->{$operator_uid}->{is_super_subclient};
            }
        }
        my $prefix = $is_sub_client? 'Sub': '';

        # решили не называть роли подобным образом
        #$prefix = 'SuperSub' if $is_super_sub_client;

        if($res->{is_client_chief}){
            $result = 'ChiefRep' . $prefix . 'Client';
        } else {
            $result = 'UnlimitedRep' . $prefix . 'Client';
        }
        
    } elsif($result eq 'media' && $res->{is_super_media_planner}){
        $result = 'SuperMedia';
    
    } elsif($result eq 'placer' && $res->{is_super_placer}){
        $result = 'SuperPlacer';
    
    } elsif($res->{superreader_control}){ 
        $result = 'SuperReader';
    
    } elsif($res->{super_control}){ 
        $result = 'SuperUser';
    }
    
    # для клиентов и агентств, такое преобразование кажется не нужно
    #   делается на всякий пожарный и для единообразия при рефакторинге
    
    if ($result eq 'client') {
        $result = 'Client';
    } elsif ($result eq 'manager') {
        $result = 'Manager';
    } elsif ($result eq 'media') {
        $result = 'Media';
    } elsif ($result eq 'agency') {
        $result = 'Agency';
    } elsif ($result eq 'placer') {
        $result = 'Placer';
    } elsif ($result eq 'client') {
        $result = 'Client';
    }
    
    return $result; 

}

=head2 get_client_object

    Возвращает массив параметров клиентов

=cut

sub get_client_object
{
    my ($self, %O) = @_;

    # если к нам пришло агентство, вычислем его ClientID
    my $cur_agency_client_id;

    if ($self->{rbac_login_rights}->{role} && $self->{rbac_login_rights}->{role} eq 'agency') {
        $cur_agency_client_id = rbac_get_agency_clientid_by_uid( $self->{uid}) || dieSOAP("BadParams", iget("Клиент не найден"));
    }

    my ($where_params, $shard_key);
    if ( defined $O{uid} ) {
        $where_params = { 'u.uid' => SHARD_IDS };
        $shard_key = 'uid';
    } elsif ( defined $O{login} ) {
        $where_params = { 'u.login' => SHARD_IDS };
        $shard_key = 'login';
    } else {
        return undef;
    }

    my $result = get_all_sql( PPC($shard_key => $O{$shard_key}), ['SELECT u.uid, u.Login, u.FIO, u.ClientID,
                                             u.Email, date(FROM_UNIXTIME(u.createtime)) as DateCreate, 
                                             u.Phone, u.statusArch as StatusArch, u.not_resident as NonResident,
                                             u.sendNews as SendNews,
                                             u.sendWarn as SendWarn, 
                                             u.sendAccNews as SendAccNews,
                                             u.description,
                                             uo.statusPostmoderate as IsGoodUser, 
                                             uo.geo_id,
                                             IFNULL(uo.sendAgencyMcbLetters, \'No\') as SendAgencyMcbLetters,
                                             IFNULL(uo.sendAgencyDirectLetters, \'No\') as SendAgencyDirectLetters
                                      FROM users u 
                                            left join users_options uo on u.uid = uo.uid
                                      WHERE ', $where_params]);

    return [] if !$result || ! scalar @$result;

    my @uids = map {$_->{uid}} @$result;
    my $users_detailed_roles = rbac_multi_who_is_detailed($self->{rbac}, \@uids);
    
    # for agencies
    my @agencies_uids = grep { $users_detailed_roles->{$_}->{role} eq 'agency' } @uids;

    my $uid2agency_vars = get_agencies_client_vars($self->{rbac}, \@agencies_uids);
    my $agency_uid2client_id = rbac_get_agencies_clientids_by_uids( \@agencies_uids);

    my %client_client_ids = map { $_->{uid} => $_->{ClientID}} @$result;

    my ($subclients_rights, $clientid2login);
    if ( $O{get_ext_params}
            && $self->{rbac_login_rights}->{role} =~ /^(manager|agency|super|support|superreader|media|client)$/
            && scalar keys %client_client_ids) {

        $subclients_rights = Agency::get_client_agency_options( operator_uid => $self->{uid},
                                                        client_uids2_client_ids => \%client_client_ids,
                                                        for_cur_agency => $cur_agency_client_id ? 1 : 0,
                                                        operator_role => $self->{rbac_login_rights}->{role},
                                                        not_filter_by_perm => $self->{api_version_full} > 4 && $self->{rbac_login_rights}->{role} =~ /^(client)$/ ? 1 : 0,
                                                    );
        if (keys %$subclients_rights) {
            my $chief_reps_of_agencies = rbac_get_chief_reps_of_agencies([uniq map {keys %{$_}} values %{$subclients_rights}]);
            # TODO: частично дублируется с получением информации 
            # по агентству для кампаний, перетащить бы в отдельную функцию
            my $chief_agencies_uids = [values %$chief_reps_of_agencies];
            $clientid2login = get_hashes_hash_sql(PPC(uid => $chief_agencies_uids), [
                'select u.clientID, u.login, u.fio, c.name 
                from users u left join clients c on u.ClientID=c.ClientID 
                ', 
                where => {uid => SHARD_IDS}]);
        }
    }

    my ($managers_of_clients, $managers_of_agencies, $uid2manager_login, $clientid2arch_status);
    my @client_ids = grep {$_} uniq map {$_->{ClientID}} @$result;
    
    if ($self->{rbac_login_rights}->{role} eq 'agency') {
    
        my $client_relations = get_agency_clients_relations_data(\@client_ids);

        foreach my $rel (@$client_relations) {
            $clientid2arch_status->{ $rel->{client_client_id} }->{ $rel->{agency_client_id} } = $rel->{client_archived};
        }
      

    } elsif ($self->{rbac_login_rights}->{role} =~ /^(manager|super|media|placer|support|superreader)$/) {

        $managers_of_clients = rbac_get_managers_of_clients($self->{rbac}, \@uids);
        $managers_of_agencies = rbac_get_all_managers_of_agencies_clientids($self->{rbac}, [uniq values %$agency_uid2client_id]);

        # получаем для uid'ов - логины
        my @managers_logins = map {@$_} values %$managers_of_clients;
        push @managers_logins, map {@$_} values %$managers_of_agencies;
        @managers_logins = uniq @managers_logins;

        # собираем все uid'ы менеджеров
        $uid2manager_login = get_uid2login(uid => \@managers_logins);
    }

    my $clients_currencies = mass_get_client_currencies(\@client_ids);
    my $clients_nds = mass_get_client_NDS(\@client_ids);
    my $clients_discount = mass_get_client_discount(\@client_ids);
    
    my $clients_market_rating = mass_get_clients_data(\@client_ids, ['hide_market_rating']);

    my $clientid2overdraft_info = get_mass_overdraft_info(\@client_ids, clients_discount => $clients_discount, clients_nds => $clients_nds, clients_currencies => $clients_currencies);

    # получаем email с кампаний пользователей
    my $campaign_emails = get_users_campaigns_emails(\@uids);

    my %client_chiefs;
    if ($self->{api_version_full} > 4) {
        my %chiefs_by_clientid = %{rbac_get_chief_reps_of_clients(\@client_ids)};
        %client_chiefs = map {$_ => $chiefs_by_clientid{$client_client_ids{$_}}} keys %client_client_ids; # uid => chief_uid
    }

    my $wallets;
    unless (exists $self->{preprocess}{clientuid2wallet} && defined $self->{preprocess}{clientuid2wallet}) {
        if (@client_ids) {
            $wallets = get_all_wallet_camps(client_client_id => \@client_ids);
        }
    }

    foreach my $u (@$result) {
        my $user_role_detailed = $users_detailed_roles->{ $u->{uid} };
        $u->{Role} = $user_role_detailed->{role};

        my $work_currency = $clients_currencies->{$u->{ClientID}}->{work_currency};
        
        # для клиентов возвращаем валюту, в которой необходимо создавать кампании

        if ($self->{api_version_full} > 4) {
            if ( $u->{Role} eq 'client') {
                push @{$u->{ClientCurrencies}}, $work_currency if $work_currency ne 'YND_FIXED';
            } elsif ($u->{Role} eq 'agency'){
                my $agency_currencies = get_agency_allowed_currencies_hash($u->{ClientID}, is_direct => 1);
                push @{$u->{ClientCurrencies}}, map { $_ eq 'YND_FIXED' ? 'cu' : $_ } sort keys %$agency_currencies;
            }
        }
        
        if ($u->{Role} eq 'agency' && $self->{rbac_login_rights}->{role} =~ /^(manager|media|placer|super|support|superreader)$/) {

            my $agency_vars     = $uid2agency_vars->{ $u->{uid} };
            $u->{AgencyName}    = $agency_vars->{agency_name};
            $u->{AgencyUrl}     = $agency_vars->{agency_url};
            $u->{AgencyStatus}  = $agency_vars->{agency_status};

        } else {

            delete $u->{SendAgencyMcbLetters};
            delete $u->{SendAgencyDirectLetters};
        } 

        if ($self->{rbac_login_rights}->{role} =~ /^(manager|super|media|placer|support|superreader)$/) {

            if($u->{Role} eq 'client') {

                my $manager_uids = $managers_of_clients->{ $u->{uid} };
                $u->{ManagersLogins} = @$manager_uids ? 
                                            [map {$uid2manager_login->{$_}} @$manager_uids] : undef;

            } elsif ($u->{Role} eq 'agency') {

                 my $manager_uids = $managers_of_agencies->{ $u->{ClientID} };
                 $u->{ManagersLogins} = @$manager_uids ? 
                                             [ map {$uid2manager_login->{$_}} @$manager_uids] : undef;
            }
        }

        if ($self->{rbac_login_rights}->{role} eq 'agency') {
            $u->{StatusArch} =  $clientid2arch_status->{$u->{ClientID}}{$cur_agency_client_id} || 'No';
        }

        # Получаем размер скидки для данного пользователя
        $u->{Discount} = $clientid2overdraft_info->{ $u->{ClientID} }->{discount} || 0;
        if ($self->{api_version_full} > 4) {
            $u->{VATRate} = $clients_nds->{$u->{ClientID}} if defined $clients_nds->{$u->{ClientID}};

            my $over = $clientid2overdraft_info->{ $u->{ClientID} }->{overdraft_rest} || 0;
            my $magic = 0.50000000001 - 0.5; # по-хорошему здесь нужен модуль десятичной арифметики

            $u->{OverdraftSumAvailable} = undef;
            $u->{OverdraftSumAvailableInCurrency} = 0.01 * POSIX::floor(($over) * 100 + $magic) + 0;

            my $is_show_market_rating = $clients_market_rating->{$u->{ClientID}}{hide_market_rating}
                                    ? 0 : 1;
            $u->{DisplayStoreRating} = $is_show_market_rating ? 'Yes' : 'No';
        }

        if ($O{get_ext_params}) {
            $u->{CampaignEmails} = [ sort {$a cmp $b} @{ $campaign_emails->{$u->{uid}} || [] } ];
            $u->{SmsPhone} = sms_check_user($u->{uid}, $self->{plack_request}->address);
        }

        # добавляем структуру с правами пользователя
        if ( $O{get_ext_params}
            && $self->{rbac_login_rights}->{role} =~ /^(super|superreader|manager|agency|client|media|support|placer)$/
            && $u->{Role} eq 'client'
        ) {
            if ($self->{api_version_full} > 4.5) {
                if ($self->{rbac_login_rights}->{role} eq 'agency') {
                    $u->{Description} = $subclients_rights->{$u->{ClientID}}{$cur_agency_client_id}{client_description};
                }
            }

            my @descriptions;

            foreach my $agClientID (keys %{$subclients_rights->{$u->{ClientID}}}) {
                if ($self->{rbac_login_rights}->{role} =~ /^(super|superreader|manager)$/) {
                    push @descriptions, {
                            AgencyLogin => $clientid2login->{$agClientID}{login}, 
                            Description => $subclients_rights->{$u->{ClientID}}{$agClientID}{client_description}
                        };
                }

                # переименовываем права для отображения в API
                my %client_rights;
                @client_rights{qw/AllowEditCampaigns AllowImportXLS AllowTransferMoney/} = 
                    @{$subclients_rights->{$u->{ClientID}}{$agClientID}}{qw/isSuperSubClient allowImportXLS allowTransferMoney/};

                foreach my $right_name (sort keys %client_rights) {

                    my $right_item = {
                                        RightName => $right_name, 
                                        Value => $client_rights{$right_name} ? 'Yes' : 'No', 
                                        };

                    # для агенства - только права клиента в его агенстве
                    if ($self->{rbac_login_rights}->{role} eq 'agency' && $agClientID == $cur_agency_client_id) { 
                        push @{$u->{ClientRights}}, $right_item;

                    # для менеджера - права клиента во всех его агенствах, + логин агенства
                    } elsif ($self->{rbac_login_rights}->{role} =~ /^(super|support|superreader|manager|client|media|placer)$/) {
                        if ($self->{rbac_login_rights}->{role} !~ /^(client)$/) {
                            $right_item->{AgencyLogin} = $clientid2login->{$agClientID}{login};
                        }
                        if ($self->{api_version_full} > 4) {
                            $right_item->{AgencyName} = $clientid2login->{$agClientID}{name};
                        }
                        push @{$u->{ClientRights}}, $right_item;
                    }
                }
            }
            if ($self->{rbac_login_rights}->{role} =~ /^(super|superreader|manager)$/ && $self->{api_version_full} > 4.5) {
                $u->{Descriptions} = \@descriptions;
            }
        }

        if (! ($u->{Role} eq 'client' && $self->{rbac_login_rights}->{role} =~ /^(super|support|superreader|manager)$/)){
            delete $u->{IsGoodUser};
        }
        
        if ($u->{Role} eq 'manager') {
            delete $u->{SendNews};
            delete $u->{SendAccNews};
            delete $u->{SendWarn};
            delete $u->{SendAgencyMcbLetters};
            delete $u->{SendAgencyDirectLetters};  

            # для менеджеров уточняем роль, для всех остальных оставляем стандартную краткую запись
            # нужно для Direct Client
            $u->{Role} = get_client_role($self->{rbac}, undef, undef, rbac_who_is_detailed => $user_role_detailed);
        }
        
        if ($self->{rbac_login_rights}->{role} =~ /^(super|support|superreader|manager)$/) {
            if ($self->{api_version_full} > 4.5) {
                $u->{CityID} = $u->{geo_id};
                $u->{Description} = $u->{description};
            }
        }

        if ($self->{rbac_login_rights}->{role} =~ /^(super|support|superreader|media|placer|manager)$/) {
            if ($self->{api_version_full} > 4) {
                $u->{UserID} = $u->{uid};
            }
        } else {
            delete $u->{ClientID};
        }
        
        if ($self->{api_version_full} > 4) {

            if (defined $self->{preprocess}{clientuid2wallet}{ $u->{uid} } ) { # забирают все данные о кошельке
                _copy_wallet_options($self, $u);
            } else { # узнаем, будет ли общий счёт у следующей созданной кампании в нашей группе сервисирования
                if ($self->{rbac_login_rights}->{role} eq 'agency') {
                    my $record = firstval {
                        $_->{'agency_client_id'} == $self->{ClientID}
                        && $_->{uid} eq ( $client_chiefs{$u->{uid}} // '' )
                        && $_->{currency} eq $work_currency
                    } @$wallets;
                    $u->{SharedAccountEnabled} = ($record && $record->{is_enabled}) ? "Yes" : "No";
                } elsif ($self->{rbac_login_rights}->{role} =~ /^(super|support|superreader|media|placer|manager)$/) {
                    $u->{SharedAccountEnabled} = undef;
                } else {
                    my @record = grep {!$_->{'agency_client_id'} && $_->{currency} eq $work_currency} @$wallets;
                    $u->{SharedAccountEnabled} = (scalar @record && $record[0]->{is_enabled})? "Yes" : "No";
                }
            }
        }
        
        delete $u->{geo_id};
        delete $u->{description};
        delete $u->{uid};

        if ($u->{Role} eq 'client') {
            $u->{Role} = 'Client';
        } elsif ($u->{Role} eq 'agency') {
            $u->{Role} = 'Agency';
        }

    }

    return $result;
}

# TODO перенести саму функцию в Filter, её вызов — в AccountManagement
=head2 _copy_wallet_options(self, user_info)

    user_info как элемент массива, получаемого в начале get_client_object

    Копирует данные из $self->{preprocess}{clientuid2wallet} в user_info->{WalletsOptions}

=cut

sub _copy_wallet_options($$) {
    my ($self, $u) = @_;

    foreach my $wallet ( @{$self->{preprocess}{clientuid2wallet}{ $u->{uid} }} ) {
        Campaign::correct_sum_and_total($wallet);
        # -- WO - wallet options
        my %WO;
        $WO{Currency}           = $wallet->{currency} if defined $wallet->{currency} && $wallet->{currency} ne 'YND_FIXED';
        $WO{Amount}             = $wallet->{total};

        @{\%WO}{qw/AccountID sms_flags email_notifications sms_time email money_warning_value day_budget/} = 
        @{$wallet}{qw/wallet_cid sms_flags email_notifications sms_time email money_warning_value day_budget/};

        if ( exists $wallet->{agency_name} || exists $wallet->{agency_fio} ) {
            $WO{AgencyName} = length($wallet->{agency_name}) ? $wallet->{agency_name}: $wallet->{agency_fio};
        } elsif ( exists $u->{AgencyName} ) {
            $WO{AgencyName} = $u->{AgencyName};
        }

        $WO{SumAvailableForTransfer} = $wallet->{sum_available};

        push @{$u->{WalletsOptions}}, \%WO;
    }

    return;
}

=head2 api_check_limit_autoload

  api_check_limit_autoload($cid, 'MethodName', $rbac_login_rights);
  при превышении лимита - dieSOAP

=cut

sub api_check_limit_autoload
{
    my ($self, $uid, $method_name, $rbac_login_rights) = @_;    

    # для менеджеров технических лимитов нет
    return if $self->{rbac_login_rights}->{role} =~ /^(manager|super|media|placer|support|superreader)$/;

    my $lim;

    my $has_no_object_limit = 1;
    $has_no_object_limit = 0 if ((defined $OBJECT_LIMITS{$method_name})
        || defined get_spec_method_limit($self->{user_info}->{ClientID}, $method_name));

    if ($self->{user_info}->{ClientID}
        && defined (my $limit = $self->{special_options}->{method_limits}->{$method_name})) {
        $lim = $limit;
    } elsif (defined $USER_LIMITS->{$method_name}) {
        $lim = $USER_LIMITS->{$method_name};
    } elsif ($has_no_object_limit) {
        if ($self->{user_info}->{ClientID}
        && defined (my $limit = $self->{special_options}->{method_limits}->{'other_methods'})) {
            $lim = $limit;
        } else {
            $lim = get_spec_limit($self->{user_info}->{ClientID} || 0, 'api_special_tech_limit') || 50_000;
        }
    }

    if ($lim) {
        api_check_limit($self, _api_limit_key_uid($uid), $method_name, $lim, 1, obj_name => $self->{operator_login});
    }
}

=head2 api_check_limit_by_method(object_id, method_name, count_objects, OPTS)

    Проверяет ограничение на кол-во объектов на пользователя
    OPTS{object_errors} — возвращать объект ошибки

=cut

sub api_check_limit_by_method {
    my ($self, $id, $method_name, $new_count_calls, %OPTS) = @_;    

    # если пользователь в специальном списке - отключаем ограничения
    return 1 if $Settings::UIDS_NO_LIMITS_API{$self->{uid}} || $self->{rbac_login_rights}->{role} eq 'manager';

    if ($method_name
            && $self->{user_info}->{ClientID}
            && has_spec_limits($self->{user_info}->{ClientID})
            && get_spec_method_limit($self->{user_info}{ClientID}, $method_name)) {

        api_check_limit($self, $id, $method_name, get_spec_method_limit($self->{user_info}{ClientID}, $method_name), $new_count_calls, %OPTS);
    } elsif ($self->{user_info}->{ClientID}
        && defined (my $limit = $self->{special_options}->{method_limits}->{$method_name})) {
        api_check_limit($self, $id, $method_name, $limit, $new_count_calls, %OPTS);
    } elsif ($method_name && defined $OBJECT_LIMITS{$method_name}) {
        api_check_limit($self, $id, $method_name, $OBJECT_LIMITS{$method_name}, $new_count_calls, %OPTS);
    } else {
        $OPTS{object_errors}? return(get_error_object('500')) : dieSOAP('500');
    }
}


=head2 my ($calls_count_key, $last_time_key) = _api_limit_keys($id, $method_name);

    сформировать ключи, используемые в memcached для хранения
    - количества совершённых вызовов
    - времени последнего вызова (unix timestamp)
    
=cut

sub _api_limit_keys {
    my ($id, $method_name) = @_;
    my $md_prefix = "api_tl:{$id}:$method_name:".today(); # значение в фигурных скобках для определения слота в redis-кластере
    return "$md_prefix:c", "$md_prefix:t";
}

=head2 api_check_limit

    Проверяет технологический лимит
    OPTS{object_errors} — возвращать объект ошибки

=cut

sub api_check_limit
{
    my ($self, $id, $method_name, $limit, $new_count_calls, %OPTS) = @_;

    # если пользователь в специальном списке - отключаем ограничения
    return 1 if $Settings::UIDS_NO_LIMITS_API{$self->{uid}} || $self->{rbac_login_rights}->{role} eq 'manager';

    my $redis = DirectRedis::get_redis();
    my ($calls_count_key, $last_time_key) = _api_limit_keys($id, $method_name);
    my $cur_limit_data = try { $redis->mget($calls_count_key, $last_time_key) };
    my ($count_calls, $last_time);
    if ($cur_limit_data && @$cur_limit_data) {
        ($count_calls, $last_time)  = @$cur_limit_data;
    } else {
        warn "Can't get tech_limit data from redis";
    }

    $count_calls //= 0;
    $last_time //= 0;
    
    my $count_increment = $new_count_calls || 1;

    if ($count_increment > $limit
        || $count_calls + $count_increment > $limit 
           && (
              $API_LIMIT_IGNORE_DELAY->{$method_name} 
              || time() - $last_time < $API_LIMIT_MAX_TIME_DELAY
           )
    ) {
        if ($OPTS{object_errors}) {
            return get_error_object('LimitExceed', iget('Лимит для %s -  %d, уже вызвано %d, попытка вызвать %d', ($OPTS{obj_name} || $id), $limit, $count_calls, $count_increment));
        } else {
            dieSOAP('LimitExceed', iget('Лимит для %s -  %d, уже вызвано %d, попытка вызвать %d', ($OPTS{obj_name} || $id), $limit, $count_calls, $count_increment));
        }
    }

    return $count_calls;
}

=head2 _api_limit_key_uid

    Формирует часть ключа для подсчета лимита вызовов метода на пользователя.
    Голый uid использовать нельзя, т.к. могут быть коллизии с другими типами Id

=cut

sub _api_limit_key_uid  {
    my $uid = shift;
    return "UID$uid";
}

=head2 api_update_limit_uid

    Обновляет технологический лимит на uid

=cut


sub api_update_limit_uid {
    my ($self, $uid, @params) = @_;
    return api_update_limit($self, _api_limit_key_uid($uid), @params);
}

=head2 api_update_limit

    Обновляет технологический лимит

=cut

sub api_update_limit {
    my ($self, $id, $method_name, $new_count_calls) = @_;

    # если пользователь в специальном списке - отключаем ограничения
    return if $Settings::UIDS_NO_LIMITS_API{$self->{uid}} || $self->{rbac_login_rights}->{role} eq 'manager';

    my $redis = DirectRedis::get_redis();
    my ($calls_count_key, $last_time_key) = _api_limit_keys($id, $method_name);

    my $counter = try { $redis->incrby($calls_count_key, $new_count_calls) }; # cоздаем ключ в redis, если его нет
    if (!$counter) {
        warn "Can't increment tech limit $calls_count_key in redis";
        return;
    } else {
        if ($counter == $new_count_calls) { # создали новый ключ
            if (!try { $redis->expire($calls_count_key, 24*60*60) }) {
                warn "Can't set expiration for tech limit $calls_count_key in redis";
            }
        }
        my $log_data = {
            uid => $self->{uid},
            span_id => Yandex::Trace::current_span_id(),
            method => $method_name,
            limit_for_id => $id,
            calls_count_key => $calls_count_key,
            calls_count_value => $new_count_calls
        };
        log_api_tech_limit_update($log_data)
    }

    if (!$API_LIMIT_IGNORE_DELAY->{$method_name}) {
        if (!try { $redis->setex($last_time_key, 24*60*60, time()) }) {
            warn "Can't set record $last_time_key in redis";
        }
    }
}


=head2 check_avaliable_version(api_version)

    Проверяет доступна ли сейчас указанная версия API

=cut

sub check_avaliable_version
{
    my $api_version = shift;
    my $latest_flag = shift;

    my $version_label = get_api_version_label({ api_version => $api_version, latest => $latest_flag });

    if (! $api_version || !defined $Settings::AVAILABLE_API_VERSIONS{$version_label} ) {
        return 'not';
    }

    if ( is_production() && $Settings::AVAILABLE_API_VERSIONS{$version_label}{hidden} ) {
        return 'not';
    }

    if ($Settings::AVAILABLE_API_VERSIONS{$version_label}{blocked}) {
         # означает, что версия заблокирована и возможен доступ только с разрешения
        return 'blocked';
    }

    return 'ok';
}

sub get_api_version_label
{
    my $param = shift;

    my $version_label = $param->{api_version};
    $version_label .= $param->{latest} ? "-latest" : "";

    return $version_label;
}


=head2 check_access_user_version

    Проверяем доступна ли пользователю заблокированная версия API
    Проверяем по специальному флагу в настройках пользователя -- устанавливается в интерфейсе
    Так же разрешаем супер пользователям

=cut

sub check_access_user_version
{
    my ($self) = @_;
    my $allowed_flag = $self->{user_info}{api_allow_old_versions} || 'No';

    if ($allowed_flag ne 'Yes' && ! $self->{rbac_login_rights}{super_control}) {
        return 0;
    }

    return 1;
}

=head2 get_user_advq_queries_limit
    
    Возвращает дневной лимит пользователя на запросы к ADVQ,
    если значение в базе нет возвращает лимит по умолчанию

=cut

sub get_user_advq_queries_limit
{
    my $uid = shift;
    my $limit = get_one_field_sql(PPC(uid => $uid), 'select advq_queries_lim from users_api_options where uid = ?', $uid);

    if ( ! $limit ) {
        $limit = $Settings::DEFAULT_WORDSTAT_DAILY_LIMIT;

        create_update_user( $uid, {advq_queries_lim => $limit} );

    }

    return $limit;

}

=head3 %price_base

    Переименование названий позиций во внутренние

=cut

my %price_base = (
    min => PlacePrice::get_guarantee_entry_place(),
    max => $PlacePrice::PLACES{GUARANTEE1},
    pmin => PlacePrice::get_premium_entry_place(),
    pmax => $PlacePrice::PLACES{PREMIUM1},
    P11 => $PlacePrice::PLACES{PREMIUM1},
    P12 => $PlacePrice::PLACES{PREMIUM2},
    P13 => $PlacePrice::PLACES{PREMIUM3},
    P14 => $PlacePrice::PLACES{PREMIUM4},
);

=head2 create_auto_price_structure

    Формирует структуру для функции set_auto_price из параметров API

=cut

sub create_auto_price_structure
{
    my ($vars, %O) = @_;

    # multicurrency: копируем для возможности конвертации валют "на месте"
    my $params = {%$vars};

    die "Not specified dest_currency" unless $O{dest_currency};

    foreach my $field (qw/Price ContextPrice SinglePrice MaxPrice/) {
        next unless $params->{$field};

        # multicurrency
        if (($params->{Currency} || 'YND_FIXED') ne $O{dest_currency}) {
            $params->{$field} = currency_price_rounding(
                convert_currency($params->{$field}, $params->{Currency} || 'YND_FIXED', $O{dest_currency}), $O{dest_currency}, up => 1);
        } else {
            $params->{$field} = round2s($params->{$field});
        }
    }

    my $func_params;
    if ($O{is_different_places}) {
        $func_params->{for_different_places} = 1;
        $func_params->{currency} = $O{dest_currency};
        
        if ($params->{Mode} eq 'SinglePrice') {

            $func_params->{single} = {
                                price => $params->{Price} || $params->{SinglePrice} || undef,
                                price_ctx => $params->{ContextPrice} || $params->{SinglePrice} || undef
                                };
            
        } elsif ($params->{Mode} eq 'Wizard') {
            my $tmp_struct = {
                    price_base        => $price_base{$params->{PriceBase}} || $price_base{'min'} # должно быть определено, если используется
                  , proc              => $params->{Proc}     || 0
                  , proc_base         => $params->{ProcBase} || 'diff' # допустимо только при неопределённом proc и тогда не повлияет на результат
                  , max_price         => $params->{MaxPrice} || get_currency_constant($O{dest_currency}, 'MAX_PRICE')
                  , scope             => $params->{Scope} || 0
                  , update_phrases    => 1
            };
            
            if ( $params->{PhrasesType} && $params->{PhrasesType} =~ m/^(Both|Network)$/ ) {
                $tmp_struct->{auto_max_price} = 1 unless $params->{MaxPrice}; # поправка на r40002 (DIRECT-20545)
                $func_params->{on_ctx} = $tmp_struct;
            }

            if (! defined $params->{PhrasesType} || $params->{PhrasesType} =~ m/^(Both|Search)$/) {
                $func_params->{on_search} = yclone($tmp_struct); # Иначе JSON::Syck::Dump не сработает
            }
        }
    } else {
        if ($params->{Mode} eq 'SinglePrice') {
            
            unless ($params->{Price} || $params->{SinglePrice}) {
                dieSOAP('BadParams', iget('Должно быть указано либо поле Price, либо поле SinglePrice'));
            }
        
            $func_params = {
                                single_price => $params->{Price} || $params->{SinglePrice}
                              , platform => 'both'
                              , currency => $O{dest_currency},
                           };
            
        } elsif ($params->{Mode} eq 'Wizard') {
            $func_params = {
                    price_base        => $price_base{$params->{PriceBase}} || $price_base{'min'} # должно быть определено, если используется
                  , proc              => $params->{Proc}     || 0
                  , proc_base         => $params->{ProcBase} || 'diff' # допустимо только при неопределённом proc и тогда не повлияет на результат
                  , max_price         => $params->{MaxPrice} || get_currency_constant($O{dest_currency}, 'MAX_PRICE')
                  , scope             => $params->{Scope} || 0
                  , update_phrases    => 1
                  , platform          => ($params->{PhrasesType}? lc($params->{PhrasesType}) : 'search')
                  , currency          => $O{dest_currency},
                };               
        }
    }
    $func_params->{change_all_banners} = ($params->{BannersType} && $params->{BannersType} eq 'All' ? 1 : 0);
    return $func_params;
}

=head2 create_pay_campaign_request($self, \@payments; %O)

    Формирует запрос для счета в Биллинге и создает его на указанные в запросе суммы.

    @payments = ( {
        CampaignID => номер кампании
        , Sum => сумма для счета (будет сконвертирована в валюту кампании!)
        , Currency => валюта, в которой указана сумма
    }, ...)

    Options:
        prepare_only => 1 - только сформировать структуру для счета, но не создавать его

    (!) функция теперь ожидает наличие $self->{_preprocessed} следующей структуры:
    cid => {cid => cid, uid => ..., type => ...}
    TODO:
        вынести из функции все посторонние для нее действия

    Возвращает ссылку на хеш {result => ..., error => ...}

=cut

sub create_pay_campaign_request($$;%)
{
    my ($self, $payments, %O) = @_;

    # берем первый попавшийся uid и флаг multiuser. Если последний = 1, то в процессе выполнения 
    # prepare_and_validate_pay_camp uid будет подменен
    my ($client_uid, $multiuser, %cids2uids, $client_chief_uid);
    foreach my $camp(map {$self->{_preprocessed}->{$_->{CampaignID}}} @$payments) {
        $client_uid = $camp->{uid} unless $client_uid;
        $multiuser = 1 if $client_uid ne $camp->{uid};
        $cids2uids{$camp->{cid}} = $camp->{uid};
    }
    
    my $uids2wallets = {};
    my @wallet_request;
    my %uids2clientids = %{get_uid2clientid(uid => [uniq values %cids2uids])};
    my %chiefs_by_clientid = %{rbac_get_chief_reps_of_clients([uniq values %uids2clientids])};
    my %chiefs = map {$_ => $chiefs_by_clientid{$uids2clientids{$_}}} keys %uids2clientids;

    # проверяем целостность
    if (any {!defined($_)} values %chiefs) {
        return {error => get_error_object('500')};
    }

    if ((scalar (uniq map {$self->{_preprocessed}->{$_->{CampaignID}}->{type}} @$payments)) > 1) {
        # на входе либо все кошельки, либо все кампании. Если иначе — сломана проверка
        return {error => get_error_object('InternalLogicError')};
    }

    my @clientids = values %uids2clientids;
    my $clients_currencies = mass_get_client_currencies(\@clientids);

    # общий счёт: подменяем номера кампаний на номера кошельков и группируем
    foreach my $uid(uniq values %cids2uids) {
        my $agency_client_id = 0;
        if ($self->{rbac_login_rights}->{role} && $self->{rbac_login_rights}->{role} eq 'agency') {
            $agency_client_id = $self->{ClientID};
        }

        $client_chief_uid = $chiefs{$uid} if ($uid == $client_uid);
        my $client_id = $uids2clientids{$uid};
        my $c = DirectContext->new({
                is_direct        => 1,# Директ/Баян
                UID              => $self->{uid},
                uid              => $uid,
                client_chief_uid => $chiefs{$uid},
                rbac             => $self->{rbac},
                rights           => $self->{rbac_rights},
                login_rights     => $self->{rbac_login_rights},
                client_client_id => $client_id,
                user_ip          => $self->{plack_request}->address || '127.0.0.1',
            });

        push @wallet_request, {agency_client_id => $agency_client_id, c => $c, client_currency => $clients_currencies->{$client_id}->{work_currency}};
    }
    my $is_services_app = ($self->{application_id} eq $API::Settings::YNDX_SERVICES_APP_ID) ? 1 : 0;
    my $wallets = Wallet::get_wallets_by_uids(\@wallet_request, force_pay_before_moderation => $is_services_app);
    my @records;
    if ($self->{rbac_login_rights}->{role} eq 'agency') {
        @records = grep {$_->{'agency_client_id'} == $self->{ClientID}} @$wallets;
    } else {
        @records = grep {!$_->{'agency_client_id'}} @$wallets;
    }

    foreach my $uid(uniq values %cids2uids) {
        my @record = grep {$_->{'uid'} == $chiefs{$uid}} @records; # два grep подряд должны выбрать единственное значение по связке (client_chief_uid, agency_client_id, currency)
        my $client_id = $uids2clientids{$uid};
        my $client_currency = $clients_currencies->{$client_id}->{work_currency};
        my $wallet = firstval {$_->{wallet_camp}->{enabled}} @record;
        if ($wallet->{wallet_camp}->{enabled}) {
            unless ($wallet->{wallet_camp}->{allow_pay}) {
                return {error => get_error_object('BadPayCamp', iget("Невозможно оплатить общий счёт %s", $wallet->{wallet_camp}->{wallet_cid}))};
            }
            $uids2wallets->{$uid} = $wallet->{wallet_camp};
        }
    }

    my $wallet_payments = {}; # здесь только непрямые платежи на кошельки
    foreach my $payment(@$payments) { # если на входе нашёлся кошелёк, значит, что тут только кошельки
        last if $self->{_preprocessed}->{$payment->{CampaignID}}->{type} && $self->{_preprocessed}->{$payment->{CampaignID}}->{type} eq 'wallet';

        if ($uids2wallets->{$cids2uids{$payment->{CampaignID}}} && $uids2wallets->{$cids2uids{$payment->{CampaignID}}}->{enabled}) {
            # валюта внутри одного запроса одна и та же
            $wallet_payments->{$uids2wallets->{$cids2uids{$payment->{CampaignID}}}->{wallet_cid}}->{Sum} += $payment->{Sum};
            $wallet_payments->{$uids2wallets->{$cids2uids{$payment->{CampaignID}}}->{wallet_cid}}->{Currency} = $payment->{Currency};
        }
    }

    # сначала добавим обычные кампании, затем прицепим кошельки, если есть (или только кошельки, если они были на входе)
    my @payments2;
    foreach my $payment(@$payments) {
        my $uid = $cids2uids{$payment->{CampaignID}};
        next if (exists $uids2wallets->{$uid} && exists $wallet_payments->{$uids2wallets->{$uid}->{wallet_cid}} && $wallet_payments->{$uids2wallets->{$uid}->{wallet_cid}}); # пропускаем, добавим к платежам позже
        push @payments2, $payment;
    }

    foreach my $wallet_cid(keys %$wallet_payments) {
        push @payments2, {CampaignID => $wallet_cid, Sum => $wallet_payments->{$wallet_cid}->{Sum}, Currency => $wallet_payments->{$wallet_cid}->{Currency}};
    }
    my $cids_array = [map {$_->{CampaignID}} @payments2];
    my $cid2currency = get_hash_sql(PPC(cid => $cids_array), ["select cid, IFNULL(currency, 'YND_FIXED') currency from campaigns", where => { cid => SHARD_IDS }]);

    my $sums = {};
    my %with_nds_flag;
    foreach my $p(@payments2) { # у всех кампаний одна и та же валюта
        my $sum;
        if ($p->{Sum} == 0
            || $cid2currency->{$p->{CampaignID}} eq 'YND_FIXED'
            || ($self->{api_version_full} > 4 && $p->{Currency} && $cid2currency->{$p->{CampaignID}} eq $p->{Currency})) {
            $sum = $p->{Sum};
            $with_nds_flag{$p->{CampaignID}} = 1;
        } else { # иначе нужно переводить валюту. При этом валюта запроса — фишки, и нужно добавить НДС клиента
            $sum = convert_currency($p->{Sum}, 'YND_FIXED', $cid2currency->{ $p->{CampaignID} }, with_nds => 0 );
            # математическое округление. Если конвертируются из фишек, потом всё равно добавится НДС
            $sum = round2s($sum);
            $with_nds_flag{$p->{CampaignID}} = 0;
        }

        $sums->{ $p->{CampaignID} } = $sum;
    }

    my $options = {
        sums => $sums,
        multiuser => $multiuser,
        is_easy_payment => 0,
        pseudo_currency => get_pseudo_currency( id => 'rub' ),
        client_chief_uid => $client_chief_uid,
        with_nds => \%with_nds_flag
    };

    if (scalar keys %{$options->{sums}}) {
        $options->{agency_uid} = rbac_is_agencycampaign($self->{rbac}, [keys %{$options->{sums}}]->[0]);
    }

    $options = prepare_and_validate_pay_camp($client_uid, $self->{uid}, $self->{rbac}, $self->{rbac_login_rights}, %{$options});

    if (defined $options->{error}) {
        return {error => get_error_object('BadPayCamp', $options->{error})};
    }

    if ( !defined $options->{error_code} ) {

        return {result => $options} if $O{prepare_only};
        return _create_pay_campaign_request_in_balance($self, $client_uid, $options, $payments);

    } else {
        return {error => get_error_object('BadPayCamp', pay_error_code_2_text(%$options, is_direct => 1, easy_direct => 0))};
    }
}

=head2 _create_pay_campaign_request_in_balance(self, client_uid, options, payments)

    Передать запрос на выставление счёта в баланс и разослать уведомления

    payments передаётся только для рассылки уведомлений, информация для баланса в options
            # отправляем нотификации
            my $cid2currency = get_hashes_hash_sql(PPC, [
                "SELECT cid, IFNULL(currency, 'YND_FIXED') as currency
                FROM campaigns",
                WHERE => {cid => [map {$_->{CampaignID}} @$payments]}
            ]);
            for (@{$options->{pay_notification_data}}) {
                mail_notification('camp', 'c_pay_multicurrency', $_->[0], 'com', "$_->[1]:$cid2currency->{$_->[0]}->{currency}", $_->[2]);
            }

    uid оператора ($self->{uid})

    TODO: отцепить код отправки уведомлений.

=cut

sub _create_pay_campaign_request_in_balance($$$$) {
    my ($self, $client_uid, $options, $payments) = @_;
    my $billing_url;
    my $cids_str = join (',', map {$_->{CampaignID}} @{$payments});
    my $uid = $self->{uid};
    eval {
        # делаем запрос в баланс

        # определяем общие параметры реквеста
        my $request_params = {
            Overdraft =>  $options->{overdraft_bill},
            UiType => 'std',
            ReturnPath => "http://direct.yandex.ru/registered/main.pl?cmd=showCamps&ulogin=" . uri_escape_utf8(get_login(uid => $client_uid)),
            DenyPromocode => 1,  # Никаких промокодов в счетах, выставленных из API
        };

        my $can_pay_before_moderation = get_client_data($self->{ClientID}, ['feature_payment_before_moderation'])->{feature_payment_before_moderation};
        my $is_services_app = ($self->{application_id} eq $API::Settings::YNDX_SERVICES_APP_ID) ? 1 : 0;
        $request_params->{ForceUnmoderated} = ($can_pay_before_moderation || $is_services_app) ? 1 : 0;

        $billing_url = balance_create_request($uid, $options->{client_id}, $options->{CreateRequest}, $request_params);
        # обновляем campaigns.sum_to_pay
        while( my ( $cid, $sum ) = each %{$options->{sums}} ) {
            do_sql(PPC(cid => $cid), "UPDATE campaigns set sum_to_pay = ? where cid=?", $sum, $cid);
        }

        my $cids_array = [map {$_->{CampaignID}} @$payments];
        # отправляем нотификации
        my $cid2currency = get_hash_sql(PPC(cid => $cids_array), [
            "SELECT cid, IFNULL(currency, 'YND_FIXED') as currency
            FROM campaigns",
            WHERE => {cid => SHARD_IDS}
        ]);

        for (@{$payments}) {
            mail_notification('camp', 'c_pay_multicurrency', $_->{CampaignID}, 'com', "$_->{Sum}:$cid2currency->{$_->{CampaignID}}", $uid);
        }

    };

    if ($billing_url) {
        return {result => $billing_url};
    } elsif ($@) {
        my $msg = "Error creating balance invoice for campaigns $cids_str: $@";
        send_alert(Carp::longmess($msg), 'create_pay_campaign_request error');
        return {error => get_error_object('FinOpsTmpUnavail')};
    }

    return {error => get_error_object('BadPayCamp')};
}

=head2 get_user_api_params

    Возвращает некоторые параметры пользователя, нужно для контроллеров API в интерфейсе

=cut

sub get_user_api_params
{
    my ($rbac, $cl_uid, $UID, %O) = @_;

    my $params = get_user_data($cl_uid, [qw/api_offer 
                                            api_developer_name 
                                            api_developer_email 
                                            api_allowed_ips 
                                            api_allow_finance_operations 
                                            api_send_mail_notifications
                                            email 
                                            fio
                                            login/]);

    $params->{api_enabled} = rbac_can_use_api($rbac, {uid => $cl_uid, UID => $UID});
    $params->{api_allow_finance_operations} = ($params->{api_allow_finance_operations} && $params->{api_allow_finance_operations} eq 'Yes') ? 1 : 0;
    $params->{api_send_mail_notifications} = ($params->{api_send_mail_notifications} && $params->{api_send_mail_notifications} eq 'Yes') ? 1 : 0;

    hash_merge $params, get_one_line_sql(PPCDICT, 'select master_token_timecreated, finance_cnt_dropped_time 
                                            from api_finance_tokens where uid = ?', $cl_uid);

    my $uhost = new APIUnits({ scheme => 'API' });
    $params->{api_units} = $uhost->check_or_init_user_units($cl_uid)->{$cl_uid}{units_rest};

    return $params if $O{ext};

    return hash_cut($params, qw/api_units
                                api_enabled
                                api_offer
                                master_token_timecreated
                                finance_cnt_dropped_time
                                email
                                fio
                                application_queries
                                login/);
}

=head2 get_user_api_apps_used_last_approx_2d($uid)

    Получаем данные на uid по приложениям которыми пользователь ходил в API, с количеством вызовов, для сортировки.
    Получаем за вчера и сегодня.

    Результат {
        application_id => requests_count
    }

=cut

sub get_user_api_apps_used_last_approx_2d {
    my $uid = shift;
    my $more_that_24h_hours_seconds = 25*60*60;
    my $sql = "SELECT application_id, count() as cnt
        FROM ppclog_api
        PREWHERE
            ". sql_condition({uid__int => $uid}) . "
        WHERE log_date >= toDate(now()-$more_that_24h_hours_seconds)
        GROUP BY application_id
        FORMAT JSON";

    my $clh = get_clickhouse_handler('cloud');
    local $clh->{timeout} = 5;
    my $result;
    eval {
        $result = $clh->query($sql)->json->{data};
    };
    warn $@ if $@;
    return {} unless $result;

    my $application_requests = {};
    foreach my $row (@$result) {
        $application_requests->{$row->{application_id}} = $row->{cnt};
    }

    return $application_requests;
}

=head2 notification_api_finance

=cut

sub notification_api_finance
{
    my ($cl_uid, $rbac, $operator_uid, $template_name, $vars) = @_;

    $vars ||= {};

    if (! is_sandbox()) {
        $vars = hash_merge $vars, get_user_api_params($rbac, $cl_uid, $operator_uid, ext => 1);
        if ($vars->{api_send_mail_notifications} && $template_name && $vars->{email}) {
            send_prepared_mail($template_name, $cl_uid, $Settings::NOTIFICATION_EMAIL_FROM, $vars);
        }
    }
    return 1;
}

=head2 get_max_available_api_version

=cut

sub get_max_available_api_version
{
    return max(
                grep {
                        ! $Settings::AVAILABLE_API_VERSIONS{$_}{hidden} 
                        && ! $Settings::AVAILABLE_API_VERSIONS{$_}{blocked} 
                        && /^\d+$/ 
                } keys %Settings::AVAILABLE_API_VERSIONS
    );
}

sub get_campaigns_tags_struct {
    my ($cids) = @_;

    my $tags = mass_get_all_campaign_tags($cids);

    my @result;

    foreach my $cid (keys %$tags) {
        my $campaign_tags = {CampaignID => $cid, Tags => []};
        foreach my $tag (@{$tags->{$cid}}) {
            push @{$campaign_tags->{Tags}}, {TagID => $tag->{tag_id}, Tag => $tag->{value}};
        }

        push @result, $campaign_tags;
    }

    return \@result;
}

=head2 check_edit_phrase(new_phrase, old_phrase)

    Проверяет изменилась ли фраза, или параметры фразы в запросе

=cut

sub check_edit_phrase
{
    my ($new, $old, $campaign_currency, $phrase_bids_currency) = @_;

    my $edit_phrase_flag = ModerateChecks::check_moderate_phrase($new->{phrase}, $old->{phrase});
    my $params_changed_flag;
    if ($edit_phrase_flag != 1) { # фраза не изменилась, сравниваем ставки и приоритет фразы
        foreach my $f (qw/price price_context/) {
            next unless defined $new->{$f};

            my $old_price = $old->{$f} + 0;
            if ($old_price && $phrase_bids_currency ne $campaign_currency) { # если ставка задана не в валюте кампании, чтобы избежать ошибок округления, переводим ставку из базы в валюту фразы, как это делается в GetBanners
                $old_price = round2s(convert_currency($old_price, $campaign_currency, $phrase_bids_currency));
            }
            my $new_price = $new->{$f} + 0;

            if ("$old_price" ne "$new_price") {
                $params_changed_flag = 1;
                last;
            }
        }

        if (   !$params_changed_flag
            && defined $new->{autobudgetPriority}
            && $Phrase::PRIORITY_VALUES{ $new->{autobudgetPriority} } != $old->{autobudgetPriority}) {
            $params_changed_flag = 1;
        }

        $params_changed_flag //= 0;
    } else {
        $params_changed_flag = 1;
    }

    return {
        flag => $params_changed_flag
        , edit_phrase => $edit_phrase_flag
    };
}

sub status_moderate_ext_to_int {
    my $statuses = shift;
    my @result;
    for my $status (@$statuses) {
        if ($status =~ m/^(New|Yes|No)$/) {
            push @result, $status;
        } elsif ($status eq 'Pending') {
            push @result, 'Sent','Sending','Ready';
        }
    }
    return \@result;
}

sub get_modreasons_into_banners {
    my ($banners, %types) = @_;
 
    my @banner_ids = map { $_->{BannerID} }
                grep {
                    $types{banner} && defined $_->{bannerStatusModerate} && $_->{bannerStatusModerate} eq 'No'
                         || $types{phrases} && defined $_->{phrasesStatusModerate} && $_->{phrasesStatusModerate} eq 'No'
                         || $types{contactinfo} && defined $_->{PhoneFlag} && $_->{PhoneFlag} eq 'No'
                         || $types{sitelinks_set} && defined $_->{statusSitelinksModerate} && $_->{statusSitelinksModerate} eq 'No'
                         || $types{image} && defined $_->{image_statusModerate} && $_->{image_statusModerate} eq 'No'
                } @$banners;
    
    if (@banner_ids) {

        my $mass_diags = mass_get_diags(\@banner_ids, 'banner', banner_diags => 1);
        my $objtype_int2ext = {
            'banner' => 'Banner',
            'phrases' => 'Phrases',
            'contactinfo' => 'ContactInfo',
            'sitelinks_set' => 'Sitelinks',
            'image' => 'AdImage',
        };

        foreach my $banner (@$banners) {
            my $diags = $mass_diags->{$banner->{BannerID}};
            
            foreach my $obj_type (keys %{$diags->{banner_diags} || {}}) {
                next if ! $types{$obj_type};
                next if !defined $objtype_int2ext->{$obj_type};
                
                my $list = $diags->{banner_diags}{$obj_type} || [];
                foreach my $reason (@$list) {
                    push @{$banner->{ModerateRejectionReasons}}, {
                        Type => $objtype_int2ext->{$obj_type},
                        Text => iget($reason->{diag_text}),
                    };
                }
            }
        }
    }
}

1;
