package APIMethods;

# $Id$

=head1 NAME

APIMethods - Старые методы АПИ к Директу.

=head1 DESCRIPTION

    Методы АПИ к Директу.
    Вызов метода и расчет необходимых параметров происходят в API.pm

    Ранее данный пакет с методами находился внутри API.pm

=cut

use strict;
use warnings;

use Data::Dumper;
use List::Util qw/max min sum/;
use Digest::MD5 qw(md5_hex);
use YAML qw(Dump Load); # Load никто уже не зовёт
use List::MoreUtils qw/uniq any firstval/;

use Settings;
use Common qw(:globals :subs);
use Forecast;
use Suggestions;
use TimeTarget;
use Yandex::DBTools;
use Yandex::DBShards;
use YaCatalogApi;
use geo_regions;

use EnvTools;
use TextTools;
use Tools;
use API::Errors;
use API::Filter;
use API::ReportCommon;
use APICommon qw(:subs);

use Primitives;
use PrimitivesIds;
use Campaign;
use CampaignTools;
use PhraseText;
use Phrase;

use Client;
use GeoTools;
use CampStat;

use Yandex::I18n;
use Yandex::DateTime;
use Yandex::MyGoodWords;

use APIUnits;

use MetrikaCounters ();
use Client::ClientFeatures ();

use utf8;

use base qw/DoCmd::Base/;
# определяем специальные правила для проверок в rbac
sub rbac_rights
    :Rbac(Cmd => api_userSettings, Code => rbac_cmd_by_owners)
    :Rbac(Cmd => api_userSettingsModify, Code => rbac_cmd_by_owners, ExceptRole => [media, placer, superreader])
    :Rbac(Cmd => api_showCampStat, Code => rbac_cmd_showCampStat)
    :Rbac(Cmd => api_updPrices, Code => rbac_cmd_user_allow_edit_camps, ExceptRole => media)
    :Rbac(Cmd => api_editCamps, Code => rbac_cmd_user_allow_edit_camps, ExceptRole => media)
    :Rbac(Cmd => api_showCamp, Code => rbac_cmd_user_allow_show_camps)
    :Rbac(Cmd => api_getSubClients, Code => rbac_cmd_by_owners)
    :Rbac(Cmd => api_payCamp, Code => [ rbac_cmd_pay_api, rbac_cmd_check_client_login ])
    :Rbac(Cmd => api_transferMoney, Code => rbac_cmd_transfer)
    :Rbac(Cmd => api_deleteCamp, Code => rbac_cmd_user_allow_edit_camps)
{}

=head2 GetReportList

    Получить список всех имеющихся отчетов
    Возвращает массив хэшей - с номером отчета и его статусом

=cut

sub GetReportList
{
    my $self = shift;

    my $reports = get_queue_reports_list("report", $self->{uid});

    # вычисляем домен для возврата ссылки на отчет
    my $r = $self->{plack_request};
    my $hostname = "https://".$r->uri->host;
    my $reports_list = filter_reports_list("report", $reports, hostname => $hostname);

    return $reports_list;
}

=head2 DeleteReport

    Удаление отчета( готов или нет - не важно )
    Каждый пользователь может заказать ограниченное кол-во отчетов - 5

=cut

sub DeleteReport
{
    my ($self, $report_id) = @_;

    # получаем указанный отчет - для того чтобы узнать номер кампании
    my $job = get_queue_report_db( "report", $report_id, $self->{uid} );

    dieSOAP('NoReport') if !$job || $job->is_revoked;

    my $args = $job->args;
    my $error_code = check_rbac_rights( $self, 'api_showCampStat', { stat_type => 'custom', cid => ($args->{CampaignID} || $args->{campaignID}), UID => $self->{uid} } );
    if ($error_code) {
        dieSOAP( 'NoRights' );
    }

    # помечаем отчет в базе удаленным
    return $job->mark_revoked;
}

=head2 DeleteForecastReport

Удаляет указанный отчёт с прогнозом бюджета

=cut

sub DeleteForecastReport
{
    my ($self, $forecast_id) = @_;

    my $job = $self->{_preprocessed}->{report};

    return $job->mark_revoked;
}

=head2 GetRegions()

    Получить список регионов в Директе(!) и их ID - список отличается от геобазы кол-вом городов
    Язык названий регионов задается в заголовке запроса (locale)

=cut

sub GetRegions
{
    my $self = shift;

    my @fields;
    if ( $self->{api_version_full} > 4 ) {
        @fields = qw( RegionID RegionName ParentID RegionType );
    } else {
        @fields = qw( RegionID RegionName ParentID );

    }

    my $regions = get_geo_regions_data( fields => \@fields );
    return $regions;
}

=head2 GetMetroStations()

=cut

sub GetMetroStations
{
    my ($self, $params) = @_;

    my %cities;
    if ($params && $params->{RegionIDS}) {
        my @cities = @{$params->{RegionIDS}};
        %cities = map {$_ => 1} @cities;
        $cities{1} = 1 if $cities{213};
    }

    my @result;
    while( my( $geo_id, $stations ) = each %geo_regions::METRO ) {
        next if scalar keys %cities && ! exists $cities{$geo_id};

        foreach my $station (@$stations) {
            push @result, {
                            RegionID   => $geo_id,
                            MetroStationName  => $station->{name},
                            MetroStationID    => $station->{region_id},
                          };
        }
    }
    return \@result;
}

=head2 GetRubrics()

    Получить список рубрик Я.Каталога и их ID
    Язык названий рубрик задается в заголовке запроса (locale)

=cut

sub GetRubrics
{
    my $self = shift;
    my $cat = get_filtered_categories();
    my $ya_catalog = [];

    my $rub_field = Yandex::I18n::current_lang() eq 'en' ? 'short_engl_name' : 'name';

    while( my ( $id, $h ) = each %$cat ) {
        my $r = {
                RubricID => $id,
                RubricName => $h->{$rub_field} || $h->{name}, # name - на случай отсутствия перевода
                Url => get_category_url( $id ),
                ParentID => $h->{path}->[-1],
        };

        $r->{RubricFullName} = get_category_name( $id ) || $r->{RubricName} || '';

        $r->{Checkable} = $h->{checkable} ? 'Yes' : 'No';

        push @$ya_catalog, $r;
    }

    $ya_catalog = [ sort {$a->{ParentID} <=> $b->{ParentID}} sort {$a->{RubricID} <=> $b->{RubricID}} @$ya_catalog ];

    return $ya_catalog;

}

=head2 GetTimeZones

    Получить список таймзон, доступных в Директе
    Язык названий таймзон задается в заголовке запроса (locale)

=cut

sub GetTimeZones
{
    my $self = shift;
    my @ret;

    for my $group (@{TimeTarget::get_timezone_groups()}) {
        for my $tz (@{$group->{timezones}}) {
            my $item = {
                TimeZone => $tz->{timezone},
                GMTOffset => $tz->{offset},
                Name => $tz->{name},
            };
            
            push @ret, $item;
        }
    }

    return \@ret;
}

=head2 int S<CreateNewForecast>( FORECAST_REQUEST )

    Создает запрос на расчет бюджета.

    Формат FORECAST_REQUEST следующий:

        array GeoID — массив GeoID регионов показа объявления - список можно получить с помощью метода L<GetRegions>
        array Phrases - массив фраз( string ) для расчета бюджета

    Все поля кроме Phrases являются необязательными.
    Все фразы должны удовлетворять требованиям к ключевым словам (http://direct.yandex.ru/help/?id=990408)
    Запрещено использование в GeoID буквенных символов.
    Если требуется явно указать, что прогноз должен быть расчитан для всех регионов - необходимо в качестве номера региона указать - `0`
    (по умолчанию GeoID = 0)

    Функция возвращает ID номера заявки, либо сообщение об ошибке посредством SOAP::Fault.

    Возможно одновременное размещение не более 5 ( пяти ) заявок на расчет бюджета.
    Время доступа к расчету ограничено и составляет 5 часов.

=cut

sub CreateNewForecast
{
    my ($self, $params) = @_;

    my $options = parse_forecast_options($self, $params);

    my $result = create_queue_report_request( 'forecast', $self->{uid}, $options );

    if (ref($result) eq 'ARRAY') {
        # если функция вернула ошибку
        dieSOAP(@$result);
    } else {
        return ($result + 0);
    }
}

=head2 GetForecast

    функция проверяет готов ли отчет, и возвращает его, если он готов

=cut

sub GetForecast
{
    my ($self, $forecast_id) = @_;

    my $report = $self->{_preprocessed}->{report};
    my $forecast_report = get_queue_report_file('forecast', $report->job_id, $self->{uid}, $report->result->{filename}) || dieSOAP("ReportTmpUnavailable");

    if ($self->{api_version_full} > 4) {
        return filter_forecast_report($self, $forecast_report, auction_bids => $report->args->{auction_bids} ? 1 : 0);
    }
    return filter_forecast_report($self, $forecast_report);
}

=head2 GetForecastList()

    Возвращает список отчетов прогноза бюджета и статус готовности

=cut

sub GetForecastList
{
    my $self = shift;
    my $reports = get_queue_reports_list( "forecast", $self->{uid} );

    my $list = filter_reports_list( "forecast", $reports );

    return $list;
}

=head2 GetVersion

    Получить текущую версию API, в которую делает запрос

=cut

sub GetVersion
{
    my $self = shift;

    # TODO: что делать с версией latest ? в wsdl прописан тип int
    return $self->{api_version};
}

=head2 PingAPI_X

    Проверка работы API

    Требуется передать на вход следующую структуру:
    $VAR = [
        {
            'IntValue' => 1,
            'DateValue' => '2008-01-01',
            'StrValue' => 'Первый элемент',
            'IntArray' => [1, 2, 3, 4],
            'StrArray' => ['Первый', 'Второй', 'Третий'],
            'StructArray' => [
                    {'Field1' => 'Значение1', 'Field2' => 1},
                    {'Field1' => 'Значение2', 'Field2' => 2}
                ],
        },
        {
            'IntValue' => 2,
            'DateValue' => '2008-12-01',
            'StrValue' => 'Второй элемент',
            'IntArray' => [5, 6, 7, 8],
            'StrArray' => ['Пятый', 'Шестой', 'Седьмой', 'Восьмой'],
            'StructArray' => [
                    {'Field1' => 'Значение3', 'Field2' => 3},
                    {'Field1' => 'Значение4', 'Field2' => 4},
                    {'Field1' => 'Значение5', 'Field2' => 5},
                ],
        }
    ]

    В случае успеха вернет ее же. В случае ошибки - ошибку SOAP::Fault 

=cut

sub PingAPI_X
{
    shift;
    my $input = shift;
### sleep server handle start
unless (is_sandbox() || is_production()) {
    if ((ref $input eq 'HASH') && exists $input->{sleep} && defined $input->{sleep} && (ref $input->{sleep} eq 'HASH')) {

        if (defined $input->{sleep}->{clients} && (ref $input->{sleep}->{clients} eq 'ARRAY' && scalar @{$input->{sleep}->{clients}})) {
            foreach my $client(@{$input->{sleep}->{clients}}) {
                if ((ref $client eq 'HASH')
                    && (defined $client->{client_id})
                    && ($client->{client_id} =~ m/^\d+$/a)
                    && (defined $client->{max_time})
                    && $client->{max_time} =~ m/^\d+$/a
                    && (defined $client->{min_time})
                    && $client->{min_time} =~ m/^\d+$/a
                    && (defined $client->{start_time})
                    && $client->{start_time} =~ m/^\d+$/a
                    && (defined $client->{end_time})
                    && $client->{end_time} =~ m/^\d+$/a
                    && (defined $client->{probability})
                    && $client->{probability} =~ /^\d{1,4}$/a
                    && $client->{probability} <= 1000) {
                    set_special_user_option($client->{client_id}, 'sleep_max_time_calls_limit', $client->{max_time});
                    set_special_user_option($client->{client_id}, 'sleep_min_time_calls_limit', $client->{min_time});
                    set_special_user_option($client->{client_id}, 'sleep_start_time_calls_limit', $client->{start_time});
                    set_special_user_option($client->{client_id}, 'sleep_end_time_calls_limit', $client->{end_time});
                    set_special_user_option($client->{client_id}, 'sleep_probability_calls_limit', $client->{probability});
                } elsif ((defined $client->{client_id})
                    && ($client->{client_id} =~ m/^\d+$/a)
                    && (exists $client->{time})
                    && (!defined $client->{time})) {
                    set_special_user_option($client->{client_id}, 'sleep_max_time_calls_limit', '');
                    set_special_user_option($client->{client_id}, 'sleep_start_time_calls_limit', '');
                    set_special_user_option($client->{client_id}, 'sleep_end_time_calls_limit', '');
                    set_special_user_option($client->{client_id}, 'sleep_probability_calls_limit', '');
                    set_special_user_option($client->{client_id}, 'sleep_min_time_calls_limit', '');
                }
            }
        }

        return 0;
    }
}
### sleep server handle end
    my $original = [
        {
            'IntValue' => 1,
            'DateValue' => '2008-01-01',
            'StrValue' => 'Первый элемент',
            'IntArray' => [1, 2, 3, 4],
            'StrArray' => ['Первый', 'Второй', 'Третий'],
            'StructArray' => [
                    {'Field1' => 'Значение1', 'Field2' => 1},
                    {'Field1' => 'Значение2', 'Field2' => 2}
                ],
        },
        {
            'IntValue' => 2,
            'DateValue' => '2008-12-01',
            'StrValue' => 'Второй элемент',
            'IntArray' => [5, 6, 7, 8],
            'StrArray' => ['Пятый', 'Шестой', 'Седьмой', 'Восьмой'],
            'StructArray' => [
                    {'Field1' => 'Значение3', 'Field2' => 3},
                    {'Field1' => 'Значение4', 'Field2' => 4},
                    {'Field1' => 'Значение5', 'Field2' => 5},
                ],
        }
    ];
        
    # таким извращённым способом мы сравниваем эквивалентность структур рекурсивно    
    local $YAML::Syck::ImplicitUnicode = 1;
    local $YAML::Syck::SortKeys = 1;    
    
    my $input_dump = Dump($input);
    my $original_dump = Dump($original);

    if ($input_dump ne $original_dump) {
        use Text::Diff;
        print STDERR diff(\$input_dump, \$original_dump, {OUTPUT => \*STDERR});
        dieSOAP('BadRequest', iget("Ошибка при вызове PingAPI_X, выходные данные не соответствуют эталону.\nЭталон:\n %s Входные данные:\n %s", Dumper($original), Dumper($input)));
    }
    return $original;
}

=head2 PingAPI

    Тестовый метод для проверки наличия доступа к API Директа.
    В случае успешного запроса - возвращает 1.

=cut

sub PingAPI
{
    return 1;
}

#-----------------------------------------------------------

=head2 ModerateBanners

    послать на модерацию список баннеров кампании

=cut

sub ModerateBanners
{
    my ($self, $params) = @_;
    my $cid   = $params->{CampaignID};
    my $bids  = $params->{BannerIDS};
    my $camps = [$cid || ()];
    my @client_ids;

    if ($bids) {
        my $count_no_draft = get_one_field_sql(PPC(bid => $bids),
                            ["select 1 from banners where statusModerate != 'New' and ", {bid => SHARD_IDS}, 'LIMIT 1']);

        dieSOAP('NotAllowedOp', iget("Отправлять на модерацию возможно только объявления-черновики")) if $count_no_draft;

        my $ids = get_all_sql(PPC(bid => $bids), ["
            select distinct c.uid, b.cid, u.ClientID
            from banners b
            join campaigns c ON b.cid = c.cid
            join users u ON u.uid = c.uid
         ", where => {bid => SHARD_IDS}]);
        @client_ids = uniq map { $_->{ClientID} } @$ids;

        $camps = [uniq map {$_->{cid}} @$ids];

        push @{$self->{cluid}}, uniq map {$_->{uid}} @$ids;
    } else {
        # если объявления не указаны - то берем все объявления-черновики из кампании
        my $ids = get_all_sql(PPC(cid => $cid), ["
            select c.uid, b.bid, u.ClientID
            from banners b
            join campaigns c ON b.cid = c.cid
            join users u ON c.uid = u.uid
         ", where => { 'b.cid' => SHARD_IDS, 'b.statusModerate' => 'New' }]) || [];
        @client_ids = uniq map { $_->{ClientID} } @$ids;
        $bids = [map {$_->{bid}} @$ids];
        unless (@$bids) {
            # выходим, если нет объявлений-черновиков
            return 0;
        }

        push @{$self->{cluid}}, uniq map {$_->{uid}} @$ids;
    }

    my $error_code = check_rbac_rights($self, 'api_editCamps', {cid => $camps, UID => $self->{uid}});
        dieSOAP('NoRights') if $error_code;

    dieSOAP('NoRights', APICommon::msg_converting_in_progress) if Client::is_any_client_converting_soon(\@client_ids);
    dieSOAP('NoRights', APICommon::msg_must_convert) if Client::is_any_client_must_convert(\@client_ids);

    my $camps_arch_status = are_campaigns_archived($camps);
    my @arched_camps = grep {$camps_arch_status->{$_}} keys %$camps_arch_status;
    if (scalar @arched_camps) {
        dieSOAP('ArchiveEdit', iget("Кампании %s заархивированы", join ", ", @arched_camps))
    }

    my $is_moderate_accept = undef;
    my $post_moderate_allow = $self->{rbac_login_rights}->{manager_control}; # for managers only
    my $send_banners_to_moderate_result = send_banners_to_moderate($bids, {moderate_accept => $is_moderate_accept, post_moderate => $post_moderate_allow});

    $self->{syslog_data}->{cid} = $camps;

    if ($send_banners_to_moderate_result) {
        return 1;
    } else {
        dieSOAP('BadBanner');
    }
}

#-----------------------------------------------------------

=head2 GetAvailableVersions()

    Каким образом будем обеспечивать версионность внутри кода можно прочитать в wiki: 
        http://wiki.yandex-team.ru/Direkt/API/Versioning
    
    Возвращает список доступных версий API.
    Метод предлагается использовать для уведомления о появлении свежих версий API, 
        и своевременных уведомлениях об истечении срока жизни предыдущих доступных версий.
    
    Список берется из переменной Settings::AVAILABLE_API_VERSIONS
    
    Не имеет входных параметров.
    Результат имеет тип - ArrayOfVersionDesc - массив структур VersionDesc, состоящих из
        VersionNumber - номер версии,
        BorderDate - время жизни указанной версии
        
    Для самой свежей версии - BorderDate - не определяется.
    
    Как следует использовать метод:
        1. вызывать раз в день
        2. настроить уведомления(например, на свой email), при появлении новой версии с указанием даты полного прекращения работы прежней версии

    Что метод не решает:
        1. если прежняя версия СТАЛА не ПОЛНОСТЬЮ совместима с новой - как предупредить ?
                рассылкой по email или дополнительным параметром-ссылкой на документацию ?
                Например, VersionDescribeLink (ссылка на докуменацию с описанием изменений)

=cut

sub GetAvailableVersions
{
    my $versions = [];
    
    foreach my $ver (sort keys %Settings::AVAILABLE_API_VERSIONS) {
        unless ($Settings::AVAILABLE_API_VERSIONS{$ver}{hidden} 
                    || $Settings::AVAILABLE_API_VERSIONS{$ver}{blocked}
                    || $ver !~ /^\d+$/ ) {
            push @$versions, {
                VersionNumber => $ver,
                BorderDate => $Settings::AVAILABLE_API_VERSIONS{$ver}{border_date} || undef,
                LinkWSDL => "http://".$Settings::API_WSDL_PATH."/v$ver/wsdl/",
            };
        }
    }

    return $versions;
}

=head2 GetKeywordsSuggestion

    Входные параметры:
      - массив ключевым фраз
    
    Выходные параметры:
      - массив подсказок по переданному набору ключевых фраз

    Получить подсказку по ключевым фразам

=cut

sub GetKeywordsSuggestion
{
    my ($self, $params) = @_;

    my $BEST_PHRASES_CNT = 20;

    $self->{uhost}->reserve_units_by_type($self->{uid}, 'get_keywords_suggestion', scalar @{$params->{Keywords}});
    dieSOAP('NotEnoughUnits') if ! $self->{uhost}->have_enough_units(allow_monitor_fault => 1);

    my $keywords_str = join(',', grep {$_} @{$params->{Keywords}});
    $keywords_str =~ s/\[|\]//g;
    process_phrase_brackets($keywords_str);
    my $keywords = [grep {$_} split(/[ ]*[,\n][ ]*/,$keywords_str)];

    my $options = { 
        count => $BEST_PHRASES_CNT,
    };
    my $result = $keywords ? Suggestions::get_bm_suggestion($keywords, $options)->{phrases} : [];
    return $result;
}

=head2 CreateNewWordstatReport

    Заказ AdvQ отчета

    Входные параметры:
        Phrases - массив фраз
        GeoID - массив регионов, для которых считаем статистику

    Выходные параметры:
        ReportID - идентификатор отчета (заказа)

=cut

sub CreateNewWordstatReport
{
    my ($self, $params) = @_;

    $self->{uhost}->reserve_units_by_type($self->{uid}, 'advq_phrase', scalar @{$params->{Phrases}});
    dieSOAP('NotEnoughUnits') if ! $self->{uhost}->have_enough_units(allow_monitor_fault => 1);

    # проверяем наш технологический лимит
    api_check_limit(
        $self, $self->{uid}, 'CreateNewWordstatReport__day_phrases', get_user_advq_queries_limit($self->{uid}),
        scalar @{ $params->{Phrases} }, obj_name => $self->{operator_login}
    );

    my $result = create_queue_report_request( 'wordstat', $self->{uid}, $params );
    
    if (ref($result) eq 'ARRAY') {
        dieSOAP(@$result);
    } else {
        api_update_limit( $self, $self->{uid}, 'CreateNewWordstatReport__day_phrases', scalar @{$params->{Phrases}});
        return $result;
    }
}

=head2 GetWordstatReportList

    Получить AdvQ отчеты

    Выходные параметры:
        ReportsStatus - Массив состояний отчетов в виде:
                ReportID => 456,
                Status => 'Pending'

=cut

sub GetWordstatReportList
{
    my $self = shift;

    my $reports = get_queue_reports_list('wordstat', $self->{uid});

    my $list = filter_reports_list("wordstat", $reports);

    return $list;
}

=head2 GetWordstatReport

    Получить AdvQ отчеты

    Входные параметры:
        ReportID - идентификатор отчета

    Выходные параметры:
        SearchedWith - Массив фраз "что искали с данным словом" в виде:
                    Phrase - фраза 
                    Shows - число показов
        SearchedAlso - Массив фраз "что ещё искали люди" в виде:
                    Phrase - фраза 
                    Shows - число показов

=cut

sub GetWordstatReport
{
    my ($self, $report_id) = @_;

    my $report = get_queue_report_db('wordstat', $report_id, $self->{uid});

    unless ($report) {
        dieSOAP('NoWordstatReport');
    }

    if ($report->status ne 'Finished') {
        dieSOAP('WordstatReportIsPending');
    }

    unless (defined $report->result->{filename}) { # не должно быть готовых задач без имени файла
        dieSOAP('InternalLogicError', iget('Невозможно получить отчёт.'));
    }

    # get file from storage
    my $wordstat_report = get_queue_report_file('wordstat', $report_id, $self->{uid}, $report->result->{filename}) || dieSOAP("ReportTmpUnavailable");
    return $wordstat_report;
}

=head2 DeleteWordstatReport

    Удаляет указанный отчет advq

    Входные параметры:
        wordstat_id

    Выходные параметры:
        1 - в случае успеха

=cut

sub DeleteWordstatReport
{
    my ($self, $wordstat_id) = @_;

    my $job = get_queue_report_db("wordstat", $wordstat_id, $self->{uid});

    if ($job) {
        return $job->mark_revoked;
    }

    return 0;
}

=head2 GetStatGoals

    Возвращает список целей кампании.
    NB! Возвращаются не все типы целей, а только перечисленные в allowed_goal_types!

    Входные параметры:
        cid
    
    Выходные параметры:
        массив целей

=cut

sub GetStatGoals
{
    my ($self, $params) = @_;
    my ($campaigns_goals_list, %O, $client_id_by_cid, $cids_by_client_id, %accessible_goal_ids_by_client_id, %accessible_goal_ids_by_cid);

    my $allowed_goal_types = ['', qw/url number step action offline call ecommerce form email phone cdp_order_in_progress cdp_order_paid messenger file search button e_cart e_purchase a_cart a_purchase conditional_call social payment_system contact_data/];
    if (defined $params->{CampaignIDS} && $self->{api_version_full} > 4)
    {
        push @{$self->{cluid}}, @{get_uids(cid => $params->{CampaignIDS})};

        $client_id_by_cid = get_key2clientid(cid => $params->{CampaignIDS});
        $campaigns_goals_list = CampaignTools::get_campaigns_goals($params->{CampaignIDS}, get_goal_types => $allowed_goal_types, include_multi_goals => 1);
        $O{include_campaignIDS} = 1;
        $self->{syslog_data}->{cid} = $params->{CampaignIDS};
    }
    else
    {
        $campaigns_goals_list = CampaignTools::get_campaigns_goals([$params->{CampaignID}], get_goal_types => $allowed_goal_types, include_multi_goals => 1);
        push @{$self->{cluid}}, get_owner(cid => $params->{CampaignID});
        $client_id_by_cid = get_key2clientid(cid => $params->{CampaignID});
        $self->{syslog_data}->{cid} = [$params->{CampaignID}];
    }

    foreach my $cid (keys %$client_id_by_cid) {
        push @{$cids_by_client_id->{$client_id_by_cid->{$cid}}}, $cid;
    }

    foreach my $client_id (keys %$cids_by_client_id) {
        if (Client::ClientFeatures::has_goals_only_with_campaign_counters_used($client_id)) {
            my $client_goal_ids = [ map { keys %{$campaigns_goals_list->{campaigns_goals}{$_}} } @{$cids_by_client_id->{$client_id}} ];

            my $extra_counter_ids;
            if (Client::ClientFeatures::has_goals_from_all_orgs_allowed($client_id)) {
                $extra_counter_ids = MetrikaCounters::get_spav_counter_ids_from_db_by_cids($cids_by_client_id->{$client_id});
            }

            $accessible_goal_ids_by_client_id{$client_id} = { map { $_ => 1 }
                @{MetrikaCounters::filter_inaccessible_goal_ids($client_id, $client_goal_ids, extra_counter_ids => $extra_counter_ids)} };
        }
    }

    foreach my $cid (keys %$client_id_by_cid) {
        $accessible_goal_ids_by_cid{$cid} = $accessible_goal_ids_by_client_id{$client_id_by_cid->{$cid}};
    }

    return filter_stat_goals($self, $campaigns_goals_list->{campaigns_goals}, \%accessible_goal_ids_by_cid, %O);
}


=head2 GetSummaryStat

    Сводная статистика по списку кампаний по датам

=cut

sub GetSummaryStat
{
    my ($self, $params) = @_;
    # проверяем наш технологический лимит
    foreach my $cid (@{$params->{CampaignIDS}}) {
        api_check_limit_by_method($self, $cid, 'GetSummaryStat__daily', 1);
    }

    $params->{CampaignIDS} = [uniq @{$params->{CampaignIDS}}];

    # Получаем OrderID
    my $orderid2cid = get_orderid2cid(cid => $params->{CampaignIDS});

    $self->{cluid} = get_uids(cid => $params->{CampaignIDS});

    my $is_beta = is_beta();

    my $camp_currency = get_one_field_sql(PPC(cid => $params->{CampaignIDS}),
            ['select DISTINCT( IFNULL(currency, "YND_FIXED") ) from campaigns', where => {cid => SHARD_IDS}]);

    my $options = {
        UID => $self->{uid},
        cid => join(', ', @{$params->{CampaignIDS}}),
        login_rights => $self->{rbac_login_rights},
        date_from => $params->{StartDate},
        date_to => $params->{EndDate},
        group => 'day',
        plot_group => 'day',
        stat_type => 'campdate',
        target_0 => 1,
        target_1 => 1,
        is_beta => $is_beta,
        is_direct => 1,
        uid => $self->{cluid},
        currency => (defined $params->{Currency})? $params->{Currency} : 'YND_FIXED',
        work_currency => (defined $params->{Currency})? $params->{Currency} : 'YND_FIXED',
        with_nds => ($params->{Currency} && defined $params->{IncludeVAT} && $params->{IncludeVAT} eq 'No')? 0 : 1,
        with_discount => (!$params->{Currency} || (defined $params->{IncludeDiscount} && $params->{IncludeDiscount} eq 'No'))? 0 : 1,
        no_nds_and_discount_fix => ($camp_currency eq 'YND_FIXED') ? 0 : 1, # не подменять значение флагов учета скидки и НДС
        single_currency => (defined $camp_currency && !defined $params->{Currency})? 1 : undef,
        consumer => 'api4',
    };
    if ($self->{rbac_login_rights}->{role} eq 'manager' && $params->{Geo}) {
        $options->{is_geo} = 1;
        delete $options->{is_direct};
    }

    $options->{no_need_days_num} = 1;

    my $stat = CampStat::get_camp_stat($self->{rbac}, $options);

    my @result = ();

    foreach my $order (@{$stat->{orders}}) {
        @{$order->{dates}} = sort {$a->{stat_date} cmp $b->{stat_date}} @{$order->{dates}};
        foreach my $stat_item (@{$order->{dates}}) {
            push @result, {
                StatDate => $stat_item->{stat_date},
                CampaignID => $orderid2cid->{$order->{OrderID}},
                SumSearch => round2s($stat_item->{sum_0}),
                SumContext => round2s($stat_item->{sum_1}),
                ShowsSearch => $stat_item->{shows_0},
                ShowsContext => $stat_item->{shows_1},
                ClicksSearch => $stat_item->{clicks_0},
                ClicksContext => $stat_item->{clicks_1},
                SessionDepthSearch => $stat_item->{adepth_0},
                SessionDepthContext => $stat_item->{adepth_1},
                GoalConversionSearch => $stat_item->{aconv_0},
                GoalConversionContext => $stat_item->{aconv_1},
                GoalCostSearch => $stat_item->{agoalcost_0},
                GoalCostContext => $stat_item->{agoalcost_1},
            };
        }
    }
    # апдейтим  наш технологический лимит
    foreach my $cid (@{$params->{CampaignIDS}}) {
        api_update_limit($self, $cid, 'GetSummaryStat__daily', 1);
    }

    $self->{syslog_data}->{cid} = $params->{CampaignIDS};

    return \@result;
}

=head2 GetWordstatSync(self, params)

    Синхронный вариант CreateNewWordstatReport

=cut

sub GetWordstatSync($$) {
    my ($self, $params) = @_;
    return sync_wordstat($params->{Phrases}, $params->{GeoID});
}

=head2 GetForecastSync($$)

    Синхронный вариант CreateNewForecast для медиапланнеров

=cut

sub GetForecastSync($$) {
    my ($self, $params) = @_;

    my %forecast_options;
    if (defined $params->{PeriodType} && $params->{PeriodType} eq "Month") {
        %forecast_options = (period => {month => $params->{PeriodValue}});
    } elsif (defined $params->{PeriodType} && $params->{PeriodType} eq "Quarter") {
        %forecast_options = (period => {quarter => $params->{PeriodValue}});
    } elsif (defined $params->{PeriodType} && $params->{PeriodType} eq "Year") {
        %forecast_options = (period => {year => 1});
    }

    my $options = parse_forecast_options($self, $params);

    my $results = get_forecast($options, %forecast_options);

    if (exists $results->{error}) {
        my $error_info = $results->{error};
        my $err_code = $error_info->{error} eq 'INVALID_QUERY' ? 'BadParams' : '500';
        dieSOAP($err_code);
    }

    return filter_forecast_report($self, $results, advanced_forecast => 1,
        auction_bids => $options->{auction_bids});
}

=head2 GetNormalizedKeywordsData(self, phrase)

    Принимает массив фраз, и возвращает данные нормализации,
    а именно для каждого слова в фразе: список форм, список лемм,
    а так же само слово в той форме в которой пришло

=cut

sub GetNormalizedKeywordsData($$) {
    my ($self, $params) = @_;
    my @result;
    for my $phrase (@{$params->{Keywords}}) {
        my @word_list;
        for my $word (grep {!/^-/} split /\s+/, $phrase) {
            my ($lemmas, $formas) = Yandex::MyGoodWords::MyGetGoodWords($word, 1);
            push @word_list, {Word => $word, Lemmas => $lemmas, Formas => $formas};
        }
        push @result, \@word_list;
    }
    return \@result;
}

=head2 GetNormalizedKeywords(self, phrase)

    Нормализация (и сортировка) массива строк.

=cut

sub GetNormalizedKeywords($$) {
    my ($self, $params) = @_;
    my @result;
    foreach my $phrase (@{$params->{Keywords}}) {
        my $normalized = Yandex::MyGoodWords::norm_words($phrase);
        push @result, "$normalized";
    }

    return \@result;
}

1;
