package APIUnits;

=pod

Модуль для ограничения вызова функций по пользовательским баллам.

Авторы: 
Лохмачев Сергей sergeysl@yandex-team.ru
Журавлев Сергей zhur@yandex-team.ru

=cut

use warnings;
use strict;

require Exporter;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;

use Settings;
use PrimitivesIds;
use LogTools;

our @ISA = qw(Exporter);

our @EXPORT = qw(
  have_enough_units
  have_enough_user_units
  reset_reserved_units
  reserve_units
  reserve_units_by_type
  get_reserved_units
  update_units
  get_units
  get_units_stats
  check_or_init_user_units
  get_user_units_stats
  log_reserved_units
);

=pod
our %unit_prices = (
            edit_banner => 5, 
            create_banner => 10, 
            edit_phrase => 1, 
            create_phrase => 1, 
    );
=cut

our %SCHEME_SETTINGS = (
    "API"   => {
        unit_prices => {
            edit_banner       => 4,
            create_banner     => 12,
            edit_phrase       => 1,
            create_phrase     => 2,
            advq_phrase       => 10,
            get_keywords_suggestion => 3,
            load_image        => 2,
            image_set_remove  => 1,
            image_set_update  => 2,
        },
        DEF_UNITS_COUNT => 32000,
    },

    "XLS"   => {
        unit_prices => {
            edit_banner     => 4,
            create_banner   => 12,
            remove_banner   => 0,
            edit_phrase     => 1,
            create_phrase   => 2,
            remove_phrase   => 0,
        },
        DEF_UNITS_COUNT => 64000,
    },
);

# Создаём объект контроллера бальной системы
#
#   Пример использования:
#   my $uhost = new APIUnits({ scheme="API" });
#
#   scheme : (API|XLS);
#
sub new($$){
    my ($class, $data) = @_;
    
    my $self = {
        is_internal_user => 0,
        client_units    => {},
        units_stats     => {},
        reserved_units  => {}
    };
    hash_merge $self, $data;

    die "scheme option is missed or of wrong type in APIUnits object init" unless defined $self->{scheme} and exists $SCHEME_SETTINGS{$self->{scheme}};
    
    bless $self, $class;
    return $self;
}


sub get_default_units($$){
    my ($scheme, $uid) = @_;

    return $SCHEME_SETTINGS{$scheme}{DEF_UNITS_COUNT};

}

sub have_enough_units ($;%){
    my ($self, %O) = @_;

    for (keys %{$self->{reserved_units}}) {
        return 0 if !$self->have_enough_user_units($_, allow_monitor_fault => $O{allow_monitor_fault});
    }
    return 1;
}

=head2 have_enough_user_units($uid, %O)

 проверяет есть ли указанное кол-во баллов у пользователя

=cut

sub have_enough_user_units ($$;%){
    my ($self, $uid, %O) = @_;
    # специальным пользователям и менеджерам продаж снимаем балльные ограничения
    return 1 if $Settings::UIDS_NO_LIMITS_API{$uid} || $self->{is_internal_user};

    if (!exists $self->{client_units}{$uid}) {
        my $user_units = $self->check_or_init_user_units($uid);

        $self->{client_units}{$uid} = $user_units->{$uid}{units_rest};
        return 1 if ($user_units->{$uid}{data_unavailable} && $O{allow_monitor_fault});
    }
    
    $self->{reserved_units}{$uid} = 0 if ! exists $self->{reserved_units}{$uid};

    return $self->{client_units}{$uid} >= $self->{reserved_units}{$uid} ? 1 : 0;
}

=head2 update_user_daily_units($uid,$units)

 обновляет пользовательскую дневную карму

=cut

sub update_user_daily_units($$$) {
    my ($self, $uid, $units) = @_;
    if($uid && defined $units) {
        do_sql(PPC(uid => $uid), "insert into api_users_units_consumption (daily_units, uid, scheme, units) 
                                             values (?, ?, ?, 0)
                                      on duplicate key update daily_units = ?, units=values(units)", 
                                      $units, $uid, $self->{scheme}, $units);
        return 1;
    }
    return 0;
}

# обнуляет хэш с данными по зарезервированным баллам
sub reset_reserved_units($){
    my $self = shift;
    hash_merge $self, { 
        client_units    => {},
        units_stats     => {},
        reserved_units  => {}
    };
}

# резервирует баллы для указанного пользователя
#
#   $uhost->reserve_units_by_type($uid, 'edit_banner')  - отредактировать баннер
#   $uhost->reserve_units_by_type($uid, 'create_phrase', 12) - добавить 12 фраз
#
sub reserve_units_by_type($$$;$){    
    my $self = shift;
    my $uid = shift;
    my $type = shift;
    my $count = scalar(@_) ? shift : 1;

    $self->{reserved_units}{$uid} = 0 if ! defined $self->{reserved_units}{$uid};
    $self->{units_stats}{$uid}->{$type} = 0 if ! defined $self->{units_stats}{$uid}->{$type};
    
    $self->{units_stats}{$uid}->{$type}+= $count;
    $self->{reserved_units}{$uid} += $count * $SCHEME_SETTINGS{$self->{scheme}}{unit_prices}{$type};
}

sub unreserve_units_by_type($$$;$){    
    my ($self, $uid, $type, $count) = @_;
    $self->reserve_units_by_type($uid, $type, -$count);
}

# резервируем баллы без указания операции
#
#
sub reserve_units($$$) {
    my $self = shift;
    my $uid = shift;
    my $count = shift;

    ($self->{reserved_units}{$uid}||=0) += $count;
}

# возвращает кол-во зарезервированных баллов для указанного пользователя
sub get_reserved_units($$){
    my $self = shift;
    my $uid = shift;
    return $self->{reserved_units}{$uid} || 0;
}


# запись в базу данных всех зарезервированных баллов
sub update_units($){
    my $self = shift;
    for my $uid (keys %{$self->{reserved_units}}){
        do_sql(PPC(uid => $uid), "
            update api_users_units_consumption
               set units=(units + ?)
             where uid = ?
               and scheme = ?
                ", $self->{reserved_units}{$uid},
                   $uid,
                   $self->{scheme}
            );
    }
}

# 4 функции get(_user)?_units(_stats)? называются похоже, а делают разное.
# у нас так паранойя разовьётся...
# UPDATE: удалось от одной избавиться ;)

# возвращает список хэшей с данными по зарезервированным баллам
sub get_units($){
    my $self = shift;
    return [map {
        {
            Login => get_login(uid => $_), 
            UnitsRest => (defined $self->{client_units}{$_}) ? (($self->{client_units}{$_} || 0) - ($self->{reserved_units}{$_} || 0)) : undef, 
            UnitsCallCost => $self->{reserved_units}{$_}
        }
    } keys %{$self->{reserved_units}}];    
}

# возвращает информацию о выполненых операциях и их кол-во
sub get_units_stats($){
    my $self = shift;
    return [map {
        {
            uid => $_,
            Login => get_login(uid => $_),             
            UnitsStats => $self->{units_stats}{$_},            
        }
    } keys %{$self->{units_stats}}];         
}

=head2 check_or_init_user_units(@uids)

 1) проверяет запись в базе о количестве юнитов для пользователей,
 2) если записи нет, добавляет дефолтное значение
 3) если запись есть, но вчерашняя, обновляет количество юнитов из daily_units

 на вход получает массив из uids, на выходе хэш в формате:

    {
        uid => {
                  units => ...,
                  daily_units => ...,
                  units_rest => ...
               }
    }

=cut

sub check_or_init_user_units($@) {
    my $self = shift;
    my @uids =  @_;       
    my %result;

    if (scalar @uids) {
        my $client_units = get_hashes_hash_sql(PPC(uid => \@uids), ['
                    SELECT uid
                         , units
                         , daily_units
                         , unix_timestamp(date) AS lastdate
                         , unix_timestamp(DATE(NOW())) AS today
                      FROM api_users_units_consumption
              ', where => {
                    uid     => SHARD_IDS,
                    scheme  => $self->{scheme},
              },
            ]);
        foreach my $uid (@uids) {
            # Если в базе нет записи по этому юзеру
            if (! defined $client_units->{$uid}{daily_units}){
                $client_units->{$uid}{daily_units} = get_default_units($self->{scheme}, $uid);
                $client_units->{$uid}{units} = 0;
                $self->update_user_daily_units($uid, $client_units->{$uid}{daily_units});
            # Если дата вчерашняя - сбрасываем units, обновляем дату
            } elsif ( $client_units->{$uid}{lastdate} < $client_units->{$uid}{today} ) {
                do_sql(PPC(uid => $uid), "update api_users_units_consumption set date=DATE(NOW()), units = 0 where uid = ? and scheme = ?",
                                    $uid, $self->{scheme});
                $client_units->{$uid}{units} = 0;
            }
            $result{$uid} = {units => $client_units->{$uid}{units}, 
                             daily_units => $client_units->{$uid}{daily_units},
                             units_rest => $client_units->{$uid}{daily_units} - $client_units->{$uid}{units}};
            if ($result{$uid}{units_rest} < 0) {   
                $result{$uid}{units_rest} = 0;
            }
        }
        

    }
    return \%result;
}

=head2 get_user_units_stats($uid[, $type])

 Возвращает информацию о выполненых операциях и их кол-во в виде ссылки на хеш {<тип выполненной операции> => <количество операций>}
 Если указан тип операции $type, возвращает количество выполненных операций типа $type.

=cut

sub get_user_units_stats($$;$){
    my $self = shift;
    my $uid = shift;
    my $type = (@_) ? shift : undef;
    if (defined $type) {
        my $stats = $self->{units_stats}{$uid};
        return ($stats && ref($stats) eq 'HASH' && $stats->{$type}) ? $stats->{$type} : 0;
    } else {
        return $self->{units_stats}{$uid};
    }
}

sub log_reserved_units($$;$) {
    my ($self, $uid, $vars)  = @_;
   
    die "No cid specified" unless $vars->{cid};

    LogTools::log_xls_user_units([{
            scheme => $self->{scheme},
            uid => $uid,
            (map {$_ => $vars->{$_}||0} qw/cid ManagerUID AgencyUID/),
            units => $self->get_reserved_units($uid),
            (map {$_ => $self->get_user_units_stats($uid, $_)} qw/create_banner edit_banner create_phrase edit_phrase/),
    }]);
}

1;
