=head1 NAME

AccountScore

=head1 DESCRIPTION

Предоставляет функции и константы для работы с показателями качества фраз.

=cut
package AccountScore;

use warnings;
use strict;
use utf8;

use List::MoreUtils qw/uniq/;
use JSON;

use Settings;
use Yandex::DBShards qw/SHARD_IDS/;
use Yandex::DBTools;
use Yandex::TimeCommon;

use Direct::YT::account_score_yt qw/
    
     calc_factors_index
     calc_score_by_factors

/;

use base qw/Exporter/;
our @EXPORT = qw/


    get_account_score
    get_scores_list

    HISTORY_DAYS
    STOP_DAYS_STAT_LENGTH

/;

our $STOP_DAYS_STAT_LENGTH = $Direct::YT::account_score_yt::STOP_DAYS_STAT_LENGTH;  # используется в ppcCalcAccountScoreFactors.pl
our $HISTORY_DAYS ||= 5;

=head2 get_account_score

    Посчитать рейтинг рекламодателя.
    На входе:
        ClientID - ClientID рекламодателя
    На выходе:
        хеш с полями:
        score - число, рейтинг;
        progress - прогресс изменений: -1 - рейтинг падает, 0 - рейтинг не изменяется, 1 - рейтинг растет;
        verdicts - массив чисел, означающий какие рекомендации необходимо показать пользователю.

=cut
sub get_account_score {


    my $ClientID = shift;

    my $score_data = get_all_sql(PPC(ClientID => $ClientID), "SELECT factors_json, score FROM account_score WHERE ClientID = ? ORDER BY date DESC LIMIT ?", $ClientID, 2) || [];

    foreach my $r (@$score_data) {
        my $f = from_json($r->{factors_json});
        my $indexes = calc_factors_index($f);
        $r->{verdicts} = get_score_verdicts($indexes);
    }

    my $today_score_data = $score_data->[0] || {};

    my $prev_score = scalar(@$score_data) > 1 ? $score_data->[1]->{score} : 0;

    return { score      => $today_score_data->{score},
             progress   => (defined ($today_score_data->{score})) ? ($today_score_data->{score} <=> $prev_score) : undef,
             verdicts   => $today_score_data->{verdicts},
             prev_score => $prev_score
           };
}

=head2 get_score_verdicts

    Возвращает список рекомендаций, которые требуется показать клиенту
    на входе:
        хеш с факторами (после calc_factors_index)
    на выходе:
        массив строк, означающий краткое имя рекомендации.

=cut
sub get_score_verdicts {
    my $factors = shift;
    # Считаем, что фактор сработал, если он больше 0
    return [uniq map {$Direct::YT::account_score_yt::FACTORS_DATA{$_}->{verdict_name}} grep {$factors->{$_} > 0} keys %$factors];
}

=head2 get_scores_list

    Получить показатели качества для списка клиентов/агентств/менеджеров.
    на входе:
        список ClientID
    на выходе
        хэш, содержащий пары вида <ClientID> => <score>

=cut
sub get_scores_list {
    my ($client_ids) = @_;
    my $data = get_hash_sql(PPC(ClientID => $client_ids), [qq/SELECT t1.ClientID, t1.score FROM account_score t1 
                                                               JOIN (SELECT ClientID, MAX(date) as max_date FROM account_score/,
                                                               WHERE => {"ClientID" => SHARD_IDS},
                                                               qq/ GROUP BY ClientID) t2
                                                               ON (t1.ClientID = t2.ClientID AND t1.date = t2.max_date)
                                                               /, WHERE => {"t1.ClientID" => SHARD_IDS}]);
    return { map { $_ => $data->{$_} } @$client_ids };  # тем, для кого ничего не нашлось, выставим undef
}

1;
