package Agency;

# $Id$

=head1 NAME
    
    Agency

=head1 DESCRIPTION

    Работа с сущностью "агентство" (основной идентификатор -- ClientID)

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq firstval/;
use Readonly;

use Settings;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;

use Campaign::Types qw(get_camp_kind_types);
use Client;
use Currencies;
use Primitives;
use RBACElementary;
use RBACDirect;
use Rbac qw/:const/;
use User;
use JSON qw/to_json/;

use base qw/Exporter/;
our @EXPORT = qw/
    agency_total_sums
    get_agency_additional_currencies
    get_agency_allowed_currencies_hash

    mass_get_agency_currency
    get_agency_currency

    set_show_agency_contacts_flag
    get_agency_contacts_by_cid

    get_agency_representatives

    get_lim_rep_group_id
    get_lim_reps_group_ids
    get_lim_rep_group_chief_uid
    get_lim_rep_group_main_uids
    get_lim_rep_group

    mass_has_lim_rep_group_main_reps

    unlink_lim_reps_from_client
    link_lim_reps_to_client
/;

my $AGENCY_UIDS_CHUNK_SIZE = 25;

Readonly our $ERROR_CREATE_AGENCY_RBAC_ERROR => 'ERROR_CREATE_AGENCY_RBAC_ERROR';

=head2 create_agency($rbac, $UID, $is_direct, $params)

    Создает агентство в Директе. Параметры агентства указаны в хеше $params
    К этому моменту агенство уже должно быть создано в Паспорте и в Балансе.


    В $params ожидаются следующие обязательные параметры
        uid             - паспортный uid представителя агентства
        name            - название агентства
        phone
        email
        url
        client_id       - ClientID из Баланса
        country_region_id
        subregion_id
        currency

    необязательные параметры
        agency_status   =~ /^(?:SA|AA|HY|WC|NR)$/
        show_agency_contacts    - настройка на представителе агентства, если 1, то субклиентам можно показывать
                                  контактную информацию агентства

=cut

sub create_agency {
    my ($rbac, $UID, $is_direct, $params) = @_;

    my ($uid, $name, $phone, $email, $url, $client_id, $country_region_id, $subregion_id, $currency) = @$params{
      qw/uid   name   phone   email   url   client_id   country_region_id   subregion_id   currency/
    };
    my ($agency_status, $show_agency_contacts) = @$params{
      qw/agency_status   show_agency_contacts/
    };

    my $user_data = {
        fio => $name,
        phone => $phone,
        email => $email,
        ClientID => $client_id,
        lang => Yandex::I18n::current_lang(),
        initial_country => $country_region_id,
        initial_currency => $currency,
        initial_subregion => $subregion_id,
    };

    # Заносим данные в users
    # также в этот момент создается запись в clients с ролью 'empty'
    create_update_user($uid, $user_data);

    my %new_data = (REGION_ID => $country_region_id);
    if ($currency ne 'YND_FIXED') {
        $new_data{CURRENCY} = $currency;
        $new_data{MIGRATE_TO_CURRENCY} = Client::get_client_migrate_to_currency($client_id);
    }
    my $is_error = update_client_id($UID, $client_id, \%new_data);
    die "Error updating ClientID $client_id in Balance" if $is_error;


    # Прописываем в RBAC информацию о агентстве
    if (rbac_create_agency($rbac, $UID, $uid, $client_id)) {
        die {code => $ERROR_CREATE_AGENCY_RBAC_ERROR};
    }

    my $primary_manager_type = !$is_direct ? 'primary_bayan_manager_uid' : 'primary_manager_uid';
    my $agency_client_data = {
        ClientID => $client_id,
        role => 'agency',
        name => $name,
        agency_url => $url,
        agency_status => $agency_status,
        $primary_manager_type => $UID,
        country_region_id => $country_region_id,
        work_currency => $currency,
        subregion_id => $subregion_id,
    };
    do_in_transaction sub {
        create_update_client({client_data => $agency_client_data});
        my $agency_options = {
            ClientID => $client_id,
            allow_clients_without_wallet => 0,
            default_clients_with_wallet => 1,
        };
        do_insert_into_table(PPC(ClientID => $client_id), 'agency_options', $agency_options, ignore => 1);
        set_show_agency_contacts_flag($uid, $show_agency_contacts ? 1 : 0);
        do_insert_into_table(PPC(ClientID => $client_id), 'clients_to_fetch_nds', {ClientID => $client_id}, ignore => 1);
    };
}

=head2 agency_total_sums

    Считает суммарные и потраченные деньги по всем клиентам агентства в единой валюте (либо указанной, либо превалирующей).

    agency_total_sums($agency_client_id, %O);
    %O => (
        rbac => $rbac,    # обязательный
        type => ['text', 'dynamic', 'performance'] | 'mcb' | 'geo' | ..., # типы кампаний, по которым считать суммы; обязательный
        currency => undef | 'YND_FIXED' | 'RUB' | 'USD' | ..., # валюта, в которой будут рассчитаны итоговые суммы
                                                               # если не указана, суммы будут в превалирующей валюте
    );

=cut

sub agency_total_sums {
    my ($agency_client_id, %O) = @_;

    my $agency_uid = rbac_get_chief_rep_of_agency($agency_client_id) or die "agency chief uid not found";
    my $client_clientids = rbac_get_subclients_clientids($O{rbac}, $agency_uid);

    # определяем, к какой валюте все приводить
    my $target_currency = ($O{currency}) ? $O{currency} : get_agency_currency($O{rbac}, $agency_client_id);

    my $sums = {
        currency => $target_currency, 
        sum => 0, 
        sum_spent => 0, 
        total => 0,
        type => $O{type},
    };

    if ($client_clientids && @$client_clientids) {
        # для каждого клиента
        # берем его суммы в целевой валюте
        # добавляем к сумме
        my $client_sums = mass_client_total_sums(ClientIDs => $client_clientids, currency => $target_currency, type => $O{type}, agency_client_id => $agency_client_id);
        while (my($client_id, $client_sums) = each %$client_sums) {
            die 'invalid currency' if $client_sums->{currency} ne $target_currency;
            for my $f (qw/sum sum_spent total/){
                $sums->{$f} += $client_sums->{$f};
            }
        }
    }

    return $sums;
}

=head2 get_agency_additional_currencies

    Возвращает список дополнительных валют на агенства. Список заполняется нотификациями со стороны Баланса.

    Принимает позиционным параметром ClientID агентства и необязательные именованные параметры:
        include_expired => 0|1 -- если 0 [по умолчанию], то будут возвращены только валюты, на которые сейчас 
                                  действует доп.соглашение. в противном случае будут возвращены ещё и просроченные валюты.  
        currency_codes_only => 0|1 -- если 1, то вернутся только коды доп. валют (в виде ссылки на массив). в противном же 
                                      случае [по умолчанию] вернётся массив валют, представленных ссылками на хеши с ключами:
                                          code -- символьный код валюты (USD/EUR/UAH/...)
                                          expiration_date -- дата, до которой агентство может пользоваться доп. валютой
                                          LastChange -- дата последнего изменения записи о доп.валюте

    my $additional_currency_codes = get_agency_additional_currencies(currency_codes_only => 1);
    $additional_currency_codes => ['USD', 'UAH', 'KZT'];

    my $additional_currencies = get_agency_additional_currencies(include_expired => 1);
    $additional_currency_codes => [
        {
            code => 'USD',
            expiration_date => '1987-10-28',
            LastChange => '2012-02-17 14:15:21',
            expired => 1,
        },
        {
            code => 'UAH',
            expiration_date => '2013-01-01',
            LastChange => '2012-02-18 12:34:56',
            expired => 0,
        },
    ];

=cut

sub get_agency_additional_currencies {
    my ($ClientID, %O) = @_;

    my %cond = (ClientID => $ClientID);
    $cond{'expiration_date__ge__dont_quote'} = 'DATE(NOW())' unless $O{include_expired};

    if ($O{currency_codes_only}) {
        return get_one_column_sql(PPC(ClientID => $ClientID), ['SELECT currency FROM agency_currencies', WHERE => \%cond]) || [];
    } else {
        return get_hashes_hash_sql(PPC(ClientID => $ClientID), 
            ['SELECT currency AS code, expiration_date, (expiration_date < DATE(NOW())) AS expired, LastChange FROM agency_currencies', WHERE => \%cond]) || {};
    }
}

=head2 _get_agency_allowed_pay_currencies

    Возвращает список валют, которыми может платить агентство.
    Использует кешированные списки стран-валют из Баланса, которые периодически обновляются скриптом ppcFetchAgencyCountryCurrency.pl.

    $allowed_pay_currencies = _get_agency_allowed_pay_currencies($agency_client_id);
    $allowed_pay_currencies => ['USD', 'EUR', 'CHF'];

=cut

sub _get_agency_allowed_pay_currencies {
    my ($client_id) = @_;

    my $country_currencies = get_all_sql(PPC(ClientID => $client_id),
        ['SELECT country_region_id, currency FROM client_firm_country_currency', WHERE => {ClientID => SHARD_IDS}]);

    # Все варианты == клиент не платил и может в чём угодны создавать субклиентов. Но после первой оплаты останется только валюта оплаты.
    # Чтобы сам себе не выстрелил в ногу, создав клиента, за которого не может платить — не даём клиентам с таким набором выбирать реальные валюты.
    if ($country_currencies && @$country_currencies && scalar(@$country_currencies) != Currencies::get_any_country_currency_cnt_for_agency()) {
        return [ uniq map { $_->{currency} } @$country_currencies ];
    } else {
        return [];
    }
}

=head2 get_agency_allowed_currencies_hash

    Возвращает список валют, которыми может пользоваться агентство, в виде хеша.
    Принимает ClientID агентства.

    my $allowed_currencies_hash = get_agency_allowed_currencies_hash($ClientID, is_direct => $c->is_direct);
    $allowed_currencies_hash => {
        KZT => 1,
        USD => 1,
        UAH => 1,
    };

=cut

sub get_agency_allowed_currencies_hash {
    my ($ClientID, %O) = @_;

    my $work_currency = get_client_currencies($ClientID)->{work_currency};
    my @currencies;

    if ($work_currency && $O{is_direct}) {
        if ($work_currency eq 'YND_FIXED') {
            my $client_data = get_client_data($ClientID, [qw(country_region_id)]);
            if ($client_data && $client_data->{country_region_id} && $client_data->{country_region_id} == Currencies::REGION_ID_BY()) {
                # DIRECT-56867, для Белорусских фишечных агентств - прибиваем гвоздями BYN
                push @currencies, 'BYN';
            } else {
                # уешные агентства могут создавать субклиентов в доступных им для оплаты валютах
                my $pay_currencies = _get_agency_allowed_pay_currencies($ClientID);
                push @currencies, @$pay_currencies;
            }
        } else {
            push @currencies, $work_currency;
        }
        # агентствам по доп.соглашению могут быть на ограниченный период доступны дополнительные валюты
        my $additional_currencies = get_agency_additional_currencies($ClientID, currency_codes_only => 1);
        if ($additional_currencies && @$additional_currencies) {
            push @currencies, @$additional_currencies;
        }
    } else {
        push @currencies, $work_currency;
    }

    return {map {$_ => 1} @currencies};
}

=head2 mass_get_agency_currencies

    Определяет "валюту агентства" — ту валюту, в которой агентство видит список клиентов.
    Возвращает самую частую валюту среди неархивных клиентов.
    Если клиентов нет — основную валюту самого агентства.

    $agencyclientid2currency = mass_get_agency_currency($rbac, [$agency_clientid1, $agency_clientid2, ...]);
    $agencyclientid2currency => { $agency_clientid1 => $currency1, $agency_clientid2 => $currency2, ... }

=cut

sub mass_get_agency_currency {
    my ($rbac, $agency_clientids) = @_;

    die 'no agency ClientIDs given' unless $agency_clientids && ref($agency_clientids) eq 'ARRAY';

    my %agency_currencies;
    for my $agency_client_id (@$agency_clientids) {
        my $agency_uids = Rbac::get_reps(ClientID => $agency_client_id, role => $ROLE_AGENCY);
        next unless @$agency_uids;
        my $subclients_clientids = rbac_get_subclients_clientids($rbac, $agency_uids, chunk_size => $AGENCY_UIDS_CHUNK_SIZE);

        # смотрим только на неархивных субклиентов агентства
        if ( $subclients_clientids && @$subclients_clientids ) {
            my $relations = get_agency_client_relations($agency_client_id, $subclients_clientids);
            $subclients_clientids = [grep { ! $relations->{$_}->{client_archived} } @$subclients_clientids];
        }

        # отсекаем баяновских и пустых клиентов (должна быть хотя бы одна агентская кампания в директе)
        if ( $subclients_clientids && @$subclients_clientids ) {
            $subclients_clientids = [ uniq @{ get_one_column_sql(PPC(ClientID => $subclients_clientids), [q/
                SELECT DISTINCT u.ClientID
                FROM users u
                INNER JOIN campaigns c ON u.uid = c.uid
            /,  WHERE => {
                    'u.ClientID' => SHARD_IDS,
                    'c.type' => get_camp_kind_types('web_edit_base'),
                    'c.AgencyID' => $agency_client_id,
                    'c.statusEmpty' => 'No',
            }]) || [] } ];
        }

        my $currency;
        if ( $subclients_clientids && @$subclients_clientids ) {
            # выбираем самую частую валюту среди неархивных клиентов агентства
            my $client_currencies = mass_get_client_currencies($subclients_clientids);
            my @currencies = map { $_->{work_currency} } values %$client_currencies;
            $currency = Currencies::get_dominant_currency(\@currencies);
        } else {
            # если клиентов нет, то используем валюту самого агентства
            my $agency_currencies = get_client_currencies($agency_client_id);
            $currency = $agency_currencies->{work_currency};
        }
        $agency_currencies{$agency_client_id} = $currency;
    }

    return \%agency_currencies;
}

=head2 get_agency_currency

    $currency = get_agency_currency($rbac, $agency_clientid);

=cut

sub get_agency_currency {
    my ($rbac, $agency_clientid) = @_;

    return mass_get_agency_currency($rbac, [$agency_clientid])->{$agency_clientid};
}

=head2 set_show_agency_contacts_flag

    set_show_agency_contacts_flag($agency_uid, $flag);

    $flag - is boolean, 1 or 0

=cut

sub set_show_agency_contacts_flag {
    my ($agency_uid, $flag) = @_;

    $flag = $flag ? 1 : 0;
    do_insert_into_table(PPC(uid => $agency_uid), 'users_agency', 
        {
            uid => $agency_uid, 
            show_agency_contacts => $flag
        }, 
        on_duplicate_key_update => 1, key => 'uid'
    );

}

=head2 get_agency_contacts_by_cid

    возвращает контакты агенства для клиента

    get_agency_contacts_by_cid($rbac, $cid);

=cut

sub get_agency_contacts_by_cid {
    my ($rbac, $cid) = @_;

    return undef unless $cid;

    my $agency_uid = get_one_field_sql(PPC(cid => $cid), "SELECT AgencyUID FROM campaigns WHERE cid = ?", $cid);
    return undef unless $agency_uid;
    my $chief_rep_uid = rbac_get_chief_rep_of_agency_rep($agency_uid);

    my $show_agency_contacts = get_hash_sql(PPC(uid => [$agency_uid, $chief_rep_uid]), [
        "SELECT uid, show_agency_contacts FROM users_agency",
        WHERE => {uid => SHARD_IDS},
    ]);

    my $contacts;
    if (my $uid = firstval { $show_agency_contacts->{$_} } ($agency_uid, $chief_rep_uid)) {
        $contacts = get_one_line_sql(PPC(uid => $uid), [
            "SELECT name, fio, phone, email FROM users u JOIN clients c USING(ClientID)",
            WHERE => {uid => SHARD_IDS},
        ]);
        $contacts->{is_chief} = 1 if $uid == $chief_rep_uid;
    }

    return $contacts;
}

=head2 get_agency_representatives

    Возвращает представителей агентства по ClientID агентства

=cut

sub get_agency_representatives {
    my ($agency_clientid) = @_;

    return undef unless $agency_clientid;

    get_hashes_hash_sql(PPC(ClientID => $agency_clientid), [
        'SELECT
            u.uid, u.login, u.ClientID,
            uag.show_agency_contacts, uag.is_no_pay as agency_no_pay, uag.disallow_money_transfer as agency_disallow_money_transfer
        FROM
            users u join users_agency uag using (uid)',
        WHERE => { 'u.ClientID' => $agency_clientid }
    ]);
}

=head2 refresh_clients_of_agency_limited_reps

    Отправляет в баланс клиентов ограниченных представителей агентств,
    чтобы ограниченные представители видели в балансе только своих клиентов
    нужно вызывать при изменении списка клиентов, назначенных ограниченному представителю
    возвращает хеш {uid => success}

=cut

sub refresh_clients_of_agency_limited_reps
{
    my ($rep_uids, $operator_uid, $log, $force_reset) = @_;

    my $limited_reps = _get_limited_agency_reps($rep_uids);
    return _do_refresh_clients($limited_reps, $operator_uid, $log, $force_reset);
}

sub _do_refresh_clients
{
    my ($limited_reps, $operator_uid, $log, $force_reset) = @_;
    return {} unless @$limited_reps;

    my $has_feature = Client::ClientFeatures::_is_feature_allowed_for_client_ids(
        [uniq map { $_->{ClientID} } @$limited_reps], 'balance_limited_agency_rep');

    # находим клиентов ограниченных представителей
    # если фича для агентства выключена, то список клиентов не нужен
    my $client_ids_by_rep_uid = _get_clients_of_agency_reps(
        [map { $_->{uid} } grep { $has_feature->{ $_->{ClientID} } } @$limited_reps]);

    my $result = {};
    foreach my $limited_rep (@$limited_reps) {
        my $limited_rep_uid = $limited_rep->{uid};
        my $agency_client_id = $limited_rep->{ClientID};
        my $client_client_ids;
        $log->out(sprintf('limited_rep_uid: %s, agency_client_id: %s, has_feature: %s', 
            $limited_rep_uid, $agency_client_id, $has_feature->{ $agency_client_id } // '')) if $log;

        # если для агентства включена фича, то ограниченный представитель должен видеть в балансе только своих клиентов
        # для этого передадим в баланс список клиентов этого представителя
        if ($has_feature->{ $agency_client_id }) {

            $client_client_ids = $client_ids_by_rep_uid->{ $limited_rep_uid };
 
            unless ($client_client_ids && @$client_client_ids) { # у представителя нет клиентов
                # чтобы представитель в балансе не видел никаких клиентов
                # необходимо в limited_list передать [null], то есть список с единственным элементом со значением null
                $client_client_ids = $limited_rep->{rep_type} eq $REP_LIMITED ? [undef] : [];
                $log->out("limited_rep_uid: $limited_rep_uid has no clients") if $log;
            } elsif ($limited_rep->{rep_type} ne $REP_LIMITED) {
                $client_client_ids = [];
            }
        # если фича для агентства выключена, то должно быть старое поведение (представитель видит в балансе всех клиентов агентства)
        # если есть флаг force_reset (например выключаем фичу для агенства и хотим сбросить список клиентов чтобы снова видел всех), то передадим []
        # если нет флага, то не делаем запрос в баланс 
        } else {
            if ($force_reset) { 
                $client_client_ids = [];
            } else {
                $log->out("limited_rep_uid: $limited_rep_uid, force_reset is false, skip this rep") if $log;
                $result->{ $limited_rep_uid } = 1;
                next;
            }
        }
        $log->out("limited_rep_uid: $limited_rep_uid, clients: " . to_json($client_client_ids)) if $log;

        # устанавливаем в балансе список клиентов для ограниченного представителя 
        $result->{ $limited_rep_uid } = balance_set_clients_of_agency_limited_rep($operator_uid, $agency_client_id, $limited_rep_uid, $client_client_ids, $log);
        $log->out(sprintf('limited_rep_uid: %s, success: %s', $limited_rep_uid, $result->{ $limited_rep_uid })) if $log;
    } 
    return $result;
}

sub _get_limited_agency_reps
{
    my ($uids) = @_;

    my $filtered_uids = [uniq grep { $_ } @{ $uids // [] }];
    return [] unless @$filtered_uids;

    Rbac::clear_cache();
    my $uid2info = Rbac::get_key2perminfo(uid => $filtered_uids);

    my @limited_reps = map { {uid => $_, ClientID => $uid2info->{$_}->{ClientID}, rep_type => $uid2info->{$_}->{rep_type}} } 
        grep {  $uid2info->{$_} 
            &&  $uid2info->{$_}->{ClientID} 
            && ($uid2info->{$_}->{role} eq $ROLE_AGENCY) } 
        @$filtered_uids;
    return \@limited_reps;
}

sub _get_clients_of_agency_reps
{
    my ($agency_uids) = @_;
    return {} unless $agency_uids && @$agency_uids;

    my $rows = get_all_sql(PPC(shard=>'all'), [
        'SELECT c.ClientID, c.agency_uid FROM clients c',
         WHERE => { 'c.agency_uid' => $agency_uids },
        'UNION',
        'SELECT ClientID, agency_uid FROM agency_lim_rep_clients',
         WHERE => { agency_uid => $agency_uids }
    ]);

    my $result = {};
    foreach my $row (@$rows) {
        push @{ $result->{ $row->{agency_uid} } }, $row->{ClientID};
    }

    return $result;
}

=head2 balance_set_clients_of_agency_limited_rep

 передать список клиентов ограниченного представителя агентства в баланс

=cut

sub balance_set_clients_of_agency_limited_rep {
    my ($operator_uid, $agency_client_id, $agency_rep_uid, $client_client_ids, $log) = @_;
    $operator_uid = $agency_rep_uid unless $operator_uid;

    my @params = ($operator_uid, $agency_client_id, $agency_rep_uid, $client_client_ids);
    my ($code) = eval {
        my ($balance_call_res, $text) = Yandex::Balance::balance_call('Balance.CreateUserClientAssociation', \@params, {write_log => 1});
        if ($balance_call_res) { # логируем ошибку
            my $error = sprintf('Balance error in balance_set_clients_of_agency_limited_rep(): Balance.CreateUserClientAssociation return (%s, %s) for params: %s', 
                $balance_call_res, $text, to_json(\@params));
            if ($log) {
                $log->out($error);
            } else {
                warn $error;
            }
        }
        $balance_call_res;
    };
 
    if ( $@ || $code ) {
        return 0;
    }
    return 1;
}

=head2 get_client_agency_options

    Возвращает для клиента его права и description на все агентства
        если передан параметр for_cur_agency, то возвращаются права только для агентства-оператора

=cut

sub get_client_agency_options
{
    my (%O) = @_;

    my $subclients_rights = rbac_mass_get_subclients_rights(undef, $O{operator_uid}, $O{client_uids2_client_ids}, 
        for_cur_agency_flag => $O{for_cur_agency},
        not_filter_by_perm => $O{not_filter_by_perm},
        operator_role => $O{operator_role},
        );

    if (%$subclients_rights) {
        my @agency_client_ids = uniq map {keys %{$_}} values %{$subclients_rights};
        my $agencies_info = get_hashes_hash_sql(PPC(ClientID => \@agency_client_ids), ["select cl.ClientID, IFNULL(cl.name, u.FIO) as agency_name
                                                         from users u
                                                         left join clients cl using(ClientID)
                                                     ", where => {'u.ClientID' => SHARD_IDS}
                                                     ]);

        my $client_descriptions_raw = get_all_sql(PPC(ClientID => [keys %$subclients_rights]), ["select client_client_id, agency_client_id, client_description
                                                               from agency_client_relations
                                                           ", where => { agency_client_id => \@agency_client_ids
                                                                       , client_client_id => SHARD_IDS
                                                                       }
                                                            ]);
        my $client_descriptions;
        foreach my $row (@$client_descriptions_raw) {
            $client_descriptions->{$row->{client_client_id}}{$row->{agency_client_id}} = $row->{client_description};
        }


        foreach my $client_id (keys %$subclients_rights) {
            for my $agency_client_id (keys %{$subclients_rights->{$client_id}}) {
                $subclients_rights->{$client_id}{$agency_client_id}{client_description} = $client_descriptions->{$client_id}{$agency_client_id};
                $subclients_rights->{$client_id}{$agency_client_id}{agency_name} = $agencies_info->{$agency_client_id}{agency_name};
                $subclients_rights->{$client_id}{$agency_client_id}{ClientID} = $agency_client_id;
            }
        }
    }

    return $subclients_rights;

}

=head2 get_lim_rep_group_id

    Возвращает для ограниченного представителя номер группы

=cut


sub get_lim_rep_group_id {
    my ($lim_rep_uid) = @_;

    return get_one_field_sql(PPC(uid => $lim_rep_uid), 'SELECT group_id FROM users_agency WHERE uid=?', $lim_rep_uid);
}

=head2 get_lim_reps_group_ids

    Возвращает для списка ограниченных представителей номера групп

=cut


sub get_lim_reps_group_ids {
    my ($lim_rep_uids) = @_;

    return get_hash_sql(PPC(uid => $lim_rep_uids), ['SELECT uid, IFNULL(group_id, 0) FROM users_agency', WHERE => { uid => SHARD_IDS }]);
}

=head2 get_lim_rep_group_chief_uid

    Возвращает uid тимлида группы для заданного uid'a ограниченного представителя

=cut


sub get_lim_rep_group_chief_uid {
    my ($lim_rep_uid) = @_;

    return get_one_field_sql(PPC(uid => $lim_rep_uid), 'SELECT uid FROM users_agency WHERE group_id = (SELECT group_id FROM users_agency WHERE uid = ?) AND lim_rep_type="chief"', $lim_rep_uid);
}

=head2 get_lim_rep_group_main_uids

    Возвращает uid'ы всех ограниченных представителей группы

=cut


sub get_lim_rep_group_main_uids {
    my ($lim_rep_uid) = @_;

    return get_one_column_sql(PPC(uid => $lim_rep_uid), 'SELECT uid FROM users_agency WHERE group_id = (SELECT group_id FROM users_agency WHERE uid = ?) AND lim_rep_type="main"', $lim_rep_uid);
}

=head2 has_lim_rep_group_main_reps

    Возвращает признак наличия в группе менеджеров

=cut


sub has_lim_rep_group_main_reps {
    my ($lim_rep_uid) = @_;
    return @{get_lim_rep_group_main_uids($lim_rep_uid)} ? 1 : 0;
}

=head2 mass_has_lim_rep_group_main_reps

    Возвращает признак наличия в группе менеджеров, для заданных тимлидов

=cut


sub mass_has_lim_rep_group_main_reps {
    my ($chief_lim_reps_with_group_ids) = @_;

    my $chief_lim_rep_uids = [keys %$chief_lim_reps_with_group_ids];
    my %groups_with_main = map { $_ => 1 } @{get_one_column_sql(PPC(uid => $chief_lim_rep_uids),
        ['SELECT group_id FROM users_agency',
        WHERE => { group_id => [values %$chief_lim_reps_with_group_ids], lim_rep_type => 'main' },
        'GROUP BY' => 'group_id'])};
    return { map { $_ => $groups_with_main{$chief_lim_reps_with_group_ids->{$_}} // 0 } @$chief_lim_rep_uids };
}

=head2 get_lim_rep_group

    Возвращает хэш с uid'ами группы по типам ограниченных представителей

=cut


sub get_lim_rep_group {
    my ($lim_rep_uid) = @_;

    my $rows = get_all_sql(PPC(uid => $lim_rep_uid),
        'SELECT lim_rep_type, uid FROM users_agency WHERE group_id = (SELECT group_id FROM users_agency WHERE uid = ?)', $lim_rep_uid);
    my $group = {};
    for my $row (@$rows) {
        if ($row->{lim_rep_type} eq 'chief') {
            $group->{chief} = $row->{uid}; # тимлид в группе только один, не делаем массив
        } else {
            push @{$group->{$row->{lim_rep_type}}}, $row->{uid};
        }
    }

    return $group;
}

=head2 unlink_lim_reps_from_client

    Удаляем связки клиента с ограниченными представителями

=cut

sub unlink_lim_reps_from_client {
    my ($client_id, $lim_rep_uids) = @_;

    die 'empty lim rep list' unless ($lim_rep_uids && @$lim_rep_uids);

    do_delete_from_table(PPC(ClientID => $client_id), 'agency_lim_rep_clients', where => { ClientID => $client_id, agency_uid => $lim_rep_uids });

    return;
}

=head2 unlink_all_clients_from_lim_rep

    Отвязываем всех клиентов от указанного представителя

=cut

sub unlink_all_clients_from_lim_rep {
    my ($lim_rep_uid, $agency_chief_uid) = @_;
    die 'lim_rep_uid is required param' unless $lim_rep_uid;
    die 'agency_chief_uid is required param' unless $agency_chief_uid;

    do_update_table(PPC(shard => 'all'), 'campaigns', {AgencyUID => $agency_chief_uid}, where=>{AgencyUID => $lim_rep_uid});
    do_update_table(PPC(shard => 'all'), 'clients', {agency_uid => $agency_chief_uid}, where=>{agency_uid => $lim_rep_uid});
    do_delete_from_table(PPC(shard => 'all'), 'agency_lim_rep_clients', where => { agency_uid => $lim_rep_uid });

    return;
}

=head2 link_lim_reps_to_client

    Создаем связи клиента с ограниченными представителями

=cut

sub link_lim_reps_to_client {
    my ($client_id, $lim_rep_uids) = @_;

    die 'empty lim rep list' unless ($lim_rep_uids && @$lim_rep_uids);

    my $tuples_to_insert = [ map { [$client_id, $_] } @$lim_rep_uids ];
    do_mass_insert_sql(PPC(ClientID => $client_id), 'INSERT IGNORE INTO agency_lim_rep_clients (ClientID, agency_uid) values %s', $tuples_to_insert);

    return;
}

=head2 get_agency_lim_rep_chiefs

    Получаем список менеджеров ограниченных представителей агентства

=cut

sub get_agency_lim_rep_chiefs {
    my ($agency_client_id) = @_;

    return get_all_sql(PPC(ClientID => $agency_client_id),
        [ "SELECT uid, login, FIO, group_id FROM users u JOIN users_agency ac USING(uid)", WHERE => { ClientID => $agency_client_id, rep_type => 'limited', lim_rep_type => 'chief' } ]);
}

=head2 relink_lim_rep_clients_old_schema

    Перепривязываем по новой схеме клиентов, привязанных к ограниченному представителю по старой схеме

=cut

sub relink_lim_rep_clients_old_schema {
    my ($lim_rep_uid) = @_;


    my $client_ids = get_one_column_sql(PPC(shard=>'all'), [ 'SELECT ClientID FROM clients', WHERE => { 'agency_uid' => $lim_rep_uid } ]);
    if (@$client_ids) {
        my $agency_chief_rep_uid = rbac_get_chief_rep_of_agency_rep($lim_rep_uid);
        do_update_table(PPC(shard => 'all'), 'campaigns', {AgencyUID => $agency_chief_rep_uid}, where=>{AgencyUID => $lim_rep_uid});
        do_update_table(PPC(shard => 'all'), 'clients', {agency_uid => $agency_chief_rep_uid}, where=>{agency_uid => $lim_rep_uid});

        foreach my $client_id (@$client_ids) {
            link_lim_reps_to_client($client_id, [$lim_rep_uid]);
        }
    }

    return;
}

=head2 link_lim_rep_clients_to_other_lim_rep

    Привязываем клиентов ограниченного представителя к заданному ограниченному представителю, не отвязывая от текущего

=cut

sub link_lim_rep_clients_to_other_lim_rep {
    my ($source_lim_rep_uid, $target_lim_rep_uid) = @_;

    my $client_ids = get_one_column_sql(PPC(shard=>'all'), [ 'SELECT ClientID FROM agency_lim_rep_clients', WHERE => { 'agency_uid' => $source_lim_rep_uid } ]);

    foreach my $client_id (@$client_ids) {
        link_lim_reps_to_client($client_id, [$target_lim_rep_uid]);
    }

    return;
}

=head2 swap_lim_rep_chiefs

    Меняем тимлидов в группах ограниченных представителей агентства, одна из групп без менеджеров

=cut

sub swap_lim_rep_chiefs {
    my ($empty_lim_rep_chief_uid, $lim_rep_chief_uid) = @_;

    my $group_ids = get_lim_reps_group_ids([$empty_lim_rep_chief_uid, $lim_rep_chief_uid]);

    link_lim_rep_clients_to_other_lim_rep($lim_rep_chief_uid, $empty_lim_rep_chief_uid);
    do_update_table(PPC(uid => $lim_rep_chief_uid), 'users_agency', { group_id => $group_ids->{$lim_rep_chief_uid} }, where => { uid => $empty_lim_rep_chief_uid });
    do_update_table(PPC(uid => $empty_lim_rep_chief_uid), 'users_agency', { group_id => $group_ids->{$empty_lim_rep_chief_uid} }, where => { uid => $lim_rep_chief_uid });

    return;
}

1;
