package Apache::AuthAdwords;
## no critic (TestingAndDebugging::RequireUseWarnings)

use strict;
use Apache2::Const qw(:common);
use Apache2::RequestUtil ();
use Apache2::Connection;
use URI::Escape;

use Yandex::Trace;
use Yandex::Blackbox;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::SendMail;
use Yandex::Log;

use LogTools qw/log_messages/;
use IpTools;
use EnvTools;
use HttpTools;

use CGIParams;
use URI::Escape qw/uri_escape uri_escape_utf8 uri_unescape/;

use Settings;
use RBACElementary;
use RBAC2::Extended;
use Direct::ResponseHelper;
use TvmChecker;
use Property;

# TODO: перенести константу в lib/Yandex/Blackbox.pm
# https://doc.yandex-team.ru/Passport/AuthDevGuide/concepts/DB_About.html#DB_About__aliases
my $BB_PORTAL_ALIAS_TYPE = 1;

my $BLACKBOX_USE_MULTISESSION_PROPERTY //= Property->new('BLACKBOX_USE_MULTISESSION');

sub handler
{
    my $r = shift;
    my $ttime = time;

    # у нас это самая ранняя точка обработки запроса, поэтому стартуем таймер тут
    my $trace = Yandex::Trace->new(method => 'auth', service => 'direct.perl.web');
    $r->pnotes(trace => $trace);

    Apache2::RequestUtil->request($r);

    local $Yandex::TVM2::APP_ID;      $Yandex::TVM2::APP_ID //= $Settings::TVM2_APP_ID{web};
    local $Yandex::TVM2::SECRET_PATH; $Yandex::TVM2::SECRET_PATH //= $Settings::TVM2_SECRET_PATH{web};
    no warnings 'once';
    local $Yandex::Blackbox::BLACKBOX_USE_TVM_CHECKER //= \&TvmChecker::use_tvm;

    # без subprocess_env нет %ENV
    $r->subprocess_env;

    my $authredirlevel = 0;
    my $uid = auth2($r);
    my $scheme = http_server_scheme($r);

    my $args = $r->args;
    my $arg_authredirlevel = '';

    if ($args =~ /authredirlevel=(\d+).(\d+)/ && time() - $1 < 60) {
        #found authredirlevel bookmark & time of bookmark not expired
        my $level = $2 + 1;
        $arg_authredirlevel = "authredirlevel=$1.$level";
        die "too many redirs $1" if ($level > 10);
    } else {
        #not found authredirlevel bookmark or time of bookmark expired
        $arg_authredirlevel = "authredirlevel=".time().".0";
    }

    if ( !defined $uid  || $uid < 0 ) {
        my $uri = ($args =~ /is_welcome=1/) ? '/dna/welcome/' : $r->uri; # костылик для редиректа с /dna/welcome/
        my $retpath = $scheme.'://'.http_server_host($r).$uri.'?'.$arg_authredirlevel;
        $retpath = build_retpath($r, $retpath);

        my $params = {retpath => $retpath};

        if (!defined $uid) {
            # нет куки Session_id или не авторизован
            redirect($r, get_passport_url($r, 'auth', $params));

            return REDIRECT;
        } elsif ($uid == -1) {
            hash_merge $params, {
                mode => 'postregistration',
                create_login => 1,
                create_password => 1,
            };
        } elsif ($uid == -2) {
            hash_merge $params, {
                mode => 'light2full',
            };
        } elsif ($uid == -3) {
            # обновление сессии
            redirect($r, get_passport_url($r, 'auth/update', $params));
            return REDIRECT;
        } else {
            die "unknown error code: $uid";
        }
        # залогинен через социальный профиль - нужно создать нормальный логин
        redirect($r, get_passport_url($r, 'passport', $params));
        return REDIRECT;        
    } else {
        return OK;
    }
}

sub build_retpath
{
    my($r, $retpath) = @_;
    my $headers_in = $r->headers_in();
    # загрузка файла через HTML5 File API -- сохранять форму не имеет смысла, т.к. ajax все равно не обработает редирект
    # а также, это приведет к ошибке "The LENGTH argument can't be negative" в CGI.pm
    if ($headers_in->{'X-Requested-With'} eq 'XMLHttpRequest' and $headers_in->{'Content-Type'} eq 'multipart/form-data' ) {
        return $retpath;
    }
    my $args = $r->args;
    $args =~ s/\bauthredirlevel=[0-9]+\.[0-9]+&?//;
    $args =~ s/cmd=chooseCountryCurrency&is_welcome=1&//; # костылик для редиректа в паспорт с /dna/welcome
    my ($session_id) = get_cookie($r, 'Session_id');

    my $current_retpath = uri_escape_utf8($retpath . "&" . $args);
    # assumes that query string to passport not more than 548 symbols
    # so resultant query string not more than 2048 symbols
    if ($r->method() eq 'POST' || length($current_retpath) > 1500) {
        my $cgi_form_id =  save_cgi_params(additional_key => $session_id, method => $r->method());
        $retpath .= "&" . $CGIParams::param_name . "=$cgi_form_id" if $cgi_form_id;
    } else {
        $retpath .= "&" . $args if $args;
    }

    return $retpath;
}

sub auth2
{
    my $r = shift;

    # extract sessionid from cookie
    my ($session_id) = get_cookie($r, 'Session_id');
    return undef if !$session_id;
    
    my $remote_ip = http_remote_ip($r);

    my $logger = sub {
        my ($msg) = @_;
        log_messages("auth_error", "$remote_ip: $msg");
    };

    my ($session_id2) = get_cookie($r, 'sessionid2');
    if(!$session_id2 && http_server_scheme($r) eq 'https'){
        $logger->("no sessionid2 cookie");
        $r->pnotes(sessionid2_is_needed => 1) if $r->can('pnotes');
    }

    no warnings 'once';
    local $Yandex::Blackbox::USE_MULTISESSION_IN_SESSIONID = $BLACKBOX_USE_MULTISESSION_PROPERTY->get(60);

    # blackbox request
    my $bb_res = eval { bb_sessionid($session_id, $remote_ip, 'direct.'.yandex_domain($r)
                              , [$BB_LOGIN, $BB_FIO, $BB_NICKNAME, $BB_SUID_DIRECT], 'getdefault', $session_id2, [ $BB_PORTAL_ALIAS_TYPE ]) };

    if ($@) {
        send_alert($@, 'BlackBox error'); 
        die $@;
    }

    # Session_id doesn't cor with sessionid2
    if($session_id2 && $bb_res->{auth}->{secure}->{content} == 0){
        $logger->("incorrect sessionid2 cookie: uid=$bb_res->{uid}");
        $r->pnotes(sessionid2_is_needed => 1) if $r->can('pnotes');
    }

    # если аккаунт пользователя не имеет портального алиаса, и пользователь не вводил пароль в Яндексе - просим ввести пароль или
    # дорегистироваться до полноценного портального аккаунта
    # по сути sessionid2_is_needed означает необходимость доавторизации или ввода пароля
    my $verif_age = $bb_res->{auth}->{password_verification_age}->{content};
    my $password_not_verified_in_session = !defined $verif_age || $verif_age < 0;
    my $is_portal_user = exists $bb_res->{aliases} && exists $bb_res->{aliases}{alias} && $bb_res->{aliases}{alias}{type} == $BB_PORTAL_ALIAS_TYPE;
    if ($password_not_verified_in_session && !$is_portal_user) {
        $logger->("blackbox alias type is not portal and incorrect password_verification_age for uid=$bb_res->{uid}");
        $r->pnotes(sessionid2_is_needed => 1) if $r->can('pnotes');
    }

=pod

    Для не продакшен конфигураций разрешаем супер-пользователям представляться любым пользователем Директа
    
    Условия:
        1. исходный пользователь - обладает ролью 'super'
        2. конфигурация сервера - все, кроме продакшен
        3. запрос из внутренней сети

=cut

    if (! is_production()) {
        my $log = Yandex::Log->new(log_file_name => 'fake-auth.log');

        # кука для авторизации под другим логином в Директе
        my ($fake_name) = get_cookie($r, 'fake_name');
        if ($fake_name) {
            
            # иногда браузер может записывать в куку - null
            if ($fake_name =~ /^\S+$/ && $fake_name ne 'null') {
    
                # проверяем, что пользователь авторизован и запрос из внутренней сети
                if ($bb_res->{uid}
                        && is_ip_in_list($ENV{REMOTE_ADDR}, $Settings::INTERNAL_NETWORKS)
                ) {
                    ### проверка роли пользователя в Директе - разрешаем только у супер-пользователям
                        my $local_rbac = RBAC2::Extended->get_singleton($bb_res->{uid});
                        my $rbac_info = rbac_who_is_detailed($local_rbac, $bb_res->{uid});
                    ####
                    
                    if ($rbac_info->{role} eq 'super' || $rbac_info->{role} eq 'superreader' || $rbac_info->{role} eq 'support') {
                        if ($fake_name && $fake_name =~ /^\S+$/i) {
                            my $real_login = $bb_res->{'dbfield'}{$BB_LOGIN};
                            
                            # если все проверки успешно пройдены
                            $log->out(["AuthAdword", "fake authorization"
                                                , $bb_res->{uid}, $real_login, $fake_name]);
                            
                            my $bb_params;
                            
                            if ($fake_name =~ /^\d+$/) {
                                # если указан UID
                                $bb_params = {uid => $fake_name};
                            } else {
                                # если указан логин - то получаем по логину uid
                                $bb_params = {login => $fake_name};
                            }

                            my $bb_fake_user = bb_userinfo( $bb_params
                                                                , $ENV{REMOTE_ADDR} || '127.0.0.1'
                                                                , "direct.yandex.ru");
                            
                            if ($bb_fake_user->{uid}) {
                                # передаем внутрь контроллера реальные данные клиента
                                if ($r->can('pnotes')) {
                                    $r->pnotes(real_uid => $bb_res->{uid});
                                    $r->pnotes(real_login => $real_login);
                                }
                                else{
                                    $r->{_REAL_UID} = $bb_res->{uid};
                                    $r->{_REAL_LOGIN} = $real_login;
                                }
                                
                                # мимикрируем
                                $bb_res->{uid} = $bb_fake_user->{uid};
                            } else {
                                # для печати сообщения об ошибке в интефейсе
                                $r->pnotes(fake_auth_error => 'incorrect-login');
                                
                                $log->out(["AuthAdword", "Error: Incorrect fake user login", {AuthAdword_ENV => \%ENV}]);
                            }
                        } else {
                            $log->out(["AuthAdword", "Error: Incorrect fake user login", {AuthAdword_ENV => \%ENV}]);
                        }
                    } else {
                        # не супер-пользователь пытается представиться другим логином
                        $log->out(["AuthAdword", "Error: Not superuser try to login as other user", {AuthAdword_ENV => \%ENV}]);
                    }
                } else {
                    $log->out(["AuthAdword", "Error: Try unauthorized using fake auth", {AuthAdword_ENV => \%ENV}]);
                }
            }
        } else {
            # 
        }
    }

    if ($bb_res->{valid} && !defined $bb_res->{login}) {
        # залогинен через социальный профиль - нужно создать нормальный логин
        return -1;
    } elsif ($bb_res->{valid} && !defined $bb_res->{uid} && $bb_res->{liteuid}) {
        # недорегистрированный лайт-пользователь - нужно создать нормальный логин
        return -2;
    } elsif ($bb_res->{renew} && $r->method() eq 'GET') {
        # нужно обновить сессию
        return -3;
    } elsif (!$bb_res->{valid}) {
        # не авторизован
        return undef;
    } elsif ($bb_res->{valid} ) {
        # all ok
        # в коде используем pnotes, т.к. в них сохраняется utf'ность строк. 
        # отдельно uid пишем и в обычные notes -- для access log'а

        if ($r->isa('Apache2::RequestRec')){
            #если ПДД-пользователь
            $r->pnotes(is_pdd_user => 1) if ($bb_res->{hosted}); #reject_pdd_users

            $r->notes->set(uid       => $bb_res->{uid});
            $r->err_headers_out->set('X-Accel-Uid' => $bb_res->{uid});
            $r->headers_out->set('X-Accel-Uid' => $bb_res->{uid});

            $r->pnotes(uid               => $bb_res->{uid});
            $r->pnotes(user_login        => $bb_res->{dbfield}{$BB_LOGIN});
            $r->pnotes(user_fio          => $bb_res->{dbfield}{$BB_FIO});
            $r->pnotes(user_email        => $bb_res->{default_email});
            $r->pnotes(nickname          => $bb_res->{dbfield}{$BB_NICKNAME});
            $r->pnotes(karma             => $bb_res->{karma}{content});
            $r->pnotes(display_name      => $bb_res->{display_name});
            $r->pnotes(address_list      => $bb_res->{'address-list'}) if defined $bb_res->{'address-list'};
            $r->pnotes(blackbox_users    => $bb_res->{user}) if defined $bb_res->{user};
            $r->pnotes(allow_more_users  => $bb_res->{allow_more_users}->{content}) if defined $bb_res->{allow_more_users};
        }
        else {
            $r->{_DISPLAY_NAME} = $bb_res->{display_name};
            $r->{_LOGIN} = $bb_res->{dbfield}{$BB_LOGIN};
            $r->{_ADDRESS_LIST} = $bb_res->{'address-list'} if defined $bb_res->{'address-list'};
            $r->{_BLACKBOX_USERS} = $bb_res->{user} if defined $bb_res->{user};
            $r->{_ALLOW_MORE_USERS} = $bb_res->{allow_more_users}->{content} if defined $bb_res->{allow_more_users};
        }

        return $bb_res->{uid};
    }
}

1;
__END__
