package AutoBroker;

=head1 NAME

    AutoBroker
    $Id$
    Вычисление действующей на поиске, ...

=head1 DESCRIPTION

=cut

use strict;
use warnings;

use List::Util qw/max min/;
use Carp qw/longmess/;

use Currencies;
use PlacePrice;

use base qw/Exporter/;
our @EXPORT = qw/calc_price calc_coverage/;

=head2 calc_price

  Вычисление действующей на поиске позиции, ...
  На входе: 
        {
            price => 0.XX, # в долларах
            camp_rest => 0.XX, # остаток на кампании в валюте кампании без НДС
            day_budget => 0.XX, # дневное ограничение бюджета на сегодня в валюте кампании без НДС
                                # 0, если ограничение не установлено
            spent_today => 0.XX, # истраченная кампанией за сегодня сумма в валюте кампании без НДС
            premium  => [{bid_price=>4640000, amnesty_bid=>5500000},...], 
            guarantee  => [{bid_price=>5760000, amnesty_bid=>6110000},...], 
            bottom_prices => [1,2,3,4], # в центах
            bottom_probs => [0.12, 0.23, 0.45, 0.99], 
            bottom_autobroker => [1, 4, 8], # в центах
            autobudget = Yes/No - включен или выключен автобюджет
            autobudget_bid - в уе - ставка автобюджета
            min_price - в уе - минимальная ставка из БК
            currency => 'YND_FIXED'|'RUB'|'USD'|... - валюта
            only_first_guarantee => 0/1 # признак, если задан - считаем что у нас только одна позиция показа (используем для mcbanner)
        }
  На выходе:
        {
            broker_price => X.XX, # 0 - если не показываем
            broker_coverage => 0.XX,
            broker_truncated => 1,
            broker_place => int,
            broker_place_name => '1-premium'|'2-premium'|'3-premium'|'1-guarantee'|'2-guarantee'|'3-guarantee'|'4-guarantee'|'bottom', 
        }

    Учитывает 4 существующие в данный момент стратегии: автобюджет, минимальная цена в блоке, наивысшая позиция, показ справа
    а также корректировку цены по временному таргетингу
    Учитывает дневное ограничение бюджета

    Внутри все деньги обрабатываются в виде целых чисел, умноженных на 100.

=cut

# Такой же расчет цены происходит в autobroker.js :: setPrice. Как бы объединить...
# Такой же расчет цены происходит в data3/desktop.blocks/i-utils/__autobroker/i-utils__autobroker.utils.js caclPrice

sub calc_price {
    my ($data) = @_;
    return {} if $data->{banner_without_text};

    # задан коэффициент временного таргетинга
    my $timetarget_coef = 1;
    my $broker_place_without_coef;
    if ($data->{timetarget_coef} && $data->{timetarget_coef} > 0 && $data->{timetarget_coef} < 100
        &&
        ! ($data->{autobudget} && $data->{autobudget} eq 'Yes')
       )
    {
        $timetarget_coef = $data->{timetarget_coef} / 100;
        my $price_data_without_coef = calc_price({%$data, timetarget_coef => 100});
        $broker_place_without_coef = $price_data_without_coef->{broker_place};
    }

    my $currency = $data->{currency};
    die 'no currency given' unless $currency;
    my $min_price_constant = get_currency_constant($currency, 'MIN_PRICE');

    my $price = int(100 * $data->{price} * $timetarget_coef + 0.5);
    $price = $min_price_constant * 100 if $price == 0;

    # при автобюджете с указанной максимальной ставкой цена не может быть выше этой ставки
    if ($data->{autobudget} && $data->{autobudget} eq 'Yes' && $data->{autobudget_bid}) {
        $price = min($price, int($data->{autobudget_bid} * 100));
    }

    my $camp_rest = int(100*($data->{camp_rest}||0)+0.5);
    my $eff_price = $camp_rest > 0 && $camp_rest < $price ? $camp_rest : $price;

    my ($truncated, $broker_price, $coverage);

    # вычисляем цену
    # стратегия: наивысшая доступная позиция
    for my $pos (@{PlacePrice::get_usedto_show_places($data->{only_first_guarantee})}) {
        my $pos_price = PlacePrice::get_data_by_place($data, $pos);
        if ($pos_price->{bid_price} <= $price && $pos_price->{bid_price} > $eff_price) {
            # ставки рекламодателя достаточно для попадания на позицию, но остаток средств на кампании - меньше ставки
            $truncated = 1;
        } elsif ($pos_price->{bid_price} <= $eff_price) {
            $broker_price = $pos_price;
            $coverage = 1;
            last;
        }
    }

    my $place;
    my $gprice = PlacePrice::get_data_by_place($data, PlacePrice::get_guarantee_entry_place());
    my $pprice = PlacePrice::get_data_by_place($data, PlacePrice::get_premium_entry_place());
    my $zero_broker_price = {bid_price => 0, amnesty_price => 0};

    if ($broker_price->{bid_price} && $data->{strategy_no_premium}) {
        # выставленна стратегия "показ справа"
        my @possible_prices = grep {$_->{bid_price} <= $price}
                                map { PlacePrice::get_data_by_place($data, $_) }
                                    @PlacePrice::INTERFACE_GUARANTEE_PLACES;
        if ($data->{strategy_no_premium} eq 'highest_place') {
            $broker_price = $possible_prices[0];
        }
        if ($broker_price->{bid_price} > $price) {
            $broker_price = $zero_broker_price;
        } else {
            $truncated = 0;
        }
    }

    # если не нашлось место в гарантии - считаем покрытие
    if (!$broker_price->{bid_price}) {
        if ($data->{only_first_guarantee}) {
            $broker_price = $zero_broker_price;
        } else {
            $coverage = calc_coverage($eff_price, $data->{bottom_prices}, $data->{bottom_probs});
            if ($coverage) {
                $truncated = $price > $eff_price ? 1 : 0;
                # вычисляем цену автоброкера
                my $low_price =[ sort {$b <=> $a} grep {$_ <= $eff_price} @{$data->{bottom_autobroker}} ]->[0]
                                || $eff_price;
                $broker_price = { bid_price => $low_price,
                                  amnesty_price => $low_price
                                };
            } else {
                $broker_price = $zero_broker_price;
            }
        }
    }

    # учитываем автобюджет и обнуляем ставку на поиске, если она меньше минимальной

    # "Нижний" минимум - до 8-го места
    my $lmin = $data->{bottom_autobroker}[0];
    $lmin = max($min_price_constant, $lmin || 0);
    $lmin = min($pprice->{bid_price}, $lmin);

    my $eff_min_price = $data->{min_price} && $data->{min_price} > 0 ? max($data->{min_price}, $lmin) : $lmin;

    if ( $broker_price->{bid_price} < $eff_min_price ) {
        $broker_price = $zero_broker_price;
    }

    if ($data->{autobudget} && $data->{autobudget} eq 'Yes') {

        if ( $data->{autobudget_bid} && $data->{min_price} && $data->{autobudget_bid} < $data->{min_price}
            || $data->{min_price} && $data->{price} < $data->{min_price}
        ) {
            $broker_price = $zero_broker_price;
        }
    }

    # вычисляем место
    if ($data->{strategy_no_premium} && $broker_price->{bid_price} > 0) {
        $eff_price = min($eff_price, $broker_price->{bid_price});
    }

    # если имя блока ещё не определили (при стратегии min_price), то вычисляем сейчас.
    unless (defined($place)) {
        $place = $PlacePrice::PLACES{ROTATION};
        # В стратегии "показывать под результатами поиска" запретить рассматривать СР.
        my $forbidden_positions = ($data->{strategy_no_premium}) ? {map {$_ => 1} @$PlacePrice::INTERFACE_PREMIUM_PLACES} : {};
        foreach (@{PlacePrice::get_usedto_show_places($data->{only_first_guarantee})}) {
            next if $forbidden_positions->{$_} || $eff_price < PlacePrice::get_data_by_place($data, $_)->{bid_price};
            $place = $_;
            last;
        }
    }

    $broker_place_without_coef = $place unless $broker_place_without_coef;

    # округляем получившуюся ставку до ближайшего шага торгов в меньшую сторону
    my $broker_price_round = round_price_to_currency_step($broker_price->{amnesty_price} / 100, $currency, down => 1);

    return {
        broker_price => sprintf("%0.2f", $broker_price_round),
        broker_coverage => $coverage || 0,
        broker_truncated => $truncated ? 1 : 0,
        broker_place => $place,
        broker_place_without_coef => $broker_place_without_coef,
    };
}

=head2 calc_coverage

  Вычисление вероятности показа в ротации
  На входе: price, bottom_prices, bottom_probs
  На выходе: coverage (вероятность)
  Единицы измерения должны быть согласованы

  Сейчас нигде не используется

=cut
sub calc_coverage {
    my ($eff_price, $prices, $probs) = @_;
    return 0 if @$prices < 2 || @$prices != @$probs || $eff_price < $prices->[0];
    # находим, между какими элементами попадает цена
    my $low_i = 0;
    for(my $i = 0; $i <= $#{$prices}-1; $i++) {
        if ($prices->[$i] <= $eff_price) {
            $low_i = $i;
        } else {
            last;
        }
    }
    # находим пропорцию
    my $low_price = $prices->[$low_i];
    my $high_price = $prices->[$low_i+1] >= $eff_price ? $prices->[$low_i+1] : $eff_price;
    my $k = ($eff_price - $low_price) / (($high_price - $low_price)||1);
    return $probs->[$low_i] + $k * ( $probs->[$low_i+1] - $probs->[$low_i] );
}


=head2 parse_prices 

    разбираем цены из строк в массивы

    на входе: цены в формате bs_auction
        {
            premium => [ { amnesty_price => 2500100000, bid_price => 2500100000 }, { amnesty_price => 105600000, bid_price => 115300000 }, { amnesty_price => 104100000, bid_price => 104100000 } ],
            guarantee => [ { amnesty_price => 154500000, bid_price => 202500000 }, { amnesty_price => 137800000, bid_price => 189900000 }, { amnesty_price => 131000000, bid_price => 156900000 }, { amnesty_price => 129000000, bid_price => 129000000 } ],
            larr => "9800000:1272,21700000:13660,33600000:49480,45500000:117031,57400000:217161,69400000:342933,81300000:477412,93200000:605703,105100000:1000000,117000000:1000000,128900000:1000000|9800000,15100000,23900000,25800000,26600000,32900000,37800000,51200000,52100000,62400000,122400000",
        }        
    на выходе: цены в формате для AutoBroker::calc_price 
        {
            bottom_prices => [980, 2170, 3360, 4550, 5740, 6940, 8130, 9320, 10510, 11700, 12890 ], # в центах
            bottom_probs => [0.001272, 0.01366, 0.04948, 0.117031, 0.217161, 0.342933, 0.477412, 0.605703, 1, 1, 1], 
            bottom_autobroker => [ 980, 1510, 2390, 2580, 2660, 3290, 3780, 5120, 5210, 6240, 12240 ], # в центах

            guarantee =>[ { amnesty_price => 15450, bid_price => 20250 }, { amnesty_price => 13780, bid_price => 18990 }, { amnesty_price => 13100, bid_price => 15690 }, { amnesty_price => 12900, bid_price => 12900 } ],
            premium => [ { amnesty_price => 250010, bid_price => 250010 }, { amnesty_price => 10560, bid_price => 11530 }, { amnesty_price => 10410, bid_price => 10410 } ]
        }

=cut
sub parse_prices
{
    my ($raw_data) = @_;
    my ($larr_probs, $larr_autobroker) = split /\|/, $raw_data->{larr} || '';
    my $prices = {
        bottom_prices => [],
        bottom_probs => [],
        bottom_autobroker => [map {int($_/10_000)} grep {/^\d+$/} split(',', $larr_autobroker||'')]
    };
    for my $price_prob (split ',', $larr_probs||'') {
        if ($price_prob =~ /^(\d+):(\d+)$/) {
            push @{$prices->{bottom_prices}}, int($1/10_000);
            push @{$prices->{bottom_probs}}, $2/1_000_000;
        }
    }
    for my $type (qw/premium guarantee/) {
        $prices->{$type} = [
            map { +{ bid_price => int($_->{bid_price}/10_000), amnesty_price => int($_->{amnesty_price}/10_000) } } 
            @{$raw_data->{$type}}
            ];
    }

    return $prices;
}


1;
