package AutobudgetAlerts;

=head1 NAME
    
    AutobudgetAlerts

=head1 DESCRIPTION

    Module for working with autobudget alerts

    Удаление неактуальных алертов переехало на java:
    https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/java/ru/yandex/direct/jobs/autobudget/OutdatedAutobudgetAlertsCleaner.java

=cut

use Direct::Modern;

use Carp;
use DateTime;
use Readonly;

use List::Util qw/minstr/;
use Yandex::DateTime;
use DateTime::Duration;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log;

use Settings;
use PrimitivesIds;

use Campaign::Types qw/get_camp_type_multi camp_kind_in camp_type_in/;

=head2 %ALERT_TYPE

Описание типов алёртов

    ВНИМАНИЕ!!!
    Пожалуйста, добавляйте параметры в оба проекта:
    https://a.yandex-team.ru/arc/trunk/arcadia/direct/core/src/main/java/ru/yandex/direct/core/entity/autobudget/model/AutobudgetAlertProperties.java

=cut

our %ALERT_TYPE = (
    hourly => {
        db_table => 'autobudget_alerts',
        fields => [ qw/ problems overdraft / ],
        log => 'ordersNotExceededBudget.log',
        ttl_for_active_alerts => DateTime::Duration->new( hours => 1, minutes => 10 ),
        ttl_for_frozen_alerts => [ days => -1 ],
        update_sub => \&_update_hourly_alert,
    },
    ab_cpa => {
        db_table => 'autobudget_cpa_alerts',
        fields => [ qw/ cpa_deviation apc_deviation / ],
        bs_fields => [ qw/ cpa apc / ],
        log => 'ordersWithCpaWarnings.log',
        ttl_for_active_alerts => DateTime::Duration->new( days => 1, hours => 2 ),
        ttl_for_frozen_alerts => [ days => -2 ],
        update_sub => \&_update_cpa_alert,
    },
);


our $AB_PROBLEMS = {
    NO_PROBLEMS              => 0,      # all goes normal
    IN_ROTATION              => (1 << 0), # there are not enouth budget to put all priorities in rotation
    MAX_BID_REACHED          => (1 << 1), # bid reached maximum, chosen by user
    MARGINAL_PRICE_REACHED   => (1 << 2), # bid reached max marginal price
    UPPER_POSITIONS_REACHED  => (1 << 3), # all priorities reached maximum possible positions, can't spent all budget
    ENGINE_MIN_COST_LIMITED  => (1 << 4), # minimum engine bid more that order max_click_cost
    LIMITED_BY_BALANCE       => (1 << 5), # order bids were limited because of low balance
    NO_BANNERS               => (1 << 6), # order has no active priorities
    LIMIT_BY_AVG_COST        => (1 << 7), # all banners reached avg click/action cost
    WALLET_DAILY_BUDGET_REACHED => (1 << 8), # wallet daily budget reached
};

# описание проблем см. в DIRECT-21638
Readonly our $HOURLY_NO_PROBLEM                     => 0;
Readonly our $HOURLY_MAX_BID_REACHED                => 1;
Readonly our $HOURLY_UPPER_POSITIONS_REACHED        => 2;
Readonly our $HOURLY_ENGINE_MIN_COST_LIMITED        => 3;
Readonly our $HOURLY_WALLET_DAY_BUDGET_REACHED      => 4;


sub update_alerts
{
    my ($mode, $data) = @_;

    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    my $log = new Yandex::Log(log_file_name => $self->{log}, date_suf => "%Y%m%d");
    $log->out("Start new update");

    eval {
        my @orders_ids = keys %$data;
        my $orderId_to_cid = get_orderid2cid(OrderID => \@orders_ids);
        my @cids = map { $orderId_to_cid->{$_} } keys %$orderId_to_cid;

        delete_old_stopped_alerts($mode);

        # Для некоторых кампаний могут игнорироваться алерты
        my $id2camp_type = get_camp_type_multi(cid => \@cids);
        my $ignored_cids = [];

        # выбираем активные алерты или алерты замороженные более суток назад
        my $day_ago = now()->add( @{ $self->{ttl_for_frozen_alerts} } );
        my $current_alerts = get_hashes_hash_sql(PPC(cid => \@cids), [
            "select cid,",
            ( map {"$_,"} @{ $self->{fields} } ),
            "status, last_update from $self->{db_table}",
            where => {
                cid => SHARD_IDS,
                _OR => {
                    _AND => { status => 'frozen', last_update__lt => "$day_ago" },
                    status => 'active',
                }
            }
        ]);

        my $all_cid_alerts = get_hashes_hash_sql(PPC(shard => 'all'), "select cid from $self->{db_table}");
        my %alert;
        for my $orderId (keys %$data) {
            my $bs_notification = $data->{$orderId};

            my $cid = $orderId_to_cid->{$orderId};
            
            # у нас нет такой кампании
            unless($cid){
                $log->out("CampaignID for OrderID $orderId doesn't exists");
                next;
            }

            if (_get_autobudget_problem_value($bs_notification->{problems}) == $HOURLY_UPPER_POSITIONS_REACHED && camp_type_in(type => $id2camp_type->{$cid}, 'performance')) {
                # для смартов не приходит из БК причина №2 "Достигнута максимальная позиция выдачи"
                push @$ignored_cids, $cid;
                next;
            }
            
            $log->out($orderId, $bs_notification);

            if ( not exists $all_cid_alerts->{$cid} ) {
                # пришла нотификация из БК, но алерта нет
                $alert{$cid} = [
                        $cid,
                        (map {$bs_notification->{$_}} @{ $self->{bs_fields} || $self->{fields} }),
                        'active',
                    ];
            } elsif ( exists $current_alerts->{$cid} && $current_alerts->{$cid}->{status} ne 'stopped' ) {
                # пришла нотификация из БК и есть не остановленный алерт
                my $updated_alert = $self->{update_sub}->($bs_notification, $current_alerts->{$cid});
                $alert{$cid} = $updated_alert;
            }
        }


        if ( %alert ) {
            for my $chunk ( sharded_chunks cid => [ keys %alert ] ) {
                my $shard = $chunk->{shard};
                my $cids = $chunk->{cid};
                my @records = map { $alert{$_} } @$cids;
                do_mass_insert_sql( PPC(shard => $shard),
                    join( q{ }, 
                        "insert $self->{db_table} (cid,",
                        ( map {"$_,"} @{$self->{fields}} ),
                        "status) values %s
                        on duplicate key update",
                        ( map {"$_=values($_),"} @{$self->{fields}} ),
                        "status=values(status),
                        last_update=now()",
                    ),
                    \@records , {max_row_for_insert => 10000}
                );
            }
        }

        # если замороженные более суток назад алерты не обновились до active или stop, то их не было в уведомлениях от БК
        do_delete_from_table(PPC(shard => 'all'), $self->{db_table},
            where => {status => 'frozen', last_update__lt => "$day_ago"}
        );
    
        $log->out({"Ignoring next campaigns" => $ignored_cids}) if @$ignored_cids;
    };

    my $response;
    if($@){
        $log->out($@);
        $response = { success => 0, error => 1 };
    } else {
        $response = { success => 1, error => 0 };
    }

    return $response;
}

sub _update_hourly_alert
{
    my $bs_data = shift;
    my $alert = shift;

    my $overdraft = $alert->{overdraft} || 0;
    my $problems = $alert->{problems} || 0;
    my $status = $alert->{status} || 'active';

    # если недотрата на заказе увеличилась, то активируем замороженные более суток назад алерты
    # если уменьшилась, то выключаем алерт
    if($status eq 'frozen'){
        # здесь $bs_data->{overdraft} > $alert->{overdraft} - правильно, потому что здесь overdraft - отрицательное число, т.е. недотрата
        $status = $bs_data->{overdraft} > $alert->{overdraft} ? 'stopped' : 'active';
    }

    # для всех активных алертов корректируем недотрату
    # и причину недотраты
    if($status eq 'active'){
        $overdraft = $bs_data->{overdraft};
        $problems = $bs_data->{problems};
    }

    return [$alert->{cid}, $problems, $overdraft, $status];
}

sub _update_cpa_alert
{
    my ($bs_data, $old_alert) = @_;
    my $status = $old_alert->{status};

    # активируем алерт, если отклонение cpa увеличилось
    if (
        $status eq 'frozen' && $bs_data->{cpa} && $bs_data->{cpa} > $old_alert->{cpa_deviation} ) {
        $status = 'active';
    }

    return [ $old_alert->{cid}, $bs_data->{cpa}, $bs_data->{apc}, $status ];
}

sub set_last_update
{
    my ($mode, $cid, $last_update) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    do_update_table(PPC(cid => $cid), $self->{db_table}, {last_update => "$last_update"}, where => {cid => $cid});
}


sub get_alert_data
{
    my ($mode, $cid) = @_;

    return mass_get_campaigns_alert_data($mode, cid => $cid)->{$cid};
}

=head2 mass_get_campaigns_alert_data($mode, $by, $what)

    Массово выгрузить автобюджетные проблемы типа $mode одной или нескольких кампаний.

    Аргумент $by обозначает ключ, по которому будет делаться выборка, и может принимать следующие значения
        cid  - выборка делается по отдельным кампаниям. В $what должен быть либо cid, либо массив из cid
        wallet_cid - выборка делается по кампаниям под указанными общими счетами. В $what должен быть либо cid общего счета,
            либо массив из cid общих счетов

    Возвращает хеш c автобюджетными проблемами по каждой кампании
    {
        cid => { problem data },
        cid => { problem data },
    }


=cut
sub mass_get_campaigns_alert_data
{
    my ($mode, $by, $what) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    if ($by eq 'cid') {
        return get_hashes_hash_sql(PPC(cid => $what), ["
                SELECT cid, status, ".join(', ', @{ $self->{fields} })."
                FROM ".$self->{db_table},
                WHERE => {cid => SHARD_IDS}
            ]);
    } elsif ($by eq 'wallet_cid') {
        return get_hashes_hash_sql(PPC(cid => $what), ["
                SELECT a.cid, a.status, ".join(', ', map { "a.$_" } @{ $self->{fields} })."
                FROM campaigns wc
                    JOIN campaigns c ON (c.ClientID = wc.ClientID AND c.wallet_cid = wc.cid)
                    JOIN ".$self->{db_table}." a ON a.cid = c.cid",
                WHERE => {'wc.cid' => SHARD_IDS}
            ]);
    } else {
        croak "unknown by param $by";
    }
}


sub get_autobudget_problem
{
    my ($cid) = @_;

    return mass_get_autobudget_problems(cid => $cid)->{$cid} // $HOURLY_NO_PROBLEM;
}

=head2 mass_get_autobudget_problems($by, $what)

    Массово выгружает коды автобюджетных проблем типа hourly
    $by может принимать следующие значения:
        cid - в $what должен быть cid или массив cid кампаний, по которым нужно получить текущую проблему
        wallet_cid - в $what должен быть cid или массив cid общих счетов. Для всех кампаний под этими общими счетами
            будут выведены текущие проблемы.

    Возвращает hashref вида
    {
        cid => $HOURLY_NO_PROBLEM,
        cid => $HOURLY_WALLET_DAY_BUDGET_REACHED,
        ...
    }

    если для кампании проблемы "заморожены" или "остановлены", для нее будет возвращено значение $HOURLY_NO_PROBLEM

=cut

sub mass_get_autobudget_problems
{
    my ($by, $what) = @_;

    my $alerts = mass_get_campaigns_alert_data('hourly', $by => $what);
    my $result = {};
    for my $cid (keys %$alerts) {
        my $alert = $alerts->{$cid};
        if ($alert->{status} ne 'active') {
            $result->{$cid} = $HOURLY_NO_PROBLEM;
            next;
        }

        $result->{$cid} = _get_autobudget_problem_value($alert->{problems}) // $HOURLY_NO_PROBLEM;
    }

    return $result;
}

=head2 _get_autobudget_problem_value

По битовой маске (значению от БК) получит численное значение, которое
используется в Директе

=cut

sub _get_autobudget_problem_value
{
    my $problems = shift;

    # описание в DIRECT-21638
    my $result;
    if (!$problems) {
        $result = $HOURLY_NO_PROBLEM; # нет проблем
    } elsif ($problems & $AutobudgetAlerts::AB_PROBLEMS->{WALLET_DAILY_BUDGET_REACHED}) {
        $result = $HOURLY_WALLET_DAY_BUDGET_REACHED;
    } elsif ($problems & $AutobudgetAlerts::AB_PROBLEMS->{ENGINE_MIN_COST_LIMITED}) {
        $result = $HOURLY_ENGINE_MIN_COST_LIMITED;
    } elsif ($problems & $AutobudgetAlerts::AB_PROBLEMS->{UPPER_POSITIONS_REACHED}) {
        $result = $HOURLY_UPPER_POSITIONS_REACHED;
    } elsif ($problems & ($AutobudgetAlerts::AB_PROBLEMS->{MAX_BID_REACHED} | $AutobudgetAlerts::AB_PROBLEMS->{MARGINAL_PRICE_REACHED})) {
        $result = $HOURLY_MAX_BID_REACHED;
    }

    return $result;
}

sub get_cpa_problems
{
    my ($cid) = @_;

    my $alert = get_alert_data( ab_cpa => $cid );
    return if !$alert;

    return (
        $alert->{status} eq 'active' ? $alert->{cpa_deviation} : 0,
        # отклонение конверсии отдаём даже для неактивных алертов
        $alert->{apc_deviation},
    );
}

sub freeze_alert
{
    my ($mode, $cid) = @_;

    mass_freeze_alert($mode, [$cid]);
}

=head2 mass_freeze_alert($mode, $cids)

    Массово "замораживает" автобюджетные проблемы типа $mode для кампаний из массива $cids

=cut

sub mass_freeze_alert
{
    my ($mode, $cids) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    do_update_table(PPC(cid => $cids), $self->{db_table}, {status => 'frozen'}, where => {cid => SHARD_IDS});
}

sub get_alert_status
{
    my ($mode, $cid) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    return get_one_field_sql(PPC(cid => $cid), "select status from $self->{db_table} where cid = ?", $cid);
}

sub set_alert_status
{
    my ($mode, $cid, $status) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    do_update_table(PPC(cid => $cid), $self->{db_table}, {status => $status}, where => {cid => $cid});
}

sub update_on_strategy_change
{
    my $params = shift;
    my ($cid, $old, $new) = @{$params}{qw(cid old_strategy new_strategy)};

    # проверяем часовые алерты
    if ( check_alert(hourly => $cid) ) {
        if (
            $old->{name} ne $new->{name} || 
            $old->{search}->{name} ne $new->{search}->{name} ||
            $old->{net}->{name} ne $new->{net}->{name}
        ) {
            # замораживаем алерт при изменении типа автобюджетной стратегии
            freeze_alert(hourly => $cid);
        } else {
            # замораживаем алерт если увеличилась максимальная ставка (описание проблем см. в DIRECT-21638)
            my $problem = get_autobudget_problem($cid);
            if (
                $problem == $HOURLY_MAX_BID_REACHED ||
                $problem == $HOURLY_ENGINE_MIN_COST_LIMITED ||
                $problem == $HOURLY_WALLET_DAY_BUDGET_REACHED
            ) {
                for my $type (qw(net search)){ # для поиска и сети
                    for my $bid_field (qw/bid avg_bid filter_avg_bid/) {
                        # для разных стартегий - разное название поля со ставкой
                        next unless defined $old->{$type}->{$bid_field} && defined $new->{$type}->{$bid_field};
                        if ($old->{$type}->{$bid_field} < $new->{$type}->{$bid_field}) {
                            freeze_alert(hourly => $cid);
                        }
                    }
                }
            }
        }
    }

    # алерты для стратегии по средней конверсии
    if ( check_alert( ab_cpa => $cid ) ) {
        if (
            # замораживаем при изменении типа автобюджетной стратегии
            $old->{name} ne $new->{name}
            || $old->{search}->{name} ne $new->{search}->{name}
            || $old->{net}->{name} ne $new->{net}->{name}
                    # или если изменилась целевая стоимость конверсии
            || !$new->{search}->{avg_cpa}
            || $new->{search}->{avg_cpa} != $old->{search}->{avg_cpa}
            || !$new->{net}->{avg_cpa}
            || $new->{net}->{avg_cpa} != $old->{net}->{avg_cpa}
        ) {
            freeze_alert( ab_cpa => $cid );
        }
    }

    return;
}

sub update_on_new_phrases_add
{
    my $cid = shift;
    my $problem = get_autobudget_problem($cid);
    if(($problem == $HOURLY_MAX_BID_REACHED || $problem == $HOURLY_UPPER_POSITIONS_REACHED) && check_alert(hourly => $cid)){
        freeze_alert(hourly => $cid);
    }
}

sub update_on_context_limit_change
{
    my ($cid, $old_context_limit, $new_context_limit) = @_;

    return unless check_alert(hourly => $cid); # нет алертов, нечего апдейтить
    # описание проблем см. в DIRECT-21638
    my $problem = get_autobudget_problem($cid);
    if($problem == $HOURLY_MAX_BID_REACHED || $problem == $HOURLY_UPPER_POSITIONS_REACHED){
        # морозим алерт если

        # включены показы на тематических площадках
        freeze_alert(hourly => $cid) if $old_context_limit == 254 && $new_context_limit <= 100 && $new_context_limit > 0;

         # увеличен % расхода на тематические площадки, но не морозим, если отлючаем РСЯ
        freeze_alert(hourly => $cid) if $old_context_limit < $new_context_limit && $new_context_limit != 254;
    }
}

sub update_on_broad_match_change
{
    my ($cid, $old_flag, $new_flag, $old_limit, $new_limit) = @_;

    return unless check_alert(hourly => $cid); # нет алертов, нечего апдейтить

    # описание проблем см. в DIRECT-21638
    my $problem = get_autobudget_problem($cid);
    if(($problem == $HOURLY_MAX_BID_REACHED || $problem == $HOURLY_UPPER_POSITIONS_REACHED) && $new_flag ){
        if(
            !$old_flag ||                                   # если это включение
            ($old_limit > 0 && $new_limit == -1) ||         # или поставили неограниченный лимит
            ($old_limit < $new_limit && $old_limit > 0)     # или если увеличили лимит
        ){
            freeze_alert(hourly => $cid);
        }
    }
}

=head2 update_on_wallet_day_budget_change($wallet_cid, $old_day_budget, $new_day_budget)

    Работа с автобюджетными проблемами при смене дневного бюджета на общем счету.
    Если дневной бюджет выключили, или увеличили, нужно заморозить проблемы на всех кампаниях под этим общим счетом
    про которые автобюджет сообщал о наличии проблем из-за дневного бюджета на общем счету

    $wallet_cid - cid общего счета
    $old_day_budget - сумма бывшего дневного бюджета (или 0, если он был выключен)
    $new_day_budget - сумма нового дневого бюджета (или 0, если его выключили)

=cut

sub update_on_wallet_day_budget_change
{
    my ($wallet_cid, $old_day_budget, $new_day_budget) = @_;

    if ($old_day_budget == 0 || $new_day_budget != 0 && $new_day_budget <= $old_day_budget) {
        return;
    }
    my $cids_problems = mass_get_autobudget_problems(wallet_cid => $wallet_cid);
    my @cids_with_wallet_problems = grep { $cids_problems->{$_} == $HOURLY_WALLET_DAY_BUDGET_REACHED } keys %$cids_problems;

    if (@cids_with_wallet_problems) {
        mass_freeze_alert(hourly => \@cids_with_wallet_problems);
    }
}


sub check_alert
{
    my ($mode, $cid) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    return get_one_field_sql(PPC(cid => $cid), "select 1 from $self->{db_table} where cid = ?", $cid);
}


sub delete_old_stopped_alerts
{
    my ($mode) = @_;
    my $self = $ALERT_TYPE{$mode}  or croak "Unknown mode: $mode";

    my $now = now()->truncate(to => 'day');
    my $min = minstr grep {defined $_} @{ get_one_column_sql(PPC(shard => 'all'), "select min(last_update) from $self->{db_table} where status = 'stopped'") || [] };                    
    return unless $min; # нет остановленный алертов, нечего удалять
    $min = datetime($min)->truncate(to => 'day');

    # каждый понедельник удаляем остановленные алерты, которые остановились не на текущей неделе
    if(DateTime->compare($now, $min) > 0 && $now->day_of_week() == 1){
        do_delete_from_table(PPC(shard => 'all'), $self->{db_table}, where => {status => 'stopped', last_update__lt => "$now"});
    }

    return;
}

1;
