package BS::CheckUrlAvailability;

=encoding utf8

=head1 NAME

    BS::CheckUrlAvailability

=head1 DESCRIPTION

    Модуль про БКшный мониторинг доступности доменов и работу с ним.

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq/;
use Yandex::DBTools;
use Yandex::URL qw/get_host get_protocol/;

use Campaign::Types;
use CampaignQuery;
use Client::ClientFeatures;
use Settings;
use Direct::Model::BannerText;
use JavaIntapi::Notifications qw/_get_url_monitoring_notifications/;

=head1 CONSTANTS

=head2 IGNORE_MONITORING_INTERVAL

    Время в часах, на которое игнорируем мониторинг и все же показываем объявления

=cut

use constant IGNORE_MONITORING_INTERVAL => 2;

=head2 MAX_DOMAIN_LENGTH

    Максимально-допустимая длина домена, зависит от размерности поля domain в ppcdict.domains_dict и ppc.domains

=cut

use constant MAX_DOMAIN_LENGTH => 100;

=head1 SUBROUTINES

=head2 off_by_monitoring_domain_id_joins_sql($banners_table_alias)

    Массив строк для добавления джойнов нужных для вычисления отключенности баннера по данным мониторинга БК

    $banners_table_alias - алиас таблицы banners, должен совпадать с алиасом в get_by, по умолчанию banners

=cut

sub off_by_monitoring_domain_id_joins_sql {
    my $banners_table_alias = shift // 'banners';
    return (
        "LEFT JOIN domains ON domains.domain = ${banners_table_alias}.domain",
        "LEFT JOIN bs_dead_domains ON bs_dead_domains.domain_id = domains.domain_id",
    );
}

=head2 get_monitoring_stopped_banners

    Возвращает все объявления группы, которые остановлены мониторингом.
    На входе:
        cid - campaigns.cid, номер кампании
    На выходе:
        ArrayRef[Direct::Model::Banner] - список моделей баннеров, которые остановлены.

=cut

sub get_monitoring_stopped_banners {
    my ($cid, $banner_ids) = @_;
    return [] unless (camp_kind_in(cid => $cid, 'allow_domain_monitoring'));

    my $campaign_data = CampaignQuery->get_campaign_data( cid => $cid, [qw/statusMetricaControl archived statusEmpty ClientID/]);
    return [] if (($campaign_data->{statusMetricaControl} // '') ne 'Yes' ||
                  ($campaign_data->{archived} // '') ne 'No' ||
                  ($campaign_data->{statusEmpty} // '') ne 'No');

    my @fields = (qw/bid pid href/, "bs_dead_domains.domain_id AS domain_id");
    my @from =  (qw/banners b/, off_by_monitoring_domain_id_joins_sql('b'));
    my $where = {'b.cid' => $cid, 'b.statusArch' => "No", 'b.statusShow' => "Yes", 'b.statusModerate' => "Yes", 'b.statusPostModerate' => "Yes"};
    $where->{'b.bid'} = $banner_ids if ($banner_ids && @$banner_ids);
    $where->{_TEXT} = "bs_dead_domains.domain_id IS NOT NULL";

    my $banners = Direct::Model::BannerText->from_db_hash_multi(get_all_sql(PPC(cid => $cid),
        ["SELECT", join (', ', @fields), "FROM", join (' ', @from),
         where => $where]));

    if (Client::ClientFeatures::get_is_feature_url_monitoring_online_request_enabled(client_id => $campaign_data->{ClientID})) {
        my @urls_to_check = [ uniq map {get_protocol($_->{href}) . get_host($_->{href})} @$banners ];
        if (@urls_to_check) {
            my ($dead_urls, $error) = JavaIntapi::Notifications::_get_url_monitoring_notifications(@urls_to_check);
            if (defined $dead_urls) {
                my $dead_urls_hash = { map {$_ => 1} @$dead_urls };
                $banners = [ grep {$dead_urls_hash->{get_protocol($_->{href}) . get_host($_->{href})}} @$banners ];
            }
        }
    }
    return $banners;
}

1;
