package BS::Export;
# $Id$

=encoding utf8

=head1 NAME

    BS::Export - утилиты для экспорта в БК (преимущественно bsClientData.pl)

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut

use Direct::Modern;

use Digest::CRC qw/crc32/;
use Readonly;
use List::MoreUtils qw/uniq pairwise any/;
use List::Util qw/min max/;
use URI::QueryParam;
use URI::Escape;

use Yandex::DBTools;
use Yandex::IDN;
use Yandex::MirrorsTools::Hostings qw/strip_www/;
use Yandex::ScalarUtils;
use Yandex::HashUtils qw(hash_cut);
use Yandex::ListUtils qw/nsort/;
use Yandex::Trace;
use Yandex::HTTP qw/http_fetch/;

use BS::URL;

use Campaign::Types qw/get_camp_kind_types camp_kind_in is_wallet_camp/;
use Client qw(get_client_NDS);
use Currencies qw(get_currency_constant);
use HashingTools ();
use MirrorsTools;
use Settings ();
use Property;
use WalletUtils;
use EnvTools qw/is_sandbox/;
use JSON;


# PriorityID, который пишем в базу, если ничего не получили от БК
# нужен, чтобы по-прежнему работала проверка было ли условие в БК вида PriorityID > 0
our $DEFAULT_PRIORITYID = 1;

use base qw/Exporter/;
our %EXPORT_TAGS = (
    sql => [qw/
        $SQL_CALC_ISIZE
        $SQL_SELECT_FIELDS_FOR_EXTRACT_SUM
        $SQL_MONEY_COND
        $SQL_ARCHIVED_UAC_BANNERS_COND
        $SQL_INTERNAL_CAMPAIGN_TYPES
        $SQL_NOT_INTERNAL_CAMPAIGN_TYPES
        /],
    limits => [qw/
        $PRICES_LIMIT
        $CAMPS_LIMIT
        $BANNERS_LIMIT
        $CONTEXTS_LIMIT
        $BIDS_LIMIT
        $MAX_ROWS_FOR_SNAPSHOT
    /]
    );
our @EXPORT_OK = (
    (map {@$_} values %EXPORT_TAGS),
    'get_value_for_sum',
    'is_strategy_roi_or_crr',
    'is_strategy_roi',
    'is_strategy_crr',
    'is_strategy_cpa',
    'is_strategy_cpa_per_filter',
    'is_strategy_cpc',
    'is_strategy_cpm',
    'is_strategy_avg_cpi',
    'is_strategy_autobudget',
    'get_target_type',
    'get_bs_expression_for_modifier',
    'get_strategy_mobile_goals',
    'convert_template_variables_to_bs_format'
    );

our @EXPORT = qw/
    $PRICES_LIMIT
    $CAMPS_LIMIT
    $BANNERS_LIMIT
    $CONTEXTS_LIMIT
    $BIDS_LIMIT
    $MAX_ROWS_FOR_SNAPSHOT
    $BS_MOON_REGION_ID
    @BIDS_BASE_ENABLED_TYPES
    $SQL_BIDS_BASE_TYPE_ENABLED
    @RELEVANCE_MATCH_BIDS_BASE_TYPES
/;

use utf8;


# NB! еще значения переопределены явно в bsClientData для buggy-потоков.
# при изменении их тоже стоит пересмотреть
our $PRICES_LIMIT = 40_000;
our $CAMPS_LIMIT = 800;
our $BANNERS_LIMIT = 2_500;
our $CONTEXTS_LIMIT = 450;
our $BIDS_LIMIT = 7_500;

# сколько максимум строк данных можем выбрать за одну итерацию
our $MAX_ROWS_FOR_SNAPSHOT = 15_000;

# id региона "Луна" в БК (используется в связке с минус-регионами, в случае когда в геотаргетингом нужно сказать "баннер не показывать ни в одном регионе")
our $BS_MOON_REGION_ID = 20001;

# id цели для CPI-кампаний
our $BS_AVG_CPI_GOAL_ID = 4;

# текущая обрабатываемая (например в ExportQuery) кампания. заполняется не везде, используется для buggy_cid
our $CURRENT_CID;

my $DIRECT_TEMPLATE_RESOURCE_ID_START_VALUE = 1_000_000;

=head2 SKIP_LOCKED_WALLETS_PROP_NAME

    Имя свойства в БД, в котором хранится "выключатель" для логики
    пропуска отправки в БК кампаний с изменениями под одним общим
    счетом в разных потоках экспорта.

=cut

use constant SKIP_LOCKED_WALLETS_PROP_NAME => 'skip_camps_in_locked_wallets_in_bs_export';

=head2 BSEXPORT_WORKERS_NUM_CONTROLLED_MANUALLY

    Имя свойства в БД, в котором хранится "выключатель" автоматической балансировки
    количества std/heavy/buggy воркеров транспорта в БК. Управление осуществляется
    с помощью внутреннего инстумента в интерфейсе.

=cut

use constant BSEXPORT_WORKERS_NUM_CONTROLLED_MANUALLY => 'bsexport_workers_num_controlled_manually';

sub get_bsexport_workers_num_controlled_manually {
    my $property = Property->new(BSEXPORT_WORKERS_NUM_CONTROLLED_MANUALLY);
    my $value = $property->get();
    if ($value) {
        $property->set(1) if $value ne '1';
        return 1;
    } else {
        $property->set(0) if $value ne '0';
        return 0;
    }
}

=head2 PREPROD_LIMTEST_PROP_PREFIX

    Имя свойства в БД, в котором хранится "выключатель" для переключения
    limtest с/на препрод БК

=cut

use constant PREPROD_LIMTEST_PROP_PREFIX => "USE_BS_PREPROD_LIMTEST";

=head2 %ADDITION_TYPES

    Опции указывающие какие дополнения нужно выбирать из БД, к чему они привязаны в Директе

=cut

our %ADDITION_TYPES = (
    callout => {get_by => 'banner_id',
                get_using_model_opts => {get_banner_id => 1, status_moderate => 'Yes'}},
    image_ad => {get_by => 'banner_id',
                 get_using_model_opts => {get_banner_id => 1, status_moderate => 'Yes'}},
    canvas => {get_by => 'banner_id',
               get_using_model_opts => {get_banner_id => 1, status_moderate => 'Yes'}},
    html5_creative => {get_by => 'banner_id',
               get_using_model_opts => {get_banner_id => 1, status_moderate => 'Yes'}},
    auto_video => {get_by => 'banner_id'},
    pixel => {get_by => 'banner_id'},
    experiment => {get_by => 'banner_id'},
    banner_price => {get_by => 'banner_id'},
    content_promotion_video => {get_by => 'banner_id'},
    content_promotion => {get_by => 'banner_id'},
    );

=head2 %ADDITION_TARGET_TO_SNAPSHOT_COLUMN

    Соответствие объектов к которым привязываются дополнения, и наименований колонок в снапшоте

=cut

our %ADDITION_TARGET_TO_SNAPSHOT_COLUMN = (
    banner_id => 'banner_pId',
    pid => 'context_pId',
    cid => 'campaign_pId'
    );

# хост logbroker'а для заливки логов экспорта в БК
# из https://wiki.yandex-team.ru/statbox/push-client/#postavkavlogbroker
our $LOGBROKER_HOST //= 'bs-prod.logbroker.yandex.net';
our $LOGBROKER_TVM_ID //= 2001059;

our $LOGBROKER_IDENT //= 'direct';
our $LOGBROKER_LOG_TYPE //= 'direct-banners-log';
our $LOGBROKER_TIMEOUT //= 90;    # в секундах
our $LOGBROKER_NETWORK_TIMEOUT //= 24;    # в секундах
our $LOGBROKER_LOG_LEVEL //= 9;

=head2 process_href_params(href, camp_type)

    Преобразует пользовательский формат параметров урла к внутреннему БК. Еще есть BS::ExportMobileContent

=cut

our $TRANSLATE_PARAMS = {
    'position' => 'POS'
    , 'position_type' => 'PTYPE'
    , 'source' => 'SRC'
    , 'source_type' => 'STYPE'
    , 'addphrases' => 'BM'
    , 'param1' => 'PARAM1'
    , 'param2' => 'PARAM2'
    , 'phraseid' => 'PHRASE_EXPORT_ID'
    , 'phrase_id' => 'PHRASE_EXPORT_ID'
    , 'adtarget_id' => 'PHRASE_EXPORT_ID'
    , 'retargeting_id' => 'PARAM126'
    , 'adtarget_name' => 'PARAM126'
    , 'interest_id' => 'PARAM125'
    , 'keyword' => 'PHRASE'
    , 'gbid' => 'GBID'
    , 'device_type' => 'DEVICE_TYPE'
    , 'logid' => 'LOGID'
    , 'trackid' => 'TRACKID'
    , 'android_id' => 'ANDROID_ID_LC'
    , 'androidid' => 'ANDROID_ID_LC'
    , 'android_id_lc_sh1' => 'ANDROID_ID_LC_SH1_HEX'
    , 'google_aid' => 'GOOGLE_AID_LC'
    , 'googleaid' => 'GOOGLE_AID_LC'
    , 'google_aid_lc_sh1' => 'GOOGLE_AID_LC_SH1_HEX'
    # https://st.yandex-team.ru/DIRECT-79877#1529353684000
    , 'ios_ifa' => 'IDFA_UC'
    , 'iosifa' => 'IDFA_UC'
    , 'idfa_lc_sh1' => 'IDFA_UC_SH1_HEX'
    , 'idfa_lc_sh1_hex' => 'IDFA_UC_SH1_HEX',
    , 'idfa_lc' => 'IDFA_UC',
    , 'idfa_lc_md5' => 'IDFA_UC_MD5_HEX',
    , 'idfa_lc_md5_hex' => 'IDFA_UC_MD5_HEX'
    # / https://st.yandex-team.ru/DIRECT-79877#1529353684000
    , 'region_id' => 'REG_BS'
    , 'region_name' => 'REGN_BS'
    , 'addphrasestext' => 'PHRASE_BM'
    , 'smartbanner_id' => 'CREATIVE_ID'
    , 'offer_id' => 'OFFER_ID'
    , 'coef_goal_context_id' => 'COEF_GOAL_CONTEXT_ID'
    , 'creative_id' => 'CREATIVE_ID'
    , 'match_type' => 'MATCH_TYPE'
    , 'matched_keyword' => 'PHRASE_RKW'
    , 'oaid' => 'OAID_LC'
    , 'oaid_lc' => 'OAID_LC'
    , 'oaid_lc_sh1' => 'OAID_LC_SH1_HEX'
    , 'oaid_lc_sh1_hex' => 'OAID_LC_SH1_HEX',
    , 'oaid_lc_md5' => 'OAID_LC_MD5_HEX',
    , 'oaid_lc_md5_hex' => 'OAID_LC_MD5_HEX'
    , 'client_ip' => 'CLIENTIP'
    , 'user_agent' => 'USER_AGENT'
    , 'device_lang' => 'DEVICE_LANG'
    , 'current_unixtime' => 'CURRENT_UNIXTIME'
    , 'yclid' => 'YCLID'
};

our $TRANSLATE_PARAMS_GENERATED_BANNERS = {
    %$TRANSLATE_PARAMS,
    bannerid => 'PEID',
    adid => 'PEID',
    banner_id => 'PEID',
    ad_id => 'PEID',
};

# копия этой мапы есть в java: AdGroupShowConditionAdditionalTargetingHandler.kt https://nda.ya.ru/t/JqKstsXb45gVoY
# если правите тут, то не забудьте поправить и в java
# для внутренней рекламы добавлять новые таргетинги достаточно только в java.
our $DIRECT_TO_BS_TARGETING_FORMAT = {
    yandexuids => {
        bs_name => 'uniq-id',
        targeting_operation => 'like',
        filtering_operation => 'not like',
        type => 'list'
    },
    device_names => {
        bs_name => 'uatraits-device-name',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    interface_langs => {
        bs_name => 'lang',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    desktop_installed_apps => {
        bs_name => 'installed-yasoft',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list'
    },
    query_referers => {
        bs_name => 'referer',
        targeting_operation => 'like',
        filtering_operation => 'not like',
        type => 'list'
    },
    user_agent => {
        bs_name => 'user-agent',
        targeting_operation => 'like',
        filtering_operation => 'not like',
        type => 'list'
    },
    show_dates => {
        bs_name => 'timestamp',
        targeting_operation => 'like',
        filtering_operation => 'not like',
        type => 'list'
    },
    query_options => {
        bs_name => 'options',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    clids => {
        bs_name => 'stat-id',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list'
    },
    clid_types => {
        bs_name => 'clid-type',
        targeting_operation => 'match',
        filtering_operation => 'not match',
        type => 'list'
    },
    test_ids => {
        bs_name => 'test-ids',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list'
    },
    ys_cookies => {
        bs_name => 'cookie-ys',
        targeting_operation => 'like',
        filtering_operation => 'not like',
        type => 'list'
    },
    yp_cookies => {
        bs_name => 'cookie-yp',
        targeting_operation => 'like',
        filtering_operation => 'not like',
        type => 'list'
    },
    internal_network => {
        bs_name => 'network-id',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'boolean',
        targeting_value => '2',
        filtering_value => '2'
    },
    is_mobile => {
        bs_name => 'device-is-mobile',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    has_l_cookie => {
        bs_name => 'cookie-l',
        targeting_operation => 'not equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '0',
        filtering_value => '0'
    },
    has_passport_id => {
        bs_name => 'passport-id',
        targeting_operation => 'greater',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '0',
        filtering_value => '0'
    },
    is_pp_logged_in => {
        bs_name => 'authorized',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    is_tablet => {
        bs_name => 'device-is-tablet',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    is_touch => {
        bs_name => 'device-is-touch',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    is_virused => {
        bs_name => 'virus-mark',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    is_yandex_plus => {
        bs_name => 'yandex-plus-enabled',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    is_default_yandex_search => {
        bs_name => 'search-antitargeting',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    is_tv => {
        bs_name => 'device-is-tv',
        targeting_operation => 'equal',
        filtering_operation => 'equal',
        type => 'boolean',
        targeting_value => '1',
        filtering_value => '0'
    },
    visit_goals => {
        bs_name => 'visit-goal',
        targeting_operation => 'match',
        filtering_operation => 'not match',
        type => 'list'
    },
    auditorium_geosegments => {
        bs_name => 'auditorium-geosegments',
        targeting_operation => 'match',
        filtering_operation => 'not match',
        type => 'list'
    },
    sids => {
        bs_name => 'sids',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list'
    },
    device_id => {
        bs_name => 'device-id',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    uuid => {
        bs_name => 'uuid',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    plus_user_segments => {
        bs_name => 'plus-user-segments',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list'
    },
    search_text => {
        bs_name => 'search-text',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    features_in_pp => {
        bs_name => 'enabled_features',
        targeting_operation => 'icase match',
        filtering_operation => 'icase not match',
        type => 'list'
    },
    browser_names => {
        bs_name => 'browser-name-and-version',
        targeting_operation => 'match name and version range',
        filtering_operation => 'not match name and version range',
        type => 'list',
        list_values_mapping => \&_convert_versioned_targeting_value_to_bs_format
    },
    browser_engines => {
        bs_name => 'browser-engine-name-and-version',
        targeting_operation => 'match name and version range',
        filtering_operation => 'not match name and version range',
        type => 'list',
        list_values_mapping => \&_convert_versioned_targeting_value_to_bs_format
    },
    os_families => {
        bs_name => 'os-family-and-version',
        targeting_operation => 'match name and version range',
        filtering_operation => 'not match name and version range',
        type => 'list',
        list_values_mapping => \&_convert_versioned_targeting_value_to_bs_format
    },
    os_names => {
        bs_name => 'os-name',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list',
        list_values_mapping => \&_convert_uatraits_targeting_value_to_bs_format
    },
    device_vendors => {
        bs_name => 'device-vendor',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list',
        list_values_mapping => \&_convert_uatraits_targeting_value_to_bs_format
    },
    mobile_installed_apps => {
        bs_name => 'except-apps-on-cpi',
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list',
        list_values_mapping => \&_convert_mobile_installed_apps_targeting_value_to_bs_format
    },
    content_categories => {
        # название кейворда зависит от типа цели крипты,
        # см bs_name_by_crypta_type чуть ниже
        bs_name => undef,
        targeting_operation => 'equal',
        filtering_operation => 'not equal',
        type => 'list',
        bs_name_by_crypta_type => {
            content_genre => 'content-genre',
            content_category => 'content-category',
        }
    },    
};

Hash::Util::lock_hash_recurse(%$DIRECT_TO_BS_TARGETING_FORMAT);

=head2 %TRAFARET_POSITION_TO_BS

    Преобразует название позиции из формата базы Директа к внутреннему БК

=cut

our %TRAFARET_POSITION_TO_BS = (
    alone => 'Alone',
    suggest => 'Suggest'
);

=head3 priceprefix2bscode($prefix)

    Перевод префикса из базы в префикс БК

=cut
sub priceprefix2bscode($) {
    my ($prefix) = @_;

    my %BANNER_PRICE_PREFIX = (
        from => 1,
        to => 2
    );

    return $prefix ? $BANNER_PRICE_PREFIX{$prefix} : 0;
}

=head3 @BIDS_BASE_ENABLED_TYPES

    Допустимые типы из bids_base

=cut

our @BIDS_BASE_ENABLED_TYPES = qw/relevance_match relevance_match_search offer_retargeting/;

=head3 @RELEVANCE_MATCH_BIDS_BASE_TYPES

    Типы из bids_base, которые являются бесфразными таргетингами

=cut

our @RELEVANCE_MATCH_BIDS_BASE_TYPES = qw/relevance_match relevance_match_search/;


{
    my $translate_params_re = join "|", map {quotemeta} keys %$TRANSLATE_PARAMS;
    my $translate_params_re_generated_banners = join "|", map {quotemeta} keys %$TRANSLATE_PARAMS_GENERATED_BANNERS;

sub process_href_params($$)
{
    my ($href, $camp_type) = @_;

    my $banners_generated = camp_kind_in(type => $camp_type, 'sub_banners_generated');
    my $translate_re = ($banners_generated) ? $translate_params_re_generated_banners : $translate_params_re;
    my $translate_params_hash = ($banners_generated) ? $TRANSLATE_PARAMS_GENERATED_BANNERS : $TRANSLATE_PARAMS;

    $href =~ s/\{($translate_re)\}/\{$translate_params_hash->{lc($1)}\}/ig;

    return $href;
}
}

our @SUBSTITUTE_PARAMS = qw(
    adgroupid adgroup_id
    adid ad_id
    bannerid banner_id
    campaignid campaign_id
    campaigntype campaign_type
    campaignname campaign_name
    campaignnamelat campaign_name_lat
    campaigncurrency campaign_currency
    campaigncurrencycode campaign_currency_code
    campaigncosttype campaign_cost_type
    campaigncost campaign_cost
);

our @SUBSTITUTE_PARAMS_GENERATED_BANNERS = qw(
    adgroupid adgroup_id
    campaignid campaign_id
    campaigntype campaign_type
    campaignname campaign_name
    campaignnamelat campaign_name_lat
    campaigncurrency campaign_currency
    campaigncurrencycode campaign_currency_code
    campaigncosttype campaign_cost_type
    campaigncost campaign_cost
);

=head2 normalize_href_param_name

    По пользовательскому имени подставляемого параметра (adgroup_id/adgroupid/...)
    возвращет имя соответствующего ключа в хеше из BS::ExportQuery::_get_href_params_for_substitute.

=cut

sub normalize_href_param_name($) {
    my $string = lc shift;
    $string =~ s/\_//g;
    return $string;
}

{
    my $substitute_href_params_re = join "|", map {quotemeta} @SUBSTITUTE_PARAMS;
    my $substitute_href_params_re_generated_banners = join "|", map {quotemeta} @SUBSTITUTE_PARAMS_GENERATED_BANNERS;

=head2 substitute_href_params(href, data, camp_type)

    Преобразует пользовательский формат параметров урла к внутреннему БК

=cut

sub substitute_href_params($$$)
{
    my ($href, $data, $camp_type) = @_;

    my $banners_generated = camp_kind_in(type => $camp_type, 'sub_banners_generated');
    my $substitute_re = ($banners_generated) ? $substitute_href_params_re_generated_banners : $substitute_href_params_re;
    $href =~ s/\{($substitute_re)\}/$data->{normalize_href_param_name($1)}/ig;

    return $href;
}
}

=head2  merge_href_with_template_params

    Объединяет ссылку и дополнительные параметры из шаблона.

    Логика повторяет аналогичную в java: пересекающиеся параметры берутся из шаблона, новые параметры добавляются в ссылку.
    Отличается наличием неявного escape и unescape внутри метода, а также порядком параметр в итоговой ссылке:
    сперва идут параметры с теми ключами, что есть и в ссылке, и в шаблоне; потом идут остальные параметры из ссылки; потом остальные из шаблона.

    https://a.yandex-team.ru/arcadia/direct/core/src/main/java/ru/yandex/direct/core/entity/hrefparams/service/HrefWithParamsBuildingService.java

=cut

sub merge_href_with_template_params {
    my ($href, $href_params) = @_;

    my $origin_href = URI->new($href);
    my $template_href = URI->new();
    $template_href->query($href_params);

    for my $key ($template_href->query_param) {
        # query_param подменяет существующие ключи и добавляет новые
        $origin_href->query_param($key, $template_href->query_param($key));
    }

    # URI по дефолту эскейпит (в частности, скобки), поэтому возвращаем к оригиналу
    return uri_unescape($origin_href->as_iri);
}

=head2 get_query_data

Получить или создать в БК-шном запросе нужную подструктуру
get_query_data($object, 'NAME', $ids_array, $ids_names)

=cut

sub get_query_data
{
    my ($hash, $type, $ids, $id_names) = @_;
    die "invalid use of get_query_data: ids and id_names must be same size!" unless @$ids == @$id_names;
    my $key = join '_', map { $_||0 } (substr $type, 0,1), @$ids;
    return $hash->{$type}->{$key} ||= {
        pairwise {our($a, $b); defined $b ? ($a => $b) : ()} @$id_names, @$ids
    };
}


my $SQL_WHERE_BANNER_ACTIVE = qq{
        b.statusShow = 'Yes' and b.statusPostModerate != 'Rejected' and b.statusArch = 'No'
        and (
            b.banner_type NOT IN
                ('image_ad', 'cpc_video', 'mcbanner', 'cpm_banner', 'cpm_outdoor', 'cpm_indoor', 'cpm_audio')
            OR IF(images.bid is not null, images.statusModerate, b_perf.statusModerate) = 'Yes'
        )
};

my $SQL_WHERE_BANNER_HAS_SOMETHING_TO_SHOW = qq{
    IFNULL(b.href,'') != ''
    OR (btl.tl_id IS NOT NULL AND btl.statusModerate = 'Yes')
    OR b.banner_type IN ('dynamic', 'mobile_content', 'performance', 'performance_main', 'cpm_outdoor', 'internal', 'cpm_indoor')
    OR (b.banner_type IN ('image_ad', 'cpc_video') AND c.type = 'mobile_content')
    OR (b.phoneflag = 'Yes' AND b.vcard_id IS NOT NULL)
    OR (bpml.permalink_assign_type = 'manual' AND bpml.prefer_vcard_over_permalink = 0 AND (org.status_publish = 'published' OR bpml.is_sent_to_bs = 1 OR bpml.chain_id IS NOT NULL))
};

my $SQL_BANNER_POST_MODERATE_COND = "b.statusPostModerate = 'Yes' AND (b.sitelinks_set_id IS NULL
                                                                       OR b.sitelinks_set_id = 0
                                                                       OR b.statusSitelinksModerate in ('Yes', 'No')
                                                                       OR b.statusShow = 'No')
                                     OR
                                     (
                                        b.statusPostModerate = 'Rejected'
                                        OR perf_cr.statusModerate = 'AdminReject'
                                     ) AND IFNULL(b.BannerID, 0) > 0";
my $SQL_PHRASES_POST_MODERATE_COND = "p.statusPostModerate = 'Yes' OR p.statusPostModerate = 'Rejected' AND IFNULL(p.PriorityID, 0) > 0";

our $SQL_BIDS_BASE_TYPE_ENABLED = qq{bi_b.bid_type IN (} . join(',', map {"'$_'"} @BIDS_BASE_ENABLED_TYPES) . qq{)};

# sql для вычисления "интегрального" размера очереди
our $SQL_CALC_ISIZE = "camps_num*20 + banners_num*10 + contexts_num*10 + bids_num + prices_num*0.1";

Readonly our $SQL_INTERNAL_CAMPAIGN_TYPES => "c.type IN ('internal_autobudget','internal_free','internal_distrib')";
Readonly our $SQL_INTERNAL_CAMPAIGN_TYPES_WITHOUT_MONEY => "c.type IN ('internal_free','internal_distrib')";
Readonly our $SQL_NOT_INTERNAL_CAMPAIGN_TYPES => "c.type NOT IN ('internal_autobudget','internal_free','internal_distrib')";

# либо на кампании есть деньги, либо кампания неархивная, уже была в БК и текущий баннер тоже уже был в БК
# Наличие денег на кампании под ОС определяется не очень честно, т.к. честно проверить - сложно.
# В итоге условие может срабатывать в случаях, когда деньги были, но все потрачены.
our $SQL_MONEY_COND = "((c.sum + IFNULL(wc.sum, 0) - (c.sum_spent + IFNULL(wc.sum_spent, 0))) > $Currencies::EPSILON
                        OR clo.auto_overdraft_lim > 0
                        OR c.archived = 'No' AND c.OrderID > 0 AND b.BannerID > 0
                        OR c.archived = 'No' AND $SQL_INTERNAL_CAMPAIGN_TYPES_WITHOUT_MONEY
                       )";

# Условие для отбора не архивных uac баннеров, которые уже отправляли ранее
our $SQL_ARCHIVED_UAC_BANNERS_COND = "((c.source != 'uac' AND c.source != 'widget')
                                        OR b.statusArch = 'No'
                                        OR b.statusBsSynced != 'Yes'
                                        )";

# JAVA-EXPORT: вынесено в код и реализовано в виде методов:
# PreFilterCampaignsStep#wasInBsOrAcceptedOnModeration
# PreFilterCampaignsStep#isNotEmpty
our $SQL_WHERE_CAMP_COMMON = q/(c.OrderID > 0 OR c.statusModerate = 'Yes') AND c.statusEmpty = 'No'/;

# кусок условий, влияющий на неотправку кампаний с ОС без OrderID
our $NO_WALLET_OR_WALLET_HAS_ORDERID = q/c.wallet_cid > 0 AND IFNULL(wc.OrderID, 0) > 0 OR c.wallet_cid = 0/;

# типовые par_type которые потоки не должны брать. используется для camp и full_lb_export
Readonly::Array our @COMMON_EXCLUDE_PAR_TYPES => ('dev1', 'dev2', 'nosend', 'buggy', 'internal_ads_dev1', 'internal_ads_dev2');

# Описание правил перекладывания универсальных корректировок, задаваемых формулами
# Тип корректировки => [название проперти в Директе, название проперти в БК]
# Например, для express_traffic_multiplier это означает, что в multipliers_dict данные по этой
# корректировке будут храниться в поле 'expression_traffic', а в БК оно поедет в поле с названием 'TrafficJam'
# Значение direct_property не должно пересекаться с ключами других корректировок в словаре $ExportQuery::MULTIPLIERS
Readonly::Hash our %EXPRESSION_MULTIPLIER_TYPES => (
    'express_traffic_multiplier' => {
        direct_property => 'expression_traffic',
        bs_property     => 'TrafficJam'
    },
    # Когда будем переносить старые погодные
    #'express_weather_multiplier' => {
    #    direct_property => 'expression_weather',
    #    bs_property     => 'Weather'
    #},
);

=head2 get_sql_from_stat($camps_only)

    Вовзращает FROM часть SQL запроса для получения объектов, нуждающихся
    в синхронизации с БК.
    Подразумевается, что она будет джойниться с таблицей campaigns c

    Если передать в параметре $camps_only правдивое значение,
    джойн будет поправлен так, как будто у кампании нет групп.
    Это может быть полезно, чтобы отобрать только кампании, нуждающиеся в
    сихнронизации.

=cut

sub get_sql_from_stat {
    my ($camps_only) = @_;

    my $phrases_join_cond;
    if ($camps_only) {
        $phrases_join_cond = "p.cid = NULL";
    } else {
        $phrases_join_cond = "p.cid = c.cid AND u.statusBlocked = 'No'";
    }

    return qq[
        left join campaigns wc on wc.cid = c.wallet_cid
        left join camp_options co on co.cid = c.cid
        left join clients_options clo on clo.ClientID = c.ClientID
        join users u on u.uid = c.uid
        left join phrases p on $phrases_join_cond
        left join banners b on b.pid = p.pid
                                 and c.archived = 'No'
                                 and $SQL_MONEY_COND
                                 and $SQL_ARCHIVED_UAC_BANNERS_COND
        left join banner_images bim on b.bid = bim.bid
        left join images on images.bid = b.bid
        left join banners_performance b_perf on b_perf.bid = b.bid
        left join perf_creatives perf_cr on perf_cr.creative_id = b_perf.creative_id
        left join banner_turbolandings btl on b.bid = btl.bid
        left join banner_permalinks bpml on b.bid = bpml.bid
        left join organizations org on bpml.permalink = org.permalink_id
                                 and org.ClientID = c.ClientID
    ];
}

=head2 get_sql_where_exclude_special_par_types

    Получить часть условия для where, исключающую кампании с s.par_type из списка @COMMON_EXCLUDE_PAR_TYPES

    Результат:
        $sql_where_exclude_special_par_types - строка (без обрамляющих скобок)

=cut

sub get_sql_where_exclude_special_par_types {
    my $params = shift;
    my @exclude_par_types = (@COMMON_EXCLUDE_PAR_TYPES, @{$params->{additional_par_types} // []});
    return sql_condition(
        [
            's.par_type__is_null' => 1,
            's.par_type__not_in' => \@exclude_par_types,
        ],
        operation => 'OR',
    );
}

=head2 get_sql_where_data

    Получить кусок sql-условия для where, проверяющий что какие-либо данные нужно отправить в БК
    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where_data - строка (без обрамляющих скобок)

=cut

sub get_sql_where_data {
    my $ignore_status_bs_synced = shift;
    if ($ignore_status_bs_synced) {
        return '1';
    } else {
        return q/c.statusBsSynced = 'Sending' OR b.statusBsSynced = 'Sending' OR p.statusBsSynced = 'Sending'/;
    }
}

=head2 get_sql_where_camp_unsync

    Получить кусок sql-условия для where, проверяющий что кампания несинхронна с БК
    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where_camp_unsync - строка (без обрамляющих скобок)

=cut

sub get_sql_where_camp_unsync {
    my $full_export = shift;
    if ($full_export) {
        return '1';
    } else {
        return q/c.statusBsSynced = 'Sending'/;
    }
}

=head2 get_sql_where_banner_unsync

    Получить кусок sql-условия для where, проверяющий что есть какие-либо данные по баннерам
    для отправки в БК и их можно отправлять

    NB: чтобы не усложнять запрос - перловое условие внутри BS::ExportQuery::get_query - строже
    (для графических объявлений с BannerID), подробнее в DIRECT-58058
    Также немного ослаблено условие на наличие промодерированной визитки с телефоном. Здесь проверяется только
    наличие промодерированной визитки, а телефон проверяется в BS::ExportQuery::get_query

    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where_banner_unsync - строка (без обрамляющих скобок)

=cut

sub get_sql_where_banner_unsync {
    my $ignore_status_bs_synced = shift;

    my ($b_statusBsSynced_cond, $p_statusBsSynced_cond);
    if ($ignore_status_bs_synced) {
        ($b_statusBsSynced_cond, $p_statusBsSynced_cond) = ('1', '1');
    } else {
        $b_statusBsSynced_cond = q/b.statusBsSynced = 'Sending'/;
        $p_statusBsSynced_cond = q/p.statusBsSynced = 'Sending'/;
    }

    return qq{
        $b_statusBsSynced_cond
        AND (
            ($SQL_BANNER_POST_MODERATE_COND)
            OR b.statusShow = 'No'
        )
        AND (
            b.BannerID > 0
            OR (
                ($SQL_WHERE_BANNER_ACTIVE)
                AND ($SQL_PHRASES_POST_MODERATE_COND)
            )
            AND ($p_statusBsSynced_cond OR p.PriorityID > 0)
        )
        AND (
            ($SQL_WHERE_BANNER_HAS_SOMETHING_TO_SHOW)
            OR b.statusShow = 'No'
            OR b.statusPostModerate = 'Rejected'
            OR (b.BannerID > 0 AND btl.tl_id IS NOT NULL AND btl.statusModerate = 'No')
        )
    };
}

=head2 get_sql_where_context_or_bid_unsync

    Получить кусок sql-условия для where, проверяющий что есть какие-либо данные по группе или баннеру
    для отправки в БК и их можно отправлять

    NB: чтобы не усложнять запрос - перловое условие внутри BS::ExportQuery::get_query - строже
    (для графических объявлений с BannerID), подробнее в DIRECT-58058

    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where_context_or_bid_unsync - строка (без обрамляющих скобок)

=cut

sub get_sql_where_context_or_bid_unsync {
    my $ignore_status_bs_synced = shift;

    my ($b_statusBsSynced_cond, $p_statusBsSynced_cond);
    if ($ignore_status_bs_synced) {
        ($b_statusBsSynced_cond, $p_statusBsSynced_cond) = ('1', '1');
    } else {
        $b_statusBsSynced_cond = q/b.statusBsSynced = 'Sending'/;
        $p_statusBsSynced_cond = q/p.statusBsSynced = 'Sending'/;
    }

    return qq{
        ($SQL_PHRASES_POST_MODERATE_COND)
        AND $p_statusBsSynced_cond
        AND (
            b.BannerID > 0
            OR (
                ($SQL_WHERE_BANNER_ACTIVE)
                AND ($SQL_BANNER_POST_MODERATE_COND)
            )
            AND (
                ($SQL_WHERE_BANNER_HAS_SOMETHING_TO_SHOW)
                OR b.statusShow = 'No'
            )
            AND $b_statusBsSynced_cond
        )
    };
}

=head2 get_sql_where_unsync

    TODO: Написать что-то осмысленное
    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where_unsync - строка (без обрамляющих скобок)

=cut

sub get_sql_where_unsync {
    my $ignore_status_bs_synced = shift;

    my $sql_where_camp_unsync = get_sql_where_camp_unsync($ignore_status_bs_synced);
    my $sql_where_banner_unsync = get_sql_where_banner_unsync($ignore_status_bs_synced);
    my $sql_where_context_or_bid_unsync = get_sql_where_context_or_bid_unsync($ignore_status_bs_synced);

    # JAVA-EXPORT: вынесено в код и частично реализовано как PreFilterCampaignsStep#hasMoneyOrNeedSync
    return qq{
        (
            $SQL_MONEY_COND AND co.statusPostModerate = 'Accepted'
            OR ($sql_where_camp_unsync AND (c.OrderID > 0 OR c.type = 'wallet'))
        )
        AND (
            $sql_where_camp_unsync
            OR $sql_where_banner_unsync
            OR $sql_where_context_or_bid_unsync
        )
    };
}

=head2 get_sql_where

    Получить кусок sql-условия для where, проверяющий что есть какие-либо данные для отправки в БК, и их можно отпрвлять.
    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where - строка (без обрамляющих скобок)

=cut

sub get_sql_where {
    my $ignore_status_bs_synced = shift;

    my $sql_where_unsync = get_sql_where_unsync($ignore_status_bs_synced);
    return qq/($SQL_WHERE_CAMP_COMMON) AND ($sql_where_unsync)/;
}

=head2 get_sql_where_banner_should_sync_minus_geo

    Получить кусок sql-условия для where, проверяющий что баннер показывается, или будет показываться в БК,
    и минус-гео по нему нужно учесть в регионах показа группы.
    Из "старых" баннеров в выборку должны попадать только те, которые мы считаем что крутятся в БК (предыдущая версия)

    Параметры:
        $ignore_status_bs_synced - 1 - игнорировать статусы синхронизации с БК (по умолчанию - 0, учитываются)
    Результат:
        $sql_where_banner_unsync - строка (без обрамляющих скобок)

=cut

sub get_sql_where_banner_should_sync_minus_geo {
    my $ignore_status_bs_synced = shift;

    my $b_statusBsSynced_cond;
    if ($ignore_status_bs_synced) {
        $b_statusBsSynced_cond = '1';
    } else {
        $b_statusBsSynced_cond = q/b.statusBsSynced = 'Sending'/;
    }

    return qq{
        ($SQL_WHERE_BANNER_ACTIVE)
        AND (
            (b.BannerID > 0
                AND p.statusPostModerate <> 'Rejected')
            OR (p.statusPostModerate = 'Yes'
                AND b.statusPostModerate = 'Yes'
                AND ($SQL_WHERE_BANNER_HAS_SOMETHING_TO_SHOW)
                AND $b_statusBsSynced_cond
            )
        )
    };
}

my $convertable_types_sql = join(',', map { "'$_'" } @{ get_camp_kind_types('with_currency') });

=head2 $SQL_SELECT_FIELDS_FOR_EXTRACT_SUM

    Кусок SQL - поля, относящиеся к сумме заказа и ее валюте, и дате перехода на валюту.
    SQL используется в запросах для UpdateData2 и set-order-sum
    Выбранные поля используются в extract_sum.

=cut
our $SQL_SELECT_FIELDS_FOR_EXTRACT_SUM = qq{
    , c.sum
    , IFNULL(c.currency, 'YND_FIXED') AS currency
    , IF(ccq.convert_type = 'MODIFY' AND c.type IN ($convertable_types_sql), DATE_FORMAT(ccc.date, '%Y%m%d'), NULL) AS currency_change_date
    , IF(ccq.convert_type = 'MODIFY' AND c.type IN ($convertable_types_sql), ccc.currency_to, NULL) AS future_currency
    , cms.chips_cost
    , cms.chips_spent
    , wwc.is_sum_aggregated
    , wwc.total_chips_cost
    , IFNULL(clo.debt, 0) AS client_debt
    , IFNULL(clo.overdraft_lim, 0) AS overdraft_lim
    , IFNULL(clo.auto_overdraft_lim, 0) AS auto_overdraft_lim
    , IFNULL(clo.statusBalanceBanned, 'No') AS statusBalanceBanned
};

=head2 currency2bsisocode

    Возвращает ISO код для валюты, пригодный для отправки в БК.
    -1 для у.е.
    643 для рубля

    my $iso_code = currency2bsisocode('RUB');
    $iso_code => 643

=cut
sub currency2bsisocode {
    my ($currency) = @_;

    die 'no currency given' unless $currency;

    return ($currency eq 'YND_FIXED') ? -1 : get_currency_constant($currency, 'ISO_NUM_CODE');
}

=head2 extract_sum

    extract_sum($order, $row, $c_type, $client_info);

    На основании сырых данных ($row):
        currency        - валюта заказа
        sum             - остаток на счете
        currency_change_date - дата перехода заказа на мультивалютность
        future_currency - валюта, в которую переводится заказ
        chips_spent     - количество потраченных на кампании фишек до перехода на мультивалютность
        chips_cost      - стоимость в деньгах потраченных на кампании фишек
        auto_overdraft_lim - порог отключения в деньгах (>0, если у клиента подключён автоовердрафт)
        is_sum_aggregated - перешел ли общий счет в новую схему учета зачислений.
                            В этом случае стоимость всех потраченных фишек под ОС суммируется в поле total_chips_cost
                            общего счета, а поле chips_cost дочерних кампаний перестает использоваться.
        total_chips_cost - суммарная стоимость потраченных фишек под общим счетом,
                            имеет смысл только если is_sum_aggregated eq 'Yes'
        wallet_sum_debt      - сумма расходов на кошельке (должно присутствовать, если у клиента есть auto_overdraft_lim > 0)

    Заполняет поля заказа ($order), относящиеся к сумме и ее валюте:
        SUM             - сумма в фишках
        SUMCur          - сумма в валюте
        CurrencyISOCode - ISO код валюты
        CurrencyConvertDate - дата перехода заказа на валюту

    Для клиентов с включённым автоовердрафтом (порогом отключения) к SUMCur добавляется значение auto_overdraft_lim
    (если включена соответствующая опция, а к кошельку применимы автоовердрафты)

    $c_type             - тип кампании
    $client_info        - информация о клиенте (для учёта автоовердрафта)

    $client_info = {
        clientID,            # идентификатор клиента
        debt,                # текущий долг клиента (размер выставленного, но ещё не оплаченного счёта)
        overdraft_lim,       # 0 или лимит овердрафта
        auto_overdraft_lim,  # 0 или порог отключения, заданный пользователем
        statusBalanceBanned  # "забанен" ли клиент в Балансе ('Yes' или 'No')
    }

=cut
sub extract_sum {
    my ($order, $row, $c_type, $client_info) = @_;

    my $currency_after_convert = $row->{future_currency} || $row->{currency};

    if ($row->{currency} eq 'YND_FIXED') {
        $order->{SUM} = $row->{sum};    # остаток на счёте в фишках
        $order->{SUMCur} = 0;
    } else {
        $order->{SUM} = $row->{chips_spent} // 0;

        my $chips_cost;
        if (($row->{is_sum_aggregated} // 'No') eq 'Yes') {
            if (Campaign::Types::is_wallet_camp(type => $c_type)) {
                $chips_cost = $row->{total_chips_cost};
            } else {
                # у дочерних кампаний все потраченные фишки суммируются на общем счету
                $chips_cost = 0;
            }
        } else {
            $chips_cost = ($row->{chips_cost} // 0);
        }

        my $auto_overdraft_addition = 0;

        # Учитываем порог отключения в логике расчёта SUMCur
        my $wallet_info = {
            type            => $c_type,
            currency        => $row->{currency},
            sum             => $row->{sum},
            wallet_sum_debt => $row->{wallet_sum_debt}
        };
        $auto_overdraft_addition = WalletUtils::get_auto_overdraft_addition($wallet_info, $client_info);

        $order->{SUMCur} = sprintf('%.6f', $row->{sum} + $auto_overdraft_addition - $chips_cost);
    }

    # отправляем будущую валюту кампании как только она стала нам известна
    $order->{CurrencyISOCode} = currency2bsisocode($currency_after_convert);

    # дату перевода отправляем только для кампаний в валюте или собирающихся переводиться
    if ($order->{CurrencyISOCode} != -1) {
        # дата перехода на валюту (первый день, когда заказ работает в реальной валюте)
        # либо для сразу мультивалютных (фейковая старая дата)
        $order->{CurrencyConvertDate} = $row->{currency_change_date} || $Client::BEGIN_OF_TIME_FOR_MULTICURRENCY_CLIENT;
    }

    return 1;
}

=head2 Получение структур для формирования запроса к БК
=head3 _get_empty_query_struct

    Получить базовую структуру для пустого запроса к БК.

    Параметры:
        $par_type
    Результат:
        $query  - hashref с базовым набором полей для запроса.

=cut
sub _get_empty_query_struct {
    my $par_type = shift;
    my $is_internal_ads = (any {$par_type eq $_} qw (internal_ads internal_ads_dev1 internal_ads_dev2)) ? 1 : 0;
    return {
        RequestUUID => HashingTools::generate_uuid(),
        EngineID => $is_internal_ads ? $Settings::SERVICEID{banana} : $Settings::SERVICEID{direct},
    };
}

=head2 get_empty_query_struct_for_UpdateData2

    Получить пустую (без данных) структуру для метода UpdateData2
    Параметры:
        $par_type
    Результат:
        $query  - hashref с базовым набором полей для запроса.

=cut
sub get_empty_query_struct_for_UpdateData2 {
    my $par_type = shift;
    my $query = _get_empty_query_struct($par_type);
    $query->{ORDER} = {};
    return $query;
}

=head2 get_request_uuid($query)

    Получить UUID текущего запроса ($query)

    Параметры:
        $query  - hashref сформированного запроса к БК
    Результат:
        $uuid   - строка с UUID запроса

=cut
sub get_request_uuid {
    my $query = shift;
    return $query->{RequestUUID};
}

=head2 %MOVE_CIDS_TO_BUGGY

    хеш, ключами которого записаны cid'ы, которые нужно переставить в buggy-очередь

=cut

our %MOVE_CIDS_TO_BUGGY;

=head2 buggy_cid($cid)

    Пометить кампанию как buggy, чтобы в конце итерации переставить в buggy-очередь

    Параметры:
        $cid    - номер кампании

=cut

sub buggy_cid {
    my $cid = shift;
    $MOVE_CIDS_TO_BUGGY{$cid} = undef;
    delay_cid($cid);    # чтобы buggy-очередь не переставала "ротироваться" на проблемных кампаниях
}

=head2 %CIDS_FOR_DELAY

    Хеш, ключами которого записаны cid'ы, которые нужно "задержать" (по seq_time) в очереди

=cut

our %CIDS_FOR_DELAY;

=head2 delay_cid($cid)

    Пометить кампанию как требующую задержки в очереди.

    Параметры:
        $cid - номер кампании

=cut

sub delay_cid {
    my $cid = shift;
    $CIDS_FOR_DELAY{$cid} = undef;
}

=head2 $ITERATION_FAILED

    признак неуспешности итерации. используется при принятии решения
    о переселении кампаний в buggy-очередь

=cut

our $ITERATION_FAILED;

=head2 get_domain_filter($domain)

    Получить фильтр-домен для отправки в БК. Это главный домен (с учётом редиректов)
        без www и в нижнем регистре. Домен конвертируется в punycode (ASCII).
    Если главным доменом получился yandex.ru или www.yandex.ru, то возращается $domain (без www, в punycode)

    Параметры:
        $domain - домен
    Результат:
        $main_domain  - фильтр-домен

=cut
sub get_domain_filter {
    my $domain = shift;
    state $mirrors;
    $mirrors //= MirrorsTools->new(use_db => 1, dont_load_file => 1);

    my $domain_ascii = Yandex::IDN::idn_to_ascii($domain);
    my $main_domain = $mirrors->domain_filter($domain_ascii);
    if ($main_domain) {
        $main_domain = strip_www($main_domain);

        if ($main_domain eq 'yandex.ru' || $main_domain eq 'www.yandex.ru') {
            # для Яндекса можно показывать параллельно несколько объявлений, т.к. проекты более-менее отдельные
            # при этом после склейки доменов может получиться, что не-яндексные домены являются зеркалами к яндексовым
            # такие надо по-прежнему фильтровать, поэтому подставляем исходный домен для тех доменов, где главным получается yande.ru
            $main_domain = lc($domain_ascii);
        }
    }
    return $main_domain;
}

=head2 get_domain_bs_id

    Получить для домена его id из БК
    Если домен отсутствует или работаетм в Песочнице API возвращаем ноль

    Параметры:
        $domain - домен
    Результат:
        $domain_bs_id  - id доменa

=cut

sub get_domain_bs_id {
    my $domain = shift;

    return 0 unless $domain;

    # В песочнице API не ходим в ручку БК
    return 0 if is_sandbox();

    my $url = BS::URL::get('domain_id');
    my $ticket = eval { Yandex::TVM2::get_ticket($Settings::YABS_ID_GENERATOR_TVM2_ID) } or die "Cannot get ticket for $Settings::YABS_ID_GENERATOR_TVM2_ID: $@";

    my $get_domain_bs_id_profile = Yandex::Trace::new_profile('bs_export:get_domain_bs_id', obj_num => 1);
    my $body = encode_json({ domains => [$domain] });
    my %opts = (
        timeout => 5,
        num_attempts => 2,
        ipv6_prefer => 1,
        headers => {
            'Content-Type' => 'application/json',
            'X-Ya-Service-Ticket' => $ticket,
        },
    );
    my $content = http_fetch('POST', $url, $body, %opts);
    my $response = decode_json($content);

    my $domain_bs_id;
    if ($response->{ids}) {
        $domain_bs_id = $response->{ids}->[0];
    } else {
        die "got malformed response: $content\n";
    }

    return $domain_bs_id;
}


=head2 target_funnel_for_BS

    Функция превращает строку из bids_performance.target_funnel в значения OnlyOfferRetargeting и OnlyNewAuditory для отправки в БК

    $hashref_for_bs = BS::Export::target_funnel_for_BS($target_funnel);
    $hashref_for_bs => {
        OnlyOfferRetargeting => 1,
        OnlyNewAuditory => 0,
    }

=cut

sub target_funnel_for_BS
{
    my $funnel_str = shift;

    my $only_offer_retargeting = {
        same_products => 0,
        product_page_visit => 1,
        new_auditory => 0,
    }->{ $funnel_str // '' } // 0;
    my $only_new_auditory = {
        same_products => 0,
        product_page_visit => 0,
        new_auditory => 1,
    }->{ $funnel_str // '' } // 0;

    return {
        OnlyOfferRetargeting => $only_offer_retargeting,
        OnlyNewAuditory => $only_new_auditory,
    };
}

=head3 _get_client_nds_with_cache

    Получает НДС клиента с использованием кеша.

    my %cache;
    $nds = _get_client_nds_with_cache($client_id, \%cache);
    $nds => 18

=cut

sub _get_client_nds_with_cache {
    my ($client_id, $client_nds_cache) = @_;

    if (!defined $client_nds_cache->{$client_id}) {
        $client_nds_cache->{$client_id} = get_client_NDS($client_id);
    }
    if (!defined $client_nds_cache->{$client_id}) {
        if ($CURRENT_CID) {
            buggy_cid($CURRENT_CID);
        }
        die "No NDS found for ClientID $client_id";
    }
    return $client_nds_cache->{$client_id};
}

=head2 get_value_for_sum

    Вспомогательная функция, которая умеет:
    - накидывать НДС [если не фишки]
    - приводить в диапазон между валютными константами

    Принимает позиционные параметры:
    - сумма
    - валюта

    И именованные:
    - with_nds => 1/0 — накидывать ли НДС [по умолчанию 0]
    - ClientID — ClientID для получения графика НДС [обязателен при with_nds => 1]
    - client_nds_cache — ссылка на хеш для хранения кеша графиков НДС [не обязательный]
    - min_constant — имя валютной константы для минимального значения [не обязательный]
    - max_constant — имя валютной константы для максимального значения [не обязательный]
    - keep_zero => 1/0 - оставлять ли ноль нолём или приводить к минимальным-максимальным константам
    - undef_is_zero => 1/0 - считать ли undef нулём

=cut

sub get_value_for_sum {
    my ($sum, $currency, %O) = @_;

    if ($O{undef_is_zero} && !defined $sum) {
        $sum = 0;
    }

    if ($O{keep_zero} && $sum == 0) {
        return 0;
    }

    my $sum_new;
    if ($O{with_nds} && $currency ne 'YND_FIXED') {
        my $client_nds = _get_client_nds_with_cache($O{ClientID}, $O{client_nds_cache});
        $sum_new = Currencies::add_nds($sum, $client_nds);
    } else {
        $sum_new = $sum;
    }

    my $opts = hash_cut \%O, qw/min_constant max_constant/;
    $sum_new = Currencies::inscribe_to_constants($sum_new, $currency, %$opts);

    return $sum_new;
}

=head2 is_strategy_roi

    Проверка является ли стратегия AUTOBUDGET_ROI

=cut

sub is_strategy_roi
{
    return shift->{strategy_name} eq 'autobudget_roi' ? 1 : 0;
}

=head2 is_strategy_crr

    Проверка является ли стратегия AUTOBUDGET_CRR

=cut

sub is_strategy_crr {
    return shift->{strategy_name} eq 'autobudget_crr' ? 1 : 0;
}

=head2 is_strategy_roi_or_crr

    Проверка является ли стратегия AUTOBUDGET_ROI или AUTOBUDGET_CRR

=cut

sub is_strategy_roi_or_crr {
    my $strategy = shift;
    return is_strategy_roi($strategy) || is_strategy_crr($strategy);
}

=head2 is_strategy_avg_cpi

    Проверка является ли стратегия AUTOBUDGET_AVG_CPI

=cut

sub is_strategy_avg_cpi {
    return (shift->{strategy_name} // '') eq 'autobudget_avg_cpi' ? 1 : 0;
}

=head2 is_strategy_autobudget

    Проверка является ли стратегия AUTOBUDGET

=cut
sub is_strategy_autobudget {
    return (shift->{strategy_name} // '') eq 'autobudget' ? 1 : 0;
}

=head2 is_strategy_cpa

=cut

sub is_strategy_cpa {
    my ($row) = @_;

    return (any { ($row->{strategy} // '') eq $_ } ('autobudget_avg_cpa_per_filter', 'autobudget_avg_cpa_per_camp'))
        || (any { ($row->{strategy_name} // '') eq $_ } ('autobudget_avg_cpa_per_filter', 'autobudget_avg_cpa_per_camp', 'autobudget_avg_cpa'));
}

=head2 is_strategy_cpa_per_filter

    Возвращает 1 если стратегия кампании -- autobudget_avg_cpa_per_filter

=cut

sub is_strategy_cpa_per_filter {
    my ($row) = @_;

    return (any { ($row->{strategy} // '') eq $_ } ('autobudget_avg_cpa_per_filter'))
        || (any { ($row->{strategy_name} // '') eq $_ } ('autobudget_avg_cpa_per_filter'));
}

=head2 is_strategy_cpc

=cut

sub is_strategy_cpc {
    my ($row) = @_;

    return (any { ($row->{strategy} // '') eq $_ } ('autobudget_avg_cpc_per_filter', 'autobudget_avg_cpc_per_camp'))
        || (any { ($row->{strategy_name} // '') eq $_ } ('autobudget_avg_cpc_per_filter', 'autobudget_avg_cpc_per_camp', 'autobudget_avg_click'));
}

=head2 is_strategy_cpm

=cut

sub is_strategy_cpm {
    my ($row) = @_;

    return ($row->{strategy} // '') eq 'cpm_default' || ($row->{strategy_name} // '') eq 'cpm_default';
}

=head2 get_strategy_goals($strategy_data_json, $meaningful_goals_json)

    Извлекает цели из стратегии и ключевых целей кампании

=cut
sub get_strategy_goals {
    my ($strategy_data_json, $meaningful_goals_json) = @_;
    my $strategy_data = from_json($strategy_data_json || '{}');
    if (!defined $strategy_data->{goal_id} || $strategy_data->{goal_id} == $Settings::MEANINGFUL_GOALS_OPTIMIZATION_GOAL_ID) {
        my $meaningful_goals = from_json($meaningful_goals_json || '[]');
        return [map {$_->{"goal_id"}} @$meaningful_goals];
    } else {
        return [$strategy_data->{goal_id}];
    }
}

=head2 get_strategy_mobile_goals($strategy_data_json, $meaningful_goals_json)

    Извлекает мобильные цели из стратегии и ключевых целей кампании

=cut
sub get_strategy_mobile_goals {
    my ($strategy_data_json, $meaningful_goals_json) = @_;
    my $goal_ids = get_strategy_goals($strategy_data_json, $meaningful_goals_json);
    return [grep { Primitives::get_goal_type_by_goal_id($_) eq 'mobile'} @$goal_ids];
}

=head3 %WORKERS_NUM_PROP_NAME

    Названия значений из ppc_properties, которые динамически определяют число
    рабочих процессов экспорта в БК

=cut

my %WORKERS_NUM_PROP_NAME = (
    std => 'bs_export_std_workers_num',
    heavy => 'bs_export_heavy_workers_num',
    buggy => 'bs_export_buggy_workers_num',
    full_lb_export => 'bs_full_lb_export_workers_num',
);

=head2 has_par_type_workers_num_prop($par_type)

    Проверить, есть ли для $par_type свойство, задающее число воркеров

=cut

sub has_par_type_workers_num_prop {
    return exists $WORKERS_NUM_PROP_NAME{shift()};
}

=head2 get_workers_num_prop_name($par_type, $shard)

    По типу очереди $par_type и номеру шарда $shard - получить строку с названием свойства, хранящего текущее число воркеров

=cut

sub get_workers_num_prop_name {
    my ($par_type, $shard) = @_;

    my $prefix = $WORKERS_NUM_PROP_NAME{$par_type};
    die "no property name for par_type $par_type" unless $prefix;

    return "${prefix}_shard_${shard}";
}

=head2 $BS::Export::BOTH

    DEPRECATED
    Пока со стороны БК не убрана поддержка RelevanceMatchType, отправляем RelevanceMatchType: both
=cut

our $BOTH = 'both';

=head3 get_target_type($c_type, $platform, $statusYandexAdv, $showOnYandexOnly)

    Определяет TargetType (запрещенные площадки для показа) для отправки в БК.

    $show_condition->{TargetType} = get_target_type(
        _camp($row, 'c_type'),
        _camp($row, 'platform'),
        _camp($row, 'statusYandexAdv'),
        _camp($row, 'showOnYandexOnly'));

    Параметры:
        $c_type - тип кампании (campaigns.type)
        $platform - площадки для показа (campaigns.platform)
        $statusYandexAdv - 'Yes'/'No', "Клиент размещает рекламу Яндекса, только в РСЯ" (users.statusYandexAdv)
        $showOnYandexOnly - 'Yes'/'No', "Клиент размещает рекламу Яндекса, только в Яндексе" (users.showOnYandexOnly)

=cut

sub get_target_type {
    my ($c_type, $platform, $statusYandexAdv, $showOnYandexOnly) = @_;

    my @TargetType;
    if ($c_type eq 'geo') {
        @TargetType = (2, 3);
    } elsif ($c_type =~ /^(?:cpm_banner|cpm_deals|cpm_yndx_frontpage|cpm_price|internal_autobudget)$/) {
        @TargetType = (0, 1, 2);
    } elsif ($c_type =~ /^(?:internal_distrib|internal_free)$/) {
        @TargetType = ();
    } else {

        $platform ||= '';

        if ($platform eq 'search') {
            push @TargetType, 3;
        }

        if ($statusYandexAdv eq 'Yes' || $platform eq 'context') {
            push @TargetType, (0, 1, 2);
        }

        if ($showOnYandexOnly eq 'Yes') {
            push @TargetType, (2, 3);
        }
    }

    return [map { int($_) } uniq nsort grep {defined $_} @TargetType];
}

=head3 %DIRECT_TO_BS_EXPRESSION_OPERATION_MAP

    Операции используемые в логических выражениях БК

=cut

my %DIRECT_TO_BS_EXPRESSION_OPERATION_MAP = (
    eq => 'equal',
    ne => 'not equal',
    gt => 'greater',
    lt => 'less',
    ge => 'greater or equal',
    le => 'less or equal'
  );

Hash::Util::lock_hash(%DIRECT_TO_BS_EXPRESSION_OPERATION_MAP);

=head3 get_bs_expression_for_modifier()

    Преобразует корректировки с условиями в конъюнктивной нормальной форме (КНФ) в формат БК
    Подробнее про форматы корректировок: https://wiki.yandex-team.ru/direct/TechnicalDesign/weather_bid_modifiers/

    Принимает на вход данные корректировки в виде:
        condition  =>  условие в формате директа (см. ниже)
        multiplier_pct  =>  множитель корректировки в процентах

    Пример условия в формате директа:
    condition: [  # КНФ с логическими опрерациями в виде json-объектов
        [{"parameter": "temp", "operation": "ge", "value": 10}],
        [{"parameter": "prec_type", "operation": "eq", "value": 2}],
        [{"parameter": "cloudness", "operation": "ge", "value": 50}]
    ]

    Возвращает корректировки в формате БК:
    {
     Expression: [    # КНФ с логическими опрерациями в формате БК
        [["temp", "greater or equal", "10"]],
        [["prec_type", "equal",  "2"]],
        [["cloudness", "greater or equal", "50"]]
        ],
     Coef: 150
    }

=cut

sub get_bs_expression_for_modifier {
    my (%modifier) = @_;

    my $bs_expression = [];
    foreach my $or_operations (@{$modifier{condition}}) {
        my $bs_or_operations = [map { _convert_to_bs_operation($_) } @$or_operations];
        push @$bs_expression, $bs_or_operations;
    }

    return {
        Expression => $bs_expression,
        Coef => $modifier{multiplier_pct} + 0  # numify it, ensuring it will be dumped as a number
    }
}

=head3 get_bs_expression_for_geo_modifier()

    Конвертирует корректировку по Geo из старого формата в новый (для ExpressionCoefs)

    Принимает на вход hash
    {
        region_id: 977,
        multiplier_pct: 150
    }

    Возвращает hash
    {
        Expression: [[["reg-id", "equal", "977"]]],
        Coef: 150
    }

=cut

sub get_bs_expression_for_geo_modifier {
    my ($modifier) = @_;

    return {
        Expression => [[["reg-id", "equal", "".$modifier->{region_id}]]],
        Coef       => 0+$modifier->{multiplier_pct}
    };
}

=head3 get_bs_expression_for_socdem_modifier()

    Преобразует соцдем-корректировку в формат БК

    На входе хеш с полями записи demography_multiplier_values
    На выходе корректировка в формате БК (для ExpressionCoefs)

=cut

sub get_bs_expression_for_socdem_modifier {
    my ($modifier) = @_;

    # https://st.yandex-team.ru/DIRECT-96262#5cbdce445bd695001f44df20
    state $crypta_age = {
        "0-17" => "0",
        "18-24" => "1",
        "25-34" => "2",
        "35-44" => "3",
        "45-54" => "4",
        "55-" => "5",
    };
    state $crypta_gender = {
        male => "0",
        female => "1",
    };

    my @condition;
    if (my $age = $modifier->{age}) {
        my $bs_age = $crypta_age->{$age};
        if ($age eq "unknown") {
            # Неизвестный возраст описывается  через отрицание известных возрастов
            foreach my $socdem_age (keys %$crypta_age) {
                push @condition, [['crypta-socdem-age', 'not equal', $socdem_age]];
            }
        } else {
            croak "Unknown age range <$age>" if !defined $bs_age;
            push @condition, [['crypta-socdem-age', 'equal', $bs_age]];
        }
    }
    if (my $gender = $modifier->{gender}) {
        my $bs_gender = $crypta_gender->{$gender};
        croak "Unknown gender <$gender>" if !defined $bs_gender;
        push @condition, [['crypta-socdem-gender', 'equal', $bs_gender]];
    }

    return {
        Expression => \@condition,
        Coef       => 0+$modifier->{multiplier_pct}
    }
}


=head3 _convert_to_bs_operation()

   Преобразует элементарную логическую операцию из формата директа в формат БК.
   https://wiki.yandex-team.ru/direct/TechnicalDesign/weather_bid_modifiers/#3.transportvbk

   формат директа:
   {
     parameter: <имя параметра> В терминах БК - кейворд.
     operation: eq | ne | gt | lt | le  (см %DIRECT_TO_BS_EXPRESSION_OPERATION_MAP)
     value:  <Integer>
   }

   формат БК:
   [ <keyword>, <operation>, <value> ]

=cut

sub _convert_to_bs_operation() {
    my ($atom) = @_;

    my $keyword = $atom->{parameter};
    my $bs_operation = $DIRECT_TO_BS_EXPRESSION_OPERATION_MAP{$atom->{operation}};
    die "Can't convert to BS operation" if (!$keyword || !$bs_operation);

    # В БК значение должно отправляться в виде строки
    return [$keyword, $bs_operation, "".$atom->{value}];
}

=head3 %DIRECT_TO_BS_EXPRESSION_OPERATION_MAP

    Операции используемые в логических выражениях БК

=cut

my %DIRECT_TO_BS_TEMPLATE_KEYS_MAP = (
    value => 'Value',
    template_resource_id => 'TemplateResourceID',
    template_resource_no => 'TemplateResourceNo',
    template_part_no => 'TemplatePartNo',
    width => 'Width',
    height => 'Height',
    unified_template_resource_no => 'UnifiedTemplateResourceNo', # гарантированно удаляется, может заменить TemplateResourceNo
    unified_template_resource_id => 'UnifiedTemplateResourceID', # гарантированно удаляется, может заменить TemplateResourceID
  );

Hash::Util::lock_hash(%DIRECT_TO_BS_TEMPLATE_KEYS_MAP);

=head3 convert_template_variables_to_bs_format()

    Преобразует ключи template_variables из формата БД Директа в формат контент-системы БК

    $template_variables_direct - это данные из banners_internal.template_variables + параметры ресурса из ppcdict.template_resource
        + дополнительные параметры по переменным. Например для картиночного ресурса будут данные по размерам

    Содержит массив структур:
    {
        value : значение переменной для отправки в БК
        internal_value : значение переменной для использования внутри Директа
        template_resource_id : часть составного ключа ресурса для БК
        template_resource_no : часть составного ключа ресурса для БК
        template_part_no : часть составного ключа ресурса для БК
        width : ширина картинки переменной
        height : высота картинки переменной
        unified_template_resource_no : часть составного ключа ресурса единого шаблона для БК
        unified_template_resource_id : часть составного ключа ресурса единого шаблона для БК
    }

    Опция unified:
        0 — просто выкинуть ключи UnifiedTemplateResourceNo и UnifiedTemplateResourceID,
        1 — заменить ключи TemplateResourceNo и TemplateResourceID на UnifiedTemplateResourceNo и UnifiedTemplateResourceID и выкинуть последние

=cut

sub convert_template_variables_to_bs_format {
    my ($template_variables_direct, %O) = @_;
    my $direct_internal_variable = "internal_value"; #ключ в template_variables, который используется только для внутренних нужд Директа и не должен отправляться в БК

    my @template_variables_bs;
    foreach my $direct_variable (@$template_variables_direct) {
        next if !(defined $direct_variable->{value});
        my $bs_variable;
        foreach my $direct_key (keys %$direct_variable) {
            next if $direct_key eq $direct_internal_variable;
            my $bs_key = $DIRECT_TO_BS_TEMPLATE_KEYS_MAP{$direct_key};
            die "Can't convert template_variables to BS format. Invalid key - $direct_key" if (!$bs_key);
            $bs_variable->{$bs_key} = $direct_variable->{$direct_key};
        }
        if ($O{unified}) {
            $bs_variable->{TemplateResourceNo} = $bs_variable->{UnifiedTemplateResourceNo};
            $bs_variable->{TemplateResourceID} = $bs_variable->{UnifiedTemplateResourceID};
        }
        delete $bs_variable->{UnifiedTemplateResourceNo};
        delete $bs_variable->{UnifiedTemplateResourceID};
        # может не оказаться id, если ресурса нет в едином шаблоне;
        # новые ресурсы (с id начиная с $DIRECT_TEMPLATE_RESOURCE_ID_START_VALUE) можно отправлять только с опцией unified
        if ($bs_variable->{TemplateResourceID}
            && ($bs_variable->{TemplateResourceID} < $DIRECT_TEMPLATE_RESOURCE_ID_START_VALUE || $O{unified})) {
            push @template_variables_bs, $bs_variable;
        }
    }

    return \@template_variables_bs
}

=head3 _convert_versioned_targeting_value_to_bs_format($value)

    Преобразует значение из списка adgroup_additional_targetings.value для версионного таргетинга
    из формата БД Директа в формат контент-системы БК

=cut

sub _convert_versioned_targeting_value_to_bs_format {
    my ($value) = @_;

    return sprintf('%s:%s:%s', $value->{targetingValueEntryId} // "", $value->{minVersion} // "", $value->{maxVersion} // "");
}

=head3 _convert_uatraits_targeting_value_to_bs_format($value)

    Преобразует значение из списка adgroup_additional_targetings.value для uatraits таргетинга (не версионного)
    из формата БД Директа в формат контент-системы БК

=cut

sub _convert_uatraits_targeting_value_to_bs_format {
    my ($value) = @_;

    return "". $value->{targetingValueEntryId};
}

=head3 _convert_mobile_installed_apps_targeting_value_to_bs_format($value)

    Преобразует значение из списка adgroup_additional_targetings.value для mobile_installed_apps таргетинга
    из формата БД Директа в формат контент-системы БК

=cut

sub _convert_mobile_installed_apps_targeting_value_to_bs_format {
    my ($value) = @_;

    return "". crc32( ($value->{ContentStoreAppID} || "") . $value->{ContentStoreName});
}

1;
