package BS::Export::Queues;

=encoding utf8

=head1 NAME

    BS::Export::Queues

=head1 DESCRIPTION

    Модуль содержит в себе описания очередей (потоков) экспорта в БК.

=cut

use warnings;
use strict;
use utf8;

use Settings;

use Hash::Util;

=head1 VARIABLES

=head2 %QUEUES

    Описание потоков экспорта в БК

=cut

our %QUEUES = (
    (map {
        0 + $_ => {
            type => "std",
            preprod => 0,
            order => $_,
            desc => "стандартная очередь $_",
            db_ref => \&PPC,
        }} (1..10)),
    (map {
        10 + $_ => {
            type => "camp",
            preprod => 0,
            order => $_,
            desc => "очередь заказов и новых кошельков $_",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        15 + $_ => {
            type => "camps_only",
            preprod => 0,
            order => $_,
            desc => "очередь для отправки только заказов $_",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        29 + $_ => {
            type => "heavy",
            preprod => 0,
            order => $_,
            desc => "тяжёлая очередь $_",
            db_ref => \&PPC,
        }} (1..10)),
    (map {
        40 + $_ => {
            type => "fast",
            preprod => 0,
            order => $_,
            desc => "быстрая очередь $_",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        50 + $_ => {
            type => "dev1",
            preprod => 0,
            order => $_,
            desc => "разработческая очередь 1",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        52 + $_ => {
            type => "internal_ads_dev1",
            preprod => 0,
            order => $_,
            desc => "разработческая очередь для внутренних объявлений 1",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        55 + $_ => {
            type => "dev2",
            preprod => 0,
            order => $_,
            desc => "разработческая очередь 2",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        57 + $_ => {
            type => "internal_ads_dev2",
            preprod => 0,
            order => $_,
            desc => "разработческая очередь для внутренних объявлений 2",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        74 + $_ => {
            type => "preprod",
            preprod => 1,
            order => $_,
            desc => "очередь preprod",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        90 + $_ => {
            type => 'buggy',
            preprod => 0,
            order => $_,
            desc => "buggy очередь $_",
            db_ref => \&PPC,
        }} (1..2)),
    (map {
        90 + $_ => {
            type => "std",
            preprod => 0,
            order => $_,
            desc => "стандартная очередь $_",
            db_ref => \&PPC,
        }} (11..25)),
    (map {
        120 + $_ => {
            type => "internal_ads",
            preprod => 0,
            order => $_,
            desc => "очередь для внутренних объявлений $_",
            db_ref => \&PPC,
        }} (1..1)),
    (map {
        130 + $_ => {
            type => "heavy",
            preprod => 0,
            order => $_,
            desc => "тяжёлая очередь $_",
            db_ref => \&PPC,
        }} (11..18)),
    (map {
        209 + $_ => {
            type => "full_lb_export",
            preprod => 0,
            logbroker_only => 1,
            ignore_bs_synced => 1,
            # при полной перепосылке вычислять время рестарта автобюджета не имеет смысла,
            # БК реагирует на него только в bssoap
            dont_calculate_autobudget_restart => 1,
            order => $_,
            desc => "очередь для полной переотправки в LB $_",
            db_ref => \&PPC,
        }} (1..10)),
);
$QUEUES{$_}->{logbroker_only} //= 0 for keys %QUEUES;
$QUEUES{$_}->{ignore_bs_synced} //= 0 for keys %QUEUES;
$QUEUES{$_}->{dont_calculate_autobudget_restart} //= 0 for keys %QUEUES;
Hash::Util::lock_hash_recurse(%QUEUES);

=head2 %SPECIAL_PAR_TYPES

    Список специльных номеров потоков (название => PAR_ID)

=over 4

=item ExportMaster

    Мастер-процесс экспорта в БК

=item nosend_for_drop_sandbox_client

    Кампанию никто не будет обрабатывать (мастер или воркеры).
    С точки зрения функции Campaign::mass_is_camp_deletable_by_hash
        такая кампания отсутствует в очереди экспорта

=back

=cut

our %SPECIAL_PAR_TYPES = (
    ExportMaster => 99,
    nosend_for_drop_sandbox_client => 201,
);

Hash::Util::lock_hash(%SPECIAL_PAR_TYPES);

=head1 SUBROUTINES

=head2 get_par_id_by_nick()

    Получить словарь соответствия "человекопонятное название и номер потока" - id потока

    Результат:
        $par_id_by_nick - hashref вида { "std:1" => 1, ... }

=cut

sub get_par_id_by_nick {
    my %result;

    for my $par_id (keys %QUEUES) {
        my $par_data = $QUEUES{$par_id};

        my $nick = "$par_data->{type}:$par_data->{order}";
        $result{$nick} = $par_id;
    }

    return \%result;
}

1;
