package BS::ExportMaster;

=head1 NAME

BS::ExportMaster

=head1 DESCRIPTION

    Функции для работы bsExportMaster.pl

=cut

use Direct::Modern;

use Hash::Util ();
use List::MoreUtils qw/any none uniq/;
use List::Util qw/sum max min/;
use Storable qw/dclone/;

use Yandex::DateTime;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::Interpolate;
use Yandex::ListUtils;
use Yandex::Runtime;
use Yandex::TimeCommon;
use Yandex::Trace;
use Yandex::Validate;

use Settings;

use BS::Export qw/:sql $SQL_BIDS_BASE_TYPE_ENABLED/;
use BS::Export::Queues;
use Campaign::Types qw/get_camp_kind_types/;
use Property;


# номер потока мастера для лока кампаний
my $PAR_ID = $BS::Export::Queues::SPECIAL_PAR_TYPES{ExportMaster};

=head2 $LOCK_NAME

    именованный sql-лок

=cut

our $LOCK_NAME = 'BS_EXPORT_MASTER';

=head2 HEAVY_LIMITS

    ограничения для переноса в тяжёлую очередь

=over 4

=item $HEAVY_CLIENT_CAMPAIGNS_BORDER

=item $HEAVY_CLIENT_BANNERS_BORDER

=item $HEAVY_CLIENT_CONTEXTS_BORDER

=item $HEAVY_CLIENT_BIDS_BORDER

=back

=cut

our $HEAVY_CLIENT_CAMPAIGNS_BORDER = 100;
our $HEAVY_CLIENT_BANNERS_BORDER = 20_000;
our $HEAVY_CLIENT_CONTEXTS_BORDER = 20_000;
our $HEAVY_CLIENT_BIDS_BORDER = 100_000;

=head2 $SHAKE_HEAVY_LIMIT

    если в тяжёлой очереди у клиента интегральный размер данных больше N -
    хвост его кампаний перемещается в конец очереди

=cut

my $SHAKE_HEAVY_LIMIT = 500_000;

=head2 $DEFAULT_RESYNC_CHUNK_SIZE

    количество элементов для единоразовой пересылки из bs_resync_queue
    значение умолчательное, используется если не задано в property

=cut

our $DEFAULT_RESYNC_CHUNK_SIZE = 10_000;
our $RESYNC_CHUNK_SIZE_PROP_NAME = 'bsexport_resync_chunk_size';

=head2 $RESYNC_PRIORITY_BORDER

    Начиная с какого значения приоритета считать записи в bs_resync_queue
    важными и отправлять вне зависимости от загрузки транспорта

=cut

our $RESYNC_PRIORITY_BORDER = 100;

=head2 $DEFAULT_RESYNC_PRIORITY_CHUNK_SIZE

    количество элементов "приоритетной переотправки" для единоразовой пересылки из bs_resync_queue
    значение умолчательное, используется если не задано в property

=cut

our $DEFAULT_RESYNC_PRIORITY_CHUNK_SIZE = 1_000;
our $RESYNC_PRIORITY_CHUNK_SIZE_PROP_NAME = 'bsexport_resync_priority_chunk_size';

=head2 $DEFAULT_RESYNC_ISIZE_BORDER

    интегральная длина очереди, меньше которой можно отправлять данные из очереди resync
    значение умолчательное, используется если не задано в property

=cut

our $DEFAULT_RESYNC_ISIZE_BORDER = 200_000;
our $RESYNC_ISIZE_BORDER_PROP_NAME = 'bsexport_resync_isize_border';

=head2 $DEFAULT_RESYNC_AGE_BORDER

    возраст очереди (в минутах), меньше которого можно отправлять данные из очереди resync
    значение умолчательное, используется если не задано в property

=cut

our $DEFAULT_RESYNC_AGE_BORDER = 15;
our $RESYNC_AGE_BORDER_PROP_NAME = 'bsexport_resync_age_border';

=head2 $DEFAULT_RESYNC_HEAVY_ISIZE_BORDER

    интегральная длина heavy-очереди, меньше которой можно отправлять данные из очереди resync
    значение умолчательное, используется если не задано в property

=cut

our $DEFAULT_RESYNC_HEAVY_ISIZE_BORDER = 500_000;
our $RESYNC_HEAVY_ISIZE_BORDER_PROP_NAME = 'bsexport_resync_heavy_isize_border';

=head2 $DEFAULT_ISIZE_CAMPS_NUM_COEF

    коэффициент по умолчанию для camps_num для вычисления интегрального размера очереди (isize)

=cut

our $DEFAULT_ISIZE_CAMPS_NUM_COEF = 20;
our $ISIZE_CAMPS_NUM_COEF_PROP_NAME = 'bsexport_isize_camps_num_coef';

=head2 $DEFAULT_ISIZE_BANNERS_NUM_COEF

    коэффициент по умолчанию для banners_num для вычисления интегрального размера очереди (isize)

=cut

our $DEFAULT_ISIZE_BANNERS_NUM_COEF = 10;
our $ISIZE_BANNERS_NUM_COEF_PROP_NAME = 'bsexport_isize_banners_num_coef';

=head2 $DEFAULT_ISIZE_CONTEXTS_NUM_COEF

    коэффициент по умолчанию для contexts_num для вычисления интегрального размера очереди (isize)

=cut

our $DEFAULT_ISIZE_CONTEXTS_NUM_COEF = 10;
our $ISIZE_CONTEXTS_NUM_COEF_PROP_NAME = 'bsexport_isize_contexts_num_coef';

=head2 $DEFAULT_ISIZE_BIDS_NUM_COEF

    коэффициент по умолчанию для bids_num для вычисления интегрального размера очереди (isize)

=cut

our $DEFAULT_ISIZE_BIDS_NUM_COEF = 1;
our $ISIZE_BIDS_NUM_COEF_PROP_NAME = 'bsexport_isize_bids_num_coef';

=head2 $DEFAULT_ISIZE_PRICES_NUM_COEF

    коэффициент по умолчанию для prices_num для вычисления интегрального размера очереди (isize)

=cut

our $DEFAULT_ISIZE_PRICES_NUM_COEF = 0.1;
our $ISIZE_PRICES_NUM_COEF_PROP_NAME = 'bsexport_isize_prices_num_coef';

=head2 @STATS_PAR_TYPES

    типы очередей для мониторинга

=cut

our @STATS_PAR_TYPES = qw/std stdprice fast heavy dev1 dev2 devprice1 devprice2 nosend buggy full_lb_export preprod camps_only internal_ads internal_ads_dev1 internal_ads_dev2/;

=head2 @STATS_PERCENTILES

    перцентили для мониторинга

=cut

our @STATS_PERCENTILES = qw/80 90 95 99 100/;

=head2 @STATS_NUMS

    типы статистик для мониторинга

=cut

our @STATS_NUMS = qw/max_age_minutes
                    max_sequence_minutes
                    campaigns_count
                    integral_size integral_size_with_props
                    camps_num banners_num contexts_num bids_num prices_num
                   /;

=head2 @UNSYNC_ITEMS_TYPES

    Типы объектов, нуждающиеся в отправке в БК

    Порядок элементов важен (возможно, устарело).
    Обновление bids_performance нужно только для триггера ess-отправки ставок

=cut

my @UNSYNC_ITEMS_TYPES = (
    { field => 'perf_id',
      select_sql => "
            SELECT bi_perf.perf_filter_id, p.cid
            FROM bids_performance bi_perf
            JOIN phrases p on p.pid = bi_perf.pid and p.adgroup_type = 'performance'
            WHERE bi_perf.statusBsSynced = 'No'
        ",
        update_sql => "UPDATE bids_performance set statusBsSynced = 'Sending', LastChange = LastChange where statusBsSynced = 'No' and perf_filter_id in ",
        cid_field => 'p.cid',
    },
    { field => 'bid',
      select_sql => "
             SELECT b.bid, p.cid
               FROM banners b
               JOIN phrases p USING(pid)
              WHERE b.statusBsSynced = 'No'
        ",
      update_sql => "UPDATE banners SET statusBsSynced = 'Sending', LastChange=LastChange WHERE statusBsSynced = 'No' and bid in ",
      cid_field => 'p.cid',
    },
    { field => 'pid',
      select_sql => "
             SELECT p.pid, p.cid
               FROM phrases p
              WHERE p.statusBsSynced = 'No'

        ",
      update_sql => "UPDATE phrases SET statusBsSynced = 'Sending', LastChange=LastChange WHERE statusBsSynced = 'No' and pid in ",
      cid_field => 'p.cid',
    },
    { field => 'cid',
      select_sql => [
        'SELECT c.cid, c.cid FROM campaigns c',
        WHERE => {
            'c.statusBsSynced' => 'No',
            'c.type' => get_camp_kind_types('bs_export'),
        },
      ],
      update_sql => "UPDATE campaigns SET statusBsSynced = 'Sending', LastChange = LastChange WHERE statusBsSynced = 'No' and cid in ",
      cid_field => 'c.cid',
    },
);

=head2 init(%options)

    Первоначальная установка переменных
    %options:
        shardid - номер шарда

=cut

my ($SHARDID);
sub init {
    my %options = @_;
    ($SHARDID) = @options{qw/shardid/};
}

=head2 get_unsync_items

    получить хеш из несинхронизованных id
    $RET{ $cid }{ cid|bid|pid|bid_id|ret_id|dyn_cond_id|perf_id }{ $id } = undef

=cut

sub get_unsync_items {
    my ($log, $only_cids) = @_;

    my %unsync;
    do_sql(PPC(shard => $SHARDID), "SET TRANSACTION ISOLATION LEVEL REPEATABLE READ");
    do_in_transaction {
        for my $type (@UNSYNC_ITEMS_TYPES) {
            $log->out("Fetching unsync items for field $type->{field}");
            my $sth = exec_sql(PPC(shard => $SHARDID), [xflatten($type->{select_sql}), _get_cids_only_sql($type->{cid_field}, $only_cids)]);
            while(my ($lid, $cid) = $sth->fetchrow_array()) {
                $unsync{$cid}{$type->{field}}{$lid} = undef;
            }
        }
    };
    return %unsync;
}


sub _calculate_statistics {
    my ($only_cids) = @_;

    my $sql_where_data = BS::Export::get_sql_where_data();
    my $sql_where_banner_unsync = BS::Export::get_sql_where_banner_unsync();
    my $sql_where_context_or_bid_unsync = BS::Export::get_sql_where_context_or_bid_unsync();
    my $sql_where = BS::Export::get_sql_where();

    do_sql(PPC(shard => $SHARDID), "SET TRANSACTION ISOLATION LEVEL REPEATABLE READ");
    my (%to_sync, $campaigns);
    my %default = (contexts_num => 0, banners_num => 0, prices_num => 0, bids_num => 0);

    do_in_transaction {

=head2 COMMENT

    Упрощенный расчет статистики - учитываем только статус синхронизации с БК

=cut

        my $statistics = get_all_sql(PPC(shard => $SHARDID), [qq[
            SELECT
                c.cid,
                SUM(
                    IF(b.statusBsSynced = 'Sending', 1, 0)
                ) banners_num
              , SUM(
                    IF(p.statusBsSynced = 'Sending', 1, 0)
                ) contexts_num
            FROM
                bs_export_candidates bec
                JOIN campaigns c ON c.cid = bec.cid
                LEFT JOIN phrases p ON p.cid = c.cid
                LEFT JOIN banners b ON b.pid = p.pid
            WHERE
                $sql_where_data
        ], _get_cids_only_sql('bec.cid' => $only_cids), qq[
            GROUP BY c.cid
            ORDER BY null
        ]]);

        foreach my $c (@$statistics) {
            $to_sync{$$c{cid}} = {%default} unless exists $to_sync{$$c{cid}};
            for my $field (qw/banners_num contexts_num/) {
                $to_sync{$$c{cid}}->{$field} += $c->{$field};
            }
        }

        my @bids_sql = (
            qq[
                SELECT STRAIGHT_JOIN p.cid, COUNT(*) AS bids_num
                FROM phrases p FORCE INDEX(p_cid)
                    JOIN bids bi FORCE INDEX(bid_pid) ON bi.pid = p.pid
                            AND p.statusPostModerate IN ('Yes', 'Rejected')
                            AND p.statusBsSynced = 'Sending'
                            AND bi.is_suspended = '0'
                WHERE p.cid IN (%1\$s)
                GROUP BY p.cid
                ORDER BY null
            ],
            qq[
                SELECT STRAIGHT_JOIN p.cid, COUNT(*) AS bids_num
                FROM phrases p FORCE INDEX(p_cid)
                    JOIN bids_retargeting bi_ret FORCE INDEX(pid) ON bi_ret.pid = p.pid
                            AND p.statusPostModerate IN ('Yes', 'Rejected')
                            AND p.statusBsSynced = 'Sending'
                            AND bi_ret.is_suspended = '0'
                WHERE p.cid IN (%1\$s)
                GROUP BY p.cid
                ORDER BY null
            ],
            qq[
                SELECT STRAIGHT_JOIN p.cid, COUNT(*) AS bids_num
                FROM phrases p FORCE INDEX(p_cid)
                    JOIN bids_dynamic bi_dyn FORCE INDEX(i_pid_dyn_cond_id) ON bi_dyn.pid = p.pid
                            AND p.adgroup_type = 'dynamic'
                            AND p.statusPostModerate IN ('Yes', 'Rejected')
                            AND p.statusBsSynced = 'Sending'
                            AND NOT FIND_IN_SET('suspended', bi_dyn.opts)
                WHERE p.cid IN (%1\$s)
                GROUP BY p.cid
                ORDER BY null
            ],
            qq[
                SELECT STRAIGHT_JOIN p.cid, COUNT(*) as bids_num
                FROM phrases p FORCE INDEX(p_cid)
                    JOIN bids_performance bi_perf FORCE INDEX(pid) on bi_perf.pid = p.pid
                        AND p.adgroup_type = 'performance'
                        AND p.statusPostModerate in ('Yes', 'Rejected')
                        AND p.statusBsSynced = 'Sending'
                        AND bi_perf.is_deleted = '0'
                        AND bi_perf.is_suspended = '0'
                WHERE p.cid IN (%1\$s)
                GROUP BY p.cid
                ORDER BY null
            ],
            qq[
                SELECT STRAIGHT_JOIN p.cid, COUNT(*) as bids_num
                FROM phrases p FORCE INDEX(p_cid)
                    JOIN bids_base bi_b FORCE INDEX(idx_bids_base_pid) on bi_b.pid = p.pid
                        AND p.statusPostModerate in ('Yes', 'Rejected')
                        AND p.statusBsSynced = 'Sending'
                        AND $SQL_BIDS_BASE_TYPE_ENABLED
                        AND NOT FIND_IN_SET('suspended', bi_b.opts)
                        AND NOT FIND_IN_SET('deleted', bi_b.opts)
                WHERE p.cid IN (%1\$s)
                GROUP BY p.cid
                ORDER BY null
            ],
        );

        my @cids = sort {$a <=> $b} keys %to_sync;
        foreach my $chunk (chunks(\@cids, 200)) {
            my $cids = join ',', @$chunk;
            foreach my $sql (@bids_sql) {
                my $groups = get_all_sql(PPC(shard => $SHARDID), sprintf $sql, $cids);
                foreach my $g (@$groups) {
                    $to_sync{$g->{cid}} = {%default} unless exists $to_sync{$g->{cid}};
                    $to_sync{$g->{cid}}->{$_} += $g->{$_} foreach qw/bids_num/;
                }
            }
        }

        $campaigns = get_all_sql(PPC(shard => $SHARDID), [
                        "SELECT
                            c.cid, bes.par_type, c.statusBsSynced,
                            u.ClientID, ua.ClientID AgencyID,
                            FIND_IN_SET('as_soon_as_possible', clo.client_flags) as asap
                        FROM
                            campaigns c
                            LEFT JOIN bs_export_specials bes USING(cid)
                            LEFT JOIN users ua ON c.AgencyUID = ua.uid
                            JOIN users u ON u.uid = c.uid
                            LEFT JOIN clients_options clo ON clo.ClientID = c.ClientID",
                        WHERE => {'c.cid' => [keys %to_sync]}]);
    };

    return [map {
        my $c = $_;
        hash_merge \my %r,
            hash_cut($to_sync{$c->{cid}}, qw/contexts_num banners_num prices_num bids_num/),
            hash_cut $c, qw/par_type ClientID AgencyID cid asap/;
        $r{camps_num} = $c->{statusBsSynced} eq 'Sending' ? 1 : 0;
        \%r;
    } @$campaigns]
}

sub _get_cids_only_sql {
    my ($field, $only_cids, %O) = @_;
    need_list_context();

    return ($only_cids && @$only_cids) ? (($O{add_where} ? 'WHERE' : 'AND'), {$field => $only_cids}) : ();
}

=head2 update_status_bs_synced

    обновить во всех таблицах statusBsSynced: 'No' -> 'Sending',
    добавить/обновить записи в bs_export_queue

=cut

# номер итерации, для того, чтобы отличить первую итерацию
my $ITERATION_NUM = 0;
sub update_status_bs_synced {
    my ($log, $only_cids) = @_;
    # выбираем неотправленные данные
    my %unsync = get_unsync_items($log, $only_cids);

    # довыбираем кампании, которые, возможно, остались заблокированы с предыдущего запуска
    if (not $ITERATION_NUM++) {
        my @old_cids = grep {!$unsync{$_}}
                @{get_one_column_sql(PPC(shard => $SHARDID), ["SELECT cid FROM bs_export_queue WHERE par_id = ?", _get_cids_only_sql(cid => $only_cids)], $PAR_ID) || []};
        $log->out({old_locked_cids => \@old_cids});
        $unsync{$_}{cid}{$_} = undef for @old_cids;
    }

    my @all_cids = nsort keys %unsync;
    $log->out("locked ".scalar(@all_cids)." unsync cids:", \@all_cids);

    my @cids_chunks = chunks \@all_cids, 2_000;
    my $chunk_idx = 0;
    for my $cids (@cids_chunks) {
        $chunk_idx++;
        $log->out("processing cids chunk [$chunk_idx/".scalar(@cids_chunks)."]:", $cids);
        # пытаемся залочить заказы
        # чтобы не обновлять statusBsSynced No -> Sending у кампаний, который сейчас отправляются, иначе изменения потеряются
        do_update_table(PPC(shard => $SHARDID), "bs_export_queue", {par_id => $PAR_ID}, where => {par_id__is_null => 1, cid => $cids});
        # отбрасываем те, которые не смогли залочить
        my (@busy_cids, @locked_cids);
        for my $row (@{get_all_sql(PPC(shard => $SHARDID), ["SELECT cid, par_id FROM bs_export_queue", WHERE => {cid => $cids}])}) {
            if (defined $row->{par_id} && $row->{par_id} == $PAR_ID) {
                # залочена нами
                push @locked_cids, $row->{cid};
            } else {
                push @busy_cids,  $row->{cid};
            }
        }
        if (@busy_cids) {
            $log->out("busy cids: ".join(", ", @busy_cids));
            delete @unsync{@busy_cids};
            $cids = xminus $cids, \@busy_cids;
        }

        # записываем в _candidates
        $log->out("insert cids into bs_export_candidates");

        do_mass_insert_sql(PPC(shard => $SHARDID), "INSERT IGNORE INTO bs_export_candidates (cid) VALUES %s", [map {[$_]} @$cids]);
        # обновляем statusBsSynced
        $log->out("update statusBsSynced");
        my $profile = Yandex::Trace::new_profile('bs_export_master:update_status_bs_synced', tags => 'update_status');
        for my $type (@UNSYNC_ITEMS_TYPES) {
            my @ids = nsort map {keys %$_} grep {$_} map {$unsync{$_}{$type->{field}}} @$cids;
            my $iter_cnt = 0;
            $log->out('type ' . $type->{field} . ', to update '. (scalar @ids). ' items');
            for my $ids_chunk (chunks \@ids, 2_000) {
                do_sql(PPC(shard => $SHARDID), $type->{update_sql} . " (".join(",", @$ids_chunk).")");
                $iter_cnt++;
                $log->out("Updating " . $type->{field} . ", done iteration number $iter_cnt") if $iter_cnt % 10 == 0;
            }
        }
        undef $profile;

        # отпускаем локи. в bs_export_queue временно оказываются заказы с неверной статистикой,
        # считаем это погрешностью и не обращаем внимания, это лучше, чем надолго лочить голову очереди
        $log->out("unlock camps in queue");
        do_update_table(PPC(shard => $SHARDID), "bs_export_queue", {par_id => undef}, where => {par_id => $PAR_ID, cid => \@locked_cids});

        # считаем статистику по всем кандидатам
        $profile = Yandex::Trace::new_profile('bs_export_master:update_status_bs_synced', tags => 'calc_stat');
        $log->out("calc stat");
        my $stat_rows = _calculate_statistics($only_cids);
        undef $profile;

        $stat_rows = [grep {$_->{camps_num} || $_->{banners_num} || $_->{contexts_num} || $_->{bids_num} || $_->{prices_num}} @$stat_rows];
        $log->out("camps for queue: ".join(", ", map {$_->{cid}} @$stat_rows));
        do_mass_insert_sql(PPC(shard => $SHARDID), "
                    INSERT INTO bs_export_queue (cid, queue_time, camps_num, banners_num, contexts_num, bids_num, prices_num)
                                         VALUES %s
                        ON DUPLICATE KEY UPDATE sync_val = sync_val + 1,
                                                queue_time = IF(sync_val > 5 OR (is_full_export = 1 AND camps_num = 0 AND banners_num = 0 AND contexts_num = 0 AND bids_num = 0 AND prices_num = 0), now(), queue_time),
                                                seq_time = IF(is_full_export = 1 AND camps_num = 0 AND banners_num = 0 AND contexts_num = 0 AND bids_num = 0 AND prices_num = 0, NOW(), seq_time),
                                                camps_num = VALUES(camps_num),
                                                banners_num = VALUES(banners_num), contexts_num = VALUES(contexts_num),
                                                bids_num = VALUES(bids_num), prices_num = VALUES(prices_num)
                        ",
                     [map {[$_->{cid}, unix2mysql(time), $_->{camps_num}, $_->{banners_num}, $_->{contexts_num}, $_->{bids_num}, $_->{prices_num}]} @$stat_rows]);

        do_sql(PPC(shard => $SHARDID), ['DELETE FROM bs_export_candidates', _get_cids_only_sql(cid => $only_cids, add_where => 1)]);
        delete_cids_without_changes($log, $only_cids);

        # Находим кампании с быстрой доставкой до показов (флаг asap, as_soon_as_possible) и переносим их в fast очередь
        my @asap_cids = map {$_->{cid}} grep { $_->{asap} } @$stat_rows;
        do_mass_insert_sql(PPC(shard => $SHARDID), "INSERT INTO bs_export_specials (cid, par_type) VALUES %s ON DUPLICATE KEY UPDATE par_type = VALUES(par_type)",
            [map {[$_, 'fast']} @asap_cids]
        );
    }
    if (!@cids_chunks) {
        # все равно делаем попытку почистить очередь
        delete_cids_without_changes($log, $only_cids);
    }
}

=head2 delete_cids_without_changes($log, $only_cids)

    Удалет из очереди незалоченные записи со всеми нулями.
    Могут образоваться в экзотической ситуации - статистика пересчиталась, пока кампания была залочена.
    Еще неэкзотический случай - снятие флага is_full_export.

    Параметры:
        $log - объект Yandex::Log для логирования
        $only_cids - arrayref cid'ов для ограниения работы

=cut

sub delete_cids_without_changes {
    my ($log, $only_cids) = @_;
    my $cids_wo_changes = get_one_column_sql(PPC(shard => $SHARDID), ["
                                    SELECT cid
                                      FROM bs_export_queue
                                     WHERE par_id is null
                                       AND camps_num = 0
                                       AND banners_num = 0
                                       AND contexts_num = 0
                                       AND bids_num = 0
                                       AND prices_num = 0
                                       AND is_full_export = 0
                                    ", _get_cids_only_sql(cid => $only_cids)]) || [];
    if (@$cids_wo_changes) {
        $log->out("Delete cid without changes from bs_export_queue: ".join(", ", @$cids_wo_changes));
        do_sql(PPC(shard => $SHARDID), ["DELETE FROM bs_export_queue", WHERE => {cid => $cids_wo_changes}]);
    }
}

=head2 move_heavy_campaigns

    переносим заказы в тяжёлую очередь

=cut

sub move_heavy_campaigns {
    my ($log, $only_cids) = @_;
    my $profile = Yandex::Trace::new_profile('bs_export_master:move_heavy_campaigns');
    # выбираем все данные
    my $camps = get_all_sql(PPC(shard => $SHARDID), ["
                            SELECT c.uid, c.cid, s.par_type,
                                   camps_num, banners_num, contexts_num, bids_num
                              FROM bs_export_queue q
                                   JOIN campaigns c on q.cid = c.cid
                                   LEFT JOIN bs_export_specials s on s.cid = q.cid
                            WHERE $SQL_NOT_INTERNAL_CAMPAIGN_TYPES", _get_cids_only_sql('q.cid' => $only_cids)]);
    # группируем по уидам
    my $stat = {};
    for my $c (@$camps) {
        for my $field (qw/camps_num contexts_num banners_num bids_num/) {
            $stat->{$c->{uid}}{$field} += $c->{$field};
        }
    }

    # находим "тяжёлые уиды"
    my %HEAVY_UIDS;
    while(my ($uid, $s) = each %$stat) {
        if ($s->{banners_num} > $HEAVY_CLIENT_BANNERS_BORDER
            || $s->{camps_num} > $HEAVY_CLIENT_CAMPAIGNS_BORDER
            || $s->{contexts_num} > $HEAVY_CLIENT_CONTEXTS_BORDER
            || $s->{bids_num} > $HEAVY_CLIENT_BIDS_BORDER
        ) {
            $HEAVY_UIDS{$uid} = undef;
        }
    }

    # какие кампании надо перенести
    my @cids_to_heavy = map {$_->{cid}} grep { exists $HEAVY_UIDS{$_->{uid} } && !defined $_->{par_type}} @$camps;
    if (@cids_to_heavy) {
        $log->out("move camps to heavy queue: ".join(',', @cids_to_heavy));
        do_mass_insert_sql(PPC(shard => $SHARDID), "INSERT IGNORE INTO bs_export_specials (cid, par_type) VALUES %s",
                           [map {[$_, 'heavy']} @cids_to_heavy]
            );
    }
}

=head2 shake_heavy_queue

    "перетряхиваем" тяжёлую очередь - если у одного клиента в тяжёлой очереди стоит
    более N элементов (по интегральной оценке) - у хвоста обновляем seq_time на now()

=cut

sub shake_heavy_queue {
    my ($log, $only_cids) = @_;

    my $profile = Yandex::Trace::new_profile('bs_export_master:shake_heavy_queue');

    # выбираем все данные
    my $camps = get_all_sql(PPC(shard => $SHARDID), ["
                            SELECT c.uid, q.cid
                                 , sum($SQL_CALC_ISIZE) AS isize
                              FROM bs_export_queue q
                                   JOIN bs_export_specials s on s.cid = q.cid
                                   JOIN campaigns c on q.cid = c.cid
                             WHERE s.par_type = 'heavy'
                            ", _get_cids_only_sql('q.cid' => $only_cids), "
                             GROUP BY c.uid, q.cid
                             ORDER BY q.seq_time
                            "]);

    # суммируем isize по клиентам, когда дойдём до границы - начинаем перекладывать
    # кампании в move_cids - массив кампаний для перенося в конец очереди
    my %uid_isize;
    my @move_cids;
    for my $camp (@$camps) {
        if ($uid_isize{$camp->{uid}} && $uid_isize{$camp->{uid}} >= $SHAKE_HEAVY_LIMIT) {
            push @move_cids, $camp->{cid};
        }
        $uid_isize{$camp->{uid}} += $camp->{isize};
    }

    $log->out("cids to move to end of heavy queue: ".join(", ", @move_cids));
    if (@move_cids) {
        do_sql(PPC(shard => $SHARDID), ["UPDATE bs_export_queue SET seq_time = now()", WHERE => {cid => \@move_cids}]);
    }
}

=head2 add_resync_items

    добавляет некоторое количество элементов для перепосылки в очередь
    (фактически, апдейтит statusBsSynced)

    возвращает ссылку на хэш с количеством добавленных элементов:
    { banners => XXX, ... }

=cut

{
my %resync_sql = (
    priority => q[
        SELECT r.id, r.cid, r.bid, r.pid
        FROM bs_resync_queue r
        WHERE
            r.priority_inverse < - ?
        ORDER BY
            r.priority_inverse,
            r.sequence_time
        LIMIT ?],
    not_priority => q[
        SELECT r.id, r.cid, r.bid, r.pid
        FROM bs_resync_queue r
        ORDER BY
            r.priority_inverse,
            r.sequence_time
        LIMIT ?]
);

sub _get_resync_chunk_size {
    return eval { Property->new($RESYNC_CHUNK_SIZE_PROP_NAME)->get(); } || $DEFAULT_RESYNC_CHUNK_SIZE;
}

sub _get_resync_priority_chunk_size {
    return eval { Property->new($RESYNC_PRIORITY_CHUNK_SIZE_PROP_NAME)->get(); } || $DEFAULT_RESYNC_PRIORITY_CHUNK_SIZE;
}

sub _get_resync_isize_border {
    return eval { Property->new($RESYNC_ISIZE_BORDER_PROP_NAME)->get(); } || $DEFAULT_RESYNC_ISIZE_BORDER;
}

sub _get_resync_heavy_isize_border {
    return eval { Property->new($RESYNC_HEAVY_ISIZE_BORDER_PROP_NAME)->get(); } || $DEFAULT_RESYNC_HEAVY_ISIZE_BORDER;
}

sub _get_resync_age_border {
    return eval { Property->new($RESYNC_AGE_BORDER_PROP_NAME)->get(); } || $DEFAULT_RESYNC_AGE_BORDER;
}

sub add_resync_items {
    my ($is_priority, $log) = @_;

    my $profile = Yandex::Trace::new_profile('bs_export_master:add_resync_items');
    my @queue = @{get_all_sql(PPC(shard => $SHARDID),
        $resync_sql{$is_priority ? 'priority' : 'not_priority'},
        ($is_priority ? ($RESYNC_PRIORITY_BORDER, _get_resync_priority_chunk_size()) : (_get_resync_chunk_size())),
    )};

    my $stat = {};
    if (@queue) {
        # не смотрим на statusBsSynced - нам в любом случае для чистоты эксперимента нужно прогнать данные через bsExportMaster
        if (my @bids = grep {$_} map {$_->{bid}} @queue) {
            $log->out('Resetting statusBsSynced during resync for banners:', \@bids) if $log;
            $stat->{banners} = do_update_table(PPC(shard => $SHARDID), "banners",
                                      {statusBsSynced => 'No', LastChange__dont_quote => 'LastChange'},
                                      where => {bid => \@bids});
        }
        if (my @pids = grep {$_} map {$_->{pid}} @queue) {
            $log->out('Resetting statusBsSynced during resync for phrases:', \@pids) if $log;
            $stat->{contexts} = do_update_table(PPC(shard => $SHARDID), "phrases",
                                      {statusBsSynced => 'No', LastChange__dont_quote => 'LastChange'},
                                      where => {pid => \@pids});
        }
        if (my @cids = map {$_->{cid}} grep {!$_->{bid} && !$_->{pid}} @queue) {
            $log->out('Resetting statusBsSynced during resync for campaigns:', \@cids) if $log;
            $stat->{campaigns} = do_update_table(PPC(shard => $SHARDID), "campaigns",
                                      {statusBsSynced => 'No', LastChange__dont_quote => 'LastChange'},
                                      where => {cid => \@cids});
        }
        do_delete_from_table(PPC(shard => $SHARDID), "bs_resync_queue", where => {id => [map {$_->{id}} @queue]} );
    }
    return $stat;
}
}


sub exists_resync_priority {
    my $shard_id = shift;
    return get_one_field_sql(PPC(shard => $shard_id), 'SELECT 1 FROM bs_resync_queue WHERE priority_inverse < ? LIMIT 1', -$RESYNC_PRIORITY_BORDER) || 0;
}

{
    my $isize_camps_num_coef_prop = Property->new($ISIZE_CAMPS_NUM_COEF_PROP_NAME);
    my $isize_banners_num_coef_prop = Property->new($ISIZE_BANNERS_NUM_COEF_PROP_NAME);
    my $isize_contexts_num_coef_prop = Property->new($ISIZE_CONTEXTS_NUM_COEF_PROP_NAME);
    my $isize_bids_num_coef_prop = Property->new($ISIZE_BIDS_NUM_COEF_PROP_NAME);
    my $isize_prices_num_coef_prop = Property->new($ISIZE_PRICES_NUM_COEF_PROP_NAME);

=head3 get_isize_with_props

    Вычислить интегральный размер очереди в зависимости от набора соответствующих пропертей

    Параметры:
        $camps_num - количество кампаний
        $banners_num - количество баннеров
        $contexts_num - количество условий
        $bids_num - количество фраз
        $prices_num - количество ставок

=cut

sub get_isize_with_props {
    my ($camps_num, $banners_num, $contexts_num, $bids_num, $prices_num) = @_;

    my ($camps_num_coef, $banners_num_coef, $contexts_num_coef, $bids_num_coef, $prices_num_coef);

    foreach my $vars (
        [\$camps_num_coef, $isize_camps_num_coef_prop, $DEFAULT_ISIZE_CAMPS_NUM_COEF],
        [\$banners_num_coef, $isize_banners_num_coef_prop, $DEFAULT_ISIZE_BANNERS_NUM_COEF],
        [\$contexts_num_coef, $isize_contexts_num_coef_prop, $DEFAULT_ISIZE_CONTEXTS_NUM_COEF],
        [\$bids_num_coef, $isize_bids_num_coef_prop, $DEFAULT_ISIZE_BIDS_NUM_COEF],
        [\$prices_num_coef, $isize_prices_num_coef_prop, $DEFAULT_ISIZE_PRICES_NUM_COEF]) {

        my ($num_coef_ref, $prop, $default) = @$vars;

        ${$num_coef_ref} = $prop->get(30) // $default;
        if (!is_valid_float(${$num_coef_ref})) {
            ${$num_coef_ref} = $default;
        }
    }

    return $camps_num * $camps_num_coef
         + $banners_num * $banners_num_coef
         + $contexts_num * $contexts_num_coef
         + $bids_num * $bids_num_coef
         + $prices_num * $prices_num_coef;
}
}

=head3 _get_queue_stats(%params)

    Получить объекты в очереди у заданного шарда, по типам очередей

    Параметры именованные:
        shard   - для какого шарда получать статистику (обязательный)

    Результат:
    {
        partype => [{....}],
    }

=cut

sub _get_queue_stats {
    my %params = @_;

    # в продолжение TODO-комментария из BS::ExportWorker::lock_campaigns_in_queue
    # часть логики про то, какие потоки что умеют отправлять - продублирована и здесь
    # например в том считать ли prices_num для потока или нет.

    # выбираем статистику
    my $sth = exec_sql(PPC(shard => $params{shard}), [
                            'SELECT q.cid',
                                 ', s.par_type',
                                 ', TIMESTAMPDIFF(SECOND, q.queue_time, NOW()) AS age',
                                 ', TIMESTAMPDIFF(SECOND, q.seq_time, NOW()) AS sequence_age',
                                 ', TIMESTAMPDIFF(SECOND, q.full_export_seq_time, NOW()) AS full_export_sequence_age',
                                 ", $SQL_CALC_ISIZE AS integral_size",
                                 ', q.camps_num, q.banners_num, q.contexts_num, q.bids_num',
                                 ', q.prices_num',
                                 ', q.is_full_export',
                                 ', c.statusBsSynced',
                            'FROM bs_export_queue q',
                            'LEFT JOIN bs_export_specials s ON s.cid = q.cid',
                            'LEFT JOIN campaigns c ON c.cid = q.cid'
                       ]);

    my %result;
    while (my $db_row = $sth->fetchrow_hashref) {
        my $par_type = delete $db_row->{par_type};
        my $full_export_sequence_age = delete $db_row->{full_export_sequence_age};

        my $camps_num_value = $db_row->{camps_num};
        if (defined($db_row->{"statusBsSynced"})
            && $db_row->{"statusBsSynced"} eq "Yes") {
            $camps_num_value = 0;
        }

        $db_row->{integral_size_with_props} = get_isize_with_props(
            $camps_num_value, $db_row->{banners_num}, $db_row->{contexts_num}, $db_row->{bids_num}, $db_row->{prices_num});

=head3 TODO

    На самом деле heavy, fast и buggy потоки могут отправлять "только цены"
    std может отправлять цены, если у той же кампании есть данные.

    Кроме того в devprice - два разных потока (раньше считалось для devprice1 и devprice2
    но реально в данных был только devprice)

    Ну и хорошо бы *_num и integral_size иметь в виде чисел, а не строк

=cut

        if (any { $db_row->{$_} > 0 } qw/camps_num banners_num contexts_num bids_num/) {
            my $row = dclone($db_row);
            $row->{par_type} = $par_type // 'std';
            $row->{prices_num} = "0" unless $par_type;
            push @{ $result{ $row->{par_type} } }, $row;
        }
        if ($db_row->{is_full_export} > 0
            && none { ($par_type // 'std') eq $_ } @BS::Export::COMMON_EXCLUDE_PAR_TYPES
        ) {
            my $row = dclone($db_row);
            $row->{par_type} = 'full_lb_export';
            $row->{camps_num} = "1";

            $row->{sequence_age} = $full_export_sequence_age;
            $row->{integral_size} = "0.0";
            $row->{integral_size_with_props} = "0.0";
            $row->{$_} = "0" for qw/banners_num contexts_num bids_num prices_num/;
            push @{ $result{ $row->{par_type} } }, $row;
        }
    }
    $sth->finish();

    return \%result;
}

=head2 calc_queue_stats(%params)

    Посчитать статистику по объектам в очереди для каждого @STATS_PAR_TYPES и @STATS_NUMS
    Формат результата частично описан в DIRECT-50507 (как новое дерево).
    Примерный вид: {
        метрика из @STATS_NUMS {
            очередь из @STATS_PAR_TYPES => {
                by_percentile => {
                    перцентиль из @STATS_PERCENTILES => {
                        shard_X => value (метрики)
                    }
                },
            }
        }
    }

    Параметры именованные:
        shard   - для какого шарда получать статистику (опциональный, если не задан - используется $SHARDID)

    Результат:
        $queue_stats    - hasherf

=cut

sub calc_queue_stats {
    my %params = @_;
    my $shard = $params{shard} || $SHARDID;

    my $by_par_type = _get_queue_stats(shard => $shard);

    # сортируем данные по всем очередям
    for my $arr (values %$by_par_type) {
        @$arr = sort { $a->{age} <=> $b->{age} or $a->{cid} <=> $b->{cid} } @$arr;
    }

    my %stats;
    # аггрегируем по par_type и перцентилям
    for my $par_type (@STATS_PAR_TYPES) {
        my $data = $by_par_type->{$par_type} // [];

        for my $percentile (@STATS_PERCENTILES) {
            my @chunk = @$data ? @$data[0 .. int($#{$data} * $percentile / 100 + 0.5)] : ();

            for my $field (@STATS_NUMS) {
                # разрез про процентилям
                $stats{$field}->{$par_type}->{by_percentile}->{$percentile}->{"shard_$shard"} = _get_field_value($field, \@chunk);
            }
        }
    }
    # во избежание "не попадания" по ключам, т.к. структура сложная и глубокая
    Hash::Util::lock_hash_recurse(%stats);

    return \%stats;
}

=head3 _get_field_value($field, $arr)

    Вспомогательная функция к calc_flow_stats - получить значение метрики $field из массива данных $arr

=cut

sub _get_field_value {
    my ($field, $arr) = @_;
    my $value;

    if ($field eq 'max_age_minutes') {
        $value = @$arr ? int($arr->[-1]->{age} / 60) : 0;
    } elsif ($field eq 'max_sequence_minutes') {
        my $max_sequence_age = max(map { $_->{sequence_age} } @$arr);
        $value = @$arr ? int($max_sequence_age / 60) : 0;
    } elsif ($field eq 'campaigns_count') {
        my @cids = uniq map { $_->{cid} } @$arr;
        $value = scalar(@cids);
    } else {
        $value = sum(map { $_->{$field} } @$arr) // 0;
    }

    return $value;
}

=head2 get_simple_stat($stats, $shard)

    Преобразовать структуру со статистикой из формата calc_queue_stats в упрощенный (старый) формат

    Параметры:
        $stats  - результат calc_queue_stats
        $shard  - для какого шарда выделяем результат
    Результат:
        $simple     - статистика в старом формате:
                      только с std/heavy очередями
                      только с полями integral_size/max_age/campaigns_count
                      max_age - в секундах, для buggy - это sequence_age

=cut

sub get_simple_stat {
    my ($stats, $shard) = @_;

    my %simple = (
        std => {
            max_age => 60 * $stats->{max_age_minutes}->{std}->{by_percentile}->{100}->{"shard_$shard"},
            integral_size => $stats->{integral_size}->{std}->{by_percentile}->{100}->{"shard_$shard"},
            campaigns_count => $stats->{campaigns_count}->{std}->{by_percentile}->{100}->{"shard_$shard"},
        },
        heavy => {
            max_age => 60 * $stats->{max_age_minutes}->{heavy}->{by_percentile}->{100}->{"shard_$shard"},
            integral_size => $stats->{integral_size}->{heavy}->{by_percentile}->{100}->{"shard_$shard"},
            campaigns_count => $stats->{campaigns_count}->{heavy}->{by_percentile}->{100}->{"shard_$shard"},
        },
        buggy => {
            max_age => 60 * $stats->{max_sequence_minutes}->{buggy}->{by_percentile}->{100}->{"shard_$shard"},
            integral_size => $stats->{integral_size}->{buggy}->{by_percentile}->{100}->{"shard_$shard"},
            campaigns_count => $stats->{campaigns_count}->{buggy}->{by_percentile}->{100}->{"shard_$shard"},
        },
    );

    return \%simple;
}

# Переменные для настройки calc_suggested_workers_count
# Ниоткуда извне не должны устанавливаться, только в юнит-тесте читаются
# избыточные воркеры только для ручного управления, в авторасчёте скрыты через min()
our $WORKERS_STD_MAX = min(18, grep {$_->{type} eq 'std' && !$_->{preprod}} values %BS::Export::Queues::QUEUES);
our $WORKERS_STD_MIN = 5;
our $WORKERS_HEAVY_MAX = min(12, grep {$_->{type} eq 'heavy' && !$_->{preprod}} values %BS::Export::Queues::QUEUES);
our $WORKERS_HEAVY_MIN = 3;
our $WORKERS_BUGGY_MAX = grep {$_->{type} eq 'buggy' && !$_->{preprod}} values %BS::Export::Queues::QUEUES;
our $WORKERS_BUGGY_MIN = 1;
our $WORKERS_STD_HEAVY_MAX = 26;

=head2 calc_suggested_workers_count

    Посчитать рекомендуемое количество воркеров разного типа (std/heavy/buggy)
    Все значения захардкожены, как их задавать декларативно не очень понятно...

    Параметры:
      - ссылку на двухуровневый хэш с текущей статистикой:
          [std|heavy|buggy] -> [intergal_size|max_age] -> значение

    Выдаёт хэш с рекомендуемым числом воркеров:
      {std => XXX, heavy => YYY, buggy => ZZZ}

=cut

sub calc_suggested_workers_count {
    my ($stat) = @_;

    my $ret = {std => $WORKERS_STD_MAX, heavy => $WORKERS_HEAVY_MIN, buggy => $WORKERS_BUGGY_MIN};

    # считаем, сколько нужно std - обычная пропорция
    # max_age > 30min  ->  max
    # max_age < 5min   ->  max / 4
    $ret->{std} = int( 0.5 + interpolate_linear($stat->{std}->{max_age},
                                                { 'x' => 5*60,   'y' => $WORKERS_STD_MIN },
                                                { 'x' => 30*60,  'y' => $WORKERS_STD_MAX   },
                       ));

    if ($stat->{std}->{integral_size} > 100_000) {
        $ret->{std} = max($ret->{std}, int($WORKERS_STD_MAX*2/3 + 0.5));
    }

    # предварительно считаем, сколько нужно heavy - обычная пропорция
    # integral_size < 100_000  ->  $WORKERS_HEAVY_MIN+1
    # integral_size > 1_000_000  ->  max
    $ret->{heavy} = int( 0.5 + interpolate_linear($stat->{heavy}->{integral_size},
                                                { 'x' =>         0, 'y' => $WORKERS_HEAVY_MIN   },
                                                { 'x' =>   100_000, 'y' => min($WORKERS_HEAVY_MIN+1, $WORKERS_HEAVY_MAX) },
                                                { 'x' => 1_000_000, 'y' => $WORKERS_HEAVY_MAX   },
                       ));

    # но всего потоков должно быть не более $WORKERS_STD_HEAVY_MAX
    $ret->{heavy} = min($ret->{heavy}, $WORKERS_STD_HEAVY_MAX - $ret->{std});
    # считаем, сколько нужно buggy некоторой эвристикой
    $ret->{buggy} = $stat->{buggy}->{campaigns_count} > 25
        || $stat->{buggy}->{max_age} > 60 * 40
        ||  $stat->{buggy}->{integral_size} > 300_000
        ? $WORKERS_BUGGY_MAX
        : $WORKERS_BUGGY_MIN;

    return $ret;
}

=head2 can_resync_queue()

    По статистике $stat проверить - можно ли сейчас
    добавить в очередь данных из ленивой переотправки

    Результат:
        $can_resync - 0/1

=cut

sub can_resync_queue {
    my $stats = calc_queue_stats();

    my $std_integral_size = $stats->{integral_size_with_props}->{std}->{by_percentile}->{100}->{"shard_$SHARDID"};
    my $std_max_age = $stats->{max_age_minutes}->{std}->{by_percentile}->{100}->{"shard_$SHARDID"};
    my $heavy_integral_size = $stats->{integral_size_with_props}->{heavy}->{by_percentile}->{100}->{"shard_$SHARDID"};

    return $std_integral_size < _get_resync_isize_border()
           && $std_max_age < _get_resync_age_border()
           && $heavy_integral_size < _get_resync_heavy_isize_border()
           ;
}

1;
