package BS::ExportMobileContent;

=encoding utf8

=head1 NAME

    BS::ExportMobileContent

=head1 DESCRIPTION

    Функции для экспорта данных по мобильному контенту (приложениям) в БК. Раньше содержал код для отправки мобильного
    контента в специальную ручку БК, сейчас - только хелперы для обычного жкспорта в БК, остальной код уехал в Java

=cut

use Direct::Modern;

use Hash::Util qw/lock_hash lock_hash_recurse/;
use List::MoreUtils qw/none/;
use List::Util qw/min/;
use URI;

use Yandex::URL qw/get_host/;

use BS::Export;
use Direct::Model::MobileContent;
use MobileContent;
use Settings;

use constant ADJUST_INSTALL_CALLBACK_KEY => 'install_callback';
use constant ADJUST_CONVERSION_CALLBACK_KEY => 'conversion_callback';
use constant ADJUST_CALLBACK_DOMAIN_PATTERN => 'postback\\.yandexadexchange\\.net';


=head2 %TRACKERS_HOSTS

    типы трекеров приложений и специфичные для них параметры. Еще есть BS::Export::$TRANSLATE_PARAMS

=cut

my %TRACKERS_HOSTS_PARAMS = (
    mat_tune => {
        hosts => {
            '(^|\.)hastrk[0-9]{1,2}\.com$'   => 1,
            '(^|\.)api-[0-9]{1,2}\.com$'     => 1,
            '(^|\.)measurementapi\.com$' => 1,
            '(^|\.)tlnk\.io$'            => 1
        },
        #значение параметра => имя параметра. Логика проверки: есть такой макрос - подставлять не надо. Отсутствует - добавить в виде
        # value={key}, т.е. для trackid => 'publisher_ref_id' это будет publisher_ref_id={trackid}
        param_aliases => {
            ios_ifa => ['ios_ifa'],
            google_aid => ['google_aid'],
            trackid => ['publisher_ref_id']
        }
    },
    appsflyer => {
        hosts => {
            '^app\.appsflyer\.com$' => 1,
            '^impression\.appsflyer\.com$' => 1
        },
        param_aliases => {
            ios_ifa => ['idfa'],
            google_aid => ['advertising_id'],
            trackid => ['clickid'],
            oaid => ['oaid'],
            campaign_id => ['af_c_id']
        },
        # ключ => значение
        mandatory_params => {
            pid => 'yandexdirect_int'
        },
        impression_params => {
            af_ip => '{client_ip}',
            af_ua => '{user_agent}',
            af_lang => '{device_lang}'
        },
        s2s_enabled => 1,
        s2s_enabled_options => {
            qr/[\?&]af_(ios_url|r|android_url|dp)=[^&]*&?/ => 0,
        },
        s2s_href_params => {
            _common_ => {
                af_ip => '{client_ip}',
                af_ua => '{user_agent}',
                af_lang => '{device_lang}',
                redirect => 'false'
            },
            Android => {
                af_ref => 'YandexDirectInt_{TRACKID}'
            }
        },
        s2s_content_store_href_params => {
            Android => {
                referrer => 'YandexDirectInt_{TRACKID}'
            }
        }
    },
    appsflyerul => {
        hosts => {
            '\.onelink\.me$' => 1
        },
        param_aliases => {
            ios_ifa => ['idfa'],
            google_aid => ['advertising_id'],
            trackid => ['clickid'],
            oaid => ['oaid'],
            campaign_id => ['af_c_id']
        },
        mandatory_params => {
            pid => 'yandexdirect_int'
        },
        tracker_name => 'appsflyer'
    },
    adjust => {
        hosts => {
           '(^|\.)adjust\.com$'    => 1,
           '(^|\.)adjust\.io$' => 1
        },
        param_aliases => {
            ios_ifa => ['idfa'],
            google_aid => ['gps_adid'],
            oaid => ['oaid'],
            trackid => ['ya_click_id']
        },
        impression_params => {
            user_agent => '{user_agent}',
            ip_address => '{client_ip}',
            language => '{device_lang}'
        },
        s2s_enabled => 1,
        s2s_href_params => {
            _common_ => {
                user_agent => '{user_agent}',
                ip_address => '{client_ip}',
                language => '{device_lang}',
                s2s => '1'
            },
            Android => {
                external_click_id => 'YandexDirectInt_{TRACKID}'
            }
        },
        s2s_content_store_href_params => {
            Android => {
                referrer => 'adjust_external_click_id%3DYandexDirectInt_{TRACKID}'
            }
        }
    },
    adjustul => {
        hosts => {
           '(^|\.)adj\.st$' => 1
        },
        param_aliases => {
            ios_ifa => ['adj_idfa'],
            google_aid => ['adj_gps_adid'],
            oaid => ['adj_oaid'],
            trackid => ['adj_ya_click_id']
        },
        tracker_name => 'adjust'
    },
    appmetrica => {
        hosts => {
            '^redirect\.appmetrica\.yandex\.com$' => 1,
            '^redirect\.appmetrica\.yandex\.ru$'  => 1
        },
        param_aliases => {
            ios_ifa => ['ios_ifa'],
            google_aid => ['google_aid'],
            trackid => ['click_id']
        },
        s2s_enabled => 1,
        s2s_href_params => {
            _common_ => {
                device_ip => '{client_ip}',
                device_ua => '{user_agent}',
                click_timestamp => '{current_unixtime}',
                noredirect => '1'
            }
        },
    },
    appmetricaul => {
        hosts => {
            '^\d+\.redirect\.appmetrica\.yandex\.(com|ru)$' => 1
        },
        param_aliases => {
            ios_ifa => ['ios_ifa'],
            google_aid => ['google_aid'],
            trackid => ['click_id']
        },
        tracker_name => 'appmetrica',
    },
    kochava => {
        hosts  => {
            '(^|\.)kochava.com$'         => 1,
        },
        param_aliases => {
            ios_ifa    => ['ios_idfa'],
            #Для kocahva нужен то ли google_aid, то ли android_id -проверяем оба для надежности
            google_aid => ['adid'],
            android_id => ['android_id']
        }
    },
    flurry => {
        hosts => {
            '(^|\.)ad\.apps\.fm$' => 1,
            '(^|\.)flurry\.com$' => 1
        },
        param_aliases => {
            ios_ifa    => ['ios_idfa'],
            google_aid => ['adid'],
            trackid => ['click_id']
        }
    },
    mytracker => {
        hosts => {
            '^trk\.mail\.ru$' => 1
        },
        param_aliases => {
            ios_ifa => ['mt_idfa'],
            google_aid => ['mt_gaid'],
            trackid => ['clickId', 'regid']
        },
        s2s_enabled => 1
    },
    branch => {
        hosts => {
            '(^|\.)app.link$' => 1,
        },
        param_aliases => {
            ios_ifa => ['%24idfa'],
            google_aid => ['%24aaid'],
            trackid => ['~click_id']
        },
        mandatory_params => {
            '%243p' => 'a_yandex_direct'
        }
    },
    singular => {
        hosts => {
            '(^|\.)sng\.link$' => 1
        },
        param_aliases => {
            ios_ifa => ['idfa'],
            google_aid => ['aifa'],
            android_id => ['andi'],
            oaid => ['oaid'],
            trackid => ['cl']
        }
    }
);
lock_hash_recurse(%TRACKERS_HOSTS_PARAMS);

=head2 %DOMAIN_FILTER_BLACKLIST

    список популярных доменов для исключения их из DomainFilter + оригинальные package_name для исключений

=cut

my %DOMAIN_FILTER_BLACKLIST = (
    'akusherstvo.ru' => [],
    'cloud.mail.ru' => [],
    'facebook.com' => [],
    'firebaseapp.com' => [],
    'foodsoul.pro' => [],
    'foodsoul.ru' => [],
    'freeprivacypolicy.com' => [],
    'github.com' => [],
    'goo.gl' => [],
    'google.ad' => [],
    'google.com' => [],
    'google.ru' => [],
    'instagram.com' => ['com.instagram.android'],
    'pastebin.com' => [],
    'taximaster.ru' => [],
    'vk.com' => ['com.vkontakte.android'],
);
lock_hash_recurse(%DOMAIN_FILTER_BLACKLIST);

=head2 @SITELINK_TEXTS

    список пар заголовка и описания для генерации сайтлинков

=cut

my @SITELINK_TEXTS = (
    ['Рейтинг и отзывы', 'Посмотреть рейтинг приложения и отзывы'],
    ['О приложении', 'Описание приложения и скриншоты'],
    ['Что нового', 'Новая функциональность и доработки'],
    ['Дополнительная информация', 'Разработчик приложения, категория, системные требования'],
);

=head1 VARIABLES

=head2 $SQL_STORE_DATA_SELECT_FIELDS

    Часть SQL-запроса для выборки данных, идентифицирующих мобильный контент в сторе

=cut
our $SQL_STORE_DATA_SELECT_FIELDS = ', mobc.os_type, mobc.store_content_id, mobc.bundle_id';

=head2 INTERNAL VARIABLES

=head3 $log

    Объект Yandex::Log для логирования событий

=cut

my $log;
sub init {
    my %params = @_;
    $log = $params{log};
}

=head3 get_apptracker_from_domain(host)

    возвращает тип трекера приложений по хосту.

=cut

sub get_apptracker_from_domain {
    my $host = shift;

    foreach my $tracker_type (keys %TRACKERS_HOSTS_PARAMS) {
        my $tracker_info  = $TRACKERS_HOSTS_PARAMS{$tracker_type};
        my $tracker_hosts = $tracker_info->{hosts};
        for my $cur_host (keys %$tracker_hosts) {
            if ($host =~ /$cur_host/) {
                return $tracker_type;
            }
        }
    }
    return undef;
}

=head2 get_min_os_version_for_bs($targeting_version, $mobile_content_data)

    Получить минимальную версию ОС для отправки в БК.
    Выбирается наибольшая версия из заданной рекламодателем и данных из стора,
        если данных нет - выбирается максимальная известная версия ОС для типа
        контента. Если и ее определить не удалось - умирает.

    Для выбранной версии XX.YY результат V рассчитывается по следующей формуле:
        V = 1000 * XX + min(999, YY)

    Параметры:
        $targeting_version  - строка с минимальной версией, заданной рекламодателем
        $mobile_content_data    - hashref с данными о мобильном контенте, используются:
                                  min_os_version - данные о минимальной версии ОС из стора
                                  os_type - тип ОС, для получения дефолтной версии
    Результат:
        $min_os_version     - версия, приведенная к одному числу

=cut
sub get_min_os_version_for_bs {
    my ($targeting_version, $mobile_content_data) = @_;

    my $targeting_version_parts = MobileContent::get_version_parts($targeting_version);
    my $content_version_parts = MobileContent::get_version_parts($mobile_content_data->{min_os_version});

    my $maximum_parts = MobileContent::cmp_os_versions_by_parts($targeting_version_parts, $content_version_parts) >= 0
                            ? $targeting_version_parts
                            : $content_version_parts;

    if (!MobileContent::is_version_non_zero_by_parts($maximum_parts)) {
        my $maximum_os_version = $MobileContent::OS_VERSIONS{ $mobile_content_data->{os_type} };
        unless ($maximum_os_version) {
            die "No maximum OS version for os_type: $mobile_content_data->{os_type}";
        }
        $maximum_parts = MobileContent::get_version_parts($maximum_os_version);
    }

    my $min_os_version = 1_000 * $maximum_parts->[0] + min(999, $maximum_parts->[1]);

    return $min_os_version;
}

=head2 merge_content_store_data($context, $row)

    Добавляет в $context данные, идентифицирующие контент в сторе (имена полей - для UpdateData2)

    Параметры:
        $entry  - ссылка на хеш с данными по контексту для отправки в БК
        $row    - ссылка на хеш с исходными данными по мобильному контенту из базы

=cut
sub merge_content_store_data {
    my ($context, $row) = @_;
    @{ $context }{qw/ContentOSType ContentStoreName ContentStoreAppID/} = @{ _get_store_data($row) };
}

=head2 get_mobile_tracker_type($href)

    Возвращает строковое обозначение используемого трекера, либо undef, если трекер не удалось определить.

    Параметры:
        $href           - трекинговая ссылка
    Результат:
        Строковое обозначение используемого трекера

=cut
sub get_mobile_tracker_type {
    my $href = shift;
    return undef unless $href;

    my $processed_href = $href;
    my $tracker_domain = get_host($processed_href);
    return get_apptracker_from_domain($tracker_domain);
}

=head2 get_mobile_tracker_name($tracker_type)

    По обозначению трекера возвращает его название, если таковое имеется.

    Параметры:
        $tracker_type           - строковое обозначение трекера
    Результат:
        Название используемого трекера

=cut
sub get_mobile_tracker_name {
    my $tracker_type = shift;
    return '_unknown_' unless defined $tracker_type;
    if (exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{tracker_name}) {
        return $TRACKERS_HOSTS_PARAMS{$tracker_type}->{tracker_name};
    }
    return $tracker_type;
}

=head2 get_mobile_tracker($href)

    Возвращает название используемого трекера, либо _unknown_, если трекер не удалось определить.

    $banner->{MobileTrackerId} = get_mobile_tracker($row->{href});

    Параметры:
        $href           - трекинговая ссылка
    Результат:
        Название используемого трекера

=cut
sub get_mobile_tracker {
    my $href = shift;
    my $tracker_type = get_mobile_tracker_type($href);
    return get_mobile_tracker_name($tracker_type);
}

=head2 is_s2s_enabled($href)

    Возвращает признак того, включен ли S2S для трекерной ссылки

    my $s2s_enabled = is_s2s_enabled($row->{href});

    Параметры:
        $href           - трекинговая ссылка
    Результат:
        0/1 - включен ли s2s

=cut
sub is_s2s_enabled {
    my $href = shift;
    return 0 unless $href;

    my $tracker_type = get_mobile_tracker_type($href);
    if (defined $tracker_type) {
        if (exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_enabled_options}) {
            my $s2s_enabled_options = $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_enabled_options};
            foreach my $href_re (keys %$s2s_enabled_options) {
                if ($href =~ /$href_re/) {
                    return $s2s_enabled_options->{$href_re};
                }
            }
        }

        if (exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_enabled}) {
            return $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_enabled};
        }
    }
    return 0;
}

=head2 add_params_to_tracking_href($href)

    Подставляет специальные параметры в трекинговую ссылку в зависимости от домена трекера.

    $banner->{Href} = _prepare_href(add_params_to_tracking_href($row->{href}));

    Параметры:
        $href           - трекинговая ссылка
        $os_type        - мобильная ОС (iOS/Android)
        $href_type      - тип ссылки (0 - редиректная, 1 - s2s, 2 - impression)
    Результат:
        $processed_href - трекинговая ссылка, возможно с подствеленными параметрами

=cut
sub add_params_to_tracking_href {
    my ($href, $os_type, $href_type) = @_;
    my $processed_href = $href;

    return undef unless $href;

    my $tracker_domain = get_host($processed_href);
    my $tracker_type = get_apptracker_from_domain($tracker_domain);
    my $tracker_name = get_mobile_tracker_name($tracker_type);
    if (defined $tracker_type) {
        my $param_aliases = $TRACKERS_HOSTS_PARAMS{$tracker_type}->{param_aliases};
        if ($os_type eq 'Android') {
            $processed_href = _add_macro_param_if_needed("google_aid", $processed_href, $param_aliases);
            $processed_href = _add_macro_param_if_needed("android_id", $processed_href, $param_aliases);
            $processed_href = _add_macro_param_if_needed("oaid", $processed_href, $param_aliases);
        } elsif ($os_type eq 'iOS') {
            $processed_href = _add_macro_param_if_needed("ios_ifa", $processed_href, $param_aliases);
        } else {
            $log->die("bad mobile OS ".($os_type//'undef')." for href $href");
        }
        $processed_href = _add_macro_param_if_needed("campaign_id", $processed_href, $param_aliases);

        $processed_href = _replace_macro("logid", "trackid", $processed_href);
        $processed_href = _add_macro_param_if_needed("trackid", $processed_href, $param_aliases);

        if (exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{mandatory_params}) {
            my $mandatory_params = $TRACKERS_HOSTS_PARAMS{$tracker_type}->{mandatory_params};
            $processed_href = _add_mandatory_params($processed_href, $mandatory_params);
        }
        if ($href_type eq 1 && exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_href_params}) {
            my @param_keys = ('_common_', $os_type);
            foreach (@param_keys) {
                if (exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_href_params}->{$_}) {
                    $processed_href = _add_mandatory_params($processed_href, $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_href_params}->{$_});
                }
            }
        }
        if ($href_type eq 2 && exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{impression_params}) {
            $processed_href = _add_mandatory_params($processed_href, $TRACKERS_HOSTS_PARAMS{$tracker_type}->{impression_params});
        }

        if ($tracker_name eq 'adjust') {
            # разрешаем колбеки в сторонние сервисы, но не к нам
            $processed_href = remove_href_param($processed_href, ADJUST_INSTALL_CALLBACK_KEY, ADJUST_CALLBACK_DOMAIN_PATTERN);
            $processed_href = remove_href_param($processed_href, ADJUST_CONVERSION_CALLBACK_KEY, ADJUST_CALLBACK_DOMAIN_PATTERN);
        }
        if ($tracker_name eq 'appsflyer') {
            unless ($processed_href =~ /(^.*[\?&])c=[^&]+&?(.*$)/) {
                $processed_href = _add_mandatory_params($processed_href, {c => '{campaign_name}'})
            }
        }
    } else {
        $processed_href = _replace_macro("logid", "trackid", $processed_href);
    }

    return $processed_href;
}

=head2 add_params_to_s2s_store_href($href)

    Подставляет специальные параметры в ссылку на стор для S2S в зависимости от домена трекера.

    $banner->{ContentStoreHrefS2S} = _prepare_href(add_params_to_s2s_store_href($row->{store_content_href}));

    Параметры:
        $href           - ссылка на стор
    Результат:
        $processed_href - ссылка на стор для S2S, возможно с подствеленными параметрами

=cut
sub add_params_to_s2s_store_href {
    my ($href, $tracker_type, $os_type) = @_;
    my $processed_href = $href;

    return undef unless $href;

    if (defined $tracker_type && exists $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_content_store_href_params}) {
        my $params = $TRACKERS_HOSTS_PARAMS{$tracker_type}->{s2s_content_store_href_params};
        my @param_keys = ('_common_', $os_type);
        foreach (@param_keys) {
            if (exists $params->{$_}) {
                $processed_href = _add_mandatory_params($processed_href, $params->{$_});
            }
        }
    }

    return $processed_href;
}

sub _replace_macro {
    my ($old_macro, $new_macro, $href) = @_;
    $href =~ s/{$old_macro}/{$new_macro}/ig;
    return $href;
}

sub _add_macro_param_if_needed {
    my ($macro, $href, $param_aliases) = @_;
    my $processed_href = $href;
    if (exists $param_aliases->{$macro}) {
        my $param_keys = $param_aliases->{$macro};
        foreach my $param_key (@$param_keys) {
            $processed_href = enrich_href($processed_href, $param_key, "{$macro}");
        }
    }
    return $processed_href;
}

sub _add_mandatory_params {
    my ($href, $mandatory_params) = @_;
    my $processed_href = $href;
    foreach my $param_key (sort keys %$mandatory_params) {
        $processed_href = enrich_href($processed_href, $param_key, $mandatory_params->{$param_key});
    }
    return $processed_href;
}


=head3 remove_href_param($href, $param_key, $param_value_regexp_pattern)

    удаляет из ссылки параметр $param_key, если его значение содержит param_value_regexp_pattern
    если param_value_regexp_pattern не задан - удаляет параметр всегда

=cut

sub remove_href_param {
    my ($href, $param_key, $param_value_regexp_pattern) = @_;
    # удаляем все вхождения параметра, если есть
    my $pattern = "(^.*[\\?&])" . $param_key . "=[^&]*&?(.*\$)";
    if (defined $param_value_regexp_pattern) {
        $pattern = "(^.*[\\?&])" . $param_key . "=[^&]*" . $param_value_regexp_pattern . "[^&]*&?(.*\$)";
    }
    while ($href =~ /$pattern/) {
        $href =~ s/$pattern/$1$2/g;
    }

    # удаляем лишний ? или & в конце урла, если остался после удаления параметра
    if ($href =~ m/[\?&]$/) {
        chop($href);
    }

    return $href;
}

=head3 enrich_href($href, $param_key, $param_value)

    добавляет параметр к url

=cut


sub enrich_href {
    my ($href, $param_key, $param_value) = @_;
    $href = remove_href_param($href, $param_key);
    # и добавляем нужный
    my $additional_url_part = "$param_key=$param_value";
    if ($href =~ m/\?/) {
        return "$href&$additional_url_part";
    } else {
        return "$href?$additional_url_part";
    }
}

=head3 is_glued_domain($domain, $package_name)

    проверяет, является ли домен/пакет приклеенным к общеупотребимому

=cut

sub is_glued_domain {
    my ($domain, $package_name) = @_;
    my $domain_glued = _get_matched_blacklist_domain($domain);
    $domain_glued = _check_domain_against_original_package_names($domain_glued, $package_name);
    return $domain_glued;
}

sub _get_matched_blacklist_domain {
    my $domain = shift;
    foreach my $blacklisted_domain (keys %DOMAIN_FILTER_BLACKLIST) {
        if ($domain =~ /^(.*\.)?$blacklisted_domain$/i) {
            return $blacklisted_domain;
        }
    }
    return undef;
}

sub _check_domain_against_original_package_names {
    my ($domain, $package_name) = @_;
    if (defined $domain) {
        my $original_package_names = $DOMAIN_FILTER_BLACKLIST{$domain};
        for (@$original_package_names) {
            if ($package_name eq $_) {
                return undef;
            }
        }
    }
    return $domain;
}

=head3 generate_sitelinks($href)

    генерирует набор сайтлинков с одинаковой ссылкой

=cut

sub generate_sitelinks {
    my $href = shift;
    return map { +{
        Title => $_->[0],
        Description => $_->[1],
        Href => $href,
    } } @SITELINK_TEXTS;
}

=head2 INTERNAL SUBROUTINES

=head3 _get_store_data($row)

    Получить из $row данные, идентифицирующие контент в сторе
    Параметры:
        $row    - ссылка на хеш с исходными данными из базы
    Результат:
        $data   - ссылка на массив со следующими полями:
                    $os_type        - тип ОС
                    $store_name     - название стора
                    $app_id         - идентификатор приложения для БК

=cut
sub _get_store_data {
    my $row = shift;
    my @data;

    push @data, $row->{os_type};
    push @data, MobileContent::get_store_name_from_mobile_content($row);

    if (!defined $row->{os_type}
        || none { $_ eq $row->{os_type} } @{Direct::Model::MobileContent->meta->find_attribute_by_name('os_type')->params->{values}}) {

        $log->die("undefined mobile OS type " . join "-", map { $_ // '' } @{$row}{qw/os_type store_content_id bundle_id/});
    }
    my $store_app_id = Direct::Model::MobileContent->get_store_app_id($row);
    push @data, $store_app_id;

    return \@data;
}


=head3 get_app_data($banner_mobile_content, $banner_mobile_app)

    Формирует информацию о приложении

=cut
sub get_app_data {
    my ($banner_mobile_content, $banner_mobile_app) = @_;

    my $mobile_content_source = {
        MOBILE_CONTENT_INVALID => 0,
        MOBILE_CONTENT_GOOGLE_PLAY_STORE => 1,
        MOBILE_CONTENT_APPLE_APP_STORE => 2,
    };

    my $url = URI->new($banner_mobile_app->{store_href});
    my $is_bad_url = not ($url->scheme and $url->host);
    my %url_params = $url->query_form();
    my @url_path_segments = $url->path_segments();

    my $regionName = "ru";
    my $localeName = "ru";
    my $bundleId = "";
    my $source_id = $mobile_content_source->{MOBILE_CONTENT_INVALID};
    if ($banner_mobile_content->{os_type} eq 'Android') {
        $source_id = $mobile_content_source->{MOBILE_CONTENT_GOOGLE_PLAY_STORE};
        if (defined $url_params{gl}) {
            $regionName = lc $url_params{gl};
        }
        if (defined $url_params{hl}) {
            $localeName = lc $url_params{hl};
        }
        $bundleId = $banner_mobile_content->{store_content_id};
    } elsif ($banner_mobile_content->{os_type} eq 'iOS') {
        $source_id = $mobile_content_source->{MOBILE_CONTENT_APPLE_APP_STORE};
        if (not $is_bad_url) {
            foreach my $segment (@url_path_segments) {
                if (lc $segment eq "app") {
                    last;
                }
                if ($segment ne "") {
                    $regionName = lc $segment;
                    last;
                }
            }
        }

        if (defined $url_params{l}) {
            $localeName = lc $url_params{l};
        }
        if (defined $banner_mobile_content->{bundle_id}) {
            $bundleId = $banner_mobile_content->{bundle_id};
        }
    }

    my $app_data = {
        BundleId => $bundleId,
        SourceID => $source_id,
        RegionName => $regionName,
        LocaleName => $localeName
    };

    return $app_data;
}


1;
