package BS::ExportWorker;

=encoding utf8

# $Id$

=head1 NAME

BS::ExportWorker - Функции для экспорта данных и цен в БК.

=head1 DESCRIPTION

    Модуль содержит различные функции следующего назначения:
        логирование
        получение данных из базы
        работа с кампаниями в очереди
        обработка ответов
        логика, специфичная для разнных типов "потоков"
    Функционал используется в bsClientData.pl.

=cut

use Direct::Modern;

use Readonly;
use JSON;
use List::MoreUtils qw/uniq any none/;
use List::UtilsBy qw(partition_by);
use List::Util qw(max min);
use POSIX qw/strftime/;
use Time::HiRes qw//;
use Hash::Util;

use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::ListUtils;
use Yandex::Log;
use Yandex::ScalarUtils;
use Yandex::Trace;

use EnvTools;
use BS::Export qw/:sql :limits get_value_for_sum is_strategy_roi_or_crr is_strategy_cpa get_target_type get_bs_expression_for_modifier get_strategy_mobile_goals $SQL_BIDS_BASE_TYPE_ENABLED @RELEVANCE_MATCH_BIDS_BASE_TYPES/;
use BS::ExportMobileContent;
use BS::ExportWorker::LogBrokerBufferNewProto;
use BS::ResyncQueue;
use Campaign;
use Client;
use Common qw//;
use Direct::Validation::HierarchicalMultipliers qw(
    $DEMOGRAPHY_MULTIPLIER_AGES
    $DEMOGRAPHY_MULTIPLIER_GENDERS
    $ALLOWED_DEMOGRAPHY_MULTIPLIER_AGES
    %DEPRECATED_DEMOGRAPHY_MULTIPLIER_AGE
);
use Direct::TurboLandings;
use Direct::AdditionsItemDisclaimers;
use Direct::BannersAdditions;
use Direct::BillingAggregates;
use Direct::Banners::Measurers qw//;
use Direct::Banners::AdditionalHrefs qw//;
use DirectRedis;
use AggregatorDomains qw//;
use MinusWordsTools;
use RedisLock;
use Sitelinks ();
use Settings;
use WalletUtils;
use Campaign::Types;
use Property;
use JavaIntapi::GetCampaignsAssetHashes;

use base qw/Exporter/;
our @EXPORT = qw/
                lock_campaigns_in_queue
                wait_synchronization
                unlock_campaigns
                set_sync_campaigns
                set_banner_permalinks_sent_to_bs
            /;

=head2 $DELAY_TIME

    Время, на которое сдвигаем (по seq_time) в очереди кампании из %BS::Export::CIDS_FOR_DELAY при их разблокировании.
    Формат MySQL INTERVAL expr, без указания ключевого слова INTERVAL

=cut

our $DELAY_TIME = '2 MINUTE';

=head2 $LOCK_NAME

    префикс именованного лока mysql

=cut

our $LOCK_NAME = 'BS_EXPORT';

=head2 $LOG_FILE_PREFIX

    префикс имени файла для логирования запросов и ответов

=cut

our $LOG_FILE_PREFIX ||= 'bsexport';

=head2 $LOG_DIVIDER

    сколько лог-файлов на каждый день создаётся (остаток от деления номера кампании на divider)

=cut

our $LOG_DIVIDER ||= 10;

=head2 $MIN_ITERATION_DURATION

    минимальное время итерации
    если итерация закончилась быстрее - делаем sleep

=cut

our $MIN_ITERATION_DURATION ||= 10;

=head2 $DONT_MOVE_CAMPS_TO_BUGGY

    Флажок для отключения перемещения кампаний в buggy-очередь.

=cut

our $DONT_MOVE_CAMPS_TO_BUGGY;

=head2 $SUM_MIGRATION_COOLDOWN_SECONDS

    В течение скольки секунд еще нужно пытаться взять Redis лок на общие счета,
    после того, как была выключена возможность мигрировать их в новую схему
    учета зачислений.

=cut

our $SUM_MIGRATION_COOLDOWN_SECONDS = 15 * 60;

=head2 $SUM_MIGRATION_LOCK_TTL_SECONDS

    TTL Redis лока на общие счета, которые могут мигрироваться в новую схему
    учета зачислений.

=cut

our $SUM_MIGRATION_LOCK_TTL_SECONDS = 10 * 60;

=head2 $LB_EXPORT_IMAGES_LIMIT

    Максимум картиночных баннеров, которые берем в пачку отправки

=cut

our $LB_EXPORT_IMAGES_LIMIT //= 25_000;

=head2 $LB_EXPORT_PHRASES_LIMIT

    Максимум фраз, которые берем в пачку отправки для реэкспорта

=cut

our $LB_EXPORT_PHRASES_LIMIT //= 200_000_000;

=head2 LOGBROKER_DISABLED_CACHE_LIFETIME

    На сколько секунд кешировать выключатель отправки данных в БК через logbroker

=cut

use constant LOGBROKER_DISABLED_CACHE_LIFETIME => 5*60;

=head2 LOGBROKER_SOURCE_ID_DIVIDER

    На сколько source_id бить отправляемые в logbroker
    См. _get_source_id
    должно быть не больше, чем число партиций в kafka-топике
    https://st.yandex-team.ru/LOGBROKER-1548#1462451215000

=cut

use constant LOGBROKER_SOURCE_ID_DIVIDER => 26;

# Сдвиг относительно cid для получения OrderID
use constant ORDER_ID_OFFSET => 100_000_000;

=head2 log_data

    логирование запроса и ответа UpdateData2

=cut

sub log_data {
    my ($start_tag, $req) = @_;
    my $profile = Yandex::Trace::new_profile('bs_export_worker:log_data');

    my $json = JSON->new()->allow_blessed()->convert_blessed();
    no warnings 'once';
    local *SOAP::Data::TO_JSON = sub {return $_[0]->value};

    my %LOG;
    for my $order (values %{$req->{ORDER}}) {
        my $log_arr = $LOG{$order->{EID} % $LOG_DIVIDER} ||= [];
        my @tags = ("cid=$order->{EID}", $start_tag);
        push @$log_arr, join(',', @tags)."\t".$json->encode(hash_kgrep {!/CONTEXT/} $order);
        for my $context (values %{$order->{CONTEXT}}) {
            unshift @tags, "pid=$context->{EID}";
            push @$log_arr, join(',', @tags)."\t".$json->encode(hash_kgrep {!/BANNER/} $context);
            for my $banner (values %{$context->{BANNER}}) {
                push @$log_arr, "bid=$banner->{EID},".join(',', @tags)."\t".$json->encode($banner);
            }
            shift @tags;
        }
    }

    _write_to_logs('data', \%LOG);
}

# вспомогательная процедура для записи данных в файлы
sub _write_to_logs {
    my ($suffix, $log_by_rem) = @_;

    state $prop = Property->new('BSEXPORT_DISABLE_DATA_LOG');
    return if $prop->get(60);

    for my $rem (sort {$a <=> $b} keys %$log_by_rem) {
        my $log = Yandex::Log->new(log_file_name => "$LOG_FILE_PREFIX.$suffix.$rem", date_suf => "%Y%m%d", lock => 1);
        $log->out(@{$log_by_rem->{$rem}});
    }
}


my ($dbh, $dbh2, $PAR_ID, $PAR_TYPE, $PAR_NORM_NICK, $log, $error_logger, $shard);
# массив хеше с данными для переотправки (добавления в bs_resync_queue) по окончании итерации
my @RESYNC_DATA;

=head3 $DEBUG_MODE

    Флаг, выключающий некоторые модифицирующие состояние в базе операции - например работу с очередью.
    А также исключающий таблицу bs_export_queue из join'ов.

    Выставляется из bsClientData, там же есть более подробное описание - о предназначении и влиянии.

=cut

my $DEBUG_MODE;

=head3 $IGNORE_BS_SYNCED

    Флаг, что нужно игнорировать статусы синхронизации объектов и брать все подходящие под другие условия

=cut

my $IGNORE_BS_SYNCED;

=head3 $LOAD_AUTOBUDGET_RESTART

    Нужно ли загружать из базы последнее подсчитанное время рестарта автобюджета (для full-lb-export)

=cut

my $LOAD_AUTOBUDGET_RESTART;

=head2 init(%options)

    Первоначальная установка переменных, а также сброс "итерационных" переменных
    %options:
        dbh - коннект к БД (БД в которую будет записан результат синхронизации с БК)
        dbh2 - коннект к БД (в этой БД будет создана временная таблица с данными, которые будут отправлены в БК)
        parid - номер очереди в БК (для bsClientData.pl)
        partype - тип очереди в БК (стандартная, заказов, цен ....)
        par_norm_nick - описание очереди (для логгирования)
        log - объект Yandex::Log
        error_logger - обертка для Yandex::Log::Messages
        load_autobudget_restart - нужно ли загружать данные из таблицы camp_autobudget_restart

=cut

sub init {
    my %options = @_;
    ($dbh, $dbh2, $PAR_ID, $PAR_TYPE, $PAR_NORM_NICK, $log, $error_logger, $IGNORE_BS_SYNCED, $LOAD_AUTOBUDGET_RESTART, $shard) =
        @options{qw/dbh dbh2 parid partype par_norm_nick log error_logger ignore_bs_synced load_autobudget_restart shard/};
    $BS::Export::ITERATION_FAILED = 0;
    %BS::Export::MOVE_CIDS_TO_BUGGY = ();
    %BS::Export::CIDS_FOR_DELAY = ();
    @RESYNC_DATA = ();
    $DEBUG_MODE = $options{debug_mode} ? 1 : 0;
}

=head2 wait_synchronization()

    Обертка над wait_master. Дожидается, когда $dbh2 догонит $dbh до позиции на момент вызовова.
    Ждет с таймаутом 600 секунд, не дождавшись - умирает.

    С отказом от bs и heavy реплик - потеряло актуальность

=cut

sub wait_synchronization {
    return wait_master([$dbh, $dbh2], 600);
}

=head2 is_par_type (@par_types)

    Проверить, совпадает ли тип текущей очереди с одним из перечисленных

=cut

sub is_par_type {
    my $par_type;
    if ($PAR_TYPE =~ /^(\D+)\d+$/) {
        $par_type = $1; # отрезаем цифру от очередей dev1/dev2
    } else {
        $par_type = $PAR_TYPE;
    }
    return any {$par_type eq $_} @_;
}

=head2 actions_on_iteration_end($camps)

    Выполнить необходимые действия (в том числе с кампаниями, оставшимися в $camps)
    по окончанию итерации:
      если массив @RESYNC_DATA не пустой - добавляем данные в ленивую переотправку
        и если в массиве было больше 100 записей на кампанию - еще закидываем ее в buggy
      разлочить все кампании, которые остались залоченными

=cut

sub actions_on_iteration_end {
    # JAVA-EXPORT: BsExportIterationContext#close
    my $camps = shift;

    my @all_resync;
    if (@RESYNC_DATA) {
        my %count_by_cid;
        for my $row (@RESYNC_DATA) {
            push @all_resync, $row;
            $count_by_cid{ $row->{cid} }++;
        }
        # DIRECT-60340: закидывать в buggy-очередь кампании с больши числом "лениво переотправляемых" попозже баннеров
        for my $cid (keys %count_by_cid) {
            if ($count_by_cid{$cid} > 100) {
                BS::Export::buggy_cid($cid);
                $error_logger->({
                        message       => "Too many resync data for campaign",
                        cid           => $cid,
                        type          => "buggy",
                        stage         => "response",
                        resync_records_count => $count_by_cid{$cid},
                    });
            }
        }
    }

    if (@all_resync) {
        bs_resync(\@all_resync);
    }

    if (is_par_type('buggy')) {
        if ($BS::Export::ITERATION_FAILED) {
            # двигаем все кампании дальше в очереди
            for my $cid (keys %$camps) {
                BS::Export::delay_cid($cid);
            }
        }
    } else {
        my @cids_to_move;
        if ($BS::Export::ITERATION_FAILED) {
            # переставить все
            @cids_to_move = keys(%$camps);
        } elsif (%BS::Export::MOVE_CIDS_TO_BUGGY) {
            # переставить только эти
            @cids_to_move = keys(%BS::Export::MOVE_CIDS_TO_BUGGY);
        }
        move_buggy_campaigns(\@cids_to_move);
    }
    # разлочим все, что осталось
    # JAVA-EXPORT: BsExportIterationContext#unlockAllCampaigns
    unlock_campaigns($camps, all => [keys %$camps]);
}

=head2 lock_campaigns_in_queue(%options)

    Выборка и блокировка кампаний из очереди для синхронизации с БК.
    Отбор кампаний происходит по типу обслуживаемой очереди BS::ExportWorker::init(partype => )

    Параметры именованные:
        cids    - ссылка на массив кампаний для ограничения выборки
        limit   - ограничение на количество блокируемых кампаний.
                  по умолчанию 2 * $BS::Export::CAMPS_LIMIT
        already_locked_only - флажок (в булевом смысле), отменяющий блокировку новых кампаний
                              при этом возвращаются уже заблокированные
        skip_locked_wallets - флажок (в булевом смысле), отменяющий блокировку новых кампаний,
                              расположенных под одним общим счетом с уже заблокированными
                              другими потоками кампаниями. по умолчанию включен.
    Результат:
    {
        cid => {
            # количество объектов, ожидающих синхронизации
            prices_num, camps_num, banners_num, contexts_num, bids_num
            # флаги "что нужно отправлять"
            send_data, send_other, send_camp,
            # время обработки в очереди
            seq_time,
        },
        cids_removed_from_specials => [
            #список cid-ов удаленных из таблицы specials. Нужно чтобы можно было понять сделана ли полезная работа
            cid1, cid2...
        ]
    }

=cut

sub lock_campaigns_in_queue {
    # JAVA-EXPORT: (BsExportQueueService#lockCampaignsInQueue)
    my %options = @_;

    # Если опция не определена, считаем её включенной
    $options{skip_locked_wallets} //= 1;

    $options{limit} ||= 2 * $BS::Export::CAMPS_LIMIT;

    ################################
    # выбираем все подходящие заказы

    # проверка _specials
    #JAVA-EXPORT(BsExportQueueRepository#getExportSpecialsCondition)
    my $SPEC_SQL;
    if (is_par_type( qw{ fast heavy dev buggy preprod camps_only } )) {
        # берут только свои заказы
        $SPEC_SQL = "s.par_type = '$PAR_TYPE'";
    } elsif (is_par_type("camp")) {
        # берут и свои, и чужие заказы (кроме разработческих и совсем специальных)
        $SPEC_SQL = BS::Export::get_sql_where_exclude_special_par_types({ additional_par_types => [qw/preprod camps_only/] });
    } elsif (is_par_type("internal_ads_dev")) {
        # берут только заказы из соответствующей dev очереди
        my $internal_dev_par_type = $PAR_TYPE;
        $internal_dev_par_type =~ s/^internal_ads_dev/dev/;
        $SPEC_SQL = "s.par_type = '$internal_dev_par_type'";
    } elsif (is_par_type(qw/std internal_ads/)) {
        # берут только не-специальные заказы
        $SPEC_SQL = "s.par_type IS NULL";
    } elsif (is_par_type(qw/full_lb_export/)) {
        # берут любые заказы (кроме разработческих и совсем специальных)
        # NB! эта же логика продублирована при отборе кампаний для статистики в cmd_bsExportQueue
        $SPEC_SQL = BS::Export::get_sql_where_exclude_special_par_types();
    } else {
        die "No spec_sql for $PAR_TYPE";
    }

    my $INTERNAL_ADS_SQL;
    #JAVA-EXPORT: вместо этого SQL используется надтип internal (BsExportQueueRepository#getInternalAdsCondition)
    # являетсяя ли заказ внутренней рекламой (должен отправляться с другим EngineID)
    if (is_par_type(qw/internal_ads internal_ads_dev/)) {
        $INTERNAL_ADS_SQL = $SQL_INTERNAL_CAMPAIGN_TYPES;
    } elsif (is_par_type(qw/full_lb_export/)) {
        $INTERNAL_ADS_SQL = 1;
    } else {
        $INTERNAL_ADS_SQL = $SQL_NOT_INTERNAL_CAMPAIGN_TYPES
    }

    # проверка специализации очереди
    #JAVA-EXPORT(BsExportQueueRepository#getWorkerSpecializationCondition)
    my $TYPE_SQL;
    my @par_types_without_type_sql = qw/heavy fast buggy/;
    if (is_par_type("camp")) {
        # при подключенном общем счете, "кампания счет" уже должна уйти в БК, иначе не отправляем
        # еще отправляем новые (без OrderID) кампании-кошельки в БК
        $TYPE_SQL = "
            q.camps_num > 0
            AND ((c.type = 'wallet' AND c.OrderID = 0)
                 OR (c.statusActive = 'Yes'
                     AND (c.statusShow = 'No'
                          OR (c.sum + IFNULL(wc.sum, 0) <= c.sum_spent + IFNULL(wc.sum_spent, 0))
                         )
                    )
                )
        ";
    } elsif (is_par_type( qw{ std dev internal_ads internal_ads_dev} )) {
        $TYPE_SQL = "q.camps_num > 0 OR q.banners_num > 0 OR q.contexts_num > 0 OR q.bids_num > 0";
    } elsif (is_par_type(@par_types_without_type_sql, 'preprod')) {
        # JAVA-EXPORT: кажтся тут баг, нужно не брать из очереди записи у которых только is_full_export
        $TYPE_SQL = "1";
    } elsif (is_par_type(qw/full_lb_export/)) {
        $TYPE_SQL = "q.is_full_export = 1";
    } elsif (is_par_type(qw/camps_only/)) {
        $TYPE_SQL = "q.camps_num > 0";
    } else {
        die "No type_sql for $PAR_TYPE";
    }

    # выполняем запрос
    $log->out(sprintf("start select cids for lock with options: only_cids - %s, locked_only - %s, skip_locked_wallets - %s, limit: %d",
                      $options{cids} ? 'yes' : 'no',
                      $options{already_locked_only} ? 'yes' : 'no',
                      $options{skip_locked_wallets} ? 'yes' : 'no',
                      $options{limit},
                      ));
    my $CIDS_ONLY_SQL = $options{cids}
        ? sprintf 'AND c.cid IN (%s)', join ',', @{$options{cids}}
        : '';

    # чтобы схема с избыточной блокировкой хорошо работала и на маленьких лимитах
    # если buggy-потоков будет больше одного - стоит увеличить запас
    # JAVA-EXPORT: задавать 4ку явно только для BUGGY-воркеров
    my $lock_limit = max($options{limit} + 4, $options{limit});

    # выбираем только залоченные нами заказы
    my $ORDERS_TO_LOCK_SQL = '0';
    if (!$options{already_locked_only}) {
        # будем выбирать еще не залоченные никем заказы

        #JAVA-EXPORT: (BsExportQueueRepository#getLockedWalletsCondition)
        my $LOCKED_WALLETS_SQL = "1";

        #JAVA-EXPORT TODO: is_par_type делать снаружи, и на его основании ставить флаг skip_locked_wallets
        if ($options{skip_locked_wallets} && !is_par_type(qw/full_lb_export/)) {
            my $locked_wallets_limit = 10 * $lock_limit;

            my $par_id_by_nick = BS::Export::Queues::get_par_id_by_nick();
            my @nicks = grep {
                $_ =~ /^(full_lb_export)/
            } keys %{$par_id_by_nick};
            my @irrelevant_par_ids = @{$par_id_by_nick}{@nicks};
            push @irrelevant_par_ids, $PAR_ID;

            # выбираем wallet_cid-ы заказов, заблокированных не нами
            #JAVA-EXPORT: (BsExportQueueRepository#getLockedWallets)
            my $locked_wallets = get_one_column_sql($dbh, ["
                                        SELECT distinct IF(c.type = 'wallet', c.cid, c.wallet_cid)
                                          FROM bs_export_queue q
                                               JOIN campaigns c on c.cid = q.cid",
                                         WHERE  => {
                                            "q.par_id__is_not_null" => 1,
                                            "q.par_id__not_in" => \@irrelevant_par_ids,
                                            _OR => {"c.type" => 'wallet', "c.wallet_cid__gt" => 0},
                                            "q.camps_num__gt" => 0,
                                            _NOT => {"c.statusBsSynced" => 'Yes'}
                                        },
                                        LIMIT => $locked_wallets_limit]);
            if (@$locked_wallets) {
                $LOCKED_WALLETS_SQL = "
                            ifnull(c.wallet_cid, 0) = 0 and c.type != 'wallet'
                            OR IF(c.type = 'wallet', c.cid, c.wallet_cid) not in (".join(',', @$locked_wallets).")
                           ";
            }
        }

        $ORDERS_TO_LOCK_SQL = "( q.par_id is null
                                 AND ($LOCKED_WALLETS_SQL)
                                 AND ($SPEC_SQL)
                                 AND ($INTERNAL_ADS_SQL)
                                 AND ($TYPE_SQL)
                                 AND ( $BS::Export::NO_WALLET_OR_WALLET_HAS_ORDERID )
                               )";
    }

    # в первую очередь, получаем уже заблокированные нами заказы
    # поэтому хитрый order by if()
    my $SEQ_TIME_COLUMN = is_par_type('full_lb_export') ? 'full_export_seq_time' : 'seq_time';
    #JAVA-EXPORT: (BsExportQueueRepository#getCandidatesForExport)
    #JAVA-EXPORT TODO: получать wwc.wallet_cid, wwc.is_sum_aggregated отдельным запросом
    my $camps = get_all_sql($dbh, "
                            SELECT q.cid, q.camps_num, q.banners_num, q.contexts_num, q.bids_num, c.statusBsSynced as c_statusBsSynced
                                 , q.is_full_export, wwc.wallet_cid, wwc.is_sum_aggregated
                              FROM bs_export_queue q
                                   JOIN campaigns c on c.cid = q.cid
                                   LEFT JOIN campaigns wc on c.wallet_cid = wc.cid
                                   LEFT JOIN bs_export_specials s ON s.cid = q.cid
                                   LEFT JOIN wallet_campaigns wwc ON wwc.wallet_cid = IF(c.type = 'wallet', c.cid, c.wallet_cid)
                             WHERE (q.par_id = ?
                                        OR $ORDERS_TO_LOCK_SQL
                                   ) $CIDS_ONLY_SQL

                             ORDER BY IF(q.par_id = ?, 0, 1),
                                      $SEQ_TIME_COLUMN,
                                      queue_time,
                                      q.cid
                             LIMIT ?
                             ",
                             $PAR_ID, $PAR_ID, $lock_limit,
        );

    # по cid'ам из этих хешей будем выставлять флажки send_data и send_camp
    # означающие, что хотим отправлять цены, данные кампании, или саму кампанию
    my (%data_cids, %camp_cids);
    # для флажка "отправить целиком" - send_full
    my %full_cids;
    # по cid'ам из этого массива будем пытаться залочить кампании своим parid
    # и по нему же оставлять кампании, если залочили больше лимита (в нем правильный порядок)
    my @all_cids_to_attempt_lock;
    # берём, сколько сможем
    # общее условие - заполненность не меньше чем на половину, не больше, чем X
    my ($camps_num, $banners_num, $contexts_num, $bids_num) = (0, 0, 0, 0);
    # номера кампаний, у которых нужно удалить запись и _specials (не подходят по условиям)
    my @delete_from_specials;
    # номера кампаний, у которых есть ОС, который еще не перешел на новую схему учета зачислений.
    # Их могут начать мигрировать в любой момент, поэтому берем локи на их ОС: DIRECT-86129
    my %camps_with_separate_sums;
    # словарь соответствия cid кампании id ее общего счета, если он есть.
    # если кампания является общим счетом, значением является его cid
    my %camp_wallets;

=head2 TODO

    Навести здесь порядок.
    Сейчас в одну кучу смешаны следующее:
        знание какой par_type что отправляет
        знание о лимитах для каждого par_type (в том числе это выражается в том,
          что часть потоков не проверяет остальные лимиты, т.е. они эффективно равны 0)
        разное поведение при заполнении лимитов на данные, цены и кампании:
          при заполнении лимита на кампании - продолжаем брать данные
          при заполнении лимита на баннеры - делаем last и вообще ничего больше не проверяем
          при заполнении лимита не цены - больше не берем цены

    + добавить логику, что если ничего еще не выбрали - то первый объект берем несмотря на лимиты
    т.к. уже встречаются кампании, которые не пролезают по условию
        $banners_num + $c->{banners_num} < 3*$BANNERS_LIMIT
    и их "спасает" только несинхронность самой кампании (camps_num != 0)

=cut

    for my $c (@$camps) {
        my $cid = $c->{cid};
        if (defined($c->{is_sum_aggregated}) && $c->{is_sum_aggregated} eq 'No') {
            $camps_with_separate_sums{$cid} = undef;
        }
        if (defined($c->{wallet_cid})) {
            $camp_wallets{$cid} = $c->{wallet_cid};
        }
        if (is_par_type('full_lb_export')) {
            # статистики в очереди нет, поэтому ограничиваемся только кампаниями
            if ($c->{is_full_export} && $camps_num < $CAMPS_LIMIT) {
                push @all_cids_to_attempt_lock, $cid;
                $full_cids{$cid} = undef;
                $camps_num++;
            }
            # весь остальной код в этом цикле - нерелевантен.
            next;
        }
        if (is_par_type(@par_types_without_type_sql)
            && none { $c->{$_} } qw/camps_num banners_num contexts_num bids_num/
        ) {
            push @delete_from_specials, $cid;
            next;
        }
        # кампании
        if (is_par_type( qw{ camp std heavy fast dev buggy preprod camps_only internal_ads internal_ads_dev } )) {
            if ($c->{camps_num} && $camps_num < $CAMPS_LIMIT) {
                push @all_cids_to_attempt_lock, $cid;
                $camp_cids{$cid} = undef;
                $camps_num++;
            }
        }
        # данные
        if (is_par_type( qw{ camp std heavy fast dev buggy preprod internal_ads internal_ads_dev } )) {
            if ($c->{banners_num} || $c->{contexts_num} || $c->{bids_num}) {
                if (
                    ($banners_num < $BANNERS_LIMIT / 2 || $banners_num + $c->{banners_num} < 3*$BANNERS_LIMIT)
                    &&
                    ($contexts_num < $CONTEXTS_LIMIT / 2 || $contexts_num + $c->{contexts_num} < 3*$CONTEXTS_LIMIT)
                    &&
                    ($bids_num < $BIDS_LIMIT / 2 || $bids_num + $c->{bids_num} < 3*$BIDS_LIMIT)
                ) {
                    $banners_num += $c->{banners_num};
                    $contexts_num += $c->{contexts_num};
                    $bids_num += $c->{bids_num};
                    push @all_cids_to_attempt_lock, $cid;
                    $data_cids{$cid} = undef;
                } else {
                    last;
                }
            }
        }
    }
    $log->out("selected cids stats: camps: $camps_num, banners: $banners_num, contexts: $contexts_num, bids: $bids_num");
    # кампании, неподходящие текущему типу по условиям удаляем из _specials
    if (@delete_from_specials) {
        $log->out("delete from _specials cids: ".join(',', @delete_from_specials));
        do_sql($dbh, ["DELETE FROM bs_export_specials", WHERE => {cid => \@delete_from_specials}]);
        $error_logger->([map{{
                    message => "cid is deleted from bs_export_specials in lock_campaigns_in_queue",
                    cid     => $_,
                    type    => "remove_from_specials",
                }} @delete_from_specials]);
    }
    @all_cids_to_attempt_lock = uniq(@all_cids_to_attempt_lock);

    # список кампаний, у которых хотим отправлять данные, и которые могут мигрироваться в новую схему учета зачислений
    my @cids_to_lock_sum_migration = grep {
            exists $camps_with_separate_sums{$_} && (exists $data_cids{$_} || exists $camp_cids{$_})
        } @all_cids_to_attempt_lock;
    my $sum_migration_locks = {};
    if (@cids_to_lock_sum_migration && _is_sum_migration_possible()) {
        my $cids_failed_locks;
        ($sum_migration_locks, $cids_failed_locks) = _try_lock_sum_migration(\@cids_to_lock_sum_migration, \%camp_wallets);
        @all_cids_to_attempt_lock = @{xminus(\@all_cids_to_attempt_lock, $cids_failed_locks)};
    }

    my %result = (camps => {}, cids_removed_from_specials => \@delete_from_specials);
    if (@all_cids_to_attempt_lock) {
        # пытаемся залочиться
        $log->out("start lock cids: ".join(",", @all_cids_to_attempt_lock));
        do_sql($dbh, ["UPDATE bs_export_queue SET par_id = ? WHERE (par_id is null OR par_id = ?) AND ", {cid => \@all_cids_to_attempt_lock}], $PAR_ID, $PAR_ID);
        if (%$sum_migration_locks) {
            _unlock_sum_migration($sum_migration_locks);
        }
        my @QUEUE_CIDS_ONLY_SQL = ( $options{cids} && @{$options{cids}} ) ? (cid => $options{cids}) : ();
        my $SEQ_TIME_COLUMN = is_par_type('full_lb_export') ? 'full_export_seq_time' : 'seq_time';
        my $locked = get_hashes_hash_sql($dbh, ["
                                                SELECT cid, sync_val, $SEQ_TIME_COLUMN AS seq_time,
                                                       camps_num, banners_num, contexts_num, bids_num
                                                     , is_full_export
                                                  FROM bs_export_queue
                                              ", WHERE => {par_id => $PAR_ID, @QUEUE_CIDS_ONLY_SQL},
                                            ]);

        # NB! залоченные кампании могли остаться от предыдущих запусков
        # и этих кампаний может не быть в $camps и @all_cids_to_attempt_lock
        # стреляет в случае, когда были залоченные кампании, а затем снизили лимиты
        $log->out("locked cids: ".join(",", sort {$a <=> $b} keys %$locked));

        if (scalar(keys(%$locked)) > $options{limit}) {
            #JAVA-EXPORT: FindExcessCampaignsStep#getExcessCampaignIds
            $log->out(sprintf('locked camps count %d, but limit was %d', scalar(keys(%$locked)), $options{limit}));
            # если получилось залочить больше, чем просили - разлочим лишнее
            # используем тот факт, что @all_cids_to_attempt_lock отсортирован как нужно

            my @good_cids;
            for my $cid (@all_cids_to_attempt_lock) {
                if (!exists $locked->{$cid}) {
                    # не смогли залочить, не рассматриваем
                    next;
                } elsif (@good_cids < $options{limit}) {
                    # оставим эту кампанию в $locked
                    push @good_cids, $cid;
                } else {
                    # все что сверх лимита - разлочится
                    # здесь же - залоченные кампании, которые мы не хотели лочить
                }
            }
            my $cids_to_unlock = xminus([ keys(%$locked) ], \@good_cids);
            unlock_campaigns($locked, all => $cids_to_unlock);
        }

        # в ре-экспорте самое объемное к отправке - MdsMeta в картиночных баннерах.
        # поэтому смотрим на то, сколько у нас картинок
        # фразы также могут занимать значительный объём - учитываем это
        if (is_par_type(qw/full_lb_export/)) {
            #JAVA-EXPORT: FindExcessCampaignsStep#getExcessImageCampaignIds
            my $image_stat = get_hash_sql($dbh, ['SELECT b.cid, count(*)',
                                                 'FROM banners b',
                                                 'JOIN banner_images bi USING(bid)',
                                                 WHERE => {'b.cid' => [keys %$locked]},
                                                 'GROUP BY b.cid',
                                          ]);
            my $total_images = 0;

            # возможно общая длина фраз - не самая оптимальная эвристика для выкидывания кампаний из реэкспорта, но пока смотрим на неё
            my $phrases_stat =  get_hash_sql($dbh, ['SELECT cid, sum(length(phrase)) FROM bids b', WHERE => {'cid' => [keys %$locked]}, 'GROUP BY cid']);
            my $total_phrases_length = 0;

            my $has_at_least_one_good_camp = 0;
            my @excess;
            my @to_check = xsort { $locked->{$_}->{seq_time}, \$locked->{$_}->{cid} } keys %$locked;
            for my $cid (@to_check) {
                my $camp_images = $image_stat->{$cid} // 0;
                my $camp_phrases_length = $phrases_stat->{$cid} // 0;

                $total_images += $camp_images;
                $total_phrases_length += $camp_phrases_length;

                if (    $has_at_least_one_good_camp
                    && ($camp_images || $camp_phrases_length)
                    && ($total_images > $LB_EXPORT_IMAGES_LIMIT || $total_phrases_length > $LB_EXPORT_PHRASES_LIMIT))
                {
                    push @excess, $cid;
                } else {
                    # Даже если кампания имеет очень много картинок или фраз - все равно ее берем в работу
                    # (не добавляем в @excess - список кого из работы исключаем)
                    $has_at_least_one_good_camp = 1;
                }
            }
            if (@excess) {
                $log->out("Got excess campaigns with lot of image banners or phrases, unlock");
                unlock_campaigns($locked, all => \@excess);
            }
        }

        for my $camp (values %$locked) {
            my $cid = $camp->{cid};
            if (exists $full_cids{$cid}) {
                $camp->{send_full} = 1;
            }
            if (exists $data_cids{$cid}) {
                $camp->{send_data} = 1;
            }
            if (exists $camp_cids{$cid}) {
                $camp->{send_camp} = 1;
            }
            if (    !$camp->{send_camp}     && (any {$camp->{$_}} qw/camps_num/)
                ||  !$camp->{send_data}     && (any {$camp->{$_}} qw/banners_num contexts_num bids_num/)
                ||  !$camp->{send_full}     && (any {$camp->{$_}} qw/is_full_export/)
            ) {
                $camp->{send_other} = 1;
            }
        }
        $result{camps} = $locked;
    }
    return \%result;
}

=head2 _is_sum_migration_possible()

    Возвращает 1, если сейчас может происходить миграция зачислений на ОС в новую схему.
    Если миграция включена через ppc_property, или выключена меньше, чем
    $SUM_MIGRATION_COOLDOWN_SECONDS секунд назад.

=cut

sub _is_sum_migration_possible {
    state $migration_state_prop = Property->new('WALLET_SUMS_MIGRATION_STATE');
    my $state_json = $migration_state_prop->get(60);

    if (!defined($state_json)) {
        return 0;
    }

    my $state = eval { from_json($state_json) };
    if (!defined($state)
        || !JSON::is_bool($state->{enabled})
        || $state->{time} !~ /^\d+$/
    ) {
        $log->out("error decoding WALLET_SUMS_MIGRATION_STATE: $state_json");
        return 0;
    }

    if ($state->{enabled}) {
        return 1;
    }

    # если с момента выключения миграции прошло меньше 15 минут, считаем, что она может еще работать.
    if ((time() - $state->{time}) <= $SUM_MIGRATION_COOLDOWN_SECONDS) {
        return 1;
    }

    return 0;
}

=head2 _try_lock_sum_migration(\@cids, \%camp_wallets)

    Пробует взять Redis лок на общие счета кампаний @cids, чтобы можно было
    приостановить миграцию ОС отправляемых кампаний путем выставления им
    bs_export_queue.par_id.

    %camp_wallets - словарик соответствия cid и wallet_cid. Если cid - ID общего счета, то wallet_cid = cid

    Если лок взять не удалось, такие кампании отправлять нельзя.

    Возвращает (\%locks, \@cids_failed_locks)
        %locks - словарик с успешными локами, пригодится для снятия локов
        @cids_failed_locks - список cid, у которых не удалось получить лок

=cut

sub _try_lock_sum_migration {
    my ($cids, $camp_wallets) = @_;

    my %lock_name_to_cids;
    for my $cid (@$cids) {
        my $wallet_cid = $camp_wallets->{$cid};

        my $lock_name = _get_sum_migration_lock_name($wallet_cid);
        push @{$lock_name_to_cids{$lock_name}}, $cid;
    }

    my $redis = DirectRedis::get_redis();
    my @lock_keys = keys %lock_name_to_cids;

    $log->out("trying to get a Redis lock on ".join(',', @lock_keys));
    my ($locks, $failed) = RedisLock::lock_multi($redis, \@lock_keys, $SUM_MIGRATION_LOCK_TTL_SECONDS);
    my @cids_failed_locks = map { @{$lock_name_to_cids{$_}} } @$failed;
    if (@cids_failed_locks) {
        $log->out("failed to get Redis locks ".join(',', @$failed).'; cids '.join(',', @cids_failed_locks));
    }

    return ($locks, \@cids_failed_locks);
}

=head2 _unlock_sum_migration(\%locks)

    Снимает локи, взятые с помощью _try_lock_sum_migration
    Если снять лок не получилось, ругается в лог.

=cut

sub _unlock_sum_migration {
    my ($locks) = @_;

    my $redis = DirectRedis::get_redis();

    $log->out("dropping Redis locks ".join(',', keys %$locks));
    my ($unlocked, $failed) = RedisLock::unlock_multi($redis, $locks);
    if (@$failed) {
        $log->out("failed to drop Redis locks ".join(',', @$failed));
    }
}

=head2 _get_sum_migration_lock_name($wallet_cid)

    Получение названия ключа из $wallet_cid, который будет использоваться для
    взятия Redis локов.

=cut

sub _get_sum_migration_lock_name {
    my ($wallet_cid) = @_;

    return 'sum-migration-'.$wallet_cid.'{group'.($wallet_cid % 3).'}1';
}

=head2 unlock_campaigns

    Разблокировка кампаний в таблице bs_export_queue, а также увеличение seq_time на $DELAY_TIME
        для кампаний, присутствующих в хеше %BS::Export::CIDS_FOR_DELAY
    На входе три аргумента:
        $camps - ссылка на хэш cid => параметры кампании
        $kind - что разлочиваем:
            all - все кампании
            camps_and_data - все кампании
            camps_only - только те, у которых нет send_data
        $cids - массив cid-ов, которые нужно разлочить
    В результате меняется $camps и %BS::Export::CIDS_FOR_DELAY - оттуда удаляются разблокированные кампании

=cut

sub unlock_campaigns {
    my ($camps, $kind, $cids) = @_;
    return if $DEBUG_MODE;
    my @cids_to_unlock;

    if ($kind eq 'all') {
        @cids_to_unlock = @$cids;
    } elsif ($kind eq 'camps_and_data') {
        @cids_to_unlock = @$cids;
    } elsif ($kind eq 'camps_only') {
        @cids_to_unlock = grep { !$camps->{$_}->{send_data} } @$cids;
    } else {
        die "unknown kind $kind";
    }

    $log->out("attempt to unlock camps: (".join(',', @$cids).")");
    $log->out("camps to really unlock: (".join(',', @cids_to_unlock).")") if @cids_to_unlock != @$cids;
    return if !@cids_to_unlock;

    my @delay_cids = grep { exists $BS::Export::CIDS_FOR_DELAY{$_} } @$cids;
    my @update_seq_time;
    if (@delay_cids) {
        $log->out("camps to delay in queue: (" . join(',', @delay_cids) . ")");
        my $cids_cond = sql_condition({cid__int => \@delay_cids});

        if (is_par_type('buggy')) {
            # LEAST(GREATEST) - короче, чем IF(seq_time + INTERVAL $DELAY_TIME > NOW(), NOW(), seq_time + INTERVAL $DELAY_TIME)
            @update_seq_time = ('seq_time__dont_quote' => "IF($cids_cond, LEAST(NOW(), GREATEST(NOW(), seq_time + INTERVAL $DELAY_TIME)), seq_time)");
        } elsif (is_par_type('full_lb_export')) {
            # двигаем кампанию в очереди, но не в будущее
            @update_seq_time = ('full_export_seq_time__dont_quote' => "IF($cids_cond, LEAST(NOW(), full_export_seq_time + INTERVAL $DELAY_TIME), full_export_seq_time)");
        } else {
            # двигаем кампанию в очереди, но не в будущее
            @update_seq_time = ('seq_time__dont_quote' => "IF($cids_cond, LEAST(NOW(), seq_time + INTERVAL $DELAY_TIME), seq_time)");
        }
    }

    do_update_table($dbh, 'bs_export_queue',
                    {
                        par_id => undef,
                        @update_seq_time,
                    },
                    where => {par_id => $PAR_ID, cid => \@cids_to_unlock});
    delete @{ $camps }{ @cids_to_unlock };
    delete @BS::Export::CIDS_FOR_DELAY{ @delay_cids };
}

=head2 move_buggy_camapaigns($cids)

    Переместить кампании в buggy-очередь.
    Если кампания уже есть в bs_export_specials и это не dev1, dev2, nosend - то
        текущая специализация заменяется на buggy.

=cut

sub move_buggy_campaigns {
    my $cids = shift;
    return if !@$cids;
    return if $DEBUG_MODE;
    return if is_par_type(qw/internal_ads internal_ads_dev/);

    # логируем
    $log->out(($DONT_MOVE_CAMPS_TO_BUGGY ? 'FAKE ' : '') . 'move camps to buggy queue: ' . join(',', @$cids));

    # если отключено - реальный перенос не выполняем
    return if $DONT_MOVE_CAMPS_TO_BUGGY;

    do_mass_insert_sql($dbh, 'INSERT INTO bs_export_specials (cid, par_type)
                              VALUES %s
                              ON DUPLICATE KEY UPDATE par_type = IF(par_type NOT IN ("nosend", "dev1", "dev2", "preprod"),
                                                                    VALUES(par_type),
                                                                    par_type
                                                                    )',
                       [ map { [ $_, 'buggy' ] } @$cids ]);
}

=head2 resync_at_end(@resync_data)

    Запомнить хеши с данными для переотправки в @RESYNC_DATA,
    чтобы в конце итерации они были добавлены в ленивую очередь.

    В потоках типа "full_lb_export" ничего не делает.

    Параметры:
        @resync_data - массив или ссылка на массив, содержащие хеши
                       в формате для bs_resync

=cut

sub resync_at_end {
    if (is_par_type('full_lb_export')) {
        return;
    }
    push @RESYNC_DATA, grep { ref $_ eq 'HASH' } xflatten(@_);
}

=head3 _determine_sent_objects($query, %options)

    my %sent_objects = _determine_sent_objects($query,
                                               forcedly_stopped_camps => $additional{forcedly_stopped_camps},
                                               forcedly_stopped_banners => $additional{forcedly_stopped_banners},
                                               is_second_query => 1,
                                              );

    По данным сформированного запроса определяет, какие объекты в нем содержатся, а также
        каким объектам можно обновлять statusBsSynced

    Кампаниям можно обновлять statusBsSynced, если:
        - это второй запрос (в нем всегда передаются "настоящие" данные, без подмен)
        - если кампании не подменяли значение Stop (0 -> 1)
    Контексты считаются отправленными, если представлены полностью с UpdateInfo=1.
    Баннерам можно обновлять statusBsSynced, если:
        - это первый запрос, подмены Stop не было, баннер обновляется (UpdateInfo=1) или выключается(Stop=1)
        - это второй запрос и баннеру подменялось (на первом запросе) значение Stop (0->1)

    Параметры:
        $query_hashref  - сформированный запрос к БК (результат функции BS::ExportQuery::get_query)
        is_second_query =>          # флаг (в булевом смысле), что обрабатывается второй запрос к БК.
                                    # по умолчанию - ложь (т.е. обрабатывается первый запрос)
        forcedly_stopped_camps =>   # hashref, ключи - номера кампаний, которым подменялся Stop в первом запросе
        forcedly_stopped_banners => # hashref, ключи - номера баннеров, которым подменялся Stop в первом запросе
    Результат:
        campaigns_in_request => $cids_hashref,  # ключами хеша являются номера кампаний, представленных
                                                # в запросе любыми данными (сама кампания, условия, баннеры
                                                # фразы, ...)
        synced_campaigns => $cids_hashref2,     # ключами хеша являются номера кампаний (cid)
                                                # которым можно обновить statusBsSynced
        contexts_sent => $contexts_hashref,     # ключами хеша являются номера отправленных групп (pid)
        synced_banners => $banners_hashref,     # ключами хеша являются номера баннеров (bid или image_id)
                                                # которым можно обновить statusBsSynced
        synced_banners_with_update_info => $banners_with_update_info
                                                # ключами хеша являются номера баннеров (bid или image_id)
                                                # которые ушли в БК с UpdateInfo=1

=cut

sub _determine_sent_objects {
    my ($query, %options) = @_;

    my (%campaigns_in_request, %camps, %contexts, %banners, %banners_with_update_info, %order_types);
    foreach my $o (values %{$query->{ORDER}}) {
        $campaigns_in_request{ $o->{EID} } = undef;
        $order_types{ $o->{EID} } = $o->{OrderType};
        # Подумать, нужно ли безусловно пытаться обновить c.statusBsSynced, если UpdateInfo=0?
        if ($options{is_second_query}
            || !exists $options{forcedly_stopped_camps}->{ $o->{EID} }
        ) {
            $camps{ $o->{EID} } = undef;
        }
        foreach my $c (values %{$o->{CONTEXT}}) {
            if ($c->{UpdateInfo}) {
                $contexts{$c->{EID}} = undef;
            }
            foreach my $banner (values %{$c->{BANNER}}) {
                if (($options{is_second_query}
                        && exists $options{forcedly_stopped_banners}->{ $banner->{EID} }
                     )
                    || (!$options{is_second_query}
                        && !exists $options{forcedly_stopped_banners}->{ $banner->{EID} }
                        && ($banner->{UpdateInfo}
                            || $banner->{Stop}
                            )
                        )
                ) {
                    $banners{$banner->{EID}} = undef;
                }
                if ($banner->{UpdateInfo}) {
                    $banners_with_update_info{$banner->{EID}} = undef;
                }
            }
        }
    }

    return (
        campaigns_in_request => \%campaigns_in_request,
        synced_campaigns => \%camps,
        contexts_sent => \%contexts,
        synced_banners => \%banners,
        synced_banners_with_update_info => \%banners_with_update_info,
        order_types_sent => \%order_types,
    );
}

=head2 do_response_actions_without_bssoap_data

    обработать запрос в БК без ответа bssoap

=cut

sub do_response_actions_without_bssoap_data {
    my ($query, %additional) = @_;
    my $campaigns_activization = $additional{wait_activization} // {};
    my $new_campaigns = $additional{new_campaigns} // {};
    my $new_banners = $additional{new_banners} // {};
    my $contexts_mapping = $additional{contexts_eid2id} // {};
    my $minus_geo = $additional{minus_geo} // {};
    my %sent_objects = _determine_sent_objects($query,
                                               forcedly_stopped_camps => $additional{forcedly_stopped_camps},
                                               forcedly_stopped_banners => $additional{forcedly_stopped_banners},
                                               is_second_query => $additional{is_second_query},
                                               );

    # 1. camp_activization
    my @activization_cids = grep { exists $campaigns_activization->{$_} } keys %{ $sent_objects{campaigns_in_request} };
    if (@activization_cids) {
        my @insert_data = map { [ $_ ] } @activization_cids;
        do_mass_insert_sql($dbh, "insert into camp_activization (cid) values %s ON DUPLICATE KEY UPDATE send_time = now()", \@insert_data);
        undef @activization_cids;
    }

    # 2. camp_order_types (выкинуть бы его вообще)
    my $order_types_sent = $sent_objects{order_types_sent};
    my @camp_order_types_data = map { [ $_, $order_types_sent->{$_} ] } keys %$order_types_sent;
    do_mass_insert_sql($dbh, "insert ignore into camp_order_types (cid, order_type) values %s ", \@camp_order_types_data, { sleep => 1 });
    undef @camp_order_types_data;

    # раскладываем запрос на айдишники
    my (@sbs_cids, %orderid_map);
    my @sbs_pids;
    my (@sbs_bids, %banner_active_map, %bannerid_map, @bmg_update, @bmg_delete, %hidden_image_bid_to_bannerid);
    for my $o (values(%{ $query->{ORDER} || {} })) {
         if (exists $new_campaigns->{$o->{EID}}) {
            # прокидываем наружу
            $new_campaigns->{$o->{EID}} = $o->{ID};
            # для апдейта в базу
            $orderid_map{ $o->{EID} } = $o->{ID};

        }
        if (exists $sent_objects{synced_campaigns}->{ $o->{EID} }) {
            push @sbs_cids, $o->{EID};
        }


        for my $c (values %{ $o->{CONTEXT} || {} }) {
            # нужно только чтобы собрать структуру для второго запроса, само значение c.ID вроде как из direct-banners-log не читается
            # пишем единичку, чтобы сработал if внутри get_second_query
            $contexts_mapping->{$c->{EID}} = "1"; # aka DEFAULT_PRIORITYID

            if (exists $sent_objects{contexts_sent}->{ $c->{EID} }) {
                push @sbs_pids, $c->{EID};
            }

            for my $ba (values %{ $c->{BANNER} || {} }) {
                if (exists $new_banners->{$ba->{EID}}) {
                    # может быть как баннер, так и banner_images.image_id
                    $new_banners->{$ba->{EID}} = $ba->{ID};
                    $bannerid_map{ $ba->{EID} } = $ba->{ID}
                }

                if (exists $additional{bid_to_hidden_image_id}->{ $ba->{EID} }) {
                    $hidden_image_bid_to_bannerid{ $ba->{EID} } = $ba->{ID};
                } elsif (!exists $additional{images_sent}->{ $ba->{EID} }) {
                    # не картинка
                    $banner_active_map{ $ba->{EID} } = ($ba->{Stop} == 1) ? "No" : "Yes";
                    if (exists $sent_objects{synced_banners}->{ $ba->{EID} }) {
                        push @sbs_bids, $ba->{EID};
                    }
                    # раньше был какой-то баг и мы пытались перекладываать минус-регионы на картиночный id (но у картинок нет своих флагов!)
                    if (exists $sent_objects{synced_banners_with_update_info}->{ $ba->{EID} }) {
                        my $banner_minus_geo = $minus_geo->{ $c->{EID} }->{ $ba->{EID} };
                        if (defined $banner_minus_geo->{current} && $banner_minus_geo->{current} ne '') {
                            push @bmg_update, [$ba->{EID}, 'bs_synced', $banner_minus_geo->{current}];
                        } elsif (defined $banner_minus_geo->{bs_synced}) {
                            push @bmg_delete, $ba->{EID};
                        }
                    }
                }
            }

        }
    }

    if (%orderid_map) {
        # 3. метабаза
        save_shard(OrderID => $orderid_map{$_}, cid => $_) for keys(%orderid_map);
        # 4. разбиваем исходный один запрос (sbs / OrderID) на два отдельных, зато массовые
        do_update_table($dbh, 'campaigns', {OrderID__dont_quote => sql_case(cid => \%orderid_map )}, where => { cid__int => [keys(%orderid_map)]});
        undef %orderid_map;
    }
    if (@sbs_cids) {
        do_update_table($dbh, 'campaigns', {
                statusBsSynced => "IF(statusBsSynced='Sending', 'Yes', statusBsSynced)",
                LastChange => 'LastChange',
            },
            where => { cid__int => \@sbs_cids },
            dont_quote => [ 'LastChange', 'statusBsSynced' ],
        );
        undef @sbs_cids;
    }

    # 5. phrases
    if (@sbs_pids) {
        for my $chunk (chunks(\@sbs_pids, 200)) {
            do_update_table($dbh, 'phrases', {
                    PriorityID => $BS::Export::DEFAULT_PRIORITYID,
                    statusBsSynced => "IF(statusBsSynced='Sending', 'Yes', statusBsSynced)",
                    LastChange => "IF(PriorityID = 0, NOW(), LastChange)",
                },
                where => { pid__int => $chunk },
                dont_quote => [ 'LastChange', 'statusBsSynced' ],
            );
        }
        undef @sbs_pids;
    }

    # 6. минус-регионы
    if (@bmg_update) {
        do_mass_insert_sql($dbh, "INSERT INTO banners_minus_geo (bid, type, minus_geo) VALUES %s ON DUPLICATE KEY UPDATE minus_geo = VALUES(minus_geo)", \@bmg_update, { max_row_for_insert => 200 });
        undef @bmg_update;
    }
    for my $chunk (chunks(\@bmg_delete, 200)) {
        do_delete_from_table($dbh, "banners_minus_geo", where => {bid__int => $chunk, type => 'bs_synced'});
    }
    undef @bmg_delete;

    # 7. banner_images
    my @image_ids = grep { exists $additional{images_sent}->{ $_ } }  keys %bannerid_map;
    for my $chunk (chunks(\@image_ids, 200)) {
        do_update_table($dbh, 'banner_images', {
                PriorityID => $BS::Export::DEFAULT_PRIORITYID,
                BannerID__dont_quote => sql_case(image_id => hash_cut(\%bannerid_map, $chunk)),
            },
            where => {image_id__int => $chunk},
        );
    }
    undef @image_ids;

    # banner_images: Если это картиночная версия баннера, которая отправляется только в родительском тикете
    my @hidden_image_bids = grep { exists $additional{bid_to_hidden_image_id}->{ $_ } }  keys %hidden_image_bid_to_bannerid;
    for my $chunk (chunks(\@hidden_image_bids, 200)) {
        do_update_table($dbh, 'banner_images', {
                PriorityID => $BS::Export::DEFAULT_PRIORITYID,
                BannerID__dont_quote => 'IF(BannerID = 0, '.sql_case(bid => hash_cut(\%hidden_image_bid_to_bannerid, $chunk)).', BannerID)',
            },
            where => {bid__int => $chunk},
        );
    }

    # 8. баннеры разбиваем на 3 апдейта, зато массовые: BannerID, statusActive, statusBsSycned
    my @new_banner_ids = grep { !exists $additional{images_sent}->{ $_ } } keys %bannerid_map;
    for my $chunk (chunks(\@new_banner_ids, 200)) {
        do_update_table($dbh, 'banners', {
                LastChange__dont_quote => 'NOW()',
                BannerID__dont_quote => sql_case(bid => hash_cut(\%bannerid_map, $chunk)),
            },
            where => {bid__int => $chunk},
        );
    }
    undef @new_banner_ids;

    for my $chunk (chunks([keys %banner_active_map], 200)) {
        do_update_table($dbh, 'banners', {
                LastChange__dont_quote => 'LastChange',
                statusActive__dont_quote => sql_case(bid => hash_cut(\%banner_active_map, $chunk)),
            },
            where => {bid__int => $chunk},
        );
    }

    # чтобы не потерять изменения минус-регионов, обновление statusBsSynced делаем последним (раньше было в одной транзакции)
    for my $chunk (chunks(\@sbs_bids, 200)) {
        do_update_table($dbh, 'banners', {
                statusBsSynced => "IF(statusBsSynced='Sending', 'Yes', statusBsSynced)",
                LastChange => 'LastChange',
            },
            where => { bid__int => $chunk },
            dont_quote => [ 'LastChange', 'statusBsSynced' ],
        );
    }
    undef @sbs_bids;

}

=head2 set_sync_campaigns

    Пометить кампании, как просинхронизированные
    Если sync_val не изменился
      если для кампаний нужно синхнонизировать что-то, кроме того, что отправили(send_other) - обнуляем поля $fields_to_reset
      иначе - удаляем из bs_export_queue
    Если sync_val изменился - кампании разблокируются

    Параметры:
        $camps - ссылка на хэш cid => хэш, содержащий старый sync_val
        $kind - что отправляли перед этим: camps_and_data/camps_only/prices/full
        $cids - массив cid-ов, которые нужно разлочить

    В результате меняется camps:
        если больше данных для отправки в БК нет - переданные кампании удаляются
        иначе - для них зануляется соответствующая статистика и
            удаляется соответствующий флаг отправки (send_data/send_camp)

=cut

sub set_sync_campaigns {
    my ($camps, $kind, $cids) = @_;
    return if $DEBUG_MODE;
    my @fields_to_reset;
    my $sent_flags;

    if ($kind eq 'camps_and_data') {
        @fields_to_reset = qw/camps_num banners_num contexts_num bids_num/;
        $sent_flags = [qw/send_data send_camp/];
    } elsif ($kind eq 'full') {
        @fields_to_reset = qw/is_full_export/;
        $sent_flags = [qw/send_full/];
    } elsif ($kind eq 'camps_only') {
        @fields_to_reset = qw/camps_num/;
        $sent_flags = [qw/send_camp/];
    } else {
        die "unknown kind $kind";
    }

    $log->out("camps to set synced: (".join(',', @$cids).")");
    return if !@$cids;

    my $sync_vals = {map {$_ => $camps->{$_}->{sync_val}} keys %$camps};

    # раскладываем список кампаний на две кучки:
    # кампании, по которым что-то еще нужно отправить в БК
    # и те, у кого все уже отправлено (и можно удалить из очереди)
    my (@sync_cids, @send_other_cids);
    for my $cid (@$cids) {
        # удаляем признак необходимости отправки того, что уже отправили ($kind)
        delete $camps->{$cid}->{$_} for @$sent_flags;
        if (any { $camps->{$cid}->{$_} } qw/send_other send_data send_full send_camp/) {
            # по кампании еще что-то нужно отправить
            push @send_other_cids, $cid;
        } else {
            push @sync_cids, $cid;
        }
    }

    # те, у которых не изменился sync_val - удаляем
    if (@sync_cids) {
        $log->out("camps to delete from queue: (".join(',', @sync_cids).")");
        my $SYNC_VAL_CASE = sql_case(cid => hash_cut($sync_vals, @sync_cids), default => undef);
        my $deleted = do_sql($dbh, ["DELETE FROM bs_export_queue", WHERE => {par_id => $PAR_ID, cid => \@sync_cids, sync_val__dont_quote => $SYNC_VAL_CASE}]);
        if ($deleted != @$cids) {
            # остальных(которых изменили, пока мы отвлеклись) разблокируем и переставляем в конец очереди
            my $updated_cids = get_one_column_sql($dbh, ["SELECT cid FROM bs_export_queue", WHERE => {cid => \@sync_cids, sync_val__dont_quote__gt => $SYNC_VAL_CASE}]) || [];
            $log->out("updated camps in background: (".join(',', @$updated_cids).")");
            my $SEQ_TIME_COLUMN = is_par_type('full_lb_export') ? 'full_export_seq_time' : 'seq_time';
            do_sql($dbh, ["UPDATE bs_export_queue SET par_id = NULL, $SEQ_TIME_COLUMN = now(), queue_time = now()", WHERE => {par_id => $PAR_ID, cid => $updated_cids,}]);
        }

        delete @{ $camps }{ @sync_cids };
    }

    # удаляем из спец. очереди, если нужно
    my $cids_to_delete_from_spec;
    my $spec_types;
    if ($kind eq 'camps_only') {
        # отправка только кампаний - особый случай, при этом удаляем из спец. очереди даже кампании, отправленные не целиком.
        # Иначе кампания застрянет в bs_export_queue, т.к. воркер camps_only не подберет кампанию с camps_num == 0,
        # а другим воркерам запрещено брать кампани из этой спец. очереди.
        $spec_types = ['camps_only'];
        $cids_to_delete_from_spec = $cids;
    } else {
        # для других типов отправки, чистим спец. очередь только после того, как все отправлено.
        $spec_types = [ qw{ heavy fast buggy } ];
        $cids_to_delete_from_spec = \@sync_cids;
    }
    my $spec_cids = get_one_column_sql($dbh, ["SELECT cid FROM bs_export_specials", WHERE => {cid => $cids_to_delete_from_spec, par_type => $spec_types} ]);
    if (@$spec_cids) {
        do_sql($dbh, ["DELETE FROM bs_export_specials", WHERE => {cid => $spec_cids, par_type => $spec_types}]);
        $error_logger->([map{{
                    message => "cid is deleted from bs_export_specials in set_sync_campaigns",
                    cid     => $_,
                    type    => "remove_from_specials",
                }} @$spec_cids]);
    }

    # кампании, по которым нужно что-то отправлять, но мы это не отправляем
    if (@send_other_cids) {
        $log->out("camps to reset stat (send something else): (".join(',', @send_other_cids).")");
        my $SYNC_VAL_CASE = sql_case(cid => hash_cut($sync_vals, @send_other_cids), default => undef);
        do_update_table($dbh, "bs_export_queue",
                        {
                            map {
                                $_."__dont_quote" => "IF(sync_val = $SYNC_VAL_CASE, 0, $_)"
                            } @fields_to_reset
                        },
                        where => {par_id => $PAR_ID, cid => \@send_other_cids, }
            );
        for my $cid (@send_other_cids) {
            $camps->{$cid}->{$_} = 0 for @fields_to_reset;
        }
    }
}

=head3 set_banner_permalinks_sent_to_bs

    Помечает связки баннеров с пермалинками как отправленные в БК

=cut
sub set_banner_permalinks_sent_to_bs {
    my ($shard, $order_by_id) = @_;

    my $banner_ids = [];
    foreach my $order (values %$order_by_id) {
        foreach my $context (values %{$order->{CONTEXT}}) {
            foreach my $banner (values %{$context->{BANNER}}) {
                if ($banner->{Permalink} && ($banner->{PermalinkAssignType}->value() // '') eq 'manual') {
                    push $banner_ids, $banner->{EID};
                }
            }
        }
    }

    if (@$banner_ids) {
        do_update_table(PPC(shard => $shard), "banner_permalinks",
            {
                "is_sent_to_bs" => 1
            },
            where => {
                "bid__in" => $banner_ids,
            }
        );
    }
}

=head3 $SQL_COMMON_SELECT_FIELDS

    Поля, которые выбираются для UpdateData
    Используется вместе с запросом из _get_sql_from_joins

=cut

my $SQL_COMMON_SELECT_FIELDS = qq{
         , b.statusBsSynced b_statusBsSynced
         , p.statusBsSynced p_statusBsSynced
         , c.cid campaign_pId
         , c.ProductID
         , c.currency AS campaign_currency
         , c.strategy_name, c.strategy_data
         , c.strategy_id
         , b.bid banner_pId, b.BannerID banner_Id, b.href Href, b.domain Site, b.title Title, b.title_extension TitleExtension, find_in_set('geoflag', b.opts) GeoFlag
         , b.type AS b_type, b.banner_type
         , b.statusArch AS banner_statusArch
         , b.phoneflag, b.vcard_id
         , b.body Body, b.statusPostModerate banner_PostModerate
         , b.statusModerate banner_statusModerate
         , b.statusShow AS banner_statusShow, b.statusActive b_statusActive
         , b.language banner_language
         , p.pid context_pId, p.ContextID context_Id, p.adgroup_type
         , p.PriorityID
         , p.geo as Geo
         , p.statusPostModerate phrases_PostModerate
         , amc.store_content_href, amc.mobile_content_id, amc.device_type_targeting AS p_device_type_targeting, amc.network_targeting, amc.min_os_version AS p_min_os_version
         , abst.page_group_tags_json, abst.target_tags_json
         , app.project_param_conditions
         , acp.content_promotion_type
         , ag_perf.feed_id AS perf_adgroup_feed_id
         , ag_perf.field_to_use_as_name AS perf_adgroup_use_as_name
         , ag_perf.field_to_use_as_body AS perf_adgroup_use_as_body
         , ag_text.feed_id AS text_adgroup_feed_id
         , ag_text.filter_data AS text_filter_data
         , ag_text.field_to_use_as_name AS text_adgroup_use_as_name
         , ag_text.field_to_use_as_body AS text_adgroup_use_as_body
         , ai.level AS adgroups_internal_level
         , ai.rf AS adgroups_internal_rf
         , ai.rfReset AS adgroups_internal_rfReset
         , DATE_FORMAT(ai.start_time, '%Y%m%d%H%i%s') AS adgroups_internal_start_time
         , DATE_FORMAT(ai.finish_time, '%Y%m%d%H%i%s') AS adgroups_internal_finish_time
         , bmc.reflected_attrs, bmc.primary_action AS b_primary_action, bmc.impression_url
         , mat.impression_url AS mobile_app_impression_url
         , b_int.template_id, b_int.template_variables
         , ad.main_domain_id AS dynamic_main_domain_id, ad.feed_id AS dyn_adgroup_feed_id
         , ad.field_to_use_as_name AS dyn_adgroup_use_as_name
         , ad.field_to_use_as_body AS dyn_adgroup_use_as_body
         , gp.href_params AS adgroup_href_params
         , b.flags
         , b.sitelinks_set_id, b.statusSitelinksModerate
         , p.mw_id
         , bim.image_hash as banner_image, bim.image_id as image_id, bim.BannerID as image_BannerID
         , bim.statusModerate as image_statusModerate, bim.statusShow as image_statusShow
         , bim.PriorityID as image_PriorityID
         , FIND_IN_SET('single_ad_to_bs', bim.opts) as is_single_image_ad_to_bs
         , perf_cr.creative_id as CreativeID
         , perf_cr.version as CreativeVersion
         , perf_cr.creative_type as creative_type
         , perf_cr.statusModerate as CreativeStatus
         , perf_cr.creative_type as perf_creative_type, perf_cr.layout_id as perf_layout_id
         , perf_cr.is_brand_lift
         , bdh.display_href, bdh.statusModerate dh_statusModerate
         , IF(
             b.banner_type in ('image_ad', 'mcbanner', 'cpm_banner', 'cpc_video', 'cpm_outdoor', 'cpm_indoor', 'cpm_audio', 'cpm_geo_pin'),
             IF(images.bid IS NOT NULL OR b.banner_type = 'mcbanner', images.statusModerate, b_perf.statusModerate),
             NULL
           ) as image_ad_statusModerate
         , IF(b.banner_type in ('image_ad', 'mcbanner', 'cpm_banner', 'cpm_geo_pin'), IF(images.bid IS NOT NULL OR b.banner_type = 'mcbanner', 'image_ad', 'canvas'), NULL) AS image_ad_type
         , cbr.categories_bs catalogia_categories
         , btl.tl_id as banner_tl_id
         , btl.statusModerate as banner_tl_statusModerate
         , btlp.href_params as banner_tl_hrefParams
         , btg.turbo_gallery_href as banner_turbo_gallery_href
         , apt.page_blocks as adgroup_page_blocks
         , ap.priority as adgroup_priority_pr
         , c.ClientID
         , FIND_IN_SET('is_auto_video_allowed', c.opts) as is_auto_video_allowed
         , bta.turbo_app_info_id as banner_turbo_app_info_id
         , bta.banner_turbo_app_type
         , bta.content as turbo_app_content
         , ahr.ret_cond_id as hyper_geo_id
         , bpn.client_phone_id
         , bia.image_hash as big_king_banner_image
         , bmcs.flags as multicards_flags
         , cmc.alternative_app_stores
};

=head3 _get_sql_from_joins($camps_only)

    Возвращает кусок SQL - join-ы, которые добавляются к таблице campaigns as c
    Используется для выборки полей $SQL_COMMON_SELECT_FIELDS

    campaigns wc и camp_options co джойнятся только для того, чтобы корректно отобрать баннеры,
    использовать данные из этих колонок в селекте не нужно. Данные о кампаниях отбираются в отдельном запросе.

    Если передан "правдивый" флаг $camps_only, запрос модифицируется так, чтобы он вернул только кампании,
    без групп и баннеров. При этом в соотв. им столбцах, типа banner_pId и context_pId будет NULL, как будто
    в кампании нет ни одной группы.

=cut

sub _get_sql_from_joins {
    my ($camps_only) = @_;

    my $phrases_join_cond;
    if ($camps_only) {
        $phrases_join_cond = 'p.cid = NULL';
    } else {
        $phrases_join_cond = 'p.cid = c.cid AND ban.statusBlocked = "No"';
    }
    return qq{
               /* joins for where clause only */
               left join campaigns wc on wc.cid = c.wallet_cid
               left join camp_options co on co.cid = c.cid
               left join clients_options clo on clo.ClientID = c.ClientID
               /* joins for select */
               left join campaigns_mobile_content cmc on cmc.cid = c.cid
               left join mobile_app_trackers mat on mat.mobile_app_id = cmc.mobile_app_id
               left join users ban on ban.uid = uid_for_join
               left join phrases p on $phrases_join_cond
               left join group_params gp on gp.pid = p.pid
               left join adgroups_dynamic ad on ad.pid = p.pid
                             and p.adgroup_type = 'dynamic'
               left join adgroups_mobile_content amc on amc.pid = p.pid
                             and p.adgroup_type = 'mobile_content'
               left join adgroups_content_promotion acp on acp.pid = p.pid
                             and p.adgroup_type = 'content_promotion'
               left join adgroups_internal ai on ai.pid = p.pid
                             and p.adgroup_type = 'internal'
               left join adgroup_page_targets apt on apt.pid = p.pid
                             and p.adgroup_type in ('cpm_outdoor', 'cpm_indoor')
               left join adgroup_priority ap on ap.pid = p.pid
                             and p.adgroup_type in ('cpm_yndx_frontpage')
               left join adgroup_bs_tags abst on abst.pid = p.pid
               left join adgroup_project_params app on app.pid = p.pid
               left join banners b on b.pid = p.pid
                             and c.archived = 'No'
                             and $SQL_MONEY_COND
                             and $SQL_ARCHIVED_UAC_BANNERS_COND
               left join banners_mobile_content bmc on bmc.bid = b.bid
                             and b.banner_type = 'mobile_content'
               left join banners_internal b_int on b_int.bid = b.bid
                             and b.banner_type = 'internal'
               left join banner_images bim on bim.bid = b.bid
               left join catalogia_banners_rubrics cbr on cbr.bid = b.bid
               left join adgroups_performance ag_perf on ag_perf.pid = p.pid and p.adgroup_type = 'performance'
               left join adgroups_text ag_text on ag_text.pid = p.pid and p.adgroup_type = 'base'
               left join banners_performance b_perf on b_perf.bid = b.bid
               left join perf_creatives perf_cr on perf_cr.creative_id = b_perf.creative_id
               left join banner_display_hrefs bdh on bdh.bid = b.bid
               left join images on images.bid = b.bid
               left join banner_turbolandings btl on btl.bid = b.bid
               left join banner_turbolanding_params btlp on btlp.bid = b.bid
               left join banner_permalinks bpml on b.bid = bpml.bid
               left join organizations org on bpml.permalink = org.permalink_id
                             and org.ClientID = c.ClientID
               left join banner_turbo_galleries btg on btg.bid = b.bid
               left join banner_turbo_apps bta on bta.bid = b.bid
               left join adgroups_hypergeo_retargetings ahr on p.pid = ahr.pid
               left join banner_phones bpn on bpn.bid = b.bid
               left join banner_image_asset bia on bia.bid = b.bid
               left join banner_multicard_sets bmcs on bmcs.bid = b.bid
                             and bmcs.statusModerate = 'Yes'
    };

}

=head2 @SELECT_FIELDS_BIDS

    набор полей, для выборки данных (в get_snapshot) по фразам из bids.
    должен быть одинаковым с @SELECT_FIELDS_BIDS_RET, @SELECT_FIELDS_BIDS_DYN, @SELECT_FIELDS_BIDS_PERF, @SELECT_FIELDS_BIDS_BASE (проверяется юнит-тестом)

=cut

our @SELECT_FIELDS_BIDS = (
    q!'bids' AS phrase_type!,
    q!p.pid!,
    q!bi.id AS phrase_pId!,
    q!bi.PhraseID AS phrase_Id!,
    q!bi.phrase AS Phrase!,
    q!bi.price AS Price!,
    q!bi.price_context AS PriceContext!,
    q!bi.optimizeTry AS optimizeTry!,
    q!bi.autobudgetPriority AS autobudgetPriority!,
    q!bi.showsForecast AS showsForecast!,
    q!bi.statusModerate AS bi_statusModerate!,
    q!bi.is_suspended AS phrase_is_suspended!,
    q!bhp.param1 AS UrlParam1!,
    q!bhp.param2 AS UrlParam2!,
    q!p.PriorityID!,
    q!gp.has_phraseid_href!,
    q!NULL as target_funnel!,
);

=head2 @SELECT_FIELDS_BIDS_RET

    набор полей, для выборки данных (в get_snapshot) по условиям ретаргетинга из bids_retargeting.
    должен быть одинаковым с @SELECT_FIELDS_BIDS, @SELECT_FIELDS_BIDS_DYN, @SELECT_FIELDS_BIDS_PERF, @SELECT_FIELDS_BIDS_BASE (проверяется юнит-тестом)

=cut

our @SELECT_FIELDS_BIDS_RET = (
    q!'retargetings' AS phrase_type!,
    q!p.pid!,
    q!bi_ret.ret_id AS phrase_pId!,
    q!bi_ret.ret_cond_id AS phrase_Id!,
    q!NULL AS Phrase!,
    q!NULL AS Price!,
    q!bi_ret.price_context AS PriceContext!,
    q!NULL AS optimizeTry!,
    q!bi_ret.autobudgetPriority AS autobudgetPriority!,
    q!NULL AS showsForecast!,
    q!'Yes' AS bi_statusModerate!,
    q!bi_ret.is_suspended AS phrase_is_suspended!,
    q!NULL AS UrlParam1!,
    q!NULL AS UrlParam2!,
    q!p.PriorityID!,
    q!gp.has_phraseid_href!,
    q!NULL as target_funnel!,
);

=head2 @SELECT_FIELDS_BIDS_DYN

    набор полей, для выборки данных (в get_snapshot) по нацеливаниям из bids_dynamic.
    должен быть одинаковым с @SELECT_FIELDS_BIDS, @SELECT_FIELDS_BIDS_RET, @SELECT_FIELDS_BIDS_PERF, @SELECT_FIELDS_BIDS_BASE (проверяется юнит-тестом)

=cut

our @SELECT_FIELDS_BIDS_DYN = (
    q!'dynamic' AS phrase_type!,
    q!p.pid!,
    q!bi_dyn.dyn_cond_id AS phrase_pId!,
    q!bi_dyn.dyn_cond_id AS phrase_Id!,
    q!NULL AS Phrase!,
    q!bi_dyn.price AS Price!,
    q!bi_dyn.price_context AS PriceContext!,
    q!NULL AS optimizeTry!,
    q!bi_dyn.autobudgetPriority AS autobudgetPriority!,
    q!NULL AS showsForecast!,
    q!'Yes' AS bi_statusModerate!,
    q!FIND_IN_SET('suspended', bi_dyn.opts) AS phrase_is_suspended!,
    q!NULL AS UrlParam1!,
    q!NULL AS UrlParam2!,
    q!p.PriorityID!,
    q!gp.has_phraseid_href!,    # unused in BS::ExportQuery::_extract_phrase
    q!NULL as target_funnel!,
);

=head2 @SELECT_FIELDS_BIDS_PERF

=cut

our @SELECT_FIELDS_BIDS_PERF = (
    q!'performance' AS phrase_type!,
    q!p.pid!,
    q!bi_perf.perf_filter_id AS phrase_pId!,
    q!bi_perf.perf_filter_id AS phrase_Id!,
    q!bi_perf.name AS Phrase!,
    q!bi_perf.price_cpc AS Price!,
    q!bi_perf.price_cpa AS PriceContext!,
    q!NULL AS optimizeTry!,
    q!bi_perf.autobudgetPriority AS autobudgetPriority!,
    q!NULL AS showsForecast!,
    q!'Yes' AS bi_statusModerate!,
    q!bi_perf.is_suspended AS phrase_is_suspended!,
    q!NULL AS UrlParam1!,
    q!NULL AS UrlParam2!,
    q!NULL as PriorityID!,
    q!NULL as has_phraseid_href!,
    q!bi_perf.target_funnel!,
);

=head2 @SELECT_FIELDS_BIDS

    набор полей, для выборки данных (в get_snapshot) по бесфразным таргетингам из bids_base.
    должен быть одинаковым с @SELECT_FIELDS_BIDS, @SELECT_FIELDS_BIDS_RET, @SELECT_FIELDS_BIDS_DYN, @SELECT_FIELDS_BIDS_PERF (проверяется юнит-тестом)

=cut

our @SELECT_FIELDS_BIDS_BASE = (
    q!bi_b.bid_type AS phrase_type!,
    q!p.pid!,
    q!bi_b.bid_id AS phrase_pId!,
    q!NULL AS phrase_Id!,
    q!NULL AS Phrase!,
    q!bi_b.price AS Price!,
    q!bi_b.price_context AS PriceContext!,
    q!NULL AS optimizeTry!,
    q!bi_b.autobudgetPriority AS autobudgetPriority!,
    q!NULL AS showsForecast!,
    q!'Yes' AS bi_statusModerate!,
    q!FIND_IN_SET('suspended', bi_b.opts) AS phrase_is_suspended!,
    q!bhp.param1 AS UrlParam1!,
    q!bhp.param2 AS UrlParam2!,
    q!NULL AS PriorityID!,
    q!gp.has_phraseid_href!,
    q!NULL AS target_funnel!,
);

=head2 @SELECT_FIELDS_ADGROUP_ADDITIONAL_TARGETINGS

=cut

our @SELECT_FIELDS_ADGROUP_ADDITIONAL_TARGETINGS = (
    q!'adgroup_additional_targetings' AS phrase_type!,
    q!p.pid!,
    q!aat.id AS phrase_pId!,
    q!aat.targeting_type!,
    q!aat.targeting_mode!,
    q!aat.value_join_type!,
    q!aat.value!,
    q!0 AS phrase_is_suspended!,
);

=head2 общий комментарий про все $SQL_JOIN_BIDS*

    Раньше при выборке "фраз" (get_snapshot->{phrases}) были следующие условия:
        * таблица bids* join'илась по условию bi*.statusBsSynced = 'Sending'
            или p.statusBsSynced = 'Sending'
        * в условии WHERE запроса на выборку было AND p.statusBsSynced = 'Sending'

    Что произошло:
        * условие на statusBsSynced bids* или phrases удалено из join'а,
            так как всегда выполнялось
        * p.statusBsSynced = 'Sending' переехало из условия для всех таблиц
            в потабличные условия, так как нацеливания нужно выбирать всегда,
            даже если отправлять их в контексте мы не планируем
        * логика про то, что p.statusBsSynced должен быть Sending продублирована
            в BS::ExportQuery::_are_we_want_to_send_phrases, которая должна вызываться
            при использовании фраз для отправки в БК.

    Если потребуется вернуть проверку bids*.statusBsSynced, нужно будеть:
        * добавить его в выборку @SELECT_FIELDS_BIDS*
        * добавить туда же p.statusBsSynced
        * переделать _are_we_want_to_send_phrases с проверки $row (элемент
            снепшота) на $phrases_row (элемент из $PHRASES)

=head2 $SQL_JOIN_BIDS

    Джойн с bids для выборки данных по "фразам" для отправка в UpdateData2

=cut

our $SQL_JOIN_BIDS = qq{
           join bids bi FORCE INDEX(bid_pid) on bi.pid = p.pid
                                and p.statusPostModerate in ('Yes', 'Rejected')
                                and bi.statusModerate = 'Yes'
           left join bids_href_params bhp on bhp.cid = bi.cid and bhp.id = bi.id
};

=head2 $SQL_JOIN_BIDS_RET

    Джойн с bids_retargeting для выборки данных по "фразам" для отправка в UpdateData2

=cut

our $SQL_JOIN_BIDS_RET = qq{
           join bids_retargeting bi_ret FORCE INDEX(pid) on
                bi_ret.pid = p.pid
                and p.statusPostModerate in ('Yes', 'Rejected')
};

=head2 $SQL_JOIN_BIDS_DYN

    Джойн с bids_dynamic для выборки данных по "фразам" для отправка в UpdateData2

=cut

our $SQL_JOIN_BIDS_DYN = qq{
    JOIN bids_dynamic bi_dyn FORCE INDEX(i_pid_dyn_cond_id) ON bi_dyn.pid = p.pid
                                AND p.statusPostModerate IN ('Yes', 'Rejected')
};

=head2 $SQL_JOIN_BIDS_PERF

    Джойн с bids_performance для выборки данных по "фразам" для отправки в UpdateData2

=cut

our $SQL_JOIN_BIDS_PERF = qq{
    JOIN bids_performance bi_perf FORCE INDEX(pid) ON bi_perf.pid = p.pid AND p.statusPostModerate in ('Yes', 'Rejected') AND bi_perf.is_deleted = 0
};

=head2 $SQL_JOIN_BIDS_BASE

    Джойн с bids_base для выборки данных по безфразному таргетингу для отправки в UpdateData2

=cut

our $SQL_JOIN_BIDS_BASE = qq{
    JOIN bids_base bi_b FORCE INDEX(idx_bids_base_pid) ON bi_b.pid = p.pid AND p.statusPostModerate in ('Yes', 'Rejected')
        AND NOT FIND_IN_SET('deleted', bi_b.opts)
        AND $BS::Export::SQL_BIDS_BASE_TYPE_ENABLED
    LEFT JOIN bids_href_params bhp on bhp.cid = bi_b.cid and bhp.id = bi_b.bid_id
};

=head2 $SQL_JOIN_ADGROUP_ADDITIONAL_TARGETINGS

    Джойн с adgroup_additional_targetings для отправки в UpdateData2

=cut

our $SQL_JOIN_ADGROUP_ADDITIONAL_TARGETINGS = qq{
    JOIN adgroup_additional_targetings aat FORCE INDEX(pid_targeting) ON aat.pid = p.pid
};

=head2 get_snapshot(%options)

    Получение данных по кампаниям, готовым (залоченным) к отправке в БК (не больше чем $BS::Export::MAX_ROWS_FOR_SNAPSHOT строк)
    Все данные выбираются в пределах одной транзакции в БД.

    Параметры именованные:
        camps - ссылка на хеш со статистикой по кампаниям. обязательный параметр.
                {
                    cid1 => {
                        # количество объектов, ожидающих синхронизации
                        prices_num, camps_num, banners_num, contexts_num, bids_num,
                        # что будет отправлено
                        send_data, send_camp, send_other
                    },
                    cid2 => { ... },
                }
        only_cids - ссылка на массив номеров кампаний, которыми требуется ограничить выборку.
                    параметр опциональный, обычно используется в отладочном режиме.
    Результат:
    {                   # ссылка на хеш со следующими ключами:
        data => ...,    - ссылка на массив хешей с данными (выборка из снепшота)
        phrases => ..., - ссылка на хеш массивов, в котором данные по фразам сгруппированы по pid
                          (выборка производится по номерам кампаний из $data). Пример:
                          {
                            pid1 => [
                                $row_hashref1,
                                $row_hashref2,
                                ...
                            ],
                            pid2 => [ ... ],
                            ...
                          }
        domains_dict => ...,    - ссылка на хеш со словарем доменов: ключ - domin_id, значение - домен
                                (выборка по dynamic_main_domain_id из $data)
        dynamic_conditions => ...,  - ссылка на хеш со словарем условий нацеливания:
                                    ключ: dyn_cond_id, значение - строка с condition_json
                                    (выборка по phrase_Id для phrase_type = 'dynamic' из $data)
        retargeting_conditions => ...,  - ссылка на хеш со словарем условий ретаргетинга
                                        ключ: ret_cond_id, значение - строка с condition_json
        brandsafety_retargeting_conditions => - ссылка на хещ со словарём условий ретаргетинга Brand Safety
                                                ключ: ret_cond_id, значение - строка с condition_json
        mobile_content => ...,  - ссылка на хеш хешей - словарь данных о мобильных приложениях (ключ mobile_content_id)
        mobapp_info_by_cid => ...,  - ссылка на хеш хешей - словарь данных о предопределённых мобильных приложениях клиента(ключ cid)

        additions => ..., ссылка на хеш с дополнениями, ключ - тип дополнения (например callout),
                          значение - хеш, где ключ - id объкта (например bid), значение - массив объектов дополнений.

        feeds => ..., ссылка на хеш с данными по фидам: ключ - feed_it, значение - хеш с данными о фиде

        performance_conditions => ..., - ссылка на хеш - данные по фильтрам перфоманс баннеров

        performance_counters => ..., - ссылка на хеш со счётчиками для перфоманс-кампаний: cid => counter_id
                                       если счётчиков несколько, используется счётчик с минимальным номером
        experiments => ..., - ссылка на хеш с данными по AB-тестам, ключ - cid, значение - хеш вида
                          { experiment_id => NNN, role => "primary/secondary", percent => XX, }

        campaigns_descriptions => ..., - ссылка на хеш с описанием кампаний, а также пользователей и клиентов, связаных с кампаниями.
                                         ключ - cid, значение - хеш с описанием

        sitelinks_sets => ..., - ссылка на хеш с сайтлинками, ключ - sitelinks_set_id, значения - массив сайтлинков.

        turbolandings => ...., ссылка на хеш с турболендингами, ключ - tl_id, значение - href.

        vcards => ..., - ссылка на хеш с контактной информацией. Ключ - vcard_id, значения - хеши с контактной информацией такого вида:
                        {   # столбцы из vcards
                            phone => ..., name => ..., street => ..., house => ..., build => ..., apart => ..., metro => ...,
                            contactperson => ..., worktime => ..., city => ..., country => ..., geo_id => ..., extra_message => ...,
                            contact_email => ..., im_client => ..., im_login => ..., permalink => ...,
                            # столбцы из org_details
                            ogrn => ...,
                            # столбцы из addresses
                            precision => ..., map_id => ..., map_id_auto => ...,
                            # столбцы из maps
                            mid => ..., x => ..., y => ..., x1 => ..., y1 => ..., x2 => ..., y2 => ...,
                        }

        template_properties => ..., — ссылка на хеш с format_name и status единого шаблона, ключ — direct_template_id
                                    Если ключа нет, то шаблон ещё не переведён на единый формат.

        template_resources => ..., - ссылка на хеш с информацией по ресурсу шаблона: ключ - template_resource_id, значение - хеш с данными о ресурсе

        template_variables_images_formats => ..., - ссылка на хеш с информацией по картинке для картиночных переменных шаблона:
                                            ключ - image_hash, значение - хеш с данными о картинке

        banner_permalinks => ..., - ссылка на хеш побаннерной привязкой карточек организации:
                                bid => {permalink => ..., chain_ids => [...]}

        client_phones => ..., - ссылка на хеш с телефонами, ключ - client_phone_id:
                                client_phone_id => { phone => ... }

        disclaimers => ..., - ссылка на хеш с дисклеймерами, ключ - bid, значения - дисклеймеры.

        minus_geo => ..., - ссылка на хеш с выбранными из БД минус-регионами баннеров

        aggregator_domains => ..., - ссылка на хеш с доменами агрегаторами. Ключ - bid, значение - домен

        billing_aggregates => { - ссылка на хеш с биллинговыми агрегатами, сгруппированными по ID общего счета и типу продукта
            $wallet_cid => {
                $product_type => Direct::Model::BillingAggregate->new(),
                ...
            }
        }

        banner_page_moderation => { - ссылка на хеш с результатами внешней модерации, сгруппированными по ID баннера
            bid => [{
                pageId => ...
                statusModerate => ...
            }
             ...
            ]
        }

        banner_measurers => { - ссылка на хеш с измерителями видимости баннеров
            bid => {
                admetrica => { // имя конкретного измерителя
                   // параметры измерителя, для адметрки см. в class BannerMeasurerParamsAdmetrica
                   .......
                },
                .......
            }
            ...
        }

        banner_tns_id => { - TNS ID баннеров
            bid => tns_id
        }

        internal_ad_products => { - продукты внутренней рекламы
            $ClientID => { ClientID => 123, product_name => "" }
        }

        skadnetwork_slots => { - SKAdNetworkCampaignID для РМП кампании cid
            cid => slot
        }

        mobile_goal_app_info => { - os_type и store_content_id для мобильных целей, используемых в стратегиях кампаний
            goal_id => {
                os_type => "iOS",
                store_content_id => "id2342928342"
            }
        }

        banner_additional_hrefs => { - ссылка на массив хешов со ссылками для креативов у HTML5 баннеров
            bid => [
                [0] = { href  => "http://ya.ru",
                        index => "0",
                        ...
                },
                [1] = { href  => "http://yandex.ru",
                        index => "1",
                        ...
                }
                ],
                .......
            }
            ...
        }

        asset_hashes => { - ссылка на хеш с информацией о id ассетах в кампаниях и их баннерах
            <campaign_id> => {
                <banner_id 1> => {
                    TitleAssetHash => '395766436205812',
                    TextBodyAssetHash => '395767351324926',
                    ImageAssetHash => '395767741545527',
                    VideoAssetHash => '395766269052275'
                },
                <banner_id 2> => {
                    TitleAssetHash => '395767825866396',
                    TextBodyAssetHash => '395767607000723',
                },
                ...
            },
            ...
        }

        select_time => ... - время выборки их из БД( фиксируется перед началом запросов к БД)
    }

=cut

sub get_snapshot {
    # JAVA-EXPORT: SnapshotDataFetcher#getSnapshot
    my %options = @_;

    my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot');

    my ($camps, $only_cids) = @options{qw/camps only_cids/};
    state ($SQL_SELECT_FIELDS_BIDS, $SQL_SELECT_FIELDS_BIDS_RET, $SQL_SELECT_FIELDS_BIDS_DYN, $SQL_SELECT_FIELDS_BIDS_PERF, $SQL_SELECT_FIELDS_BIDS_BASE, $SQL_SELECT_FIELDS_ADGROUP_ADDITIONAL_TARGETINGS);
    state $json_obj;
    $json_obj //= JSON->new->utf8(0);

    Readonly my $DUMMY_DATE_TIME => '1980-01-01 00:00:00';

    # описание условий для выборки данных по фразам: поля, таблицы, условия
    my %tables_specs = (
        bids => {
            fields => \$SQL_SELECT_FIELDS_BIDS,
            joins => \$SQL_JOIN_BIDS,
            extra_where => !$IGNORE_BS_SYNCED ? ['p.statusBsSynced' => 'Sending'] : undef,
        },
        bids_retargeting => {
            fields => \$SQL_SELECT_FIELDS_BIDS_RET,
            joins => \$SQL_JOIN_BIDS_RET,
            extra_where => !$IGNORE_BS_SYNCED ? ['p.statusBsSynced' => 'Sending'] : undef,
        },
        bids_dynamic => {
            fields => \$SQL_SELECT_FIELDS_BIDS_DYN,
            joins => \$SQL_JOIN_BIDS_DYN,
            # Подгружаем данные вне зависимости от phrases.statusBsSynced,
            # так как они нужны для составления BannerLandData в баннере
        },
        bids_performance => {
            fields => \$SQL_SELECT_FIELDS_BIDS_PERF,
            joins => \$SQL_JOIN_BIDS_PERF,
            # Подгружаем данные вне зависимости от phrases.statusBsSynced,
            # так как они нужны для составления BannerLandData в баннере
        },
        bids_base => {
            fields => \$SQL_SELECT_FIELDS_BIDS_BASE,
            joins => \$SQL_JOIN_BIDS_BASE,
            extra_where => !$IGNORE_BS_SYNCED ? ['p.statusBsSynced' => 'Sending'] : undef,
        },
        adgroup_additional_targetings => {
            fields => \$SQL_SELECT_FIELDS_ADGROUP_ADDITIONAL_TARGETINGS,
            joins => \$SQL_JOIN_ADGROUP_ADDITIONAL_TARGETINGS,
            extra_where => !$IGNORE_BS_SYNCED ? ['p.statusBsSynced' => 'Sending'] : undef,
        },
    );

    my %result = (
        data => [],
        phrases => {},
        domains_dict => {},
        dynamic_conditions => {},
        multipliers => _get_empty_multipliers_dict(),
        retargeting_conditions => {},
        ab_segment_retargeting_conditions => {},
        brandsafety_retargeting_conditions => {},
        mobile_content => {},
        feeds => {},
        performance_conditions => {},
        mobapp_info_by_cid => {},
        performance_counters => {},
        additions => {},
        experiments => {},
        campaign_minus_objects => {},
        minus_phrases => {},
        lib_minus_phrases_by_pid => {},
        campaigns_descriptions => {},
        sitelinks_sets => {},
        vcards => {},
        template_properties => {},
        template_resources => {},
        template_variables_images_formats => {},
        banner_permalinks => {},
        client_phones => {},
        disclaimers => {},
        minus_geo => {},
        turbolandings => {},
        aggregator_domains => {},
        billing_aggregates => {},
        banner_page_moderation => {},
        banner_measurers => {},
        camp_measurers => {},
        mediascope_prefixes => {},
        cid_to_bs_order_id => {},
        internal_ad_products => {},
        skadnetwork_slots => {},
        mobile_goal_app_info => {},
        project_param_conditions => {},
        banner_additional_hrefs => {},
        asset_hashes => {},
        camp_autobudget_restarts => {},
    );

    # $QUERY_CAMPS_ONLY означает, что в снепшоте будут только кампании без групп, баннеров и прочих вложенных объектов
    my ($SET_SYNC_KIND, $UNLOCK_KIND, $SEND_FLAGS, $QUERY_CAMPS_ONLY);
    if (is_par_type(qw/full_lb_export/)) {
        ($SET_SYNC_KIND, $UNLOCK_KIND, $SEND_FLAGS) = ('full', 'all', [qw/send_full/]);
        $QUERY_CAMPS_ONLY = 0;
    } elsif (is_par_type(qw/camps_only/)) {
        ($SET_SYNC_KIND, $UNLOCK_KIND, $SEND_FLAGS) = ('camps_only', 'camps_only', [qw/send_camp/]);
        $QUERY_CAMPS_ONLY = 1;
    } else {
        ($SET_SYNC_KIND, $UNLOCK_KIND, $SEND_FLAGS) = ('camps_and_data', 'camps_and_data', [qw/send_camp send_data/]);
        $QUERY_CAMPS_ONLY = 0;
    }

    my @cids;
    for my $cid (keys %$camps) {
        if (any { $camps->{$cid}->{$_} } @$SEND_FLAGS) {
            push @cids, $cid;
        }
    }
    return \%result unless @cids;

    $SQL_SELECT_FIELDS_BIDS //= join(', ', @SELECT_FIELDS_BIDS);
    $SQL_SELECT_FIELDS_BIDS_RET //= join(', ', @SELECT_FIELDS_BIDS_RET);
    $SQL_SELECT_FIELDS_BIDS_DYN //= join(', ', @SELECT_FIELDS_BIDS_DYN);
    $SQL_SELECT_FIELDS_BIDS_PERF //= join(', ', @SELECT_FIELDS_BIDS_PERF);
    $SQL_SELECT_FIELDS_BIDS_BASE //= join(', ', @SELECT_FIELDS_BIDS_BASE);
    $SQL_SELECT_FIELDS_ADGROUP_ADDITIONAL_TARGETINGS //= join(', ', @SELECT_FIELDS_ADGROUP_ADDITIONAL_TARGETINGS);

    $log->out("Let's get snapshot data...");
    my $phrases_cnt = 0;
    my $multipliers_cnt = 0;
    my $banners_with_minus_geo_cnt = 0;
    my $cids_in = join(',', @cids);
    my $only_cids_sql = $only_cids && @$only_cids ? sprintf('AND c.cid IN (%s)', join(',', @$only_cids)) : '';
    # ретаргетинг сначала откладываем "в сторонку", потому что нужно фильтровать по недоступности целей
    my @phrases_bids_retargeting;
    my %retargeting_multipliers_for;
    my $dont_send_extra_banner_image_formats = _dont_send_extra_banner_image_formats();
    my %bids_to_log_for_images_with_deleted_formats;

    my $sql_where = BS::Export::get_sql_where($IGNORE_BS_SYNCED);
    my $sql_where_data = BS::Export::get_sql_where_data($IGNORE_BS_SYNCED);
    my $sql_where_unsync = BS::Export::get_sql_where_unsync($IGNORE_BS_SYNCED);

    my $select_time = Time::HiRes::time;
    do_sql($dbh2, "SET TRANSACTION ISOLATION LEVEL REPEATABLE READ");
    do_in_transaction {
        my $data_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'data');
        my $SEQ_TIME_COLUMN = is_par_type('full_lb_export') ? 'q.full_export_seq_time' : 'q.seq_time';
        my $QUEUE_TIME_COLUMN = 'q.queue_time';
        my $QUEUE_TABLE = 'bs_export_queue';
        if ($DEBUG_MODE) {
            # поскольку кампания может в очереди и не стоять, а запрос сформировать хочется
            # джойним campaigns саму с собой, а поля очереди заменяем нулями
            $QUEUE_TABLE = 'campaigns';
            $SEQ_TIME_COLUMN = sql_quote($DUMMY_DATE_TIME);
            $QUEUE_TIME_COLUMN = sql_quote($DUMMY_DATE_TIME);
        }
        my $SQL_FROM_JOINS = _get_sql_from_joins($QUERY_CAMPS_ONLY);

        # https://st.yandex-team.ru/DIRECT-106676#5daf71ae701665001dc56149
        do_sql($dbh2, "SET SESSION optimizer_switch='derived_merge=off'");

        # JAVA-EXPORT: BsExportSnapshot#fetch
        # запрос распилен на несколько отдельных частей, и в нем оставлены только id и вспомогательные данные
        my $data = get_all_sql($dbh2, "
                                  SELECT STRAIGHT_JOIN
                                         c.seq_time, c.queue_time
                                         $SQL_COMMON_SELECT_FIELDS
                                    FROM (SELECT
                                            q.cid, c.wallet_cid,
                                            c.sum, c.sum_spent,
                                            c.uid AS uid_for_join,
                                            c.archived, c.OrderID, c.ProductID,
                                            c.statusBsSynced, c.type, c.ClientID, c.opts, c.source,
                                            c.currency, c.strategy_name, c.strategy_data, c.strategy_id,
                                            $SEQ_TIME_COLUMN AS seq_time, $QUEUE_TIME_COLUMN AS queue_time
                                          FROM $QUEUE_TABLE q
                                            JOIN campaigns c ON c.cid = q.cid
                                          WHERE q.cid IN ($cids_in)
                                                AND $BS::Export::SQL_WHERE_CAMP_COMMON
                                          ORDER BY $SEQ_TIME_COLUMN, $QUEUE_TIME_COLUMN, q.cid) c
                                         $SQL_FROM_JOINS
                                   WHERE $sql_where_unsync
                                     AND ($sql_where_data)
                                     $only_cids_sql
                                   LIMIT $MAX_ROWS_FOR_SNAPSHOT
                       ");
        undef $data_profile;

        # turn it back
        do_sql($dbh2, "SET SESSION optimizer_switch='derived_merge=on'");

        if (scalar(@$data) == $MAX_ROWS_FOR_SNAPSHOT) {
            # в BS::ExportQuery::get_query будем исключать последнюю группу из пачки, поэтому сортируем данные.
            # при этом полагаемся на то, что MySQL join'ит phrases к campaigns по Nested-Loop алгоритму,
            # т.е. данные из phrases c одинаковым cid - идут подряд, а не вперемешку
            # сохраняем исходную сортировку + pid
            my $sort_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'sorting_data');
            $result{data} = [ xsort { $_->{seq_time}, $_->{queue_time}, $_->{campaign_pId}, ($_->{context_pId}//0) } @$data ];

            # JAVA-EXPORT: на nested-join не полагаемся, используем явную сортировку в базе
            # для кампаний реализовано внутри SnapshotDataFetcher#preFetch с использованием PreFilterCampaignsStep#getExcessCampaignIds
            # для групп - пока еще нет.
        } else {
            $result{data} = $data;
        }
        undef $data;

        # выборка id'шников для получения вспомогательных данных
        my (%domain_id_to_fetch, %dyn_cond_id_to_fetch, %ret_cond_id_to_fetch, %mobile_content_id_to_fetch, %perf_cond_id_to_fetch, %cid_to_fetch_mobapp_info, %cid_to_fetch_perf_counters);
        my %cid_to_fetch_campaign;
        my %cid_to_fetch_experiments;
        my (%dynamic_pid_to_fetch, %perf_pid_to_fetch, %pid_to_fetch_minus_geo, %pid_to_fetch_additional_targetings, %pid_to_fetch_lib_minus_words);
        my (%pid_to_fetch_bids, %pid_to_fetch_retargeting_and_relevance_match);
        my (%cid_to_fetch_multipliers, %cid2pids_to_fetch_multipliers);
        my %ab_segment_stat_ret_cond_ids_by_cid_to_fetch;
        my %ab_segment_ret_cond_ids_to_fetch;
        my %brandsafety_ret_cond_ids_to_fetch;
        my %feed_id_to_fetch;
        my %project_param_condition_id_to_fetch;
        my %cids_to_fetch_ignoring_bs_monitoring;
        my (%cids_to_fetch_campaign_minus_objects, %mw_id_to_fetch);
        my %sitelinks_set_id_to_fetch;
        my %vcard_id_to_fetch;
        my %bids_to_fetch_banner_permalinks;
        my %phone_ids_to_fetch_client_phone;
        my %bids_to_fetch_disclaimers;
        my %turbolandings_to_fetch_hrefs;
        my %banner_images_formats_to_fetch;
        my %client_ids_for_banner_images_formats_to_fetch;
        my %bids_to_fetch_aggregator_domains;
        my %clients_to_fetch;
        my %bids_to_fetch_banner_page_moderation;
        my %bids_to_fetch_banner_measurers;
        my %cids_to_fetch_camp_measurers;
        my %cids_to_fetch_bs_order_id;
        my %template_ids_to_fetch;
        my %template_resource_id_to_fetch;
        my %bids_to_fetch_banner_tns_id;
        my %turbo_app_info_id_to_fetch_content;
        my %client_id_to_fetch_internal_ad_product;
        my %cid_to_fetch_skadnetwork_slots;
        my %goal_ids_to_fetch_goal_store_content_ids;
        my %bids_to_fetch_banner_additional_hrefs;

        # список выгруженных переменных баннеров внутренней рекламы. Получаем из banners_internal.template_variables
        my @fetched_banners_template_variables;

        # инициализируем типы объектов к которым привязаны отправляемые сейчас в транспорте дополнения
        my %targets_to_fetch_additions = map { $_ => [] } uniq map { $_->{get_by} } values %BS::Export::ADDITION_TYPES;

        for my $row (@{ $result{data} }) {
            # удаляем вспомогательные данные
            delete @{ $row }{qw/ seq_time queue_time /};

            # Для получения подробных данных о кампании
            $cid_to_fetch_campaign{ $row->{campaign_pId} } = undef;

            # для получения коэффициентов цены на кампанию
            $cid_to_fetch_multipliers{ $row->{campaign_pId} } = undef;
            # и для получения значений "игнорировать мониторинг объявлений до"
            $cids_to_fetch_ignoring_bs_monitoring{ $row->{campaign_pId} } = undef;

            # для выборки данных по A/B экспериментам
            $cid_to_fetch_experiments{ $row->{campaign_pId} } = undef;

            $cids_to_fetch_campaign_minus_objects{ $row->{campaign_pId} } = undef;

            # для получаения данных об измерителях
            $cids_to_fetch_camp_measurers{$row->{campaign_pId}} = undef;

            if ($row->{context_pId} && $row->{context_pId} > 0) {
                if ($row->{adgroup_type} eq 'base') {
                    # номера групп для выборки данных о фразах/ретаргетинге
                    $pid_to_fetch_bids{ $row->{context_pId} } = undef;
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                    $pid_to_fetch_additional_targetings{ $row->{context_pId} } = undef;

                    if ($row->{text_adgroup_feed_id}) {
                        $feed_id_to_fetch{ $row->{text_adgroup_feed_id} } = undef;
                    }
                } elsif ($row->{adgroup_type} eq 'dynamic' ) {
                    # номера групп для выборки данных о нацеливаниях
                    $dynamic_pid_to_fetch{ $row->{context_pId} } = undef;

                    if ($row->{dynamic_main_domain_id}) {
                        # domain_id для выборки доменов
                        $domain_id_to_fetch{ $row->{dynamic_main_domain_id} } = undef;
                    }
                    if ($row->{dyn_adgroup_feed_id}) {
                        # номера фидов для выборки данных по фидам
                        $feed_id_to_fetch{ $row->{dyn_adgroup_feed_id} } = undef;
                    }
                } elsif ($row->{adgroup_type} eq 'mobile_content') {
                    # номера групп для выборки данных о фразах/ретаргетинге
                    $pid_to_fetch_bids{ $row->{context_pId} } = undef;
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'performance') {
                    # номера групп для выборки фильтров перфоманс баннеров (bids_performance)
                    $perf_pid_to_fetch{ $row->{context_pId} } = undef;
                    # номера фидов для выборки данных по фидам
                    $feed_id_to_fetch{ $row->{perf_adgroup_feed_id} } = undef;
                } elsif ($row->{adgroup_type} eq 'mcbanner') {
                    # номера групп для выборки данных о фразах
                    $pid_to_fetch_bids{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_banner' || $row->{adgroup_type} eq 'cpm_video') {
                    # номера групп для выборки данных о фразах
                    $pid_to_fetch_bids{ $row->{context_pId} } = undef;
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                    $pid_to_fetch_additional_targetings{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_outdoor') {
                    # номера групп для выборки данных о ретаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_yndx_frontpage') {
                    # номера групп для выборки данных о ретаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_geoproduct') {
                    # номера групп для выборки данных о ретаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'internal') {
                    $pid_to_fetch_additional_targetings{ $row->{context_pId} } = undef;
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'content_promotion_video') {
                    # номера групп для выборки данных о фразах
                    $pid_to_fetch_bids{ $row->{context_pId} } = undef;
                    # номера групп для выборки данных об автотаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'content_promotion') {
                    # номера групп для выборки данных о фразах
                    $pid_to_fetch_bids{ $row->{context_pId} } = undef;
                    # номера групп для выборки данных об автотаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_indoor') {
                    # номера групп для выборки данных о ретаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_audio') {
                    # номера групп для выборки данных о ретаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } elsif ($row->{adgroup_type} eq 'cpm_geo_pin') {
                    # номера групп для выборки данных о ретаргетинге
                    $pid_to_fetch_retargeting_and_relevance_match{ $row->{context_pId} } = undef;
                } else {
                    # при появлении таких предупреждений (т.е. новых видов групп) нужно проверить
                    # и подумать: для каких еще выборок данных нужно запоминать pid этой группы,
                    # дописать elsif ($row->{adgroup_type} eq 'new_cool_type') {}
                    # даже если идентификаторы групп ни для каких выборок не нужны.
                    $log->warn("pid: $row->{context_pId}: unknown adgroup_type: " . ($row->{adgroup_type} // 'undef'));
                }
                # номера групп для выборки минус-регионов относящимся к баннерам
                $pid_to_fetch_minus_geo{ $row->{context_pId} } = undef;

                # номера минус-фраз для получения минус-слов и минус-фраз
                if (defined $row->{mw_id}) {
                    $mw_id_to_fetch{ $row->{mw_id} } = undef;
                };

                # номера групп для выборки библиотечных минус-слов и минус-фраз
                $pid_to_fetch_lib_minus_words{ $row->{context_pId} } = undef;

                # для получения коэффициентов цены на группу
                # группируем по cid, т.к. таблица не индексирована по pid (но есть индекс cid-pid)
                push @{ $cid2pids_to_fetch_multipliers{ $row->{campaign_pId} } }, $row->{context_pId};
            }

            if ($row->{banner_pId} && $row->{banner_pId} > 0) {
                if ($row->{sitelinks_set_id} && $row->{statusSitelinksModerate} eq "Yes") {
                    $sitelinks_set_id_to_fetch{ $row->{sitelinks_set_id} } = undef;
                }
                $bids_to_fetch_disclaimers{$row->{banner_pId}} = undef;

                if ($row->{banner_tl_statusModerate} && $row->{banner_tl_statusModerate} eq 'Yes') {
                    $turbolandings_to_fetch_hrefs{$row->{banner_tl_id}} = undef;
                }

                $bids_to_fetch_banner_permalinks{$row->{banner_pId}} = undef;
                $bids_to_fetch_banner_measurers{$row->{banner_pId}} = undef;
                $bids_to_fetch_banner_tns_id{$row->{banner_pId}} = undef;
                $bids_to_fetch_banner_additional_hrefs{$row->{banner_pId}} = undef;

                if ($row->{banner_type} eq 'cpm_outdoor') {
                    $bids_to_fetch_banner_page_moderation{$row->{banner_pId}} = undef;
                }

                if ($row->{client_phone_id}) {
                    $phone_ids_to_fetch_client_phone{$row->{client_phone_id}} = undef;
                }
            }

            # mobile_content_id для получения данных о мобильном контенте
            if ($row->{mobile_content_id}) {
                $mobile_content_id_to_fetch{ $row->{mobile_content_id} } = undef;
            }

            # bid/pid/cid (при надобности), для выборки дополнений
            for my $target (keys %targets_to_fetch_additions) {
                if ($BS::Export::ADDITION_TARGET_TO_SNAPSHOT_COLUMN{$target}) {
                    if (defined $row->{$BS::Export::ADDITION_TARGET_TO_SNAPSHOT_COLUMN{$target}}) {
                        push @{$targets_to_fetch_additions{$target}}, $row->{$BS::Export::ADDITION_TARGET_TO_SNAPSHOT_COLUMN{$target}};
                    }
                } else {
                    die("Unsupportable target for addition: $target");
                }
            }

            # для подгрузки контактной информации из таблиц vcards, org_details, addresses и maps
            # подгружаем только такие визитки, которые прошли модерацию хотя бы с одним из привязанных баннеров
            if ($row->{vcard_id} && $row->{phoneflag} eq 'Yes') {
                $vcard_id_to_fetch{ $row->{vcard_id} } = undef;
            }

            # для подгрузки информации по форматам картинок для баннеров
            my $banner_image = $row->{banner_image} // $row->{big_king_banner_image};
            if ($banner_image) {
                $banner_images_formats_to_fetch{ $banner_image } = undef;
                $client_ids_for_banner_images_formats_to_fetch{ $row->{ClientID} } = undef;
                if ($row->{banner_pId} && $dont_send_extra_banner_image_formats) {
                    $bids_to_log_for_images_with_deleted_formats{ $row->{banner_pId} } = undef;
                }
            }

            # для подгрузки информации по ресурсам, необходимых для шаблона внутренней рекламы
            if ($row->{template_variables}) {
                my $template_variables = $json_obj->decode(Encode::decode('UTF-8', $row->{template_variables}));

                for my $variable (@$template_variables) {
                    # игнорим переменные без template_resource_id. Такого быть не должно, но защититься будет не лишним
                    next unless $variable->{template_resource_id};

                    $template_resource_id_to_fetch{ $variable->{template_resource_id} } = undef;
                    push @fetched_banners_template_variables, $variable;
                }

                $template_ids_to_fetch{$row->{template_id}} = undef;
            }

            # для подгрузки информации о доменах агрегаторах
            if ($row->{banner_pId} && $row->{Href}) {
                $bids_to_fetch_aggregator_domains{ $row->{banner_pId} } = undef;
            }

            # для подгрузки информации о турбо-аппах
            if ($row->{banner_pId} && $row->{banner_turbo_app_info_id}) {
                $turbo_app_info_id_to_fetch_content{ $row->{banner_turbo_app_info_id} } = $row->{banner_turbo_app_type};
            }

            # для подгрузки информации о формулах таргетинга на параметры проектов
            if ($row->{project_param_conditions}) {
                map { $project_param_condition_id_to_fetch{ $_ } = undef } @{from_json($row->{project_param_conditions})};
            }

            $clients_to_fetch{ $row->{ClientID} } = undef;
        }

        if (%clients_to_fetch) {
            $result{mediascope_prefixes} = get_hash_sql($dbh2,
                                             ["SELECT ClientID, settings->>'\$.tmsec_prefix'
                                               FROM client_measurers_settings",
                                             WHERE => {ClientID => [keys %clients_to_fetch], measurer_system => 'mediascope'}]);
        }

        if (%pid_to_fetch_bids) {
            my @pids = keys(%pid_to_fetch_bids);
            %pid_to_fetch_bids = ();
            for my $table (qw/bids/) {
                $tables_specs{$table}->{pids} = \@pids;
            }
        }
        if (%pid_to_fetch_retargeting_and_relevance_match) {
            my @pids = keys(%pid_to_fetch_retargeting_and_relevance_match);
            %pid_to_fetch_retargeting_and_relevance_match = ();
            for my $table (qw/bids_retargeting bids_base/) {
                $tables_specs{$table}->{pids} = \@pids;
            }
        }
        if (%dynamic_pid_to_fetch) {
            my @pids = keys(%dynamic_pid_to_fetch);
            %dynamic_pid_to_fetch = ();
            $tables_specs{bids_dynamic}->{pids} = \@pids;
        }
        if (%perf_pid_to_fetch) {
            $tables_specs{bids_performance}->{pids} = [ keys %perf_pid_to_fetch ];
            %perf_pid_to_fetch = ();
        }
        if (%pid_to_fetch_additional_targetings) {
            $tables_specs{adgroup_additional_targetings}->{pids} = [ keys %pid_to_fetch_additional_targetings ];
            %pid_to_fetch_additional_targetings = ();
        }
        for my $table (qw/bids bids_retargeting bids_dynamic bids_performance bids_base adgroup_additional_targetings/) {
            my $spec = $tables_specs{$table};
            # если групп для выборки данных из таблицы нет - пропускаем таблицу
            next unless $spec->{pids} && @{ $spec->{pids} };

            my $phrases_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => "phrases,$table");
            my $sth = exec_sql($dbh2, ['SELECT STRAIGHT_JOIN', ${ $spec->{fields} },
                                       'FROM phrases p', ${ $spec->{joins} },
                                       'LEFT JOIN group_params gp ON gp.pid = p.pid',
                                       WHERE => {
                                          'p.pid__int' => $spec->{pids},
                                          ($spec->{extra_where} ? @{ $spec->{extra_where} } : ()),
                                       },
                                      ]);
            while ( my $row = $sth->fetchrow_hashref() ) {
                # все кроме ретаргетинга сразу сохраняем в результат, его - позже и с фильтрацией
                if ($table ne 'bids_retargeting') {
                    $phrases_cnt++;
                    push @{ $result{phrases}->{ $row->{pid} } }, $row;
                }

                if ($table eq 'bids_retargeting') {
                    # выбираем ret_cond_id для получения condition_json и проверки на недоступность целей
                    $ret_cond_id_to_fetch{ $row->{phrase_Id} } = undef;
                    # откладываем для последующего разбора
                    push @phrases_bids_retargeting, $row;
                } elsif ($table eq 'bids_dynamic') {
                    # выбираем IDшники условий нацеливания
                    # NB: можно не подгружать условия для выключенных нацеливаний
                    $dyn_cond_id_to_fetch{ $row->{phrase_Id} } = undef;
                } elsif ($table eq 'bids_performance') {
                    $perf_cond_id_to_fetch{ $row->{phrase_Id} } = undef;
                }
            }
            $sth->finish();
        }

        if (%cid_to_fetch_multipliers || %cid2pids_to_fetch_multipliers) {
            my $multipliers_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot',
                                                                 tags => "hierarchical_multipliers");

            my %common_params = (
                multipliers_ref => $result{multipliers},
                ret_cond_id_to_fetch => \%ret_cond_id_to_fetch,
                retargeting_multipliers => \%retargeting_multipliers_for,
            );

            if (%cid_to_fetch_multipliers) {
                my @cids_to_fetch_multipliers = keys(%cid_to_fetch_multipliers);
                %cid_to_fetch_multipliers = ();

                $multipliers_cnt += _load_multipliers(key => 'cid', ids_ref => \@cids_to_fetch_multipliers, %common_params);
            }

            if (%cid2pids_to_fetch_multipliers) {
                my @cid_pid_pairs;
                for my $cid (keys(%cid2pids_to_fetch_multipliers)) {
                    push @cid_pid_pairs, _AND => [cid__int => $cid, pid__int => $cid2pids_to_fetch_multipliers{$cid}];
                }
                %cid2pids_to_fetch_multipliers = ();

                $multipliers_cnt += _load_multipliers(key => 'pid', ids_ref => \@cid_pid_pairs, %common_params);
            }
        }

        if (%ret_cond_id_to_fetch) {
            my $retargeting_conditions_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot',
                                                                            tags => "retargeting_conditions");

            my @ret_cond_ids = keys(%ret_cond_id_to_fetch);
            %ret_cond_id_to_fetch = ();

            my $inaccessible_retargetings = get_hash_sql($dbh2, ['SELECT DISTINCT ret_cond_id',
                                                                 'FROM retargeting_goals',
                                                                  WHERE => {
                                                                    ret_cond_id__int => \@ret_cond_ids,
                                                                    is_accessible => 0,
                                                                  },
                                                                ]);

            if (%$inaccessible_retargetings) {
                # убираем из выборки условия с недоступными целями, если такие есть
                @ret_cond_ids = grep { !exists $inaccessible_retargetings->{$_} } @ret_cond_ids;
            }

            # выбираем сами условия
            $result{retargeting_conditions} = get_hashes_hash_sql($dbh2, ['SELECT ret_cond_id, condition_json, retargeting_conditions_type, FIND_IN_SET("interest", properties) as is_interest',
                                                                   'FROM retargeting_conditions',
                                                                   WHERE => {
                                                                        ret_cond_id__int => \@ret_cond_ids,
                                                                   },
                                                                  ]);
            if (any {$_->{is_interest}} values %{$result{retargeting_conditions}}) {
                my $interests = get_hash_sql(PPCDICT,
                    "SELECT import_id, name
                       FROM targeting_categories
                      WHERE state = 'Submitted' AND
                            targeting_type = 'rmp_interest'");
                my $i18n_guard = Yandex::I18n::init_i18n_guard('en');

                for my $cond (values %{ $result{retargeting_conditions} }) {
                    next if !$cond->{is_interest};

                    # Здесь всегда by design должен быть JSON со списком из одного элемента,
                    # содержащий еще один список из одного элемента:
                    # "[{"goals":[{"goal_id":118654,"time":90}],"type":"all"}]"
                    my $goals = from_json($cond->{condition_json});
                    for my $goal_data (@$goals) {
                        for my $goal (@{$goal_data->{goals} || []}) {
                            if (exists $interests->{$goal->{goal_id}}) {
                                $cond->{interest_id} = $goal->{goal_id};
                                $cond->{interest_name} = iget($interests->{$goal->{goal_id}});
                            }
                        }
                    }
                }
            }

            if (any {defined $_->{retargeting_conditions_type} && $_->{retargeting_conditions_type} eq 'interests'} values %{$result{retargeting_conditions}}) {
                my $crypta_goals = get_hashes_hash_sql(PPCDICT,
                    "SELECT goal_id, interest_type, crypta_goal_type, bb_keyword, bb_keyword_value, bb_keyword_short, bb_keyword_value_short FROM crypta_goals");

                for my $cond (values %{ $result{retargeting_conditions} }) {
                    next if !(defined $cond->{retargeting_conditions_type} && $cond->{retargeting_conditions_type} eq 'interests');

                    my $goals = from_json($cond->{condition_json});
                    for my $goal_data (@$goals) {
                        for my $goal (@{$goal_data->{goals} || []}) {
                            next if !(defined $crypta_goals->{$goal->{goal_id}});
                            hash_merge($goal, $crypta_goals->{$goal->{goal_id}});
                        }
                    }
                    $cond->{condition_json} = to_json($goals);
                }
            }
        }

        if (%dyn_cond_id_to_fetch) {
            my $dynamic_conditions_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'dynamic_conditions');
            $result{dynamic_conditions} = get_hashes_hash_sql($dbh2, ['SELECT dyn_cond_id, condition_name, condition_json',
                                                               'FROM dynamic_conditions',
                                                               WHERE => {
                                                                    dyn_cond_id__int => [keys(%dyn_cond_id_to_fetch)],
                                                               },
                                                              ]);
            %dyn_cond_id_to_fetch = ();
        }

        if (%perf_cond_id_to_fetch) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'bids_performance');
            $result{performance_conditions} = get_hashes_hash_sql($dbh2, [
                'SELECT p.perf_filter_id, p.name, p.condition_json, p.target_funnel,
                        p.ret_cond_id, rc.condition_json retargeting_condition_json
                   FROM bids_performance p
              LEFT JOIN retargeting_conditions rc using (ret_cond_id)',
                where => {
                    'p.perf_filter_id__int' => [ keys %perf_cond_id_to_fetch ],
                },
            ]);

            %perf_cond_id_to_fetch = ();
        }

        # для таргетингов mobile_installed_apps нужно выгрузить mobileContent
        if ($tables_specs{adgroup_additional_targetings}->{pids}) {
            for my $pid (@{$tables_specs{adgroup_additional_targetings}->{pids}}) {
                next unless exists $result{phrases}->{$pid};

                my @mobile_installed_apps_targetings = grep { exists $_->{targeting_type} && $_->{targeting_type} eq 'mobile_installed_apps' } @{$result{phrases}->{$pid}};

                for my $targeting (@mobile_installed_apps_targetings) {
                    map { $mobile_content_id_to_fetch{ $_->{mobileContentId} } = undef } grep { exists $_->{mobileContentId} } @{from_json($targeting->{value})};
                }
            }
        }

        if (%mobile_content_id_to_fetch) {
            my $mobile_content_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'mobile_content');
            my $sth_mobc = exec_sql($dbh2, ['SELECT mobc.mobile_content_id',
                                                 ', mobc.min_os_version',
                                                 ', mobc.publisher_domain_id',
                                            $BS::ExportMobileContent::SQL_STORE_DATA_SELECT_FIELDS,
                                            'FROM mobile_content mobc',
                                            WHERE => {
                                                'mobc.mobile_content_id__int' => [keys(%mobile_content_id_to_fetch)],
                                            },
                                           ]);
            %mobile_content_id_to_fetch = ();
            while ( my $row = $sth_mobc->fetchrow_hashref() ) {
                if ($row->{publisher_domain_id}) {
                    $domain_id_to_fetch{ $row->{publisher_domain_id} } = undef;
                }
                $result{mobile_content}->{ $row->{mobile_content_id} } = $row;
            }
            $sth_mobc->finish();
        }

        if (%cid_to_fetch_campaign) {
            my $campaigns_descriptions_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'campaigns_descriptions');
            my $QUEUE_TABLE = 'bs_export_queue';
            my $QUEUE_TIME_COLUMN = 'bsq.queue_time';
            if ($DEBUG_MODE) {
                $QUEUE_TABLE = 'campaigns';
                $QUEUE_TIME_COLUMN = sql_quote($DUMMY_DATE_TIME);
            }
            $result{campaigns_descriptions} = get_hashes_hash_sql($dbh2, ["
                SELECT
                  c.cid AS campaign_pId,
                  c.agencyId,
                  c.strategy_id AS strategy_id,
                  c.statusBsSynced AS c_statusBsSynced,
                  c.OrderId AS campaign_Id,
                  c.type AS c_type,
                  c.statusActive != $Common::CAMP_ACTIVE_SQL AS c_statusActive_sync,
                  c.statusShow AS Stop,
                  c.name AS Description,
                  DATE_FORMAT(c.start_time, '%Y%m%d000000') AS Start_time,
                  c.archived AS campaign_arc,
                  DATE_FORMAT(c.finish_time, '%Y%m%d') AS finish_date,
                  c.statusOpenStat AS campaign_openstat,
                  c.ManagerUID,
                  c.uid,
                  c.platform, c.strategy_name, c.strategy_data,
                  c.autobudget,
                  c.autoOptimization,
                  c.DontShow AS DontShowDomains,
                  c.disabled_ssp,
                  c.disabled_video_placements,
                  c.disabledIps,
                  c.timeTarget,
                  c.timezone_id,
                  IFNULL(c.AgencyID, 0) AS AgencyID,
                  c.ContextLimit,
                  c.ContextPriceCoef,
                  c.day_budget,
                  c.day_budget_show_mode,
                  c.paid_by_certificate,
                  c.attribution_model,
                  c.ab_segment_stat_ret_cond_id,
                  c.ab_segment_ret_cond_id,
                  c.brandsafety_ret_cond_id,
                  c.opts,
                  c.rf,
                  c.rfReset
                  $SQL_SELECT_FIELDS_FOR_EXTRACT_SUM,
                  c.sum_spent,
                  c.source,
                  c.metatype as c_metatype,
                  wc.cid AS wallet_campaign_pId,
                  wc.OrderID AS wallet_campaign_Id,
                  wc.paid_by_certificate AS wallet_paid_by_certificate,
                  wc.day_budget AS wallet_day_budget,
                  wc.sum AS wallet_sum,
                  co.broad_match_flag = 'Yes' AS broad_match_flag,
                  co.broad_match_limit,
                  co.broad_match_rate,
                  co.stopTime,
                  co.competitors_domains,
                  co.status_click_track,
                  co.statusMetricaControl,
                  co.strategy,
                  co.content_lang AS CampLang,
                  co.device_targeting,
                  co.meaningful_goals,
                  co.allowed_page_ids,
                  co.disallowed_page_ids,
                  co.allowed_domains AS camp_allowed_domains,
                  co.allowed_ssp AS camp_allowed_ssp,
                  co.create_time,
                  co.impression_standard_time,
                  co.eshows_banner_rate,
                  co.eshows_video_rate,
                  co.eshows_video_type,
                  co.brand_survey_id,
                  co.is_search_lift_enabled,
                  co.is_cpm_global_ab_segment,
                  co.href_params AS campaign_href_params,
                  u.ClientID,
                  u.statusYandexAdv,
                  u.showOnYandexOnly,
                  cl.is_favicon_blocked,
                  clo.hide_market_rating,
                  clo.non_resident,
                  clo.is_business_unit,
                  clo.social_advertising,
                  clo.default_allowed_domains,
                  find_in_set('no_display_hrefs', clo.client_flags) AS no_display_hrefs,
                  cms.avg_discount,
                  mc.metrika_counters,
                  CAST(c.strategy_data->>'\$.filter_avg_bid' as decimal(8,2)) as filter_avg_bid,
                  CAST(c.strategy_data->>'\$.filter_avg_cpa' as decimal(8,2)) as filter_avg_cpa,
                  DATE_FORMAT($QUEUE_TIME_COLUMN, '%Y%m%d%H%i%s') AS queue_time_for_bs,
                  cmc.mobile_app_id,
                  cyf.allowed_frontpage_types,
                  cyf.is_cpd_paused as cpm_price_is_cpd_paused,
                  FIND_IN_SET('is_virtual', c.opts) as is_virtual,
                  FIND_IN_SET('is_alone_trafaret_allowed', c.opts) as is_alone_trafaret_allowed,
                  FIND_IN_SET('require_filtration_by_dont_show_domains', c.opts) as require_filtration_by_dont_show_domains,
                  FIND_IN_SET('has_turbo_smarts', c.opts) as has_turbo_smarts,
                  FIND_IN_SET('has_turbo_app', c.opts) as has_turbo_app,
                  FIND_IN_SET('as_soon_as_possible', clo.client_flags) as asap,
                  FIND_IN_SET('s2s_tracking_enabled', c.opts) as s2s_tracking_enabled,
                  ccs.calltracking_settings_id as calltracking_settings_id,
                  ccp.status_correct,
                  ccp.status_approve,
                  ccp.package_id as price_package_id,
                  ccp.auction_priority as camp_auction_priority,
                  ccp.is_cpd_paused as cpm_yndx_frontpage_is_cpd_paused,
                  cot.order_type
                FROM campaigns c
                  JOIN $QUEUE_TABLE bsq ON c.cid = bsq.cid
                  LEFT JOIN campaigns wc ON wc.cid = c.wallet_cid
                  LEFT JOIN camp_options co ON co.cid = c.cid
                  LEFT JOIN campaigns_mobile_content cmc ON c.cid = cmc.cid AND c.type = 'mobile_content'
                  LEFT JOIN campaigns_cpm_yndx_frontpage cyf ON c.cid = cyf.cid
                  JOIN users u ON u.uid = c.uid
                  LEFT JOIN campaigns_cpm_price ccp on ccp.cid = c.cid and c.type = 'cpm_price'
                  LEFT JOIN clients cl ON cl.ClientID = u.ClientID
                  LEFT JOIN clients_options clo ON clo.ClientID = u.ClientID
                  LEFT JOIN campaigns_multicurrency_sums cms ON cms.cid = c.cid
                  LEFT JOIN client_currency_changes ccc ON ccc.ClientID = u.ClientID
                  LEFT JOIN currency_convert_queue ccq ON ccq.ClientID = u.ClientID
                  LEFT JOIN camp_metrika_counters mc ON mc.cid = c.cid
                  LEFT JOIN wallet_campaigns wwc ON wwc.wallet_cid = IF(c.type = 'wallet', c.cid, c.wallet_cid)
                  LEFT JOIN camp_calltracking_settings ccs ON ccs.cid = c.cid
                  LEFT JOIN camp_order_types cot ON cot.cid = c.cid
             ", WHERE => {
                    'c.cid' => [keys %cid_to_fetch_campaign],
                }]);

            # Если среди кампаний есть кошельки клиентов, у которых подключен автоовердрафт,
            # нужно дополнительно для них затянуть суммы открученных денег (они понадобятся в extract_sum)
            my @wallet_campaigns_to_get_debts = ();
            my %wallet_to_campaigns;
            my $uac_campaign_ids = [];
            my $uac_heavy_campaign_ids = [];
            foreach my $campaign_description (values $result{campaigns_descriptions}) {
                if (Campaign::Types::is_wallet_camp(type => $campaign_description->{c_type})
                    && $campaign_description->{auto_overdraft_lim} != 0) {
                    push @wallet_campaigns_to_get_debts, $campaign_description;
                }
                # для получения сгенерированныx в Директе OrderID
                if ($campaign_description->{campaign_Id} == 0) {
                    $cids_to_fetch_bs_order_id{ $campaign_description->{campaign_pId} } = undef;
                }
                if (my $wallet_cid = $campaign_description->{wallet_campaign_pId}) {
                    push @{$wallet_to_campaigns{$wallet_cid}}, $campaign_description->{campaign_pId};
                }

                if (
                    ($campaign_description->{source} eq 'uac' || $campaign_description->{source} eq 'widget')
                    && ($campaign_description->{c_type} eq 'text' || _is_send_asset_hash_included_for_all_types_enabled())
                ) {
                    if ($campaign_description->{c_type} eq 'mobile_content') {
                        push $uac_heavy_campaign_ids, $campaign_description->{campaign_pId};
                    } else {
                        push $uac_campaign_ids, $campaign_description->{campaign_pId};
                    }
                }
            }

            if (@wallet_campaigns_to_get_debts || %wallet_to_campaigns) {
                my $sum_debts = WalletUtils::get_sum_debts_for_wallets(
                    [uniq keys(%wallet_to_campaigns), map { $_->{campaign_pId} } @wallet_campaigns_to_get_debts]
                );
                foreach my $campaign_description (@wallet_campaigns_to_get_debts) {
                    $campaign_description->{wallet_sum_debt} = $sum_debts->{$campaign_description->{campaign_pId}} // 0;
                }
                while(my ($wallet_id, $cids) = each %wallet_to_campaigns) {
                    for my $cid (@$cids) {
                        $result{campaigns_descriptions}->{$cid}->{wallet_sum_debt_for_camp} = $sum_debts->{$wallet_id} // 0;
                    }
                }
            }

            if (%cids_to_fetch_bs_order_id) {
                $result{cid_to_bs_order_id} = fetch_bs_order_ids([keys %cids_to_fetch_bs_order_id]);
                %cids_to_fetch_bs_order_id = ();
            }

            my $campid2bannerid2assetids = get_campaigns_asset_hashes($uac_campaign_ids);
            my $campid2bannerid2assetids_heavy = get_campaigns_asset_hashes($uac_heavy_campaign_ids, is_heavy => 1);
            if (( $campid2bannerid2assetids && %$campid2bannerid2assetids )
                  || ( $campid2bannerid2assetids_heavy && %$campid2bannerid2assetids_heavy )) {
                $result{asset_hashes} = hash_merge $campid2bannerid2assetids, $campid2bannerid2assetids_heavy;
            }

            if ($LOAD_AUTOBUDGET_RESTART) {
                $result{camp_autobudget_restarts} = _get_camp_autobudget_restarts([keys %cid_to_fetch_campaign]);
            }

            #заполняем значениями из пакета, если они есть.
            my @price_package_ids = uniq map {$_->{price_package_id}} values $result{campaigns_descriptions};
            my $price_packages = get_hashes_hash_sql(PPCDICT,
                ['SELECT package_id, allowed_domains AS package_allowed_domains, available_ad_group_types, auction_priority as package_auction_priority, '.
                 'bu.id as business_unit, bu.partner_share, is_cpd AS package_is_cpd, ',
                 'is_frontpage as package_is_frontpage',
                 'FROM cpm_price_packages cpp',
                 'LEFT JOIN products p ON p.ProductID=cpp.ProductID',
                 'LEFT JOIN business_units_info bu on p.business_unit = bu.id',
                  WHERE => { package_id => \@price_package_ids }
                ]);

            foreach my $campaign_description (values $result{campaigns_descriptions}) {
                my $price_package_id = $campaign_description->{price_package_id};
                if (defined $price_package_id && defined $price_packages->{$price_package_id}) {
                    hash_copy $campaign_description,
                              $price_packages->{$price_package_id},
                              qw/package_allowed_domains price_markups available_ad_group_types package_auction_priority
                                 package_is_cpd package_is_frontpage/;
                }

                if (defined $price_package_id && defined $price_packages->{$price_package_id}{business_unit}) {
                    $campaign_description->{partner_share} = $price_packages->{$price_package_id}{partner_share};
                }
                else {
                    $campaign_description->{partner_share} = undef;
                }
            }

            #Подтянем цели турболендингов
            my $camp_goal_expressions;
            my $tl_goals = get_all_sql($dbh2, [
                'SELECT cid, goal_id FROM camp_metrika_goals',
                    WHERE => {cid => [keys %cid_to_fetch_campaign], links_count__gt => 0, _TEXT => 'find_in_set("combined",goal_role)' }
            ]);
            foreach my $row (@$tl_goals){
                push @{$camp_goal_expressions->{$row->{cid}}}, $row->{goal_id};
            }

            #Проверим, есть ли сделки по кампаниям, соберем id агенств для подтягивания preferred_deal_fee_percent
            my $deals;
            my %agency_ids;
            my $cids = [keys %cid_to_fetch_campaign];
            my $deals_data = get_all_sql($dbh2, [
                'SELECT c.cid, deal_id, agencyId FROM campaigns_deals cd JOIN campaigns c ON (c.cid = cd.cid)',
                    WHERE => {
                        'cd.cid' => $cids,
                        'cd.is_deleted' => 0
                    }
            ]);

            my $active_deals = get_hash_sql(PPC(ClientID => [uniq map { $_->{agencyId} } @$deals_data]),  [
                    'SELECT deal_id, 1 FROM deals',
                    WHERE => {deal_id => [uniq map { $_->{deal_id} } @$deals_data], direct_deal_status => 'Active'}
            ]);

            foreach my $row (@$deals_data){
                next unless $active_deals->{$row->{deal_id}};
                push @{$deals->{$row->{cid}}}, $row->{deal_id};
                $agency_ids{$row->{agencyId}} //= 1;
            }

            #Подтянем диалоги
            my $dialogs = get_hashes_hash_sql($dbh2, [
                'SELECT cd.cid
                   , cd.client_dialog_id
                   , cld.skill_id AS skill_id
                   , cld.bot_guid AS bot_guid
                FROM camp_dialogs cd
                LEFT JOIN client_dialogs cld ON cd.client_dialog_id = cld.client_dialog_id',
                WHERE => {
                    'cd.cid' => [keys %cid_to_fetch_campaign],
                    'cld.is_active' => 1,
                }]);

            my $preferred_deal_fee_percents;
            if (keys %agency_ids){
                $preferred_deal_fee_percents = get_hash_sql(PPC(ClientID => [keys %agency_ids]), [
                    'SELECT ClientID, preferred_deal_fee_percent FROM agency_options',
                        WHERE => {ClientID => SHARD_IDS}
                    ]);
            }

            my $campaigns_internal = get_hashes_hash_sql($dbh2, [
                    "SELECT c.cid,
                            ci.is_mobile,
                            ci.restriction_type,
                            ci.restriction_value,
                            ci.rf_close_by_click,
                            ci.page_ids,
                            ci.place_id,
                            ci.rotation_goal_id AS cint_rotation_goal_id
                     FROM campaigns c JOIN campaigns_internal ci ON c.cid = ci.cid AND $SQL_INTERNAL_CAMPAIGN_TYPES",
                    WHERE => {'c.cid' => [keys %cid_to_fetch_campaign]}
                ]);

            my $agencies_client_ids = [uniq map { $_->{AgencyID} } values $result{campaigns_descriptions}];
            my $agencies_data = get_hashes_hash_sql(PPC(ClientID => $agencies_client_ids), [
                "SELECT co.ClientID, co.social_advertising
                 FROM clients_options co",
                WHERE => {'co.ClientID' => $agencies_client_ids}
            ]);

            my $campaigns_promotions;
            my $campaigns_promotions_data = get_all_sql($dbh2, [
                q/SELECT c.cid,
                            DATE_FORMAT(cp.start, "%Y%m%d") as start,
                            DATE_FORMAT(cp.finish, "%Y%m%d") as finish,
                            cp.percent
                     FROM campaigns c JOIN campaigns_promotions cp ON c.cid = cp.cid/,
                WHERE => {'c.cid' => [keys %cid_to_fetch_campaign]}
            ]);
            foreach my $row (@$campaigns_promotions_data){
                push @{$campaigns_promotions->{$row->{cid}}}, {
                    date_from => $row->{start},
                    date_to => $row->{finish},
                    conversion_boost_percent => $row->{percent} * 10000
                };
            }

            %cid_to_fetch_campaign = ();
            undef $cids;
            undef $active_deals;
            undef %agency_ids;

            for my $row (values %{$result{campaigns_descriptions}}) {

                $row->{social_advertising_agency} = $row->{AgencyID} && $agencies_data->{$row->{AgencyID}} &&
                    $agencies_data->{$row->{AgencyID}}->{social_advertising};
                $row->{send_extended_relevance_match_flag} = _send_extended_relevance_match_flag($row->{c_type}, $row->{mobile_app_id});

                if ($row->{c_type} eq 'text' && $camp_goal_expressions->{$row->{campaign_pId}}) {
                    $row->{autobudget_goal_expression} = join '|', map {"$_:1"} @{$camp_goal_expressions->{$row->{campaign_pId}}};
                }
                else{
                    $row->{autobudget_goal_expression} = undef;
                }

                my $camp_dialog = $dialogs->{$row->{campaign_pId}};
                if ($camp_dialog) {
                    $row->{dialog_skill_id} = $camp_dialog->{skill_id};
                    $row->{dialog_bot_guid} = $camp_dialog->{bot_guid};
                } else {
                    $row->{dialog_skill_id} = undef;
                    $row->{dialog_bot_guid} = undef;
                }

                if ($row->{c_type} eq 'mobile_content') {
                    $cid_to_fetch_mobapp_info{ $row->{campaign_pId} } = undef;
                    $cid_to_fetch_skadnetwork_slots{ $row->{campaign_pId} } = undef;
                }

                if ($row->{c_type} eq 'performance' || ($row->{c_type} eq 'text' && $row->{c_metatype} eq 'ecom') ) {
                    $cid_to_fetch_perf_counters{ $row->{campaign_pId} } = undef;
                }
                $row->{deals} = $deals->{$row->{campaign_pId}} // undef;
                $row->{preferred_deal_fee_percent} = $preferred_deal_fee_percents->{$row->{agencyId}} if $row->{deals};

                $ab_segment_stat_ret_cond_ids_by_cid_to_fetch{$row->{campaign_pId}} = $row->{ab_segment_stat_ret_cond_id} if $row->{ab_segment_stat_ret_cond_id};
                $ab_segment_ret_cond_ids_to_fetch{$row->{ab_segment_ret_cond_id}} = undef if $row->{ab_segment_ret_cond_id};

                $brandsafety_ret_cond_ids_to_fetch{$row->{brandsafety_ret_cond_id}} = undef if $row->{brandsafety_ret_cond_id};

                if ($campaigns_internal->{$row->{campaign_pId}}) {
                    $row->{is_mobile} = $campaigns_internal->{$row->{campaign_pId}}->{is_mobile};
                    $row->{restriction_type} = $campaigns_internal->{$row->{campaign_pId}}->{restriction_type};
                    $row->{restriction_value} = $campaigns_internal->{$row->{campaign_pId}}->{restriction_value};
                    $row->{rf_close_by_click} = $campaigns_internal->{$row->{campaign_pId}}->{rf_close_by_click};
                    $row->{page_ids} = $campaigns_internal->{$row->{campaign_pId}}->{page_ids};
                    $row->{place_id} = $campaigns_internal->{$row->{campaign_pId}}->{place_id};
                    $row->{cint_rotation_goal_id} = $campaigns_internal->{$row->{campaign_pId}}->{cint_rotation_goal_id};

                    $client_id_to_fetch_internal_ad_product{$row->{ClientID}} = undef;
                }

                $row->{campaigns_promotions} = $campaigns_promotions->{$row->{campaign_pId}} // undef;

                if ($row->{s2s_tracking_enabled}) {
                    my $mobile_goals = get_strategy_mobile_goals($row->{'strategy_data'}, $row->{'meaningful_goals'});
                    foreach my $mobile_goal (@$mobile_goals) {
                        $goal_ids_to_fetch_goal_store_content_ids{$mobile_goal} = undef;
                    }
                }
            }
        }

        if (%ab_segment_stat_ret_cond_ids_by_cid_to_fetch) {
            my $ab_segment_ret_conds_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'ab_segment_retargeting_conditions');

            my $ab_segment_ret_cond_ids_hash = hash_merge {}, \%ab_segment_ret_cond_ids_to_fetch;
            $ab_segment_ret_cond_ids_hash = hash_merge $ab_segment_ret_cond_ids_hash, {map {$_ => undef} values %ab_segment_stat_ret_cond_ids_by_cid_to_fetch};

            my $inaccessible_ab_segments = get_hash_sql($dbh2, ['SELECT DISTINCT ret_cond_id',
                    'FROM retargeting_goals',
                    WHERE => {
                        ret_cond_id__int => [keys %$ab_segment_ret_cond_ids_hash],
                        is_accessible => 0,
                    },
                ]);

            if (%$inaccessible_ab_segments) {
                # убираем из выборки условия с недоступными целями, если такие есть
                delete $ab_segment_ret_cond_ids_hash->{$_} foreach keys %$inaccessible_ab_segments;
                foreach my $cid (keys %ab_segment_stat_ret_cond_ids_by_cid_to_fetch) {
                    delete $ab_segment_stat_ret_cond_ids_by_cid_to_fetch{$cid} if exists $inaccessible_ab_segments->{$ab_segment_stat_ret_cond_ids_by_cid_to_fetch{$cid}};
                }
            }

            #Удаляем корректировки на кампанию, относящиеся к недоступным аб-сегментам и добавляем идетификаторы из корректировок для загрузки
            #должно выполнятся после _load_multipliers
            for my $for_key (keys(%{$result{multipliers}->{ab_segment}})) {
                next unless $for_key eq 'for_cid';
                for my $key (keys %{$result{multipliers}->{ab_segment}->{$for_key}}) {
                    my $multipliers = $result{multipliers}->{ab_segment}->{$for_key}->{$key};
                    if ($ab_segment_stat_ret_cond_ids_by_cid_to_fetch{$key}) {
                        $ab_segment_ret_cond_ids_hash = hash_merge $ab_segment_ret_cond_ids_hash, {map {$_->{ab_segment_ret_cond_id} => undef} @$multipliers};
                    } else {
                        $result{multipliers}->{ab_segment}->{$for_key}->{$key} = undef;
                    }
                }
            }

            $result{ab_segment_retargeting_conditions} = get_hash_sql($dbh2, ['SELECT ret_cond_id, condition_json',
                    'FROM retargeting_conditions',
                    WHERE => {
                        ret_cond_id__int => [keys %$ab_segment_ret_cond_ids_hash],
                        retargeting_conditions_type => 'ab_segments',
                    },
                ]);
            %ab_segment_stat_ret_cond_ids_by_cid_to_fetch = ();
            %ab_segment_ret_cond_ids_to_fetch = ();
        }

        if (%brandsafety_ret_cond_ids_to_fetch) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'brandsafety_retargeting_conditions');

            $result{brandsafety_retargeting_conditions} = get_hash_sql($dbh2, ["
                SELECT ret_cond_id, condition_json
                FROM retargeting_conditions",
                WHERE => [
                    ret_cond_id__int => [keys %brandsafety_ret_cond_ids_to_fetch],
                    retargeting_conditions_type => 'brandsafety',
                ]
            ]);
            %brandsafety_ret_cond_ids_to_fetch = ();
        }


        if (%cid_to_fetch_mobapp_info) {
            my $perf_counters_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'mobapp_info_by_cid');
            my $sth_mobapp = exec_sql($dbh2, ['
                SELECT cmc.cid, ma.mobile_content_id, ma.domain_id, ma.store_href
                FROM campaigns_mobile_content AS cmc INNER JOIN mobile_apps AS ma ON cmc.mobile_app_id = ma.mobile_app_id
             ', WHERE => {
                    'cmc.cid' => [keys %cid_to_fetch_mobapp_info],
                }
            ]);

            my %mobapp_info_by_cid;
            while ( my $mobapp_info = $sth_mobapp->fetchrow_hashref() ) {
                $mobapp_info_by_cid{ $mobapp_info->{cid} } = $mobapp_info;
                if ( my $domain_id = $mobapp_info->{domain_id} ) {
                    $domain_id_to_fetch{$domain_id} = undef;
                }
            }
            $sth_mobapp->finish();

            $result{mobapp_info_by_cid} = \%mobapp_info_by_cid;
            %cid_to_fetch_mobapp_info = ();
        }

        if (%goal_ids_to_fetch_goal_store_content_ids) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'fetch_goal_store_content_ids');
            # s2s не имеет смысла для аппметрики, поэтому ищем только среди целей внешних трекеров
            $result{mobile_goal_app_info} = get_hashes_hash_sql(PPC(shard => 'all'), [
                'SELECT et.goal_id,',
                    'mc.os_type as os_type, mc.store_content_id as store_content_id',
                'FROM mobile_app_goals_external_tracker et',
                'JOIN mobile_apps ma ON ma.mobile_app_id=et.mobile_app_id',
                'JOIN mobile_content mc ON mc.mobile_content_id=ma.mobile_content_id',
                WHERE => [
                    'et.goal_id' => [keys %goal_ids_to_fetch_goal_store_content_ids],
                ]
            ]);
            %goal_ids_to_fetch_goal_store_content_ids = ();
        }

        if (%cid_to_fetch_skadnetwork_slots) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'fetch_skadnetwork_slots');
            $result{skadnetwork_slots} = get_hash_sql(PPCDICT, ['
                SELECT cid, slot
                FROM ios_skadnetwork_slots',
                WHERE => [
                    cid => [keys %cid_to_fetch_skadnetwork_slots],
                ]
            ]);
            %cid_to_fetch_skadnetwork_slots = ();
        }

        if (%cid_to_fetch_perf_counters) {
            my $perf_counters_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'performance_counters');
            $result{performance_counters} = get_hash_sql($dbh2, ['
                SELECT mc.cid, MIN(mc.metrika_counter)
                FROM metrika_counters mc
             ', WHERE => {
                    'mc.cid' => [keys %cid_to_fetch_perf_counters],
                }, '
                GROUP BY mc.cid
            ']);
            %cid_to_fetch_perf_counters = ();
        }

        if (%feed_id_to_fetch) {
            my $feeds_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'feeds');
            $result{feeds} = get_hashes_hash_sql($dbh2, [
                'SELECT feed_id',
                     ', url, login, encrypted_password, is_remove_utm, business_type, feed_type, target_domain',
                'FROM feeds',
                WHERE => {
                    feed_id__int => [ keys (%feed_id_to_fetch) ],
                },
            ]);
            %feed_id_to_fetch = ();
        }

        if (%domain_id_to_fetch) {
            my $domains_dict_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'domains_dict');
            $result{domains_dict} = get_hash_sql($dbh2, ['SELECT domain_id, domain',
                                                         'FROM domains',
                                                         WHERE => {
                                                             domain_id__int => [ keys(%domain_id_to_fetch) ],
                                                         },
                                                        ]);
            %domain_id_to_fetch = ();
        }

        for my $addition_type (keys %BS::Export::ADDITION_TYPES) {
            my $addition_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => $addition_type);
            my $addition_opts = $BS::Export::ADDITION_TYPES{$addition_type};
            if (@{$targets_to_fetch_additions{$addition_opts->{get_by}} // []}) {
                if ($addition_type eq 'callout') {
                    $result{additions}->{$addition_type} = Direct::BannersAdditions->get_by(additions_type => $addition_type,
                                                                                            $addition_opts->{get_by} => $targets_to_fetch_additions{$addition_opts->{get_by}},
                                                                                            %{$addition_opts->{get_using_model_opts} // {}}
                                                                                           )->items_callouts_by($addition_opts->{get_by});
                } elsif ($addition_type eq 'experiment') {
                    $result{additions}->{$addition_type} = get_hash_sql($dbh2, ['
                        SELECT ba.bid, aie.experiment_json
                          FROM additions_item_experiments aie
                          JOIN banners_additions ba USING(additions_item_id)
                     ', WHERE => {
                            'ba.bid' => $targets_to_fetch_additions{$addition_opts->{get_by}},
                            'ba.additions_type' => $addition_type,
                        }]);
                } elsif ($addition_type eq 'image_ad') {
                    # TODO: Ядро должно добавить метод Direct::Images->get_by, возможно стоит перейти на него.
                    $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT i.bid, i.image_text,
                               bif.mds_group_id AS banner_image_mds_group_id,
                               bif.namespace AS banner_image_namespace,
                               bif.image_hash AS banner_image,
                               bif.width, bif.height,
                               bif.avatars_host AS banner_image_avatars_host,
                               c.ClientID
                          FROM images i
                          JOIN banners b USING(bid)
                          JOIN banner_images_formats bif USING(image_hash)
                          JOIN campaigns c ON (b.cid = c.cid)
                     ', WHERE => {
                            'i.bid' => $targets_to_fetch_additions{$addition_opts->{get_by}},
                            'b.statusModerate' => 'Yes',
                            'i.statusModerate' => 'Yes',
                        }]);
                } elsif ($addition_type eq 'canvas') {

                    $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT b_perf.bid, b_perf.creative_id as CreativeID, b_perf.extracted_text, perf_cr.moderate_info,
                               perf_cr.statusModerate as CreativeStatus, perf_cr.version as CreativeVersion,
                               perf_cr.creative_type as perf_creative_type, perf_cr.layout_id as perf_layout_id
                          FROM banners_performance  b_perf
                          JOIN banners b USING(bid)
                          JOIN perf_creatives perf_cr USING(creative_id)
                     ', WHERE => {
                            'b_perf.bid' => $targets_to_fetch_additions{$addition_opts->{get_by}},
                            'b_perf.statusModerate' => 'Yes',
                            'perf_cr.creative_type' => 'canvas',
                        }]);
                } elsif ($addition_type eq 'html5_creative') {

                    $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT b_perf.bid, b_perf.creative_id as CreativeID, b_perf.extracted_text, perf_cr.moderate_info,
                               perf_cr.statusModerate as CreativeStatus, perf_cr.version as CreativeVersion,
                               perf_cr.yabs_data as RenderInfo,
                               perf_cr.source_media_type as CreativeComposedFrom
                          FROM banners_performance  b_perf
                          JOIN banners b USING(bid)
                          JOIN perf_creatives perf_cr USING(creative_id)
                     ', WHERE => {
                            'b_perf.bid' => $targets_to_fetch_additions{$addition_opts->{get_by}},
                            'b_perf.statusModerate' => 'Yes',
                            'perf_cr.creative_type' => 'html5_creative',
                        }]);
                } elsif ($addition_type eq 'auto_video') {

                    $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT bp.bid, perf_cr.stock_creative_id, bp.extracted_text, perf_cr.moderate_info, acv.is_non_skippable
                          FROM perf_creatives perf_cr
                          JOIN banners_performance bp ON bp.creative_id = perf_cr.creative_id
                          JOIN banners b ON b.bid = bp.bid
                          LEFT JOIN adgroups_cpm_video acv on acv.pid = b.pid
                     LEFT JOIN clients_options clo on clo.ClientID = perf_cr.ClientID
                     ', WHERE => {
                            'bp.bid' => $targets_to_fetch_additions{$addition_opts->{get_by}},
                            'bp.statusModerate' => 'Yes',
                            'b.statusModerate' => 'Yes',
                            'perf_cr.creative_type__in' => [qw/video_addition bannerstorage/],
                            _OR => [
                                'clo.client_flags__is_null' => 1,
                                'clo.client_flags__scheck' => { suspend_video => 0 },
                            ],
                        },
                    ]);

                } elsif ($addition_type eq 'pixel') {
                    $result{additions}->{$addition_type} = Direct::BannersPixels->get_by(additions_type => $addition_type,
                        $addition_opts->{get_by} => $targets_to_fetch_additions{$addition_opts->{get_by}},
                        %{$addition_opts->{get_using_model_opts} // {}}
                    )->items_by($addition_opts->{get_by});

                } elsif ($addition_type eq 'banner_price') {
                    $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT bid, prefix, currency, price, price_old
                          FROM banner_prices
                    ', WHERE => {
                           bid => $targets_to_fetch_additions{$addition_opts->{get_by}},
                        },
                    ]);
                } elsif ($addition_type eq 'content_promotion_video') {
                     $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT b_cpv.bid, b_cpv.packshot_href, cpv.video_metadata, cpv.is_inaccessible
                          FROM banners_content_promotion_video b_cpv
                          JOIN content_promotion_video cpv on b_cpv.content_promotion_video_id = cpv.content_promotion_video_id
                    ', WHERE => {
                           bid => $targets_to_fetch_additions{$addition_opts->{get_by}},
                        },
                    ]);
                } elsif ($addition_type eq 'content_promotion') {
		    $result{additions}->{$addition_type} = get_hashes_hash_sql($dbh2, ['
                        SELECT bcpromo.bid, bcpromo.visit_url, cpromo.metadata, cpromo.is_inaccessible
                          FROM banners_content_promotion bcpromo
                          JOIN content_promotion cpromo on bcpromo.content_promotion_id = cpromo.id
                    ', WHERE => {
			bid => $targets_to_fetch_additions{$addition_opts->{get_by}},
                        },
                    ]);
		}
            }
        }
        %targets_to_fetch_additions = ();

        if (%cid_to_fetch_experiments) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'experiments');
            my $sth_experiments = exec_sql($dbh2, [
                                            'SELECT ce.cid, ce.experiment_id, ce.role, e.percent',
                                            'FROM campaigns_experiments ce',
                                            'JOIN experiments e ON e.experiment_id = ce.experiment_id',
                                            WHERE => {
                                                'ce.cid__int' => [keys(%cid_to_fetch_experiments)],
                                                'e.status' => 'Started',
                                            },
                                       ]);
            while (my $row = $sth_experiments->fetchrow_hashref() ) {
                my $cid = delete $row->{cid};
                $result{experiments}->{$cid} = $row;
            }
            $sth_experiments->finish();
            %cid_to_fetch_experiments = ();
        }

        if (%cids_to_fetch_campaign_minus_objects) {
            my $campaigns_minus_objects = get_hash_sql($dbh2,
                ['SELECT cid, minus_words FROM camp_options',
                    WHERE => {
                        cid__int => [keys %cids_to_fetch_campaign_minus_objects],
                        minus_words__is_not_null => 1,
                    },
                ]);
            for my $cid (keys %$campaigns_minus_objects) {
                my $minus_objects = MinusWordsTools::minus_words_str2array($campaigns_minus_objects->{$cid});
                if (@$minus_objects) {
                    $result{campaign_minus_objects}->{$cid} = $minus_objects;
                }
            }
            %cids_to_fetch_campaign_minus_objects = ();
        }

        if (%mw_id_to_fetch) {
            my $minus_phrases = get_hash_sql($dbh2,
                ['SELECT mw_id, mw_text FROM minus_words',
                    WHERE => {
                        mw_id__int => [keys %mw_id_to_fetch],
                    },
                ]);
            for my $mw_id (keys %$minus_phrases) {
                my $minus_objects = MinusWordsTools::minus_words_str2array($minus_phrases->{$mw_id});
                if (@$minus_objects) {
                    $result{minus_phrases}->{$mw_id} = $minus_objects;
                }
            }
            %mw_id_to_fetch = ();
        }

        if (%sitelinks_set_id_to_fetch) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'sitelinks_sets');
            $result{sitelinks_sets} = Sitelinks::get_sitelinks_by_set_id_multi_without_sharding($dbh2, [keys %sitelinks_set_id_to_fetch]);
            %sitelinks_set_id_to_fetch = ();
        }

        if (%turbolandings_to_fetch_hrefs) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'turbolandings');
            $result{turbolandings} = Direct::TurboLandings::get_turbolandings_by_id_multi_without_sharding($dbh2, [keys %turbolandings_to_fetch_hrefs]);
        }


        if (%vcard_id_to_fetch) {
            $result{vcards} = get_hashes_hash_sql($dbh2, ['SELECT vc.vcard_id, vc.phone, vc.name, vc.street, vc.house, vc.build, vc.apart, vc.metro, vc.contactperson, vc.worktime,
                                                                    vc.city, vc.country, vc.geo_id, vc.extra_message, vc.contact_email, vc.im_client, vc.im_login, vc.permalink,
                                                                    org.ogrn,
                                                                    a.precision, a.map_id, a.map_id_auto,
                                                                    m.mid, m.x, m.y, m.x1, m.y1, m.x2, m.y2
                                                            FROM vcards vc
                                                                LEFT JOIN org_details org ON org.org_details_id = vc.org_details_id
                                                                LEFT JOIN addresses a ON a.aid = vc.address_id
                                                                LEFT JOIN maps m ON m.mid = a.map_id',
                                                            WHERE => { 'vc.vcard_id__int' => [keys %vcard_id_to_fetch] }
                                                        ]);
            %vcard_id_to_fetch = ();
        }

        if (%banner_images_formats_to_fetch) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'banner_images_formats');
            my $data = get_all_sql($dbh2,
                ['SELECT
                      bif.image_hash
                    , p.ClientID
                    , bif.formats as image_formats
                    , bif.mds_meta as image_mds_meta
                    , p.mds_meta_user_override as image_mds_meta_user_override
                    , bif.image_type
                    , bif.mds_group_id as banner_image_mds_group_id
                    , bif.namespace as banner_image_namespace
                    , bif.avatars_host as banner_image_avatars_host
                FROM
                    banner_images_formats bif
                    JOIN banner_images_pool p ON (bif.image_hash = p.image_hash)',
                    WHERE => {
                        'bif.image_hash' => [keys %banner_images_formats_to_fetch],
                        'p.ClientID' => [keys %client_ids_for_banner_images_formats_to_fetch]
                    }
                ]);

            for my $bif ( @$data ) {
                if ($dont_send_extra_banner_image_formats && $bif->{image_formats}) {
                    my $new_image_formats = from_json($bif->{image_formats});

                    #Убираем лишние размеры для отправки (DIRECT-159642)
                    delete @$new_image_formats{qw/x80 x160 x320 x360 x378 x410 x756 x820 y65 y80 y110 y129 y160 y320 y360 y680/};
                    $bif->{image_formats} = to_json($new_image_formats);
                }
                my ($ClientID, $image_hash) = @$bif{qw/ClientID image_hash/};
                foreach my $meta_field (qw/image_mds_meta image_mds_meta_user_override/) {
                    my $meta = delete $bif->{$meta_field};
                    if (defined $meta) {
                        my $field_json = $meta_field.'_json';
                        $bif->{$field_json} = from_json($meta);
                    }
                }
                if ($bif->{image_mds_meta_user_override_json}){
                    $bif->{image_mds_meta_json} = _hash_deep_merge(
                            $bif->{image_mds_meta_json}, delete $bif->{image_mds_meta_user_override_json} );
                }
                Hash::Util::lock_hashref_recurse($bif->{image_mds_meta_json}) if $bif->{image_mds_meta_json};
                $result{banner_images_formats}->{$ClientID}->{$image_hash} = $bif;
            }
            %banner_images_formats_to_fetch = ();
            %client_ids_for_banner_images_formats_to_fetch = ();
        }

        if (%template_resource_id_to_fetch) {
            $result{template_resources} = get_hashes_hash_sql(PPCDICT,
                ['SELECT id
                    , resource_no
                    , template_part_no
                    , FIND_IN_SET("banana_image", options) AS is_image
                  FROM template_resource',
                  WHERE => { id => [keys %template_resource_id_to_fetch] }
                ]);

            # Дополняем данными о смигрированных ресурсах, затирая старые неполные сведения.
            hash_merge($result{template_resources}, get_hashes_hash_sql(PPCDICT,
                ['SELECT direct_template_resource_id as id
                    , resource_no
                    , unified_resource_no
                    , unified_template_resource_id
                    , 0 as template_part_no
                    , FIND_IN_SET("banana_image", options) AS is_image
                  FROM direct_template_resource',
                  WHERE => { direct_template_resource_id => [keys %template_resource_id_to_fetch] }
                ]
            ));

            my @image_hashes = ();
            for my $variable (@fetched_banners_template_variables) {
                my $variable_resource = $result{template_resources}->{$variable->{template_resource_id}};
                next if !defined $variable_resource;

                if ($variable_resource->{is_image} && $variable->{internal_value}) {
                    push @image_hashes, $variable->{internal_value};
                }
            }

            # выгружаем параметры картинок для переменных шаблона
            $result{template_variables_images_formats} = get_hashes_hash_sql($dbh2,
                ['SELECT image_hash as banner_image
                    , width
                    , height
                    , mds_group_id as banner_image_mds_group_id
                    , namespace as banner_image_namespace
                    , avatars_host as banner_image_avatars_host
                  FROM banner_images_formats',
                  WHERE => { image_hash => \@image_hashes }
                ]);

            %template_resource_id_to_fetch = ();
        }

        if (%template_ids_to_fetch) {
            $result{template_properties} = get_hashes_hash_sql(PPCDICT,
                ['SELECT direct_template_id
                    , format_name
                    , state
                  FROM direct_template',
                  WHERE => { direct_template_id => [keys %template_ids_to_fetch] }
                ]
            );
            %template_ids_to_fetch = ();
        }

        if (%bids_to_fetch_banner_permalinks) {
            for my $bids_chunk (chunks([keys %bids_to_fetch_banner_permalinks], 5_000)) {
                my $sth = exec_sql($dbh2, [ 'SELECT
                                                    bpml.bid
                                                    , bpml.permalink
                                                    , bpml.chain_id
                                                    , bpml.permalink_assign_type
                                                    , bpml.is_sent_to_bs
                                                    , bpml.prefer_vcard_over_permalink
                                                    , org.status_publish
                                            FROM banner_permalinks bpml
                                            JOIN banners b ON bpml.bid = b.bid
                                            JOIN campaigns c ON b.cid = c.cid
                                            LEFT JOIN organizations org ON bpml.permalink = org.permalink_id
                                                    AND org.ClientID = c.ClientID',
                                             WHERE => { 'bpml.bid' => $bids_chunk },
                                            'ORDER BY' => 'bpml.bid, bpml.permalink, bpml.chain_id'
                                        ]);

                while (my $row = $sth->fetchrow_hashref()) {
                    my $data = $result{banner_permalinks}{$row->{bid}} // {permalinks => {}, chain_ids => []};
                    push @{$data->{chain_ids}}, $row->{chain_id} if $row->{chain_id};
                    if ($row->{permalink}) {
                        $data->{permalinks}->{$row->{permalink_assign_type}} = hash_cut($row, qw/permalink status_publish is_sent_to_bs prefer_vcard_over_permalink/);
                    }
                    $result{banner_permalinks}{$row->{bid}} //= $data;
                }

                $sth->finish();
            }

            %bids_to_fetch_banner_permalinks = ();
        }

        if (%phone_ids_to_fetch_client_phone) {
            my $client_phones = get_all_sql($dbh2, ['
                SELECT client_phone_id, coalesce(telephony_phone, phone) as phone
                FROM client_phones',
                WHERE => {
                    client_phone_id => [ keys %phone_ids_to_fetch_client_phone ]
                }
            ]);

            for my $row (@$client_phones) {
                $result{client_phones}->{$row->{client_phone_id}} = {
                    phone => $row->{phone},
                };
            }

            %phone_ids_to_fetch_client_phone = ();
        }

        if (%bids_to_fetch_disclaimers) {
            my $disclaimers = Direct::AdditionsItemDisclaimers->
                get_by(banner_id => [keys %bids_to_fetch_disclaimers], get_banner_id => 1)->
                items_by('banner_id');
            for my $bid (keys %$disclaimers){
                $result{disclaimers}->{$bid} = $disclaimers->{$bid}[0]->{disclaimer_text};
            }
            %bids_to_fetch_disclaimers = ();
        }

        if (%pid_to_fetch_minus_geo) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'minus_geo');
            my $sql_where_banner_should_sync_minus_geo = BS::Export::get_sql_where_banner_should_sync_minus_geo($IGNORE_BS_SYNCED);
            my $bid_minus_geo = get_all_sql($dbh2, ["
                SELECT p.pid, b.bid, b.BannerID, bmg.type, bmg.minus_geo
                  FROM campaigns c
                  JOIN phrases p ON c.cid = p.cid
                  JOIN banners b ON p.pid = b.pid
                  JOIN banners_minus_geo bmg ON b.bid = bmg.bid
             LEFT JOIN images on images.bid = b.bid
             LEFT JOIN banner_turbolandings btl on b.bid = btl.bid
             LEFT JOIN banners_performance b_perf on b_perf.bid = b.bid
             LEFT JOIN banner_permalinks bpml on bpml.bid = b.bid
             LEFT JOIN organizations org on org.permalink_id = bpml.permalink AND org.ClientID = c.ClientID
                 WHERE ($sql_where_banner_should_sync_minus_geo)
                   AND ", {
                          'p.pid' => [keys %pid_to_fetch_minus_geo],
                          'bmg.minus_geo__ne' => ''
                       }
                ]);
            for my $mg (@$bid_minus_geo) {
                $result{minus_geo}->{$mg->{pid}}->{$mg->{bid}}->{$mg->{type}} = $mg->{minus_geo};
                $result{minus_geo}->{$mg->{pid}}->{$mg->{bid}}->{is_new} = 1 if $mg->{BannerID} == 0;
            }
            %pid_to_fetch_minus_geo = ();

            for my $group_minus_geo (values %{$result{minus_geo}}) {
                $banners_with_minus_geo_cnt += scalar(keys %$group_minus_geo);
            }
        }

        if (%pid_to_fetch_lib_minus_words) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'lib_minus_words_by_pid');

            my $pid_with_lib_mw_id = get_all_sql($dbh2, ["
                SELECT pid, mw_id
                  FROM adgroups_minus_words",
                WHERE => { 'pid' => [keys %pid_to_fetch_lib_minus_words] }
            ]);
            my $lib_mw_id_to_fetch = [uniq map {$_->{mw_id}} @$pid_with_lib_mw_id ];

            my $lib_minus_word_strs = get_all_sql($dbh2, ["
                SELECT mw_id, mw_text
                  FROM minus_words",
                WHERE => { 'mw_id' => $lib_mw_id_to_fetch }
            ]);

            my %lib_minus_words = map {$_->{mw_id} => MinusWordsTools::minus_words_str2array($_->{mw_text})} @$lib_minus_word_strs;

            for my $lib_mw (@$pid_with_lib_mw_id) {
                my $minus_objects = $lib_minus_words{$lib_mw->{mw_id}};
                if (@$minus_objects) {
                    push @{$result{lib_minus_phrases_by_pid}->{$lib_mw->{pid}} ||= []}, $minus_objects;
                }
            }
            %pid_to_fetch_lib_minus_words = ();
        }

        if (%bids_to_fetch_aggregator_domains) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'aggregator_domains');

            $result{aggregator_domains} = AggregatorDomains::get_aggregator_domains_by_bids_without_sharding($dbh2, [keys %bids_to_fetch_aggregator_domains]);

            %bids_to_fetch_aggregator_domains = ();
        }

        if (%bids_to_fetch_banner_page_moderation) {
            my @bids = keys(%bids_to_fetch_banner_page_moderation);
            %bids_to_fetch_banner_page_moderation = ();

            #отправляем в БК только промодерированные и загруженные баннеры. получение statusModerate - легаси
            my $bid_page_moderation = get_all_sql($dbh2, ['
                        SELECT bid, PageID, statusModerate, version, task_url
                          FROM moderate_banner_pages
                    ', WHERE => {
                bid        => \@bids,
                is_removed => 0,
                'statusModerate' => 'Yes',
                'statusModerateOperator' => 'Yes',
            },
            ]);
            for my $pm (@$bid_page_moderation) {
                push @{$result{banner_page_moderation}->{$pm->{bid}}}, { pageId => $pm->{PageID}, statusModerate => $pm->{statusModerate}, Version => $pm->{version}, TaskURL => $pm->{task_url} };
            }
        }

        if (%bids_to_fetch_banner_measurers) {
            $result{banner_measurers} = Direct::Banners::Measurers->get_by(banner_id => [keys %bids_to_fetch_banner_measurers])->items_by('banner_id');
        }

        if (%cids_to_fetch_camp_measurers) {
            my $camp_measurers = get_all_sql($dbh2, ['SELECT cid, measurer_system, params FROM camp_measurers',
                WHERE => {cid => [keys %cids_to_fetch_camp_measurers]}]);
            for my $camp_measurer (@$camp_measurers) {
                push @{$result{camp_measurers}->{$camp_measurer->{cid}}}, $camp_measurer;
            }
        }

        if (%bids_to_fetch_banner_tns_id) {
            $result{banner_tns_id} = get_hash_sql($dbh2, ["
                SELECT bid, tns_id
                FROM banners_tns",
                WHERE => [
                    bid => [keys %bids_to_fetch_banner_tns_id]
                ]
            ]);
        }

        if (%project_param_condition_id_to_fetch) {
            my $project_param_conditions = get_all_sql(PPCDICT, ['
                SELECT condition_id, condition_json
                FROM project_param_conditions',
                WHERE => {
                    condition_id => [ keys %project_param_condition_id_to_fetch ]
                }
            ]);

            for my $row (@$project_param_conditions) {
                $result{project_param_conditions}->{$row->{condition_id}} = from_json($row->{condition_json});
            }

            %project_param_condition_id_to_fetch = ();
        }

        if (%turbo_app_info_id_to_fetch_content) {
            my $turbo_app_content = get_all_sql($dbh2, ['
                SELECT turbo_app_info_id, turbo_app_id, content
                FROM turbo_apps_info',
                WHERE => {
                    turbo_app_info_id => [ keys %turbo_app_info_id_to_fetch_content ]
                }
            ]);

            for my $row (@$turbo_app_content) {
                $result{turbo_app_content}->{$row->{turbo_app_info_id}} = {
                    TurboAppId => $row->{turbo_app_id},
                    TurboAppContent => $row->{content},
                };
            }

            %turbo_app_info_id_to_fetch_content = ();
        }

        if (%bids_to_fetch_banner_additional_hrefs) {
            $result{banner_additional_hrefs} = Direct::Banners::AdditionalHrefs->get_by(banner_id => [keys %bids_to_fetch_banner_additional_hrefs])->items_by('banner_id');
        }

        if (%clients_to_fetch) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'billing_aggregates');

            $result{billing_aggregates} =
                Direct::BillingAggregates->get_by(client_id => [keys %clients_to_fetch])
                    ->items_by_wallet_and_product();

            %clients_to_fetch = ();
        }

        if (%client_id_to_fetch_internal_ad_product) {
            my $profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => 'internal_ad_products');
            $result{internal_ad_products} = get_hashes_hash_sql($dbh2, [
                'SELECT ClientID, product_name',
                'FROM internal_ad_products',
                WHERE => { ClientID => [ keys %client_id_to_fetch_internal_ad_product ] },
            ]);

            %client_id_to_fetch_internal_ad_product = ();
        }
    };

    if (@phrases_bids_retargeting) {
        # перекладываем отложенные "фразы" по ретаргетингу в результат
        # при этом неявно (по наличию данных условия) проверяется соблюдение двух условий:
        #   все цели в соответствующем условии ретаргетинга доступны клиенту
        #   не нарушена связь bids_retargeting <-> retargeting_conditions
        my $phrases_profile = Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => "phrases,bids_retargeting");

        for my $row (@phrases_bids_retargeting) {
            if (exists $result{retargeting_conditions}->{ $row->{phrase_Id} } ){
                $phrases_cnt++;
                push @{ $result{phrases}->{ $row->{pid} } }, $row;
            }
        }

        @phrases_bids_retargeting = ();
    }

    if (%retargeting_multipliers_for) {
        # перекладываем отложенные коэффиценты по ретаргетингу в результат
        my $reatrgeting_profile;

        for my $key (keys(%retargeting_multipliers_for)) { # 'cid', 'pid'
            my $ret_multipliers_by_key_id = $retargeting_multipliers_for{$key}; # $key_id => []
            next unless $ret_multipliers_by_key_id && %$ret_multipliers_by_key_id;

            my $for_key = "for_$key";
            $reatrgeting_profile //= Yandex::Trace::new_profile('bs_export_worker:get_snapshot', tags => "hierarchical_multipliers");

            for my $key_id (keys(%$ret_multipliers_by_key_id)) {
                my @good_retargeting_multipliers;
                my @good_retargeting_filter_multipliers;
                for my $one_ret_multiplier (@{ $ret_multipliers_by_key_id->{$key_id} }) {
                    if (exists $result{retargeting_conditions}->{ $one_ret_multiplier->{ret_cond_id} }) {
                        if ($one_ret_multiplier->{type} eq 'retargeting_multiplier') {
                            push @good_retargeting_multipliers, $one_ret_multiplier;
                        } else {
                            push @good_retargeting_filter_multipliers, $one_ret_multiplier;
                        }
                        ++$multipliers_cnt;
                    }
                }
                # запаковываем в json
                if (@good_retargeting_multipliers) {
                    $result{multipliers}->{retargeting}->{$for_key}->{$key_id} = $json_obj->encode(\@good_retargeting_multipliers);
                }
                if (@good_retargeting_filter_multipliers) {
                    $result{multipliers}->{retargeting_filter}->{$for_key}->{$key_id} = $json_obj->encode(\@good_retargeting_filter_multipliers);
                }
            }

            %{ $retargeting_multipliers_for{$key} } = ();
        }

        %retargeting_multipliers_for = ();
    }

    $log->out(sprintf("Snapshot data fetched, %.2f seconds elapsed. Selected: %d, limit: %d rows. Extra: ",
                      (Time::HiRes::time - $select_time),
                      scalar(@{ $result{data} }),
                      $MAX_ROWS_FOR_SNAPSHOT,
              ));
    if (%bids_to_log_for_images_with_deleted_formats) {
        $log->out("bids for which was deleted images extra formats: (".join(',', nsort keys %bids_to_log_for_images_with_deleted_formats).")");
    }
    $log->out({
        phrases => $phrases_cnt,
        multipliers => $multipliers_cnt,
        domains_dict => scalar(keys(%{ $result{domains_dict} })),
        dynamic_conditions => scalar(keys(%{ $result{dynamic_conditions} })),
        retargeting_conditions => scalar(keys(%{ $result{retargeting_conditions} })),
        ab_segment_retargeting_conditions => scalar(keys(%{ $result{ab_segment_retargeting_conditions} })),
        brandsafety_retargeting_conditions => scalar(keys(%{ $result{brandsafety_retargeting_conditions} })),
        mobile_content => scalar(values(%{ $result{mobile_content} })),
        additions => { map { $_ => scalar(keys(%{ $result{additions}->{$_} })) } keys($result{additions}) },
        feeds => scalar(keys(%{ $result{feeds} })),
        performance_conditions => scalar(values(%{ $result{performance_conditions} })),
        mobapp_info_by_cid => scalar(keys(%{ $result{mobapp_info_by_cid} })),
        performance_counters => scalar(keys(%{ $result{performance_counters} })),
        experiments => scalar(keys(%{ $result{experiments} })),
        campaign_minus_objects => scalar(keys(%{ $result{campaign_minus_objects} })),
        minus_phrases => scalar(keys(%{ $result{minus_phrases} })),
        lib_minus_phrases_by_pid => scalar(keys(%{ $result{lib_minus_phrases} })),
        campaigns_descriptions => scalar(keys(%{ $result{campaigns_descriptions} })),
        cid_to_bs_order_id => scalar(keys(%{ $result{cid_to_bs_order_id} })),
        sitelinks_sets => scalar(keys(%{ $result{sitelinks_sets} })),
        vcards => scalar(keys(%{ $result{vcards} })),
        banner_permalinks => scalar(keys(%{ $result{banner_permalinks} })),
        client_phones => scalar(keys(%{ $result{client_phones} })),
        disclaimers => scalar(keys(%{ $result{disclaimers} })),
        banners_with_minus_geo => $banners_with_minus_geo_cnt,
        banner_images_formats => scalar((map {keys %$_} values(%{ $result{banner_images_formats} }))),
        template_properties => scalar(keys(%{ $result{template_properties} })),
        template_resources => scalar(keys(%{ $result{template_resources} })),
        template_variables_images_formats => scalar(keys(%{ $result{template_variables_images_formats} })),
        aggregator_domains => scalar(keys(%{ $result{aggregator_domains} })),
        billing_aggregates => scalar(keys(%{ $result{billing_aggregates} })),
        turbo_app_content => scalar(keys(%{ $result{turbo_app_content} })),
        project_param_conditions => scalar(keys(%{ $result{project_param_conditions} })),
        internal_ad_products => scalar(keys(%{ $result{internal_ad_products} })),
        skadnetwork_slots => scalar(keys(%{ $result{skadnetwork_slots} })),
        mobile_goal_app_info => scalar(keys(%{ $result{mobile_goal_app_info} })),
        camp_autobudget_restarts => scalar(keys(%{ $result{camp_autobudget_restarts} })),
    });

    # кампании, которые попали в снэпшот
    my %IN_SNAPSHOT = map { $_->{campaign_pId} => 1 } @{ $result{data} };

    # кампании, которые вообще нужно отправить
    my %UNSYNCED;
    my $SQL_FROM_STAT = BS::Export::get_sql_from_stat($QUERY_CAMPS_ONLY);
    my $unsynced_arr = get_one_column_sql($dbh2, "
        SELECT STRAIGHT_JOIN c.cid
          FROM campaigns c $SQL_FROM_STAT
         WHERE c.cid in ($cids_in) AND ($sql_where) AND ($sql_where_data)
         GROUP BY c.cid
    ");
    for my $unsync_cid (@$unsynced_arr) {
        $UNSYNCED{$unsync_cid} = 1;
    }
    undef $unsynced_arr;

    # если при наличии send_data окажется, что отправлять данные не нужно (%UNSYNCED)
    # и ничего не выбралось (%IN_SNAPSHOT) - считаем, что /данные/ все отправили
    set_sync_campaigns($camps, $SET_SYNC_KIND => [ grep {!$UNSYNCED{$_} && !$IN_SNAPSHOT{$_}} @cids ]);
    # если кампания не влезла по лимитам в отправку данных - освобождаем
    unlock_campaigns($camps, $UNLOCK_KIND => [ grep {$UNSYNCED{$_} && !$IN_SNAPSHOT{$_}} @cids ]);
    $result{select_time} = int($select_time);

    return \%result;
}

sub _send_extended_relevance_match_flag {
    my ($ctype, $mobile_app_id) = @_;
    if (none { $ctype eq $_ } qw/performance mobile_content internal_autobudget internal_distrib internal_free/) {
        return 1;
    }
    elsif (($ctype eq 'mobile_content') && $mobile_app_id) {
        return 1;
    }
    return 0;
}

=head3 _hash_deep_merge
    Простое рекурсивное объединение двух хешей

=cut

my $MAX_HDM_RECURSION_LEVEL = 7;

sub _hash_deep_merge {
    my ($left, $right, $current_recursion_level) = @_;
    $current_recursion_level += 1;

    foreach my $key ( keys %$right) {
        if ( exists $left->{$key} && defined $left->{$key} && $current_recursion_level < $MAX_HDM_RECURSION_LEVEL) {
            #Сверять будем копии значений, чтобы не затрагивать perl'овые флаги
            my ($lval, $rval) = ($left->{$key}, $right->{$key});

            #https://st.yandex-team.ru/DIRECT-107783
            #Если правое значение не определено - не меняем левое, т.е. затереть существующее значение нельзя
            next unless defined $rval;

            # Если элементы совпадают - делать ничего не нужно
            next if $lval eq $rval;
            # Если и слева и справа хеши - мержим их
            if (ref $lval eq 'HASH' && ref $rval eq 'HASH') {
                $left->{$key} = _hash_deep_merge($left->{$key}, $right->{$key}, $current_recursion_level);
            } else {
                #Если это не два хеша - просто используем правое значение
                $left->{$key} = $right->{$key};
            }

        } else {
            #если слева ничего не было - используем правый элемент
            $left->{$key} = $right->{$key};
        }
    }

    return $left;
}

=head3 _get_empty_multipliers_dict

    Получить пустую структуру под словарь с коэффициентами цен
    Описание структуры - хеш со следующими полями:
        mobile => { # множители для мобильных устройств
            for_cid => {    # множители для кампаний
                cid1 => mobile_multipliers_json_data,    # ключ - номер кампании
                cid2 => mobile_multipliers_json_data,    # значение - json-строка, см. описание ниже
                ...
            },
            for_pid => {    # множители для групп
                pid1 => mobile_multipliers_json_data,    # ключ - номер группы
                ...
            },
        },
        desktop => { # множители для десктопных устройств
            for_cid => {    # множители для кампаний
                cid1 => multiplier1,    # ключ - номер кампании
                cid2 => multiplier2,    # значение - процент к ставке (50, 200, ...)
                ...
            },
            for_pid => {    # множители для групп
                pid1 => multiplier3,    # ключ - номер группы
                ...
            },
        },
        demography => { # коэффициенты по соцдему
            for_cid => {
                cid3 => 'dem_multipliers_json_data',    # значение - json-строка, см. описание ниже
                ...,
            },
            for_pid => {...},
        },
        retargeting => {    # коэффициенты по условию ретаргетинга
            for_cid => {
                cid4 => 'ret_multipliers_json_data',    # значение - json-строка, см. описание ниже
                ...
            },
            for_pid => {...},
        },
        retargeting_filter => {
            for_cid => {
                cid4 => 'ret_multipliers_json_data',    # значение - json-строка, см. описание ниже
                ...
            },
            for_pid => {...},
        },
        weather => {    # коэффициенты по погоду
            for_cid => {
                cid4 => 'weather_multipliers_json_data',    # значение - json-строка, см. описание ниже
                ...
            },
            for_pid => {...},
        },

    Формат содержимого dem_multipliers_json_data (т.е. в распакованном виде) описан в _get_demography_multipliers_dict

    Формат содержимого ret_multipliers_json_data (т.е. в распакованном виде) - массив хешей с ключами:
        ret_cond_id => XXX,             # id условия
        multiplier_pct => multiplier21  # значение к-та

    Формат содержимого mobile_multipliers_json_data (т.е. в распакованном виде) - массив хешей с ключами:
        multiplier_pct => multiplier56  # значение к-та
        os_type => ios|android,         # тип мобильной ОС

    Формат содержимого weather_multipliers_json_data (т.е. в распакованном виде) - массив хешей с ключами:
        expression => [ ] - логическое выражене в КНФ, при истинности которого применяется коэффициент.
        multiplier_pct => multiplier77  # значение к-та в процентах

    Результат:
        $multipliers_dict   - ссылка на хеш, структура описана выше

=cut

sub _get_empty_multipliers_dict {
    my $result = {
        mobile => {
            for_cid => {},
            for_pid => {},
        },
        demography => {
            for_cid => {},
            for_pid => {},
        },
        retargeting => {
            for_cid => {},
            for_pid => {},
        },
        retargeting_filter => {
            for_cid => {},
            for_pid => {},
        },
        geo => {
            for_cid => {},
        },
        weather => {
            for_cid => {},
            for_pid => {},
        }
        # not implemented yet
        # distance => {
        #     for_cid => {},
        #     for_pid => {},
        # },
    };
    for my $type ( keys(%BS::Export::EXPRESSION_MULTIPLIER_TYPES) ) {
        my $info_ref = $BS::Export::EXPRESSION_MULTIPLIER_TYPES{$type};
        my $direct_property_name = $info_ref->{direct_property};
        $result->{$direct_property_name} = {
            for_cid => {},
            for_pid => {},
        };
    }
    return $result;
}

=head3 _get_demography_multipliers_dict

    Получить словарь соцдем коэффициентов с умолчаниями (т.е. незаданными значениями)

    Результат:
      $demography_multipliers - ссылка хеш со следующими полями, где все multiplierХХ - undef:
        male => {
            '0-17' => multiplier4,
            '18-24' => multiplier5,
            '25-34' => multiplier6,
            '35-44' => multiplier7,
            '45-'   => multiplier8,
            '45-54' => multiplier9,
            '55-'   => multiplier10,
            'unknown' => multiplier11,   # возраст определить не удалось. возникает из "Возраст: Любой"
        },
        female => {
            '0-17' => multiplier12,
            '18-24' => multiplier13,
            '25-34' => multiplier14,
            '35-44' => multiplier15,
            '45-'   => multiplier16,
            '45-54' => multiplier17,
            '55-'   => multiplier18,
            'unknown' => multiplier19,
        },
        unknown => {    # пол определить не удалось. возникает из "Пол: Все" (м + ж + неизвестные)
            '0-17' => multiplier20,
            '18-24' => multiplier21,
            '25-34' => multiplier22,
            '35-44' => multiplier23,
            '45-'   => multiplier24,
            '45-54' => multiplier25,
            '55-'   => multiplier26,
            # unknown - быть не может - он мог получиться только для пары "Все"-"любого возраста",
            # которую мы запрещаем выбирать
        },

=cut

sub _get_demography_multipliers_dict {
    my %demography_multipliers;

    for my $gender (@$DEMOGRAPHY_MULTIPLIER_GENDERS, 'unknown') {
        my $age_coeffs = {};
        for my $age (@$ALLOWED_DEMOGRAPHY_MULTIPLIER_AGES) {
            $age_coeffs->{$age} = undef;
        }
        if ($gender ne 'unknown') {
            $age_coeffs->{unknown} = undef;
        }
        $demography_multipliers{$gender} = $age_coeffs;
    }

    return \%demography_multipliers;
}

=head3 _load_multipliers(%params)

    my $multipliers_dict = _get_empty_multipliers_dict();
    $multipliers_cnt += _load_multipliers(key => 'cid',
                                          ids => $cids,
                                          multipliers_ref => $multipliers_dict,
                                          ret_cond_id_to_fetch => \%ret_cond_id_to_fetch,
                                          retargeting_multipliers => \%retargeting_multipliers,
                                          );

    Загрузить в словарь $multipliers_dict коэффициенты цен:
        для мобильных
        демографические
        по условиям ретаргетинга
        гео
        погоды

    Структура multipliers_dict описана в документации на функцию _get_empty_multipliers_dict
    Структура retargeting_multipliers: {
        $key => {
            $key_id => [
                {
                    ret_cond_id => $ret_cond_id,
                    multiplier_pct => $multiplier_value,
                    type => $type
                },
                ...
            ],
            ...
        },
        ...
    }

   Структура mobile_multipliers: {
        $key => {
            $key_id => [
                {
                    multiplier_pct => $multiplier_value,
                    os_type => $os_type
                },
                ...
            ],
            ...
        },
        ...
    }

    Структура weather_multipliers: {
        $key => {
            $key_id => [
                {
                    multiplier_pct => $multiplier_value,
                    condition => $condition #логическое условие, при истинности которого применяется коэффициент
                },
                ...
            ],
            ...
        },
        ...
    }

    Параметры именованные:
        key     - строка с типом ключа, по которому нужно выбирать коэффициенты (cid или pid)
        ids_ref - ссылка на данные с id (в зависмости от key), по которым нужно загрузить коэффициенты
                    для key=cid - массив cid
                    для key=pid - массив массивов (пар соответствий cid и pid)
        multipliers_ref - ссылка на хеш, в который нужно сохранять коэффициенты
        ret_cond_id_to_fetch    - ссылка на хеш, в ключах которого будут запомнены ret_cond_id,
                                используемые в к-тах по ретаргетингу
        retargeting_multipliers - ссылка на хеш, в котором внутри будут сохранены промежуточные данные
                                по ретаргетингу, для последующей фильтрации по недоступности условий
    Результат:
        $multipliers_cnt    - количество загруженных в словарь коэффициентов

=cut

sub _load_multipliers {
    my (%params) = @_;
    my ($key, $ids_ref) = @params{qw/key ids_ref/};
    my $multipliers = $params{multipliers_ref};
    my $ret_cond_id_to_fetch = $params{ret_cond_id_to_fetch};
    $params{retargeting_multipliers}->{$key} //= {};
    my $retargeting_multipliers = $params{retargeting_multipliers}->{$key};

    my $multipliers_cnt = 0;
    my $for_key = "for_$key";
    state $json_obj;
    $json_obj //= JSON->new->utf8(0);

    my %where = (
        is_enabled => 1,
    );
    if ($key eq 'cid') {
        $where{cid__int} = $ids_ref;
        $where{pid__is_null} = 1;
    } elsif ($key eq 'pid') {
        $where{_OR} = $ids_ref;
    } else {
        $log->die('unknown multipliers key: ' . ($key // 'undef'));
    }

    # словари hierarchical_multiplier_id => $key для загрузки значений и мапинга обратно в $key
    my %hmid_to_fetch_retargeting;
    my %hmid_to_fetch_demography;
    my %hmid_to_fetch_geo;
    my %hmid_to_fetch_ab_segment;
    my %hmid_to_fetch_inventory;
    my %hmid_to_fetch_banner_type;
    my %hmid_to_fetch_mobile;
    my %hmid_to_fetch_weather;
    my %hmid_to_fetch_expression;
    my %hmid_to_fetch_trafaret_position;

    # Таблица expression_multiplier_values хранит корректировки для наборов разных типов одновременно
    # Поэтому нужно запоминать, к какому типу относится тот или иной набор. Для этого храним здесь
    # hierarchical_multiplier_id => type для универсальных корректировок, задаваемых формулами
    my %hmid_to_fetch_expression_types;
    # То же самое с двумя типами корретировок по ретаргетингу
    my %hmid_to_fetch_retargeting_types;

    my $sth = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, cid, pid, type, multiplier_pct',
                               'FROM hierarchical_multipliers',
                               WHERE => \%where,
                              ]);

    while ( my $row = $sth->fetchrow_hashref() ) {
        if ($row->{type} eq 'mobile_multiplier') {
            ++$multipliers_cnt;
            if (defined $row->{multiplier_pct}){
                $multipliers->{mobile}->{$for_key}->{ $row->{$key} } = {
                    multiplier_pct => $row->{multiplier_pct},
                    os_type => undef,
                };
            } else {
                $hmid_to_fetch_mobile{ $row->{hierarchical_multiplier_id} } = $row->{$key};
            }
        } elsif ($row->{type} eq 'desktop_multiplier') {
            ++$multipliers_cnt;
            $multipliers->{desktop}->{$for_key}->{ $row->{$key} } = $row->{multiplier_pct};
        } elsif ($row->{type} eq 'video_multiplier') {
            $multipliers->{video}->{$for_key}->{ $row->{$key} } = $row->{multiplier_pct};
        } elsif ($row->{type} eq 'performance_tgo_multiplier') {
            $multipliers->{performance_tgo}->{$for_key}->{ $row->{$key} } = $row->{multiplier_pct};
        } elsif ($row->{type} eq 'demography_multiplier') {
            $hmid_to_fetch_demography{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif ($row->{type} eq 'retargeting_multiplier' || $row->{type} eq 'retargeting_filter') {
            $hmid_to_fetch_retargeting{ $row->{hierarchical_multiplier_id} } = $row->{$key};
            $hmid_to_fetch_retargeting_types{ $row->{hierarchical_multiplier_id} } = $row->{type};
        } elsif ($row->{type} eq 'geo_multiplier') {
            $hmid_to_fetch_geo{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif ($row->{type} eq 'ab_segment_multiplier') {
            $hmid_to_fetch_ab_segment{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif ($row->{type} eq 'inventory_multiplier') {
            $hmid_to_fetch_inventory{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif ($row->{type} eq 'banner_type_multiplier') {
            $hmid_to_fetch_banner_type{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif ($row->{type} eq 'weather_multiplier') {
            $hmid_to_fetch_weather{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif (exists $BS::Export::EXPRESSION_MULTIPLIER_TYPES{ $row->{type} }) {
            $hmid_to_fetch_expression{ $row->{hierarchical_multiplier_id} } = $row->{$key};
            $hmid_to_fetch_expression_types{ $row->{hierarchical_multiplier_id} } = $row->{type};
        # } elsif ($row->{type} eq 'distance_multiplier') {
        # not implemented yet
        } elsif ($row->{type} eq 'trafaret_position_multiplier') {
            $hmid_to_fetch_trafaret_position{ $row->{hierarchical_multiplier_id} } = $row->{$key};
        } elsif (any {$row->{type} eq $_ } qw/prisma_income_grade_multiplier smarttv_multiplier desktop_only_multiplier tablet_multiplier/) {
            # do nothing; only for ESS-export
        } else {
            $log->warn("hierarchical_multiplier_id: $row->{hierarchical_multiplier_id} unknown type: " . ($row->{type} // 'undef'));
        }
    }
    $sth->finish();
    undef $sth;

    if (keys %hmid_to_fetch_mobile) {
        my $sth_mobile = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, os_type, multiplier_pct',
                'FROM mobile_multiplier_values',
                WHERE => {
                    hierarchical_multiplier_id__int => [keys %hmid_to_fetch_mobile],
                },
            ]);
        while ( my ($hmid, $os_type, $multiplier_value) = $sth_mobile->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_mobile{$hmid};
            $multipliers->{mobile}->{$for_key}->{ $key_id } = {
                os_type => $os_type,
                multiplier_pct => int($multiplier_value),
            };
        }

        $sth_mobile->finish();
        %hmid_to_fetch_mobile = ();
    }

    if (keys %hmid_to_fetch_retargeting) {
        my $sth_ret = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, ret_cond_id, multiplier_pct',
                                       'FROM retargeting_multiplier_values',
                                       WHERE => {
                                            hierarchical_multiplier_id__int => [keys %hmid_to_fetch_retargeting],
                                       },
                                      ]);
        while ( my ($hmid, $ret_cond_id, $multiplier_value) = $sth_ret->fetchrow_array() ) {
            $ret_cond_id_to_fetch->{$ret_cond_id} = undef;
            push @{ $retargeting_multipliers->{ $hmid_to_fetch_retargeting{$hmid} } } , {
                ret_cond_id => int($ret_cond_id),
                multiplier_pct => int($multiplier_value),
                type => $hmid_to_fetch_retargeting_types{ $hmid }
            };
        }

        $sth_ret->finish();
        %hmid_to_fetch_retargeting = ();
    }

    if (keys %hmid_to_fetch_geo) {
        my $sth_geo = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, region_id, multiplier_pct',
                'FROM geo_multiplier_values',
                WHERE => {
                    hierarchical_multiplier_id__int => [keys %hmid_to_fetch_geo],
            },
        ]);
        my %data_by_key;
        while ( my ($hmid, $region_id, $multiplier_value) = $sth_geo->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_geo{$hmid};
            push @{ $data_by_key{$key_id} }, {
                    region_id => int($region_id),
                    multiplier_pct => int($multiplier_value),
                };
        }

        $sth_geo->finish();

        for my $key_id (keys(%data_by_key)) {
            $multipliers->{geo}->{$for_key}->{ $key_id } = $data_by_key{$key_id};
        }

        ### New style geo multiplier
        for my $key_id (keys(%data_by_key) ) {
            my @multipliers = map {BS::Export::get_bs_expression_for_geo_modifier($_)} @{$data_by_key{$key_id}};
            if (@multipliers) {
                $multipliers->{geo_new}->{$for_key}->{$key_id} = $json_obj->encode(\@multipliers);
            }
        }

        %hmid_to_fetch_geo = ();
    }

    if (keys %hmid_to_fetch_ab_segment) {
        my $sth_ab_segment = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, ab_segment_ret_cond_id, multiplier_pct',
                'FROM ab_segment_multiplier_values',
                WHERE => {
                    hierarchical_multiplier_id__int => [keys %hmid_to_fetch_ab_segment],
                },
            ]);
        my %data_by_key;
        while ( my ($hmid, $ab_segment_ret_cond_id, $multiplier_value) = $sth_ab_segment->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_ab_segment{$hmid};
            push @{$data_by_key{$key_id}}, {
                    ab_segment_ret_cond_id => int($ab_segment_ret_cond_id),
                    multiplier_pct => int($multiplier_value),
                };
        }

        $sth_ab_segment->finish();

        for my $key_id (keys(%data_by_key)) {
            $multipliers->{ab_segment}->{$for_key}->{$key_id} = $data_by_key{$key_id};
        }

        %hmid_to_fetch_ab_segment = ();
    }

    if (keys %hmid_to_fetch_weather) {
        my $sth_weather = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, multiplier_pct, condition_json',
            'FROM weather_multiplier_values',
             WHERE => { hierarchical_multiplier_id__int => [keys %hmid_to_fetch_weather] },
        ]);

        my %data_by_key;
        while ( my ($hmid, $multiplier_value, $weather_condition_json) = $sth_weather->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_weather{$hmid};
            push @{ $data_by_key{$key_id} }, {
                multiplier_pct => int($multiplier_value),
                condition_json => $weather_condition_json
            };
        }
        $sth_weather->finish();

        for my $key_id (keys(%data_by_key) ) {
            my $multipliers_set = [];
            foreach my $modifier ( @{$data_by_key{$key_id}} ) {
                my $condition_json =$json_obj->decode($modifier->{condition_json});
                my $bs_expression = get_bs_expression_for_modifier(
                        condition => $condition_json,
                        multiplier_pct => $modifier->{multiplier_pct});

                push @$multipliers_set, $bs_expression;
            }
            if(@$multipliers_set) {
                $multipliers->{weather}->{$for_key}->{$key_id} = $json_obj->encode($multipliers_set);
            }
        }


        %hmid_to_fetch_weather = ();
    }

    if (keys %hmid_to_fetch_expression) {
        my $sth_expression = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, multiplier_pct, condition_json',
            'FROM expression_multiplier_values',
            WHERE => { hierarchical_multiplier_id__int => [keys %hmid_to_fetch_expression] },
        ]);

        my %data_by_key;
        while ( my ($hmid, $multiplier_value, $expression_condition_json) = $sth_expression->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_expression{$hmid};
            push @{ $data_by_key{$key_id} }, {
                type => $hmid_to_fetch_expression_types{ $hmid },
                multiplier_pct => int($multiplier_value),
                condition_json => $expression_condition_json
            };
        }
        $sth_expression->finish();

        for my $key_id (keys(%data_by_key) ) {
            my $multipliers_sets_by_type = {};
            foreach my $modifier ( @{$data_by_key{$key_id}} ) {
                my $condition_json =$json_obj->decode($modifier->{condition_json});
                my $bs_expression = get_bs_expression_for_modifier(
                    condition => $condition_json,
                    multiplier_pct => $modifier->{multiplier_pct});
                if (!exists $multipliers_sets_by_type->{ $modifier->{type} }) {
                    $multipliers_sets_by_type->{ $modifier->{type} } = [];
                }
                push @$multipliers_sets_by_type{ $modifier->{type} }, $bs_expression;
            }
            for my $type (keys(%$multipliers_sets_by_type)) {
                my $multipliers_set = $multipliers_sets_by_type->{ $type };
                if (@$multipliers_set) {
                    my $direct_property_name = $BS::Export::EXPRESSION_MULTIPLIER_TYPES{$type}->{direct_property};
                    $multipliers->{$direct_property_name}->{$for_key}->{$key_id} = $json_obj->encode($multipliers_set);
                }
            }
        }

        %hmid_to_fetch_expression = ();
        %hmid_to_fetch_expression_types = ();
    }

    if (keys %hmid_to_fetch_demography) {
        # может приезжать много строк на одно значение $key, поэтому режем на чанки
        for my $hmid_chunk (chunks([ keys(%hmid_to_fetch_demography) ], 1_000)) {
            my %data_by_key;
            my $sth_dem = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, demography_multiplier_value_id,',
                                                  'gender, age, multiplier_pct',
                                           'FROM demography_multiplier_values',
                                           WHERE => {
                                                hierarchical_multiplier_id__int => $hmid_chunk,
                                           },
                                          ]);
            while ( my ($hmid, $dmvid, $gender, $age, $multiplier_value) = $sth_dem->fetchrow_array() ) {
                my $key_id = $hmid_to_fetch_demography{$hmid};

                if ( $age && defined $DEPRECATED_DEMOGRAPHY_MULTIPLIER_AGE{$age}){
                    for my $new_age (@{$DEPRECATED_DEMOGRAPHY_MULTIPLIER_AGE{$age}}) {
                        push @{ $data_by_key{$key_id} }, {
                            gender => $gender,
                            age => $new_age,
                            multiplier_pct => $multiplier_value,
                            # используется только для логирования при ошибках
                            demography_multiplier_value_id => $dmvid,
                        };
                    }
                } else {
                    push @{ $data_by_key{$key_id} }, {
                        gender => $gender,
                        age => $age,
                        multiplier_pct => $multiplier_value,
                        # используется только для логирования при ошибках
                        demography_multiplier_value_id => $dmvid,
                    };
                }
            }
            $sth_dem->finish();

            ### New style demography multiplier
            for my $key_id (keys(%data_by_key) ) {
                my @multipliers = map {BS::Export::get_bs_expression_for_socdem_modifier($_)} @{$data_by_key{$key_id}};
                if (@multipliers) {
                    $multipliers->{demography_new}->{$for_key}->{$key_id} = $json_obj->encode(\@multipliers);
                }
            }

            ### Old-style
            for my $key_id (keys(%data_by_key)) {
                # пустой словарь с умолчаниями, который будем заполнять
                my $multipliers_set = _get_demography_multipliers_dict();
                my $at_least_one_coef;

                for my $row (@{ $data_by_key{$key_id} }) {
                    my (@genders, @ages);

                    if (defined $row->{gender}) {
                        @genders = $row->{gender};
                    } else {
                        # gender = NULL - это "любой" пол
                        @genders = (@$DEMOGRAPHY_MULTIPLIER_GENDERS, 'unknown');
                    }

                    if (defined $row->{age}) {
                        @ages = $row->{age};
                    } else {
                        # age = NULL - это "любой" возраст
                        @ages = (@$ALLOWED_DEMOGRAPHY_MULTIPLIER_AGES, 'unknown');
                    }

                    for my $gender (@genders) {
                        for my $age (@ages) {
                            if (exists $multipliers_set->{$gender}
                                && exists $multipliers_set->{$gender}->{$age}
                                && !defined $multipliers_set->{$gender}->{$age}
                            ) {
                                $multipliers_set->{$gender}->{$age} = int($row->{multiplier_pct});
                                $at_least_one_coef //= 1;
                            } else {
                                # что-то пошло не так - дублирующееся или неизвестное значение
                                $error_logger->({
                                        message           => 'duplicate or invalid multiplier',
                                        multiplier        => $multipliers_set,
                                        snapshot_row      => $row,
                                        multiplier_id     => $key_id,
                                        type              => "demography_multipliers",
                                        multipliers_scope => $key,
                                        stage             => "request",
                                    });
                            }
                        }
                    }
                }
                # запаковываем готовый словарь с коэффициентами
                if ($at_least_one_coef) {
                    $multipliers->{demography}->{$for_key}->{$key_id} = $json_obj->encode($multipliers_set);
                }
            }
        }

        %hmid_to_fetch_demography = ();
    }
    if (keys %hmid_to_fetch_inventory) {
        my $sth_inventory = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, inventory_type, multiplier_pct',
                'FROM inventory_multiplier_values',
                WHERE => {
                    hierarchical_multiplier_id__int => [keys %hmid_to_fetch_inventory],
            },
        ]);
        my %data_by_key;
        while ( my ($hmid, $inventory_type, $multiplier_value) = $sth_inventory->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_inventory{$hmid};
            push @{ $data_by_key{$key_id} }, {
                    inventory_type => $inventory_type,
                    multiplier_pct => int($multiplier_value),
                };
        }

        $sth_inventory->finish();

        for my $key_id (keys(%data_by_key)) {
            $multipliers->{inventory}->{$for_key}->{ $key_id } = $data_by_key{$key_id};
        }

        %hmid_to_fetch_inventory = ();
    }

    if (keys %hmid_to_fetch_banner_type) {
        my $sth_banner_type = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, banner_type, multiplier_pct',
                'FROM banner_type_multiplier_values',
                WHERE => {
                    hierarchical_multiplier_id__int => [keys %hmid_to_fetch_banner_type],
            },
        ]);
        my %data_by_key;
        while ( my ($hmid, $banner_type, $multiplier_value) = $sth_banner_type->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_banner_type{$hmid};
            push @{ $data_by_key{$key_id} }, {
                    banner_type => $banner_type,
                    multiplier_pct => int($multiplier_value),
                };
        }

        $sth_banner_type->finish();

        for my $key_id (keys(%data_by_key)) {
            $multipliers->{banner_type}->{$for_key}->{ $key_id } = $data_by_key{$key_id};
        }

        %hmid_to_fetch_banner_type = ();
    }

    if (keys %hmid_to_fetch_trafaret_position) {
        my $sth_trafaret_position = exec_sql($dbh2, ['SELECT hierarchical_multiplier_id, trafaret_position, multiplier_pct',
            'FROM trafaret_position_multiplier_values',
            WHERE => {
                hierarchical_multiplier_id__int => [keys %hmid_to_fetch_trafaret_position],
            },
        ]);
        my %data_by_key;
        while ( my ($hmid, $trafaret_position, $multiplier_value) = $sth_trafaret_position->fetchrow_array() ) {
            my $key_id = $hmid_to_fetch_trafaret_position{$hmid};
            push @{ $data_by_key{$key_id} }, {
                trafaret_position => $trafaret_position,
                multiplier_pct => int($multiplier_value),
            };
        }

        $sth_trafaret_position->finish();

        for my $key_id (keys(%data_by_key)) {
            $multipliers->{trafaret_position}->{$for_key}->{ $key_id } = $data_by_key{$key_id};
        }

        %hmid_to_fetch_trafaret_position = ();
    }

    return $multipliers_cnt;
}

=head2 _get_source_id

    Определяет source_id для отправки в logbroker по номеру кампании

    $source_id = _get_source_id($cid);

=cut

sub _get_source_id {
    my ($shard, $cid) = @_;

    return (($shard - 1) % LOGBROKER_SOURCE_ID_DIVIDER) + 1;
}

=head2 logbroker_debug_info(shard)

    Создание словаря с дебаг-информацией для первой записи каждой строки, отправляемой в LB

=cut

sub logbroker_debug_info($$) {
    my ($shard, $params) = @_;
    return {
        host => $EnvTools::hostname,
        reqid => Yandex::Trace::current_span_id(),
        shard => $shard, # NB! значение используется в БК - BSSERVER-14180: Протащить до EventLog'а номер шарда в директе
        par_norm_nick => $PAR_NORM_NICK,
        $params->{with_send_time} ? (send_time => strftime("%Y-%m-%d %H:%M:%S", localtime)) : (),
    };
}

=head2 send_to_logbroker

    Отправляет в БК через logbroker теже данные, что и через UpdateData2

    Принимает позиционные параметры:
        - номер шарда
        - ссылка на хеш с запросом, отправляемым в БК через UpdateData2
        - UUID запроса
        - опция "не использовать логброкер" (не для продакшена, сбрасывать буфер "вникуда" вместо push-client'а)
        - флажок "рекэспорт" для добавления в метаданные

=cut

sub send_to_logbroker {
    my ($shard, $query, $request_uuid, %options) = @_;

    my %common_params = (
        uuid => $request_uuid,
        iter_id => get_new_id('bsexport_iter_id'),
        full_export_flag => ($options{full_export_flag} ? 1 : 0),
        debug_info => logbroker_debug_info($shard, { with_send_time => 1 }),
        actual_in_direct_db_at => $options{actual_in_direct_db_at},
    );

    my $json = JSON->new->canonical->utf8->allow_blessed->convert_blessed;
    local *SOAP::Data::TO_JSON = sub {return $_[0]->value};

    my $orders_hash = $query->{ORDER};
    my @orders = values %$orders_hash;
    my %source_id2orders_chunk = partition_by { _get_source_id($shard, $_->{EID}) } @orders;
    for my $source_id (keys %source_id2orders_chunk) {
        my $orders_chunk = $source_id2orders_chunk{$source_id};
        next unless $orders_chunk && @$orders_chunk;

        my $logbroker_buffer = BS::ExportWorker::LogBrokerBufferNewProto->new(source_id => $source_id, prefix => $request_uuid, profile_tag => 'data', dont_real_flush => $options{no_logbroker});

        for my $order (@$orders_chunk) {
            my $EngineID = $options{cid_to_engine_id}->{ $order->{EID} };
            my $order_for_rec = hash_kgrep { $_ ne 'CONTEXT' } $order;
            my $order_for_rec_json = $json->encode({
                level => 'ORDER',
                OrderID => $order->{ID},
                EngineID => $EngineID,
                cid => $order->{EID},
                data => $order_for_rec,
                %common_params,
            });
            $logbroker_buffer->append( \$order_for_rec_json );
            undef $order_for_rec;
            undef $order_for_rec_json;

            my $contexts = $order->{CONTEXT};
            for my $context (values %$contexts) {
                my $context_json = $json->encode({
                    level => 'CONTEXT',
                    OrderID => $order->{ID},
                    EngineID => $EngineID,
                    cid => $order->{EID},
                    pid => $context->{EID},
                    data => $context,
                    %common_params,
                });
                $logbroker_buffer->append( \$context_json );
            }
        }

        $logbroker_buffer->close();
    }
}

=head2 fetch_bs_order_ids

    Получаем для каждого переданного cid-a новой кампании соответсвующий OrderID для БК, сгенерированный в Директе.
    OrderID получается прибавлением 100_000_000 к cid'у
    NB! Предполагается, что уже сделаны проверки на то что все cid-ы принадлежат новым кампаниям которые ещё не были в БК и не имеют записанного OrderID в таблице ppc.campaigns

    Принимает на вход ссылку на массив cid-ов:
     [1, 2, ...]
    Возвращает ссылку на хеш вида:
     { cid => OrderID, ... }

=cut

sub fetch_bs_order_ids {
    my $cids = shift;
    return { map { $_ => "".($_ + ORDER_ID_OFFSET) } @$cids };
}

=head2 get_campaigns_asset_hashes

    Принимает на вход ссылку на массив campaign_id
    Возвращает ссылку на хеш campaign_id => banner_id => id ассетов баннера
    {
        238192593 => {
            11027950388 => {
                TitleAssetHash => '395766436205812',
                TextBodyAssetHash => '395767351324926',
                ImageAssetHash => '395767741545527',
                VideoAssetHash => '395766269052275'
            },
            11027952443 => {
                TitleAssetHash => '395767825866396',
                TextBodyAssetHash => '395767607000723',
            },
            ...
        },
        ...
    }

=cut

sub get_campaigns_asset_hashes {
    my ($uac_campaign_ids, %opts) = @_;

    return undef if !@$uac_campaign_ids || !_is_send_asset_hash_enabled();

    my $campaigns_chunk = _get_campaigns_chunk_for_getting_asset_hashes(%opts);

    my %campid2bannerid2assetids_bucket;
    foreach my $uac_campaign_ids_chunk (chunks($uac_campaign_ids, $campaigns_chunk)) {
        my $campid2bannerid2assetids_chunk = JavaIntapi::GetCampaignsAssetHashes->new(campaign_ids => $uac_campaign_ids_chunk)->call();

        foreach my $key ( keys %$campid2bannerid2assetids_chunk ) {
            if( exists $campid2bannerid2assetids_bucket{$key} ) {
                $log->warn("Key $key is in both hashes!");
                next;
            } else {
                $campid2bannerid2assetids_bucket{$key} = $campid2bannerid2assetids_chunk->{$key};
            }
        }
    }
    my $campid2bannerid2assetids = \%campid2bannerid2assetids_bucket;

    # Логируем id кампаний для которых не получили хеши ассетов
    my @campaign_ids_without_assets = ();
    foreach my $campaign_id (@$uac_campaign_ids) {
        next if $campid2bannerid2assetids && $campid2bannerid2assetids->{$campaign_id};
        push @campaign_ids_without_assets, $campaign_id;
    }

    # Данный кейс может происходить, пока не все uac кампании в ydb
    if (@campaign_ids_without_assets) {
        $log->out("Cannot find assets for campaign ids: @campaign_ids_without_assets");
    }
    return $campid2bannerid2assetids;
}

=head2 _is_send_asset_hash_enabled

    Определить, доступна ли отправка хешей ассетов у баннеров.
    При включении ходим в kotlin через intapi, для получения хешей ассетов

    Результат:
        0/1 -  добавлять ли отправку хешей ассетов

=cut

sub _is_send_asset_hash_enabled {
    state $is_send_asset_hash_enabled_by_property = Property->new('SEND_ASSET_HASH_ENABLED');
    return $is_send_asset_hash_enabled_by_property->get(60) ? 1 : 0;
}

=head2 _is_send_asset_hash_included_for_all_types_enabled

    Определить, доступна ли отправка хешей ассетов для всех типов UC/UAC или только для ТГО

    Результат:
        0/1 -  добавлять отправку хешей ассетов для всех типов кампаний

=cut

sub _is_send_asset_hash_included_for_all_types_enabled {
    state $is_send_asset_hash_enabled_by_property = Property->new('SEND_ASSET_HASH_INCLUDED_FOR_ALL_TYPES_ENABLED');
    return $is_send_asset_hash_enabled_by_property->get(60) ? 1 : 0;
}

=head2 _get_campaigns_chunk_for_getting_asset_hashes

    Определить размер чанка id кампаний для отправки в intapi ручку get-campaigns-asset-hashes

=cut

sub _get_campaigns_chunk_for_getting_asset_hashes {
    my (%opts) = @_;

    if ($opts{is_heavy}) {
        state $heavy_campaigns_chunk_for_getting_asset_hashes = Property->new('UAC_CAMPAIGNS_CHUNK_FOR_GETTING_ASSET_HASHES');
        return eval { $heavy_campaigns_chunk_for_getting_asset_hashes->get(60); } || 1;
    } else {
        state $campaigns_chunk_for_getting_asset_hashes = Property->new('CAMPAIGNS_CHUNK_FOR_GETTING_ASSET_HASHES');
        return eval { $campaigns_chunk_for_getting_asset_hashes->get(60); } || 70;
    }
}

=head2 _get_camp_autobudget_restarts

    Получает для каждого переданного cid'а информацию о последнем рестарте автобюджета из таблицы camp_autobudget_restart.
    Используется только в полном ре-экспорте (поток full_lb_export).

    Возвращает ссылку на хэш следующего формата:
    {
        <campaign_id> => {
            cid => <campaign_id>
            restart_time => '2020-07-14 10:50:00',
            soft_restart_time => '2020-07-14 10:50:00',
            restart_reason => 'BS_RESTART'
        },
        ...
    }

=cut

sub _get_camp_autobudget_restarts {
    my ($cids) = @_;
    my $restarts_sql = 'SELECT cid, restart_time, soft_restart_time, restart_reason FROM camp_autobudget_restart';
    my $restarts = get_hashes_hash_sql($dbh, [$restarts_sql, WHERE => { cid__int => $cids }]);
    return $restarts;
}

=head2 _dont_send_extra_banner_image_formats

    Не отправлять лишние размеры картинок

=cut

sub _dont_send_extra_banner_image_formats {
    state $dont_send_extra_banner_image_formats_by_property = Property->new('DO_NOT_SEND_EXTRA_BANNER_IMAGE_FORMATS_TO_BS');
    return $dont_send_extra_banner_image_formats_by_property->get(60) ? 1 : 0;
}

1;
