package BS::ExportWorker::LogBrokerBuffer;

use Direct::Modern;

use Carp qw(croak);
use Yandex::Shell qw(yash_quote);

use BS::Export ();


=head2 $LOG

    Объект Yandex::Log для записи сообщений об ошибках

=cut

our $LOG;

=head2 MAX_BUFFER_LENGTH

    Максимальный размер данных в байтах, отправляемый за один раз в logbroker

=cut

use constant MAX_BUFFER_LENGTH => 12 * 1024 * 1024;

=head2 new

    Считываем общие аргументы для всех наследников LogBrokerBuffer 

=cut

sub new {
    my ($class, %O) = @_;

    my $source_id = delete $O{source_id};
    croak 'no source_id given' unless defined $source_id;

    my $prefix = delete $O{prefix};
    croak 'no prefix given' unless defined $prefix;

    my $profile_tag = delete($O{profile_tag}) // 'unknown';
    my $dont_real_flush = delete $O{dont_real_flush} ? 1 : 0;

    croak 'Unsupported options: '.join(', ', keys %O) if %O;

    bless {
        buf => '',
        buf_len => 0,
        source_id => $source_id,
        prefix => $prefix,
        profile_tag => $profile_tag,
        _dont_real_flush => $dont_real_flush,
    }, $class;
}

=head2 append

    Добавляет строку байт к буферу на отправку в БК через logbroker.
    Если в буфере не осталось места, то предварительно отправляет его в logbroker.

    На вход принимает ссылку на скаляр со строкой байт для отправки.

    $lobroker_buffer->append(\$data);

=cut

sub append {
    my ($self, $data_ref) = @_;

    use bytes;

    if (!$data_ref || ref($data_ref) ne 'SCALAR') {
        croak '$data_ref argument to append should be a reference to scalar';
    }

    my $data_len = length($$data_ref);
    return if !$data_len;

    if ($data_len + length("[]\n") > MAX_BUFFER_LENGTH) {
        $self->_log_fail_and_croak("Too long data given to append: length is $data_len");
    }

    if ($self->{buf_len} > 0 && $self->{buf_len} + $data_len + length("[,]\n") >= MAX_BUFFER_LENGTH) {
        $self->flush();
    }

    $self->{buf} .= $$data_ref;
    $self->{buf} .= ',';
    $self->{buf_len} += $data_len + length(',');
}

sub close {
    die "should be overrided";
}

=head3 _log_fail_and_croak

    Пишет сообщения (через метод out) в $LOG если он определен и делает croak

=cut

sub _log_fail_and_croak {
    my ($self, $msg) = @_;
    if ($LOG) {
        $LOG->out($msg);
    }
    croak $msg;
}

=head3 _log

    Пишет сообщения (через метод out) в $LOG если он определен

=cut

sub _log {
    my $self = shift;
    if ($LOG) {
        $LOG->out(@_);
    }
}

sub _get_pushclient_logname {
    return $Settings::LOG_ROOT . '/bsexport-pushclient-log' . '.' . Yandex::TimeCommon::today();
}

sub _get_pushclient_common_params {
    my $self = shift;

    return join(' ', (map { yash_quote($_) } (
            '/usr/bin/push-client',
            '-f',
            "--ident=$BS::Export::LOGBROKER_IDENT",
            "--log_type=$BS::Export::LOGBROKER_LOG_TYPE",
            "--network:master-addr=$BS::Export::LOGBROKER_HOST",
            '--network:transport=ipv6',
            "--network:timeout=$BS::Export::LOGBROKER_NETWORK_TIMEOUT",
            '--logger:mode=file',
            '--logger:file='._get_pushclient_logname(),
            "--logger:level=$BS::Export::LOGBROKER_LOG_LEVEL",
            '--logger:timeformat=(%b %d %H:%M:%S.\S) \H '."[$self->{prefix}]".' [\P.\T]',
            '--logger:telemetry_interval=0',
            '--stdin',
        )),
    );
}

sub _prepare_data {
    my $self = shift;

    # удаляем последнюю запятую, дописываем скобки, чтобы получился корректный json-array
    return '[' . substr($self->{buf}, 0, -1) . "]\n";
}

sub _clear_buffer {
    my $self = shift;
    $self->{buf} = '';
    $self->{buf_len} = 0;
}

=head2 _get_alarm_sub

    Получаем сабу для задания в $SIG{ALRM} для реализации таймаута (на запись в push-client или закрытие пайпа)

    Аргументы:

        $msg - префикс сообщения-ошибки о таймауте
        $killed_ref - ссылка на переменную, в которую положим индиактор того, упали ли мы по таймауту

=cut

sub _get_alarm_sub {
    my ($self, $msg, $killed_ref) = @_;

    return sub {
        if ($self->{push_client}->{pid} > 0) {
            my $cnt = kill KILL => $self->{push_client}->{pid};
            $self->{is_killed} = 1;
            $msg .= " (killed $cnt processes, pid=$self->{push_client}->{pid})";
            $$killed_ref = $cnt;
        } else {
            $$killed_ref = 0;
        }
        # не используем здесь croak, потому что исполнение передается не в ближайший наружный eval, а в "дальний" (тот что в bsClientData)
        # при этом если правда прибили пуш-клиент, то упадем на close(), иначе - на проверке $killed
        $self->_log($msg);
    };
}

sub _check_exit_codes {
    my ($self, $child_exit_status, $killed) = @_;

    if (($child_exit_status != -1) && (!defined $child_exit_status || $child_exit_status != 0)) {
        $self->_log_fail_and_croak("ERROR sending data to push-client for source_id $self->{source_id}: exit status is $child_exit_status");
    } elsif (defined $killed) {
        $self->_log_fail_and_croak("ERROR sending data to push-client for source_id $self->{source_id}: process timed out");
    }
}

1;
