package BS::ExportWorker::LogBrokerBufferPipeEach;

=encoding utf8

=head1 NAME

    BS::ExportWorker::LogBrokerBufferPipeEach

=head1 DESCRIPTION

    !!! Запись по старому протоколу закрыта, используйте LogBrokerBufferNewProto !!!

    Объект для накапливания данных на отправку в БК через logbroker.
    Копит строку с байтами и по достижению в буфере MAX_BUFFER_LENGTH байт
    отправляет весь буфер в logbroker через push-client.

=cut

use Direct::Modern;

use Guard qw(scope_guard);

use Yandex::Trace;

use BS::Export ();
use Settings ();

use parent 'BS::ExportWorker::LogBrokerBuffer';

=head2 flush

    Отправляет накопленный буфер в БК через logbroker
    Если данных в буфере нет, ничего не делает
    Если истинна опция _dont_real_flush - данные в logboker не отправляются

    $lobroker_buffer->flush();

=cut

sub flush {
    my ($self) = @_;

    return unless $self->{buf_len} > 0;

    my $data = $self->_prepare_data();

    if ($self->{_dont_real_flush}) {
        $self->_clear_buffer();
        return;
    }

    my $profile_tags = join ',', grep {defined $_ && $_ ne ''} 'bs', $self->{profile_tag};
    my $profile = Yandex::Trace::new_profile('logbroker:send', tags => $profile_tags, obj_num => length($data));

    {
        scope_guard { alarm 0; };

        local $SIG{ALRM} = $self->_get_alarm_sub("timeout sending data to logbroker ($BS::Export::LOGBROKER_TIMEOUT seconds)", \(my $killed));

        $self->_check_iteration_time();

        alarm $BS::Export::LOGBROKER_TIMEOUT;

        $self->_get_push_client();

        $self->_log("writing to push-client (pid=$self->{push_client}->{pid}, buf_len: $self->{buf_len})");

        select $self->{push_client}->{pipe_fh}; $| = 1; # make unbuffered
        print { $self->{push_client}->{pipe_fh} } $data or $self->_log_fail_and_croak("ERROR writing to push-client for source_id $self->{source_id}: $!");

        $self->_check_exit_codes(-1, $killed);

        $self->_clear_buffer();

        $self->_after_print();
    }

}

=head2 close

    Закрываем открытый пайп push-client

    Параметры:

        skip_flush - не отправлять накопленное перед закрытием

=cut

sub close {
    my ($self, %O) = @_;

    # перед закрытием отправляем все, что осталось перед закрытием
    if (!($O{skip_flush} // 0)) {
        $self->flush();
    }

    if (!defined $self->{push_client} || $self->{is_killed}) {
        return;
    }

    my $profile = Yandex::Trace::new_profile('logbroker:close');
    {
        scope_guard { alarm 0; };

        local $SIG{ALRM} = $self->_get_alarm_sub("timeout closing pushclient", \(my $killed));

        $self->_check_iteration_time();

        $self->_log("close push-client");

        alarm $BS::Export::LOGBROKER_TIMEOUT;
        CORE::close $self->{push_client}->{pipe_fh} or $self->_log_fail_and_croak("ERROR closing push-client for source_id $self->{source_id}: $!");

        $self->_check_exit_codes($? >> 8, $killed);
    }

    $self->{push_client} = undef;
}

sub _get_pushclient_params {
    my $self = shift;
    return $self->_get_pushclient_common_params()." --partition=$self->{source_id}";
}

sub _open_pushclient {
    my $self = shift;

    my $cmd = $self->_get_pushclient_params();

    $self->_log("open push-client: $cmd");

    my $push_client_pid = open(my $pushclient, '|-', $cmd) or $self->_log_fail_and_croak("ERROR opening push-client for source_id $self->{source_id}: $!");

    $self->{push_client} = {
        pipe_fh => $pushclient,
        pid => $push_client_pid,
    };
}

sub _get_push_client {
    my $self = shift;
    $self->_open_pushclient();
}

sub _after_print {
    my ($self) = @_;
    $self->close();
}

sub _check_iteration_time {
    my $self = shift;
    # empty
}

1;
