##################################################
#
#  Direct.Yandex.ru
#
#  BS::History
#  парсинг и редактирование строки истории статистики фраз/баннеров
#
##################################################

=head1 NAME

BS::History

=head1 DESCRIPTION

парсинг и редактирование строки истории статистики фраз/баннеров

=cut

package BS::History;

use warnings;
use strict;

use Data::Dumper;

use Yandex::DBTools;

use Settings;
use ShardingTools;
use PrimitivesIds;

=head2 group_bids_history

    парсит исходную phraseIdHistory и отдает историю по группе
    в формате G<pid>,PhraseID,...

=cut

sub group_bids_history ($;$) {
    my $phraseIdHistory = shift || '';
    my $GroupID_to_add = shift;
    my %history = parse_bids_history($phraseIdHistory);

    my @ids = ();
    if ($history{GroupID} || $GroupID_to_add || !$phraseIdHistory) {
        foreach my $GroupID (($history{GroupID} || ()), ($GroupID_to_add || ())) {
            foreach my $ph (@{$history{phrases}}) {
                push @ids, 'G'.$GroupID, $ph;
            }
        }
        return join ',', @ids;
    } else {
        return undef;
    }
}

=head2 parse_bids_history

    парсит исходную phraseIdHistory и отдает полную историю по каждому баннеру в виде хеша
    {
        OrderID => 123,
        GroupID => 321,
        phrases => [PhraseID1,PhraseID2,...],
        banners => {
            bid1 => [BannerID1,BannerID2,...],
            bid2 => [BannerID3,BannerID4,...],
        }
    }
    !! кешировать через Memoize

=cut
sub parse_bids_history ($) {
    my $phraseIdHistory = shift || '';

    my %history = make_empty_bids_history();
    foreach (split ';', $phraseIdHistory) {
        if (/^O(.+)$/) {
            $history{OrderID} = $1;
        } elsif (/^G(.+)$/) {
            $history{GroupID} = $1;
        } elsif (/^P(.+)$/) {
            $history{phrases} = [split /,/, $1];
        } else {
            my @pair = split(':', $_);
            if (defined ($pair[0]) && $pair[1]) {
                $history{banners}->{$pair[0]} = [split /,/, $pair[1]];
            }
        }
    }           
    return %history;
}

=head2 convert_to_banner_history

    переводит строку phraseIdHistory формата для групп (группа-фраза) к формату баннеров(баннер-фараза)

=cut
sub convert_to_banner_history {

    my $phraseIdHistory = shift;

    my %history = parse_bids_history($phraseIdHistory);
    return undef unless $history{banners} && $history{phrases};

    my @pairs;
    foreach my $BannerID (map {@$_} values %{$history{banners}}) {
       push @pairs, "B${BannerID},${_}" for @{$history{phrases}}; 
    }
    return join ',', @pairs;
}

=head2 serialize_bids_history

    сериализует хеш аналогичный возвращаемому в функции parse_bids_history и возвращает строку совместимую с phraseIdHistory

=cut
sub serialize_bids_history ($) {
    my $history = shift;
    return unless $history && ref($history) eq 'HASH';

    my @parts = ();
    if ($history->{OrderID}) {
        push @parts, 'O'.$history->{OrderID};
    }
    if ($history->{GroupID}) {
        push @parts, 'G'.$history->{GroupID};
    }
    if ($history->{phrases} and @{$history->{phrases}}) {
        push @parts, 'P' . join ',', splice(@{$history->{phrases}}, 0, 3);
    }

    push @parts, join ';', map { $_ .':'. join ',', @{ref $history->{banners}->{$_} ? $history->{banners}->{$_} : [$history->{banners}->{$_}]}} sort keys %{$history->{banners}};

    return join ';', @parts;
}

=head2 prepend_history

    добавляет в начало строки истории еще одну или несколько записей

=cut
sub prepend_history ($;@) {
    my $history_str = shift;

    my @prefix_list = ();
    foreach my $prefix (@_) {
        next unless $prefix && ref($prefix) eq 'HASH' && $prefix->{PhraseID} && ($prefix->{BannerID} || $prefix->{GroupID});
        push @prefix_list, ($prefix->{BannerID} ? 'B'.$prefix->{BannerID} : 'G'.$prefix->{GroupID}).','.$prefix->{PhraseID};
    }

    return join(',', @prefix_list, ($history_str ? $history_str : ()));
}


=head2 make_empty_bids_history

Возвращает пустую историю, чтобы более ясно было, чем вызов
C<parse_bids_history('')> или чем создание аналогичного хэша явно.

    my %history = make_empty_bids_history();

=cut
sub make_empty_bids_history() {
    return (
        OrderID => undef,
        GroupID => undef,
        phrases => [],
        banners => {}
    );
}


=head2 merge_mediaplan_history

    обновляет историю текущими BannerID/PhraseID при копировании фраз в медиаплан

=cut
sub merge_mediaplan_history ($$$$$) {
    my ($history_str, $OrderID, $PhraseID, $GroupID, $group_banners) = @_;

    if (!$OrderID and $PhraseID) {
        warn "merge_mediaplan_history called for bid with PhraseID but without OrderID: " . Dumper({history_str => $history_str, OrderID => $OrderID, PhraseID => $PhraseID, GroupID => $GroupID, group_banners => $group_banners});
        return serialize_bids_history({make_empty_bids_history()});
    }

    my %history = parse_bids_history($history_str);

    my $history_kind;
    if (!$history{OrderID}) {
        $history_kind = 'empty';
    } elsif ($history{OrderID} != ($OrderID // 0)) {
        $history_kind = 'other';
    } else {
        $history_kind = 'ours';
    }

    if ($history_kind eq 'other' and !$PhraseID) {
        return $history_str;
    }
    if (!$OrderID and $history_kind eq 'empty') {
        return serialize_bids_history({make_empty_bids_history()});
    }

    if ($OrderID && $history_kind eq 'other') {
        %history = make_empty_bids_history();
    }

    $history{OrderID} = $OrderID if !$history{OrderID} || $PhraseID;
    $history{GroupID} = $GroupID if !$history{GroupID} || $PhraseID;

    if ($PhraseID) {
        if (grep { !$_ } values %$group_banners) {
            # Приписывем $PhraseID в начало списка, удаляем дубликаты с сохранением порядка.
            my %already_seen;
            @{$history{phrases}} = map { $already_seen{$_}++ ? () : $_ } ($PhraseID, @{$history{phrases}});
        } else {
            $history{phrases} = [$PhraseID];
        }
    }

    foreach my $gr_bid (keys %$group_banners) {
        if ($group_banners->{$gr_bid}) {
            $history{banners}->{$gr_bid} = [$group_banners->{$gr_bid}];
        }
        elsif (@{ $history{bannners}->{old} || [] }) {
            $history{banners}->{$gr_bid} = [@{$history{bannners}->{old}}];
        }
    }

    return '' unless @{$history{phrases}};
    return serialize_bids_history(\%history);
}


=head2 get_img_bid
    
    возвращает код, в phraseIdHistory соответствующий баннеру-картинке заданного bid

=cut

sub get_img_bid ($) {
    my $bid = shift;
    return 'im' . $bid;
}

=head2 get_keywords_with_history

Получает данные для копирования CTR по указанным фразам.

    # История по фразам для указанной кампании
    my $phrases = get_keywords_with_history({cid => 1234});

    # История по фразам для указанной группы
    my $phrases = get_keywords_with_history({pid => 1234});

    # Все фразы, относящиеся к указанному баннеру (связь через группу)
    my $phrases = get_keywords_with_history({bid => 1234});

    # Явно указанные идентификаторы ключевых фраз
    my $phrases = get_keywords_with_history({ph_ids => [bid.id 1, bid.id 2, ...]});

Возвращает ссылку на массив из хэшей вида:

    {
         id => bids.id,
         pid => phrases.pid,
         phrase => bids.phrase,
         norm_phrase => bids.norm_phrase,
         price => bids.price,
         price_context => bids.price_context,
         phraseIdHistory => вычисленная история
    }

=cut
sub get_keywords_with_history {
    my ($filter) = @_;
    my ($tables, $where);
    if ($filter->{ph_ids}) {
        my $keywords = ref($filter->{ph_ids}) ? $filter->{ph_ids} : [$filter->{ph_ids}];
        $where = {
            "bi.id" => $keywords,
            ($filter->{cid} ? ("c.cid" => $filter->{cid}) : ()),
            ($filter->{pid} ? ("p.pid" => $filter->{pid}) : ()),
        };
        $tables = "
            bids bi
            join phrases p on bi.pid = p.pid
            join campaigns c on c.cid = p.cid
            join banners b on b.pid = p.pid";
    } elsif ($filter->{bid} || $filter->{pid}) {
        my $pid = $filter->{pid} // get_pid(bid => $filter->{bid});
        $where = {"p.pid" => $pid};
        $tables = "
            phrases p
            join campaigns c on c.cid = p.cid
            join bids bi on bi.pid = p.pid
            join banners b on b.pid = p.pid";
    } elsif ($filter->{cid}) {
        $where = {"c.cid" => $filter->{cid}};
        $tables = "
            campaigns c
            join phrases p on p.cid = c.cid
            join banners b on b.pid = p.pid
            join bids bi on bi.pid = p.pid";
    } else {
        die "invalid filter";
    }
    my $dbh = PPC(choose_shard_param($where, [qw/pid cid/], allow_shard_all => 1));
    do_sql($dbh, "SET SESSION group_concat_max_len = 1000000");
    $tables .= "
        left join banner_images bim ON (bim.bid = b.bid and bim.statusShow = 'Yes')
        left join bids_phraseid_history bph ON bph.cid = bi.cid and bph.id = bi.id
    ";

    my $bid_phrases = get_all_sql(
        $dbh, [
            "select
             bi.id, bi.phrase, bi.norm_phrase, bi.price, bi.price_context,
             -- keyword data
                bi.id, bi.phrase, bi.norm_phrase, IFNULL(bi.place, 0) place,
                bi.cid, bi.showsForecast, IF(bi.statusModerate='No', 'No', 'Yes') statusModerate,
                bi.is_suspended,
             -- data for phraseid_history
                c.OrderID, p.pid, bi.PhraseID,
                GROUP_CONCAT(CONCAT(b.bid, ',', IFNULL(b.BannerID, 0),
                                 IF(bim.bid IS NOT NULL,
                                    CONCAT(';im',bim.bid,',',IFNULL(bim.BannerID, 0)),
                                    ''))
                              SEPARATOR ';') group_banners,
             bph.phraseIdHistory
             from $tables
            ", where => $where, "GROUP BY bi.id"
        ],
    );
    foreach my $ph (@$bid_phrases) {
        my %group_banners = map { my @pair = split ',', $_; ($pair[0] => $pair[1]) } split(';', $ph->{group_banners});
        $ph->{phraseIdHistory} = BS::History::merge_mediaplan_history($ph->{phraseIdHistory}, $ph->{OrderID}, $ph->{PhraseID}, $ph->{pid}, \%group_banners);
    }
    return $bid_phrases;
}
1;

