package BS::ResyncQueue;

=encoding utf8

=head1 NAME

    BS::ResyncQueue - медленная переотправка данных в БК

=head1 DESCRIPTION

=cut

use Direct::Modern;

use List::MoreUtils qw/uniq/;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::IDN qw/idn_to_ascii idn_to_unicode/;

use Settings;

use base qw/Exporter/;
our @EXPORT = qw/bs_resync bs_resync_camps bs_resync_whole_camps/;


=head2 Именованные константы с приоритетами для ленивой очереди: PRIORITY_NNNNNN

    Пожалуйста, добавляйте константы в оба проекта. Список в Java:
    https://a.yandex-team.ru/arc/trunk/arcadia/direct/core/src/main/java/ru/yandex/direct/core/entity/bs/resync/queue/model/BsResyncPriority.java


    Нужны для того, чтобы:
        * все приоритеты были в одном месте (вместо россыпи "магических чисел"),
        * можно было легко изменить приоритеты
        * легко понимать и насколько заполнена очередь переотправки.
    Допустимый диапазон -128:127.
    Чем больше приоритет, тем раньше переотправится объект.
    Приоритет > 100 - переотправка вне зависимости от текущей нагрузки на транспорт

    Все приоритеты должны иметь различные значения (проверяется юнит-тестами).
    При добавлении новых приоритетов - сохраняйте отсортированность по приоритету

=over 4

=item PRIORITY_UNARC_CAMP_IN_BS_ON_NOTIFY_ORDER2
=item PRIORITY_MOBILE_ADS_ON_CAMP_NAME_UPDATE
=item PRIORITY_RESEND_DOMAINS_WITH_BLOCKED_TITLE
=item PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE
=item PRIORITY_INTERNAL_REPORTS_FAST_RESYNC
=item PRIORITY_INTAPI_RESYNC_CAMPAIGNS
=item PRIORITY_MULTICURRENCY_SUMS_UPDATED
=item PRIORITY_NDSDISCOUNT_GRAPH_CHANGED
=item PRIORITY_ON_BILLING_AGGREGATE_ADDED
=item PRIORITY_ON_CHANGED_RETARGETING_ACCESSIBILITY
=item PRIORITY_RESEND_SUSPENDED_VIDEO
=item PRIORITY_UPDATE_DOMAIN_RATINGS
=item PRIORITY_UPDATE_BANNER_WITH_ADDITIONS
=item PRIORITY_ON_MOBILE_CONTENT_CHANGED
=item PRIORITY_INTERNAL_REPORTS_LAZY_RESYNC
=item PRIORITY_RESEND_DOMAINS_BS
=item PRIORITY_UPDATE_PERMALINKS
=item PRIORITY_CHANGE_CAMP_CONTENT_LANG
=item PRIORITY_CATALOGIA_BANNER_RUBRICS
=item PRIORITY_RESYNC_WHOLE_CONTEXT_FOR_ABSENT_BID
=item PRIORITY_DEFAULT_WHOLE_CAMP
=item PRIORITY_DEFAULT_CAMPS
=item PRIORITY_DEFAULT
=item PRIORITY_MAINTENANCE_PREPARE_RESYNC_FILE
=item PRIORITY_MAINTENANCE_FIX_VCARD_GEO_ID
=item PRIORITY_ONE_SHOT_FIX_MISSING_IMAGE_FORMATS
=item PRIORITY_ONE_SHOT_FIX_MODERATION_FLAGS
=item PRIORITY_ONE_SHOT_RESYNC_BANNERS_ON_BR_FIX

=back

=cut

use constant PRIORITY_UNARC_CAMP_IN_BS_ON_NOTIFY_ORDER2     =>  119;    # in java
use constant PRIORITY_RESEND_DOMAINS_WITH_BLOCKED_TITLE     =>  109;
use constant PRIORITY_MOBILE_ADS_ON_CAMP_NAME_UPDATE        =>  107;    # https://st.yandex-team.ru/DIRECT-78409
use constant PRIORITY_ONE_SHOT_CAMP_STRATEGY_UPGRADE        =>  103;
use constant PRIORITY_INTERNAL_REPORTS_FAST_RESYNC          =>  102;
use constant PRIORITY_INTAPI_RESYNC_CAMPAIGNS               =>  101;
use constant PRIORITY_MULTICURRENCY_SUMS_UPDATED            =>   95;    # in java
use constant PRIORITY_NDSDISCOUNT_GRAPH_CHANGED             =>   93;
use constant PRIORITY_ON_BILLING_AGGREGATE_ADDED            =>   92;
use constant PRIORITY_ON_CHANGED_RETARGETING_ACCESSIBILITY  =>   90;
use constant PRIORITY_RESEND_SUSPENDED_VIDEO                =>   80;
use constant PRIORITY_UPDATE_DOMAIN_RATINGS                 =>   75;
use constant PRIORITY_UPDATE_BANNER_WITH_ADDITIONS          =>   56;
use constant PRIORITY_ON_MOBILE_CONTENT_CHANGED             =>   55;
use constant PRIORITY_MANUAL_DOMAIN_MIRROR_CORRECTION       =>   54;
use constant PRIORITY_ONE_SHOT_RESYNC_BANNERS_ON_BR_FIX     =>   51; #https://st.yandex-team.ru/DIRECT-68145
use constant PRIORITY_INTERNAL_REPORTS_LAZY_RESYNC          =>   50;
use constant PRIORITY_RESEND_DOMAINS_BS                     =>   40;
use constant PRIORITY_UPDATE_PERMALINKS                     =>   28;
use constant PRIORITY_CHANGE_CAMP_CONTENT_LANG              =>   27;
use constant PRIORITY_CATALOGIA_BANNER_RUBRICS              =>   26;
use constant PRIORITY_RESYNC_WHOLE_CONTEXT_FOR_ABSENT_BID   =>   12;
use constant PRIORITY_DEFAULT_WHOLE_CAMP                    =>    3;
use constant PRIORITY_DEFAULT_CAMPS                         =>    2;
use constant PRIORITY_DEFAULT                               =>    1;
use constant PRIORITY_MAINTENANCE_PREPARE_RESYNC_FILE       => -  3;
use constant PRIORITY_MAINTENANCE_FIX_VCARD_GEO_ID          => - 20;
use constant PRIORITY_ONE_SHOT_FIX_MISSING_IMAGE_FORMATS    => - 21;
use constant PRIORITY_ONE_SHOT_FIX_MODERATION_FLAGS         => - 74;

=head2 bs_resync([ {cid => ..., pid => ...}, ... ])
    
    Переотправка постановка данных в очередь на переотправку

    На вход принимает ссылку на массив из хэшей с ключами cid, pid, bid, priority
    cid - должен быть обязательно указан
    Подробности тут: db_schema/ppc/bs_resync_queue.text

    На выходе количество изменённых строк

=cut

sub bs_resync($) {
    my ($data) = @_;
    
    my %by_cid;
    for my $d (@$data) {
        push @{$by_cid{$d->{cid}}}, [$d->{cid}, $d->{pid}||0, $d->{bid}||0, $d->{priority}//PRIORITY_DEFAULT];
    }

    my $cnt = 0;
    foreach_shard cid => [keys %by_cid], sub {
        my ($shard, $cids) = @_;
        $cnt += do_mass_insert_sql(PPC(shard => $shard), "
                    INSERT INTO bs_resync_queue (cid, pid, bid, priority)
                    VALUES %s
                    ON DUPLICATE KEY UPDATE priority = greatest(priority, values(priority))
                    ",
                    [map {@{$by_cid{$_}}} @$cids]
            );
    };
    return $cnt;
}


=head2 bs_resync_camps([$cid1, ...], priority => BS::ResyncQueue::PRIORITY_DEFAULT_CAMPS)
    
    Переотправка данных уровня кампании
    На вход принимает:
        * ссылку на массив cid-ов
        * именованный параметр - приоритет отправки
    На выходе количество изменённых строк

=cut

sub bs_resync_camps($;@) {
    my ($cids, %O) = @_;
    $O{priority} //= PRIORITY_DEFAULT_CAMPS;

    return bs_resync([map {+{cid => $_, priority => $O{priority}}} @$cids]);
}

=head2 bs_resync_whole_camps($cids, %options)
    
    Переотправка данных уровня кампании вместе со всеми баннерами и условиями
    
    Параметры:
        $cids - ссылка на массив cid-ов
        %options - хеш с полем priority, задающий приоритет в очереди (опционально)
    
    Результат:
        Хеш с полями:
            cid_count - количество переотправленных кампаний, 
            total_count- количество переотправленных объектов,

=cut

sub bs_resync_whole_camps($;@) {
    my ($cids_req, %O) = @_;
    $O{priority} //= PRIORITY_DEFAULT_WHOLE_CAMP;
    
    my $cids = get_one_column_sql(PPC(cid => $cids_req), [
        'SELECT cid FROM campaigns',
        WHERE => {
            'cid' => SHARD_IDS,
        },
    ]);

    my $data = get_all_sql(PPC(cid => $cids), [
        "SELECT p.cid, b.bid, p.pid, $O{priority} AS priority
        FROM phrases p
        JOIN banners b ON b.pid = p.pid
        ",  
        WHERE => {'p.cid' => SHARD_IDS}
        ]);

    for my $cid (@$cids) {
       push @$data, {cid => $cid, priority => $O{priority} };
    }
    
    bs_resync($data);

    return {
            total_count => scalar(@$data),
            cid_count => scalar(@$cids),
    };
}

=head2 bs_resync_banners_by_domain

    Отправляет на ленивую переотправку все баннеры указанного домена

    Позиционные параметры
        $domain - домен, чьи баннеры нужно переотправлять. Может быть как в Punicode, так и в юникод.
    Именованные параметры
        priority - приоритет в очереди на переотправку, обязательный.

    Результат:
        Хеш с полями:
            cid_count   - количество переотправленных кампаний,
            bid_count   - количество переотправленных баннеров,

=cut
sub bs_resync_banners_by_domain {
    my ($domain, %O) = @_;

    my $priority = $O{priority};
    die "priority must be specified" unless defined $priority;

    my $unicode_domain = idn_to_unicode($domain);
    die "Error converting domain to Unicode" unless $unicode_domain;
    my $punicode_domain = idn_to_ascii($domain);
    die "Error converting domain to Punicode" unless $punicode_domain;

    my @domain_conditions =
        map {
            my $rd = reverse $_;
            (reverse_domain => $rd, reverse_domain__starts_with => "$rd.")
        }
        uniq ($unicode_domain, $punicode_domain);

    my $data = get_all_sql(PPC(shard => 'all'), ["
                        SELECT cid, bid, ? as priority
                        FROM banners
                        ", WHERE => {
                            statusModerate__ne => 'New',
                            statusArch => 'No',
                            statusShow => 'Yes',
                            _OR => \@domain_conditions,
                        }],
                    $priority);

    bs_resync($data);

    my $camps_count = scalar(uniq map { $_->{cid} } @$data);
    my $banners_count = scalar(@$data);

    return {
        bid_count   => $banners_count,
        cid_count   => $camps_count,
    }
}


=head2 get_priorities_hash

    Возвращает ссылку на хеш с именованными константами приоритетов

=cut
sub get_priorities_hash {
    no strict 'refs';
    state $hash //= { map {$_ => &{$_} } grep {$_ =~ m/PRIORITY_/} keys %{__PACKAGE__.'::'} };
    return $hash;
}

1;
