package BS::SearchQuery;

# $Id$

=head1 DESCRIPTION

    Модуль реализует интерфейс с БК по получению отчета по поисковым фразам

=cut

use Direct::Modern;

use Mouse;

use Encode;
use LWP::UserAgent;
use List::MoreUtils qw/ zip /;
use Yandex::HashUtils qw/ hash_cut hash_copy /;

use Settings;

use BS::URL;

our $TIMEOUT ||= 20; # таймаут для одиночных запросов в БК
our $READ_SIZE_HINT ||= 1_000_000; # рекомендуемый размер блока данных для потоковой обработки

has log_object => ( is => 'rw', isa => 'Maybe[Object]', default => undef);
has ua => (is => 'rw', lazy_build => 1);
has ua_timeout => (is => 'rw', isa => 'Maybe[Int]', default => undef);


sub _build_ua {
    my ($self) = @_;
    my $timeout = $self->ua_timeout // $TIMEOUT;
    return LWP::UserAgent->new(timeout => $timeout);
}



=head2 _log

    Пишет в лог

=cut

sub _log {
    my ($self, $string) = @_;
    if (defined $self->log_object) {
        $self->log_object->out($string);
    }
}


=head2 create

    my ($code, $bs_id) = $self->create(\%params);

Создание заявки на отчёт в БК. В случае ошибки - пустой ответ.

Параметры уходят в БК:
    orderlist       - список заказов через запятую
    startdate       - дата начала отчёта
    stopdate        - дата конца
    cliksthreshold  - пороговое число кликов
    showsthreshold  - пороговое число показов
    bannerdesc      - добавить к отчёту поля баннера
    metricdesc      - добавить поля метрики
    incemptyqueries - включать в отчет данные по показам без поисковой фразы (SearchQueryMD5 = 0)

=cut

{
my @allowed_bs_params = qw/
    orderlist
    startdate
    stopdate
    bannerdesc
    metricdesc
    incemptyqueries
/;

sub create {
    my ($self, $params) = @_;

    my $call_params = hash_cut $params, @allowed_bs_params;
    my $response = $self->_bs_call(POST => $call_params);

    return if $response->is_error;
    return $self->_decode_bs_response($response);
}
}


=head2 delete

    my ($code, $status) = $self->delete($bs_id);

Удаление заявки на отчёт в БК. В случае ошибки - пустой ответ.

=cut

sub delete {
    my ($self, $bs_id) = @_;

    my $call_params = { act => 'delete', taskid => $bs_id };
    my $response = $self->_bs_call(POST => $call_params);

    return if $response->is_error;
    return $self->_decode_bs_response($response);
}


=head2 check

    my ($code, $status) = $self->check($bs_id);

Проверка статуса заявки на отчёт в БК. В случае ошибки - пустой ответ.

=cut

sub check {
    my ($self, $bs_id) = @_;

    my $call_params = { act => 'check', taskid => $bs_id };
    my $response = $self->_bs_call(GET => $call_params);

    return if $response->is_error;
    return $self->_decode_bs_response($response);
}


=head2 get_chunked_data

    my $error = $self->get_chunked_data($bs_id, $chunk_callback);

Получение и обработка данных готового отчёта

=cut

sub get_chunked_data {
    my ($self, $bs_id, $chunk_cb) = @_;

    my $call_params = { act => 'get', taskid => $bs_id };
    my $response = $self->_bs_call(GET => $call_params, ':content_cb' => $chunk_cb, ':read_size_hint' => $READ_SIZE_HINT);

    return $response->status_line  if $response->is_error;
    return;
}


=head2 get_data($bs_id)

    my $records = $self->get_data($bs_id);

Получает данные отчёта в виде списка хешей; в случае ошибки - пустой ответ.

=cut

sub get_data {
    my ($self, $bs_id) = @_;

    my $status;
    my @records;
    my $chunk_cb = $self->make_chunk_callback(
        status => \$status,
        record => sub { push @records, @_ },
    );

    my $error = $self->get_chunked_data($bs_id, $chunk_cb);
    return if $error || $status;

    return \@records;
}


=head2 _decode_bs_response($response)

    my ($code, $status) = $self->_decode_bs_response($response);

Раскодированный ответ БК.
В скалярном контексте отдаётся код ответа, в списочном - все поля

=cut

sub _decode_bs_response {
    my ($self, $response) = @_;

    my @fields = split /[\r\n]+/, $response->decoded_content;
    $self->_log("BS: $fields[0] $fields[1]")  if $fields[0] && $fields[0]<0;
    return wantarray ? @fields : $fields[0];
}


=head2 _bs_call($http_method => $url_params, @opts)

    my $response = $self->_bs_call(GET => \%form);

Запрос в БК

=cut

sub _bs_call {
    my ($self, $method => $params, @opts) = @_;

    my $ua = $self->ua;
    my $url = URI->new(BS::URL::get('search_query_report'));

    my $response;
    if ($method eq 'POST') {
        $response = $ua->post($url, $params, @opts);
    }
    elsif ($method eq 'GET') {
        $url->query_form($params);
        $response = $ua->get($url, @opts);
    }
    else {
        croak "Unimplemented method <$method>";
    }

    if ($response->is_error) {
        my $action = $params->{act} || 'create';
        $self->_log('got erroneous response from BS');
        $self->_log({act => $action, params => $params, status_line => $response->status_line, url => $url});
    }

    return $response;
}


=head2 make_chunk_callback(%subcallback)

    my @records;
    my %subcallback = (
        chunk => sub { ... },
        status => ...,
        header => ...,
        record => sub { my $record = shift; push @records, $records },
    );

    my $chunk_cb = $self->make_chunk_callback(%subcallback);
    $self->bs_get_chunked_data($bs_id, $chunk_cb);

Создаёт колбек-обработчик чанка для потокового чтения отчёта.
Вызывает другие колбеки, указанные в виде именованных параметров:

    chunk  - на каждый чанк, параметр - полученные данные
    status - при чтении статуса ответа, параметром получает строку статуса
            вместо колбека можно указать ссылку на скаляр, тогда статус запишется туда
    header - при чтении заголовка, параметр - массив названий столбцов
    record - на чтение каждой строки, параметр - хеш {column_name => value}
    record_set - то же самое, но для массива record-ов из чанка

=cut

sub make_chunk_callback {
    my ($self, %subcb) = @_;

    my $status;
    my @head;
    my $is_finished;

    my $tail = q{};
    my $line_count = 0;

    return sub {
        my ($data) = @_;

        $tail .= $data;
        my @report_rows = split /[\r\n]+/, $tail, -1;
        $tail = pop @report_rows;

        my @records;

        ROW:
        while (@report_rows) {
            # не совсем корректно, надо смотреть заголовок
            my $line = decode utf8 => shift @report_rows;
            $line_count ++;

            # первая строка ответа - статус
            if (!defined $status) {
                $status = $line;
                if (my $cb = $subcb{status}) {
                    ref $cb eq 'SCALAR' ? $$cb = $status : $cb->($status);
                }
                next ROW;
            }
            # вторая строка - заголовок
            if (!@head) {
                $line =~ s/^#//;
                @head = split /\t/, $line;
                $subcb{header}->(\@head)  if $subcb{header};
                next ROW;
            }
            # последняя строка - #End
            $is_finished = 1  if $line =~ /^#End/;
            next if $is_finished;

            # остальные строки парсим как tsv-данные
            my @values = split /\t/, $line, -1;
            my %record = zip @head, @values;
            push @records, \%record;
            $subcb{record}->(\%record)  if $subcb{record};
        }

        $subcb{record_set}->(\@records)  if $subcb{record_set};
        $subcb{chunk}->($data)  if $subcb{chunk};
        return;
    };
}

__PACKAGE__->meta->make_immutable();
1;
