package BS::TrafaretAuction;

=encoding utf8

=head1 NAME

    BS::TrafaretAuction - модуль для работы с новыми торгами БК - Трафаретами

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut

use Direct::Modern;

use URI::Escape::XS qw/uri_escape/;
use HTTP::Request;
use HTTP::Request::Common;
use List::Util qw/any first min max/;
use JSON qw/decode_json/;

use Yandex::TimeCommon qw/today/;
use Yandex::Trace qw/current_trace/;
use Yandex::HashUtils;
use Yandex::Log;
use Yandex::HTTP qw/http_parallel_request/;

use Settings;
use TextTools;
use Currencies;
use BannerTemplates;
use BS::History;
use DirectContext;
use GeoTools;
use geo_regions;
use PhraseText ();
use PlacePrice ();
use DeviceTargeting qw/is_only_phone_device_targeting get_target_devices/;
use JavaIntapi::GetTrafficVolume qw//;
use BSAuction qw//;

use base qw/Exporter/;
our @EXPORT = qw/trafaret_auction/;

=head2 $LOG_REQUESTS, $LOG_REQUESTS_FILENAME

    Нужно ли логгировать запросы торгов и в какой файл

=cut
our $LOG_REQUESTS;
our $LOG_REQUESTS_FILENAME = 'bsauction_requests';


=head2 $LOG_CIDS, $LOG_CIDS_FILENAME

    Нужно ли логгировать номера кампаний и информацию о количестве запросов/фраз и в какой файл

=cut
our $LOG_CIDS //= 0;
our $LOG_CIDS_FILENAME = 'bsauction_cids';


our $BANNER_MUL_PHRASES_CHUNK_LIMIT = 150; # при формировании пачки фраз для запроса в торги, <кол-во фраз>*<кол-во баннеров в группе> не должно превышать 250
our $PHRASES_CHUNK_LIMIT = 10; # максимальное количество фраз в запросе в торги

my $TRAFARET_MCBANNER_RANK_OPTION_ID = 13; # ID опции на стороне БК для трафаретов в ГО на поиске BSSERVER-4864
my $TRAFARET_MOBILE_AND_DEVICE_TARGETING_RANK_OPTION_ID = 13; # ID опции на стороне БК для трафаретов в РМП/ТГО с таргетингом BSSERVER-4865 BSRELEASE-129507

=head2 bs_auction_banner_lang

Вычисляет значение banner-lang для передачи в БК. Делает это на основе геотаргетинга и содержимого
$DirectContext::current_context, по site_host. Если в БК не надо передавать banner-lang, то возвращает undef.

    my $banner_lang_or_undef = bs_auction_banner_lang($geo_str);

=cut
sub bs_auction_banner_lang {
    my ($geo_str) = @_;
    return unless $DirectContext::current_context && $DirectContext::current_context->site_host;
    my $translocal_opts = {ClientID => $DirectContext::current_context->client_client_id};
    if ($DirectContext::current_context->site_host =~ /\.tr$/ and is_targeting_in_region($geo_str, $geo_regions::TR, $translocal_opts)) {
        return 'tr';
    }
    return;
}

=head2 trafaret_auction

Кромер входных параметров на поведение влияет глобальный экзмемпляр DirectContext, хранящийся в $DirectContext::current_context

    Входные параметры:
    - массив баннеров
      каждый баннер - хэш с параметрами
        - cid
        - OrderID
        - adgroup_type
        - geo 
        - filter_domain
        - phone
        - fairAuction    
        - title
        - body
        - camp_rest -- сколько денег осталось на кампании в currency без НДС
        - day_budget -- дневное ограничение бюджета на кампании в currency без НДС
        - spent_today -- истраченная кампанией за сегодня сумма в currency без НДС
        - currency -- код валюты, в которой переданы суммы; обязательный параметр
        - device_targeting -- строка из поля camp_options.device_targeting БД, 
                нужна для того чтобы включить мобильные торги для кампаний таргетирванных на смартфоны
        - phrases - массив фраз
          каждая фраза - хэш с полями:
            - phrase - текст фразы или номер категории каталога
            - BannerID
            - PhraseID
            - phraseIdHistory
            - shows / showsForecast
    - ссылка на хеш опций
        - calc_rotation_factor
        - dont_use_trafaret ходить в старые торги
        - forecast_without_banners сходить в торги даже если не передали title и body баннера. Нужно для отчета "Расчёт потенциала клиента"
=cut

sub trafaret_auction {
    my ($banners, $options) = @_;
    $options //= {};

    my @norm_phrases_for_ctr;
    for my $banner (@$banners) {
        for my $ph ( @{$banner->{phrases}} ) {
            # даже без PhraseID может быть история и CTR не потребуется, т.е. выбираем "с запасом"
            if (!$ph->{PhraseID}) {
                PhraseText::ensure_phrase_have_props($ph);
                push @norm_phrases_for_ctr, $ph->{norm_phrase};
            }
        }
    }
    my $norm_phrase2ctr_data;
    if (@norm_phrases_for_ctr) {
        $norm_phrase2ctr_data = PhraseText::mass_get_norm_phrases_ctr_data(\@norm_phrases_for_ctr);
    }
    undef @norm_phrases_for_ctr;

    # создаём список запросов в БК
    my %reqs;
    my $phrases_num = 0;
    my %__data; # хеш для сопоставления данных о баннерах и ответов торгов
    my $id = 0;

    my $filtered_banners = [];
    for my $banner (@$banners) {
        die 'no currency given' unless defined $banner->{currency};
        die 'no group.is_bs_rarely_loaded given' unless exists $banner->{is_bs_rarely_loaded};

        next if $banner->{is_bs_rarely_loaded} || $banner->{camp_is_arch};

        # для баннеров продвижения контента/видео-контента не ходим в торги
        next if $banner->{adgroup_type} eq 'content_promotion_video';
        next if $banner->{adgroup_type} eq 'content_promotion';

        if (
            !($banner->{title} && $banner->{body} || $options->{forecast_without_banners})
            || ($banner->{ad_type} || '') =~ /^(image_ad|cpc_video)$/
        ) {
            for my $p (@{$banner->{phrases}//[]}) {
                $p->{banner_without_text} = 1;
            }
            next;
        }

        push @$filtered_banners, $banner;
    }

    my %STAT_BY_CID;
    for my $banner (@$filtered_banners) {
        my @phrases = @{$banner->{phrases}};
        $phrases_num += @phrases;
        my $banner_data = hash_cut $banner, qw/adgroup_type autobudget autobudget_bid strategy_no_premium camp_rest day_budget spent_today currency timetarget_coef/;
        # т.к. обрабатываем копию массива ссылок, то при изменении элементов @chunk будет меняться содержимое $banner
        my $chunk_size = min int($BANNER_MUL_PHRASES_CHUNK_LIMIT / ($banner->{banners_quantity} || 1)) || 1,
            $PHRASES_CHUNK_LIMIT;
        while (my @chunk = splice @phrases, 0, $chunk_size) {
            $STAT_BY_CID{$banner->{cid} // 0}->{requests}++;
            $STAT_BY_CID{$banner->{cid} // 0}->{phrases}+=@chunk;

            my ($url, $use_trafaret) = _compose_url($banner, \@chunk, $norm_phrase2ctr_data, hash_cut($options, qw/dont_use_trafaret/));
            $__data{$id} = {
                chunk => \@chunk,
                banner_data => $banner_data,
                use_trafaret => $use_trafaret
            };
            $reqs{$id} = {url => $url,};
            $id++;
        }
    }

    if ($LOG_CIDS) {
        # логирование статистики по номерам кампаний
        Yandex::Log->new(
            log_file_name => $LOG_CIDS_FILENAME, date_suf => "%Y%m%d", auto_rotate => 1,
            msg_prefix => sprintf("[pid:%d,reqid:%s]", $$, Yandex::Trace::trace_id()),
        )->out([map {hash_merge {cid=>int($_)}, $STAT_BY_CID{$_}} keys %STAT_BY_CID]);
    }

    # логгирование
    my $logger;
    if ($LOG_REQUESTS) {
        $logger = Yandex::Log->new(
            log_file_name => $LOG_REQUESTS_FILENAME, date_suf => "%Y%m%d", auto_rotate => 1,
            msg_prefix => sprintf("[pid:%d,reqid:%s]", $$, Yandex::Trace::trace_id()),
        );
    }

    # разбор ответов
    my $process_resp = sub {
        my ($id, $clear_resp) = @_;
        $logger->out($clear_resp) if $logger;

        my $content = $clear_resp->{content};
        my $currency = $__data{$id}->{banner_data}->{currency};

        my $bs_data;
        if ($__data{$id}->{use_trafaret}) {
            $bs_data = _parse_bs_data_trafaret($content);
        } else {
            $bs_data = _parse_bs_data_rank24($content, $currency);
        }

        _calculate_bs_prices($__data{$id}->{chunk}, $__data{$id}->{banner_data}, $bs_data, %{$options});
    };

    # параллельные запросы в БК
    my $profile = Yandex::Trace::new_profile('bsrank:bsrank_call', obj_num => $phrases_num);
    if ($phrases_num) {
        my $reqs_num = scalar keys %reqs;
        my $result = http_parallel_request(
            GET => \%reqs,
            max_req => $Settings::BS_RANK_PARALLEL_LEVEL,
            timeout => $Settings::BS_TIMEOUT,
            callback => $process_resp,
            headers => {
                "Host" => $Settings::BSRANK_HOST,
                "Referer" => undef,
            },
            num_attempts => $Settings::BS_NUM_ATTEMPTS,
            # в начале - похожа на BS_NUM_ATTEMPTS * reqs_num, в пределе - похожа на BS_TOTAL_TRIES_COEF * reqs_num
            max_total_retries => $Settings::BS_NUM_ATTEMPTS + int($Settings::BS_TOTAL_TRIES_COEF * $reqs_num),
            soft_timeout => $Settings::BS_SOFT_TIMEOUT,
            ipv6_prefer => 1,
        );
        undef $profile;

        foreach my $resp (values %$result) {
            if (!$resp->{is_success}) {
                utf8::decode($resp->{headers}->{Reason});
                warn "bs_auction got status $resp->{headers}->{Status} from $resp->{headers}->{URL}: $resp->{headers}->{Reason}";
            }
            die "bs_auction died: $resp->{callback_error}" if $resp->{callback_error};
        }

        # Принудительно проставляем флажок, что данных от БК нет, если нет флажка, что данные есть
        # подстраховка от того, что коллбек, обрабатывающий данные - умер в процессе обработки
        for my $banner (@$banners) {
            next if !($banner->{title} && $banner->{body} || $options->{forecast_without_banners} ) || ($banner->{ad_type} || '') =~ /^(image_ad|cpc_video)$/;
            for my $phrase (@{ $banner->{phrases} }) {
                $phrase->{nobsdata} ||= 1 unless $phrase->{bs_data_exists};
            }
        }
    }

    _set_traffic_volume($filtered_banners);

    if (!$options->{calc_rotation_factor}) {
        _set_phrase_statistic($filtered_banners, $options->{calc_rotation_factor});
    }
    return $banners;
}

# подготовка фразы для торгов БК
sub _prepare_text {
    my ($text) = @_;
    $text =~ s/[!+]//g;
    $text =~ s/[\r\n\"]+/ /g;
    $text =~ s/\s-.*//g; # удаляем минус-слова
    $text =~ s/[\s-]+/ /g;
    $text =~ s/^\s+|\s+$//g;
    return $text;
}

# составление url запроса
sub _compose_url {
    ## no critic (Freenode::DollarAB)
    my ($b, $phrases, $norm_phrase2ctr_data, $options) = @_;

    my @params;
    my $use_trafaret = $options->{dont_use_trafaret} ? 0 : 1;  # в какие торги ходим, в новые (трафареты) или старые (спец/гарантия)
    my $bsrank_url = $Settings::BSRANK_URL;

    # географический таргетинг
    $b->{geo} =~ s/\s+//g if $b->{geo};
    my $geo_param = "reg-id=".($b->{geo}||0);
    $geo_param =~ s/,/%0A/g;
    push @params, $geo_param;

    if (exists $b->{no_extended_geotargeting}) {
        my $no_ext_geo = 0 + !!$b->{no_extended_geotargeting};
        push @params, "no-extended-geotargeting=$no_ext_geo";
    }

    if ($b->{currency} ne 'YND_FIXED') {
        push @params, "currency=".get_currency_constant($b->{currency}, 'ISO_NUM_CODE');
    }
    my $banner_lang = bs_auction_banner_lang($b->{geo});
    if ($banner_lang) {
        push @params, "banner-lang=$banner_lang";
    }

    # домен баннера
    my $domain = 'yandex.ru'; # https://st.yandex-team.ru/DIRECT-46905
    if ($b->{filter_domain} || $b->{phone}) {
        # Если телефон не содержит '#' значит надо его пересобрать из country_code, city_code, phone и ext, чтобы при необходимости использовать далее как домен.
        my $phone = (defined($b->{phone})) ? $b->{phone} : "";
        $phone = join "#", map {$b->{$_}} grep {defined($b->{$_})} qw/country_code city_code phone ext/ if ($phone !~ /#/);

        $domain = $b->{filter_domain} || phone_domain($phone);
    }
    push @params, "domain=".uri_escape($domain);

    # учёт временного таргетинга
    if ($b->{fairAuction}) {
        push @params, 'timetable=1';
    }

    #    Запрашивать данные в формате нового VCG аукциона
    push @params, "vcg-auction=1";

    # дописываем данные для релевантности
    # https://st.yandex-team.ru/DIRECT-46905
    my @phrases_sorted = sort map { $_->{phrase} } @$phrases;
    for my $d (
        {field => 'title', param => 'ban-head', default => _prepare_text($phrases_sorted[0])}
        , {field => 'body', param => 'ban-body'}
    ) {
        my $field = $b->{$d->{field}} || $d->{default};
        next if !defined $field;
        $field =~ s/\s+/ /g;
        $field =~ s/$TEMPLATE_METKA/TMPLPHRASE/g;
        push @params, "$d->{param}=".uri_escape($field);
    }

    my $adgroup_type = $b->{adgroup_type} // 'base';
    my $is_banner_for_phone = ($adgroup_type eq 'mobile_content') || is_only_phone_device_targeting($b->{device_targeting}) ? 1 : 0;

    my $is_banner_format = $b->{banners_quantity} && $b->{banners_quantity} == 1
        && $b->{image_BannerID}
        && ($b->{strategy}||'') ne 'different_places';
    # Запрос без operation - ставка, которую надо поставить, чтобы победил именно переданный в запросе баннер
    unless ($is_banner_format) {
        push @params, 'operation=3'; # operation = 3 - ставка, которую надо поставить, чтобы хотя бы один баннер из группы победил
    }

    my $rank_option_id = 0;

        if ( ($adgroup_type eq 'mobile_content') || $b->{device_targeting} ) { # РМП или ТГО с таргетингом на устройства
            my ($target_device_group, $target_detailed_device_type);
            if ($adgroup_type eq 'base' && $b->{device_targeting}) { # ТГО с включенным таргетингом на устройства
                ($target_device_group, $target_detailed_device_type) = get_target_devices($b->{device_targeting});
            } elsif ($adgroup_type eq 'mobile_content') { # PМП
                my $device_type_targeting = $b->{device_type_targeting} // [];
                push @$target_device_group, 1 if any { $_ eq 'phone' } @$device_type_targeting;
                push @$target_device_group, 2 if any { $_ eq 'tablet' } @$device_type_targeting;

                $target_detailed_device_type = 2 if $b->{mobile_content}->{os_type} eq 'Android';
                $target_detailed_device_type = 3 if $b->{mobile_content}->{os_type} eq 'iOS';
            }

            if ($target_detailed_device_type) {
                $rank_option_id = $TRAFARET_MOBILE_AND_DEVICE_TARGETING_RANK_OPTION_ID;
                push @params, "target-detailed-device-type=$target_detailed_device_type";
            }

            if ($target_device_group && scalar @$target_device_group) {
                $rank_option_id = $TRAFARET_MOBILE_AND_DEVICE_TARGETING_RANK_OPTION_ID;
                push @params, "target-device-group=${\join('%0A', @$target_device_group)}";
            }

        }
        
        if ($adgroup_type eq 'mcbanner') { # ГО на поиске
            $rank_option_id = $TRAFARET_MCBANNER_RANK_OPTION_ID;
            push @params, 'mcbanner=1';
        }

        push @params, "rank-option-id=$rank_option_id" if $rank_option_id;


    if ($is_banner_for_phone) {
        push @params, 'only-mobile-pages=1';
    }

    if ($use_trafaret) {
        push @params, 'trafaret-auction=1';
    }

    # фразы
    my $order_id = $b->{OrderID} || 0;
    my @targets;
    my $id = 0;
    for my $ph (@$phrases) {
        $id++;
        my $text = "text:".uri_escape(_prepare_text($ph->{phrase}));

        my $ids;
        if ($is_banner_format) {
            my $banner_history = BS::History::convert_to_banner_history($ph->{phraseIdHistory});
            $ids = BS::History::prepend_history( $banner_history,
                { BannerID => $b->{BannerID}, PhraseID => $ph->{PhraseID} } ) || 'B0,0';
        } else {
            my $group_bids_history = BS::History::group_bids_history($ph->{phraseIdHistory}, $b->{pid});
            $ids = BS::History::prepend_history( $group_bids_history,
                { GroupID => $b->{pid}, PhraseID => $ph->{PhraseID} } ) || 'G0,0';
        }
        $ids =~ s/^,+|,+$//g;

        if (($ids eq 'G0,0' || $ids eq 'B0,0') && defined $ph->{norm_phrase}) {
            my $ctr_data = $norm_phrase2ctr_data->{ $ph->{norm_phrase} };
            my ($ctr, $p_ctr) = (0, 0);
            if (ref($ctr_data) eq 'HASH') {
                $ctr = $ctr_data->{ctr} || 0;
                $p_ctr = $ctr_data->{p_ctr} || 0;
            }

            # https://st.yandex-team.ru/DIRECT-46905
            $ctr ||= 0.03;
            $p_ctr ||= 0.20;

            my $shows = $ph->{shows} || $ph->{showsForecast} || 0;
            my %sf;
            if ($ctr > 0) {
                $sf{shows_forecast_right} = ($ctr < 0.005) ? 300 : $shows;
                $sf{clicks_forecast_right} = int($sf{shows_forecast_right} * $ctr + 0.5);
            }
            if ($p_ctr > 0) {
                $sf{shows_forecast_premium} = $ctr < 0.005 ? 300 : $shows;
                $sf{clicks_forecast_premium} = int($sf{shows_forecast_premium} * $p_ctr + 0.5);
            }
            $ids = 's:'.join(',', map {$sf{$_}||0} qw/shows_forecast_right clicks_forecast_right shows_forecast_premium clicks_forecast_premium/);
        }

        # 0 is market_cpms
        my @target_params = ($id, $order_id, '0', $text);

        my $target = 'target=' . join ',', @target_params, $ids;
        push @targets, $target;
        $ph->{bs_url} = "$bsrank_url?".join('&', @params, $target);
    }

    my $url = "$bsrank_url?".join('&', @params, @targets);
    return ($url, $use_trafaret);
}

sub _set_positions {
    my ($rec, $type, $position_ctr_corrections, $line, $currency) = @_;

    my @prices = map { my $p = {}; ($p->{amnesty_price}, $p->{bid_price}, ) = split(':', $_); $p; } split(',', $line);
    foreach my $i (0..$#prices) {
        if (my $position_ctr_correction = $position_ctr_corrections->{$i}) {
            push @{$rec->{clickometer}}, {
                bid_price               => $prices[$i]->{bid_price},
                amnesty_price           => $prices[$i]->{amnesty_price},
                position_ctr_correction => $position_ctr_correction,
            };
        }
        push @{$rec->{$type}}, _round_bs_data_prices($prices[$i], $currency);
    }
}
# парсинг пришедших из БК данных (старый формат - спец/гарантия)
sub _parse_bs_data_rank24 {
    my ($content, $currency) = @_;

    die 'no currency given' unless $currency;
    return {} unless $content;

    my $block = 7;
    my (%ret, $rec);
    my $lineN = 0;
    for my $line ( split /\n/, $content ) {
        if ($lineN % $block == 0) {
            my ($TargetID, $place, $BannerID, $PhraseID, $price, $min_price, $context_stop_flag, $tragic_context_flag)  = split ',', $line;
            $rec = $ret{$TargetID} = { id => $TargetID, clickometer => [], is_trafaret => 0, };
        } elsif ( $rec && $lineN % $block == 1) {
            # ctr по сумме показов и кликов,ctr в гарантии,ctr в спецразмещении
            @$rec{qw(rank ectr pectr rshows rclicks pshows pclicks)} = split ',', $line;
            $rec->{shows} = $rec->{rshows} + $rec->{pshows};
            $rec->{clicks} = $rec->{rclicks} + $rec->{pclicks};
        } elsif ( $rec && $lineN % $block == 2) {
            _set_positions($rec, 'premium', \%PlacePrice::PREMIUM_POSITION_CTR_CORRECTION, $line, $currency);
        } elsif ( $rec && $lineN % $block == 3) {
            _set_positions($rec, 'guarantee', \%PlacePrice::GUARANTEE_POSITION_CTR_CORRECTION, $line, $currency);
      # } elsif ( $rec && $lineN % $block == 4 && $line =~ /[1-9]/ ) { после перехода на трафареты, цены в ротации нас не интересуют
        } elsif ( $rec && $lineN % $block == 5) {
            my $weighted;
            @$weighted{qw/min max pmin pmax/} = map { my $p; ($p->{amnesty_price}, $p->{bid_price}) = split ':', $_; _round_bs_data_prices($p, $currency)} split ',', $line;

            $rec->{weighted}->{premium} = [$weighted->{pmax}, $weighted->{pmin}, $weighted->{pmin}, $weighted->{pmin}];
            $rec->{weighted}->{guarantee} = [$weighted->{max}, $weighted->{min}, $weighted->{min}, $weighted->{min}];
      # } elsif ( $rec && $lineN % $block == 6) { цена автоброкера ротации нас тоже более не интересуют
        }
        $lineN++;
    }

    return \%ret;
}

=head2 _parse_bs_data_trafaret

    Новые торги возвращают не позиции (спец размещение/гарантия) а "трафареты", во время перевходного периода каждый
    трафатер содержит поле Position, используем его для отображения трафаретов на старый формат.
    
    Если в ответе торгов есть несколько трафаретов с одним и тем же Position то берем трафарет с наименьшим Bid

=cut
my %_rename_bs_fields = ( X => 'position_ctr_correction', Cpc => 'amnesty_price', Bid => 'bid_price',  );
sub _parse_bs_data_trafaret {
    my $content = shift;
    return {} unless $content;

    my (%ret, $data);
    eval {
        $data = decode_json($content); 1;
    } or do {
        return {};
    };

    for my $target (@{ $data->{Targets} // [] }) {
        unless (scalar @{$target->{Clickometer} // []}) {
            $ret{$target->{TargetID}} = undef;
            next;
        }

        my $legacy_stats = $target->{LegacyFields} // {};
        $ret{$target->{TargetID}} = {
            id => $target->{TargetID},

            ectr => $legacy_stats->{ECtr},
            pectr => $legacy_stats->{PremiumECtr},
            rshows => $legacy_stats->{RightShows},
            rclicks => $legacy_stats->{RightClicks},
            pclicks => $legacy_stats->{PremiumClicks},
            pshows => $legacy_stats->{PremiumShows},
            clicks => $legacy_stats->{PremiumClicks} + $legacy_stats->{RightClicks},
            shows => $legacy_stats->{PremiumShows} +  $legacy_stats->{RightShows},

            clickometer => [
                map { hash_kmap( sub { $_rename_bs_fields{$_} }, $_) } map { hash_cut($_, qw/X Cpc Bid/) } @{$target->{Clickometer}}
            ],

            premium => [],
            guarantee => [],
            weighted => {
                premium => [],
                guarantee => [],
            },

            is_trafaret => 1,
        }
    }
    return \%ret;
}

#округление до шага торгов вверх
sub _round_bs_data_prices {
    my ($p, $currency) = @_;
    foreach my $key (keys %$p) {
        $p->{$key} = round_price_to_currency_step($p->{$key} / 1e6, $currency, up => 1) * 1e6 if $p->{$key};
    }
    return $p;
}

# расчёт цен входа
sub _calculate_bs_prices {
    my ($phrases, $banner, $bs_data, %O) = @_;

    my $currency = $banner->{currency};
    die 'no currency given' unless $currency;

    my $max_show_bid_constant = get_currency_constant($currency, 'MAX_SHOW_BID');
    # Цены иногда от БК приходят нулевые, если так, заполняем минимальными ценами.
    my $min_price_constant = get_currency_constant($currency, 'MIN_PRICE');
    my $no_bs_price = {bid_price => $min_price_constant, amnesty_price => $min_price_constant};

    for my $i (0 .. scalar(@$phrases)-1) {
        my $el = $phrases->[$i];
        my $bs = $bs_data->{$i+1};

        # убрать после https://st.yandex-team.ru/DIRECT-78371
        $el->{rank} = $Settings::DEFAULT_RANK;
        $el->{context_stop_flag} = 0;

        if (defined $bs) {
            if (!$bs->{is_trafaret} && (@{$bs->{premium}} < $PlacePrice::PREMIUM_PLACE_COUNT)) {
                $bs->{premium} = [ ($no_bs_price) x $PlacePrice::PREMIUM_PLACE_COUNT ];
                warn sprintf("premium prices are empty (PhraseID: %d\tBannerID: %d\tbs_url: %s)",
                        $el->{PhraseID}, $el->{BannerID}, $el->{bs_url});
            }
            if (!$bs->{is_trafaret} && (@{$bs->{guarantee}} < $PlacePrice::GUARANTEE_PLACE_COUNT)) {
                $bs->{guarantee} = [ ($no_bs_price) x $PlacePrice::GUARANTEE_PLACE_COUNT ];
                warn sprintf("guarantee prices are empty (PhraseID: %d\tBannerID: %d\tbs_url: %s)",
                        $el->{PhraseID}, $el->{BannerID}, $el->{bs_url});
            }

            # Вычисляем broker
            hash_copy $el, $bs, qw/premium guarantee weighted clickometer is_trafaret/;

            $el->{ectr} = $bs->{ectr} / 1_000_000;
            $el->{pectr} = $bs->{pectr} / 1_000_000;

            if ( !$O{calc_rotation_factor} ) {
                $el->{$_} = $bs->{$_} for qw/shows clicks/;
                $el->{$_} = 0 for qw/eshows effective_ctr/;
            }

            # Вычисляем rotation_factor
            for (qw/rshows rclicks pshows pclicks/) {
                $el->{"real_$_"} = $bs->{$_} if defined $bs->{$_};
            }

            # Добавляем флаг, который означает, что данные по баннеру есть.
            # В случае, если БК отвалится, поле будет отсутствовать, что даст возможность обрабатывать ситуацию корректно в других местах.
            $el->{bs_data_exists} = 1;
        } else {
            print STDERR "No bs data\n";
            $el->{nobsdata} = 1;
            $el->{premium} = [ ($no_bs_price) x $PlacePrice::PREMIUM_PLACE_COUNT ];
            $el->{guarantee} = [ ($no_bs_price) x $PlacePrice::GUARANTEE_PLACE_COUNT ];
            $el->{clickometer} = [ (hash_merge($no_bs_price, { position_ctr_correction => 100_000 })) x 2 ];
        }
        if (($banner->{adgroup_type} // '') eq 'mcbanner') {
            if (!$bs->{is_trafaret}) {
                # если ходили в старые торги, то:
                # в качестве цены за клик в новом типе кампании нужно брать цену 1-го места гарантии
                # https://st.yandex-team.ru/DIRECT-66999#1501154268000
                # TODO DIRECT-67003 - возможно потребуется продублировать и поднять этот блок выше, если нужно втащить его в AutoBroker::calc_price
                $el->{price_for_mcbanner} = [ PlacePrice::get_data_by_place($bs, $PlacePrice::PLACES{GUARANTEE1}) ];  
            } else {
                my $price_for_mcbanner = first { $_->{position_ctr_correction} == 1_000_000 } @{$bs->{clickometer}};
                if ($price_for_mcbanner) {
                    $price_for_mcbanner = _round_bs_data_prices(
                        hash_cut($price_for_mcbanner, qw/amnesty_price bid_price/), $currency
                    );
                    if ( ($price_for_mcbanner->{bid_price} / 1e6) > $max_show_bid_constant ) {
                        $price_for_mcbanner->{dont_show_bid_price} = 1;
                    }
                    $el->{price_for_mcbanner} = [ $price_for_mcbanner ];
                    # в целях обратной совместимости (с онлайн/оффлайн конструкторами) заполняем premium/guarantee
                    $el->{premium} = [ ($price_for_mcbanner) x $PlacePrice::PREMIUM_PLACE_COUNT ];
                    $el->{guarantee} = [ ($price_for_mcbanner) x $PlacePrice::GUARANTEE_PLACE_COUNT ];
                } else { # если по каким-то причинам БК нарушили контракт, и не прислали X == 1_000_000 для ГО на поиске
                    print STDERR "No bs_trafaret data for mcbanner\n";
                    $el->{nobsdata} = 1;
                    $el->{price_for_mcbanner} = [ $no_bs_price ];
                    $el->{premium} = [ ($no_bs_price) x $PlacePrice::PREMIUM_PLACE_COUNT ];
                    $el->{guarantee} = [ ($no_bs_price) x $PlacePrice::GUARANTEE_PLACE_COUNT ];
                }
            }
        }
        if (defined $el->{phr}) {
            $el->{phrase} = $el->{phr};
        }
        _set_phrase_ctr($el, $O{calc_rotation_factor});
    }
}

sub _set_phrase_ctr {
    my ($el, $calc_rotation_factor, $force) = @_;

    # Расчёт Ctr, приведение цен к приятному формату
    if ($el->{PhraseID} || $el->{phraseIdHistory} || $force) {
        $el->{Ctr} = $el->{shows} ? 100 * ($el->{clicks} || 0) / $el->{shows} : 0;
        $el->{Clicks} = $el->{clicks};
        $el->{Shows} = $el->{shows};
        $el->{$_} = sprintf("%.2f", $el->{$_}||0) for qw(broker price Ctr);
    } elsif ( !$calc_rotation_factor ) {
        for my $field(qw(shows Shows clicks Clicks ctr Ctr)) {
            $el->{$field} = 0;
        }
    }
}

=head2 _set_traffic_volume

    ходим в intapi ручку для вычисления "промежуточных" X

=cut

sub _set_traffic_volume {
    my ($banners) = @_;
    return unless scalar @$banners;

    my @items;
    foreach my $banner (@$banners) {
        # Пропускаем фразы без резултатор торгов и фразы кампаниий ГО на поиске, тк в них только одна ставка (X == 100)
        foreach my $phrase (grep { $_->{bs_data_exists} && !exists $_->{price_for_mcbanner} } @{ $banner->{phrases} }) {
            push @items, { phrase => $phrase, domain => $banner->{domain}, currency => $banner->{currency}, };
        }
    }
    return unless scalar @items;

    JavaIntapi::GetTrafficVolume->new(items => \@items)->call();

   foreach my $banner (@$banners) {
        my $max_show_bid = get_currency_constant($banner->{currency}, 'MAX_SHOW_BID');
        foreach my $phrase (grep { $_->{bs_data_exists} } @{ $banner->{phrases} }) {
            my @traffic_volume = keys %{ $phrase->{traffic_volume} // {} };
            next unless @traffic_volume;

            my $min_traffic_volume = min(@traffic_volume);
            my $max_traffic_volume = max(@traffic_volume);
            my %show_in_table = map { $_ => 1 } 
                                grep { $min_traffic_volume <= $_ && $_ <= $max_traffic_volume }  
                                (@PlacePrice::PRESET_TRAFFIC_VOLUME, $min_traffic_volume, $max_traffic_volume);

            foreach my $tv (@traffic_volume) {
                my $item = $phrase->{traffic_volume}->{ $tv };
                if ( ($item->{bid_price} / 1e6) > $max_show_bid) {
                    $item->{dont_show_bid_price} = 1;
                }
                if ($show_in_table{ $tv }) {
                    $item->{show_in_table} = 1;
                }
            }
        }
    }
}

=head2 _set_phrase_statistic

    для всех переданных фраз проставит clicks, shows из java intapi
    если у фразы PhraseID = 0 или данных для нее нет, то clicks и shows = 0
=cut
  
sub _set_phrase_statistic {
    my ($banners, $calc_rotation_factor) = @_;

    my @items;
    for my $banner (@$banners) {
        push @items, map { {id => $_->{id}, pid => $banner->{pid}, cid => $banner->{cid}} } @{ $banner->{phrases} }; 
    }
    my $statistic = BSAuction::get_show_conditions_statistic('phrase', \@items);

    for my $banner (@$banners) {
        for my $phrase (@{ $banner->{phrases} }) {
            my $s = $statistic->{ $banner->{pid} // '' }->{ $phrase->{id} // '' } // {};
            $phrase->{shows} = $s->{shows} || 0;
            $phrase->{clicks} = $s->{clicks} || 0;
            $phrase->{eshows} = sprintf("%.2f", $s->{eshows} || 0);
            $phrase->{effective_ctr} = ($phrase->{eshows} > 0.001) ? (100 * $phrase->{clicks} / $phrase->{eshows}) : 0;
            _set_phrase_ctr($phrase, $calc_rotation_factor, 1);
        }
    }
}

1;
