package BS::URL;

=encoding utf8

=head1 NAME

    BS::URL - получение URL ручек БК с учетом разных стендов и отщепления части запросов в препрод

=head1 SYNOPSIS

    use BS::URL;
    my $url2 = BS::URL::get('master-report', ClientID => 15155);

    my $export_url = BS::URL::get_by_preprod_param('update_data2', want_preprod => 1);

=head1 DESCRIPTION

    В этом модуле определены следующие знания о "ручках БК":
        какой хост использовать
        собственно адрес ручки на хосте
        есть ли препрод (другими словами - нужно ли пытаться использовать препрод для этой ручки)

    Модуль позволяет по имени ручки - получить ее адрес.

    Дополнительно передав "ключ" и "значения", может быть получен препрод-адрес.
    При этом будет процент отщепляемых запросов (по остатку от деления соответствующего ClientID на 100)
    управляется через свойство в БД.

=cut

use Direct::Modern;

use Hash::Util;
use List::MoreUtils qw/uniq/;

use Yandex::DBShards qw/get_shard_multi/;
use Yandex::ListUtils qw/nsort/;
use Yandex::Log::Messages;
use Yandex::Validate qw/is_valid_int/;

use Property;
use Settings;

=head1 CONSTANTS

=head2 EXPORT_PREPROD_USAGE_PCT_REFRESH_INTERVAL

    Время в секундах, на которое кешируем полученное из БД
    значение процента участвующих в эксперименте

=cut

use constant EXPORT_PREPROD_USAGE_PCT_REFRESH_INTERVAL => 300;

=head2 EXPORT_PREPROD_USAGE_PCT_PROP_NAME

    Название свойства в БД, в котором храним процент

=cut

use constant EXPORT_PREPROD_USAGE_PCT_PROP_NAME => 'BS_EXPORT_PREPROD_USAGE_PCT';

=head1 VARIABLES

=head2 %DATA

    Данные о ручках БК.
    Ключ - человекопонятное название ("тег") ручки
    Значение - hashref со следующими полями:
        url_path    - относительный адрес ручки, обязательный параметр
        host_type   - тип хоста, обязательный параметр, по нему из %TYPE2HOST будет получен хост
        has_preprod - может ли эта ручка использовать препрод-хост

=cut

our %DATA = (
    active_orders_incremental => { url_path => '/export/active_orders_4.cgi', host_type => 'export' },
    url_availability => { url_path => '/export/exp_stat_url_availability.cgi', host_type => 'export' },

    master_report => { url_path => '/export/master-report.cgi', host_type => 'fastexport', has_preprod => 1 },
    direct_goal_stat => { url_path => '/export/export_direct_goal_stat.cgi', host_type => 'fastexport'},

    import_auto_video_creative => { url_path =>'/export/import-auto-video-creative.cgi', host_type => 'import' },

    search_query_report => { url_path => '/export/search-query-report.cgi', host_type => 'export' },

    domain_id => { url_path => '/domain', host_type => 'id_generator' },
);
Hash::Util::lock_hash_recurse(%DATA);

=head2 %TYPE2HOST

    Расшифровка "типов хостов" в хостнеймы для продакшен и пред-продакшн стендов БК.
    Ключ - "условное название" (host_type из %DATA)
    Значение - hashref с двумя ключами:
        host    - хостнейм, обязательный параметр
        pre     - хостнейм для пред-продакшен стенда данного типа.
                  параметр опциональный: если отсутсвует - значит пред-прода нет и все запросы должны идти в host

=cut

our %TYPE2HOST = (
    export => { host => $Settings::BS_HOST_EXPORT },
    fastexport => { host => $Settings::BS_HOST_FAST, pre => $Settings::BS_HOST_FAST_PRE },
    import => { host => $Settings::BS_HOST_IMPORT, pre => $Settings::BS_HOST_IMPORT_PRE },
    id_generator => { host => $Settings::BS_HOST_ID_GENERATOR},
);
Hash::Util::lock_hash_recurse(%TYPE2HOST);

=head1 SUBROUTINES

=head2 get($name[, $key, $ids])

    Получить строку с адресом ручки $name в зависимости от значений $ids ключа $key.
    Алгоритм выбора описан ниже. Результат логирует в $log.

    Алгоритм выбора хоста в зависимости от значений $ids ключа $key:
      если у ручки нет флага has_preprod - выбирается обычный хост ( $TYPE2HOST{ $host_type }->{host} )
      если ключ не определен или неизвестен, или список значний пустой - выбирается обычный хост
      ключ ClientID: если уникальное значение только одно и остаток от его деления на 100
        меньше значения свойства EXPORT_PREPROD_USAGE_PCT_PROP_NAME,
        то выбирается пре-хост ( $TYPE2HOST{ $host_type }->{pre} ), иначе - обычный хост
      ключ OrderID: по значениям получаются ClientID (по метабазе), и для них применяется логика как для ключа ClientID
    Для случаев, когда было из чего выбирать - делается дополнительная запись в $log

    Параметры:
        $name   - "название" ручки
        $key    - тип ключа: OrderID или ClientID
        $ids    - значение, ссылка на массив значений или строка со списком значений ключа
    Результат:
        $url    - строка вида 'http://choosed_host/path/to/method'

=cut

sub get {
    my ($name, $key, $ids, %O) = @_;

    my ($HAS_PREPROD, $PATH, $TYPE) = _get_description($name);
    my ($HOST_DEFAULT, $HOST_PRE) = _get_host_by_type($TYPE);

    # дополнительно проверяем, что настройки не разошлись с требованиями (например в песочнице)
    if ($HAS_PREPROD && !$HOST_PRE) {
        die "'" . ($name // 'undef') . "' defined as 'has_preprod', but no pre-host";
    }

    if (!$ids) {
        $ids = [];
    } elsif (ref $ids ne 'ARRAY') {
        $ids = [ split qr/[^0-9]+/, $ids ];
    }

    my $CHOOSED_HOST;
    if (!$HAS_PREPROD) {
        # у ручки нет препрода
        $CHOOSED_HOST = $HOST_DEFAULT;
    } elsif (!( $key && $ids && @$ids )) {
        # параметры не переданы / неизвестны / пустой список
        $CHOOSED_HOST = $HOST_DEFAULT;
    } elsif ($O{force_preprod}) {
        $CHOOSED_HOST = $HOST_PRE;
    } else {
        # попробуем решить, какой хост нужен
        my $usage_pct = _get_exp_usage_pct();

        my $remainder;
        if ($usage_pct) {
            # обрабатываем ключи/значения только если есть процент отщепления
            if ($key eq 'OrderID') {
                my $ClientIDs = get_shard_multi(OrderID => $ids, 'ClientID');
                if (scalar(uniq(values(%$ClientIDs))) == 1) {
                    $remainder = (values(%$ClientIDs))[0] % 100;
                }
            } elsif ($key eq 'ClientID') {
                if (scalar(uniq(@$ids)) == 1) {
                    $remainder = $ids->[0] % 100;
                }
            }
        }

        if (defined $remainder && $remainder < $usage_pct) {
            $CHOOSED_HOST = $HOST_PRE;
        } else {
            $CHOOSED_HOST = $HOST_DEFAULT;
        }

        # логируем только те случаи, где выбирали между пре- и обычным хостом.
        _log(sprintf('rem: %2s, pct: %3u; by ClientID. %s => %s', ($remainder // '??'), $usage_pct, $key, join (',', nsort(@$ids))));
    }

    return _format_url($CHOOSED_HOST, $PATH);
}

=head2 get_by_params($name, %params)

    my $url1 = get_by_params('update_data2', want_preprod => 1, log_message => "UUID ...");
    my $url2 = get_by_params('update_data2');
    my $url3 = get_by_params('update_data2', want_host => 'xxx.zz');

    Получить строку с адресом ручки $name в зависимости от параметров %params. Приоритет параметров следующий:
    Если передан хост (want_host) - будет использован он.
    Если передан флаг необходимости препрода (want_preprod) - будет использован препрод-хост ручки,
        если у ручки он есть, иначе - умолчательный.
    Иначе - будет использован умолчательный хост ручки.
    Результат логирует в $log, включая log_message если он определен.

    Параметры позиционные (обязательные):
        $name           - "название" ручки
    Параметры именованные (опциональные):
        want_host       - сформировать адрес для этого хоста
        want_preprod    - сформировать адрес для препрод хоста
        log_message     - дополнительное сообщения для записи в лог
    Результат:
        $url    - строка вида 'http://choosed_host/path/to/method'

=cut

sub get_by_params {
    my ($name, %params) = @_;

    die 'name is not specified' unless $name;

    my ($HAS_PREPROD, $PATH, $TYPE) = _get_description($name);
    my ($HOST_DEFAULT, $HOST_PRE) = _get_host_by_type($TYPE);

    my ($CHOOSED_HOST, $choose_msg);
    if ($params{want_host}) {
        $CHOOSED_HOST = $params{want_host};
        $choose_msg = $params{want_host};
    } elsif ($params{want_preprod}) {
        $CHOOSED_HOST = $HAS_PREPROD ? $HOST_PRE : $HOST_DEFAULT;
        $choose_msg = 'preprod';
    } else {
        $CHOOSED_HOST = $HOST_DEFAULT;
        $choose_msg = 'default';
    }

    _log(sprintf('%swant %s host', ($params{log_message} ? "$params{log_message}: " : ''), $choose_msg));

    return _format_url($CHOOSED_HOST, $PATH);
}

=head2 INTERNAL SUBROUTINES

=head3 _get_description($name)

    По "имени" $name ручки получить тип хоста, адрес ручки и опцию "есть ли препрод"

    Параметры:
        $name   - "название" ручки
    Результат:
        $has_preprod    - флаг 0/1 есть ли у ручки препрод
        $path           - путь до ручки
        $type           - тип хоста (один из %TYPE2HOST)

=cut

sub _get_description {
    my $name = shift;
    my ($HAS_PREPROD, $PATH, $TYPE);

    if ($name && exists $DATA{ $name }) {
        $PATH = $DATA{ $name }->{url_path};
        $TYPE = $DATA{ $name }->{host_type};
        $HAS_PREPROD = exists $DATA{ $name }->{has_preprod} && $DATA{ $name }->{has_preprod} ? 1 : 0;
    } else {
        die "no description for '" . ($name // 'undef') . "'";
    }

    return ($HAS_PREPROD, $PATH, $TYPE);
}

=head3 _get_host_by_type($type)

    По типу хоста получить значения хоста и препрод-хоста

    Параметры:
        $type   - тип хоста (один из %TYPE2HOST)
    Результат:
        $host   - хост
        $pre    - препрод-хост (может быть undef)

=cut

sub _get_host_by_type {
    my $type = shift;

    my $default = $TYPE2HOST{ $type }->{host};
    my $pre = exists $TYPE2HOST{ $type }->{pre} ? $TYPE2HOST{ $type }->{pre} : undef;

    return ($default, $pre);
}

=head3 _get_exp_usage_pct()

    Получить текущее значение процента участвующих в "эксперименте".
    Результат кешируется на время EXPORT_PREPROD_USAGE_PCT_REFRESH_INTERVAL секунд.

    my $usage_pct = _get_exp_usage_pct();

=cut

sub _get_exp_usage_pct {
    state ($last_fetch_time, $usage_pct, $prop);

    $prop //= Property->new(EXPORT_PREPROD_USAGE_PCT_PROP_NAME);

    if (time() - ($last_fetch_time // 0) > EXPORT_PREPROD_USAGE_PCT_REFRESH_INTERVAL) {
        $usage_pct = $prop->get();
        if (!is_valid_int($usage_pct, 0, 100)) {
            # если свойство не задано или в БД ошибочное значение
            $usage_pct = 0;
        }
        $last_fetch_time = time();
    }

    return $usage_pct;
}

=head3 _log(@messages)

    Обертка над Yandex::Log::out:
        создает объект при необходимости
        устанавлиает в качестве префикса лога - текущий span_id

    Параметры:
        @messages - массив сообщений для логирования.
                    передаются как есть в $log->out

=cut

sub _log {
    state $log;
    $log //= Yandex::Log::Messages->new();
    $log->msg_prefix('bs_url');
    $log->out(@_);
}

=head3 _format_url($host, $path)

    Сформировать и залогировать url.
    Умирает, если $host не определен

    Параметры:
        $host   - выбранный хост для ручки
        $path   - путь до ручки
    Результат:
        $url    - готовый url

=cut

sub _format_url {
    my ($host, $path) = @_;

    die 'no host for url' unless $host;

    my $url = 'http://' . $host . $path;
    _log($url);

    return $url;
}

1;
