=pod

=encoding utf8

=head1 NAME

BSStatImport

=head1 DESCRIPTION

Designed to import BS statistics. Can returns statistics as arrayref of hashes or iterator.

BS returns different formats of data which depends on type of request:

    incremental:
    OrderID stat_date target_type shows clicks sum updated_time CostCur

    period:
    OrderID stat_date target_type shows clicks sum CostCur

All future BS exports will be in standard format:
    * first line -- status code (0 for OK)
    * second line -- #<tab-separated column names> (for example: #OrderID   UpdateTime  TargetType  Shows   Clicks  Cost    MaxHostTime CostCur)
    * <tab-separated lines of data>
    * <info line>? -- optional, only for incremental formats
    * last line -- #End

=cut

package BSStatImport;

use Direct::Modern;

use JSON;
use List::MoreUtils qw/none zip any uniq/;
use LWP::UserAgent;

use Yandex::DateTime;
use Yandex::HashUtils;
use Yandex::HTTP;
use Yandex::ListUtils;
use Yandex::Log::Messages;
use Yandex::Trace;
use Yandex::Validate;

use BS::URL;
use Settings;

# размер чанков по умолчанию для process_data_chunk
use constant DEFAULT_CHUNK_SIZE => 250_000;

our $BS_STAT_TYPES = {
    active_orders_incremental => {
        params => {
            # default request limit
            limit => 250_000
        },
        http_parallel_request => {
            method => 'POST',
            # default request timeout
            timeout => 600,
        },
        info_line => qr/^(?:[0-9]+:[0-9]+(?:,|;|$))+$/,
        incremental => 1,
        required_fields => [qw/OrderID  UpdateTime  Stop  Shows  Clicks  Cost  CostCur/],
        fields_format => {
            OrderID => 'uint',
            UpdateTime => 'datetime',
            Stop => 'uint',
            Shows => 'uint',
            Clicks => 'uint',
            Cost => 'uint',
            CostCur => 'uint',
        },
        value_modifiers => {
            UpdateTime => 'datetime',
            Cost => 'sum',
            CostCur => 'sum',
        },
    },
    broadmatch => {
        # не READONLY, этой же ручкой ставятся задания на построение отчетов!
        url => $Settings::BS_EXPORT_PROXY_READONLY.'export/export_broadmatch_stat2.cgi',
        params => {format => 1},
        requied_fields => [qw/UpdateTime    BannerID    Phrase  Shows   Clicks  Cost    CostCur/],
        fields_format => {
            UpdateTime => 'date',
            BannerID => 'uint',
            # пустые фразы в ответах БК иногда встречаются
            #Phrase => 'not_empty',
            Shows => 'uint',
            Clicks => 'uint',
            Cost => 'uint',
            CostCur => 'uint',
        },
        value_modifiers => {
            UpdateTime => 'date',
            Cost => 'sum',
            CostCur => 'sum',
        },
    },
    url_availability => {
        incremental => 1,
        info_line => qr/^[0-9]+$/,
        required_fields => [qw/Url Available/],
        http_parallel_request => {
            method => 'GET',
        },
        fields_format => {
            Url => 'not_empty',
            Available => {type => 'uint', min => 0, max => 1},
        },
        params => {
            limit => 150_000,
        },
    },
    direct_goal_stat => {
        required_fields => [qw/OrderID GoalID AttributionType SearchGoalsCount ContextGoalsCount/],
        fields_format => {
            OrderID => 'uint',
            GoalID => 'uint',
            AttributionType => 'uint',
            SearchGoalsCount => {type => 'uint', min => 0},
            ContextGoalsCount => {type => 'uint', min => 0},
        },
        # пороговое число строк, свыше которого не храним сериализованные данные готовым массивом
        deferred_serialization_lines_border_count => 300_000,
        http_parallel_request => {
            method => 'GET',
        },
    },
};

# default request timeout in seconds
our $DEFAULT_REQUEST_TIMEOUT = 1200;

my %FORMAT_VALIDATORS = (
    uint => {
        code => sub {
            my ($value, $params) = @_;
            return "invalid uint" unless is_valid_int($value, $params->{min}, $params->{max});
            return undef;
        },
        default_params => {min => 0},
    },
    float => {
        code => sub {
            my ($value, $params) = @_;
            return "invalid float" unless is_valid_float($value);
            return undef;
        },
    },
    datetime => {
        code => sub {
            my ($value, $params) = @_;
            return 'invalid datetime ' . ($value // 'undef') unless $value && $value =~ /^\d{14}$/;
            return undef;
        },
    },
    date => {
        code => sub {
            my ($value, $params) = @_;
            return 'invalid date ' . ($value // 'undef') unless $value && $value =~ /^\d{8}$/;
            return undef;
        },
    },
    not_empty => {
        code => sub {
            my ($value, $params) = @_;
            return 'invalid not_empty value ' . ($value // 'undef') unless $value;
            return undef;
        },
    },
    re_match => {
        code => sub {
            my ($value, $params) = @_;
            my $re = $params->{re};
            return "unmatched \"$re\" value " . ($value // 'undef') unless $value && $value =~ $re;
            return undef;
        }
    },
);

my %VALUE_MODIFIERS = (
    # 20120921000000 -> 2012-09-21
    date => sub {
        my ($value) = @_;
        $value =~ 
            s{^ (\d{4}) # year as yyyy
                (\d{2}) # month as mm
                (\d{2}) # day as dd
                \d{6}   # hours, minutes and seconds aren't needed
            $}{$1-$2-$3}x;
        return $value;
    },
    # 20120921000000 -> 2012-09-21 00:00:00
    datetime => sub {
        my ($value) = @_;
        $value =~ 
            s{^ (\d{4}) # year as yyyy
                (\d{2}) # month as mm
                (\d{2}) # day as dd
                (\d{2}) # hours as HH
                (\d{2}) # minutes as MM
                (\d{2}) # seconds as SS
            $}{$1-$2-$3 $4:$5:$6}x;
        return $value;
    },
    # 65190000 -> 65.190000
    sum => sub {
        my ($value) = @_;
        return sprintf('%0.6f', $value/1e6);
    },
    # -5 => 0
    zero_negative => sub {
        my ($value, $row) = @_;
        if (defined $value && $value >= 0) {
            return $value;
        } else {
            my $log = new Yandex::Log::Messages();
            $log->msg_prefix('BSStatImport_zero_negative');
            $log->out($row);
            return 0;
        }
    },
    keyval_list => sub {
        my ($value) = @_;
        my @list;
        for (split /,+/, $value) {
            my ($key, $val) = split /:/;
            push @list, [$key, $val];
        }
        return \@list;
    },
    keyval_list_sum_by_key => sub {
        my ($value) = @_;
        return [] unless $value;
        my %data;
        for my $pair (split(qr/,+/, $value)) {
            my ($key, $val) = split(qr/:/, $pair);
            $data{$key} += $val;
        }
        return [ map { [ $_, $data{$_} ] } keys %data ];
    },
);

# статусный код для внутренних ошибок валидации
my $INTERNAL_ERRORS_CODE = 255;

=head1 METHODS

=head2 new(type => 'incremental', params => {}, timeout => 123)

    Конструктор для BSStatImport

    Параметры именованные:
        type - строка, должна быть одним из ключей %$BS_STAT_TYPES
        params - hashref. Может быть, например {info => last_orderstat_info} - для инкрементальных статистик.
                 или {orderid => 123456, start => 'yyyy-mm-dd', stop => 'yyyy-mm-dd'} - для подневных статистик.
        timeout - таймаут запроса в секундых (опциональный; умолчание берется из определения типа в BS_STAT_TYPES
                  или равняется $DEFAULT_REQUEST_TIMEOUT секундам)
        separate_invalid_rows - булев параметр, отделять ли невалидные строки из входных данных (параметр опциональный)
                                умолчание - отключено - если есть строки, не проходящие валидацию get() вернет ошибку
                                включено - get() не возвращает ошибку при наличии таких строк, невалидные данные
                                    будут доступны с помощью метода get_invalid_rows
        fake_url - использовать указанный url вместо ручки БК, используется для удобства тестирования

    Результат:
        объект BSStatImport

=cut

sub new
{
    my $class = shift;
    my %args = @_;

    # type is required
    if(not exists $args{type}){
        confess "Can't create BSStatImport object without type of query.";
    }

    # type can be any from $BS_STAT_TYPES
    if ( !exists $BS_STAT_TYPES->{$args{type}} ) {
        confess "Can't create BSStatImport object. Wrong type of statistics - '$args{type}'.";
    }

    my $type_description = $BS_STAT_TYPES->{$args{type}};
    my $params = hash_merge {}, $type_description->{params}, $args{params};

    for my $date_param (qw(start stop date_to date_from)){
        if(exists $params->{$date_param} && $params->{$date_param}){
            $params->{$date_param} = date($params->{$date_param})->ymd('');
        }
    }

    my $self = bless(
        {
            type => $args{type},
            params => $params,
            timeout => $args{timeout},
            _separate_invalid_rows => ($args{separate_invalid_rows} ? 1 : 0),
            _fake_url => $args{fake_url},
        },
        $class,
    );

    return $self;
}

=head2 get()

    Connects to BS and gets statistics
    if (my $error = $bsstat->get()) {
        die $error;
    } else {
        # do something with $bsstat->{named_response}
    }

    receives:
        nothing

    returns:
        description of error if an error occured, 0 otherwise

=cut

sub get
{
    my $self = shift;

    my $type_description = $self->_type_description();
    my $response;

    if (defined $type_description->{http_parallel_request}) {
        my $request_params = $type_description->{http_parallel_request};
        my %req;

        if ($request_params->{method} eq 'POST') {
            %req = ( 1 => {
                url => $self->get_url(),
                body => $self->_get_params_for_bs(),
            });
        } else {
            %req = ( 1 => {
                url => $self->get_url(),
            });
        }
        my $profile = Yandex::Trace::new_profile('bs_stat_import:get', tags => $self->{type});
        my $resp = Yandex::HTTP::http_parallel_request(
                $request_params->{method} => \%req,
                headers => { 'Content-type' => 'application/x-www-form-urlencoded' },
                timeout => ($self->{timeout} // $request_params->{timeout} // $DEFAULT_REQUEST_TIMEOUT),
                # дефолта для connect_timeout нет, полагаемся на дефолт в http_parallel_request
                connect_timeout => $request_params->{connect_timeout},
                max_req => $request_params->{max_req},
                soft_timeout => $request_params->{soft_timeout},
                num_attempts => $request_params->{num_attempts},
                handle_params => { keepalive => 1 },
                persistent => 0,
                ipv6_prefer => 1,
            )->{1};
        undef $profile;
        if (!$resp->{is_success}) {
            utf8::decode($resp->{headers}->{Reason});
            return "$INTERNAL_ERRORS_CODE\tBS request failed: $resp->{headers}->{Status} $resp->{headers}->{Reason} $request_params->{method} url: ".$self->get_url().((defined $request_params->{method} && $request_params->{method} ne 'GET')?" Params: ".Yandex::HTTP::get_param_string($req{1}->{body}):"");
        }
        utf8::decode($resp->{content});
        $response = $resp->{content};        
    } else {
        my $profile = Yandex::Trace::new_profile('bs_stat_import:get', type => $self->{type});
        my $ua = new LWP::UserAgent(timeout => ($self->{timeout} // $DEFAULT_REQUEST_TIMEOUT));
        my $resp = $ua->get($self->get_url());
        undef $profile;
        if (!$resp->is_success()) {
            return sprintf("%d\tBS request failed: %s; URL: %s", $INTERNAL_ERRORS_CODE, $resp->status_line(), $self->get_url());
        }
        $response = $resp->decoded_content();
    }
    # data processing
    my @lines = split /\n/, $response;

    return "$INTERNAL_ERRORS_CODE\tEmpty response" if !@lines;
    $self->{count} = scalar(@lines);

    # check for errors
    $self->{bs_status_code} = shift(@lines);

    # check for correct last line
    my $endline = pop(@lines);

    return "$INTERNAL_ERRORS_CODE\tIncorrect last but one line: $endline" if $endline ne '#End';

    if ($type_description->{incremental}){
        $self->{info} = pop(@lines);
        if ($type_description->{info_line} && $self->{info} !~ /^$type_description->{info_line}$/) { 
            return "$INTERNAL_ERRORS_CODE\tIncorrect info line: $self->{info}";
        }
    }

    return "$INTERNAL_ERRORS_CODE\tNo line with field names" unless @lines;
    my $field_names_row = shift(@lines);
    return "$INTERNAL_ERRORS_CODE\tInvalid line with field names: $field_names_row" unless $field_names_row =~ /^#([\w\t]+)$/;

    my @field_names = split /\t/, $1, -1;
    if ($type_description->{required_fields}) {
        my $missing_fields = xminus($type_description->{required_fields}, \@field_names);
        if ($missing_fields && @$missing_fields) {
            my $missing_fields_str = join(', ', @$missing_fields);
            return "$INTERNAL_ERRORS_CODE\tFields $missing_fields_str are required but was not found in response header: $field_names_row";
        }
    }
    $self->{_field_names} = \@field_names;

    my $deferred_serialization = $self->_is_deferred_serialization();
    my (@good_lines, @named_lines, @invalid_data);
    for my $row (@lines) {
        # NB: такой же по смыслу код (распарсить строку и применить модификаторы)
        # еще есть ниже, в анонимной функции named_chunks_iterator_over_response
        # в случае правок здесь - нужно будет внести изменения и там.
        my $named_line = $self->_parse_named_line(\$row);

        if (my $error = $self->_check_named_row($named_line)) {
            if ($self->{_separate_invalid_rows}) {
                push @invalid_data, { error => $error, row => $named_line };
                next;
            } else {
                my $error_text = _make_error_string_with_row_dump($error, $named_line);
                return "$INTERNAL_ERRORS_CODE\t$error_text";
            }
        }

        if ($deferred_serialization && $self->{_separate_invalid_rows}) {
            push @good_lines, $row;
        } else {
            $self->_apply_value_modifiers($named_line);

            push @named_lines, $named_line;
        }
    }
    if ($deferred_serialization) {
        $self->{response} = $self->{_separate_invalid_rows} ? \@good_lines : \@lines;
        $self->{data_count} = scalar(@{ $self->{response} });
    } else {
        $self->{named_response} = \@named_lines;
        $self->{data_count} = scalar(@named_lines);
    }

    $self->{invalid_data} = \@invalid_data;
    $self->{invalid_data_count} = scalar(@invalid_data);

    # ставим флажок, что данные успешно загружены и обработаны
    $self->{_loaded} = 1;

    return $self->get_response_bs_status_code();
}

=head3 $self->_check_named_row(\%row)

    Проверяет именованный ответ на соответствие формату из хеша fields_format

    Параметры:
        $row    - ссылка на хеш, содержаший именованные значения строки данных
                  {
                    field_name_1 => YYY,
                    field_name_2 => ZZZ,
                    ...
                  }
    Результат:
        $error  - строка с текстом ошибки или undef

=cut

sub _check_named_row {
    my ($self, $row) = @_;

    my $fields_format = $self->_type_description()->{fields_format};
    return unless $fields_format && %$fields_format;

    for my $field_name (keys %$fields_format) {
        if (!exists $row->{$field_name}) {
            return "$field_name does not exists";
        } elsif (my $error = _check_field_value($row->{$field_name}, $fields_format->{$field_name})) {
            return "Error $error in field $field_name";
        }
    }
    return undef;
}

=head3 _make_error_string_with_row_dump($error, $row)

    if (my $error = some_check($row)) {
        return _make_error_string_with_row_dump($error, $row);
    }

    Возвращает строку, содержащую текст ошибки $error,
        дополненную json-дампом исходной строки $row

    Параметры:
        $error  - строка с текстом ошибки
        $row    - ссылка на хеш с данными
    Результат:
        $err_st - строка с расширенным текстом ошибки

=cut

sub _make_error_string_with_row_dump {
    my ($error, $row) = @_;
    return sprintf('%s. Row: %s', $error, to_json($row));
}

=head2 $self->_apply_value_modifiers(\%row)

    Applies type-dependent value modifiers.
    Modifies argument.

=cut

sub _apply_value_modifiers {
    my ($self, $row) = @_;

    my $value_modifiers = $self->_type_description()->{value_modifiers};
    return unless $value_modifiers && %$value_modifiers;

    while (my ($field, $modifiers) = each %$value_modifiers) {
        next unless exists $row->{$field};

        if (ref($modifiers) ne 'ARRAY') {
             $modifiers = [$modifiers];
        }
        for my $modifier_params (@$modifiers) {
            my $modifier_type;
            if (!ref($modifier_params)) {
                $modifier_type = $modifier_params;
            } elsif (ref($modifier_params) eq 'HASH') {
                $modifier_type = delete $modifier_params->{type};
            } else {
                die "invalid modifier description for field $field";
            }

            my $modifier_description = $VALUE_MODIFIERS{$modifier_type};
            die "no modifier $modifier_type found" unless $modifier_description;
            $row->{$field} = $modifier_description->($row->{$field}, $row);
        }
    }
}

=head2 $error = _check_field_value($value, $format_rules)

    Checks $value aginst field-specific rules

    $format_rules = {
        field_name1 => 'uint',
        field_name1 => {type => 'uint', min => 5, max => 42},
        field_name2 => 'datetime',
    };

=cut

sub _check_field_value {
    my ($value, $format_rules) = @_;

    my ($check_type, $params);
    if (!ref($format_rules)) {
        $check_type = $format_rules;
        $params = {};
    } else {
        $check_type = $format_rules->{type};
        $params = $format_rules;
    }

    my $validatator_data = $FORMAT_VALIDATORS{$check_type};
    die "no validator for $check_type found" unless $validatator_data;

    if ($validatator_data->{code}) {
        my $params = hash_merge {}, $validatator_data->{default_params}, $params;
        return $validatator_data->{code}->($value, $params);
    }

    return undef;
}

=head2
    get_rows_count( )

    receives:
        nothing

    returns:
        how many rows BS was returned. raw value (includes #End, status line, field names line, etc)

=cut

sub get_rows_count
{
    my $self = shift;
    confess "get() method should be called before get_rows_count()." unless $self->{_loaded};
    return $self->{count};
}

=head2 get_data_rows_count($self)

    Получить количество строк данных (размерность $self->{response} или $self->{named_response})

    Параметры:
        $self - объект BSStatImport
    Результат:
        $data_rows_count - количество строк данных (без учета служебной информации), полученных из БК

=cut

sub get_data_rows_count {
    my $self = shift;
    confess "get() method should be called before get_data_rows_count()." unless $self->{_loaded};
    return $self->{data_count};
}

=head2 $self->get_invalid_rows_count()

    Получить количество строк данных (размерность $self->{response} или $self->{named_response})

    Параметры:
        $self - объект BSStatImport
    Результат:
        $data_rows_count - количество строк данных не прошедших валидацию

=cut

sub get_invalid_rows_count {
    my $self = shift;
    confess "get() method should be called before get_invalid_rows_count()." unless $self->{_loaded};
    return $self->{invalid_data_count};
}

=head3 _get_params_for_bs($self)

    Получить хеш с параметрами для отправки в БК.

    Если в описании типа в определен (как функция) params_serializer,
    то для запроса к БК будет использован результат вызова params_serializer($self->{params}),
    иначе - просто $self->{params}

    Параметры:
        $self - объект BSStatImport
    Результат:
        $params - hasref - который $self->{params} или $self->_type_description()->{params_serializer}->($self->{params})

=cut

sub _get_params_for_bs {
    my $self = shift;
    my $type_description = $self->_type_description();

    if (defined $type_description->{params_serializer} && ref $type_description->{params_serializer} eq 'CODE') {
        return $type_description->{params_serializer}->( $self->{params} );
    } else {
        return $self->{params};
    }
}

=head2 get_url( )
    
    receives:
        nothing

    returns:
        url to get statistics from BS

=cut

sub get_url {
    my $self = shift;
    my $type_description = $self->_type_description();

    my ($base_url, $result_url);
    if (defined $self->{_fake_url}) {
        $base_url = $self->{_fake_url};                 # для тестов
    } elsif (defined $type_description->{url}) {        # Старая схема, с урлами в описании
        $base_url = $type_description->{url};
    } else {
        # новая схема (через BS::URL::get)
        my $params_for_chooser = $type_description->{params_for_preprod_chooser};
        if ($params_for_chooser && ref $params_for_chooser eq 'CODE') {
            $base_url = BS::URL::get($self->{type}, $params_for_chooser->( $self->{params} ));
        } else {
            $base_url = BS::URL::get($self->{type});
        }
    }

    if (defined $type_description->{http_parallel_request} && $type_description->{http_parallel_request}->{method} eq 'POST') {
        $result_url = $base_url;
    } else {
        $result_url = Yandex::HTTP::make_url($base_url, $self->_get_params_for_bs());
    }

    return $result_url;
}

=head2 get_info( )

    Works for incremental request only and after get().

    receives:
        nothing

    returns:
        last_orderstat_info value for ppcdict.ppc_properties

=cut

sub get_info
{
    my $self = shift;
    confess "get_info() works for incremental request only." if ! $self->_type_description->{incremental};
    confess "get() method should be called before get_info()." unless $self->{_loaded};
    return $self->{info};
}

=head2 get_response_bs_status_code( )

    Works only after get() method.

    receives:
        nothing

    returns:
        BS error code which is first row of answer from BS.

=cut

sub get_response_bs_status_code
{
    my $self = shift;
    confess "get() method should be called before get_response_bs_status_code()." unless $self->{_loaded};
    return $self->{bs_status_code};
}

=head3 _parse_named_line($self, $line_ref)

    Распарсить текстовую строку с данными в хеш с именованными полями
    my $named_line = $self->_parse_named_line(\$row)

    Параметры:
        $self       - Объект BSStatImport
        $line_ref   - ссылка на строку данных
    Результат:
        $named_line - ссылка на хеш с данными в формате (поле => значение)

=cut

sub _parse_named_line {
    my $self = shift;
    my $line_ref = shift;

    my @values = split(qr/\t/, $$line_ref);
    # отбрасываем лишние столбцы со значениями
    if ($#values > $#{ $self->{_field_names} }) {
        $#values = $#{ $self->{_field_names} };
    }

    return { zip(@{ $self->{_field_names} }, @values) };
}

=head3 _is_deferred_serialization($self)

    Действует ли для полученных данных отложенная сериализация.
    Не действует, если:
        - отложенная сериализация не включена для этого типа данных
        - включена, но число полученных строк данных меньше граничного
    Действует, если:
        - отложенная сериализация включена и число строк больше граничного

    Параметры:
        $self   - Объект BSStatImport
    Результат:
        $deferred_serialization - булев результат (1/"")

=cut

sub _is_deferred_serialization {
    my $self = shift;
    my $type_description = $self->_type_description();
    my $deferred_serialization = $type_description->{deferred_serialization_lines_border_count}
                                 && $self->{count} > $type_description->{deferred_serialization_lines_border_count};
    return $deferred_serialization;
}

=head2 get_named_chunks_iterator($self, $chunk_size)

    Получить итератор, возвращающий массивы сериализованных строк данных (по $chunk_size штук каждый)

    my $named_chunks_iterator = $self->get_named_chunks_iterator();
    while (my $data_chunk = $named_chunks_iterator->()) {
        for my $row_hashref (@$data_chunk) {
            # do something with $row_hashref
            ...
        }
    }

    Параметры:
        $self       - объект BSStatImport
        $chunk_size - максимальный размер чанка (опциональный параметр). по умолчанию равен
                      deferred_serialization_lines_border_count для этого типа статистики
                      или DEFAULT_CHUNK_SIZE, если этот порог не задан в BS_STAT_TYPES.
    Результат
        $iterator_code_ref  - функция, возвращающая ссылки на массивы (чанки) сериализованных строк данных

=cut

sub get_named_chunks_iterator {
    my $self = shift;
    my $type_description = $self->_type_description();
    my $chunk_size = shift // $type_description->{deferred_serialization_lines_border_count} // DEFAULT_CHUNK_SIZE;

    confess "get() method should be called before get_named_chunks_iterator()." unless $self->{_loaded};
    confess "incorrect chunk_size value (should be positive integer)" unless is_valid_int($chunk_size, 1);

    my $i = 0;
    if ($self->_is_deferred_serialization()) {
        return sub {
            local *__ANON__ = 'named_chunks_iterator_over_response';
            my @chunk;
            while (@chunk < $chunk_size && $i <= $#{ $self->{response} }) {
                my $row = $self->_parse_named_line(\( $self->{response}->[ $i++ ] ));
                $self->_apply_value_modifiers($row);
                push @chunk, $row;
            }
            return @chunk ? \@chunk : ();
        };
    } else {
        return sub {
            local *__ANON__ = 'named_chunks_iterator_over_named_response';
            my @chunk;
            while (@chunk < $chunk_size && $i <= $#{ $self->{named_response} }) {
                push @chunk, $self->{named_response}->[ $i++ ];
            }
            return @chunk ? \@chunk : ();
        };
    }
}

=head2 get_named_rows_iterator($self)

    Получить итератор, возвращающий (по одной) сериализованные строки данных (как ссылки на хеш).
    Является частным случаем get_named_chunks_iterator при $chunk_size = 1, при этом дополнительно
        разыменовывается результат итератора (ею возвращенного), т.е. результатом являются строки,
        а не массивы из одной строки.

    my $named_rows_iterator = $self->get_named_rows_iterator();
    while (my $row_hashref = $named_rows_iterator->()) {
        # do something with $row_hashref
        ...
    }

    Параметры:
        $self   - объект BSStatImport
    Результат
        $iterator_code_ref  - функция, возвращающая по одной за раз сериализованные строки данных

=cut

sub get_named_rows_iterator {
    my $self = shift;
    my $type_description = $self->_type_description();

    confess "get() method should be called before get_named_rows_iterator()." unless $self->{_loaded};

    my $chunks_iterator = $self->get_named_chunks_iterator(1);

    return sub {
        local *__ANON__ = 'named_rows_iterator';
        my $row = $chunks_iterator->();
        return $row ? $row->[0] : ();
    };

}

=head2 $data = $self->as_named_arrayref()

    $data = [
        {field_name1 => $field_value1, field_name2 => $field_value2, ...},
        {field_name1 => $field_value1, field_name2 => $field_value2, ...},
        [...]
    ];

=cut

sub as_named_arrayref {
    my $self = shift;
    my $type_description = $self->_type_description();

    confess "get() method should be called before as_named_arrayref()." unless $self->{_loaded};

    if ($self->_is_deferred_serialization()) {
        return $self->get_named_chunks_iterator(2 * $self->get_data_rows_count())->();
    } else {
        return $self->{named_response};
    }
}

=head2 $invalid_data = $self->get_invalid_data()

    Получить данные, не прошедшие валидацию.
    Возвращает ссылку на массив хешей следующего формата:
    {
        error => "тексто ошибки валидации",
        row => $named_line, # hashref - сериализованная строка данных БЕЗ применения value_modifiers
    }

    $invalid_data = [
        {error => "field xx doesn't exists", row => {aa => 1, bb => 22, cc => 3}},
        {error => "Error invalid uint in field Clicks", row => {Clicks => -1, Shows => 2, ...}},
    ];

=cut

sub get_invalid_data {
    my $self = shift;
    confess "get() method should be called before get_invalid_data()." unless $self->{_loaded};
    return $self->{invalid_data};
}

sub _type_description {
    my $self = shift;

    return $BS_STAT_TYPES->{$self->{type}};
}

1;
